/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		  UIS Line Parsing  Routines                  <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>		kvf_parse_startform_line()		      <<<<   
   >>>>		kvf_parse_startmaster_line()		      <<<<   
   >>>>		kvf_parse_startsubmenu_line()		      <<<<   
   >>>>		kvf_parse_startsubform_line()		      <<<<   
   >>>>         kvf_parse_startsubmenu_line()		      <<<<
   >>>>		kvf_parse_subformbutton_line()		      <<<<   
   >>>>		kvf_parse_master_action_line()		      <<<<   
   >>>>		kvf_parse_startguide_line()		      <<<<   
   >>>>		kvf_parse_startpane_line()		      <<<<   
   >>>>		kvf_parse_guide_line()			      <<<<   
   >>>>		kvf_parse_quit_line()			      <<<<   
   >>>>		kvf_parse_subform_action_line()		      <<<<   
   >>>>		kvf_parse_help_line()		      	      <<<<   
   >>>>		kvf_parse_input_line()		      	      <<<<   
   >>>>		kvf_parse_output_line()		      	      <<<<   
   >>>>		kvf_parse_stdin_line()		      	      <<<<   
   >>>>		kvf_parse_stdout_line()		      	      <<<<   
   >>>>		kvf_parse_int_line()		      	      <<<<   
   >>>>		kvf_parse_float_line()		      	      <<<<   
   >>>>		kvf_parse_double_line()		      	      <<<<   
   >>>>		kvf_parse_logic_line()		      	      <<<<   
   >>>>		kvf_parse_string_line()		    	      <<<<   
   >>>>		kvf_parse_stringlist_line()	    	      <<<<   
   >>>>		kvf_parse_routine_line()	      	      <<<<   
   >>>>		kvf_parse_toggle_line()		      	      <<<<   
   >>>>		kvf_parse_blank_line()		      	      <<<<   
   >>>>		kvf_parse_pane_action_line()		      <<<<   
   >>>>		kvf_parse_includepane_line()		      <<<<   
   >>>>		kvf_parse_includesubform_line()		      <<<<   
   >>>>		kvf_parse_mutexcl_line()		      <<<<   
   >>>>		kvf_parse_mutincl_line()		      <<<<   
   >>>>		kvf_parse_group_line()		      	      <<<<   
   >>>>		kvf_parse_workspace_line()		      <<<<   
   >>>>		kvf_parse_libcall_line()		      <<<<   
   >>>>		kvf_parse_list_line()		      	      <<<<   
   >>>>		kvf_parse_cycle_line()		      	      <<<<   
   >>>>		kvf_parse_displaylist_line()		      <<<<   
   >>>>		kvf_parse_end_line()		      	      <<<<   
   >>>>		kvf_gen_parse()	 	             	      <<<<   
   >>>>		kvf_clear_line_info()		      	      <<<<   
   >>>>		kvf_realloc_val_labelnum()	              <<<<   
   >>>>                                                       <<<<
   >>>>   Static:                                             <<<<
   >>>>		create_readerror_mesg()	             	      <<<<   
   >>>>		create_mesg()		             	      <<<<   
   >>>>		check_line_variable()	                      <<<<   
   >>>>		check_UIS_line()	 	              <<<<   
   >>>>                                                       <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"
#include "parser.h"


static int check_line_variable     PROTO((char *, char *, char *));

static char *create_readerror_mesg PROTO((char *, int, int));
static char *create_mesg           PROTO((char *, char *));
static int  check_UIS_line         PROTO((char *, Line_Info *, 
				          unsigned long, char *));
static void unescape_ticmark       PROTO((char *));



/*-----------------------------------------------------------
|
|       Routine: kvf_parse_startform_line
|       Purpose: Parses a StartForm line  (-F)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_startform_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_title[KLENGTH], tmp_var[KLENGTH], flag;
	int  release, version;
	char *mesg, temp[KLENGTH];
	int  status;
	unsigned long mask;

        status = ksscanf(line, startform_scan, &flag, 
			&release, &version,
			&line_info->activate, &line_info->selected,
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			&line_info->xpos, &line_info->ypos, 
			tmp_title, tmp_var);
	

	if (status < 12) 
	{
	    mesg = create_readerror_mesg(line, status, 12);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_startform_line", mesg);
            kfree(mesg);
	    return(FALSE);
   	} 

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);
	
	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_startform_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	ksprintf(temp, "%d.%d", release, version);
	if (kstrcmp(temp, "4.3")!= 0) 
	{
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_startform_line",
	    	   "On -F line of UIS file, 'version' field must be 4.3;  \
the outdated 4.2 version number indicates that your UIS file is out of date.");
	    return(FALSE);
	}
	else line_info->version = 4.2;
	
	line_info->typeflag = KUIS_STARTFORM;

	mask = KUIS_ACT;
	if (!(check_UIS_line(line, line_info, mask,
			         "kvf_parse_startform_line")))
	    return(FALSE);

	return(TRUE);
}



/*------------------------------------------------------------
|
|       Routine: kvf_parse_startmaster_line
|       Purpose: Parses a StartMaster line  (-C)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_startmaster_line(
   char      *line,
   Line_Info *line_info)
{
	char flag, *mesg;
	int status;
	unsigned long mask;

	status = ksscanf(line, startmaster_scan, &flag, &line_info->activate,
			      &line_info->logical_val);

	if (status < 3) 
	{
	    mesg = create_readerror_mesg(line, status, 3);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_startmaster_line", mesg);
	    kfree(mesg);
	    return(FALSE);
   	} 

	line_info->typeflag = KUIS_STARTMASTER;

	mask = KUIS_ACT | KUIS_LOGICALVAL;
	if (!(check_UIS_line(line, line_info, mask,
			         "kvf_parse_startmaster_line")))
	    return(FALSE);

	kfree(line_info->title);
	line_info->title = kstrdup("Master");
	return(TRUE);
}



/*------------------------------------------------------------
|
|       Routine: kvf_parse_startsubmenu_line
|       Purpose: Parses a StartSubmenu line  (-D)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: March 25, 1990
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_startsubmenu_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_title[KLENGTH], tmp_var[KLENGTH], flag;
	char *mesg;
	int  status;
	unsigned long mask;

        status = ksscanf(line, startsubmenu_scan, &flag, 
			&line_info->activate, &line_info->selected, 
		        &line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			tmp_title, tmp_var);
	
	if (status < 9) 
	{
	    mesg = create_readerror_mesg(line, status, 9);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_startsubmenu_line", mesg);
	    kfree(mesg);
	    return(FALSE);
   	} 

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);
	
	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_startsubmenu_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_STARTSUBMENU;

	mask = KUIS_ACT;
	if (!(check_UIS_line(line, line_info, mask,
			         "kvf_parse_startsubmenu_line")))
	    return(FALSE);


	return(TRUE);
}



/*------------------------------------------------------------
|
|       Routine: kvf_parse_startsubform_line
|       Purpose: Parses a StartSubform line  (-M)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_startsubform_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_title[KLENGTH], tmp_var[KLENGTH], flag;
	char *mesg;
	int  status;
	unsigned long mask;

        status = ksscanf(line, startsubform_scan, &flag, 
			&line_info->activate, &line_info->selected, 
		        &line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			&line_info->xpos, &line_info->ypos, 
			tmp_title, tmp_var);
	
	if (status < 11) 
	{
	    mesg = create_readerror_mesg(line, status, 11);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_startsubform_line", mesg);
	    kfree(mesg);
	    return(FALSE);
   	} 

	unescape_ticmark(tmp_title);
	kfree(line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);
	
	kfree(line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_startform_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_STARTSUBFORM;

	mask = KUIS_ACT;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_startsubform_line")))
	    return(FALSE);

	return(TRUE);
}




/*------------------------------------------------------------
|
|       Routine: kvf_parse_startguide_line(line, &line_info);
|       Purpose: Parses a StartGuide line  (-G)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_startguide_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_title[KLENGTH], flag;
	int status;
	char *mesg;
	unsigned long mask;

        status = ksscanf(line, startguide_scan, &flag, 
			&line_info->activate, &line_info->width, 
			&line_info->height, &line_info->x, &line_info->y, 
			&line_info->xpos, &line_info->ypos, tmp_title);
	
	if (status < 9) 
	{
	    mesg = create_readerror_mesg(line, status, 9);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_startguide_line", mesg);
	    kfree(mesg);
            return(FALSE);
	}

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	line_info->typeflag = KUIS_STARTGUIDE;
	mask = KUIS_ACT;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_startguide_line")))
	    return(FALSE);

	return(TRUE);
}


/*------------------------------------------------------------
|
|       Routine: kvf_parse_startpane_line
|       Purpose: Parses a Start Form line  (-P)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_startpane_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_title[KLENGTH], tmp_var[KLENGTH], flag;
	int status;
	char *mesg;
	unsigned long mask;

        status = ksscanf(line, startpane_scan, &flag, 
			&line_info->activate, &line_info->selected,
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			&line_info->xpos, &line_info->ypos, 
			tmp_title, tmp_var);
	

	if (status < 11) 
	{
	    mesg = create_readerror_mesg(line, status, 11);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_startpane_line", mesg);
	    kfree(mesg);
	    return(FALSE);
   	}

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_startpane_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_STARTPANE;

	mask = KUIS_ACT;
	if (!(check_UIS_line(line, line_info, mask,
				  "kvf_parse_startpane_line")))
	    return(FALSE);

	return(TRUE);
}


/*------------------------------------------------------------
|
|       Routine: kvf_parse_subformbutton_line
|       Purpose: Parses a SubformButton line  (-d)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_subformbutton_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_title[KLENGTH], tmp_var[KLENGTH];
	char flag;
	int status;
	char *mesg;
	unsigned long mask;

        status = ksscanf(line, subformbutton_scan, &flag, 
			&line_info->activate, &line_info->selected, 
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, tmp_title, tmp_var);

	if (status < 9) 
	{
	    mesg = create_readerror_mesg(line, status, 9);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_subformbutton_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}

	unescape_ticmark(tmp_title);
	kfree(line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);
	
	kfree (line_info->variable);
	if ((kstrcmp(tmp_var," ")) == 0) line_info->variable = NULL;
	else line_info->variable = kstrdup(tmp_var);
	
	line_info->typeflag = KUIS_SUBFORMBUTTON;

	mask = KUIS_ACT | KUIS_SEL;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_subformbutton_line")))
	    return(FALSE);

	return(TRUE);
}

/*------------------------------------------------------------
|
|       Routine: kvf_parse_master_action_line
|       Purpose: Parses a MasterAction line  (-n)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_master_action_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_desc[KLENGTH], tmp_title[KLENGTH], tmp_var[KLENGTH], 
	     flag;
	int status;
	char *mesg;
	unsigned long mask;

        status = ksscanf(line, master_action_scan, &flag, 
			&line_info->activate, &line_info->selected, 
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			tmp_title, tmp_desc, tmp_var);
	
	if (status < 10) 
	{
	    mesg = create_readerror_mesg(line, status, 10);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_master_action_line", mesg);
	    kfree(mesg);
	    return(FALSE);
    	}

	
	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_master_action_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_MASTERACTION;

	mask = KUIS_ACT | KUIS_SEL;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_master_action_line")))
	    return(FALSE);

	return(TRUE);
}


/*------------------------------------------------------------
|
|       Routine: kvf_parse_guide_line
|       Purpose: Parses a Guide line  (-g)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_guide_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_title[KLENGTH], tmp_var[KLENGTH];
	char flag;
	int status;
	char *mesg;
	unsigned long mask;

        status = ksscanf(line, guidebutton_scan, &flag, 
			&line_info->activate, &line_info->selected, 
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, tmp_title, tmp_var);

	if (status < 9) 
	{
	    mesg = create_readerror_mesg(line, status, 9);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_guide_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	kfree (line_info->variable);
	if ((kstrcmp(tmp_var," ")) == 0) line_info->variable = NULL;
	else line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_GUIDEBUTTON;

	mask = KUIS_ACT | KUIS_SEL;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_guide_line")))
	    return(FALSE);

	return(TRUE);
}


/*------------------------------------------------------------
|
|       Routine: kvf_parse_quit_line
|       Purpose: Parses a Quit line  (-Q)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_quit_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_title[KLENGTH], flag;
	int status;
	char *mesg;
	unsigned long mask;

        status = ksscanf(line, quitform_scan, &flag, 
			&line_info->activate, &line_info->selected,
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, tmp_title);
	
	if (status < 8) 
	{
	    mesg = create_readerror_mesg(line, status, 8);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_quit_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	kfree(line_info->variable);
	line_info->variable = kstrdup("quit");

	line_info->typeflag = KUIS_QUIT;

	mask = KUIS_ACT | KUIS_SEL;
	if (!(check_UIS_line(line, line_info, mask, 
				 "kvf_parse_quit_line")))
	    return(FALSE);

	return(TRUE);
}


/*------------------------------------------------------------
|
|       Routine: kvf_parse_subform_action_line
|       Purpose: Parses a SubformAction line  (-m)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_subform_action_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_title[KLENGTH], tmp_desc[KLENGTH], tmp_var[KLENGTH], 
	     flag;
	int status;
	char *mesg;
	unsigned long mask;

        status = ksscanf(line, subform_action_scan, &flag, 
			&line_info->activate, &line_info->selected, 
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			tmp_title, tmp_desc, tmp_var);
	
	if (status < 10) 
	{
	    mesg = create_readerror_mesg(line, status, 10);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_subform_action_line", mesg);
	    kfree(mesg);
	    return(FALSE);
    	}

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_subform_action_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_SUBFORMACTION;

	mask = KUIS_ACT | KUIS_SEL;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_subform_action_line")))
	    return(FALSE);

	return(TRUE);
}


/*------------------------------------------------------------
|
|       Routine: kvf_parse_help_line
|       Purpose: Parses a Help line  (-H)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_help_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_desc[KLENGTH], tmp_title[KLENGTH], 
	     tmp_file[KLENGTH], tmp_var[KLENGTH];
	char flag;
	int  status;
	char *mesg;
	unsigned long mask;

        status = ksscanf(line, help_scan, &flag, 
			&line_info->activate, &line_info->width, 
			&line_info->height, &line_info->x, &line_info->y, 
			tmp_title, tmp_desc, tmp_file, tmp_var);

	if (status < 10) 
	{
	    mesg = create_readerror_mesg(line, status, 10);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_help_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kfree (line_info->help_file);
	if ((kstrcmp(tmp_file," ")) == 0) line_info->help_file = NULL;
	else line_info->help_file = kstrdup(tmp_file);

	kfree (line_info->variable);
	if ((kstrcmp(tmp_var," ")) == 0) line_info->variable = NULL;
	else line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_HELP;

	mask = KUIS_ACT;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_help_line")))
	    return(FALSE);

	return(TRUE);
}



/*------------------------------------------------------------
|
|       Routine: kvf_parse_input_line
|       Purpose: Parses an KUIS_INPUTFILE line  (-I)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|          Date: Jul 14, 1992
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_input_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_desc[KLENGTH], tmp_title[KLENGTH], tmp_var[KLENGTH], 
	     tmp_filename[KLENGTH], tmp_default[KLENGTH], 
	     tmp_literal[KLENGTH];
	char flag;
	int status;
	char *mesg;
	unsigned long mask;

        status = ksscanf(line, input_scan, &flag, 
			&line_info->activate, &line_info->selected,  
			&line_info->optional, &line_info->opt_sel,
			&line_info->live, &line_info->file_check,
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			tmp_default, tmp_title, tmp_desc, 
			tmp_var, tmp_filename, tmp_literal);
	
	if (status == 17)
        {
            kfree (line_info->literal);
            if ((kstrcmp(tmp_literal," ")) == 0) line_info->literal = NULL;
            else line_info->literal = kstrdup(tmp_literal);

            kfree (line_info->filename);
	    if ((kstrcmp(tmp_filename," ")) == 0) line_info->filename = NULL;
            else line_info->filename = kstrdup(tmp_filename);

            kfree (line_info->file_def);
            if ((kstrcmp(tmp_default," ")) == 0) line_info->file_def = NULL;
            else line_info->file_def = kstrdup(tmp_default);
        }

	else if (status == 16) 
	{
            kfree (line_info->filename);
            kfree (line_info->literal);

	    if ((kstrcmp(tmp_filename," ")) == 0) 
	    {
		line_info->filename = NULL;
		line_info->literal = NULL;
	    }
	    else 
	    { 
	        line_info->filename = kstrdup(tmp_filename);
                line_info->literal = kstrdup(tmp_filename);
	    }

            kfree (line_info->file_def);
	    if ((kstrcmp(tmp_default," ")) == 0) line_info->file_def = NULL;
            else line_info->file_def = kstrdup(tmp_default);
	}

	else if (status == 15) 
	{
            kfree (line_info->filename);
            kfree (line_info->literal);
            kfree (line_info->file_def);

	    if ((kstrcmp(tmp_default, " ")) == 0)
	    {
	 	line_info->filename = NULL;
	 	line_info->literal  = NULL;
	 	line_info->file_def = NULL;
	    }
	    else 
	    {
                line_info->file_def = kstrdup(tmp_default);
                line_info->filename = kstrdup(tmp_default);
                line_info->literal  = kstrdup(tmp_default);
	    }
	}

	else if (status < 15) 
	{
	    mesg = create_readerror_mesg(line, status, 15);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_input_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_input_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_INPUTFILE;

	mask = KUIS_ACT | KUIS_SEL | KUIS_OPT | KUIS_LIVE | 
	       KUIS_FILECHECK | KUIS_INFILEDEF;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_input_line")))
	    return(FALSE);

	return(TRUE);
}



/*------------------------------------------------------------
|
|       Routine: kvf_parse_output_line
|       Purpose: Parses an Output line  (-O)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|          Date: Jul 14, 1992
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_output_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_desc[KLENGTH], tmp_title[KLENGTH], tmp_var[KLENGTH], 
	     tmp_filename[KLENGTH], tmp_default[KLENGTH], 
	     tmp_literal[KLENGTH];
	char flag;
	int  status;
	char *mesg;
	unsigned long mask;

        status = ksscanf(line, output_scan, &flag, 
			&line_info->activate, &line_info->selected, 
			&line_info->optional, &line_info->opt_sel,
		        &line_info->live, &line_info->file_check,
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			tmp_default, tmp_title, tmp_desc, 
			tmp_var, tmp_filename, tmp_literal);
	
	if (status == 17)
        {
            kfree (line_info->literal);
            if ((kstrcmp(tmp_literal," ")) == 0) line_info->literal = NULL;
            else line_info->literal = kstrdup(tmp_literal);

            kfree (line_info->filename);
	    if ((kstrcmp(tmp_filename," ")) == 0) line_info->filename = NULL;
	    else line_info->filename = kstrdup(tmp_filename);

            kfree (line_info->file_def);
	    if ((kstrcmp(tmp_default," ")) == 0) line_info->file_def = NULL;
	    else line_info->file_def = kstrdup(tmp_default);
        }

	else if (status == 16) 
	{
	    kfree (line_info->filename);
	    kfree (line_info->literal);

	    if ((kstrcmp(tmp_filename," ")) == 0) 
	    {
		line_info->filename = NULL;
		line_info->literal  = NULL;
	    }
	    else 
	    { 
                line_info->filename = kstrdup(tmp_filename);
                line_info->literal = kstrdup(tmp_filename);
	    }

            kfree (line_info->file_def);
	    if ((kstrcmp(tmp_default," ")) == 0) line_info->file_def = NULL;
	    else line_info->file_def = kstrdup(tmp_default);
	}

	else if (status == 15) 
	{
            kfree (line_info->filename);
            kfree (line_info->literal);
	    kfree (line_info->file_def);

	    if ((kstrcmp(tmp_default, " ")) == 0)
	    {
		line_info->filename = NULL;
                line_info->literal = NULL;
                line_info->file_def = NULL;
	    }
	    else 
	    {
                line_info->file_def = kstrdup(tmp_default);
                line_info->filename = kstrdup(tmp_default);
                line_info->literal  = kstrdup(tmp_default);
	    }
	}

	else if (status < 15) 
	{
	    mesg = create_readerror_mesg(line, status, 15);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_output_line", mesg);
	    kfree(mesg);
	    return(FALSE);
    	}

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_output_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_OUTPUTFILE;

	mask = KUIS_ACT | KUIS_SEL | KUIS_OPT | KUIS_LIVE | KUIS_FILECHECK;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_output_line")))
	    return(FALSE);

	return(TRUE);
}


/*------------------------------------------------------------
|
|       Routine: kvf_parse_stdin_line
|       Purpose: Parses a stdin line  (-e)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: April 26, 1994    
|    Written By: Danielle Argiro 
| Modifications: 
|
------------------------------------------------------------*/

int kvf_parse_stdin_line(
   char      *line,
   Line_Info *line_info)
{
	char flag;
	int status;
	char *mesg;
	char tmp_title[KLENGTH], tmp_var[KLENGTH], tmp_filename[KLENGTH];

        status = ksscanf(line, stdin_scan, &flag, &line_info->activate, 
			&line_info->optional, &line_info->opt_sel,
	                &line_info->x, &line_info->y, 
			tmp_title, tmp_var, tmp_filename); 

	if (status == 9)
	{
	    kfree (line_info->filename);
            if ((kstrcmp(tmp_filename," ")) == 0) line_info->filename = NULL;
            else line_info->filename = kstrdup(tmp_filename);
	}
	else if (status == 8)
	{
	    kfree (line_info->filename);
	    line_info->filename = NULL;
	}
	else if (status < 8) 
	{
	    mesg = create_readerror_mesg(line, status, 8);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_stdin_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	kfree (line_info->variable);
	if ((kstrcmp(tmp_var," ")) == 0) line_info->variable = NULL;
	else line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_STDIN;
	return(TRUE);
}




/*------------------------------------------------------------
|
|       Routine: kvf_parse_stdout_line
|       Purpose: Parses a stdout line  (-o)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: April 26, 1994    
|    Written By: Danielle Argiro 
| Modifications: 
|
------------------------------------------------------------*/

int kvf_parse_stdout_line(
   char      *line,
   Line_Info *line_info)
{
	char flag;
	int status;
	char *mesg;
	char tmp_title[KLENGTH], tmp_var[KLENGTH], tmp_filename[KLENGTH];

        status = ksscanf(line, stdout_scan, &flag, &line_info->activate, 
			&line_info->optional, &line_info->opt_sel,
	                &line_info->x, &line_info->y, 
			tmp_title, tmp_var, tmp_filename); 

        if (status == 9)
        {
            kfree (line_info->filename);
            if ((kstrcmp(tmp_filename," ")) == 0) line_info->filename = NULL;
            else line_info->filename = kstrdup(tmp_filename);
        }
        else if (status == 8)
        {
            kfree (line_info->filename);
            line_info->filename = NULL;
        }
        else if (status < 8) 
	{
	    mesg = create_readerror_mesg(line, status, 8);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_stdout_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	kfree (line_info->variable);
	if ((kstrcmp(tmp_var," ")) == 0) line_info->variable = NULL;
	else line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_STDOUT;
	return(TRUE);
}


/*------------------------------------------------------------
|
|       Routine: kvf_parse_int_line
|       Purpose: Parses an Integer line  (-i)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_int_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_desc[KLENGTH], tmp_title[KLENGTH], tmp_var[KLENGTH], 
	     tmp_literal[KLENGTH], temp[KLENGTH];
	char flag;
	int  status;
	char *mesg; 
	unsigned long mask;

        status = ksscanf(line, int_scan, &flag, 
			&line_info->activate, &line_info->selected, 
			&line_info->optional, &line_info->opt_sel,
			&line_info->live,
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			&line_info->lower_int, &line_info->upper_int, 
			&line_info->int_def, &line_info->special,
			&line_info->int_incr, tmp_title, tmp_desc, 
			tmp_var, &line_info->int_val, tmp_literal);

	if (status == 20)
        {
            kfree (line_info->literal);
            if ((kstrcmp(tmp_literal," ")) == 0) line_info->literal = NULL;
            else line_info->literal = kstrdup(tmp_literal);
        }

	else if (status == 19)
	{
	    if (!(kvf_check_double((double) line_info->int_val, 
			           (double) line_info->upper_int,
                                   (double) line_info->lower_int, 
				   tmp_var, tmp_title)))
		return(FALSE);
		
	    ksprintf(temp, "%d", line_info->int_val);
	    line_info->literal = kstrdup(temp);

	}
	else if (status == 18)  
	{
	    if (!(kvf_check_double((double) line_info->int_def, 
				   (double) line_info->upper_int,
                                   (double) line_info->lower_int, 
				   tmp_var, tmp_title)))
		return(FALSE);

	   line_info->int_val = line_info->int_def;
	   ksprintf(temp, "%d", line_info->int_def);

	   kfree(line_info->literal);
	   line_info->literal = kstrdup(temp);

	} /* end if */

	else if (status < 18) 
	{
	     mesg = create_readerror_mesg(line, status, 18);
	     errno = KUIS_SYNTAX;
	     kerror("kforms", "kvf_parse_int_line", mesg);
	     kfree(mesg);
	     return(FALSE);
	}

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_int_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_INTEGER;

	mask = KUIS_ACT | KUIS_SEL | KUIS_OPT | KUIS_LIVE | KUIS_MECH;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_int_line")))
	    return(FALSE);

	return(TRUE);
}



/*------------------------------------------------------------
|
|       Routine: kvf_parse_float_line
|       Purpose: Parses a Float line  (-f)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_float_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_desc[KLENGTH], tmp_title[KLENGTH], tmp_var[KLENGTH], 
	     tmp_literal[KLENGTH], temp[KLENGTH], scale[KLENGTH];
	char flag;
	int  status;
	char *mesg;
	unsigned long mask;

        status = ksscanf(line, float_scan, &flag, 
			&line_info->activate, &line_info->selected, 
			&line_info->optional,  &line_info->opt_sel,
			&line_info->live,
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			&line_info->lower_float, &line_info->upper_float, 
			&line_info->float_def, &line_info->precision,
			&line_info->special, &line_info->flt_incr,
			tmp_title, tmp_desc, tmp_var, 
			&line_info->float_val, tmp_literal);

	if (status == 21)
        {
            kfree (line_info->literal);
            if ((kstrcmp(tmp_literal," ")) == 0) line_info->literal = NULL;
            else line_info->literal = kstrdup(tmp_literal);
        }

	else if (status == 20)
	{

	    if (!(kvf_check_double((double) line_info->float_val, 
			           (double) line_info->upper_float,
			           (double) line_info->lower_float, 
				   tmp_var, tmp_title)))
		return(FALSE);

	    if (line_info->precision == 0)
	        ksprintf(scale, "%%g");
	    else if (line_info->precision == -1)
	        ksprintf(scale, "%%f");
	    else ksprintf(scale, "%%.%df", line_info->precision);
	    ksprintf(temp, scale, line_info->float_val);
	    kfree(line_info->literal);
	    line_info->literal = kstrdup(temp);
	}

	else if (status == 19)  
	{
	    if (!(kvf_check_double((double) line_info->float_def, 
				   (double) line_info->upper_float,
				   (double) line_info->lower_float,
				    tmp_var, tmp_title)))
		return(FALSE);

	    line_info->float_val = line_info->float_def;
	    if (line_info->precision == 0)
	        ksprintf(scale, "%%g");
	    else if (line_info->precision == -1)
	        ksprintf(scale, "%%f");
	    else ksprintf(scale, "%%.%df", line_info->precision);
	    ksprintf(temp, scale, line_info->float_def);
	    kfree(line_info->literal);
	    line_info->literal = kstrdup(temp);
	} 


	else if (status < 19) 
	{
	      mesg = create_readerror_mesg(line, status, 19);
	      errno = KUIS_SYNTAX;
	      kerror("kforms", "kvf_parse_float_line", mesg);
	      kfree(mesg);
	      return(FALSE);
      	}

	
	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_float_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_FLOAT;

	mask = KUIS_ACT  | KUIS_SEL  | KUIS_OPT | 
               KUIS_LIVE | KUIS_MECH | KUIS_PREC;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_float_line")))
	    return(FALSE);

	return(TRUE);
}



/*------------------------------------------------------------
|
|       Routine: kvf_parse_double_line
|       Purpose: Parses a Double line  (-h)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_double_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_desc[KLENGTH], tmp_title[KLENGTH], tmp_var[KLENGTH], 
	     tmp_literal[KLENGTH], temp[KLENGTH],   scale[KLENGTH];
	char flag;
	int  status;
	char *mesg;
	unsigned long mask;

        status = ksscanf(line, double_scan, &flag, 
			&line_info->activate, &line_info->selected, 
			&line_info->optional,  &line_info->opt_sel,
			&line_info->live,
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			&line_info->lower_double, &line_info->upper_double, 
			&line_info->double_def, &line_info->precision,
			&line_info->special, &line_info->dbl_incr,
			tmp_title, tmp_desc, 
			tmp_var, &line_info->double_val, tmp_literal);

	if (status == 21)
        {
            kfree (line_info->literal);
            if ((kstrcmp(tmp_literal," ")) == 0) line_info->literal = NULL;
            else line_info->literal = kstrdup(tmp_literal);
        }

	else if (status == 20)
	{

	    if (!(kvf_check_double(line_info->double_val, 
				   line_info->upper_double,
                                   line_info->lower_double,
				   tmp_var, tmp_title)))
		return(FALSE);

	    if (line_info->precision == 0)
	        ksprintf(scale, "%%g");
	    else if (line_info->precision == -1)
	        ksprintf(scale, "%%f");
	    else ksprintf(scale, "%%.%df", line_info->precision);
	    ksprintf(temp, scale, line_info->double_val);
	    kfree(line_info->literal);
	    line_info->literal = kstrdup(temp);

	}

	else if (status == 19)  
	{
	    if (!(kvf_check_double(line_info->double_def, 
				   line_info->upper_double,
                                   line_info->lower_double,
				   tmp_var, tmp_title)))
		return(FALSE);
	    line_info->double_val = line_info->double_def;

	    if (line_info->precision == 0)
	        ksprintf(scale, "%%g");
	    else if (line_info->precision == -1)
	        ksprintf(scale, "%%f");
	    else ksprintf(scale, "%%.%df", line_info->precision);
	    ksprintf(temp, scale, line_info->double_def);
	    kfree(line_info->literal);
	    line_info->literal = kstrdup(temp);
	} 

	else if (status < 19) 
	{
	      mesg = create_readerror_mesg(line, status, 17);
	      errno = KUIS_SYNTAX;
	      kerror("kforms", "kvf_parse_double_line", mesg);
	      kfree(mesg);
	      return(FALSE);
      	}

	
	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_double_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_DOUBLE;

	mask = KUIS_ACT  | KUIS_SEL  | KUIS_OPT | 
               KUIS_LIVE | KUIS_MECH | KUIS_PREC;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_double_line")))
	    return(FALSE);

	return(TRUE);
}



/*------------------------------------------------------------
|
|       Routine: kvf_parse_logic_line
|       Purpose: Parses a Logic line  (-l)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|          Date: Jul 14, 1992
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/
int kvf_parse_logic_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_desc[KLENGTH], tmp_title[KLENGTH], tmp_var[KLENGTH];
	char tmp_logic0[KLENGTH], tmp_logic1[KLENGTH]; 
	char flag;
	int  status;
	char *mesg;
	unsigned long mask;

        status = ksscanf(line, logic_scan, &flag, 
			&line_info->activate, &line_info->selected, 
			&line_info->optional, &line_info->opt_sel,
			&line_info->live,
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			&line_info->logical_def, tmp_title, 
			tmp_logic0, tmp_logic1, tmp_desc, tmp_var, 
			&line_info->logical_val);
	
	if (status == 16) line_info->logical_val = line_info->logical_def;

	else if (status < 16) 
	{
	    mesg = create_readerror_mesg(line, status, 16);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_logic_line", mesg);
	    kfree(mesg);
	    return(FALSE);
       	}

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kvf_realloc_val_labelnum(line_info, 2);

	unescape_ticmark(tmp_logic0);
	kfree (line_info->val_labels[0]);
	if ((kstrcmp(tmp_logic0," ")) == 0) line_info->val_labels[0] = NULL;
	else line_info->val_labels[0] = kstrdup(tmp_logic0);

	unescape_ticmark(tmp_logic1);
	kfree (line_info->val_labels[1]);
	if ((kstrcmp(tmp_logic1," ")) == 0) line_info->val_labels[1] = NULL;
	else line_info->val_labels[1] = kstrdup(tmp_logic1);

	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_logic_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_LOGICAL;

	mask = KUIS_ACT | KUIS_SEL | KUIS_OPT | KUIS_LIVE | 
		KUIS_LOGICALVAL | KUIS_LOGICALDEF;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_logic_line")))
	    return(FALSE);

	return(TRUE);
}



/*------------------------------------------------------------
|
|       Routine: kvf_parse_flag_line
|       Purpose: Parses a Flag line  (-t)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jun 26, 1992
|    Written By: Danielle Argiro 
| Modifications: 
|
------------------------------------------------------------*/
int kvf_parse_flag_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_desc[KLENGTH], tmp_title[KLENGTH], tmp_var[KLENGTH];
	char flag;
	int status;
	char *mesg;
	unsigned long mask;

        status = ksscanf(line, flag_scan, &flag, 
			&line_info->activate, &line_info->selected, 
			&line_info->optional, &line_info->opt_sel,
			&line_info->live,
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			tmp_title, tmp_desc, tmp_var);
	
	if (status < 13) 
	{
	    mesg = create_readerror_mesg(line, status, 13);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_flag_line", mesg);
	    kfree(mesg);
	    return(FALSE);
       	}

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_flag_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_FLAG;

	mask = KUIS_ACT | KUIS_SEL | KUIS_OPT | KUIS_LIVE | KUIS_FLG;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_flag_line")))
	    return(FALSE);

	return(TRUE);
}





/*------------------------------------------------------------
|
|       Routine: kvf_parse_cycle_line
|       Purpose: Parses a KUIS_CYCLE line  (-c)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|          Date: Jul 14, 1992
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_cycle_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_desc[KLENGTH], tmp_title[KLENGTH], tmp_var[KLENGTH];
	char tmp_label[KLENGTH]; 
	char flag;
	int status, indx, j;
	char *mesg, *scanstr;
	unsigned long mask;

        status = ksscanf(line, cycle_scan, &flag, 
			&line_info->activate, &line_info->selected, 
			&line_info->optional, &line_info->opt_sel,
			&line_info->live,
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			&line_info->list_num, 
			&line_info->int_val,
			&line_info->list_val,
			tmp_title, tmp_desc, tmp_var);
	
	if (status < 16) 
	{
	    mesg = create_readerror_mesg(line, status, 16);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_cycle_line", mesg);
	    kfree(mesg);
	    return(FALSE);
       	}

        indx = 0;
        if (kstrlen(tmp_title) != 0)
        {
            while ((kstrncmp(&line[indx], tmp_title, kstrlen(tmp_title))) != 0)
                indx++;
            indx+= kstrlen(tmp_title);
        }
        if (kstrlen(tmp_desc) != 0)
        {
            while ((kstrncmp(&line[indx], tmp_desc, kstrlen(tmp_desc))) != 0)
                indx++;
            indx+= kstrlen(tmp_desc);
        }
        if (kstrlen(tmp_var) != 0)
        {
            while ((kstrncmp(&line[indx], tmp_var, kstrlen(tmp_var))) != 0)
                indx++;
            indx+= kstrlen(tmp_var);
        }

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_cycle_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_CYCLE;

	mask = KUIS_ACT | KUIS_SEL | KUIS_OPT | KUIS_LIVE | KUIS_CYCLEVAL;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_cycle_line")))
	    return(FALSE);

	kvf_realloc_val_labelnum(line_info, line_info->list_num);

	scanstr = "%*[ ]%*[']%[^']";
	for (j = 0; j < line_info->list_num; j++)
	{
	    status = ksscanf(&line[indx], scanstr, tmp_label);
	    if (status != 1) 
	    {
	        mesg = create_mesg(line, 
		       "Unable to read correct number of cycle labels");
	        errno = KUIS_SYNTAX;
	        kerror("kforms", "kvf_parse_cycle_line", mesg);
		kfree(mesg);
	        return(FALSE);
	    }
	    indx = indx + kstrlen(tmp_label) + 3;

	    kfree(line_info->val_labels[j]);
	    unescape_ticmark(tmp_label);
	    line_info->val_labels[j] = kstrdup(tmp_label);

	    if (j == 0)
	       scanstr = "%*[ ]%*[']%[^']%*[ ']";
	}

	return(TRUE);
}




/*------------------------------------------------------------
|
|       Routine: kvf_parse_list_line
|       Purpose: Parses a List line  (-x)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_list_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_desc[KLENGTH], tmp_title[KLENGTH], tmp_var[KLENGTH];
	char tmp_label[KLENGTH]; 
	char flag;
	int  status, indx, j;
	char *mesg, *scanstr;
	unsigned long mask;

        status = ksscanf(line, list_scan, &flag, 
			&line_info->activate, &line_info->selected, 
			&line_info->optional, &line_info->opt_sel,
			&line_info->live,
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			&line_info->list_num, 
			&line_info->int_val,
			&line_info->list_val,
			tmp_title, tmp_desc, tmp_var);
	
	if (status < 16) 
	{
	    mesg = create_readerror_mesg(line, status, 16);
            errno = KUIS_SYNTAX;
            kerror("kforms", "kvf_parse_list_line", mesg);
            kfree(mesg);
            return(FALSE);
       	}

        indx = 0;
        if (kstrlen(tmp_title) != 0)
        {
            while ((kstrncmp(&line[indx], tmp_title, kstrlen(tmp_title))) != 0)
                indx++;
            indx+= kstrlen(tmp_title);
        }
        if (kstrlen(tmp_desc) != 0)
        {
            while ((kstrncmp(&line[indx], tmp_desc, kstrlen(tmp_desc))) != 0)
                indx++;
            indx+= kstrlen(tmp_desc);
        }
        if (kstrlen(tmp_var)!= 0)
        {
            while ((kstrncmp(&line[indx], tmp_var, kstrlen(tmp_var))) != 0)
                indx++;
            indx+= kstrlen(tmp_var);
        }

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_list_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_LIST;

	mask = KUIS_ACT | KUIS_SEL | KUIS_OPT | KUIS_LIVE | KUIS_LISTVAL;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_list_line")))
	    return(FALSE);

	kvf_realloc_val_labelnum(line_info, line_info->list_num);

	scanstr = "%*[ ]%*[']%[^']";
	for (j = 0; j < line_info->list_num; j++)
	{
	    status = ksscanf(&line[indx], scanstr, tmp_label);
	    if (status != 1) 
	    {
	        mesg = create_mesg(line, 
		       "Unable to read correct number of list labels");
	        errno = KUIS_SYNTAX;
	        kerror("kforms", "kvf_parse_list_line", mesg);
		kfree(mesg);
	        return(FALSE);
	    }
	    indx = indx + kstrlen(tmp_label) + 3;

	    kfree(line_info->val_labels[j]);
	    unescape_ticmark(tmp_label);
	    line_info->val_labels[j] = kstrdup(tmp_label);

	    if ( j == 0 )
	       scanstr = "%*[ ]%*[']%[^']%*[ ']";
	}

	return(TRUE);
}




/*------------------------------------------------------------
|
|       Routine: kvf_parse_displaylist_line
|       Purpose: Parses a DisplayList line  (-z)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jan 18, 1994
|    Written By: Danielle Argiro
| Modifications: 
|
------------------------------------------------------------*/

int kvf_parse_displaylist_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_desc[KLENGTH], tmp_title[KLENGTH], tmp_var[KLENGTH];
	char tmp_label[KLENGTH]; 
	char flag;
	int  status, indx, j;
	char *mesg, *scanstr;
	unsigned long mask;

        status = ksscanf(line, displaylist_scan, &flag, 
			&line_info->activate, &line_info->selected, 
			&line_info->optional, &line_info->opt_sel,
			&line_info->live,
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			&line_info->list_num, 
			&line_info->int_val,
			&line_info->list_val,
			&line_info->special,
			tmp_title, tmp_desc, tmp_var);
	
	if (status < 17) 
	{
	    mesg = create_readerror_mesg(line, status, 17);
	    return(FALSE);
       	}

        indx = 0;
        if (kstrlen(tmp_title) != 0)
        {
            while ((kstrncmp(&line[indx], tmp_title, kstrlen(tmp_title))) != 0)
                indx++;
            indx+= kstrlen(tmp_title);
        }
        if (kstrlen(tmp_desc) != 0)
        {
            while ((kstrncmp(&line[indx], tmp_desc, kstrlen(tmp_desc))) != 0)
                indx++;
            indx+= kstrlen(tmp_desc); 
        }
        if (kstrlen(tmp_var) != 0)
        {
            while ((kstrncmp(&line[indx], tmp_var, kstrlen(tmp_var))) != 0)
                indx++;
            indx+= kstrlen(tmp_var);
        }

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_displaylist_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_DISPLAYLIST;

	mask = KUIS_ACT | KUIS_SEL | KUIS_OPT | KUIS_LIVE | KUIS_LISTVAL;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_displaylist_line")))
	    return(FALSE);

	kvf_realloc_val_labelnum(line_info, line_info->list_num);

	scanstr = "%*[ ]%*[']%[^']";
	for (j = 0; j < line_info->list_num; j++)
	{
	    status = ksscanf(&line[indx], scanstr, tmp_label);
	    if (status != 1) 
	    {
	        mesg = create_mesg(line, 
		       "Unable to read correct number of list labels");
	        errno = KUIS_SYNTAX;
	        kerror("kforms", "kvf_parse_displaylist_line", mesg);
		kfree(mesg);
	        return(FALSE);
	    }

	    indx = indx + kstrlen(tmp_label) + 3;

	    kfree(line_info->val_labels[j]);
	    unescape_ticmark(tmp_label);
	    line_info->val_labels[j] = kstrdup(tmp_label);

	    if ( j == 0 )
	       scanstr = "%*[ ]%*[']%[^']%*[ ']";
	}

	return(TRUE);
}

/*------------------------------------------------------------
|
|       Routine: kvf_parse_string_line
|       Purpose: Parses a String line  (-s)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_string_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_desc[KLENGTH], tmp_title[KLENGTH], tmp_var[KLENGTH], 
	     tmp_default[10*KLENGTH], tmp_string[25*KLENGTH], 
	     tmp_literal[25*KLENGTH];
	char flag;
	int  status;
	char *mesg;
	unsigned long mask;

        status = ksscanf(line, string_scan, &flag, 
			&line_info->activate, &line_info->selected, 
			&line_info->optional, &line_info->opt_sel,
			&line_info->live,
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			tmp_default, tmp_title, tmp_desc, 
			tmp_var, tmp_string, tmp_literal);
	
	if (status == 16)
        {
	    unescape_ticmark(tmp_literal);
            kfree (line_info->literal);
            if ((kstrcmp(tmp_literal," ")) == 0) line_info->literal = NULL;
            else line_info->literal = kstrdup(tmp_literal);

	    unescape_ticmark(tmp_string);
            kfree (line_info->string_val);
	    if ((kstrcmp(tmp_string," ")) == 0) line_info->string_val = NULL;
            else line_info->string_val = kstrdup(tmp_string);

	    unescape_ticmark(tmp_default);
            kfree (line_info->string_def);
	    if ((kstrcmp(tmp_default," ")) == 0) line_info->string_def = NULL;
            else line_info->string_def = kstrdup(tmp_default);
        }

	else if (status == 15) 
	{
	    kfree (line_info->string_val);
	    if (line_info->literal != NULL) kfree (line_info->literal);

	    if ((kstrcmp(tmp_string," ")) == 0) 
	    {
		line_info->string_val = NULL;
		line_info->literal = NULL;
	    }
	    else 
	    { 
	        unescape_ticmark(tmp_string);
	        line_info->string_val = kstrdup(tmp_string);
	        line_info->literal = kstrdup(tmp_string);
	    }

	    unescape_ticmark(tmp_default);
            kfree (line_info->string_def);
	    if ((kstrcmp(tmp_default," ")) == 0) line_info->string_def = NULL;
            else line_info->string_def = kstrdup(tmp_default);
	 }

	else if (status == 14) 
	{
	    kfree (line_info->string_def);
	    kfree (line_info->string_val);
	    kfree (line_info->literal);

	    if ((kstrcmp(tmp_default, " ")) == 0)
	    {
	 	line_info->string_val = NULL;
	 	line_info->literal    = NULL;
	 	line_info->string_def = NULL;
	    }
	    else 
	    {
	        unescape_ticmark(tmp_default);
	        line_info->string_def = kstrdup(tmp_default);
	        line_info->string_val = kstrdup(tmp_default);
	        line_info->literal    = kstrdup(tmp_default);
	    }
	}

	if (status < 14) 
	{
	    mesg = create_readerror_mesg(line, status, 14);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_string_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}

	unescape_ticmark(tmp_default);
	kfree (line_info->string_def);
	if ((kstrcmp(tmp_default," ")) == 0) line_info->string_def = NULL;
	else line_info->string_def = kstrdup(tmp_default);

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_string_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_STRING;

	mask = KUIS_ACT | KUIS_SEL | KUIS_OPT | KUIS_LIVE;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_string_line")))
	    return(FALSE);

	return(TRUE);
}




/*------------------------------------------------------------
|
|       Routine: kvf_parse_stringlist_line
|       Purpose: Parses a StringList line  (-y)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Sep 17, 1992
|    Written By: Danielle Argiro 
| Modifications: 
|
------------------------------------------------------------*/

int kvf_parse_stringlist_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_desc[KLENGTH], tmp_title[KLENGTH], tmp_var[KLENGTH], 
	     tmp_string[25*KLENGTH], tmp_literal[25*KLENGTH], 
	     tmp_label[KLENGTH];
	char flag;
	int  status, indx, j;
	char *mesg, *scanstr; 
	unsigned long mask;
	char *scanstr1 = "%*[ ]%*[']%[^']";
	char *scanstr2 = "%*[ ]%*[']%[^']%*[ ']";

	/*
	 *  scan in & store everything up to & including variable
	 */
        status = ksscanf(line, stringlist_scan, &flag, 
			&line_info->activate, &line_info->selected, 
			&line_info->optional, &line_info->opt_sel,
			&line_info->live,
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			&line_info->list_num, &line_info->list_val, 
		        tmp_title, tmp_desc, tmp_var);
	
	if (status < 15)
        {
            mesg = create_readerror_mesg(line, status, 15);
	    errno = KUIS_SYNTAX;
            kerror("kforms", "kvf_parse_stringlist_line", mesg);
	    kfree(mesg);
            return(FALSE);
        }

        /*
         *  move thru line past variable
         */
        indx = 0;
        if (kstrlen(tmp_title) != 0)
        {
            while ((kstrncmp(&line[indx], tmp_title, kstrlen(tmp_title))) != 0)
                indx++;
            indx+= kstrlen(tmp_title);
        }
        if (kstrlen(tmp_desc) != 0)
        {
            while ((kstrncmp(&line[indx], tmp_desc, kstrlen(tmp_desc))) != 0)
                indx++;
            indx+= kstrlen(tmp_desc);
        }
        if (kstrlen(tmp_var) != 0)
        {
            while ((kstrncmp(&line[indx], tmp_var, kstrlen(tmp_var))) != 0)
                indx++;
            indx+= kstrlen(tmp_var);
        }


	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_string_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->int_val = 1;

	line_info->typeflag = KUIS_STRINGLIST;

	mask = KUIS_ACT | KUIS_SEL | KUIS_OPT | KUIS_LIVE | KUIS_LISTVAL;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_stringlist_line")))
	    return(FALSE);

	/*
	 * read in strings to appear in list
	 */
	kvf_realloc_val_labelnum(line_info, line_info->list_num);

	scanstr = scanstr1;
        for (j = 0; j < line_info->list_num; j++)
        {
            status = ksscanf(&line[indx], scanstr, tmp_label);
            if (status != 1)
            {
                mesg = create_mesg(line,
                       "Unable to read correct number of stringlist values");
		errno = KUIS_SYNTAX;
                kerror("kforms", "kvf_parse_stringlist_line", mesg);
		kfree(mesg);
                return(FALSE);
            }
            indx = indx + kstrlen(tmp_label) + 3;

            kfree(line_info->val_labels[j]);
	    unescape_ticmark(tmp_label);
            line_info->val_labels[j] = kstrdup(tmp_label);

            if (j == 0)
		scanstr = scanstr2;
        }

	if ((line_info->list_num > 0) && (line_info->list_val > 0))
	{
	    kfree(line_info->string_def);
	    if ((kstrcmp(line_info->val_labels[line_info->list_val-1], " "))
		         == 0)
	        line_info->string_def = NULL;
	    else line_info->string_def = 
		kstrdup(line_info->val_labels[line_info->list_val-1]);
	}

	/*
	 *  finally, pull in string value & literal if they are there
	 */
	status = ksscanf(&line[indx], 
			"%*[ ]%*[']%[^']%*[']%*[ ]%*[']%[^']%*[']",
		        tmp_string, tmp_literal);

	kfree (line_info->string_val);
	kfree (line_info->literal);

	/* assign both string val & literal */
	if (status == 2)
        {
	    unescape_ticmark(tmp_string);
            if ((kstrcmp(tmp_string," ")) == 0) line_info->string_val = NULL;
            else line_info->string_val = kstrdup(tmp_string);

	    unescape_ticmark(tmp_literal);
            if ((kstrcmp(tmp_literal," ")) == 0) line_info->literal = NULL;
            else line_info->literal = kstrdup(tmp_literal);
	}
	
	/* assign string val, take same value for literal */
	else if (status == 1)
	{
	    if ((kstrcmp(tmp_string," ")) == 0) 
	    {
	 	line_info->string_val = NULL;
	 	line_info->literal = NULL;
	    }
            else 
	    {
	        unescape_ticmark(tmp_string);
		line_info->string_val = kstrdup(tmp_string);
		line_info->literal = kstrdup(tmp_string);
	    }
	}

	/* get string val & literal from default */
	else if (status < 1)
	{
	    if (line_info->string_def == NULL)
	    {
		line_info->string_val = NULL;
                line_info->literal = NULL;
	    }
	    else
	    {
		line_info->string_val=kstrdup(line_info->string_def);
                line_info->literal = kstrdup(line_info->string_def);
	    }
	}

	return(TRUE);
}




/*------------------------------------------------------------
|
|       Routine: kvf_parse_routine_line
|       Purpose: Parses a Routine line  (-R)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_routine_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_desc[KLENGTH], tmp_title[KLENGTH], 
	     tmp_routine[KLENGTH], flag;
	int status;
	char *mesg;
	unsigned long mask;

        status = ksscanf(line, routine_scan, &flag, 
			&line_info->activate, &line_info->selected,
			&line_info->exec_type,
		        &line_info->width, &line_info->height, 
		  	&line_info->x, &line_info->y, 
			tmp_title, tmp_desc, tmp_routine);

	if (status < 11) 
	{
	    mesg = create_readerror_mesg(line, status, 11);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_routine_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kfree (line_info->routine);
	if ((kstrcmp(tmp_routine," ")) == 0) line_info->routine = NULL;
	else line_info->routine = kstrdup(tmp_routine);

	line_info->typeflag = KUIS_ROUTINE;

	mask = KUIS_ACT | KUIS_SEL | KUIS_EXECTYPE;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_routine_line")))
	    return(FALSE);

	return(TRUE);
}




/*------------------------------------------------------------
|
|       Routine: kvf_parse_toggle_line
|       Purpose: Parses a KUIS_TOGGLE line  (-T)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992    
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_toggle_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_desc[KLENGTH], tmp_title[KLENGTH]; 
	char tmp_var[KLENGTH];
	char flag;
	int  status;
	char *mesg;
	unsigned long mask;

        status = ksscanf(line, toggle_scan, &flag, 
			&line_info->activate, &line_info->selected, 
			&line_info->optional,  &line_info->opt_sel,
			&line_info->live,
		 	&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			&line_info->xpos, &line_info->ypos, 
			&line_info->toggle_def, tmp_title, tmp_desc, 
			tmp_var, &line_info->toggle_val);
	
	if (status < 16) 
	{
	    mesg = create_readerror_mesg(line, status, 16);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_toggle_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_toggle_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	if (line_info->toggle_def < 1)
	{
	    mesg = create_mesg(line, "toggle default must be > 0");
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_toggle_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}

	line_info->typeflag = KUIS_TOGGLE;

	mask = KUIS_ACT | KUIS_SEL | KUIS_OPT | KUIS_LIVE;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_toggle_line")))
	    return(FALSE);

	return(TRUE);
}




/*------------------------------------------------------------
|
|       Routine: kvf_parse_blank_line
|       Purpose: Parses a Blank line  (-b)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992    
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_blank_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_title[KLENGTH], tmp_var[KLENGTH], flag;
	int status;
	char *mesg;

        status = ksscanf(line, blank_scan, &flag, 
			&line_info->xpos, &line_info->ypos, 
			tmp_title, tmp_var); 

	if (status < 5) 
	{
	    mesg = create_readerror_mesg(line, status, 5);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_blank_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	kfree (line_info->variable);
	if ((kstrcmp(tmp_var," ")) == 0) line_info->variable = NULL;
	else line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_BLANK;

	line_info->optional = TRUE;
	return(TRUE);
}




/*------------------------------------------------------------
|
|       Routine: kvf_parse_pane_action_line
|       Purpose: Parses a Pane Action line  (-a)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992    
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_pane_action_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_desc[KLENGTH], tmp_title[KLENGTH], tmp_var[KLENGTH], 
	     flag;
	int status;
	char *mesg;
	unsigned long mask;

        status = ksscanf(line, pane_action_scan, &flag, 
			&line_info->activate, &line_info->selected, 
			&line_info->width, &line_info->height, 
			&line_info->x, &line_info->y, 
			tmp_title, tmp_desc, tmp_var);
	
	if (status < 10) 
	{
	    mesg = create_readerror_mesg(line, status, 10);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_pane_action_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}

	unescape_ticmark(tmp_title);
	kfree (line_info->title);
	if ((kstrcmp(tmp_title," ")) == 0) line_info->title = NULL;
	else line_info->title = kstrdup(tmp_title);

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_pane_action_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_PANEACTION;

	mask = KUIS_ACT | KUIS_SEL;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_pane_action_line")))
	    return(FALSE);

	return(TRUE);
}



/*------------------------------------------------------------
|
|       Routine: kvf_parse_includepane_line
|       Purpose: Parses an IncludePane line  (-p)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992    
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_includepane_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_file[KLENGTH], flag, *mesg;
	int status;

        status = ksscanf(line, includepane_scan, &flag, tmp_file);

	if (status < 2) 
	{
	    mesg = create_readerror_mesg(line, status, 2);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_includepane_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}
	
	kfree (line_info->filename);
	if ((kstrcmp(tmp_file," ")) == 0) 
	{
	    line_info->filename = NULL;
	    mesg = create_mesg(line, "-p MUST be followed by filename");
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_includepane_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}
	else 
	{ 
	    line_info->filename = kstrdup(tmp_file);
	}

	line_info->typeflag = KUIS_INCLUDEPANE;

	return(TRUE);
}



/*------------------------------------------------------------
|
|       Routine: kvf_parse_includesubform_line
|       Purpose: Parses an IncludeSubform line  (-k)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992    
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_includesubform_line(
   char      *line,
   Line_Info *line_info)
{
	char tmp_file[KLENGTH], flag, *mesg;
	int status;

        status = ksscanf(line, includesubform_scan, &flag, tmp_file);
	if (status < 2) 
	{
	    mesg = create_readerror_mesg(line, status, 2);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_includesubform_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}
	
	kfree (line_info->filename);
	if ((kstrcmp(tmp_file," ")) == 0) 
	{
	    line_info->filename = NULL;
	    mesg = create_mesg(line, "-k MUST be followed by filename");
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_includesubform_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}
	else 
	{ 
	    line_info->filename = kstrdup(tmp_file);
	}

	line_info->typeflag = KUIS_INCLUDESUBFORM;

	return(TRUE);
}


/*------------------------------------------------------------
|
|       Routine: kvf_parse_mutexcl_line
|       Purpose: Parses a Mutually Exclusive line  (-C 0, -C 1, or -C 2)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992    
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
------------------------------------------------------------*/

int kvf_parse_mutexcl_line(
   char      *line,
   Line_Info *line_info)
{
	char flag, *mesg;
	int status;
	unsigned long mask;

        status = ksscanf(line, "-%c %d", &flag, &line_info->logical_val);
	if (flag != 'C') return(FALSE);
	if (status < 2) 
	{
	    mesg = create_readerror_mesg(line, status, 2);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_mutexcl_line", mesg); 
	    kfree(mesg);
	    return(FALSE);
	}

	line_info->typeflag = KUIS_MUTEXCL;

	mask = KUIS_LOGICALVAL;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_mutexcl_line")))
	    return(FALSE);
	
	return(TRUE);
}


/*------------------------------------------------------------
|
|       Routine: kvf_parse_mutincl_line
|       Purpose: Parses a Mutually Inclusive line  (-B:  typeflag = KUIS_MUTINCL)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jun 06, 1992
|    Written By: Danielle Argiro 
| Modifications: 
|
------------------------------------------------------------*/

int kvf_parse_mutincl_line(
   char      *line,
   Line_Info *line_info)
{
	char flag, *mesg;
	int status;
	unsigned long mask;

        status = ksscanf(line, "-%c %d", &flag, &line_info->logical_val);
	if (flag != 'B') return(FALSE);
	if (status < 2) 
	{
	    mesg = create_readerror_mesg(line, status, 2);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_mutincl_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}

	line_info->typeflag = KUIS_MUTINCL;

	mask = KUIS_LOGICALVAL;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_mutincl_line")))
	    return(FALSE);

	return(TRUE);
}

/*------------------------------------------------------------
|
|       Routine: kvf_parse_group_line
|       Purpose: Parses a loose Group line  (-K 0)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Oct 6, 1993    
|    Written By: Danielle Argiro 
| Modifications: 
------------------------------------------------------------*/

int kvf_parse_group_line(
   char      *line,
   Line_Info *line_info)
{
	char flag, *mesg;
	int status;
	unsigned long mask;

        status = ksscanf(line, "-%c %d", &flag, &line_info->logical_val);
	if (flag != 'K') return(FALSE);
	if (status < 2) 
	{
	    mesg = create_readerror_mesg(line, status, 2);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_group_line", mesg); 
	    kfree(mesg);
	    return(FALSE);
	}

	line_info->typeflag = KUIS_GROUP;

	mask = KUIS_LOGICALVAL;
	if (!(check_UIS_line(line, line_info, mask,
				 "kvf_parse_group_line")))
	    return(FALSE);
	
	return(TRUE);
}

/*------------------------------------------------------------
|
|       Routine: kvf_parse_workspace_line
|       Purpose: Parses a Workspace line  (-w)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992   
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/


int kvf_parse_workspace_line(
   char      *line,
   Line_Info *line_info)
{
	char flag, *mesg;
	int  status;
	char tmp_desc[KLENGTH], tmp_var[KLENGTH]; 
	long tmp_wksp_address;

        status = ksscanf(line, workspace_scan, &flag,
			&line_info->width, &line_info->height,
                        &line_info->x, &line_info->y,
			tmp_desc, tmp_var, &tmp_wksp_address);
	if (flag != 'w') return(FALSE);

        if (status == 8)
	{
	    line_info->workspace = (xvobject) tmp_wksp_address;
	}
	else if (status < 7) 
	{
	    mesg = create_readerror_mesg(line, status, 7);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_workspace_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}

	unescape_ticmark(tmp_desc);
	kfree (line_info->description);
	if ((kstrcmp(tmp_desc," ")) == 0) line_info->description = NULL;
	else line_info->description = kstrdup(tmp_desc);

	kfree (line_info->variable);
	if (!(check_line_variable(line, "kvf_parse_workspace_line", 
		                      tmp_var))) return(FALSE);
        line_info->variable = kstrdup(tmp_var);

	line_info->typeflag = KUIS_WORKSPACE;

	return(TRUE);
}

/*------------------------------------------------------------
|
|       Routine: kvf_parse_end_line
|       Purpose: Parses an End line  (-E)
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992   
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_parse_end_line(
   char      *line,
   Line_Info *line_info)
{
	char flag, *mesg;
	int status;

        status = ksscanf(line, "-%c", &flag);
	if (status < 1) 
	{
	    mesg = create_readerror_mesg(line, status, 1);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_parse_end_line", mesg);
	    kfree(mesg);
	    return(FALSE);
	}

	if (flag != 'E') return(FALSE);
	
	line_info->typeflag = KUIS_END;

	return(TRUE);
}





/*------------------------------------------------------------
|
|   Routine Name: kvf_gen_parse
|        Purpose: Parses an undetermined type of UIS line
|         Input: line      - the UIS line
|                line_info - must be cleared with kvf_clear_line_info
|        Output: line_info - with fields filled out accordingly
|       Returns: TRUE on success, FALSE on failure
|          Date: Jul 14, 1992   
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int kvf_gen_parse(
   char      *line,
   Line_Info *line_info)
{
 	int	typeflag, status;

	typeflag = kvf_get_line_type(line);

	switch (typeflag) {
		case KUIS_STARTFORM:
			status = kvf_parse_startform_line(line, line_info);
			break;

		case KUIS_STARTMASTER:
			status = kvf_parse_startmaster_line(line, line_info);
			break;

		case KUIS_STARTSUBMENU:
			status = kvf_parse_startsubmenu_line(line, line_info);
			break;

		case KUIS_STARTSUBFORM:
			status = kvf_parse_startsubform_line(line, line_info);
			break;

		case KUIS_STARTPANE:
			status = kvf_parse_startpane_line(line, line_info);
			break;

		case KUIS_STARTGUIDE:
			status = kvf_parse_startguide_line(line, line_info);
			break;

		case KUIS_GUIDEBUTTON:
			status = kvf_parse_guide_line(line, line_info);
			break;

		case KUIS_MASTERACTION:
			status = kvf_parse_master_action_line(line, line_info);
			break;

		case KUIS_SUBFORMBUTTON:
			status = kvf_parse_subformbutton_line(line, line_info);
			break;
	
		case KUIS_QUIT:
			status = kvf_parse_quit_line(line, line_info);
			break;
	 
		case KUIS_SUBFORMACTION:
			status = kvf_parse_subform_action_line(line, line_info);
			break;
	
		case KUIS_INPUTFILE:
			status = kvf_parse_input_line(line, line_info);
			break;

		case KUIS_OUTPUTFILE:
			status = kvf_parse_output_line(line, line_info);
			break;

		case KUIS_STDIN:
			status = kvf_parse_stdin_line(line, line_info);
			break;

		case KUIS_STDOUT:
			status = kvf_parse_stdout_line(line, line_info);
			break;

		case KUIS_INTEGER:
			status = kvf_parse_int_line(line, line_info);
			break;

		case KUIS_FLOAT:
			status = kvf_parse_float_line(line, line_info);
			break;

		case KUIS_DOUBLE:
			status = kvf_parse_double_line(line, line_info);
			break;

		case KUIS_STRING:
			status = kvf_parse_string_line(line, line_info);
			break;

		case KUIS_STRINGLIST:
			status = kvf_parse_stringlist_line(line, line_info);
			break;

		case KUIS_LOGICAL:
			status = kvf_parse_logic_line(line, line_info);
			break;

		case KUIS_TOGGLE:
			status = kvf_parse_toggle_line(line, line_info);
			break;

		case KUIS_ROUTINE:
			status = kvf_parse_routine_line(line, line_info);
			break;

		case KUIS_HELP:
			status = kvf_parse_help_line(line, line_info);
			break;

		case KUIS_BLANK:
			status = kvf_parse_blank_line(line, line_info);
			break;

		case KUIS_PANEACTION:
			status = kvf_parse_pane_action_line(line, line_info);
			break;

		case KUIS_INCLUDEPANE:
			status = kvf_parse_includepane_line(line, line_info);
			break;

		case KUIS_INCLUDESUBFORM:
			status = kvf_parse_includesubform_line(line, line_info);
			break;

		case KUIS_WORKSPACE:
			status = kvf_parse_workspace_line(line, line_info);
			break;

		case KUIS_MUTEXCL:
			status = kvf_parse_mutexcl_line(line, line_info);
			break;

		case KUIS_MUTINCL:
			status = kvf_parse_mutincl_line(line, line_info);
			break;

		case KUIS_GROUP:
			status = kvf_parse_group_line(line, line_info);
			break;

		case KUIS_CYCLE:
			status = kvf_parse_cycle_line(line, line_info);
			break;

		case KUIS_LIST:
			status = kvf_parse_list_line(line, line_info);
			break;

		case KUIS_DISPLAYLIST:
			status = kvf_parse_displaylist_line(line, line_info);
			break;

		case KUIS_FLAG:
			status = kvf_parse_flag_line(line, line_info);
			break;

		case KUIS_END:
			status = kvf_parse_end_line(line, line_info);
			break;

                default:
	                line_info->typeflag = -1;
			status = FALSE;
			break;

		}  /* end switch */

        return(status);
}
	

/*------------------------------------------------------------
|
|  Routine Name: create_readerror_mesg 
|
|       Purpose: Creates an appropriate error message for when
|	         a UIS line cannot be fully read.
|         Input: line     - the UIS line
|                status   - number of fields read
|                expected - number of fields expected
|        Output: none
|       Returns: Returns an error message to send to kerror
|          Date: Jul 14, 1992   
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

static char *create_readerror_mesg(
   char *line,
   int  status,
   int  expected)
{
	char *mesg_return; 
	int  length; 
	char errline[KLENGTH], mesg[KLENGTH];

	ksprintf(errline, "     Error in line [%s] - ", line);
	ksprintf(mesg, "Only able to read %d out of %d required fields.",
                            status, expected);
	length = kstrlen(errline) + kstrlen(mesg) + 5;

	if ((mesg_return = (char *) kcalloc (1, (unsigned) length)) == NULL)
	{
	    kerror("kforms", "create_readerror_mesg",
		   "Unable to allocate internal message string");
	    return(NULL);
	}
	ksprintf(mesg_return,  "\n%s%s", errline, mesg);
	return (mesg_return);
}

/*------------------------------------------------------------
|
|  Routine Name: create_mesg 
|
|       Purpose: Creates an error message for when
|	         a UIS line has syntax errors.
|         Input: line     - the UIS line
|                mesg     - specific message for error 
|        Output: none
|       Returns: the error message
|          Date: Jul 14, 1992   
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA) 
|
------------------------------------------------------------*/

static char *create_mesg(
   char *line,
   char *mesg)
{
	char *mesg_return;
        char errorline[KLENGTH];
        int  length;

	ksprintf(errorline, "Error: in line [%s]", line);
	
	length = kstrlen(errorline) + kstrlen(mesg) + 5;
	if ((mesg_return = (char *) kcalloc (1,(unsigned) length)) == NULL)
	{
	    kerror("kforms", "create_mesg",
		   "Unable to allocate internal message string");
	    return(NULL);
	}
	ksprintf(mesg_return, "\n%s\n%s\n", errorline, mesg);

	kfree(mesg);
	return(mesg_return);
	
}

/*------------------------------------------------------------
|
|   Routine Name: check_UIS_line 
|
|       Purpose:  Takes a UIS line and checks fields specified by mask.
|
|         Input: line      - the UIS line to be checked
|                line_info - pointer to the line_info structure
|                mask      - indicates which fields to check
|                caller    - name of calling routine
|        Output: none
|       Returns: TRUE if UIS line (represented by Line_Info struct)
|                is ok, FALSE if it has a syntax error.
|
|          Date: Jun 25, 1992
|    Written By: Danielle Argiro
| Modifications: 
|
------------------------------------------------------------*/
static int check_UIS_line(
   char          *line,
   Line_Info     *line_info,
   unsigned long mask,
   char          *caller)
{
	char temp[KLENGTH];
	char *mesg;
	char *fullpath;

	if (mask & KUIS_ACT)
	{
	   if (!(kvf_check_boolean(line_info->activate)))
	   {
	        mesg = create_mesg(line, 
			              "'activate' field must be 0 or 1");
	        errno = KUIS_SYNTAX;
	        kerror("kforms", caller, mesg); 
	 	kfree(mesg);
	        return(FALSE);
	   }
	}

	if (mask & KUIS_SEL)
	{
	    if (!(kvf_check_boolean(line_info->selected)))
	    {
	        mesg = create_mesg(line, 
                                     "'selected' field must be 0 or 1");
	        errno = KUIS_SYNTAX;
	        kerror("kforms", caller, mesg); 
		kfree(mesg);
	        return(FALSE);
	    }
	}

	if (mask & KUIS_OPT)
	{
	    if (!(kvf_check_boolean(line_info->optional)))
	    {
	        mesg = create_mesg(line, 
                                     "'optional' field must be 0 or 1");
	        errno = KUIS_SYNTAX;
	        kerror("kforms", caller, mesg); 
		kfree(mesg);
	        return(FALSE);
	    }

	    if (!(kvf_check_opt_sel(line_info->opt_sel)))
	    {
	        mesg = create_mesg(line, 
                                      "opt_sel must be 0, 1, or 2");
	        errno = KUIS_SYNTAX;
	        kerror("kforms", caller, mesg); 
		kfree(mesg);
	        return(FALSE);
	    }
	
	    if ((line_info->opt_sel == 2) && (line_info->typeflag == KUIS_FLAG))
	    {
	        mesg = create_mesg(line, "opt_sel must be 0 or 1 for a Flag Selection -- suppressing the optional box would imply there was no way to run the program WITHOUT the flag");
	        errno = KUIS_SYNTAX;
	        kerror("kforms", caller, mesg); 
		kfree(mesg);
	        return(FALSE);
	    }

            if ((line_info->optional == 0 ) && (line_info->opt_sel == 0))
	    {
	        mesg = create_mesg(line, "if the 'optional' field = 0, then  the 'opt_sel' must be 1 or 2");
	        errno = KUIS_SYNTAX;
	        kerror("kforms", caller, mesg); 
		kfree(mesg);
	        return(FALSE);
	    }
	}

	if (mask & KUIS_LIVE)
	{
	    if (!(kvf_check_boolean(line_info->live)))
	    {
	        mesg = create_mesg(line, "'live' field must be 0 or 1");
	        errno = KUIS_SYNTAX;
	        kerror("kforms", caller, mesg); 
		kfree(mesg);
	        return(FALSE);
	    }
	}

	if (mask & KUIS_FILECHECK)
	{
            if (!(kvf_check_boolean(line_info->file_check)))
	    {
	        mesg = create_mesg(line, "'file_check' field must be 0 or 1");
	        errno = KUIS_SYNTAX;
	        kerror("kforms", caller, mesg); 
		kfree(mesg);
	        return(FALSE);
	    }
	}

	if (mask & KUIS_POS)
	{
 	    if (!(kvf_check_pos(line_info->xpos, line_info->width)))
	    {
	        mesg = create_mesg(line, "X position > than width");
	        errno = KUIS_SYNTAX;
		kerror("kforms", caller, mesg); 
		kfree(mesg);
                return(FALSE);
            }

 	    if (!(kvf_check_pos(line_info->ypos, line_info->height)))
	    {
	        mesg = create_mesg(line, "Y position > height");
	        errno = KUIS_SYNTAX;
		kerror("kforms", caller, mesg); 
		kfree(mesg);
                return(FALSE);

            }
        }

 	if (mask & KUIS_LOGICALVAL)
	{
	    if (!(kvf_check_boolean(line_info->logical_val)))
	    {
	        mesg = create_mesg(line,
			       	    "'mutually exclusive' flag must be 0 or 1");
	        errno = KUIS_SYNTAX;
	        kerror("kforms", caller, mesg); 
		kfree(mesg);
	        return(FALSE);
	    }
        }
	if (mask & KUIS_LOGICALDEF)
	{
	    if (!(kvf_check_boolean(line_info->logical_def)))
	    {
	        mesg = create_mesg(line, 
				"Logical default value must be 1 or 0");
	        errno = KUIS_SYNTAX;
	        kerror("kforms", caller, mesg); 
		kfree(mesg);
	        return(FALSE);
	    }
	}
	if (mask & KUIS_CYCLEVAL)
	{
	    if ((line_info->list_val < line_info->int_val)  &&
	        (line_info->list_val > 0))
	    {
		ksprintf(temp, "cycle value must be >= cycle start (%d)",
			line_info->int_val);
	        mesg = create_mesg(line, temp);
	        errno = KUIS_SYNTAX;
	        kerror("kforms", caller, mesg); 
		kfree(mesg);
	        return(FALSE);
	    }
            if (line_info->list_num < 0)
            {
                ksprintf(temp, "cycle number must be >= 0");
                mesg = create_mesg(line, temp);
	        errno = KUIS_SYNTAX;
                kerror("kforms", caller, mesg);
                kfree(mesg);
                return(FALSE);
            }
	    if ((line_info->list_val > line_info->int_val + line_info->list_num -1) && (line_info->list_num > 0))
	    {
	        ksprintf(temp, "cycle value must be <= %d", 
			line_info->int_val + line_info->list_num - 1);
                mesg = create_mesg(line, temp);
	        errno = KUIS_SYNTAX;
                kerror("kforms", caller, mesg);
                kfree(mesg);
                return(FALSE);
            }

	}

	if (mask & KUIS_LISTVAL)
	{
	    if ((line_info->typeflag != KUIS_DISPLAYLIST) &&
		(line_info->list_val < line_info->int_val) &&
		(line_info->list_num > 0))
            {
		ksprintf(temp, "list value must be >= list start (%d)",
			line_info->int_val);
	        mesg = create_mesg(line, temp);
	        errno = KUIS_SYNTAX;
                kerror("kforms", caller, mesg);
                kfree(mesg);
                return(FALSE);
            }
	    else if ((line_info->typeflag == KUIS_DISPLAYLIST) &&
                     (line_info->list_val < line_info->int_val-1) &&
		     (line_info->list_num > 0))
            {
		ksprintf(temp, "list value must be >= list start (%d), or set to %d to indicate that NO selection from the list is wanted",
			line_info->int_val, line_info->int_val-1);
	        mesg = create_mesg(line, temp);
	        errno = KUIS_SYNTAX;
                kerror("kforms", caller, mesg);
                kfree(mesg);
                return(FALSE);
            }

	    if (line_info->list_num < 0)
	    {
		ksprintf(temp, "list number must be >= 0");
	        mesg = create_mesg(line, temp);
	        errno = KUIS_SYNTAX;
                kerror("kforms", caller, mesg);
                kfree(mesg);
                return(FALSE);
	    }
            if ((line_info->list_val > line_info->int_val + line_info->list_num - 1) && (line_info->list_num > 0))
            {
	        ksprintf(temp, "list value must be <= %d", 
			line_info->int_val + line_info->list_num - 1);
                mesg = create_mesg(line, temp);
	        errno = KUIS_SYNTAX;
                kerror("kforms", caller, mesg);
                kfree(mesg);
                return(FALSE);
            }

	}

	if (mask & KUIS_EXECTYPE)
	{
	    if ((line_info->exec_type < 0) || (line_info->exec_type > 4))
	    {
	        mesg = create_mesg(line, 
			"'exec_type' field must be between 0 and 4");
	        errno = KUIS_SYNTAX;
	        kerror("kforms", caller, mesg); 
		kfree(mesg);
	        return(FALSE);
	    }
	}

	if (mask & KUIS_INFILEDEF)
	{
	    if (line_info->file_def != NULL)
	    {
	        fullpath = kfullpath(line_info->file_def, NULL, NULL);
		if (fullpath == NULL)
		{
		    mesg = create_mesg(line, 
			   "'file_def' field must be a valid input file");
	            errno = KUIS_SYNTAX;
		    kerror("kforms", caller, mesg);
		    kfree(mesg);
		    free(fullpath);
                    return(FALSE);
		}
		free(fullpath);
	    }
	}

	if (mask & KUIS_FLG)
	{
	    if (!line_info->optional) 
	    {
	        ksprintf(temp, "'optional' must be set to 1 (TRUE)");
                mesg = create_mesg(line, temp);
	        errno = KUIS_SYNTAX;
                kerror("kforms", caller, mesg);
                kfree(mesg);
                return(FALSE);
	    }
	}

	if (mask & KUIS_MECH)
        {
            if ((line_info->special < 0) ||
		(line_info->special > 1))
            {
                ksprintf(temp, "'mechanism' field must be set to 0 (no movement mechanism) or 1 (scrollbar). In the future, 2 (dial) will also be supported.");
                mesg = create_mesg(line, temp);
	        errno = KUIS_SYNTAX;
                kerror("kforms", caller, mesg);
                kfree(mesg);
                return(FALSE);
            }
        }

	if (mask & KUIS_PREC)
        {
            if ((line_info->precision < -1) ||
		(line_info->precision > 14))
            {
                ksprintf(temp, "'precision' field must be -1, 0, or less than 14.");
                mesg = create_mesg(line, temp);
	        errno = KUIS_SYNTAX;
                kerror("kforms", caller, mesg);
                kfree(mesg);
                return(FALSE);
            }
        }
	return(TRUE);
}


/*------------------------------------------------------------
|
|  Routine Name: kvf_clear_line_info 
|       Purpose: Initializes all fields in a Line_Info structure.
|         Input: line_info - pointer to the line_info structure
|        Output: none
|       Returns: nothing
|          Date: Jul 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

void kvf_clear_line_info(
   Line_Info *line_info)
{
	/*
	 *  clear all the line_info structure except that we need
	 *  to set the version to 4.2
	 */
	kmemset((char *) line_info, 0, sizeof(Line_Info));
	line_info->version     = 4.2;

}  /* end kvf_clear_line_info */

/*------------------------------------------------------------
|
|  Routine Name: check_line_variable
|       Purpose: Makes sure that variable name is provided on UIS line
|         Input: line     - UIS line (for error message if necessary)
|                routien  - name of calling routine (for error message)
|                variable - buffer containing variable name
|        Output: 
|       Returns: TRUE if variable is ok, FALSE if it is blank
|          Date: Jul 13, 1992
|    Written By: Danielle Argiro
| Modifications: 
|
------------------------------------------------------------*/

static int check_line_variable(
   char *line,
   char *routine,
   char *variable)
{
	char *mesg;

	if ((kstrcmp(variable," ")) == 0) 
        {
	    mesg = create_mesg(line, "'variable' field MUST be specified");
	    errno = KUIS_SYNTAX;
	    kerror("kforms", routine, mesg); 
	    kfree(mesg);
	    return(FALSE);
        }
	return(TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: kvf_realloc_val_labelnum
|       Purpose: Makes sure that the line_info->val_labels are allocated
|                according to the number needed.
|         Input: line_info - pointer to Line_Info structure
|                label_num - number of labels needed
|       Returns: TRUE if memory can be allocated, FALSE if it can't
|          Date: Nov 18, 1992
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
int kvf_realloc_val_labelnum(
   Line_Info *line_info,
   int       label_num)
{
	int i;

	if (line_info->val_labelnum == label_num) 
	    return(TRUE);

	for (i = 0; i < line_info->val_labelnum; i++)
	    kfree(line_info->val_labels[i]);
	kfree(line_info->val_labels);

	line_info->val_labelnum = label_num;

	if (label_num > 0)
	{
	    line_info->val_labels = (char **) kcalloc((unsigned) label_num, 
					      (unsigned) sizeof(char *));
	    return(TRUE);
	}
	else if (line_info->val_labels == NULL)
	    return(FALSE);

	return(TRUE);
}

/*-------------------------------------------------------------
|
|       Routine: unescape_ticmark
|       Purpose: In strings, if there are any escaped tic marks (\'), 
|	         these need to be "unescaped" (changed to ') before 
|		 they are put into the line_info structure, so that
|		 they will not appear in the GUI in escaped form. 
|         Input: string - Already-allocated string buffer
|        Output: string - string with \' changed to ' (if applicable)
|       Returns:
|          Date: Feb 24, 1994
|    Written by: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

/* ARGSUSED */
static void unescape_ticmark(
   char *string)
{
        /*
         *  if there is a \' in the string, replace it with '
         */
        if (kstrchr(string, '\'') != NULL)
            (void) kstring_replace(string, "\\\'", "\'", string);
}

