/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>                                                       <<<<
   >>>>		       Grouping Routines                      <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                kvf_call_do_group()                    <<<<
   >>>>                kvf_do_group()                         <<<<
   >>>>                kvf_set_group_selection()              <<<<
   >>>>   Static:                                             <<<<
   >>>>                do_me_group()                          <<<<
   >>>>                do_mi_group()                          <<<<
   >>>>                do_loose_group()                       <<<<
   >>>>                direct_group_sibling()                 <<<<
   >>>>                indirect_group_sibling()               <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"


/* 
 * possible values of uplevel_status
 */
#define NO_UPPER 0  /* no upper group                                        */
#define ME_ON    1  /* upper level ME group ON  => current group must be OFF */
#define ME_OFF   2  /* upper level ME group OFF => current group must be ON  */
#define MI_ON    3  /* upper level MI group ON  => current group must be ON  */
#define MI_OFF   4  /* upper level MI group OFF => current group must be OFF */
#define LOOSE    5  /* upper level loose group                               */

/*
 * declarations of static routines
 */
static int  
direct_group_sibling   PROTO((kselection *, kselection *, kselection *));

static int 
indirect_group_sibling PROTO((kselection *, kselection *, kselection *));

static int 
group_member    PROTO ((kselection *, kselection *));

static int 
do_me_group     PROTO((kselection *, kselection *, int, int, int, kfunc_void));

static int  
do_mi_group     PROTO((kselection *, kselection *, int, int, int, kfunc_void));

static int  
do_loose_group  PROTO((kselection *, kselection *, int, int, int, kfunc_void));



static int selected_item_num PROTO((kselection *));

/*-----------------------------------------------------------
|
|  Routine Name: kvf_call_do_group
|
|       Purpose: Calls the kvf_do_group() routine with the correct
|                value for the 'uplevel_status' parameter based on
|                the type of group that the selection belongs to.
|
|         Input: selection - the group member
|                value     - TRUE for turning group selection on, FALSE
|                            for turning group selection OFF
|        Output: none
|          Date: Jun 09, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/

void kvf_call_do_group(
    kselection *selection,
    int        value)
{
        kselection *start_group;

        start_group = selection->back_group;
        while (start_group->back_group != NULL)
            start_group = start_group->back_group;

        if (start_group->type == KUIS_MUTEXCL)
            (void) kvf_do_group(start_group, selection, value,
                                0, start_group->selected, 
				kvf_set_group_selection);
        else (void) kvf_do_group(start_group, selection, value,
		                0, FALSE, kvf_set_group_selection);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_do_group
|
|       Purpose: This is the subroutine for each of the buttons that make
|                up a mutually exclusive or mutually inclusive group.
|
|         Input: start_group - the beginning of the group (-C or -B selection)
|                selection   - member of the ME/MI group just clicked on
|                uplevel_status - status of UPPER LEVEL group, one of: 
|                                 NO_UPPER, ME_ON, ME_OFF, MI_ON, MI_OFF
|                required    - TRUE if group as a whole is required,
|                              FALSE otherwise
|                set_group_selection - routine to set the group selection
|        Output: none
|          Date: Jun 09, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/

int kvf_do_group(
   kselection *start_group,
   kselection *selection,
   int        value,
   int        uplevel_status,
   int        required,
   kfunc_void set_group_selection)
{
	int status;

	/*
         *  we've got a mutually exclusive group
	 */
	if (start_group->type == KUIS_MUTEXCL)
	    status = do_me_group(start_group, selection, value, 
				uplevel_status, required, set_group_selection);

	else if (start_group->type == KUIS_MUTINCL)
	    status = do_mi_group(start_group, selection, value, 
				uplevel_status, required, set_group_selection);

	else if (start_group->type == KUIS_GROUP)
	    status = do_loose_group(start_group, selection, value, 
				 uplevel_status, required, set_group_selection);
	else
	{
	    kerror(NULL, "kvf_do_group",
		   "Bogus group type sent into kvf_do_group");
	    status = FALSE;
	}
	
	return(status);
}


/*-----------------------------------------------------------
|
|  Routine Name: do_me_group
|
|       Purpose: This is the subroutine for each of the 
|                buttons that make up a mutually exclusive group.
|
|         Input: start_group - the beginning of the ME group 
|                selection   - member of the ME/MI group just clicked on
|                uplevel_status - status of UPPER LEVEL group, one of:
|                                 NO_UPPER, ME_ON, ME_OFF, MI_ON, MI_OFF
|        Output: none
|          Date: Jun 09, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/

static int do_me_group(
   kselection *start_group,
   kselection *selection,
   int        value,
   int        uplevel_status,
   int        required,
   kfunc_void set_group_selection)
{
        int        substatus;
        kselection *mut_excl;
        int        current_status; /* one of ME_ON, ME_OFF */

	/*
	 * traverse list of ME group siblings, 
	 * deal with selected member in first pass
	 */
	current_status = ME_OFF;
	mut_excl = start_group->group_next;
 	while (mut_excl != NULL) 
	{
	    /*
	     *  an MI group within the ME group, recurse
	     */
	    if (mut_excl->type == KUIS_MUTINCL)
	    {
                substatus = do_mi_group(mut_excl, selection,  value,
					current_status, required,
					set_group_selection);

		/* if lower level group was turned on, then this
		   group is now considered on */
		if (substatus == MI_ON) current_status = ME_ON;
	    }

	    else if (mut_excl->type == KUIS_BLANK)
	    {
		    mut_excl = mut_excl->next;
		    continue;
	    }

	    /* 
	     * found the new selection
	     */
	    else if (selection == mut_excl) 
	    {
	        /* ME group optional - simply reverse member's value */
	        current_status = ME_ON;
	        if ((!required) && (uplevel_status != MI_ON))
	        {
		    if (uplevel_status != MI_OFF)
		    {
		        (*set_group_selection)(selection, value);
		        if (mut_excl->opt_selected)
			    current_status = ME_ON;
			else current_status = ME_OFF;
		    }
		    else
		    {
			current_status = ME_OFF;
		    }
	        }

	        /* ME group required - turning member on */
                else if (!mut_excl->opt_selected)
                {
		    (*set_group_selection)(selection, TRUE);
		    current_status = ME_ON;
                }

		/* ME group required - don't allow member to turn off */
		else
                {
		    current_status = ME_ON;
                }
	    }
	    mut_excl = mut_excl->next;
	}

	/*
	 * traverse list of ME group siblings, 
	 * deal with un-selected member(s) in second pass
	 */
	mut_excl = start_group->group_next;
	while (mut_excl != NULL)
        {
	    if ((mut_excl->type == KUIS_MUTINCL) ||
	  	(mut_excl->type == KUIS_BLANK))
	    {
	        mut_excl = mut_excl->next;
	        continue;
	    }

	    /*
	     *  a loose group within the ME group, recurse
	     */
	    else if (mut_excl->type == KUIS_GROUP)
	    {
                substatus = do_loose_group(mut_excl, selection,  value,
					   current_status, required,
					   set_group_selection);

		/* if lower level group was turned on, then this
		   group is now considered on */
		if (substatus == ME_ON) current_status = ME_ON;
	    }

	    /* 
	     * found the old selection 
             */
	    else if ((mut_excl->opt_selected) && 
		     (mut_excl != selection)) 
	    {
	        if ((indirect_group_sibling(start_group, mut_excl, selection)) 
		    && ((uplevel_status == MI_ON) || 
		        (uplevel_status == NO_UPPER) ||
			(uplevel_status == LOOSE)))
		    current_status = ME_ON;
		else 
		{
		    (*set_group_selection)(mut_excl, FALSE);
		}
	    }
	    mut_excl = mut_excl->next;
	}

	/*
	 *  special case where we turn on one selection by default
	 */
	if ((uplevel_status == MI_ON) && (current_status == ME_OFF))
	{
	    mut_excl = start_group->group_next;
	    while ((mut_excl != NULL) && (!mut_excl->prev_selected))
	        mut_excl = mut_excl->next;

	    if (mut_excl != NULL)
	    {
                (*set_group_selection)(mut_excl, TRUE);
	    }
	    else
	    {
                (*set_group_selection)(start_group->group_next, TRUE);
	    }
        }

	if (current_status == ME_ON)
	    start_group->opt_selected = TRUE;
	else start_group->opt_selected = FALSE;

	start_group->modified = TRUE;
	return(current_status);
}


/*-----------------------------------------------------------
|
|  Routine Name: do_mi_group
|
|       Purpose: This is the subroutine for each of the
|                buttons that make up a mutually inclusive group.
|
|         Input: start_group - the beginning of the MI group
|                selection   - member of the MI group just clicked on
|                uplevel_status - status of UPPER LEVEL group, one of:
|                                 NO_UPPER, ME_ON, ME_OFF, MI_ON, MI_OFF
|        Output: none
|          Date: Jun 09, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/

static int do_mi_group(
   kselection *start_group,
   kselection *selection,
   int        value,
   int        uplevel_status,
   int        required,
   kfunc_void set_group_selection)
{
        kselection *mut_incl;
        int        substatus;
        int        current_status;  /* one of MI_ON, MI_OFF */

	/*
	 * traverse list of MI group siblings, 
	 * change values of all members to be the same value
	 */
	mut_incl = start_group->group_next;
	current_status = NO_UPPER;
	while (mut_incl != NULL)
	{
	    if ((mut_incl->type == KUIS_BLANK) ||
		(mut_incl->type == KUIS_GROUP) ||
		(mut_incl->type == KUIS_MUTEXCL))
	    {
		    mut_incl = mut_incl->next;
		    continue;
	    }

	    else if ((uplevel_status == ME_ON) &&
		    !(direct_group_sibling(start_group, mut_incl, selection)))
	    {
		if (mut_incl->opt_selected)
		{
		    (*set_group_selection)(mut_incl, FALSE);
		}
		current_status = MI_OFF;
	    }

	    else if (uplevel_status == MI_OFF)
	    {
	        if (mut_incl->opt_selected)
	        {
		    (*set_group_selection)(mut_incl, FALSE);
		}
		current_status = MI_OFF;
	    }

	    else if (direct_group_sibling(start_group, mut_incl, selection))
	    {
	        /* MI group must stay on - they try to turn it off */
	        if (!((required) && (mut_incl->opt_selected)))
	        {
		    (*set_group_selection)(mut_incl, value);
		}
		if (mut_incl->opt_selected)
		    current_status = MI_ON;
		else current_status = MI_OFF;
	    }
	    else if (uplevel_status == ME_OFF)
	    {
	        if (mut_incl->opt_selected)
	        {
	            (*set_group_selection)(mut_incl, FALSE);
	        }
	        current_status = MI_OFF;
	    }
	    else
	    {
	        if (mut_incl->opt_selected)
	            current_status = MI_ON;
	        else current_status = MI_OFF;
	    }
	    mut_incl = mut_incl->next;
	}

	mut_incl = start_group->group_next;
	while (mut_incl != NULL)
	{
	    /*
             *  a group within the MI group - use recursion later
             */
	    if (mut_incl->type == KUIS_MUTEXCL)
	    {
            	substatus = do_me_group(mut_incl, selection,  value,
					current_status, required,
					   set_group_selection);

	        /* if lower level group was turned on, then this
                   group is now considered on */
	        if (current_status == NO_UPPER)
	        {
                    if (substatus == ME_ON)
                        current_status = MI_ON;
		    else if (substatus == ME_OFF) 
                        current_status = MI_OFF;
		}
	    }

	    /*
	     *  a loose group within the ME group, recurse
	     */
	    else if (mut_incl->type == KUIS_GROUP)
	    {
                substatus = do_loose_group(mut_incl, selection,  value,
					   current_status, required,
					   set_group_selection);

		/* if lower level group was turned on, then this
		   group is now considered on */
		if (substatus == ME_ON) current_status = ME_ON;
	    }
		
	    mut_incl = mut_incl->next;
	}
	if (current_status == MI_ON)
	    start_group->opt_selected = TRUE;
	else start_group->opt_selected = FALSE;

	start_group->modified = TRUE;
	return(current_status);
}

/*-----------------------------------------------------------
|
|  Routine Name: do_loose_group
|
|       Purpose: This is the subroutine for each of the 
|                buttons that make up a loose group.
|
|         Input: start_group - the beginning of the loose group 
|                selection   - member of the group just clicked on
|                uplevel_status - status of UPPER LEVEL group, one of:
|                                 NO_UPPER, ME_ON, ME_OFF, MI_ON, MI_OFF
|        Output: none
|          Date: Oct 6, 1993
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/

static int do_loose_group(
   kselection *start_group,
   kselection *selection,
   int        value,
   int        uplevel_status,
   int        required,
   kfunc_void set_group_selection)
{
        int        selected_items, substatus;
        kselection *group;
        int        current_status; /* one of ME_ON, ME_OFF */

	current_status = ME_OFF;

	/*
	 *  see how many items in loose group are selected
	 */
	selected_items = selected_item_num(start_group);

	/*
	 * if none are on, and part of an ME group where value
	 * is already set on the upper level, can return.
	 */
	if ((selected_items == 0) && (uplevel_status == ME_ON))
	{
	    current_status = ME_OFF;
	    return(current_status);
	}
	
        /*
         * if none are on, and part of an MI group where values
         * are OFF on the upper level, can return.
         */
        if ((selected_items == 0) && (uplevel_status == MI_OFF))
        {
            current_status = ME_OFF;
            return(current_status);
        }

	/*
	 * if some are on, and part of an ME group where value
	 * is already set on the upper level, make sure all items
         * are turned off before returning.
	 */
	if ((selected_items > 0) && (uplevel_status == ME_ON))
	{
	    group = start_group->group_next;
	    while (group != NULL)
            {
	        if (group->opt_selected)
	        {
                    (*set_group_selection)(group, FALSE);
	        }
                group = group->next;
            }
	    return(current_status);
	}

	/*
	 * if some are on, and part of an MI group where values
	 * are OFF on the upper level, make sure all items
         * are turned off before returning.
	 */
	if ((selected_items > 0) && (uplevel_status == MI_OFF))
	{
	    group = start_group->group_next;
	    while (group != NULL)
            {
	        if (group->opt_selected)
	        {
                    (*set_group_selection)(group, FALSE);
	        }
                group = group->next;
            }
	    return(current_status);
	}

	/*
	 * if none are on, and part of an MI group where values
         * are ON on the upper level, turn on the first & return
	 */
	if ((selected_items == 0) && (uplevel_status == MI_ON))
        {
	    group = start_group->group_next;
            (*set_group_selection)(group, TRUE);
            current_status = ME_ON;
            return(current_status);
        }


	/*
	 * traverse list of group siblings, 
	 * deal with selected member in first pass
	 */
	group = start_group->group_next;
 	while (group != NULL) 
	{
            /* group optional - simply reverse member's value */
            if  ( (selected_items > 1) || (!group->opt_selected) ||
		  (group->type == KUIS_MUTEXCL) )
            {
	    /* an ME group within the loose group, recurse */
	    if ((group->type == KUIS_MUTEXCL) && 
	        (direct_group_sibling(group, group->group_next, selection)))
	    {
		if (selected_items > 1)
                    substatus = do_me_group(group, selection,  value,
	 				    LOOSE, FALSE,
					   set_group_selection);
		else substatus = do_me_group(group, selection, value,
                                             MI_ON, FALSE,
					   set_group_selection);

		/* if lower level group was turned on, then this
		   group is now considered on */
		if (substatus == ME_ON) current_status = ME_ON;
	    }

	    /* an MI group within the loose group, recurse */
	    else if ((group->type == KUIS_MUTINCL) &&
                    (direct_group_sibling(group, group->group_next, selection)))
	    {
                substatus = do_mi_group(group, selection,  value,
					current_status, required,
					   set_group_selection);

		/* if lower level group was turned on, then this
		   group is now considered on */
		if (substatus == MI_ON) current_status = ME_ON;
	    }
		
	    /* ignore blank selections inside the group */
	    else if (group->type == KUIS_BLANK)
	    {
		    group = group->next;
		    continue;
	    }

	    /* found the new selection */
	    else if (selection == group) 
	    {
                (*set_group_selection)(selection, value);
		break;
	    }

            }
	    group = group->next;
	}

	start_group->modified = TRUE;
	return(current_status);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_set_group_selection
|
|       Purpose: This routine sets a selection to selected or
|                non-selected, depending on the value passed in.
|
|         Input: selection   - the selection to set
|                value       - value to set it to
|        Output: none
|          Date: Jun 10, 1992 
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/
void kvf_set_group_selection(
   kselection *selection,
   int        value)
{
	kselection *save_back_group;

	save_back_group = selection->back_group;
	selection->back_group = NULL;
	kvf_set_attribute(selection->back_kformstruct, KVF_OPTSEL, value);
	if (value == TRUE)
	    selection->opt_selected = TRUE;
	selection->back_group = save_back_group;

}


/*-----------------------------------------------------------
|
|  Routine Name: direct_group_sibling
|
|       Purpose: This routine determines whether a member of a 
|                mutually inclusive or mutually exclusive group
|                is a direct sibling of another.
|
|         Input: start_group - header to list of group members
|                member1     - first member
|                member2     - second member
|        Output: none
|          Date: Jun 10, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/

static int direct_group_sibling(
   kselection *start_group,
   kselection *member1,
   kselection *member2)
{
	kselection *tmp;
	int found1 = FALSE, found2 = FALSE;

	tmp = start_group->group_next;
	while (tmp != NULL)
	{
	    if (member1  == tmp) found1 = TRUE;
	    if (member2 == tmp) found2 = TRUE;
	    tmp = tmp->next;
	}
	if (found1 && found2) return(TRUE);
	else return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: indirect_group_sibling
|
|       Purpose: This routine determines whether a member of a 
|                mutually inclusive or mutually exclusive group
|                is an indirect sibling of another.
|
|         Input: start_group - header to list of group members
|                member1     - first member
|                member2     - second member
|        Output: none
|          Date: Sept 15, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/

static int indirect_group_sibling(
   kselection *start_group,
   kselection *member1,
   kselection *member2)
{
	if (direct_group_sibling(start_group, member1, member2))
	    return(FALSE);
	
	if (start_group->back_group != NULL)
	{
	    if (  (group_member(start_group->back_group, member1) &&
	         (!group_member(start_group->back_group, member2)) ) ||
	        (group_member(start_group->back_group, member2) &&
               (!group_member(start_group->back_group, member1)) ) )
		return(FALSE);
	}

	else 
	{
	    if  ((group_member(start_group, member1) && 
		 (!group_member(start_group, member2))) ||

	         (group_member(start_group, member2) && 
		 (!group_member(start_group, member1))) )
		return(FALSE);
	}

	return(TRUE);
	
}
	
/*-----------------------------------------------------------
|
|  Routine Name: group_member
|
|       Purpose: Determines whether or not a selection is a
|                member of the ME or MI group.
|
|         Input: start_group - header of list of group members
|                member      - pointer to selection to be checked
|        Output: TRUE if member is a member of the group, FALSE otherwise
|          Date: June 10, 1992
|    Written By: Danielle Argiro
| Modifications: 
|
-----------------------------------------------------------*/

static int group_member(
   kselection *start_group,
   kselection *member)
{
        kselection *tmp;

	tmp = start_group->group_next;
        while (tmp != NULL)
	{
	    if (tmp == member) return(TRUE);
	    tmp = tmp->next;
        }
	return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: selected_item_num
|
|       Purpose: Returns the number of items in a group that are selected.
|
|         Input: start_group - header of list of group members
|        Output: 
|       Returns: number of items in group that are selected.
|          Date: June 10, 1992
|    Written By: Danielle Argiro
| Modifications: 
|
-----------------------------------------------------------*/

static int selected_item_num(
    kselection *start_group)
{
	kselection *group;
	int selected_items = 0;

	group = start_group->group_next;
 	while (group != NULL) 
	{
	    if (group->opt_selected)
	        selected_items++;
	    else if (group->group_next != NULL)
		selected_items += selected_item_num(group);
	    group = group->next;
	}
	return(selected_items);
}
