/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		Attribute Actions For Int Selections          <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                 kvf_get_nobounds_sel()                <<<<
   >>>>                 kvf_set_nobounds_sel()                <<<<
   >>>>                 kvf_get_bounds_sel()                  <<<<
   >>>>                 kvf_set_bounds_sel()                  <<<<
   >>>>	     		kvf_get_int_gt0_sel()	              <<<<
   >>>>	     		kvf_set_int_gt0_sel()	              <<<<
   >>>>	     		kvf_get_int_ge0_sel()	              <<<<
   >>>>	     		kvf_set_int_ge0_sel()	              <<<<
   >>>>	     		kvf_get_int_lt0_sel()	              <<<<
   >>>>	     		kvf_set_int_lt0_sel()	              <<<<
   >>>>	     		kvf_get_int_le0_sel()	              <<<<
   >>>>	     		kvf_set_int_le0_sel()	              <<<<
   >>>>	     		kvf_get_int_upper()		      <<<<
   >>>>	     		kvf_set_int_upper()		      <<<<
   >>>>	     		kvf_get_int_lower()		      <<<<
   >>>>	     		kvf_set_int_lower()		      <<<<
   >>>>	     		kvf_get_int_def()		      <<<<
   >>>>	     		kvf_set_int_def()		      <<<<
   >>>>	     		kvf_get_int_val()		      <<<<
   >>>>	     		kvf_set_int_val()		      <<<<
   >>>>	     		kvf_get_mechanism()		      <<<<
   >>>>	     		kvf_set_mechanism()		      <<<<
   >>>>	     		kvf_change_intbounds()	              <<<<
   >>>>                                                       <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"


/*-----------------------------------------------------------
|
|       Routine: kvf_get_nobounds_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not bounds are set on an integer selection 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_INT_NOBOUNDS_SEL ("kvf_int_nobounds_sel")
|        Output: calldata    - passes back TRUE if no bounds set
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_nobounds_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	int          *nobounds;

        /* set the pointer to be returned */
        nobounds = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	   return(FALSE);

	if (kformstruct->flag == KUIS_INTEGER)
	{
	    if ((line_info.lower_int == 0) &&
	 	(line_info.upper_int == 0))
	        *nobounds = TRUE;
	    else *nobounds = FALSE;
	}
	else if (kformstruct->flag == KUIS_FLOAT)
	{
	    if ((line_info.lower_float == 0.0) &&
		(line_info.upper_float == 0.0))
	        *nobounds = TRUE;
	    else *nobounds = FALSE;
	}
	else if (kformstruct->flag == KUIS_DOUBLE)
	{
	    if ((line_info.lower_double == 0.0) &&
		(line_info.upper_double == 0.0))
	        *nobounds = TRUE;
	    else *nobounds = FALSE;
	}

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_nobounds_sel
|       Purpose: Sets the no bounds on an integer selection 
|                according to the value of the "nobounds" 
|                selection on the pop-up menuform.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute - KVF_INT_NOBOUNDS_SEL ("kvf_int_nobounds_sel")
|                calldata  - provides boolean value of TRUE if bounds not set
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_nobounds_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (kformstruct->flag == KUIS_INTEGER)
	     kvf_change_intbounds(kformstruct, "nobounds");
	else if (kformstruct->flag == KUIS_FLOAT)
	     kvf_change_floatbounds(kformstruct, "nobounds");
	else if (kformstruct->flag == KUIS_DOUBLE)
	     kvf_change_doublebounds(kformstruct, "nobounds");

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kvf_get_bounds_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not bounds are set on an integer selection 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_INT_BOUNDS_SEL ("kvf_int_bounds_sel")
|        Output: calldata    - passes back TRUE if bounds set
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_bounds_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	int          *bounds;

        /* set the pointer to be returned */
        bounds = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	   return(FALSE);

	if (kformstruct->flag == KUIS_INTEGER)
	{
	    if (line_info.lower_int != line_info.upper_int)
	        *bounds = TRUE;
	    else *bounds = FALSE;
	}
	else if (kformstruct->flag == KUIS_FLOAT)
	{
	    if (line_info.lower_float != line_info.upper_float)
	        *bounds = TRUE;
	    else *bounds = FALSE;
	}
	else if (kformstruct->flag == KUIS_DOUBLE)
	{
	    if (line_info.lower_double != line_info.upper_double)
	        *bounds = TRUE;
	    else *bounds = FALSE;
	}

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_bounds_sel
|       Purpose: Sets the bounds on an integer selection 
|                according to the value of the "bounds" 
|                selection on the pop-up menuform.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute - KVF_INT_BOUNDS_SEL ("kvf_int_bounds_sel")
|                calldata  - provides boolean value of TRUE if bounds not set
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_bounds_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (kformstruct->flag == KUIS_INTEGER)
	     kvf_change_intbounds(kformstruct, "bounds");
	else if (kformstruct->flag == KUIS_FLOAT)
	     kvf_change_floatbounds(kformstruct, "bounds");
	else if (kformstruct->flag == KUIS_DOUBLE)
	     kvf_change_doublebounds(kformstruct, "bounds");

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kvf_get_int_gt0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an int selection are > 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_INT_GT0_SEL ("kvf_int_gt0_sel")
|        Output: calldata    - passes back TRUE if bounds > 0 set
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_int_gt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int       *gt0;
	Line_Info line_info;

        /* set the pointer to be returned */
        gt0 = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	   return(FALSE);

	if ((line_info.upper_int == line_info.lower_int) &&
	    (line_info.lower_int == 2))
  	    *gt0 = TRUE;
	else *gt0 = FALSE;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_int_gt0_sel
|       Purpose: Sets the bounds of an int selection to > 0
|                according to the value of the "int_gt0" 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_INT_GT0_SEL ("kvf_int_gt0_sel")
|                calldata    - provides value of TRUE if bounds > 0 are set
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_int_gt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int  *gt0; 
	int  int_def;

	/* value to which KVF_INT_GT0_SEL is to be set */
        gt0  = (int *) calldata;

	if (*gt0 != TRUE) return(FALSE);

	if (!(kvf_change_intbounds(kformstruct, "gt0")))
	    return(FALSE);

	kvf_get_attribute(kformstruct, KVF_INT_DEF, &int_def);
	if (int_def <= 0)
	    kvf_set_attribute(kformstruct, KVF_INT_DEF, 1);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_int_ge0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an int selection are >= 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_INT_GE0_SEL ("kvf_int_ge0_sel")
|        Output: calldata    - passes back TRUE if bounds are set >= 0
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_int_ge0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *ge0;
	Line_Info    line_info;

        /* set the pointer to be returned */
        ge0 = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	   return(FALSE);

	if ((line_info.lower_int == line_info.upper_int) &&
	    (line_info.lower_int == 1))
  	    *ge0 = TRUE;
	else *ge0 = FALSE;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_int_ge0_sel
|       Purpose: Sets the bounds of an int selection to >= 0
|                according to the value of the "int_ge0" 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_INT_GE0_SEL ("kvf_int_ge0_sel")
|                calldata    - provides boolean value of TRUE if bounds are >= 0
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_int_ge0_sel(
   kform_struct *kformstruct,
   char        *attribute,
   kaddr       calldata)
{
	int *ge0; 
	int int_def;

	/* value to which KVF_INT_GE0_SEL is to be set */
        ge0  = (int *) calldata;

	if (*ge0 != TRUE) return(FALSE);

	if (!(kvf_change_intbounds(kformstruct, "ge0")))
	     return(FALSE);

	kvf_get_attribute(kformstruct, KVF_INT_DEF, &int_def);
	if (int_def < 0)
	    kvf_set_attribute(kformstruct, KVF_INT_DEF, 0);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_int_lt0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an int selection are < 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_INT_LT0_SEL ("kvf_int_lt0_sel")
|        Output: calldata    - passes back TRUE if bounds are set to < 0
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_int_lt0_sel(
   kform_struct *kformstruct,
   char     *attribute,
   kaddr    calldata)
{
	int          *lt0;
	Line_Info    line_info;

        /* set the pointer to be returned */
        lt0 = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	   return(FALSE);

	if ((line_info.lower_int == line_info.lower_int) &&
	    (line_info.lower_int == -2))
  	    *lt0 = TRUE;
	else *lt0 = FALSE;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_int_lt0_sel
|       Purpose: Sets the bounds of an int selection to < 0
|                according to the value of the "int_lt0" 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_INT_LT0_SEL ("kvf_int_lt0_sel")
|                calldata    - provides boolean value of TRUE if bounds are < 0
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_int_lt0_sel(
   kform_struct *kformstruct,
   char        *attribute,
   kaddr       calldata)
{
	int *lt0; 
	int int_def;

	/* value to which KVF_INT_LT0_SEL is to be set */
        lt0  = (int *) calldata;

	if (*lt0 != TRUE) return(FALSE);
 
	if (!(kvf_change_intbounds(kformstruct, "lt0")))
	    return(FALSE);

	kvf_get_attribute(kformstruct, KVF_INT_DEF, &int_def);
	if (int_def >= 0)
	    kvf_set_attribute(kformstruct, KVF_INT_DEF, -1);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_int_le0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an int selection are <= 0
|
|         Input: kformstruct    - backplane of the selection
|                attribute - KVF_INT_LE0_SEL ("kvf_int_le0_sel")
|        Output: calldata  - passes back TRUE if bounds are set to <= 0
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_int_le0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *le0;
	Line_Info    line_info;

        /* set the pointer to be returned */
        le0 = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	   return(FALSE);

	if ((line_info.lower_int == line_info.lower_int) &&
	    (line_info.lower_int == -1))
  	    *le0 = TRUE;
	else *le0 = FALSE;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_int_le0_sel
|       Purpose: Sets the bounds of an int selection to <= 0
|                according to the value of the "int_le0" 
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_INT_LE0_SEL ("kvf_int_le0_sel")
|                calldata    - provides boolean value of TRUE if bounds are <= 0
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_int_le0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int *le0; 
	int int_def;

	/* value to which KVF_INT_LE0_SEL is to be set */
        le0  = (int *) calldata;

	if (*le0 != TRUE) return(FALSE);

	if (!(kvf_change_intbounds(kformstruct, "le0")))
	    return(FALSE);

	kvf_get_attribute(kformstruct, KVF_INT_DEF, &int_def);
	if (int_def > 0)
	    kvf_set_attribute(kformstruct, KVF_INT_DEF, 0);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_int_upper
|       Purpose: Gets the value of the upper bound for an int,
|                assuming that "kvf_int_upper_sel" is TRUE
|
|         Input: kformstruct    - backplane of the selection
|                attribute - KVF_INT_UPPER ("kvf_int_upper")
|        Output: calldata  - passes back int upper bound value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_int_upper(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int       *upper;
	Line_Info line_info;

        /* set the pointer to be returned */
        upper = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	   return(FALSE);

  	*upper = line_info.upper_int;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_int_upper
|       Purpose: Sets the value of the upper bound for an int,
|                assuming that "kvf_int_upper_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_INT_UPPER ("kvf_int_upper")
|                calldata    - provides int upper bound value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_int_upper(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int  *upper; 
	int  int_def;
	char temp[KLENGTH];

	/* value to which KVF_INT_UPPER is to be set */
        upper  = (int *) calldata;

	sprintf(temp, "upper = %d", *upper);

	if (!(kvf_change_intbounds(kformstruct, temp)))
	    return(FALSE);

	if (!(kvf_get_attribute(kformstruct, KVF_INT_DEF, &int_def)))
            return(FALSE);

	if (int_def > *upper)
	{
	     int_def = *upper;
	     if (!(kvf_set_attribute(kformstruct, KVF_INT_DEF, int_def)))
            	return(FALSE);
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_int_lower
|       Purpose: Gets the value of the lower bound for an int,
|                assuming that "kvf_int_lower_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_INT_LOWER ("kvf_int_lower")
|        Output: calldata    - passes back int lower bound value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_int_lower(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int       *lower;
	Line_Info line_info;

        /* set the pointer to be returned */
        lower = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	   return(FALSE);

  	*lower = line_info.lower_int;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);

}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_int_lower
|       Purpose: Sets the value of the lower bound for an int,
|                assuming that "kvf_int_lower_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_INT_LOWER ("kvf_int_lower")
|                calldata    - provides int lower bound value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_int_lower(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int  *lower; 
	int  int_def;
	char temp[KLENGTH];

	/* value to which KVF_INT_GE0_SEL is to be set */
        lower  = (int *) calldata;

	sprintf(temp, "lower = %d", *lower);

	if (!(kvf_change_intbounds(kformstruct, temp)))
	    return(FALSE);

	if (!(kvf_get_attribute(kformstruct, KVF_INT_DEF, &int_def)))
	    return(FALSE);

	if (int_def < *lower)
	{
	     int_def = *lower;
	     if (!(kvf_set_attribute(kformstruct, KVF_INT_DEF, int_def)))
            	return(FALSE);
	}
	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kvf_get_int_def
|       Purpose: Gets the value of the default for an int
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_INT_DEF ("kvf_int_def")
|        Output: calldata    - passes back int default value
|
|          Date: March 15, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_int_def(
   kform_struct *kformstruct,
   char     *attribute,
   kaddr    calldata)
{
	int       *int_def;
	Line_Info line_info;

        /* set the pointer to be returned */
        int_def = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	   return(FALSE);

  	*int_def = line_info.int_def;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_int_def
|       Purpose: Sets the value of the default for an int
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_INT_DEF ("kvf_int_def")
|                calldata    - provides int default value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_int_def(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char         temp[KLENGTH];
	int          *int_def;
	Line_Info    line_info;
	kselection   *selection;

	/* value to which KVF_INT_DEF is to be set */
        int_def  = (int *) calldata;

	if (!(kvf_set_attribute(kformstruct, KVF_INT_VAL, *int_def)))
	    return(FALSE);

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	   return(FALSE);

	line_info.int_def = *int_def;
	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	if (!(kvf_set_attribute(kformstruct, KVF_INT_VAL, *int_def)))
            return(FALSE);

	selection = kformstruct->Selptr;
	if (selection->back_toggle != NULL)
	{
	    sprintf(temp, "%d", *int_def);
	    kvf_set_attribute(kformstruct, KVF_TITLE, temp);
	}

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kvf_get_int_val
|       Purpose: Gets the value of the int
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_INT_VAL ("kvf_int_val")
|        Output: calldata    - passes back int value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_int_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int       *int_val;
	Line_Info line_info;
	char      *literal;

        /* set the pointer to be returned */
        int_val = (int *) calldata;

	/*
	 *  get & set the literal to itself before setting return
	 *  integer value.  the reason we do this is to force a re-evaluation
	 *  of integer expressions before the value is returned.  For 
	 *  example, suppose literal ="j+1", int val = "2" because j was 1
	 *  last time the integer was evaluated.  in the interim, suppose
	 *  j got changed to 10.  the correct value of j is now 11, but 
	 *  line_info.int_val will still be set to 2.  need to get into the
	 *  kvf_set_literal code, which has a side effect of calling the 
	 *  expression parser and updating the integer value.  just set the 
	 *  literal to itself in order to get there, it doesn't harm anything 
	 *  and will have no effect if the integer value is not given using
	 *  an expression.
	 */
	kvf_get_attribute(kformstruct, KVF_LITERAL, &literal);
	if (literal != NULL)
	    kvf_set_attribute(kformstruct, KVF_LITERAL, literal);

	/*
	 *  now, simply return the integer value that is in the UIS line;
	 *  it will be updated if it came from an expression, unchanged 
	 *  otherwise.
	 */
	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	   return(FALSE);
  	*int_val = line_info.int_val;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_int_val
|       Purpose: Sets the value of an int
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_INT_VAL ("kvf_int_val")
|                calldata    - provides int value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_int_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int        *int_val;
	Line_Info  line_info;

	/* value to which KVF_INT_VAL is to be set */
        int_val  = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	   return(FALSE);

        if (!(kvf_check_double((double)*int_val, 
			       (double) line_info.upper_int,
                               (double) line_info.lower_int, 
			       line_info.variable, line_info.title)))
        {
            kvf_free_line_info_strings(&line_info);
            return(FALSE);
        }
	line_info.int_val = *int_val;

	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_mechanism
|       Purpose: Gets the mechanism for an integer,
|                float, or double selection 
|                (0 = NONE, 1 = scrollbar, 2 = dial)
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_MECHANISM ("kvf_mechanism")
|        Output: calldata    - passes back mechanism value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_mechanism(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *mechanism; 
	int          bounds; 
	double       upper, lower;
	Line_Info    line_info;
	kselection   *selection;

        /* set the pointer to be returned */
        mechanism = (int *) calldata;

	selection = kformstruct->Selptr;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	   return(FALSE);

	if (selection->type == KUIS_INTEGER)
	{
	    upper = (double) line_info.upper_int;
	    lower = (double) line_info.lower_int;
	    if (upper != lower) 
		bounds = TRUE;
	    else bounds = FALSE;
	}
	else if (selection->type == KUIS_FLOAT)
        {
	    upper = (double) line_info.upper_float;
	    lower = (double) line_info.lower_float;
	    if (upper != lower) 
                bounds = TRUE;
            else bounds = FALSE;
        }
        else if (selection->type == KUIS_DOUBLE)
        {
	    upper = line_info.upper_double;
	    lower = line_info.lower_double;
	    if (upper != lower) 
                bounds = TRUE;
            else bounds = FALSE;
        }
	else
	{
	    kerror("kforms", "kvf_get_mechanism",
	           "Can only get attribute KVF_MECHANISM on integer, float, or double selections");
	    return(FALSE);
	}
	if ((bounds) && (line_info.special == 1))
	    *mechanism = TRUE;
	else *mechanism = FALSE;


	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_mechanism
|       Purpose: Sets the mechanism for an integer,
|                float, or double selection 
|                (0 = NONE, 1 = scrollbar, 2 = dial)
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_MECHANISM ("kvf_mechanism")
|                calldata    - passes in mechanism value
|        Output: none
|
|          Date: May 7, 1993
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_mechanism(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *mechanism, bounds; 
	Line_Info    line_info;
	double       upper, lower;
	kselection   *selection;

        /* the value to which KVF_MECHANISM is to be set */
        mechanism  = (int *) calldata;

	selection = kformstruct->Selptr;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	   return(FALSE);

	if (selection->type == KUIS_INTEGER)
	{
	    upper = (double) line_info.upper_int;
	    lower = (double) line_info.lower_int;
	    if (upper != lower) 
		bounds = TRUE;
	    else bounds = FALSE;
	}
	else if (selection->type == KUIS_FLOAT)
        {
	    upper = (double) line_info.upper_float;
	    lower = (double) line_info.lower_float;
	    if (upper != lower) 
                bounds = TRUE;
            else bounds = FALSE;
        }
        else if (selection->type == KUIS_DOUBLE)
        {
	    upper = line_info.upper_double;
	    lower = line_info.lower_double;
	    if (upper != lower) 
                bounds = TRUE;
            else bounds = FALSE;
        }
        else
        {
            kerror("kforms", "kvf_get_mechanism",
                   "Can only set attribute KVF_MECHANISM on integer, float, or double selections");
            return(FALSE);
        }

	if ((!bounds) && (*mechanism))
	{
	    kerror("kforms", "kvf_set_mechanism",
		   "Can't put a scrollbar on an unbounded int, float, or double"); 
	    kvf_free_line_info_strings(&line_info);
	    return(FALSE);
	}

	line_info.special = *mechanism;
	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: kvf_change_intbounds
|
|       Purpose: Changes the bounds for a int selection
|
|         Input: kformstruct - generic kform_struct pointer
|		 string    - string rep of int bounds, may be one of:
|			     "gt0"        : bounds >  0
|			     "ge0"        : bounds >= 0
|			     "lt0"        : bounds <  0
|			     "le0"        : bounds <= 0
|                            "lower = %d" : lower bound 
|                            "upper = %d" : upper bound 
|			     "nobounds"   : no bounds 
|
|        Output: Returns TRUE on success, FALSE on Failure
|          Date: March 20, 1994 
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

int kvf_change_intbounds(
   kform_struct *kformstruct,
   char         *string)
{
	kselection *selection;
	Line_Info  line_info;
	int        lower, upper;
	int        status, bounds = FALSE;

	kvf_clear_line_info(&line_info);
	selection = kformstruct->Selptr;
        kvf_gen_parse(selection->line, &line_info);

	if (kstrcmp(string, "gt0") == 0)
	{
	   line_info.lower_int = 2;
	   line_info.upper_int = 2;
	}
	else if (kstrcmp(string, "ge0") == 0)
        {
           line_info.lower_int = 1;
           line_info.upper_int = 1;
        }
	else if (kstrcmp(string, "lt0") == 0)
        {
           line_info.lower_int = -2;
           line_info.upper_int = -2;
        }
	else if (kstrcmp(string, "le0") == 0)
        {
           line_info.lower_int = -1;
           line_info.upper_int = -1;
        }
	else if (kstrcmp(string, "nobounds") == 0)
        {
           line_info.lower_int = 0;
           line_info.upper_int = 0;
        }
	else if (kstrcmp(string, "bounds") == 0)
        {
	   if (line_info.lower_int == line_info.upper_int)
	   {
               line_info.lower_int = -100;
               line_info.upper_int =  100;
	   }
	   bounds = TRUE;
        }
	else 
	{
	   status = ksscanf(string, "lower = %d", &lower);
	   if (status != 1)
	   {
	       status = ksscanf(string, "upper = %d", &upper);
	       if (status != 1)
	       {
	 	   kerror("kforms", "kvf_change_int_bounds",
		          "Invalid string provided for KVF_INTBOUNDS_CHNG");
		   return(FALSE);
	       }
	       else line_info.upper_int = upper;
	   }
	   else line_info.lower_int = lower;

           if (line_info.lower_int != line_info.upper_int)
	       bounds = TRUE;
	}

	if (line_info.upper_int < line_info.lower_int)
	{
	    kerror("kforms", "kvf_change_int_bounds",
	  	   "Attempt to set int upper bound < lower bound");
	    return(FALSE);
	}
	kvf_gen_deparse(&line_info, &selection->line);

	if (bounds)
	{
	    if (line_info.int_val < line_info.lower_int)
	        line_info.int_val = line_info.lower_int;

    	    if (line_info.int_val > line_info.upper_int)
	        line_info.int_val = line_info.upper_int;

	    kvf_set_attribute(kformstruct, KVF_INT_DEF, line_info.int_val);
	}

	kvf_free_line_info_strings(&line_info);

	return(TRUE);
}

