/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		    General Attribute Actions                 <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>	     		kvf_get_activate()		      <<<<
   >>>>	     		kvf_set_activate()		      <<<<
   >>>>	     		kvf_get_selected()		      <<<<
   >>>>	     		kvf_set_selected()		      <<<<
   >>>>	     		kvf_get_optional()		      <<<<
   >>>>	     		kvf_set_optional()		      <<<<
   >>>>	     		kvf_get_optsel()		      <<<<
   >>>>	     		kvf_set_optsel()		      <<<<
   >>>>	     		kvf_get_live()			      <<<<
   >>>>	     		kvf_set_live()			      <<<<
   >>>>	     		kvf_get_variable()		      <<<<
   >>>>	     		kvf_set_variable()		      <<<<
   >>>>	     		kvf_get_description()		      <<<<
   >>>>	     		kvf_set_description()		      <<<<
   >>>>	     		kvf_get_title()			      <<<<
   >>>>	     		kvf_set_title()			      <<<<
   >>>>	     		kvf_get_buttontitle()		      <<<<
   >>>>	     		kvf_set_buttontitle()		      <<<<
   >>>>	     		kvf_get_guidepanetitle()	      <<<<
   >>>>	     		kvf_set_guidepanetitle()	      <<<<
   >>>>	     		kvf_get_literal()		      <<<<
   >>>>	     		kvf_set_literal()		      <<<<
   >>>>	     		kvf_get_modified()		      <<<<
   >>>>	     		kvf_set_modified()		      <<<<
   >>>>	     		kvf_get_workspace()		      <<<<
   >>>>	     		kvf_set_workspace()		      <<<<
   >>>>	     		kvf_get_client_data()		      <<<<
   >>>>	     		kvf_set_client_data()		      <<<<
   >>>>	     		kvf_set_delete()		      <<<<
   >>>>	     		kvf_set_display_subform()	      <<<<
   >>>>	     		kvf_set_display_pane()		      <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"


/*-----------------------------------------------------------
|
|       Routine: kvf_get_activate
|       Purpose: Gets the value of the activate field
|                from the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_ACTIVATE ("kvf_activate")
|        Output: calldata    - passes back TRUE if GUI item is activated
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/

/* ARGSUSED */
int kvf_get_activate(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *active;
	Line_Info    line_info;

        /* set the pointer to be returned */
        active = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (line_info.activate)
	    *active = TRUE;
	else if ((line_info.typeflag == KUIS_BLANK) ||
		 (line_info.typeflag == KUIS_WORKSPACE))
	    *active = TRUE;
	else *active = FALSE;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_activate
|       Purpose: Sets the value of the activate field
|                on the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_ACTIVATE ("kvf_activate")
|                calldata    - provides boolean value of TRUE or FALSE
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_activate(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        int          *active;
	Line_Info    line_info;

	/* the value to which KVF_ACTIVATE is to be set */
        active    = (int *) calldata;
	if (!(kvf_check_boolean(*active)))
        {
	    errno = KINVALID_PARAMETER;
            kerror("kforms", "kvf_set_activate", 
		   "'kvf_activate' may only be set to 0 or 1");
	    return(FALSE);
        }

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	line_info.activate = *active;
        kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_selected
|       Purpose: Gets the value of the selected field
|                from the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_SELECTED ("kvf_selected")
|        Output: calldata    - passes TRUE if GUI item is selected
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_selected(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *selected;
	Line_Info    line_info;

        /* set the pointer to be returned */
        selected = (int *) calldata;

	if (kformstruct->type == KSELECTION)
	    *selected = kformstruct->Selptr->selected;
	else if (kformstruct->type == KSUBFORM)
	    *selected = kformstruct->Subformptr->selected;
	else if (kformstruct->type == KGUIDE)
	    *selected = kformstruct->Guideptr->selected;
	else
	{
	    if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	        return(FALSE);
	    if (line_info.selected)
	        *selected = TRUE;
	    else *selected = FALSE;
 	    kvf_free_line_info_strings(&line_info);
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_selected
|       Purpose: Sets the value of the selected field
|                on the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_SELECTED ("kvf_selected")
|                calldata    - provides boolean value of TRUE or FALSE
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_selected(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        int          *selected;
	Line_Info    line_info;

	/* the value to which KVF_SELECTED is to be set */
        selected = (int *) calldata;
	if (!(kvf_check_boolean(*selected)))
        {
	    errno = KINVALID_PARAMETER;
            kerror("kforms", "kvf_set_selected", 
		   "'kvf_selected' may only be set to 0 or 1");
	    return(FALSE);
        }

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);
	line_info.selected = *selected;
	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	if (kformstruct->type == KSELECTION)
	    kformstruct->Selptr->selected = *selected;
	else if (kformstruct->type == KSUBFORM)
	    kformstruct->Subformptr->selected = *selected;
	else if (kformstruct->type == KGUIDE)
	    kformstruct->Guideptr->selected = *selected;
	return(TRUE);

}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_optional
|       Purpose: Gets the value of the optional field
|                from the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_OPTIONAL ("kvf_optional")
|        Output: calldata    - passes back boolean value of TRUE or FALSE
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_optional(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *optional;
	Line_Info    line_info;

        /* set the pointer to be returned */
        optional = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

        if ((kformstruct->flag == KUIS_MUTEXCL) ||
            (kformstruct->flag == KUIS_MUTINCL) ||
            (kformstruct->flag == KUIS_GROUP))
        {
	    if (line_info.logical_val)
		*optional = FALSE;
	    else *optional = TRUE;
	}
	else
	{
	    if (line_info.optional)
	        *optional = TRUE;
	    else *optional = FALSE;
	}
	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_optional
|       Purpose: Sets the value of the optional field
|                on the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_OPTIONAL ("kvf_optional")
|                calldata    - provides boolean value of TRUE or FALSE
|
|        Output: none
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_optional(
   kform_struct *kformstruct,
   char        *attribute,
   kaddr       calldata)
{
        int           *optional;
	Line_Info     line_info;
	kselection    *selection;

	/* the value to which KVF_OPTIONAL is to be set */
        optional  = (int *) calldata;
	if (!(kvf_check_boolean(*optional)))
        {
	    errno = KINVALID_PARAMETER;
            kerror("kforms", "kvf_set_optional", 
		   "'kvf_optional' may only be set to 0 or 1");
	    return(FALSE);
        }

	selection = kformstruct->Selptr;
	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if ((selection->type == KUIS_MUTEXCL) ||
	    (selection->type == KUIS_MUTINCL) ||
	    (selection->type == KUIS_GROUP))
	{
	     line_info.logical_val = !(*optional);
	     kvf_gen_deparse_kformstruct(kformstruct, &line_info);
	     return(TRUE);
	}
	if ((selection->type == KUIS_FLAG) && (*optional != 1))
	{
	    kerror("kforms", "kvf_set_optional",
                   "Flag selections must always be optional");
            return(FALSE);
        }
	if ((selection->back_toggle != NULL) &&
	    (*optional == FALSE))
	{
	    kerror("kforms", "kvf_set_optional",
                   "Toggle members must always be optional");
            return(FALSE);
	}
	if ((selection->back_group != NULL) &&
	    (*optional == FALSE))
	{
	    kerror("kforms", "kvf_set_optional",
                   "Mutually inclusive/exclusive members must always be optional");
            return(FALSE);
	}

	line_info.optional = *optional;
	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_optsel
|       Purpose: Gets the value of the opt sel field
|                from the UIS line of a GUI item
|
|         Input: object    - backplane of the selection
|                attribute - KVF_OPTSEL ("kvf_optsel")
|        Output: calldata  - passes back integer value of 0, 1, or 2
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_optsel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *optsel;
	Line_Info    line_info;

        /* set the pointer to be returned */
        optsel = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	*optsel = line_info.opt_sel;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_optsel
|       Purpose: Sets the value of the opt sel field
|                on the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute - KVF_OPTSEL ("kvf_optsel")
|                calldata  - provides integer value of 0, 1, or 2
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_optsel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        int           *optsel;
	Line_Info     line_info;
	kselection    *selection;

	/* the value to which KVF_OPTSEL is to be set */
        optsel    = (int *) calldata;
	if ((*optsel < 0) || (*optsel > 2))
        {
	    errno = KINVALID_PARAMETER;
            kerror("kforms", "kvf_set_optsel",
                   "'kvf_optsel' may only be set to 0, 1, or 2");
            return(FALSE);
        }

        selection = kformstruct->Selptr;
        if ((selection->back_group != NULL) && (*optsel == 1))
        {
            kvf_call_do_group(selection, TRUE);
            return(TRUE);
        }

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if ((!line_info.optional) && (*optsel != 1))
	{
	    kerror("kforms", "kvf_set_optsel",
		   "'opt-sel' must stay 1 (Selected) when 'optional' is FALSE"); 
            return(FALSE);
	}

	line_info.opt_sel = *optsel;
        kvf_gen_deparse_kformstruct(kformstruct, &line_info);

        kformstruct->Selptr->opt_selected = *optsel;

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_live
|       Purpose: Gets the value of the live field
|                from the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_LIVE ("kvf_live")
|        Output: calldata    - passes back boolean value of TRUE or FALSE
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_live(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *live;
	Line_Info    line_info;

        /* set the pointer to be returned */
        live = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (line_info.live)
	    *live = TRUE;
	else *live = FALSE;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_live
|       Purpose: Sets the value of the live field
|                on the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_LIVE ("kvf_live")
|                calldata    - provides boolean value of TRUE or FALSE
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_live(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *live;
	Line_Info    line_info;

	/* the current value that "kvf_live" is to be set */
        live  = (int *) calldata;
	if (!(kvf_check_boolean(*live)))
        {
	    errno = KINVALID_PARAMETER;
            kerror("kforms", "kvf_set_live", 
		   "'kvf_live' may only be set to 0 or 1");
	    return(FALSE);
        }

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

        line_info.live = *live;
        kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_variable
|       Purpose: Gets the value of the variable field
|                from the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_VARIABLE ("kvf_variable")
|        Output: calldata    - passes back string value of variable
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_variable(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char         **variable;
	Line_Info    line_info;

        /* set the pointer to be returned */
        variable = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (line_info.variable == NULL)
	    *variable = NULL;
	else *variable = kstrdup(line_info.variable);

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_variable
|       Purpose: Sets the value of the variable field
|                on the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_VARIABLE ("kvf_variable")
|                calldata    - provides string value of variable
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_variable(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char         **variable;
	Line_Info    line_info;
	kguide       *guide;
	ksubform     *subform;

	/* the value to which KVF_VARIABLE is to be set */
        variable  = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (*variable == NULL)
	{
	    kerror("kforms", "kvf_set_variable",
		   "variable may never be set to NULL");
	    return(FALSE);
	}
	if (isdigit((*variable)[0]))
	{
	    kerror("kforms", "kvf_set_variable",
		   "You may not begin a variable name with a number.");
	    return(FALSE);
	}

	kfree(line_info.variable);
	line_info.variable = kstrdup(*variable);
	kvf_gen_deparse_kformstruct(kformstruct, &line_info);
	kvf_free_line_info_strings(&line_info);

	if (kformstruct->type == KSELECTION)
	{
	    kformstruct->Selptr->var_token = 
			kstring_to_token(*variable);
	}
	else if (kformstruct->type == KGUIDE)
	{
	    guide = kformstruct->Guideptr;
	    kvf_parse_startpane_line(guide->control_line, &line_info);
	    kfree(line_info.variable);
	    line_info.variable = kstrdup(*variable);
	    kvf_deparse_startpane_line(&line_info, &guide->control_line);
	    kvf_deparse_startpane_line(&line_info, &guide->pane->control_line);
	    kvf_free_line_info_strings(&line_info);
	    guide->var_token = kstring_to_token(*variable);
	    guide->pane->var_token = kstring_to_token(*variable);
	}
	else if (kformstruct->type == KPANE)
	{
	    guide = kformstruct->Controlptr->back_guide;

	    /* change variable on guide button if applicable (to match) */
	    if (guide->line != NULL)
	    {
	        kvf_parse_guide_line(guide->line, &line_info);
	        kfree(line_info.variable);
	        line_info.variable = kstrdup(*variable);
	        kvf_deparse_guide_line(&line_info, &guide->line);
	        kvf_free_line_info_strings(&line_info);
	    }

	    /* change variable on -P line as referenced by guide */
	    kvf_parse_startpane_line(guide->control_line, &line_info);
	    kfree(line_info.variable);
	    line_info.variable = kstrdup(*variable);
	    kvf_deparse_startpane_line(&line_info, &guide->control_line);
	    kvf_free_line_info_strings(&line_info);

	    /* change variable on -P line as referenced by pane */
	    kvf_parse_startpane_line(guide->pane->control_line, &line_info);
            kfree(line_info.variable);
            line_info.variable = kstrdup(*variable);
            kvf_deparse_startpane_line(&line_info, &guide->pane->control_line);
            kvf_free_line_info_strings(&line_info);

	    /* change var token as referenced by guide & pane */
	    guide->var_token = kstring_to_token(*variable);
	    guide->pane->var_token = kstring_to_token(*variable);

	}
	else if (kformstruct->type == KSUBFORM)
	{
	    subform = kformstruct->Subformptr;
	    kvf_parse_subformbutton_line(subform->line, &line_info);
	    kfree(line_info.variable);
	    line_info.variable = kstrdup(*variable);
	    kvf_deparse_subformbutton_line(&line_info, &subform->line);
	    kvf_free_line_info_strings(&line_info);

	    kvf_parse_startsubform_line(subform->control_line, &line_info);
	    kfree(line_info.variable);
	    line_info.variable = kstrdup(*variable);
	    kvf_deparse_startsubform_line(&line_info, &subform->control_line);
	    kvf_free_line_info_strings(&line_info);
	    subform->var_token = kstring_to_token(*variable);
	}

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_description
|       Purpose: Gets the value of the description field
|                from the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_DESCRIPTION ("kvf_description")
|        Output: calldata    - passes back string value of description
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_description(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char         **description;
	Line_Info    line_info;

        /* set the pointer to be returned */
        description = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (line_info.description == NULL)
	    *description = NULL;
	else *description = kstrdup(line_info.description);

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_description
|       Purpose: Sets the value of the description field
|                on the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_DESCRIPTION ("kvf_description")
|                calldata    - provides string value of description
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_description(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char          **description;
	Line_Info     line_info;

	/* the value to which KVF_DESCRIPTION is to be set */
        description  = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	kfree(line_info.description);
	if (kstrlen(*description) > 0)
            line_info.description = kstrdup(*description);
	else line_info.description = NULL;

	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_title
|       Purpose: Gets the value of the title field
|                from the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_TITLE ("kvf_title")
|        Output: calldata    - passes back string value of title
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_title(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	char         **title;

        /* set the pointer to be returned */
        title = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (line_info.title == NULL)
	    *title = NULL;
	else *title = kstrdup(line_info.title);

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_title
|       Purpose: Sets the value of the title field
|                on the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_TITLE ("kvf_title")
|                calldata    - provides string value of title
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_title(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char          **title;
	Line_Info     line_info;

	/* the current value that "kvf_title" is to be set */
        title  = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

        kfree(line_info.title);
        if (kstrlen(*title) > 0)
            line_info.title = kstrdup(*title);
        else line_info.title = NULL;

        kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_buttontitle
|       Purpose: Gets the value of the title field from the -d UIS line
|                This routine is specialized for use with subforms, where 
|                you can't tell from the kformstruct if they want to get
|                the attributes of the subform button or of the subform itself.
|                However, it will also work for any of the other button-specific
|                UIS lines (-g, -m, -n, -a, -R, -H, -Q), as it just turns
|                around and calls kvf_get_title().  
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_BUTTONTITLE ("kvf_buttontitle")
|        Output: calldata    - passes back string value of title
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_buttontitle(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	char         **title;
	char         *line = NULL;

	if (kformstruct->type != KSUBFORM)
	    return(kvf_get_title(kformstruct, KVF_TITLE, calldata));
	/*
	 * Note: we are parsing the UIS line directly here, 
	 *       because kvf_gen_parse_kformstruct isn't smart
	 *       enough to know whether you want the -M or the -d
	 *       for a subform.
	 */
	kvf_clear_line_info(&line_info);

        /* set the pointer to be returned */
        title = (char **) calldata;

	line = kformstruct->Subformptr->line;

	if (!(kvf_parse_subformbutton_line(line, &line_info)))
	    return(FALSE);

	if (line_info.title == NULL)
	    *title = NULL;
	else *title = kstrdup(line_info.title);

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_buttontitle
|       Purpose: Sets the value of the title field from the -d UIS line
|                This routine is specialized for use with subforms, where
|                you can't tell from the kformstruct if they want to set
|                the attributes of the subform button or of the subform itself.
|                However, it will also work for any of the other button-specific
|                UIS lines (-g, -m, -n, -a, -R, -H, -Q), as it just turns
|                around and calls kvf_get_title().
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_BUTTONTITLE ("kvf_buttontitle")
|                calldata    - provides string value of title
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_buttontitle(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char          **title;
	Line_Info     line_info;
	char          *line = NULL;

        if (kformstruct->type != KSUBFORM)
            return(kvf_set_title(kformstruct, KVF_TITLE, calldata));

	/* the value to which KVF_BUTTONTITLE is to be set */
        title  = (char **) calldata;

        /*
         * Note: we are parsing the UIS line directly here,
         *       because kvf_gen_parse_kformstruct isn't smart
         *       enough to know whether you want the -M or the -d
         *       for a subform.
         */
        kvf_clear_line_info(&line_info);
        line = kformstruct->Subformptr->line;
	if (!(kvf_parse_subformbutton_line(line, &line_info)))
	    return(FALSE);

        kfree(line_info.title);
	if (kstrlen(*title) > 0)
            line_info.title = kstrdup(*title);
	else line_info.title = NULL;

        kvf_gen_deparse(&line_info, &kformstruct->Subformptr->line);
	kvf_free_line_info_strings(&line_info);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kvf_get_guidepanetitle
|       Purpose: Gets the value of the title field
|                from the StartGuide -G UIS line of a subform
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_GUIDEPANETITLE ("kvf_guidepanetitle")
|        Output: calldata    - passes back string value of guide pane title
|
|          Date: November 28, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_guidepanetitle(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	char         **title;
	kcontrol     *guidepane;

        /* set the pointer to be returned */
        title = (char **) calldata;

	/* kformstruct coming in is from the subform */
	guidepane = kformstruct->Subformptr->guidepane;
	if (guidepane == NULL)
	    return(FALSE);

	if (!(kvf_gen_parse_kformstruct(guidepane->back_kformstruct, 
					&line_info)))
	    return(FALSE);

	if (line_info.title == NULL)
	    *title = NULL;
	else *title = kstrdup(line_info.title);

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_guidepanetitle
|       Purpose: Sets the value of the title field
|                on the StartGuide -G UIS line of a subform
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_GUIDEPANETITLE ("kvf_guidepanetitle")
|                calldata    - provides string value of guide pane title
|        Output: none
|
|          Date: November 28, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_guidepanetitle(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char          **title;
	Line_Info     line_info;
	kcontrol      *guidepane;

	/* the value to which KVF_GUIDEPANETITLE is to be set */
        title  = (char **) calldata;

	/* kformstruct coming in is from the subform */
        guidepane = kformstruct->Subformptr->guidepane;
        if (guidepane == NULL)
            return(FALSE);

	if (!(kvf_gen_parse_kformstruct(guidepane->back_kformstruct, 
					&line_info)))
	    return(FALSE);

        kfree(line_info.title);
	if (kstrlen(*title) > 0)
            line_info.title = kstrdup(*title);
	else line_info.title = NULL;

        kvf_gen_deparse_kformstruct(guidepane->back_kformstruct, &line_info);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|
|       Routine: kvf_get_literal
|       Purpose: Gets the value of the literal field
|                from the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_LITERAL ("kvf_literal")
|        Output: calldata    - passes back string value of literal
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_literal(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char         **literal;
	Line_Info    line_info;

        /* set the pointer to be returned */
        literal = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (line_info.literal == NULL)
	    *literal = NULL;
	else *literal = kstrdup(line_info.literal);

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_literal
|       Purpose: Sets the value of the literal field
|                on the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_LITERAL ("kvf_literal")
|                calldata    - provides string value of literal
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_literal(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char          **literal;
	Line_Info     line_info;
	char          text_contents[KLENGTH];
	kselection    *selection;

	/* the value to which KVF_LITERAL is to be set */
        literal  = (char **) calldata;

	if (*literal != NULL)
	    ksprintf(text_contents, "%s", *literal);
	else text_contents[0] = '\0';

	selection = kformstruct->Selptr;

	switch (selection->type)
	{
 	    case KUIS_STDIN:
 	    case KUIS_STDOUT:
 	    case KUIS_INPUTFILE:
 	    case KUIS_OUTPUTFILE:
		 /* expressions don't work for filenames yet
	         if (!(kvf_set_file_from_literal(selection, text_contents)))
		     return(FALSE);
	          */
		 if (!(kvf_set_attribute(selection->back_kformstruct,
                                         KVF_FILE_NAME, text_contents)))
                      return(FALSE);
	         break;

 	    case KUIS_INTEGER:
	         if (!(kvf_set_int_from_literal(selection, text_contents)))
		     return(FALSE);
	         break;

 	    case KUIS_FLOAT:
	         if (!(kvf_set_float_from_literal(selection, text_contents)))
		     return(FALSE);
	         break;

 	    case KUIS_DOUBLE:
	         if (!(kvf_set_double_from_literal(selection, text_contents)))
		     return(FALSE);
	         break;

 	    case KUIS_STRING:
 	    case KUIS_STRINGLIST:
                 if (!(kvf_set_attribute(selection->back_kformstruct,
                                         KVF_STRING_VAL, text_contents)))
                      return(FALSE);
	         break;
	}

	/*
	 *  either way, set the literal as specified (unless it's a string,
	 *  in which case it already got done in kvf_set_string_val, or
	 *  if it's a stdin or stdout, in which case there isn't one)
	 */
	if ((selection->type != KUIS_STRING)     &&
	    (selection->type != KUIS_STRINGLIST) &&
	    (selection->type != KUIS_STDIN)      &&
	    (selection->type != KUIS_STDOUT))
	{
	    if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	        return(FALSE);

	    kfree(line_info.literal);
	    if (kstrlen(text_contents) > 0)
                line_info.literal = kstrdup(text_contents);
	    else line_info.literal = NULL;

	    kvf_gen_deparse_kformstruct(kformstruct, &line_info);
	}

	return(TRUE);
}


/*-----------------------------------------------------------
|
|
|       Routine: kvf_get_modified
|       Purpose: Gets the value of the "modified" field
|                directly from the kselection structure
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_MODIFIED ("kvf_modified")
|        Output: calldata    - passes back int value of modified
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_modified(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *modified;

        /* set the pointer to be returned */
        modified = (int *) calldata;

	*modified = kformstruct->Selptr->modified;
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_modified
|       Purpose: Sets the value of the "modified" field
|                directly from the kselection structure
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_MODIFIED ("kvf_modified")
|                calldata    - provides int value of modified
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_modified(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int           *modified;

	/* the value to which KVF_MODIFIED is to be set */
        modified  = (int *) calldata;

	if ((*modified != 0) && (*modified != 1))
	{
	    kerror("kforms", "kvf_set_modified",
		   "Cannot set 'modified' field to anything other than 0 or 1");
	    return(FALSE);
	}
	kformstruct->Selptr->modified = *modified;

	return(TRUE);
}

/*-----------------------------------------------------------
|
|
|       Routine: kvf_get_workspace
|       Purpose: Gets the value of the workspace field
|                directly from the kselection structure
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_WORKSPACE ("kvf_workspace")
|        Output: calldata    - passes back xvobject address of workspace
|
|          Date: April 7, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_workspace(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        xvobject   *workspace;
        Line_Info  line_info;

        /* set the pointer to be returned */
        workspace = (xvobject *) calldata;

        if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
            return(FALSE);

        *workspace = line_info.workspace;

        kvf_free_line_info_strings(&line_info);
        return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_workspace
|       Purpose: Sets the value of the workspace field
|                directly from the kselection structure
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_WORKSPACE ("kvf_workspace")
|                calldata    - provides xvobject address of workspace
|        Output: none
|
|          Date: April 7, 1994
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_workspace(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	xvobject  *workspace;
        Line_Info line_info;

        /* the value to which KVF_LITERAL is to be set */
        workspace  = (xvobject *) calldata;

        if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
            return(FALSE);

        line_info.workspace = *workspace;

        kvf_gen_deparse_kformstruct(kformstruct, &line_info);
        return(TRUE);
}


/*-----------------------------------------------------------
|
|
|       Routine: kvf_get_clientdata
|       Purpose: Gets the value of the client_data field
|                directly from the kformstruct
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_CLIENTDATA ("kvf_clientdata")
|        Output: calldata    - passes back address of client data
|
|          Date: October 25, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_clientdata(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	kaddr  *client_data;

        /* set the pointer to be returned */
        client_data = (kaddr *) calldata;

        *client_data = kformstruct->client_data;

        return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_clientdata
|       Purpose: Sets the value of the client_data field
|                directly in the kformstruct
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_CLIENTDATA ("kvf_clientdata")
|                calldata    - provides address of client data
|        Output: none
|
|          Date: October 25, 1994
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_clientdata(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	kaddr  *client_data;

        /* the value to which KVF_CLIENTDATA is to be set */
        client_data  = (kaddr *) calldata;

        kformstruct->client_data = *client_data;

        return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_delete
|       Purpose: Deletes a selection from the form tree
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DELETE ("kvf_delete")
|                calldata    - not used
|        Output: none
|
|          Date: July 7, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_delete(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        if (kformstruct->type == KSELECTION)
            return(kvf_delete_selection(kformstruct->Selptr));

        else if (kformstruct->type == KSUBFORM)
            return(kvf_delete_subform(kformstruct->Subformptr->back_form,
				      kformstruct->Subformptr));

        else if (kformstruct->type == KGUIDE)
            return(kvf_delete_guide(kformstruct->Guideptr));

        else if (kformstruct->type == KPANE)
            return(kvf_delete_guide(kformstruct->Controlptr->back_guide));

	else
	{
	    kerror("kforms", "kvf_set_delete",
		   "kformstruct of unsupported type passed in");
	    return(FALSE);
	}
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_display_subform
|       Purpose: This routine sets a subform to displayed or un-displayed
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DISPLAY_SUBFORM ("kvf_display_subform")
|                calldata    - not used
|        Output: none
|
|          Date: April 17, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_display_subform(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        Line_Info    line_info, subform_info;
        ksubform     *subform, *old_subform;
        int          *display;

        /* the value to which XVF_DISPLAY_SUBFORM is to be set */
        display  = (int *) calldata;

        if (kformstruct->type != KSUBFORM)
        {
            kerror("kforms", "kvf_set_display_subform",
                   "kformstruct is not of type subform");
            return(FALSE);
        }

        if ((*display != 0) && (*display != 1))
        {
            errno = KINVALID_PARAMETER;
            kerror("kforms", "kvf_set_display_subform",
                   "You must provide value of either '1' or '0'");
            return(FALSE);
        }

        /*
         * if we are turning on new subform, and we have a
         * mutually exclusive form, we must first turn off the old subform
         */
        subform = kformstruct->Subformptr;
        if (subform->selected == *display)
            return(TRUE);

        kvf_clear_line_info(&line_info);
        if ((subform->back_form->master != NULL) && (*display) &&
            (subform->back_form->master->excl_subforms == 1))
        {
            old_subform = subform->back_form->master->subform_list;
            while (old_subform != NULL)
            {
                kvf_clear_line_info(&subform_info);
                if (old_subform->selected)
                {
                    old_subform->selected = FALSE;
                    kvf_parse_startsubform_line(old_subform->control_line,
                                                &subform_info);
                    subform_info.selected = FALSE;
                    kvf_deparse_startsubform_line(&subform_info,
                                                  &old_subform->control_line);
                    kvf_parse_subformbutton_line(old_subform->line,  
						 &subform_info);
                    subform_info.selected = FALSE;
                    kvf_deparse_subformbutton_line(&subform_info,
                                                   &old_subform->line);
                }
                old_subform = old_subform->next;
            }
        }

        /*
         * next, turn on subform on (or off)
         */
        subform->selected = *display;

        kvf_parse_startsubform_line(subform->control_line, &line_info);
        line_info.selected = *display;
        kvf_deparse_startsubform_line(&line_info, &subform->control_line);

        kvf_parse_subformbutton_line(subform->line, &line_info);
        line_info.selected = *display;
        kvf_deparse_subformbutton_line(&line_info, &subform->line);

        kvf_free_line_info_strings(&line_info);
        return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_display_pane
|       Purpose: This routine sets a pane to be displayed on a subform
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_DISPLAY_PANE ("kvf_display_pane")
|                calldata    - not used
|        Output: none
|
|          Date: April 18, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_display_pane(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info, guide_info;
        kguide       *guide = NULL;
        int          *display;

        /* the value to which XVF_DISPLAY_PANE is to be set */
        display  = (int *) calldata;

        if ((kformstruct->flag != KUIS_STARTPANE) &&
            (kformstruct->flag != KUIS_GUIDEBUTTON))
        {
            kerror("kforms", "xvf_set_display_pane",
                   "kformstruct is not of type pane or guide button");
            return(FALSE);
        }
        if ((*display != 0) && (*display != 1))
        {
            errno = KINVALID_PARAMETER;
            kerror("kforms", "xvf_set_display_pane",
                   "You must provide value of either 1 or 0");
            return(FALSE);
        }

        kvf_clear_line_info(&line_info);

        /*
         * if we are turning on new guide button,
         * we must first turn off the old one
         */
        if (*display)
        {
            if (kformstruct->flag == KUIS_STARTPANE)
                guide =
                  kformstruct->Controlptr->back_subform->guidepane->guide_list;
            else if (kformstruct->flag == KUIS_GUIDEBUTTON)
                guide =
                   kformstruct->Guideptr->back_subform->guidepane->guide_list;

            while (guide != NULL)
            {
                kvf_clear_line_info(&guide_info);
                if ((guide->selected == TRUE) && (guide->pane != NULL))
                {
                    guide->selected = FALSE;
                    kvf_parse_guide_line(guide->line, &guide_info);
                    guide_info.selected = FALSE;
                    kvf_deparse_guide_line(&guide_info, &guide->line);

                    kvf_parse_startpane_line(guide->pane->control_line,
                                             &guide_info);
                    guide_info.selected = FALSE;
                    kvf_deparse_startpane_line(&guide_info,
                                               &guide->pane->control_line);
                }
                guide = guide->next;
            }
        }

        /*
         * find the guide button in question
         */
        if (kformstruct->flag == KUIS_STARTPANE)
            guide = kformstruct->Controlptr->back_guide;

        else if (kformstruct->flag == KUIS_GUIDEBUTTON)
            guide = kformstruct->Guideptr;

        guide->selected = *display;
        kvf_parse_guide_line(guide->line, &line_info);
        line_info.selected = *display;
        kvf_deparse_guide_line(&line_info, &guide->line);

        kvf_parse_startpane_line(guide->control_line, &line_info);
        line_info.selected = *display;
        kvf_deparse_startpane_line(&line_info, &guide->control_line);

        kvf_free_line_info_strings(&line_info);
        return(TRUE);
}
