 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Private Khoros Expression Evaluation Routines
   >>>>
   >>>>  Private:
   >>>>             eval_function()
   >>>>             eval_ufunction()
   >>>>             eval_operator()
   >>>>             eval_instruction()
   >>>>             eval_symlist()
   >>>>
   >>>>		   kexpr_eval_symbol()
   >>>>		   kexpr_eval_symlist()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"
#include "y.tab.h"


static SymbolList *eval_symlist PROTO((SymbolList *, int, Symbol *result));


/*-----------------------------------------------------------
|
|  Routine Name: eval_function - evaluate a function
|
|       Purpose: eval_function() is called when we want to evaluate
|                a function.
|
|        Input:  symbol   - the function symbol
|		 symlist  - the argument symbols for the function
|		 abort_when_undefined - whether to abort when an undefined
|					variable is found.
|
|        Output: symbol   - the result stored in the "result" symbol
|		 RETURNS: the symbol list on success, NULL otherwise
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
------------------------------------------------------------*/


static SymbolList *eval_function(
   Symbol     *symbol,
   SymbolList *symlist,
   int        abort_when_undefined,
   Symbol     *result)
{
	Symbol   arg;
	double   vals[20];
	int	 i, num_args;
	register double	value;
	double   (*function)();


	/*
	 *  Evaluate the different argument expressions and then call the
	 *  function with the different evaluated arguments.
	 */
	num_args = symbol->num;
	if (result == NULL)
	{
	   for (i = 0; i < num_args; i++)
	      symlist = eval_symlist(symlist, abort_when_undefined, NULL);

	   return(symlist);
	}

	function = symbol->Function;
	for (i = 0; i < num_args; i++)
	{
	   symlist = eval_symlist(symlist, abort_when_undefined, &arg);
	   vals[i] = arg.Value;
	}

	switch (num_args)
	{
	   case 0:
		value = function();
		break;

	   case 1:
		value = function(vals[0]);
		break;

	   case 2:
		value = function(vals[0], vals[1]);
		break;

	   case 3:
		value = function(vals[0], vals[1], vals[2]);
		break;

	   case 4:
		value = function(vals[0], vals[1], vals[2], vals[3]);
		break;

	   case 5:
		value = function(vals[0], vals[1], vals[2], vals[3], vals[4]);
		break;

	   case 6:
		value = function(vals[0], vals[1], vals[2], vals[3], vals[4],
				vals[5]);
		break;

	   case 7:
		value = function(vals[0], vals[1], vals[2], vals[3], vals[4],
				vals[5], vals[6]);
		break;

	   case 8:
		value = function(vals[0], vals[1], vals[2], vals[3], vals[4],
				vals[5], vals[6], vals[7]);
		break;

	   case 9:
		value = function(vals[0], vals[1], vals[2], vals[3], vals[4],
				vals[5], vals[6], vals[7], vals[8]);
		break;

	   default:
		value = 0.0;
	        break;
	}
	result->type  = NUMBER;
	result->Value = value;
	return(symlist);
}

/*-----------------------------------------------------------
|
|  Routine Name: eval_ufunction - evaluate user defined function
|
|       Purpose: eval_ufunction() is called when we want to evaluate
|		 a user defined function.
|
|        Input:  symbol   - the function symbol
|		 symlist  - the argument symbols for the function
|		 abort_when_undefined - whether to abort when an undefined
|					variable is found.
|
|        Output: symbol   - the result stored in the "result" symbol
|		 RETURNS: the symbol list on success, NULL otherwise
|
|    Written By: Mark Young  
|          Date: 6/25/92
| Modifications:
|
------------------------------------------------------------*/


static SymbolList *eval_ufunction(
   Symbol     *symbol,
   SymbolList *symlist,
   int        abort_when_undefined,
   Symbol     *result)
{
	double   values[100];
	SymbolList *arglist;
	Symbol   *arg, arg1;
	register int i, num_args;


	/*
	 *  Evaluate the different argument expressions and then call the
	 *  eval_symlist() with the different evaluated arguments, but we
	 *  first need to save the argument values so that if we act just
	 *  like a stack would.
	 */
	num_args = symbol->num;
	if (result == NULL)
	{
	   for (i = 0; i < num_args; i++)
	      symlist = eval_symlist(symlist, abort_when_undefined, &arg1);

	   return(symlist);
	}

	arglist  = symbol->symlist;
	for (i = 0; i < num_args; i++)
	{
	   arg = arglist->symbol;
	   values[i] = arg->Value;
	   if (symlist->symbol->type == NUMBER)
           {
              arg->Value = symlist->symbol->Value;
              symlist = symlist->next;
           }
	   else
	   {
	      symlist = eval_symlist(symlist, abort_when_undefined, &arg1);
	      arg->Value = arg1.Value;
	   }
	   arglist = arglist->next;
	}
	(void) eval_symlist(arglist, abort_when_undefined, result);

	arglist = symbol->symlist;
	for (i = 0; i < num_args; i++)
	{
	   arg = arglist->symbol;
	   arg->Value = values[i];
	   arglist = arglist->next;
	}
	return(symlist);
}

/*-----------------------------------------------------------
|
|  Routine Name: eval_operator - evaluate an operator
|
|       Purpose: eval_operator() is called when we want to evaluate
|		 an operation.
|
|        Input:  symbol   - the operator symbol
|		 symlist  - the argument symbols for the operator
|		 abort_when_undefined - whether to abort when an undefined
|					variable is found.
|
|        Output: symbol   - the result stored in the "result" symbol
|		 RETURNS: the symbol list on success, NULL otherwise
|
|    Written By: Mark Young  
|          Date: 6/25/92
| Modifications:
|
------------------------------------------------------------*/

#ifdef CRAY
#define vpow(x,y) ((val1) < 0.0 && (val2) == ((int) (val2)))            ? \
        (((((int) (val2)) % 2 == 0) ? 1 : -1) * pow(fabs(val1),val2))   : \
        pow(val1, val2)
#else
#define vpow pow
#endif


static SymbolList *eval_operator(
   Symbol     *symbol,
   SymbolList *symlist,
   int        abort_when_undefined,
   Symbol     *result)
{
	Symbol	 arg1, arg2;
	register double	value, val1, val2;
	char	 str1[2*KLENGTH], str2[2*KLENGTH];


	if (result == NULL)
	{
	   symlist = eval_symlist(symlist, abort_when_undefined, NULL);
	   if (symbol->type == OPERATOR)
	      symlist = eval_symlist(symlist, abort_when_undefined, NULL);

	   return(symlist);
	}

	symlist = eval_symlist(symlist, abort_when_undefined, &arg1);
	val1 = arg1.Value;

	if (symbol->type == OPERATOR)
	{
	   symlist = eval_symlist(symlist, abort_when_undefined, &arg2);
	   val2 = arg2.Value;
	}
	else
	   val2 = 0.0;

	switch (symbol->Operator)
	{
	   case AND:
		value = ((val1 && val2) ? TRUE : FALSE);
		break;

	   case OR:
		value = ((val1 || val2) ? TRUE : FALSE);
		break;

	   case EQ:
		value = ((val1 == val2) ? TRUE : FALSE);
		break;

	   case NE:
		value = ((val1 != val2) ? TRUE : FALSE);
		break;

	   case LE:
		value = ((val1 <= val2) ? TRUE : FALSE);
		break;

	   case GE:
		value = ((val1 >= val2) ? TRUE : FALSE);
		break;

	   case LT:
		value = ((val1 < val2) ? TRUE : FALSE);
		break;

	   case GT:
		value = ((val1 > val2) ? TRUE : FALSE);
		break;

	   case '+':
		value = val1+val2;
		break;

	   case '-':
		value = val1 - val2;
		break;

	   case '*':
		value = val1*val2;
		break;

	   case '/':
		if (abort_when_undefined == TRUE)
	        {
		   if (val2 == 0.0)
		   {
		      kexpr_error("Floating divide by zero");
		      return(NULL);
		   }
		   value = val1/val2;
	        }
		else
		   value = (val2 != 0.0) ? val1/val2 : val1;
		break;

	   case POW:
		value = vpow(val1, val2);
		break;

	   case '%':
		value = ((((int) val2) != 0) ? ((int) val1) % ((int) val2) : 0);
		break;

	   case '&':
		value = ((int) val1) & ((int) val2);
		break;

	   case '|':
		value = ((int) val1) | ((int) val2);
		break;

	   case '^':
		value = ((int) val1) ^ ((int) val2);
		break;

	   case SL:
		value = ((int) val1) << ((int) val2);
		break;

	   case SR:
		value = ((int) val1) >> ((int) val2);
		break;

	   case '~':
		value = ~((int) val1);
		break;

	   case '!':
		value = !val1;
		break;

	   case UMINUS:
		value = -val1;
		break;

	   case CINT:
		value = (int) val1;
		break;

	   case CFLOAT:
		value = (float) val1;
		break;

	   case CSTRING:
		if (arg1.type == NUMBER)
		   (void) sprintf(str1,"%g", arg1.Value);
		else if (arg1.type == STRING)
		   (void) sprintf(str1,"%s", arg1.String);
		else
		   (void) sprintf(str1,"");

		result->type   = STRING;
		result->String = kstring_copy(str1, NULL);
		return(symlist);

	   case STRCAT:
		if (arg1.type == NUMBER)
		   (void) sprintf(str1,"%g", arg1.Value);
		else if (arg1.type == STRING)
		   (void) sprintf(str1,"%s", arg1.String);
		else
		   (void) kstrcpy(str1,"");

		if (arg2.type == NUMBER)
		   (void) sprintf(str2,"%g", arg2.Value);
		else if (arg2.type == STRING)
		   (void) sprintf(str2,"%s", arg2.String);
		else
		   (void) kstrcpy(str2,"");

		result->type   = STRING;
		result->String = kstring_cat(str1, str2, NULL);
		return(symlist);

	   default:
		value = 0.0;
		break;
	}
	result->type  = NUMBER;
	result->Value = value;
	return(symlist);
}

/*-----------------------------------------------------------
|
|  Routine Name: eval_instruction - evaluate an instruction
|
|       Purpose: eval_instruction() is called when we want to evaluate
|		 an instruction.
|
|        Input:  symbol   - the instruction symbol
|		 symlist  - the argument symbols for the instruction
|		 abort_when_undefined - whether to abort when an undefined
|					variable is found.
|
|        Output: symbol   - the result stored in the "result" symbol
|		 RETURNS: the symbol list on success, NULL otherwise
|
|    Written By: Mark Young  
|          Date: 6/25/92
| Modifications:
|
------------------------------------------------------------*/


static SymbolList *eval_instruction(
   Symbol     *symbol,
   SymbolList *symlist,
   int        abort_when_undefined,
   Symbol     *result)
{
	Symbol	arg;


	if (result == NULL)
	{
	   if (symbol->Instruction == IF)
	   {
	      symlist = eval_symlist(symlist, abort_when_undefined, NULL);
	      symlist = eval_symlist(symlist, abort_when_undefined, NULL);
	      symlist = eval_symlist(symlist, abort_when_undefined, NULL);
	   }
	   return(symlist);
	}

	switch (symbol->Instruction)
	{
	    case IF:
		 symlist = eval_symlist(symlist, abort_when_undefined, &arg);
		 if (arg.Value)
		 {
		    symlist = eval_symlist(symlist, abort_when_undefined, &arg);
		    symlist = eval_symlist(symlist, abort_when_undefined, NULL);
	         }
		 else
		 {
		    symlist = eval_symlist(symlist, abort_when_undefined, NULL);
		    symlist = eval_symlist(symlist, abort_when_undefined, &arg);
		 }
		 break;
	}

	result->type = arg.type;
	if (arg.type == NUMBER)
	   result->Value = arg.Value;
	else
	   result->String = kstring_copy(arg.String, NULL);
	return(symlist);
}

/*-----------------------------------------------------------
|
|  Routine Name: eval_symlist - evaluate a symbol list
|
|       Purpose: eval_symlist() is called when we want to evaluate
|		 a symbol list.
|
|        Input:  symlist  - the current symbol list to be evaluated
|		 abort_when_undefined - whether to abort when an undefined
|					variable is found.
|
|        Output: symbol   - the result stored in the "result" symbol
|		 RETURNS: the symbol list on success, NULL otherwise
|
|    Written By: Mark Young  
|          Date: 6/25/92
| Modifications: changed parameter list so that "result" was last in the
|		 list - (6/25/92 MY)
|
------------------------------------------------------------*/


static SymbolList *eval_symlist(
   SymbolList *symlist,
   int        abort_when_undefined,
   Symbol     *result)
{
	Symbol	*symbol;
	char	error[2*KLENGTH];


	/*
	 *  If the current symlist is NULL then the endlist will become
	 *  the beginning of the symbol list, so we just return it.
	 */
	if (symlist == NULL)
	{
	   sprintf(error,"Error! end of symbol list encountered, but more \
expected.");
	   kexpr_error(error);
	   return(NULL);
	}

	symbol = symlist->symbol;
	switch (symbol->type)
	{
	   case NUMBER:
	   case CONSTANT:
	   case EXPRESSION:
		if (result != NULL)
		{
		   result->type  = NUMBER;
		   result->Value = symbol->Value;
		}
		symlist = symlist->next;
		break;

	   case STRING:
		if (result != NULL)
		{
		   result->type  = STRING;
		   result->String = kstring_copy(symbol->String, NULL);
		}
		symlist = symlist->next;
		break;

	   case VARIABLE:
		if ((unsigned int) symbol->eval == FALSE)
		{
		   symbol->eval = TRUE;
		   (void) eval_symlist(symbol->symlist, abort_when_undefined, result);
		   symbol->eval = FALSE;
		}
		else if (result != NULL)
		{
		   result->type  = NUMBER;
		   result->Value = symbol->Value;
		}
		symlist = symlist->next;
		break;

	   case FUNCTION:
		symlist = symlist->next;
		symlist = eval_function(symbol, symlist, abort_when_undefined, result);
		break;

	   case UFUNCTION:
		symlist = symlist->next;
		symlist = eval_ufunction(symbol, symlist, abort_when_undefined, result);
		break;

	   case UOPERATOR:
	   case OPERATOR:
		symlist = symlist->next;
		symlist = eval_operator(symbol, symlist, abort_when_undefined, result);
		break;

	   case INSTRUCTION:
		symlist = symlist->next;
		symlist = eval_instruction(symbol, symlist, abort_when_undefined, result);
	        break;

	   case UNDEFINED:
		if (abort_when_undefined)
		{
		   sprintf(error,"Error! Unknown variable or constant '%s'",
				symbol->name);
		   kexpr_error(error);
		   return(NULL);
		}
		symlist = symlist->next;

		if (result != NULL)
		{
		   result->type  = UNDEFINED;
		   result->Value = 0.0;
		}
		break;

	   default:
		(void) sprintf(error,"Error!  Unknown symbol type '%d'",
				symbol->type);
		kexpr_error(error);
		return(NULL);
	}
	return(symlist);
}

/*-----------------------------------------------------------
|
|  Routine Name: kexpr_eval_symbol - evaluate a single symbol
|
|       Purpose: kexpr_eval_symbol() is called when we want to evaluate
|		 a symbol.
|
|        Input:  symbol  - the current symbol to be evaluated
|		 abort_when_undefined - whether to abort when an undefined
|					variable is found.
|
|        Output: symbol   - the result stored in the "result" symbol
|                RETURNS: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young  
|          Date: 6/25/92
| Modifications: changed parameter list so that "result" was last in the
|		 list - (6/25/92 MY)
|
------------------------------------------------------------*/


int kexpr_eval_symbol(
   Symbol *symbol,
   int    abort_when_undefined,
   Symbol *result)
{
	int	   i;
	Symbol	   temp_symbol;
	SymbolList list, *symlist;
	char       error[2*KLENGTH];


	/*
	 *  If the current symbol is NULL then print out an error
	 *  and abort.
	 */
	error_found = FALSE;
	if (symbol == NULL)
	{
	   sprintf(error,"Error! NULL symbol list encountered.\n");
	   kexpr_error(error);
	   return(FALSE);
	}
	else if (result == NULL)
	   result = &temp_symbol;

 	if (symbol->type == VARIABLE)
	{
	   symbol->eval = TRUE;
	   eval_symlist(symbol->symlist, abort_when_undefined, result);
	   symbol->Value = result->Value;
	   symbol->eval = FALSE;
	}
	else if (symbol->type == EXPRESSION)
	{
	   eval_symlist(symbol->symlist, abort_when_undefined, result);
	   symbol->Value = result->Value;
	}
	else if (symbol->type == UFUNCTION)
	{
	   symlist = symbol->symlist;
	   for (i = 0; i < (unsigned int)symbol->num; i++)
	      symlist = symlist->next;

	   (void) eval_symlist(symlist, abort_when_undefined, result);
	   symbol->Value = result->Value;
	}
	else
	{
	   list.symbol = symbol;
	   list.next   = NULL;
	   (void) eval_symlist(&list, abort_when_undefined, result);
	}
	return(error_found == FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kexpr_eval_symlist - evaluate a symbol list
|
|       Purpose: kexpr_eval_symlist() is called when we want to evaluate
|		 a list of symbols.
|
|        Input:  symlist  - the current symbol to be evaluated
|		 abort_when_undefined - whether to abort when an undefined
|					variable is found.
|
|        Output: symbol   - the result stored in the "result" symbol
|                RETURNS: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young  
|          Date: 6/25/92
| Modifications: changed parameter list so that "result" was last in the
|		 list - (6/25/92 MY)
|
------------------------------------------------------------*/


int kexpr_eval_symlist(
   SymbolList *symlist,
   int        abort_when_undefined,
   Symbol     *result)
{
	char	   error[2*KLENGTH];


	/*
	 *  If the current symlist is NULL then the endlist will become
	 *  the beginning of the symbol list, so we just return it.
	 */
	error_found = FALSE;
	if (symlist == NULL)
	{
	   sprintf(error,"Error! NULL symbol list encountered.\n");
	   kexpr_error(error);
	   return(FALSE);
	}

	if (result == NULL)
	{
	   sprintf(error,"Error! NULL result symbol encountered.\n");
	   kexpr_error(error);
	   return(FALSE);
	}
	(void) eval_symlist(symlist, abort_when_undefined, result);
	return(error_found == FALSE);
}
