/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>        Routines to generate GUI initialization code   <<<<
   >>>>             for xvroutines in form_init.c             <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>              kgen_gui_init_forminfo()                 <<<<
   >>>>              kgen_gui_init_subforminfo()              <<<<
   >>>>   Static:                                             <<<<
   >>>>              gen_init_selections()                    <<<<
   >>>>                                                       <<<<
   >>>>              print_if_protection()                    <<<<
   >>>>              print_else_protection()                  <<<<
   >>>>              print_sel_selected()                     <<<<
   >>>>                                                       <<<<
   >>>>              sel_var()                                <<<<
   >>>>              sel_desc()                               <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"  

static int 
gen_init_selections   PROTO((kfile *, kfile *, int, char *, char *,
                             int, char *, kselection *, int *));
static void 
print_sel_selected    PROTO((kfile *, int, int, int, char  *));

static void 
print_if_protection   PROTO((kfile *, char *));

static void 
print_else_protection PROTO((kfile *, int, int, char  *,
                             char  *, char  *, char  *));
static char *
sel_var               PROTO((int, char *, int, char *, char *));

static char *
sel_desc              PROTO((int, int, char *, char *, char *));



/*-----------------------------------------------------------
|
|  Routine Name: kgen_gui_init_forminfo
|
|       Purpose: This routine creates a file called "form_init.c" 
|		 in the local directory, and automatically generates in
|		 this file the routine "_xvf_init_XXX", where XXX is the
|		 variable program_name given on the -F UIS line.  
|                This "_xvf_init_XXX" routine will be used to initialize 
|                the GUI Information structure.
|
|         Input: form         - pointer to the form tree 
|                program      - program object being generated
|                program_name - name of the program being generated
|
|        Output: Returns TRUE if form_init.c was created successfully,
|                FALSE otherwise
|          Date: May 11, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/

int kgen_gui_init_forminfo(
   kform   *form,
   kobject program,
   char    *program_name)
{
        kstring    routine = "kgen_gui_init_forminfo()";
	kstring    oname   = NULL;
	int        selected, user_refused, dummy; 
	kobject    toolbox;
	char       *form_var, *subform_var, *copyright, *fullname;
	char       func_name[KLENGTH], purpose[KLENGTH]; 
	char       input[KLENGTH], temp[KLENGTH];  
 	kfile      *file, *free_file; 
	ksubform   *subform;
	kselection *selection;


	/*-- get the attributes we need from the software object --*/
	if (!kcms_get_attributes(program,
				 KCMS_NAME,   &oname,
				 KCMS_PARENT, &toolbox,
				 KCMS_END))
	   return(FALSE);

        /*
	 *  first of all, if 'form_init.c' exists, see if they want to over-
	 *  write it.  If not, return immediately.
	 */
	file = kgen_open_file(program, "form_init.c", &user_refused);
        if (user_refused) return(TRUE);
        else if (file == NULL) 
	{
	    kerror(KCODEGEN, "kgen_gui_init_forminfo", 
		   "Could not create file 'form_init.c'");
	    return(FALSE);
	}

	ksprintf(temp, "%s/free_form_init", kgen_tmpdirname);
	if ((free_file = kfopen(temp, "w")) == NULL)
	{
	    kerror(KCODEGEN, "kgen_gui_init_forminfo", 
		   "Could not create tmp file 'free_form_init'");
	    return(FALSE);
	}

	/* begin file with RCS header */
	kgen_rcs_chdr(file);

	/*
	 * generate the copyright
	 */
	if ((copyright = kcms_get_copyright(toolbox, KCMS_TB_COPYRIGHT_LONG,
					  KCMS_LANG_C)) != NULL)
	{
	   kfprintf(file, "%s\n", copyright);
	   kfree(copyright);
	}


	/* read the StartForm definition & get FormInfo variable program_name */
        form_var = ktoken_to_string(form->var_token);

	/* generate file header for form_init.c */
	kgen_create_init_srcfile_hdr(file, form_var);

	/* write to file the necessary #includes */
	kfprintf(file,"#include \"%s.h\"\n\n", oname);

	/* 
	 * write out static routine declarations
	 */
	ksprintf(func_name, "_xvf_init_%s_warning PROTO((char *))", 
		 form_var);
	kfprintf(file, "static void %s;\n", func_name);
	ksprintf(func_name, "_xvf_init_%s_error   PROTO((char *, char *))", form_var);
	kfprintf(file, "static void %s;\n\n", func_name);

       	/* 
	 * generate private header  for _xvf_init_{forminfo}
	 */
	ksprintf(func_name, "_xvf_init_%s", form_var);
	ksprintf(input, "forminfo - ptr to %s_gui_struct\n|                form     - pointer to the form tree", func_name);
	ksprintf(purpose, "Initializes the GUI Information (gui_info_struct) structure");
	kgen_privsrc_hdr(file, func_name, purpose, 
			 input, "none", "conductor", " ");
       
        
        /* write out program program_name & argument list */
        kfprintf(file, "int _xvf_init_%s(\n\n", form_var); 
        kfprintf(file, "    kform *form, \n");
        kfprintf(file, "    gui_info_struct *forminfo)\n");

        /*
         * also, start generating code to free the FormInfo structure
         * in a different file that will be catted onto form_init.c
         */
        ksprintf(func_name, "_xvf_free_%s", form_var);
        ksprintf(input, "forminfo - ptr to %s_gui_struct", func_name);
        ksprintf(purpose, "Frees the gui_info_struct");
        kgen_privsrc_hdr(free_file, func_name, purpose,
                         input, "none", "conductor", " ");

        kfprintf(free_file, "void _xvf_free_%s(\n",form_var); 
        kfprintf(free_file, "    gui_info_struct *forminfo)\n{\n");
       
       	/* write out beginning bracket & declarations */
       	kfprintf(file, "{\n"); 
       	kfprintf(file, "\tksubform   *subform;\n");
       	kfprintf(file, "\tkcontrol   *pane;\n");
       	kfprintf(file, "\tkselection *selection;\n");
	
	if (form->master != NULL)
        {
            subform    = form->master->subform_list;
            selection  = form->master->sel_list;
       	    kfprintf(file, "\tkcontrol   *master;\n");
        }
        else
        {
            subform   = form->subform;
            selection = NULL;
        }
	while (subform != NULL)
	{
	    if (subform->guidepane != NULL)
	    {
       	        kfprintf(file, "\tkcontrol   *guidepane;\n");
		break;
	    }
	    subform = subform->next;
	}
	if (form->master != NULL)
	    subform = form->master->subform_list;
	else subform = form->subform;

	if (form->master != NULL)
	    kfprintf(file, "\n\tmaster = form->master;\n");

	kfprintf(file, "\t/*\n\t * initializing '%s' Form ", form_var);
	kfprintf(file, "Info structure\n\t */\n\n");

	/*
	 *  initialize selections on master
	 */
	if (!(gen_init_selections(file, free_file, KMASTER, form_var, 
				  NULL, 0, NULL, selection, &dummy)))
	    return(FALSE);

	/*
	 *  initialize subform buttons
	 */
	while (subform != NULL)
	{
	    if (subform->type == KUIS_SUBFORMBUTTON ) 
	    {
		subform_var = ktoken_to_string(subform->var_token);
		fullname = ktoken_to_string(subform->name_token);

		kvf_get_attribute(subform->back_kformstruct,
				  KVF_SELECTED, &selected);
                kgen_gui_print_create_subform_struct(file, fullname, 
			                        form_var, subform_var);

	        kfprintf(file, "\n\t/*\n\t * initializing '%s->%s'", 
				  form_var, subform_var);
	        kfprintf(file, " Subform Info structure\n\t */\n\n");
	        kfprintf(file, "\tforminfo->%s = (%s_%s *)\n", 
				  subform_var, form_var, subform_var);
		kfprintf(file, "\t\t\t\tkcalloc(1, sizeof(%s_%s));\n",
				   form_var, subform_var);
		kfprintf(file, "\tsubform = forminfo->%s_struct->Subformptr;\n",
			 subform_var);
		if (subform->guidepane != NULL)
	            kfprintf(file, "\tguidepane = subform->guidepane;\n", 
			     subform_var);

		if (!(kgen_gui_init_subforminfo(file, free_file, subform, 
		 	                           form_var, subform_var)))
		    return(FALSE);

		kfprintf(free_file,"\n\tif (forminfo->%s_struct != NULL)\n",
			 subform_var);
		kfprintf(free_file,"\t    kfree(forminfo->%s_struct);\n", 
			 subform_var);

		kfprintf(free_file, "\tkfree(forminfo->%s);\n", 
			 subform_var);
	    } 
	    subform = subform->next;
	} 

	if (form->master != NULL)
	{
	    kfprintf(file, "\tforminfo->%s_struct =\n", form_var);
            kfprintf(file, "\t    kvf_create_struct_from_mastername(form, ");
            kfprintf(file, "\"%s\");\n", form_var);
            kfprintf(file, "\tif (forminfo->%s_struct == NULL)\n\t{\n", 
		     form_var)
;
            kfprintf(file, "\t    _xvf_init_%s_error(\"form '%s'\",\n\t\t\t    \"gui_info\");\n",
                        form_var, form_var);
            kfprintf(file, "\t    return(FALSE);\n");
            kfprintf(file, "\t}\n");
	}

        kfprintf(file, "\n\treturn(TRUE);\n");
       	kfprintf(file, "\n}\n\n");

	kfprintf(free_file, "\tkfree(forminfo);\n}\n");
	kfclose(free_file);



	kgen_gui_init_warning(file, form_var);
	kgen_gui_init_error(file, form_var);
       	kfclose(file);

	ksprintf(temp,  "%s/form_init.c", kgen_fulltopsrc);

#if 0
	/*-- seems to be a no-op - neilb 22-nov-94 --*/
	if (kaccess(temp, R_OK) != 0) kgen_newfile = TRUE;
#endif

	ksprintf(temp, "cat %s/free_form_init >> %s/form_init.c",
		kgen_tmpdirname, kgen_fulltopsrc);
	ksystem(temp);

        kannounce(KCODEGEN, routine, "done generating 'form_init.c'");

       	return(TRUE);
        
} /* end kgen_gui_init_forminfo */


/*-----------------------------------------------------------
|
|  Routine Name: kgen_gui_init_subforminfo
|
|       Purpose: Generates the initialization code for each subform
|
|  	  Input: file        - open stream to "form_init.c" file
|                free_file   - open stream to "free_form_init" file
|		 subform     - pointer to the subform
|		 form_var    - program_name of the form variable
|		 subform_var - program_name of the subform variable
|
|        Output: Returns TRUE if subform initialization code
|                was created successfully, FALSE otherwise
|          Date: May 11, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/

int kgen_gui_init_subforminfo(
   kfile        *file,
   kfile        *free_file,
   ksubform *subform,
   char         *form_var,
   char         *subform_var)
{
	char       *pane_var, *fullname;
	kguide     *guide;
	kselection *selection;
	int        subform_created = FALSE;

	if (subform->guidepane != NULL)
        {
            guide = subform->guidepane->guide_list;
            selection = subform->guidepane->sel_list;
        }
        else
        {
            guide = subform->guide;
            selection = NULL;
        }
	/*
	 *  initialize selections on guide pane
	 */
	if (!(gen_init_selections(file, free_file, KGUIDEPANE, form_var, 
				  subform_var, subform->type, NULL, 
				  selection, &subform_created)))
	    return(FALSE);

	while (guide != NULL)
       	{
	    pane_var = ktoken_to_string(guide->pane->var_token);
	    fullname = ktoken_to_string(guide->pane->name_token);

	   /*
	    *  generate code to initialize PaneInfo struct only if it's
            *  definition was generated in form_info.h
	    */
	    if  (((guide->pane != NULL) &&
                 (guide->pane->sel_list != NULL)) &&
                 (kgen_valid_selections(guide->pane->sel_list)))
            {
	        kfprintf(file, "\n\t/*\n\t * initializing '%s->%s->%s'",
			 form_var, subform_var, pane_var);
	        kfprintf(file, " Pane Info structure\n\t */\n\n");
	        if (subform->type == KUIS_SUBFORMBUTTON)
	            kfprintf(file, "\tforminfo->%s->%s = (%s_%s *)\n", 
			     subform_var, pane_var, subform_var, pane_var);
	        kfprintf(file, "\t\t\t\tkcalloc(1, sizeof(%s_%s));\n", 
			 subform_var, pane_var);
	    }

	    /*
	     *  need to generate kformstruct for pane, regardless.
	     */
	    kgen_gui_print_create_pane_struct(file, fullname, form_var, 
				      subform_var, subform->type, pane_var);
	    if (!(gen_init_selections(file, free_file, KPANE, form_var,
		 	              subform_var, subform->type, pane_var, 
			              guide->pane->sel_list, &subform_created)))
		return(FALSE);

	    if (subform->type == KUIS_SUBFORMBUTTON)
	    {
	        /* free PaneInfo struct only if it was declared */
	        if ((guide->pane->sel_list != NULL) &&
                    (kgen_valid_selections(guide->pane->sel_list)))
	  	    kfprintf(free_file, "\tkfree(forminfo->%s->%s);\n",
                             subform_var, pane_var);

	        /* free kformstruct for pane, regardless */
                kfprintf(free_file, "\tif (forminfo->%s->%s_struct != NULL)\n",
                         subform_var, pane_var);
                kfprintf(free_file, "\t    kfree(forminfo->%s->%s_struct);\n\n",
                         subform_var, pane_var);
	    }
	    guide = guide->next;
        }
	return(TRUE);
	

} /* end kgen_gui_init_subforminfo */


/*-----------------------------------------------------------
|
|  Routine Name: gen_init_selections
|
|       Purpose: Generates the initialization code for each seelction
|
| 	  Input: file        - open stream to "form_init.c" file
|                file        - open stream to "free_form_init" file
|		 location    - KMASTER, KGUIDEPANE, or KPANE
|		 subform_var - program_name of the subform variable (on -M line)
|                subform_type- type of subform (if location != KMASTER)
|		 pane_var    - program_name of the pane variable (on -P line)
|                selection   - header of the list of selections
|                subform_created - is set to TRUE if call to xvf_create_subform
|                                  gets generated for this subform (if applic.) 
|
|        Output: Returns TRUE if pane initialization code was created 
|                successfully, FALSE otherwise
|    Written By: Danielle Argiro
|          Date: May 11, 1992
| Modifications:
|
-----------------------------------------------------------*/

static int gen_init_selections(
   kfile         *file,
   kfile         *free_file,
   int           location,
   char          *form_var,
   char          *subform_var,
   int           subform_type,
   char          *pane_var,
   kselection *selection,
   int           *subform_created)
{
	int        live, optional, optsel, toggle_type; 
	int        doubleclick, pane_created = FALSE;
	char       *variable, *sel_variable, *desc, *fullname, *toggle_val;

	while (selection != NULL)
	{
	   fullname = ktoken_to_string(selection->name_token);

       	   switch(selection->type) {
      
     	   case KUIS_INPUTFILE:
		variable = ktoken_to_string(selection->var_token);
		kvf_get_attributes(selection->back_kformstruct,
				   KVF_LIVE,     &live,
				   KVF_OPTIONAL, &optional,
				   KVF_OPTSEL,   &optsel,
				   NULL);
		sel_variable = sel_var(location, subform_var, subform_type,
				       pane_var, variable);
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
		print_if_protection(file, sel_variable);
		kfprintf(file, "\t    kvf_get_attribute(%s_struct, KVF_FILE_DEF,\n", sel_variable);
		kfprintf(file, "\t                    &(%s));\n", sel_variable);
	        print_else_protection(file, selection->type, location, 
			form_var, subform_var, pane_var, variable);

		print_sel_selected(file, live, optional, optsel, sel_variable);
	        kfprintf(free_file, "\tkfree(%s);\n", sel_variable);
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
	        kfree(sel_variable);
        	break;
        
           case KUIS_OUTPUTFILE:
		variable = ktoken_to_string(selection->var_token);
		kvf_get_attributes(selection->back_kformstruct,
				   KVF_LIVE,     &live,
				   KVF_OPTIONAL, &optional,
				   KVF_OPTSEL,   &optsel,
				   NULL);
		sel_variable = sel_var(location, subform_var,subform_type,
                                       pane_var, variable);
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
		print_if_protection(file, sel_variable);
		kfprintf(file, "\t    kvf_get_attribute(%s_struct, KVF_FILE_DEF,\n", sel_variable);
		kfprintf(file, "\t                    &(%s));\n", sel_variable);
		print_else_protection(file, selection->type, location,
		      form_var,	subform_var, pane_var, variable);

		print_sel_selected(file, live, optional, optsel, sel_variable);
	        kfprintf(free_file, "\tkfree(%s);\n", sel_variable);
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
		kfree(sel_variable);
        	break;
        
           case KUIS_INTEGER:
		variable = ktoken_to_string(selection->var_token);
		kvf_get_attributes(selection->back_kformstruct,
				   KVF_LIVE,     &live,
				   KVF_OPTIONAL, &optional,
				   KVF_OPTSEL,   &optsel,
				   NULL);
		sel_variable = sel_var(location, subform_var,subform_type,
                                       pane_var, variable);
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
		print_if_protection(file, sel_variable);
		kfprintf(file, "\t    kvf_get_attribute(%s_struct, KVF_INT_DEF,\n", sel_variable);
		kfprintf(file, "\t                    &(%s));\n", sel_variable);
		print_else_protection(file, selection->type, location,
		 	form_var, subform_var, pane_var, variable);

		print_sel_selected(file, live, optional, optsel, sel_variable);
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t     kfree(%s_struct);\n", sel_variable);
		kfree(sel_variable);
        	break;
       
      	   case KUIS_FLOAT:
		variable = ktoken_to_string(selection->var_token);
		kvf_get_attributes(selection->back_kformstruct,
				   KVF_LIVE,     &live,
				   KVF_OPTIONAL, &optional,
				   KVF_OPTSEL,   &optsel,
				   NULL);
		sel_variable = sel_var(location, subform_var,subform_type,
                                       pane_var, variable);
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
		print_if_protection(file, sel_variable);
		kfprintf(file, "\t    kvf_get_attribute(%s_struct, KVF_FLOAT_DEF,\n", sel_variable);
		kfprintf(file, "\t                    &(%s));\n", sel_variable);
		print_else_protection(file, selection->type, location,
			form_var,  subform_var, pane_var, variable);
		print_sel_selected(file, live, optional, optsel, sel_variable);
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
		kfree(sel_variable);
       		break;
       
      	   case KUIS_DOUBLE:
		variable = ktoken_to_string(selection->var_token);
		kvf_get_attributes(selection->back_kformstruct,
				   KVF_LIVE,     &live,
				   KVF_OPTIONAL, &optional,
				   KVF_OPTSEL,   &optsel,
				   NULL);
		sel_variable = sel_var(location, subform_var,subform_type,
                                       pane_var, variable);
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
		print_if_protection(file, sel_variable);
		kfprintf(file, "\t    kvf_get_attribute(%s_struct, KVF_DOUBLE_DEF,\n", sel_variable);
		kfprintf(file, "\t                    &(%s));\n", sel_variable);
		print_else_protection(file, selection->type, location,
			form_var,  subform_var, pane_var, variable);
		print_sel_selected(file, live, optional, optsel, sel_variable);
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
		kfree(sel_variable);
       		break;
       
      		
       	   case KUIS_STRING:
       	   case KUIS_STRINGLIST:
		variable = ktoken_to_string(selection->var_token);
		kvf_get_attributes(selection->back_kformstruct,
				   KVF_LIVE,     &live,
				   KVF_OPTIONAL, &optional,
				   KVF_OPTSEL,   &optsel,
				   NULL);
		sel_variable = sel_var(location, subform_var,subform_type,
                                       pane_var, variable);
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
		print_if_protection(file, sel_variable);
		kfprintf(file, "\t    kvf_get_attribute(%s_struct, KVF_STRING_DEF,\n", sel_variable);
		kfprintf(file, "\t                    &(%s));\n", sel_variable);
		print_else_protection(file, selection->type, location,
		 	form_var, subform_var, pane_var, variable);
		print_sel_selected(file, live, optional, optsel, sel_variable);
	        kfprintf(free_file, "\tkfree(%s);\n", sel_variable);
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
		kfree(sel_variable);
       		break;
       
      	  case KUIS_LOGICAL:
		variable = ktoken_to_string(selection->var_token);
		kvf_get_attribute(selection->back_kformstruct,
				  KVF_LIVE, &live);
		sel_variable = sel_var(location, subform_var,subform_type,
                                       pane_var, variable);
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
		print_if_protection(file, sel_variable);
		kfprintf(file, "\t    kvf_get_attribute(%s_struct, KVF_LOGIC_DEF,\n", sel_variable);
		kfprintf(file, "\t                    &(%s));\n", sel_variable);
		kfprintf(file, "\t    kvf_get_attribute(%s_struct, KVF_LOGIC_LABEL,\n", sel_variable);
		kfprintf(file, "\t                    &(%s_label));\n", sel_variable);
		print_else_protection(file, selection->type, location,
		 	form_var, subform_var, pane_var, variable);
		if (live)
		    kfprintf(file, "\t%s_selected = FALSE;\n\n", sel_variable);
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
	        kfprintf(free_file, "\tkfree(%s_label);\n", sel_variable);
		kfree(sel_variable);
       		break;
       
      	  case KUIS_FLAG:
		variable = ktoken_to_string(selection->var_token);
		kvf_get_attribute(selection->back_kformstruct,
				  KVF_LIVE, &live);
		sel_variable = sel_var(location, subform_var,subform_type,
                                       pane_var, variable);
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
		print_if_protection(file, sel_variable);
		kfprintf(file, "\t    kvf_get_attribute(%s_struct, KVF_OPTSEL,\n", sel_variable);
		kfprintf(file, "\t                    &(%s));\n", sel_variable);
		print_else_protection(file, selection->type, location,
			form_var, subform_var, pane_var, variable);
		if (live)
		    kfprintf(file, "\t%s_selected = FALSE;\n\n", sel_variable);
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
		kfree(sel_variable);
       		break;
       
      	  case KUIS_CYCLE:
		variable = ktoken_to_string(selection->var_token);
		kvf_get_attribute(selection->back_kformstruct,
				  KVF_LIVE, &live);
		sel_variable = sel_var(location, subform_var,subform_type,
                                       pane_var, variable);
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
		print_if_protection(file, sel_variable);
		kfprintf(file, "\t    kvf_get_attribute(%s_struct, KVF_CYCLE_VAL,\n", sel_variable);
		kfprintf(file, "\t                    &(%s));\n", sel_variable);
		kfprintf(file, "\t    kvf_get_attribute(%s_struct, KVF_CYCLE_LABEL,\n", sel_variable);
		kfprintf(file, "\t                    &(%s_label));\n", sel_variable);
		print_else_protection(file, selection->type, location,
			form_var, subform_var, pane_var, variable);
		if (live)
		    kfprintf(file, "\t%s_selected = FALSE;\n\n", sel_variable);
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
	        kfprintf(free_file, "\tkfree(%s_label);\n", sel_variable);
		kfree(sel_variable);
       		break;

      	  case KUIS_LIST:
      	  case KUIS_DISPLAYLIST:
		variable = ktoken_to_string(selection->var_token);
		kvf_get_attribute(selection->back_kformstruct,
				  KVF_LIVE, &live);
		if (selection->type == KUIS_DISPLAYLIST)
		    kvf_get_attribute(selection->back_kformstruct,
                                      KVF_LIST_DOUBLECLICK, &doubleclick);
		else doubleclick = FALSE;

		sel_variable = sel_var(location, subform_var,subform_type,
                                       pane_var, variable);
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
		print_if_protection(file, sel_variable);
		kfprintf(file, "\t    kvf_get_attribute(%s_struct, KVF_LIST_VAL,\n", sel_variable);
		kfprintf(file, "\t                    &(%s));\n", sel_variable);
		kfprintf(file, "\t    kvf_get_attribute(%s_struct, KVF_LIST_LABEL,\n", sel_variable);
		kfprintf(file, "\t                    &(%s_label));\n", sel_variable);
		print_else_protection(file, selection->type, location,
		 	form_var, subform_var, pane_var, variable);
		if (live)
		    kfprintf(file, "\t%s_selected = FALSE;\n\n", sel_variable);
		if (doubleclick)
		    kfprintf(file, "\t%s_dblclick = FALSE;\n\n", sel_variable);
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
	        kfprintf(free_file, "\tkfree(%s_label);\n", sel_variable);
		kfree(sel_variable);
       		break;

	   case KUIS_TOGGLE:
		variable = ktoken_to_string(selection->var_token);
                kvf_get_attributes(selection->back_kformstruct,
                                   KVF_LIVE,        &live,
                                   KVF_OPTIONAL,    &optional,
                                   KVF_OPTSEL,      &optsel,
                                   KVF_TOGGLE_TYPE, &toggle_type,
                                   NULL);
		sel_variable = sel_var(location, subform_var,subform_type,
                                       pane_var, variable);
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
		print_if_protection(file, sel_variable);
		kfprintf(file, "\t    kvf_get_attribute(%s_struct, KVF_TOGGLE_NUM,\n", sel_variable);
		kfprintf(file, "\t                    &(%s_num));\n", sel_variable);
		kvf_get_attribute(selection->back_kformstruct,
				  KVF_TOGGLE_VAL, &toggle_val);
	        if ((toggle_type == KUIS_STRING) ||
		    (toggle_type == KUIS_INPUTFILE) ||
		    (toggle_type == KUIS_OUTPUTFILE))
		{
		    kfprintf(file,"\t    %s_val = kstrdup(\"%s\");\n",
                             sel_variable, toggle_val);
		}
	        else
		{
		    kfprintf(file,"\t    %s_val = %s;\n",
		             sel_variable, toggle_val);
		}
		print_else_protection(file, selection->type, location,
			form_var, subform_var, pane_var, variable);
		print_sel_selected(file, live, optional, optsel, sel_variable);
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
		kfree(sel_variable);
		break;
		
		
       	   case KUIS_ROUTINE:
		sel_variable = sel_var(location, subform_var,subform_type,
                                       pane_var, "routine");
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
	        kfprintf(file, "\tif (%s_struct == NULL)\n\t", sel_variable);
		desc = sel_desc(selection->type, location,
				subform_var, pane_var, "routine");
	        kfprintf(file, "\t    _xvf_init_%s_warning(\"%s\");\n", 
		        form_var, desc);
		kfree(desc);
		kfprintf(file, "\n");
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
		kfree(sel_variable);
       		break;
       
      	   case KUIS_MASTERACTION:
	        variable = ktoken_to_string(selection->var_token);
		sel_variable = sel_var(location, subform_var,subform_type,
                                       pane_var, variable);
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
	        kfprintf(file, "\tif (%s_struct == NULL)\n", sel_variable);
		desc = sel_desc(selection->type, location,
                                subform_var, pane_var, variable);
	        kfprintf(file, "\t    _xvf_init_%s_warning(\"%s\");\n", 
		        form_var, desc);
		kfree(desc);
       		kfprintf(file,"\t%s = FALSE;\n", sel_variable);
		kfprintf(file, "\tselection = %s_struct->Selptr;\n",
			     sel_variable);
		kfprintf(file, "\tkvf_set_attribute(selection->back_kformstruct,\n");
		kfprintf(file, "\t                  KVF_SELECTED, FALSE);\n");
		kfprintf(file, "\n");
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
		kfree(sel_variable);
		break;

      	   case KUIS_SUBFORMACTION:
                variable = ktoken_to_string(selection->var_token);
		sel_variable = sel_var(location, subform_var,subform_type,
                                       pane_var, variable);
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
	        kfprintf(file, "\tif (%s_struct == NULL)\n", sel_variable);
		desc = sel_desc(selection->type, location,
                                  subform_var, pane_var, variable);
	        kfprintf(file, "\t    _xvf_init_%s_warning(\"%s\");\n", 
		        form_var, desc);
		kfree(desc);
       		kfprintf(file,"\t%s = FALSE;\n", sel_variable);
		kfprintf(file, "\tselection = %s_struct->Selptr;\n",
			     sel_variable);
		kfprintf(file, "\tkvf_set_attribute(selection->back_kformstruct,\n");
		kfprintf(file, "\t                  KVF_SELECTED, FALSE);\n");
		kfprintf(file, "\n");
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
		kfree(sel_variable);
		break;

      	   case KUIS_PANEACTION:
                variable = ktoken_to_string(selection->var_token);
		sel_variable = sel_var(location, subform_var,subform_type,
                                       pane_var, variable);
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
	        kfprintf(file, "\tif (%s_struct == NULL)\n", sel_variable);
		desc = sel_desc(selection->type, location,
                                subform_var, pane_var, variable);
	        kfprintf(file, "\t    _xvf_init_%s_warning(\"%s\");\n", 
		        form_var, desc);
		kfree(desc);
       		kfprintf(file,"\t%s = FALSE;\n", sel_variable);
		kfprintf(file, "\tselection = %s_struct->Selptr;\n", 
			     sel_variable);
		kfprintf(file, "\tkvf_set_attribute(selection->back_kformstruct,\n");
		kfprintf(file, "\t                  KVF_SELECTED, FALSE);\n");
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
		kfree(sel_variable);
       		break;
       
      	   case KUIS_WORKSPACE:
                variable = ktoken_to_string(selection->var_token);
		sel_variable = sel_var(location, subform_var,subform_type,
                                       pane_var, variable);
		if (location != KMASTER) 
		{
		     if (!(*subform_created))
		     {
		         kfprintf(file, "\tif (!(xvf_create_subform(subform, -1, -1, FALSE)))\n");
                         kfprintf(file, "\t   return(FALSE);\n");
		         kfprintf(file, "\t_xvf_get_%s(forminfo->%s);\n", 
			          subform_var, subform_var);
		         *subform_created = TRUE;
		     }
		     if (location == KPANE)
		     {
			if (!(pane_created))
			{
			    kfprintf(file, "\tif (!(xvf_create_pane(pane, FALSE)))\n");
                            kfprintf(file, "\t   return(FALSE);\n");
		            kfprintf(file, "\t_xvf_get_%s(forminfo->%s->%s);\n",
			             pane_var, subform_var, pane_var);
			    pane_created = TRUE;
			}
		     }
		}
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
	        kfprintf(file, "\tif (%s_struct == NULL)\n", sel_variable);
		desc = sel_desc(selection->type, location,
                                subform_var, pane_var, variable);
	        kfprintf(file, "\t    _xvf_init_%s_warning(\"%s\");\n", 
		        form_var, desc);
		kfree(desc);
		kfprintf(file, "\telse\n\t{\n");

		kfprintf(file, "\t    kvf_get_attribute(%s_struct, KVF_WORKSPACE,\n", sel_variable);
		kfprintf(file, "\t                    &(%s));\n", sel_variable);
		kfprintf(file, "\n\t}\n");
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
		kfree(sel_variable);
       		break;

	   case KUIS_QUIT:
                variable = ktoken_to_string(selection->var_token);
		sel_variable = sel_var(location, subform_var,subform_type,
                                       pane_var, "quit");
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
	        kfprintf(file, "\tif (%s_struct == NULL)\n\t", sel_variable);
		desc = sel_desc(selection->type, location,
                                subform_var, pane_var, variable);
	        kfprintf(file, "\t    _xvf_init_%s_warning(\"%s\");\n", 
			form_var, desc);
		kfree(desc);
		kfprintf(file, "\t%s = FALSE;\n", sel_variable);
		kfprintf(file, "\n");
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
		kfree(sel_variable);
		break;

	   case KUIS_HELP:
                variable = ktoken_to_string(selection->var_token);
		sel_variable = sel_var(location, subform_var,subform_type,
                                       pane_var, variable);
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
	        kfprintf(file, "\tif (%s_struct == NULL)\n\t", sel_variable);
		desc = sel_desc(selection->type, location,
                                subform_var, pane_var, variable);
	        kfprintf(file, "\t    _xvf_init_%s_warning(\"%s\");\n", 
			form_var, desc);
		kfree(desc);
		kfprintf(file, "\n");
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
		kfree(sel_variable);
		break;

       	   case KUIS_BLANK:
       	   case KUIS_STDIN:
       	   case KUIS_STDOUT:
                variable = ktoken_to_string(selection->var_token);
		sel_variable = sel_var(location, subform_var,subform_type,
                                       pane_var, variable);
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
	        kfprintf(file, "\tif (%s_struct == NULL)\n", sel_variable);
		desc = sel_desc(selection->type, location,
                                subform_var, pane_var, variable);
	        kfprintf(file, "\t    _xvf_init_%s_warning(\"%s\");\n", 
			form_var, desc);
		kfree(desc);
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
		kfree(sel_variable);
		break;

      	   case KUIS_STARTSUBMENU:
                variable = ktoken_to_string(selection->var_token);
		sel_variable = sel_var(location, subform_var, subform_type,
                                       pane_var, variable);
                kgen_gui_print_create_sel_struct(file, fullname, location, sel_variable);
	        kfprintf(file, "\tif (%s_struct == NULL)\n", sel_variable);
		desc = sel_desc(selection->type, location,
                                  subform_var, pane_var, variable);
	        kfprintf(file, "\t    _xvf_init_%s_warning(\"%s\");\n", 
		        form_var, desc);
		kfree(desc);
		kfprintf(file, "\n");
	        kfprintf(free_file, "\tif (%s_struct != NULL)\n", sel_variable);
	        kfprintf(free_file, "\t    kfree(%s_struct);\n", sel_variable);
		kfree(sel_variable);
		break;

	   case KUIS_MUTEXCL:
	   case KUIS_MUTINCL:
                if (!(gen_init_selections(file, free_file, location, 
				form_var, subform_var, subform_type, pane_var, 
				selection->group_next, subform_created)))
		    return(FALSE);
		break;

      	   default:
       		break;
       
      	   }  /* end switch */

	   selection = selection->next;
      	} 
      
	return(TRUE);
      
} /* end gen_init_selections */


/*-----------------------------------------------------------
|
|  Routine Name: sel_var
|
|       Purpose: Returns complete selection variable for
|                use in code initialization, for example,
|                forminfo->my_subform->my_pane->my_variable
|
| 	  Input: type        - KMASTER, KGUIDEPANE, or KPANE
|                subform_var - variable on -M line (if KGUIDEPANE or KPANE)
|                subform_type - type of subform (if type != KMASTER)
|                pane_var    - variable on -P line (if KPANE)
|                variable    - variable on UIS line assoc. w/ selection
|
|        Output: string containing full selection variable
|          Date: May 12, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/

static char *sel_var(
   int  type,
   char *subform_var,
   int  subform_type,
   char *pane_var,
   char *variable)
{
	char temp[KLENGTH];

	if (type == KMASTER)
	    ksprintf(temp, "forminfo->%s", variable);
	else if (type == KGUIDEPANE)
	{
	    if (subform_type == KUIS_SUBFORMBUTTON) 
		ksprintf(temp, "forminfo->%s->%s", subform_var, variable);
	    else ksprintf(temp, "subforminfo->%s", variable);
	}
	else if (type == KPANE)
	{
	    if (subform_type == KUIS_SUBFORMBUTTON) 
		ksprintf(temp, "forminfo->%s->%s->%s", subform_var, 
		    pane_var, variable);
	    else ksprintf(temp, "subforminfo->%s->%s", pane_var, variable);
	}

	return(kstrdup(temp));
}

/*-----------------------------------------------------------
|
|  Routine Name: print_if_protection
|
|       Purpose: Generates the beginning of an 'if' statement
|                that is generated often.
|
| 	  Input: file         - open stream to form_init.c
|                sel_variable - complete selection variable 
|        Output: none
|          Date: May 12, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/

static void print_if_protection(
   kfile *file,
   char *sel_variable)
{
	kfprintf(file, "\tif (%s_struct != NULL)\n\t{\n", sel_variable);
	kfprintf(file, "\t    selection = %s_struct->Selptr;\n", sel_variable);
}

/*-----------------------------------------------------------
|
|  Routine Name: print_else_protection
|
|       Purpose: Generates the beginning of an 'if' statement
|                that is generated often.
|
| 	  Input: file         - open stream to form_init.c
|                sel_variable - complete selection variable 
|        Output: none
|          Date: May 12, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/

static void print_else_protection(
   kfile *file,
   int   flag,
   int   location,
   char  *form_var,
   char  *subform_var,
   char  *pane_var,
   char  *variable)
{
	char *desc;

	desc = sel_desc(flag,  location, subform_var, pane_var, variable);
	kfprintf(file, "\t}\n");
	kfprintf(file, "\telse\n");
	kfprintf(file, "\t    _xvf_init_%s_warning(\"%s\");\n", form_var, desc);
	kfprintf(file, "\n");
	kfree(desc);
}

/*-----------------------------------------------------------
|
|  Routine Name: sel_desc
|
|       Purpose: Generate a string describing a selection.
|
| 	  Input: flag         - type of selection
|                location     - KMASTER, KGUIDEPANE, or KPANE
|                subform_var  - variable on subform
|                pane_var     - variable on pane
|                variable     - variable on selection
|        Output: none
|          Date: May 12, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/

static char *sel_desc(
   int  flag,
   int  location,
   char *subform_var,
   char *pane_var,
   char *variable)
{
	char temp[KLENGTH];
	char *type_string;

	type_string = kgen_uis_type_string(flag);

	if (location == KMASTER)
	{
	   if (variable == NULL)
	       ksprintf(temp, "%s on master form", type_string);
	   else ksprintf(temp, "%s '%s' on master form", type_string, variable);
	}
	else if (location == KGUIDEPANE)
	{
	   if (variable == NULL)
	       ksprintf(temp, "%s on subform '%s'", type_string, 
		       subform_var);
	   else ksprintf(temp, "%s '%s' on subform '%s'", type_string, 
		       variable, subform_var);
	}
	else if (location == KPANE)
	{
	   if (variable == NULL)
	       ksprintf(temp, "%s on pane '%s' of subform '%s'",
		       type_string, pane_var, subform_var);
	   else ksprintf(temp, "%s '%s' on pane '%s' of subform '%s'",
		       type_string, variable, pane_var, subform_var);
	}

	kfree(type_string);
	return(kstrdup(temp));
}

/*-----------------------------------------------------------
|
|  Routine Name: print_sel_selected
|
|       Purpose: Generates a line of code to correctly initialize
|                the _selected field of a selection.
|
| 	  Input: file         - open stream to form_init.c
|                live         - TRUE if selection is "live"          
|                optional     - TRUE if selection is optional        
|                opt_sel      - value of the OptSel field in UIS line
|                sel_variable - complete variable describing selection
|        Output: none
|          Date: May 12, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/

static void print_sel_selected(
   kfile *file,
   int   live,
   int   optional,
   int   opt_sel,
   char  *sel_variable)
{
	if (live) 
	    kfprintf(file, "\t%s_selected = FALSE;\n", sel_variable);
	if (optional && (opt_sel == 0))
	    kfprintf(file, "\t%s_optsel = FALSE;\n", sel_variable);
	else if (optional && (opt_sel != 0))
	    kfprintf(file, "\t%s_optsel = TRUE;\n",  sel_variable);
	kfprintf(file, "\n");
}
