/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>> Routines for interactive prompting and error checking <<<<
   >>>>                                                       <<<<
   >>>>   Private:                                            <<<<
   >>>>              kgen_clui_prompt()                       <<<<
   >>>>              kgen_clui_prompt_mut_excl()              <<<<
   >>>>              kgen_clui_prompt_mut_incl()              <<<<
   >>>>              kgen_clui_prompt_loose_group()           <<<<
   >>>>              kgen_clui_prompt_toggle()                <<<<
   >>>>              kgen_clui_prompt_list()                  <<<<
   >>>>    Static:                                            <<<<
   >>>>    Public:                                            <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

/*-----------------------------------------------------------
|
|  Routine Name: kgen_clui_prompt
|
|       Purpose: This routine gets a single program argument from
|                the user when interactive prompting is being used.
|
|         Input: selection - selection corresponding to program argument
|                present   - returns TRUE if selection is given via
|                            interactive prompting, FALSE otherwise.
|        Output: none
|       Returns:
|    Written By: Danielle Argiro
|          Date: March 30, 1994
| Modifications:
|
------------------------------------------------------------*/

int kgen_clui_prompt(
   kselection  *selection,
   int         *present)
{
	kform  *form;
	char   *variable;
	int    presence_flag;
	int    ok, status = TRUE, optional;
	char   mesg[KLENGTH];
	char   *pstring;

	ok = FALSE;
	*present = presence_flag = FALSE;
	form = selection->back_form;
	variable = ktoken_to_string(selection->var_token);
	pstring  = kgen_create_pstr(selection);
	ksprintf(mesg, "       -- %s is a required argument-- \n",
	         variable);

	kvf_get_attribute(selection->back_kformstruct, KVF_OPTIONAL, &optional);

	while (!ok)
	{
	    /*
	     *  interactively prompt the user for the value, and
             *  store that information in the kform.
	     */
	    switch(selection->type)
	    {
                case KUIS_INPUTFILE:
		     if ((kstrcmp(variable, "a")  == 0) ||
			 (kstrcmp(variable, "ap") == 0))
		     {
			 ok = TRUE;
	                 kvf_set_attribute(selection->back_kformstruct,
			                   KVF_MODIFIED, TRUE);
			 break;
		     }
	             status = kclui_prompt_infile(form, variable, pstring, 
			                           NULL, &presence_flag);
		     if (!status) ok = FALSE;
		     else if (presence_flag)
		     {
			ok = TRUE;
			kvf_set_attributes(selection->back_kformstruct,
                                           KVF_OPTSEL, TRUE,
					   KVF_MODIFIED, TRUE,
				           NULL);
		     }
		     else if (optional) ok = TRUE;
		     else if (!optional)
                          kfprintf(kstderr, "%s", mesg);
                     break;

                case KUIS_OUTPUTFILE:
		     if (kstrcmp(variable, "A") == 0) 
	             {
			 ok = TRUE;
	                 if (presence_flag)
	                      kvf_set_attribute(selection->back_kformstruct,
			                        KVF_MODIFIED, TRUE);
			 break;
                     }
	             status = kclui_prompt_outfile(form, variable, pstring, 
				                    NULL, &presence_flag);
                     if (!status) ok = FALSE;
		     else if (presence_flag)
		     {
			ok = TRUE;
                        kvf_set_attributes(selection->back_kformstruct,
                                           KVF_OPTSEL, TRUE,
                                           KVF_MODIFIED, TRUE,
                                           NULL);
		     }
		     else if (optional) ok = TRUE;
		     else if (!optional)
                          kfprintf(kstderr, "%s", mesg);
                     break;

	        case KUIS_INTEGER:
	             status = kclui_prompt_int(form, variable, pstring, 
				                NULL, &presence_flag);
		     if (!status) ok = FALSE;
		     else if (presence_flag)
		     {
			ok = TRUE;
                        kvf_set_attributes(selection->back_kformstruct,
                                           KVF_OPTSEL, TRUE,
                                           KVF_MODIFIED, TRUE,
                                           NULL);
		     }
		     else if (optional) ok = TRUE;
		     else if (!optional)
                          kfprintf(kstderr, "%s", mesg);
                     break;
     
	        case KUIS_FLOAT:
	             status = kclui_prompt_float(form, variable, pstring, 
				                  NULL, &presence_flag);
		     if (!status) ok = FALSE;
		     else if (presence_flag)
		     {
			ok = TRUE;
                        kvf_set_attributes(selection->back_kformstruct,
                                           KVF_OPTSEL, TRUE,
                                           KVF_MODIFIED, TRUE,
                                           NULL);
		     }
		     else if (optional) ok = TRUE;
		     else if (!optional)
                          kfprintf(kstderr, "%s", mesg);
                     break;
     
	        case KUIS_DOUBLE:
	             status = kclui_prompt_double(form, variable, pstring, 
				                   NULL, &presence_flag);
		     if (!status) ok = FALSE;
		     else if (presence_flag)
		     {
			ok = TRUE;
                        kvf_set_attributes(selection->back_kformstruct,
                                           KVF_OPTSEL, TRUE,
                                           KVF_MODIFIED, TRUE,
                                           NULL);
		     }
		     else if (optional) ok = TRUE;
		     else if (!optional)
                          kfprintf(kstderr, "%s", mesg);
                     break;

	        case KUIS_STRING:
	        case KUIS_STRINGLIST:
	             status = kclui_prompt_string(form, variable, pstring, 
				                   NULL, &presence_flag);
		     if (!status) ok = FALSE;
		     else if (presence_flag)
		     {
			ok = TRUE;
                        kvf_set_attributes(selection->back_kformstruct,
                                           KVF_OPTSEL, TRUE,
                                           KVF_MODIFIED, TRUE,
                                           NULL);
		     }
		     else if (optional) ok = TRUE;
		     else if (!optional)
                         kfprintf(kstderr, "%s", mesg);
                     break;
     
                case KUIS_LOGICAL:
	             status = kclui_prompt_logical(form, variable, pstring, 
				                    NULL, &presence_flag);
	 	     if (!status) ok = FALSE;
	 	     else if (presence_flag)
		     {
			ok = TRUE;
                        kvf_set_attributes(selection->back_kformstruct,
                                           KVF_OPTSEL, TRUE,
                                           KVF_MODIFIED, TRUE,
                                           NULL);
		     }
		     else if (optional) ok = TRUE;
		     else if (!optional)
                          kfprintf(kstderr, "%s", mesg);
                     break;
     
	        case KUIS_FLAG:
		     if  ((kstrcmp(variable, "U")   == 0) ||
		          (kstrcmp(variable, "V")   == 0) ||
		          (kstrcmp(variable, "gui") == 0) ||
		          (kstrcmp(variable, "P")   == 0))
		     {
			return(TRUE);
		     }
	             status = kclui_prompt_flag(form, variable, pstring, 
					         &presence_flag);
		     if (!status) ok = FALSE;
		     if (presence_flag)
		     {
			ok = TRUE;
                        kvf_set_attributes(selection->back_kformstruct,
                                           KVF_OPTSEL, TRUE,
                                           KVF_MODIFIED, TRUE,
                                           NULL);
		     }
		     else if (optional) ok = TRUE;
                     break;

		case KUIS_MUTEXCL:
                     status = kgen_clui_prompt_mut_excl(selection, FALSE, 
						 	&presence_flag);
		     if (!status) ok = FALSE;
		     else if (presence_flag) ok = TRUE;
		     else if (optional) ok = TRUE;
		     break;

		case KUIS_MUTINCL:
                     status = kgen_clui_prompt_mut_incl(selection, FALSE, 
							&presence_flag);
		     if (!status) ok = FALSE;
		     else if (presence_flag) ok = TRUE;
		     else if (optional) ok = TRUE;
		     break;

		case KUIS_GROUP:
                     status = kgen_clui_prompt_loose(selection, FALSE, 
						     &presence_flag);
		     if (!status) ok = FALSE;
		     else if (presence_flag) ok = TRUE;
		     else if (optional) ok = TRUE;
	             if (presence_flag)
	                 kvf_set_attribute(selection->back_kformstruct,
			                   KVF_MODIFIED, TRUE);
		     break;

		case KUIS_TOGGLE:
                     status = kgen_clui_prompt_toggle(selection, 
					              &presence_flag);
		     if (!status) ok = FALSE;
		     else if (presence_flag)
		     {
			ok = TRUE;
                        kvf_set_attributes(selection->back_kformstruct,
                                           KVF_OPTSEL, TRUE,
                                           KVF_MODIFIED, TRUE,
                                           NULL);
		     }
		     else if (optional) ok = TRUE;
		     else if (!optional)
                          kfprintf(kstderr, "%s", mesg);
		     break;

		case KUIS_LIST:
		case KUIS_CYCLE:
		case KUIS_DISPLAYLIST:
                     status = kgen_clui_prompt_list(selection, 
						    &presence_flag);
		     if (!status) ok = FALSE;
		     else if (presence_flag)
		     {
			ok = TRUE;
                        kvf_set_attributes(selection->back_kformstruct,
                                           KVF_OPTSEL, TRUE,
                                           KVF_MODIFIED, TRUE,
                                           NULL);
		     }
		     else if (optional) ok = TRUE;
		     else if (!optional)
                          kfprintf(kstderr, "%s", mesg);
		     break;
	    }
	}

	*present = presence_flag;
	return(status);
    
} /* end kgen_clui_prompt */

/*-----------------------------------------------------------
|
|       Routine: kgen_clui_prompt_mut_excl
|
|       Purpose: This routine prompts interactively for 
|                ME groups' values from the command line when the
|                prompt (-P) flag is used
|
|         Input: group_start - header of the group list
|                inside      - KUIS_MUTINCL: ME group is inside MI group
|                              KUIS_GROUP:   ME group is inside loose group
|                              FALSE:        ME group is not nested.
|
|        Output: presence_flag - TRUE if one and only one value
|                                was entered for required ME group;
|                                TRUE if one or zero values was
|                                entered for optional ME group;
|                                FALSE otherwise
|
|       Returns: TRUE on success, FALSE on error
|    Written By: Danielle Argiro
|          Date: March 31, 1994
| Modifications: 
|
------------------------------------------------------------*/

int kgen_clui_prompt_mut_excl(
   kselection *group_start,
   int        inside,
   int        *present)
{
	int        optional, me_count, ok; 
	int        group_size, presence_flag;
        kselection *groupmember;

	ok = FALSE;

	kvf_get_attribute(group_start->back_kformstruct,
			  KVF_OPTIONAL, &optional);
	group_size = kgen_get_group_size(group_start);

	while (!ok)
	{
            me_count = 0;
    	    *present = presence_flag = FALSE;
    	    groupmember = group_start->group_next;

	    /*
	     *  special case: std args -P and -gui dealt with in kgen_initialize
	     */
	    if ((kstrcmp(ktoken_to_string(groupmember->var_token), "P") == 0)||
	        (kstrcmp(ktoken_to_string(groupmember->var_token), "khoros") == 0))
	        return(TRUE);

            /*
	     *  display introductory remarks
	     */
    	    if (!inside)
    	    {
                kfprintf(kstderr, 
    		     "The following %d arguments are mutually exclusive.\n", 
    		     group_size);
                kfprintf(kstderr, "Provide a value for ONE ");
    	        if (optional) 
    	            kfprintf(kstderr, "or NONE ");
    	        kfprintf(kstderr, "of the %d arguments:\n", group_size);
    	        kgen_print_group_choices(group_start, NULL);
                    kfprintf(kstderr, "\n--------\n");
    	    }
    	    else  
    	    {
    	        kfprintf(kstderr, "\n\t");
                kfprintf(kstderr, 
    	    	         "The following %d arguments are mutually exclusive.\n",
    		          group_size);
    	        kfprintf(kstderr, "\tProvide a value for ONE ");
    	        kfprintf(kstderr, "of the %d arguments:\n", group_size);
    	        kgen_print_group_choices(group_start, "\t");
                    kfprintf(kstderr, "\n\t--------\n");
    	    }
    
	    /*
	     *  interactively prompt for values of individual members of group
	     */
    	    while ((groupmember != NULL) && (me_count == 0))
            {
	        if (groupmember->type == KUIS_BLANK)
	        {
	    	    groupmember = groupmember->next;
		    continue;
	        }
    	        /* ME group has illegal ME group nested inside */
    	        if (groupmember->type == KUIS_MUTEXCL)
    	        {
    	    	    kerror(KCODEGEN, "print_get_mut_excl_prompt",
    		           "Nesting of ME groups within ME redundant");
    		    return(FALSE);
    	        }
    
    	        /* ME group has MI group nested inside */
    	        else if (groupmember->type == KUIS_MUTINCL)
    	        {
    	    	    if (!(kgen_clui_prompt_mut_incl(groupmember, KUIS_MUTEXCL, 
					            &presence_flag)))
    		        return(FALSE);
    
    		    if (presence_flag) me_count++;
    	        }
    
    	        /* ME group has loose group nested inside */
    	        else if (groupmember->type == KUIS_GROUP)
    	        {
    	    	    if (!(kgen_clui_prompt_loose(groupmember,  KUIS_MUTEXCL, 
						 &presence_flag)))
    		        return(FALSE);
    
    		    if (presence_flag) me_count++;
    	        }
    
    	        /* ME group has toggle inside (special case) */
    	        else if (groupmember->type == KUIS_TOGGLE)
    	        {
    		    if (!(kgen_clui_prompt_toggle(groupmember, 
						  &presence_flag)))
    		        return(FALSE);
    
    		    if (presence_flag) me_count++;
    	        }
    
    	        /* ME group has list inside (special case) */
                    else if ((groupmember->type == KUIS_LIST)  ||
                             (groupmember->type == KUIS_CYCLE) ||
                             (groupmember->type == KUIS_DISPLAYLIST))
                    {
                        if (!(kgen_clui_prompt_list(groupmember, 
						    &presence_flag)))
                            return(FALSE);
    
                        if (presence_flag) me_count++;
                    }
    
    	        /* ME group has "regular" argument inside */
    	        else
    	        {
                    if (!(kgen_clui_prompt(groupmember, &presence_flag)))
    		        return(FALSE); 
    
    		    if (presence_flag) me_count++;
    	        }
                groupmember = groupmember->next;
    	    }
    
    	   /*
    	    *  a required ME group must have one values specified, 
            *  an optional ME group may have one or none.
     	    */
    	    if ((optional) && (me_count == 0))
    	    {
    	        kfprintf(kstderr, "\n--------\n");
    	        *present = FALSE;
    	        ok = TRUE;
    	    }
    	    else if (me_count == 1)
    	    {
    	        kfprintf(kstderr, "\n--------\n");
    	        *present = TRUE;
    	        ok = TRUE;
    	    }
    	    else 
    	    {
	        kfprintf(kstderr, "\n     Error in interactive command line entry; try again...\n");
    	        *present = FALSE;
		ok = FALSE;
    	    }
	}
	return(TRUE);
}
	

/*-----------------------------------------------------------
|
|       Routine: kgen_clui_prompt_mut_incl
|
|       Purpose: This routine prints the code for getting MI groups' 
|                values from the command line when the (-P) flag is used
|
|         Input: group_start - header of the group list
|                inside      - KUIS_MUTEXCL: MI group is inside ME group
|                              KUIS_GROUP:   MI group is inside loose group
|                              FALSE:        MI group is not nested.
|
|        Output: presence_flag - TRUE if all or no values were entered 
|                                FALSE otherwise
|       Returns: TRUE on success, FALSE on error
|    Written By: Danielle Argiro
|          Date: March 31, 1994
| Modifications: 
|
------------------------------------------------------------*/

int kgen_clui_prompt_mut_incl(
   kselection *group_start,
   int        inside,
   int        *present)
{
	int        ok, mi_count, group_size;
	int        presence_flag;
	kselection *groupmember;

	ok = FALSE;
	group_size = kgen_get_group_size(group_start);

	while (!ok)
	{
            mi_count = 0;
	    *present = presence_flag = FALSE;

	    /*
	     *  display introductory remarks
	     */
	    if (!inside)
	        kfprintf(kstderr, "\n");
	    else kfprintf(kstderr, "\n\t");

            kfprintf(kstderr, 
		     "The following %d arguments are mutually inclusive.\n", 
		     group_size);
	    kfprintf(kstderr, "Provide a value for ALL or NONE ");
	    kfprintf(kstderr, " of the %d arguments:\n", group_size);
	    kgen_print_group_choices(group_start, NULL);

	    /*
	     *  interactively prompt for values of individual members of group
	     */
            groupmember = group_start->group_next;
	    while (groupmember != NULL)
            {
	        if (groupmember->type == KUIS_BLANK)
	        {
	    	   groupmember = groupmember->next;
		   continue;
	        }
                /* MI group has illegal MI group nested inside */
                if (groupmember->type == KUIS_MUTINCL)
                {
                    kerror(KCODEGEN, "kgen_clui_prompt_mut_incl",
                           "Nesting of MI groups within MI groups redundant");
                    return(FALSE);
                }
    
                /* MI group has ME group nested inside */
                else if (groupmember->type == KUIS_MUTEXCL)
                {
                    if (!(kgen_clui_prompt_mut_excl(groupmember,  KUIS_MUTINCL, 
						    &presence_flag)))
                        return(FALSE);

                    if (presence_flag) mi_count++;
                }

                /* MI group has loose group nested inside */
                else if (groupmember->type == KUIS_GROUP)
                {
                    if (!(kgen_clui_prompt_loose(groupmember,  KUIS_MUTINCL, 
					 	 &presence_flag)))
                        return(FALSE);
    
                    if (presence_flag) mi_count++;
                }
    
                /* MI group has toggle inside (special case) */
                else if (groupmember->type == KUIS_TOGGLE)
                {
                    if (!(kgen_clui_prompt_toggle(groupmember, &presence_flag)))
                        return(FALSE);
    
                    if (presence_flag) mi_count++;
                }
    
                /* MI group has list inside (special case) */
                else if ((groupmember->type == KUIS_LIST)  ||
                         (groupmember->type == KUIS_CYCLE) ||
                         (groupmember->type == KUIS_DISPLAYLIST))
                {
                    if (!(kgen_clui_prompt_list(groupmember, &presence_flag)))
                        return(FALSE);

                    if (presence_flag) mi_count++;
                }

                /* MI group has "regular" argument inside */
                else
                {
                    if (!(kgen_clui_prompt(groupmember, &presence_flag)))
                        return(FALSE);
    
                    if (presence_flag) mi_count++;
		    else if (mi_count == 0) return(TRUE);
                }

	        if ((mi_count > 0) && (!presence_flag))
		{
		    kfprintf(kstderr, "\n => In a mutually inclusive group, you must provide ALL the values, or NONE.\n");
		    break;
		}
                groupmember = groupmember->next;
	    }

	    /*
             *  an MI group may have all or no values specified.
 	     */
            if ((mi_count == 0) || (mi_count == group_size))
            {
                kfprintf(kstderr, "\n--------\n");
                *present = TRUE;
                ok = TRUE;
            }
            else 
	    {
	        kfprintf(kstderr, "\n      Error in interactive command line entry; try again... \n");
                *present = FALSE;
	    }
	}
	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kgen_clui_prompt_loose
|
|       Purpose: This routine prints the code for getting loose groups'
|                values from the command line when the (-P) flag is used
|
|         Input: group_start - header of the group list
|                inside      - KUIS_MUTEXCL: loose group is inside ME group
|                              KUIS_MUTINCL: loose group is inside MI group
|                              FALSE:        loose group is not nested.
|
|        Output: presence_flag - TRUE if at least one value was entered 
|                                FALSE otherwise
|       Returns: TRUE on success, FALSE on error
|    Written By: Danielle Argiro
|          Date: March 31, 1994
| Modifications:
|
------------------------------------------------------------*/

int kgen_clui_prompt_loose(
   kselection *group_start,
   int        inside,
   int        *present)
{
	int        i, gp_count, group_size, optional;
	int        presence_flag;
	kselection *groupmember;

	kvf_get_attribute(group_start->back_kformstruct,
			  KVF_OPTIONAL, &optional);
        gp_count = 0;
	*present = presence_flag = FALSE;
	group_size = kgen_get_group_size(group_start);

	/*
	 *  display introductory remarks
	 */
	if (!inside)
	    kfprintf(kstderr, "\n");
	else kfprintf(kstderr, "\n\t");

        kfprintf(kstderr, "The following %d arguments are a group.\n", 
		 group_size);
	kfprintf(kstderr, "Provide a value for AT LEAST ONE ");
	kfprintf(kstderr, " of the %d arguments:\n", group_size);
	kgen_print_group_choices(group_start, NULL);

	/*
	 *  interactively prompt for values of individual members of group
	 */
	i = 0;
        groupmember = group_start->group_next;
	while (groupmember != NULL)
        {
            /* loose group has illegal loose group nested inside */
            if (groupmember->type == KUIS_GROUP)
            {
                kerror(KCODEGEN, "kgen_clui_prompt_loose",
                       "Nesting of loose groups within loose groups not supported");
                return(FALSE);
            }

            /* loose group has ME group nested inside */
            else if (groupmember->type == KUIS_MUTEXCL)
            {
                if (!(kgen_clui_prompt_mut_excl(groupmember, 
                                                KUIS_GROUP, &presence_flag)))
                    return(FALSE);

                if (presence_flag) gp_count++;
            }

            /* loose group has MI group nested inside */
            else if (groupmember->type == KUIS_GROUP)
            {
                if (!(kgen_clui_prompt_mut_incl(groupmember,  
					        KUIS_GROUP, &presence_flag)))
                    return(FALSE);

                if (presence_flag) gp_count++;
            }

            /* loose group has toggle inside (special case) */
            else if (groupmember->type == KUIS_TOGGLE)
            {
                if (!(kgen_clui_prompt_toggle(groupmember,  
					      &presence_flag)))
                    return(FALSE);

                if (presence_flag) gp_count++;
            }

            /* loose group has list inside (special case) */
            else if ((groupmember->type == KUIS_LIST)  ||
                     (groupmember->type == KUIS_CYCLE) ||
                     (groupmember->type == KUIS_DISPLAYLIST))
            {
                if (!(kgen_clui_prompt_list(groupmember,  &presence_flag)))
                    return(FALSE);

                if (presence_flag) gp_count++;
            }

            /* loose group has "regular" argument inside */
            else
            {
                if (!(kgen_clui_prompt(groupmember, &presence_flag)))
                    return(FALSE);

                if (presence_flag) gp_count++;
            }

	    i++;
            groupmember = groupmember->next;
	}

	/*
	 *  a loose group must have at least one value specified
 	 */
        if (gp_count > 0)
        {
            kfprintf(kstderr, "\n--------\n");
            *present = TRUE;
        }
	else if (optional)
	{
	    return(TRUE);
	}
        else 
	{
	    kfprintf(kstderr, "\n     Error in interactive command line entry; try again...\n");
	    *present = FALSE;
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kgen_clui_prompt_toggle
|
|       Purpose: This routine prints the code for getting toggle's
|                values from the command line when the (-P) flag is used
|
|         Input: selection - selection corresponding to toggle argument
|        Output: presence_flag - TRUE if value for toggle was entered properly
|                                FALSE otherwise
|       Returns: TRUE on success, FALSE on error
|    Written By: Danielle Argiro
|          Date: March 31, 1994
| Modifications:
|
------------------------------------------------------------*/

int kgen_clui_prompt_toggle(
   kselection *selection,
   int        *present)
{
	char   **values;
        char   *variable, *return_str, *pstring;
        int    i, ok, size, optional, toggle_type, presence_flag;

	ok = FALSE;
	*present = presence_flag = FALSE;
        variable = ktoken_to_string(selection->var_token);
	pstring = kgen_create_pstr(selection);

	kvf_get_attributes(selection->back_kformstruct,
			   KVF_TOGGLE_TYPE, &toggle_type,
			   KVF_OPTIONAL,    &optional,
			   NULL);
	while (!ok)
	{
	    if(!(kclui_prompt_value(selection->back_form, variable, 
				    KVF_TOGGLE_VAL, pstring, KSTRING,
                                    &return_str, &presence_flag)))
		return(FALSE);

	    if ((!presence_flag) && (optional))
		ok = TRUE;

	    else if ((!presence_flag) && (!optional))
		kfprintf(kstderr, "Error: %s is a required argument; please re-enter:\n", variable);

	    else if (presence_flag)
	    {
		kvf_get_attributes(selection->back_kformstruct,
                                   KVF_TOGGLE_CONTENTS,   &values,
				   KVF_TOGGLE_SIZE, &size,
				   NULL);

		for (i = 0; i < size; i++)
		{
		    if (kstrcmp(return_str, values[i]) == 0)
			ok = TRUE;
		}
	    }
	}
	*present = presence_flag;
	if (presence_flag)
	    kvf_set_attribute(selection->back_kformstruct,
			      KVF_MODIFIED, TRUE);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kgen_clui_prompt_list
|
|       Purpose: This routine prints the code for getting lists's
|                values from the command line when the (-P) flag is used
|
|         Input: selection - selection corresponding to list argument
|        Output: presence_flag - TRUE if value for list was entered properly
|                                FALSE otherwise
|       Returns: TRUE on success, FALSE on error
|    Written By: Danielle Argiro
|          Date: March 31, 1994
| Modifications:
|
------------------------------------------------------------*/

int kgen_clui_prompt_list(
   kselection *selection,
   int        *present)
{
        int    ok, return_int, start; 
	int    size, optional, presence_flag;
        char   *variable;
	char   *pstring;

	ok = FALSE;
	*present = presence_flag = FALSE;
        variable = ktoken_to_string(selection->var_token);
	pstring = kgen_create_pstr(selection);

	kvf_get_attributes(selection->back_kformstruct,
			   KVF_OPTIONAL,    &optional,
			   KVF_LIST_SIZE,   &size,
			   KVF_LIST_START,  &start,
			   NULL);
	while (!ok)
	{
	    if(!(kclui_prompt_value(selection->back_form, variable, 
				    KVF_LIST_VAL, pstring, KINT,
                                    &return_int, &presence_flag)))
		return(FALSE);

	    if ((!presence_flag) && (optional))
		ok = TRUE;

	    else if ((!presence_flag) && (!optional))
		kfprintf(kstderr, "Error: %s is a required argument; please re-enter:\n", variable);

	    else if (presence_flag)
	    {
		if ((return_int >= start) && (return_int <= start + size))
		    ok = TRUE;
	    }
	}
	*present = presence_flag;
	if (presence_flag)
	    kvf_set_attribute(selection->back_kformstruct,
			      KVF_MODIFIED, TRUE);
	return(TRUE);
}
