/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>          Routines to generate l*.c file
   >>>>
   >>>>  Private:
   >>>>             kgen_clui_generate_lfile()
   >>>>             kgen_clui_genlibproto()
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

/*------------------------------------------------------------
|
|  Routine Name: kgen_clui_generate_lfile
|
|       Purpose: Creates the library file l*.c
|
|         Input: prog_spec -  pointer to the internal PS struct
|                name      -  name of program object being generated
|                lname     -  exact name of l*.c file
|		 filepath  -  full path to l*.c file location
|
|        Output: Returns TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Jul 23, 1992 10:12
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int  kgen_clui_generate_lfile(
   kobject   program,
   char    **prog_spec,
   char     *oname)
{
        kstring  routine    = "kgen_clui_generate_lfile()";
        kobject  toolbox    = NULL;
	int      force_flag = FALSE;
	int      object_type;
        kfile    *file;
        char     temp[KLENGTH], routine_name[KLENGTH]; 
	char     purpose[KLENGTH], fullpath[KLENGTH];
        char     *copyright;
	char     *filepath;
        char     **priv_routines;
	char     *purp, *ret;
	char     *lname;
	kobject  file_object;
	unsigned long update;

	/* get the name of the library */
        if (!kcms_get_attributes(program,
				 KCMS_PARENT,            &toolbox,
				 KCMS_CMOBJ_TYPE,        &object_type,
				 KCMS_CMOBJ_GEN_LIBNAME, &lname,
				 KCMS_CMOBJ_UPDATE_DB,   &update,
				 KCMS_END)
	    || !kcms_query_bit(program, KCMS_CMOBJ_FLAGS, KCMS_BIT_CMOBJ_FORCE,
			       &force_flag))
	    return(FALSE);

	/* if there is no library associated with the object, that's ok */
	if (lname == NULL)
	    return(TRUE);

	/* get the path to the library */
	if (!(kcms_get_attribute(program, KCMS_CMOBJ_GEN_LFILE, &file_object)))
	    return(FALSE);

        if (file_object == NULL)
	{
	    kerror(KCODEGEN, routine,
	           "The library routine for this program object does "
		   "not appear in database");
	    return(FALSE);
	}
	
        if (!kcms_get_attribute(file_object, KCMS_PATH, &filepath)
	    || (filepath == NULL))
	{
	    return(FALSE);
	}

        /*
         *  see if l*.c file already exists.  If so, prompt to over-write,
         *  return TRUE if the answer is NO
         */
	if (kfullpath(filepath, NULL, fullpath) == NULL)
	{
	     kerror(KCODEGEN, "kgen_clui_generate_lfile", 
                    "Could not expand path '%s'", filepath);
             return(FALSE);
	}
   
        if (force_flag == FALSE
	    && (update & KCMS_UPDATE_NEW) != KCMS_UPDATE_NEW
	    && koverwrite(kget_notify(), filepath) == FALSE)
	    return(TRUE);

        /*
         *  create & open *.c file
         */
        if ((file = kfopen(fullpath, "w"))== NULL)
        {
	     kerror(KCODEGEN, "kgen_clui_generate_lfile", 
                    "Could not create file '%s'", fullpath);
             return(FALSE);
        }
   
        /*
         * begin l*.c file with RCS header
         */
        kgen_rcs_chdr(file);
         
        /*
         * follow RCS header with copyright
         */
        copyright = kcms_get_copyright(toolbox, KCMS_TB_COPYRIGHT_LONG,
				       KCMS_LANG_C);

        if (copyright != NULL) 
            kfprintf(file, "%s\n", copyright);
	kfree(copyright);
   
        /*
         *  print file header
         */
	ksprintf(temp, "Library Routine for %s", oname);
        priv_routines = (char **) kcalloc(1, sizeof(char *));
        ksprintf(routine_name, "l%s", oname);
        priv_routines[0] = kstrdup(routine_name);
        kgen_srcfile_hdr(file, temp, NULL, 0, NULL, 0, priv_routines, 1);
	kfree(priv_routines[0]);
	kfree(priv_routines);
   
        /*
         *  add in #includes
         */
        if (object_type == KCMS_KROUTINE)
           kfprintf(file, "#include \"internals.h\"\n\n");
        else kfprintf(file, "#include \"%s.h\"\n\n", oname);

        kfprintf(file, "/* -library_includes */\n");
	if (kstrlen(prog_spec[LIBRARY_INCLUDES]) > 0) 
    	    kfprintf(file, "%s", prog_spec[LIBRARY_INCLUDES]); 
        kfprintf(file, "/* -library_includes_end */\n\n\n");
   
        /*
         * put in header for library routine
         */
	if (kstrlen(prog_spec[LIBRARY_SHORTDESC]) == 0) 
	    ksprintf(routine_name, "l%s", oname);
	else ksprintf(routine_name, "l%s - %s", oname,
		     prog_spec[LIBRARY_SHORTDESC]);
        ksprintf(purpose, "Library Routine for %s", oname);

	
	if (kstrlen(prog_spec[LIBRARY_LONGDESC]) > 0) 
	    purp = prog_spec[LIBRARY_LONGDESC];
	else purp = purpose;

	if (kstrlen(prog_spec[LIBRARY_RETURNS]) > 0)
	    ret = prog_spec[LIBRARY_RETURNS];
	else ret = "TRUE (1) on success, FALSE (0) on failure";

        kgen_pubsrc_hdr(file, routine_name, purp, prog_spec[LIBRARY_INPUT],
			prog_spec[LIBRARY_OUTPUT], ret,
                        prog_spec[LIBRARY_RESTRICT], prog_spec[AUTHORS],
			prog_spec[LIBRARY_VERIFY], prog_spec[LIBRARY_SIDEEFF],
			prog_spec[LIBRARY_MODS]);
   
        /*
         *   print out library definition
         */
	if (kstrlen(prog_spec[LIBRARY_DEF]) > 0) 
        {
            kfprintf(file, "/* -library_def */\n");
   	    kfprintf(file, "%s", prog_spec[LIBRARY_DEF]); 
            kfprintf(file, "/* -library_def_end */\n\n");
        }
        else
        {
            kfprintf(file, "/* -library_def */\n");
	    kfprintf(file, "int l%s (void)\n", oname);
	    kfprintf(file, "\n/* -library_def_end */\n");
        }
   
        /*
         *   print out library code
         */
	if (kstrlen(prog_spec[LIBRARY_CODE]) > 0) 
        {
            kfprintf(file, "/* -library_code */\n");
   	    kfprintf(file, "%s", prog_spec[LIBRARY_CODE]); 
            kfprintf(file, "/* -library_code_end */\n");
        }
        else
        {
            kfprintf(file, "/* -library_code */\n");
            kfprintf(file, "{\n   return(TRUE);\n}\n"); 
            kfprintf(file, "/* -library_code_end */\n\n");
        }

        kfclose(file);
   
        kannounce(KCODEGEN, routine, "done generating l%s.c", oname);
	kinfo(KHOSTILE, "the headers for your %s library really suck.", oname);

        return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kgen_clui_genlibproto - Generate a prototype for a lfunction call
|
|       Purpose: This routine will add the l*.c's function proto type
|		 to the library's .h file
|
|         Input: tb_name   -  name of toolbox
|		 oname     -  name of program object being generated
|		 lname     -  exact name name of l*.c file
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Jorgensen
|          Date: Jan 21, 1993
| Modifications:
|
------------------------------------------------------------*/
int
kgen_clui_genlibproto(
   kobject   program,
   char    **prog_spec,
   char     *tbname,
   char     *oname)
{
	kstring  routine = "kgen_clui_genlibproto";
	kfile   *file;
	char    path[KLENGTH],  upper[KLENGTH];
	char    temp1[KLENGTH], temp2[KLENGTH];
	char    *filecontents, *ptr;
        char    *libdef, *filepath;
	kstring  lname;			/* name of associated library */
	int     status, comment = TRUE;
	int     object_type;
	kobject file_object;


	if (!kcms_get_attributes(program,
				KCMS_CMOBJ_TYPE,        &object_type,
				KCMS_CMOBJ_GEN_LIBNAME, &lname,
				KCMS_CMOBJ_GEN_LFILE,   &file_object,
				KCMS_END))
	   return FALSE;

	if (object_type != KCMS_KROUTINE)
	  return(TRUE);

        if ((file_object == NULL) ||
            (!kcms_get_attribute(file_object, KCMS_PATH, &filepath)))
            return(FALSE);

	if (!(kcms_get_attribute(program, KCMS_CMOBJ_PROG_LIBDEF, &libdef)))
	    return(FALSE);

        if (oname == NULL)
	{
	    kerror(KCODEGEN, routine, "Object name is NULL");
	    return(FALSE);
	}

        if (lname == NULL)
	{
	    kerror(KCODEGEN, routine, "Library name is NULL");
	    return(FALSE);
	}

	if (tbname == NULL)
	{
	    kerror(KCODEGEN, routine, "Toolbox name is NULL");
	    return(FALSE);
	}

	kstring_upper(tbname, upper);
	ksprintf(path, "$%s/include/%s/%s.h", upper, lname, lname);
	if (kaccess(path, F_OK) == -1)
	{
	    kdirname(path, upper);
	    if (!kmake_dir(upper, 0777) && errno != EEXIST)
	    {
		kerror(KCODEGEN, routine, "Cannot create '%s' to install prototype for 'l%s.c', and copying a template failed", upper, oname);
		return(FALSE);
	    }
	    if (ksedfile("$BOOTSTRAP/repos/templates/src/template.h", path, FALSE,
		KFILE_UPDATE, NULL,
		"XXXX", lname, "   >>>>    Written By:",
	        "   >>>>    Written By: ghostwriter", NULL) == FALSE)
	    {
		kerror(KCODEGEN, routine, "Cannot access '%s' to install prototype for 'l%s.c', and copying a template failed", path, oname);
		return(FALSE);
	    }
	    else
		kinfo(KSTANDARD, "The include file '%s' for library '%s' does "
		      "not exist.  So, a template .h file has been created.\n",
		      path, lname);
	}
	if ((file = kfopen(path, "r")) == NULL)
	{
	    kerror(KCODEGEN, routine, "Cannot open '%s' for reading", path);
	    return(FALSE);
	}
	filecontents = kparse_file_scan(file, KPARSE_BOF, KPARSE_EOF, KLITERAL,
					NULL, NULL, &status);
	kfclose(file);
	if (status != KPARSE_OK)
	{
	    kerror(KCODEGEN, routine, "Cannot read '%s'", path);
	    return(FALSE);
	}

	ksprintf(upper, "l%s\\s*PROTO\\(\\(", oname);
	ptr = kparse_string_search(filecontents, upper, KLITERAL, NULL,&status);
	if (status == KPARSE_OK)
	  return(TRUE);
	ptr = kparse_string_search(filecontents,
			"Ghostwriter\\s*Prototypes:\\s*Do\\s*Not\\s*Erase.*$",
			KIGNORE_CASE, NULL, &status);
	if (status != KPARSE_OK)
	{
	    comment = FALSE;
	    ksprintf(upper, "^#endif\\s*/\\*\\s*_%s_h_\\s*\\*/$", lname);
	    ptr = kparse_string_search(filecontents, upper, KLITERAL, NULL,
				       &status);
	    if (status != KPARSE_OK)
	    {
		kinfo(KFORCE, "kgen_clui_genlibproto:  '%s' is not in a "
		      "standard format.  So, prototype will be added to "
		      "the end of the file\n", path);
	    }
	    else
	    {
		while (((unsigned int) *ptr) != '#')
		    --ptr;
	    }
	}
	if ((file = kfopen(path, "w")) == NULL)
	{
	    kerror(KCODEGEN, "kgen_clui_genlibproto", "Permission denied on overwriting '%s'.  Prototype for l%s() was NOT added.\n", path, lname);
	    return(FALSE);
	}
	kfwrite(filecontents, sizeof(char), ptr - filecontents, file);
	if (comment == FALSE)
	    kfprintf(file,
	     "\n\t/* Ghostwriter Prototypes:  Do Not Erase This Comment */");

	if (kstrlen(prog_spec[LIBRARY_DEF]) > 0) 
	{
	    kstring_replace(prog_spec[LIBRARY_DEF], "("," PROTO((", temp1); 
	    kstring_replace(temp1, ")", "));", temp2); 
	    kstring_replace(temp2, "\n", " ", temp1);
	    kstring_cleanup(temp1, temp2);
	    kfprintf(file, "\n%s\n", temp2);
	}
	else kfprintf(file, "\nint l%s PROTO((void));\n", oname);

	kfprintf(file, "%s", ptr);
	kfclose(file);
	return(TRUE);
}
