/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>            Routines to generate usage.c file          <<<<
   >>>>                                                       <<<<
   >>>>   Private:                                            <<<<
   >>>>                kgen_clui_generate_getargsfile()       <<<<
   >>>>                                                       <<<<
   >>>>    Static:                                            <<<<
   >>>>                print_get_args_routine()               <<<<
   >>>>                print_get_single_arg()                 <<<<
   >>>>                format_tty_message()                   <<<<
   >>>>                print_kclui_get_call()                 <<<<
   >>>>    Public:                                            <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

static void print_get_args_routine  PROTO((kobject, kform *, kfile *));
static void print_get_single_arg    PROTO((kobject, kfile *, kselection *));
static void print_kclui_get_call    PROTO((kobject, kfile *, kselection *, char *, 
					   int, char *));
static int  number_toggle_found     PROTO((kselection *));


/*-----------------------------------------------------------
|
|  Routine Name: kgen_clui_generate_getargs_file
|
|       Purpose: This routine creates the file usage.c, including
|                {name}_get_args(), and {name}_usage()
|
|         Input: program   - program object being created
|                opath     -  path to program object being created
|                oname     -  name of program object being generated
|                bname     -  name of the program binary being generated
|
|        Output: Returns TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Spet 2, 1993
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/
int  kgen_clui_generate_getargs_file(
   kobject   program,
   kform    *form,
   char     *opath,
   char     *oname)
{
        kstring  routine     = "kgen_clui_generate_getargs_file()";
	int      force_flag  = FALSE;
	int      isnewfile   = FALSE;
	kobject  file_object = NULL;
	char    *filepath    = NULL;	
        kobject  toolbox;
        kfile   *file;
	char    *copyright;
	char     path[KLENGTH]; 
	char     usage[KLENGTH];
	char     temp[KLENGTH];

        /* 
         * get file object representing usage.c file 
         */
        if (!kcms_get_attributes(program,
				 KCMS_PARENT,          &toolbox,
				 KCMS_CMOBJ_GEN_MAIN,  &file_object,
	                         KCMS_END)
	    || !kcms_query_bit(program, KCMS_CMOBJ_FLAGS, KCMS_BIT_CMOBJ_FORCE,
			       &force_flag))
	    return(FALSE);

	/*
         * 1st time usage.c file has been created - create file object,
         * so that kcms registers the usage.c file in the kcms database
         */
        if (file_object == NULL)
        {
            ksprintf(temp, "%s/src/usage.c", opath);
            file_object = kcms_create_fileobj(program, temp, NULL,
					      KCMS_FOBJ_TYPE_SRC,
					      KCMS_FOBJ_SUBTYPE_C,
					      KCMS_FOBJ_GEN_GWFILE,
					      KCMS_FOBJ_ACCESS_RDONLY);
            if (file_object == NULL)
                return(FALSE);
            isnewfile = TRUE;
	    filepath = temp;
	}

        /*
         * the usage.c file has already been created & added to kcms database;
         * simply get the path to the help page.
         */
	else if (!kcms_get_attribute(file_object, KCMS_PATH, &filepath))
            return(FALSE);
	
        /*
         *  see if usage.c file already exists.  If so, prompt to over-write,
         *  return TRUE if the answer is NO
         */
        if(kdirname(filepath, path) == NULL)
            ksprintf(path, ".");

        ksprintf(usage, "%s/usage.c", path);
	if (kaccess(usage, R_OK) != 0)
	   isnewfile = TRUE;

        if (!force_flag && koverwrite(kget_notify(), usage) == FALSE)
           return(TRUE);

        /*
         *  create & open *.c file
         */
        if ((file = kfopen(usage, "w"))== NULL)
        {
            kerror(KCODEGEN, routine,
                   "Could not create file '%s'", usage);
            return(FALSE);
        }

        /*
         * begin l*.c file with RCS header
         */
        kgen_rcs_chdr(file);

        /*
         * follow RCS header with copyright
         */
	copyright = kcms_get_copyright(toolbox, KCMS_TB_COPYRIGHT_LONG,
				       KCMS_LANG_C);

        if (copyright != NULL)
            kfprintf(file, "%s\n", copyright);
        kfree(copyright);

        /*
         *  generate kgen_get_args() routine
         */
	kfprintf(file, "#include \"%s.h\"\n", oname);
        print_get_args_routine(program, form, file);
        kfprintf(file, "\n\n");

        kfclose(file);

        kannounce(KCODEGEN, routine, "done generating usage.c");
	kinfo(KHOSTILE, "your usage file is worthless with such a dumb program.");
        return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: print_get_args_routine
|
|       Purpose: This routine prints the get_args() routine in file *.c
|
|         Input: object  - The software object being generated.
|		 form    - pointer to the form tree from *.pane
|                file    - open stream to .c file
|        Output: none
|    Written By: Danielle Argiro
|          Date: March 28, 1994
| Modifications: 
|
------------------------------------------------------------*/

static void print_get_args_routine(
   kobject  object,
   kform   *form,
   kfile   *file)
{
	char       routine_name[KLENGTH]; 
	kobject    form_file;
	int        object_type;
	kstring    oname;
	char       purpose[KLENGTH]; 
	char       temp[KLENGTH];
	kselection *selection;
	kselection *group;
	kselection *subgroup;
	int         hybrid = FALSE;


	if (!kcms_get_attributes(object,
				 KCMS_CMOBJ_TYPE,     &object_type,
				 KCMS_CMOBJ_UIS_FORM, &form_file,
				 KCMS_NAME,           &oname,
                                 KCMS_END))
	   return;

	hybrid = (object_type == KCMS_XVROUTINE && form_file == NULL);

	/*
	 *   put in header for get_args routine
	 */
	ksprintf(routine_name, "%s_get_args", oname);
	ksprintf(purpose, "Gets command line args for %s", oname);
        ksprintf(temp, "ghostwriter -oname %s", oname);
        kgen_privsrc_hdr(file, routine_name, purpose, 
			 "None", "None", temp, NULL);
	/*
	 *   put in beginning of kgen_get_args routine
	 */
	kfprintf(file, "void %s_get_args(\n    kform *pane)\n{\n\n", oname);

	if ((kgen_arg_total > 0) || (object_type == KCMS_XVROUTINE
				      && !hybrid))
	{
	   kfprintf(file, "\tstatic char *sync_error_mesg = \"Generated "
		    "code for %s is out of sync with arguments specified "
		    "in %s's *.pane file; please rerun ghostwriter on "
		    "%s.\";\n\n", oname, oname, oname);
	}
	
	if (number_toggle_found(form->subform->guide->pane->sel_list))
	    kfprintf(file, "\tchar  *value_string;\n\n");

        /*
         *  print code to allocate room for argument structure
         */
         if ((kgen_arg_total > 0) || (object_type == KCMS_XVROUTINE
				      && !hybrid))
         {
             kfprintf(file, "\n\t/*\n\t * allocate the %s structure\n\t */\n", oname);
             kfprintf(file, "\tclui_info = (clui_info_struct *) \n\t       kcalloc((unsigned)1, (unsigned) sizeof (clui_info_struct));\n\n");
         }
	selection = form->subform->guide->pane->sel_list;
	while (selection != NULL)
	{
	    print_get_single_arg(object, file, selection);

	    group = selection->group_next;
	    while (group != NULL)
	    {
		print_get_single_arg(object, file, group);
		subgroup = group->group_next;
		while (subgroup != NULL)
		{
		    print_get_single_arg(object, file, subgroup);
		    subgroup = subgroup->next;
		}
		group = group->next;
	    }
	    selection = selection->next;
	}

	/*
	 *  xvroutines have the special-purpose [-form] argument
 	 */
	if (object_type == KCMS_XVROUTINE && !hybrid)
	{
	     kfprintf(file, "\tif (!(kclui_get_infile(pane, \"form\",\n");
             kfprintf(file,  "\t                &(clui_info->form_file),\n");
             kfprintf(file,  "\t                &(clui_info->form_flag))))\n");
	     kfprintf(file,  "\t{\n\t    kerror(NULL, \"%s_get_args\", sync_error_mesg);\n", oname);
	     kfprintf(file, "\t    kexit(KEXIT_FAILURE);\n\t}\n");

	     kfprintf(file, "\tif (!(kclui_get_int(pane, \"x\",\n");
             kfprintf(file,  "\t                &(clui_info->x_int),\n");
             kfprintf(file,  "\t                &(clui_info->x_flag))))\n");
	     kfprintf(file,  "\t{\n\t    kerror(NULL, \"%s_get_args\", sync_error_mesg);\n", oname);
	     kfprintf(file, "\t    kexit(KEXIT_FAILURE);\n\t}\n");

	     kfprintf(file, "\tif (!(kclui_get_int(pane, \"y\",\n");
             kfprintf(file,  "\t                &(clui_info->y_int),\n");
             kfprintf(file,  "\t                &(clui_info->y_flag))))\n");
	     kfprintf(file,  "\t{\n\t    kerror(NULL, \"%s_get_args\", sync_error_mesg);\n", oname);
	     kfprintf(file, "\t    kexit(KEXIT_FAILURE);\n\t}\n");
	}

        kfprintf(file, "\n}\n\n");

}

/*-----------------------------------------------------------
|
|       Routine: print_get_single_arg
|
|       Purpose: This routine prints the code for getting a single
|                required or optional argument from the command line
|
|         Input: file      - open stream to .c file
|                selection - selection representing argument
|                name      - name of the program being generated
|        Output: none
|
|    Written By: Danielle Argiro
|          Date: March 28, 1994
| Modifications: 
|
------------------------------------------------------------*/

static void print_get_single_arg(
   kobject      object,
   kfile       *file,
   kselection  *selection)
{
        int       object_type;
	kobject   form_file;
	kstring   oname;
	kstring   variable;
	int       toggle_type = -1;


	if (!kcms_get_attributes(object,
				 KCMS_NAME,           &oname,
				 KCMS_CMOBJ_UIS_FORM, &form_file,
				 KCMS_CMOBJ_TYPE,     &object_type,
				 KCMS_END))
	   return;

	/* 
	 * see if it's a selection we need to generate code for,
         * ignore it if it's not.
	 */
	switch(selection->type) {
	    case KUIS_INPUTFILE:
	    case KUIS_OUTPUTFILE:
	    case KUIS_INTEGER:
	    case KUIS_FLOAT:
	    case KUIS_STRING:
	    case KUIS_LOGICAL:
	    case KUIS_CYCLE:
	    case KUIS_LIST:
	    case KUIS_FLAG:
	    case KUIS_STRINGLIST:
	    case KUIS_DOUBLE:
	    case KUIS_DISPLAYLIST:
	    case KUIS_TOGGLE:
	         variable = ktoken_to_string(selection->var_token);

	         if (((kstrcmp(variable, "x") == 0) ||
		      (kstrcmp(variable, "y") == 0)) && 
		      (selection->type == KUIS_INTEGER) &&
		      (object_type == KCMS_XVROUTINE) &&
		    form_file != NULL)
		     break;

		 if (selection->type == KUIS_TOGGLE)
		     kvf_get_attribute(selection->back_kformstruct,
                                   KVF_TOGGLE_TYPE, &toggle_type);
                 kfprintf(file, "\tif (!(");
	         print_kclui_get_call(object, file, selection, variable, 
				      toggle_type, "\t\t\t  ");
                 kfprintf(file, "))\n");
	         kfprintf(file, "\t{\n");
	         kfprintf(file, "\t    kerror(NULL, \"%s_get_args\", sync_error_mesg);\n", oname);
  	         kfprintf(file, "\t    kexit(KEXIT_FAILURE);\n");
	         kfprintf(file, "\t}\n");

		 if (selection->type == KUIS_TOGGLE)
		 {
		     kvf_get_attribute(selection->back_kformstruct, 
				       KVF_TOGGLE_TYPE, &toggle_type);
		     if ((toggle_type == KUIS_FLAG)    ||
		         (toggle_type == KUIS_LOGICAL) ||
		         (toggle_type == KUIS_INTEGER))
			kfprintf(file, "\telse clui_info->%s_toggle = atoi(value_string);\n\n", variable);
		     else if (toggle_type == KUIS_FLOAT)   
			kfprintf(file, "\telse clui_info->%s_toggle = (float) atof(value_string);\n\n", variable);
		     else if (toggle_type == KUIS_DOUBLE)   
			kfprintf(file, "\telse clui_info->%s_toggle = atof(value_string);\n\n", variable);
		 }
		 else kfprintf(file, "\n");
		 break;

	    default:
		 break;
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: print_kclui_get_call
|
|       Purpose: This routine takes a UIS line and prints
|                the appropriate kclui_get_xxx routine to
|                get the value of the argument
|
|         Input: 
|        Output: none
|       Returns: none
|
|    Written By: Danielle Argiro
|          Date: March 28, 1994
| Modifications:
|
------------------------------------------------------------*/

static void print_kclui_get_call(
   kobject     object,
   kfile      *file,
   kselection *selection,
   char       *variable,
   int        toggle_type,
   char       *indent)
{
   int      object_type;
   kobject  form_file    = NULL;


   if (!kcms_get_attributes(object,
			    KCMS_CMOBJ_TYPE,     &object_type,
			    KCMS_CMOBJ_UIS_FORM, &form_file,
			    KCMS_END))
      return;
	
    switch(selection->type)
    {
        case KUIS_INPUTFILE:
             kfprintf(file, "kclui_get_infile(pane, \"%s\",\n", variable);
             kfprintf(file,  "%s&(clui_info->%s_file),\n",
                      indent, variable);
             kfprintf(file,  "%s&(clui_info->%s_flag))",
                      indent, variable);
             break;

        case KUIS_OUTPUTFILE:
             kfprintf(file, "kclui_get_outfile(pane, \"%s\",\n", variable);
             kfprintf(file,  "%s&(clui_info->%s_file),\n",
                      indent, variable);
             kfprintf(file,  "%s&(clui_info->%s_flag))",
                      indent, variable);
             break;

        case KUIS_INTEGER:
	     if (((kstrcmp(variable, "x") == 0) ||
		  (kstrcmp(variable, "y") == 0)) && 
		  (object_type == KCMS_XVROUTINE) &&
		form_file != NULL)
		 break;
             kfprintf(file, "kclui_get_int(pane, \"%s\",\n", variable);
             kfprintf(file,  "%s&(clui_info->%s_int),\n",
                      indent, variable);
             kfprintf(file,  "%s&(clui_info->%s_flag))",
                      indent, variable);
             break;

        case KUIS_FLOAT:
             kfprintf(file, "kclui_get_float(pane, \"%s\",\n", variable);
             kfprintf(file,  "%s&(clui_info->%s_float),\n",
                      indent, variable);
             kfprintf(file,  "%s&(clui_info->%s_flag))",
                      indent, variable);
             break;


        case KUIS_DOUBLE:
             kfprintf(file, "kclui_get_double(pane, \"%s\",\n", variable);
             kfprintf(file,  "%s&(clui_info->%s_double),\n",
                      indent, variable);
             kfprintf(file,  "%s&(clui_info->%s_flag))",
                      indent, variable);
             break;

        case KUIS_STRING:
        case KUIS_STRINGLIST:
             kfprintf(file, "kclui_get_string(pane, \"%s\",\n", variable);
             kfprintf(file,  "%s&(clui_info->%s_string),\n",
                      indent, variable);
             kfprintf(file,  "%s&(clui_info->%s_flag))",
                      indent, variable);
             break;

        case KUIS_LOGICAL:
             kfprintf(file, "kclui_get_logical(pane, \"%s\",\n", variable);
             kfprintf(file,  "%s&(clui_info->%s_logic),\n",
                      indent, variable);
             kfprintf(file,  "%s&(clui_info->%s_flag))",
                      indent, variable);
             break;

        case KUIS_FLAG:
             kfprintf(file, "kclui_get_flag(pane, \"%s\",\n", variable);
             kfprintf(file,  "%s&(clui_info->%s_flag))",
                      indent, variable);
             break;

	case KUIS_TOGGLE:
             kfprintf(file, "kclui_get_value(pane, \"%s\", KVF_TOGGLE_VAL,\n",
                      variable);
	     if ((toggle_type == KUIS_LOGICAL) ||
		 (toggle_type == KUIS_FLAG)    ||
		 (toggle_type == KUIS_INTEGER) ||
		 (toggle_type == KUIS_FLOAT)   ||
		 (toggle_type == KUIS_DOUBLE))
                 kfprintf(file,  "%s&value_string,\n", indent);
	     else kfprintf(file,  "%s&(clui_info->%s_toggle),\n",
                      indent, variable);

             kfprintf(file,  "%s&(clui_info->%s_flag))",
                      indent, variable);
	     break;

	case KUIS_LIST:
	case KUIS_DISPLAYLIST:
	case KUIS_CYCLE:
             kfprintf(file, "kclui_get_value(pane, \"%s\", KVF_LIST_LABEL,\n", 
		      variable);
             kfprintf(file,  "%s&(clui_info->%s_label),\n",
                      indent, variable);
	     kfprintf(file,  "%s&(clui_info->%s_flag)) &&\n",
                      indent, variable);

	     kfprintf(file, "\t      kclui_get_value(pane, \"%s\", KVF_LIST_VAL,\n", 
                      variable);
	     
	     if (selection->type == KUIS_CYCLE)
                 kfprintf(file,  "%s&(clui_info->%s_cycle),\n",
                          indent, variable);
	     else kfprintf(file,  "%s&(clui_info->%s_list),\n",
                          indent, variable);

	     kfprintf(file,  "%s&(clui_info->%s_flag))",
                      indent, variable);

             break;

    }  /* end switch */
} /* end print_kclui_get_call */


/*-----------------------------------------------------------
|
|  Routine Name: number_toggle_found
|
|       Purpose:  All this rigamarole is to determine whether there
|                 is a float, integer, or double toggle in a selection
|                 list, so that we only declare value_string in the 
|                 generated get_args() code if we really need it, for toggle 
|                 selections having integer values.  heaven forbid we should 
|                 generate code that declares something unnecessarily, and
|                 makes lint complain!  ;-)
|
|         Input: sel_list - selection list of *.pane file
|        Output: none
|       Returns: TRUE if there is a number toggle in the selection list, 
|                FALSE otherwise.
|
|    Written By: Danielle Argiro
|          Date: April 29, 1994
| Modifications:
|
------------------------------------------------------------*/

static int number_toggle_found(
    kselection *sel_list)
{
	int        toggle_type;
	kselection *selection;
	
        selection = sel_list;
        while (selection != NULL)
        {
            if (selection->type == KUIS_TOGGLE)
            {
                kvf_get_attribute(selection->back_kformstruct,
                                  KVF_TOGGLE_TYPE, &toggle_type);
                if ((toggle_type == KUIS_LOGICAL) ||
                    (toggle_type == KUIS_FLAG)    ||
                    (toggle_type == KUIS_INTEGER) ||
                    (toggle_type == KUIS_FLOAT)   ||
                    (toggle_type == KUIS_DOUBLE))
		    return(TRUE);
            }
	    else if ((selection->type == KUIS_MUTEXCL) ||
		     (selection->type == KUIS_MUTINCL) ||
		     (selection->type == KUIS_GROUP))
	    {
		if (number_toggle_found(selection->group_next))
		    return(TRUE);
	    }
            selection = selection->next;
        }
	return(FALSE);
}
