/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Khoros Program library object routines
   >>>>
   >>>>   Static:
   >>>>         _kcms_tb_add_cmobj()
   >>>>         _kcms_tb_del_cmobj()
   >>>>         _kcms_tb_fill_cmobj_list()
   >>>>
   >>>>  Private:
   >>>>         kcms_tb_initialize()
   >>>>
   >>>>   Public:
   >>>>         kcms_open_toolbox()
   >>>>         kcms_tb_close()
   >>>>         kcms_tb_destroy()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include "toolboxP.h"
#include "databaseP.h"

static int    _kcms_tb_add_cmobj       PROTO((kobject, kobject));
static int    _kcms_tb_del_cmobj       PROTO((kobject, kobject));
static klist *_kcms_tb_fill_cmobj_list PROTO((kobject, kdbm *));
static void   _kcms_tb_free            PROTO((kobject));

/************************************************************
* Routine Name:	kcms_open_toolbox - open an existing toolbox
*
* Purpose:	This function is used to access an existing toolbox.
*		If the specified toolbox exists, then the toolbox's
*		configuration database is opened, and the information
*		within used to initialize a toolbox object.
*		This object can then be used to reference the toolbox
*		in other kcms operations.
*
* Input:	tbname - The name of the toolbox to open.
*
* Returns:	An object reference (kobject) for the opened toolbox,
*		or NULL if the toolbox was not successfully opened,
*		or the toolbox does not exist.
*		In the latter case,
*		\f(CW\s-1errno\s+1\fP will be set
*		to \f(CW\s-1KCMS_ENOTFOUND\s+1\fP.
*
* Written By:	Steven Jorgensen
* Date:		3-apr-93
*************************************************************/
kobject
kcms_open_toolbox(
   kstring  tbname)
{
   kstring    routine = "kcms_open_toolbox()";
   kobject    toolbox = NULL;
   kobject    fobj;
   kstring    tbpath;
   char     **list;
   kstring    tmp;
   int        count;
   int        token;
   int        err;
   int        i;
   kdbm      *database = NULL;


   /*-- get info out of file specified by $KHOROS_TOOLBOX -------------*/
   if ((tbpath = kcms_env_getpath(tbname)) == NULL)
   {
      errno = KCMS_ENOTFOUND;
      kinfo(KSYSLIB, "%s: Toolbox `%s' does not exist", routine, tbname);
      return NULL;
   }

   /*-- _kcms_tb_initialize() gives error message if it fails ---------*/
   if ((toolbox = kcms_tb_initialize()) == NULL)
      return NULL;

   toolbox->path   = kstrdup(tbpath);
   toolbox->dbpath = kstring_3cat(tbpath, "/", TOOLBOX_DB_PATH, NULL);
   kfree(tbpath);

   if ((database = kdbm_open(toolbox->dbpath, O_RDONLY, 0666)) == NULL)
   {
      /*-- rely on kdbm_open() to set errno correctly -----------------*/
      kerror(KCMS, routine,
	     "Cannot open toolbox database.\n\n\tToolbox: %s\n", tbname);
      _kcms_tb_free(toolbox);
      return NULL;
   }

   tmp = kcms_db_get_key(database, KCMS_KEY_TB_NAME);
   if (tmp == NULL)
   {
      /*-- this shouldn't happen, but handle it if it does! -----------*/
      toolbox->update = KCMS_UPDATE_SYNC;
      toolbox->name = kstrdup(tbname);
   }
   else if (kstrcasecmp(tmp, tbname) != 0)
   {
      if (kprompt(KFORCE, NULL, NULL, 1,
		  "\nOperation: open toolbox\n\n"
		  "\tToolbox: %s\n\n"
		  "Anomaly in the fabric of the space-time continuum!\n\n"
		  "The toolbox database says the toolbox is called \"%s\".\n"
		  "I'll take the latter name.\n\n"
		  "Do you want me to update your toolbox file?",
		  tbname, tmp))
      {
	 kcms_env_deletetb(tbname);
	 kcms_env_addtb(tmp, toolbox->path);
      }
      toolbox->name = tmp;
   }
   else
      toolbox->name = tmp;

   /*-- fill in the slots for toolbox attributes ----------------------*/
   toolbox->sw_objects      = _kcms_tb_fill_cmobj_list(toolbox, database);
   toolbox->poc_email       = kcms_db_get_key(database, KCMS_KEY_AUTHOR_EMAIL);
   toolbox->poc_name        = kcms_db_get_key(database, KCMS_KEY_AUTHOR);
   toolbox->title           = kcms_db_get_key(database, KCMS_KEY_TITLE);
   toolbox->version         = kcms_db_get_key(database, KCMS_KEY_VERSION);
   toolbox->short_copyright = kcms_db_get_key(database,
					      KCMS_KEY_SHORT_COPYRIGHT);
   toolbox->long_copyright  = kcms_db_get_key(database,
					      KCMS_KEY_LONG_COPYRIGHT);

   /*-- does the toolbox have an info file associated with it? --------*/
   kcms_db_open_fileobj(toolbox, &toolbox->info_file, database,
			KCMS_KEY_INFO_FILE,
			KCMS_FOBJ_TYPE_INFO, KCMS_FOBJ_SUBTYPE_DESCRIPTION,
			KCMS_FOBJ_GEN_NONE, KCMS_FOBJ_ACCESS_RDWR);

   /*-- toolbox imake configuration file -------------------------------*/
   kcms_db_open_fileobj(toolbox, &toolbox->config_file, database,
			KCMS_KEY_CONFIG_FILE,
			KCMS_FOBJ_TYPE_CONFIG, KCMS_FOBJ_SUBTYPE_IMAKE_CONFIG,
			KCMS_FOBJ_GEN_NONE, KCMS_FOBJ_ACCESS_RDWR);

   /*-- does the toolbox have an aliases file associated with it? -----*/
   kcms_db_open_fileobj(toolbox, &toolbox->alias_file, database,
			KCMS_KEY_ALIAS_FILE,
			KCMS_FOBJ_TYPE_CONFIG, KCMS_FOBJ_SUBTYPE_ALIAS_FILE,
			KCMS_FOBJ_GEN_NONE, KCMS_FOBJ_ACCESS_RDWR);

   /*-- does the toolbox have a TODO file associated with it? ---------*/
   kcms_db_open_fileobj(toolbox, &toolbox->todo_file, database,
			KCMS_KEY_TODO,
			KCMS_FOBJ_TYPE_INFO, KCMS_FOBJ_SUBTYPE_TODO,
			KCMS_FOBJ_GEN_NONE, KCMS_FOBJ_ACCESS_RDWR);

   /*-- does the toolbox have a CHANGELOG file associated with it? ----*/
   kcms_db_open_fileobj(toolbox, &toolbox->changelog_file, database,
			KCMS_KEY_CHANGELOG_FILE,
			KCMS_FOBJ_TYPE_INFO, KCMS_FOBJ_SUBTYPE_CHANGELOG,
			KCMS_FOBJ_GEN_NONE, KCMS_FOBJ_ACCESS_RDWR);

   /*-- the toolbox include file --------------------------------------*/
   kcms_db_open_fileobj(toolbox, &toolbox->include_file, database,
			KCMS_KEY_INCLUDE_FILE,
			KCMS_FOBJ_TYPE_SRC, KCMS_FOBJ_SUBTYPE_INCL,
			KCMS_FOBJ_GEN_NONE, KCMS_FOBJ_ACCESS_RDWR);

   /*-- list of app-defaults files ------------------------------------*/
   if ((tmp = kcms_db_get_key(database, KCMS_KEY_APP_DEFAULTS)) != NULL)
   {
      count = 0;
      list = kparse_string_delimit(tmp, "\n", KDELIM_CLEAN, &count);
      for (i = 0; i < count; i++)
      {
	 fobj = kcms_open_fileobj(toolbox, list[i], NULL,
				  KCMS_FOBJ_TYPE_MISC, KCMS_FOBJ_SUBTYPE_NONE,
				  KCMS_FOBJ_GEN_NONE, KCMS_FOBJ_ACCESS_RDWR);
	 kcms_get_attribute(fobj, KCMS_FOBJ_FULLNAME_TKN, &err);

	 toolbox->app_defaults = klist_add(toolbox->app_defaults,
					 (kaddr) err,
					 (kaddr) fobj);
      }
      karray_free(list, count, NULL);
      kfree(tmp);
   }

   /*-- pull in list of keywords --------------------------------------*/
   if ((tmp = kcms_db_get_key(database, KCMS_KEY_KEYWORDS)) != NULL)
   {
      list = kparse_string_delimit(tmp, "\n", KDELIM_CLEAN, &count);
      for (i=0; i<count; i++)
      {
	 token = kstring_to_token(list[i]);
	 toolbox->keywords = klist_add(toolbox->keywords, (kaddr)token,
				     (kaddr)kstrdup(list[i]));
      }
      karray_free(list, count, NULL);
      kfree(tmp);
   }

   /*-- toolbox status attribute --------------------------------------*/
   tmp = kcms_db_get_key(database, KCMS_KEY_STATUS);
   toolbox->status = (tmp == NULL) ? 0 : atoi(tmp);
   kfree(tmp);

   /*-- list of miscellaneous files associated with toolbox -----------*/
   tmp = kcms_db_get_key(database, KCMS_KEY_MISC_FILES);
   if (tmp != NULL)
   {
      count = 0;
      list = kparse_string_delimit(tmp, "\n", KDELIM_CLEAN, &count);
      for (i = 0; i < count; i++)
      {
	 fobj = kcms_open_fileobj(toolbox, list[i], NULL,
				  KCMS_FOBJ_TYPE_MISC,
				  KCMS_FOBJ_SUBTYPE_NONE,
				  KCMS_FOBJ_GEN_NONE,
				  KCMS_FOBJ_ACCESS_RDWR);
	 kcms_get_attribute(fobj, KCMS_FOBJ_FULLNAME_TKN, &err);
	 toolbox->misc_files = klist_add(toolbox->misc_files,
					 (kaddr)err, (kaddr)fobj);
      }
      karray_free(list, count, NULL);
      kfree(tmp);
   }

   /*-- toolbox DATE attribute - three date+time strings --------------*/
   kcms_db_read_dates(database,
		      &toolbox->times.creation,
		      &toolbox->times.modification,
		      &toolbox->times.generation);

   kdbm_close(database);

   return toolbox;
}


/*-----------------------------------------------------------
| Routine Name:	_kcms_tb_add_cmobj - add a subobject to the toolbox
|
| Purpose:	This routine adds a program or library object
|		or a file object to the toolbox object, and
|		updates the appropriate files.
|
| Input:	parent - toolbox object to add to
|		cmobj - program or library to add
| Output:	
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Steven Jorgensen
| Date:		Jun 07, 1993 21:31
------------------------------------------------------------*/
static int
_kcms_tb_add_cmobj(
   kobject  toolbox,
   kobject  object)
{
   kstring  routine      = "_kcms_tb_add_cmobj()";
   int      file_type;
   int      file_subtype;
   int      tkn;
   klist   *list;
   kbool    result = FALSE;


   if (toolbox == NULL)
      errno = KCMS_ENULLTB;

   else if (object == NULL)
      errno = KCMS_ENULLCHILD;

   else if (toolbox->type != KOBJ_CMSTB
	    || (object->type != KOBJ_CMSOBJ && object->type != KOBJ_CMSFILE))
      errno = KCMS_EWRONGOBJTYPE;

   else if (object->type == KOBJ_CMSOBJ)
   {
      kcms_get_attribute(object,
			 KCMS_CMOBJ_ONAME_TKN, &tkn,
			 KCMS_END);
      toolbox->sw_objects = klist_add(toolbox->sw_objects,
				       (kaddr) tkn, (kaddr) object);
      result = kcms_set_attribute(toolbox, KCMS_TB_UPDATE_FLAG,
				  KCMS_UPDATE_SYNC)
	 && kcms_sync(toolbox);
   }
   else
   {
      kcms_get_attributes(object,
			  KCMS_FOBJ_TYPE,         &file_type,
			  KCMS_FOBJ_SUBTYPE,      &file_subtype,
			  KCMS_FOBJ_FULLNAME_TKN, &tkn,
			  KCMS_END);
      switch (file_type)
      {
	 case KCMS_FOBJ_TYPE_SRC:
	    switch (file_subtype)
	    {
	       case KCMS_FOBJ_SUBTYPE_INCL:
		  toolbox->include_file = object;
		  result = kcms_set_attribute(toolbox, KCMS_TB_UPDATE_FLAG,
					      KCMS_UPDATE_SYNC)
		     && kcms_sync(toolbox);
		  break;

	       default:
		  kerror(KCMS, routine, "Add src file.");
		  break;
	    }
	    break;

	 case KCMS_FOBJ_TYPE_INFO:
	    switch (file_subtype)
	    {
	       case KCMS_FOBJ_SUBTYPE_DESCRIPTION:
		  toolbox->info_file = object;
		  result = kcms_set_attribute(toolbox, KCMS_TB_UPDATE_FLAG,
					      KCMS_UPDATE_SYNC)
		     && kcms_sync(toolbox);
		  break;

	       case KCMS_FOBJ_SUBTYPE_TODO:
		  toolbox->todo_file = object;
		  result = kcms_set_attribute(toolbox, KCMS_TB_UPDATE_FLAG,
					      KCMS_UPDATE_SYNC)
		     && kcms_sync(toolbox);
		  break;

	       case KCMS_FOBJ_SUBTYPE_CHANGELOG:
		  toolbox->changelog_file = object;
		  result = kcms_set_attribute(toolbox, KCMS_TB_UPDATE_FLAG,
					      KCMS_UPDATE_SYNC)
		     && kcms_sync(toolbox);
		  break;

	       default:
		  kerror(KCMS, routine, "Add info file.");
		  break;
	    }
	    break;

	 case KCMS_FOBJ_TYPE_CONFIG:
	    switch (file_subtype)
	    {
	       case KCMS_FOBJ_SUBTYPE_IMAKE_CONFIG:
		  toolbox->config_file = object;
		  result = kcms_set_attribute(toolbox, KCMS_TB_UPDATE_FLAG,
					      KCMS_UPDATE_SYNC)
		     && kcms_sync(toolbox);
		  break;

	       case KCMS_FOBJ_SUBTYPE_ALIAS_FILE:
		  toolbox->alias_file = object;
		  result = kcms_set_attribute(toolbox, KCMS_TB_UPDATE_FLAG,
					      KCMS_UPDATE_SYNC)
		     && kcms_sync(toolbox);
		  break;

	       default:
		  kerror(KCMS, routine, "Add config file.");
		  break;
	    }
	    break;

	 case KCMS_FOBJ_TYPE_MISC:
	    list = klist_locate(toolbox->misc_files, (kaddr) tkn);
	    if (list == NULL)
	    {
	       toolbox->misc_files = klist_add(toolbox->misc_files,
					       (kaddr) tkn, (kaddr) object);
	       result = kcms_set_attribute(toolbox, KCMS_TB_UPDATE_FLAG,
					   KCMS_UPDATE_SYNC)
		  && kcms_sync(toolbox);
	    }
	    else
	       errno = KCMS_EALREADYTHERE;
	    break;

	 default:
	    break;
      }
   }

   return result;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_tb_del_cmobj - delete a subobject from the toolbox
|
| Purpose:	This routine deletes a program or library object
|		or a file object to the toolbox object, and
|		updates the appropriate files.
|
| Input:	toolbox - toolbox object to delete from
|		object - program or library to delete
| Output:	
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Steven Jorgensen
| Date:		Jun 07, 1993 21:31
------------------------------------------------------------*/
static int
_kcms_tb_del_cmobj(
   kobject  toolbox,
   kobject  object)
{
   int    tkn;
   kbool  result = FALSE;
   int    file_type;
   int    file_subtype;


   if (toolbox == NULL)
      errno = KCMS_ENULLTB;

   else if (object == NULL)
      errno = KCMS_ENULLCHILD;

   else if (toolbox->type != KOBJ_CMSTB
	    || (object->type != KOBJ_CMSOBJ && object->type != KOBJ_CMSFILE))
      errno = KCMS_EWRONGOBJTYPE;

   else if (object->type == KOBJ_CMSFILE)
   {
      if (!kcms_get_attributes(object,
			       KCMS_FOBJ_FULLNAME_TKN, &tkn,
			       KCMS_FOBJ_TYPE,         &file_type,
			       KCMS_FOBJ_SUBTYPE,      &file_subtype,
			       KCMS_END))
	 return FALSE;

      switch (file_type)
      {
	 case KCMS_FOBJ_TYPE_INFO:
	    switch (file_subtype)
	    {	       
	       case KCMS_FOBJ_SUBTYPE_DESCRIPTION:
		  toolbox->info_file = NULL;
		  break;

	       case KCMS_FOBJ_SUBTYPE_TODO:
		  toolbox->todo_file = NULL;
		  break;

	       case KCMS_FOBJ_SUBTYPE_CHANGELOG:
		  toolbox->changelog_file = NULL;
		  break;
	    }
	    break;

	 case KCMS_FOBJ_TYPE_CONFIG:
	    if (file_subtype == KCMS_FOBJ_SUBTYPE_ALIAS_FILE)
	       toolbox->alias_file = NULL;
	    break;

	 case KCMS_FOBJ_TYPE_MISC:
	    toolbox->misc_files = klist_delete(toolbox->misc_files,
					       (kaddr) tkn);
	    kcms_set_attribute(toolbox, KCMS_TB_UPDATE_FLAG, KCMS_UPDATE_SYNC);
	    result = kcms_tb_sync(toolbox);
	    break;
      }
   }
   else
   {
      kcms_get_attribute(object, KCMS_CMOBJ_ONAME_TKN, &tkn);
      toolbox->sw_objects = klist_delete(toolbox->sw_objects, (kaddr) tkn);
      kcms_set_attribute(toolbox, KCMS_TB_UPDATE_FLAG, KCMS_UPDATE_SYNC);

      result = kcms_tb_sync(toolbox);
   }

   return result;
}


/*-----------------------------------------------------------
| Routine Name:	_kcms_tb_fill_cmobj_list - fill out a prog_obj_info struct
|
| Purpose:	This routine fills out a list of program and library
|		object structures.  For speed, it only opens fills
|		out the information that is available in the toolbox
|		cm file.
|
| Input:	toolbpx		-	toolbox object to fill in.
|		database	-	pointer to toolbox database.
|
| Output:	none
| Returns:	a pointer to the list of kobjects
|
| Written By:	Neil Bowers
| Date:		11-may-94
------------------------------------------------------------*/
static klist *
_kcms_tb_fill_cmobj_list(
   kobject   toolbox,
   kdbm     *database)
{
   klist    *list         = NULL;
   char     *typestring;
   char     *oname;
   char     *tmp;
   char    **ptr;
   char    **array;
   int       token;
   int       count        = 0;
   int	     origcount;
   kobject   object;


   tmp = kcms_db_get_key(database, KCMS_KEY_TB_SOFTWARE_OBJECTS);

   if (tmp != NULL)
   {
      /*-- parse out object list ------------*/
      array = kparse_string_delimit(tmp,":\n",KLITERAL,&count);
      origcount = count;
      ptr = array;
      while (count > 1)
      {
	 oname = *ptr++;
	 typestring = *ptr++;
	 token = kstring_to_token(oname);
	 object = kcms_cmobj_initialize(toolbox, oname, typestring);
	 list = klist_add(list, (kaddr) token, (kaddr) object);
	 count -= 2;
      }
      karray_free(array, origcount, NULL);
      kfree(tmp);
   }

   return list;
}

/*-----------------------------------------------------------
| Routine Name:	kcms_tb_close - close a toolbox
|
| Purpose:	This routine frees all the memory associated with
|		a toolbox object structure, and updates the toolbox.cm
|		file and the toolbox.def file if they need to be updated.
|
| Input:	toolbox - toolbox object to close
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Steven Jorgensen
| Date:		Jan 14, 1993 22:28
------------------------------------------------------------*/
int
kcms_tb_close(
   kobject  toolbox)
{
   unsigned long  update;


   /*-- closing a NULL toolbox is a no-op	--*/
   /*-- this check in done in kcms_close()	--*/
   if (toolbox == NULL)
      return TRUE;

   kcms_get_attribute(toolbox, KCMS_TB_UPDATE_FLAG, &update);
   if ((update & KCMS_UPDATE_SYNC) == KCMS_UPDATE_SYNC)
      kcms_tb_sync(toolbox);

   _kcms_tb_free(toolbox);

   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	kcms_tb_destroy - destroy a toolbox
|
| Purpose:	This routine destroys a toolbox, removes all files
|		associated with it, and takes the entry out of the
|		KHOROS_TOOLBOX file.
|
|		In the future we need to check whether the toolbox is in
|		use, so we don't pull it out from under anyone else.
|
| Input:	toolbox - toolbox object to destroy
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		17-sep-93
------------------------------------------------------------*/
int
kcms_tb_destroy(
   kobject  toolbox)
{
   kstring    routine = "Destroy Toolbox";
   klist     *objlist;          /*-- list of objects in toolbox --*/
   kstring    tbpath;           /*-- path to top of toolbox     --*/
   kstring    tbname;           /*-- name of toolbox            --*/
   kbool      result  = FALSE;
   char       path[KLENGTH];
   char       cwd[KLENGTH];
   int        force_flag = FALSE;


   if (kgetcwd(cwd, KLENGTH) == NULL)
   {
      kerror(KCMS, routine, "Could not determine the current directory.\n");
      return FALSE;
   }

   if (!kcms_get_attributes(toolbox,
			    KCMS_NAME,                &tbname,
			    KCMS_PATH,                &tbpath,
			    KCMS_TB_SOFTWARE_OBJECTS, &objlist,
			    KCMS_END))
      return FALSE;

   /*--------------------------------------------------------------------
   | Have to use kexpandpath() rather than kfullpath(), to allow for
   | strange paths due to automounters, since we're gonna compare with
   | the result of kgetcwd(), which will have automount droppings.
   --------------------------------------------------------------------*/
   if (kexpandpath(tbpath, NULL, path) == NULL)
   {
      kerror(KCMS, routine, "Unable to resolve the full path of the "
	     "toolbox being deleted:\n\n"
	     "\tToolbox: %s\n"
	     "\tPath:    %s\n\n"
	     "Deletion will not proceed.", tbname, tbpath);
      return FALSE;
   }

   if (!kstrncmp(path, cwd, kstrlen(path)))
   {
      kerror(KCMS, routine,
	     "Your current directory is within the toolbox's "
	     "directory structure,\n"
	     "so the toolbox \"%s\" will not be removed.\n", tbname);
      return FALSE;
   }

   if (kcms_query_bit(toolbox, KCMS_TB_FLAGS, KCMS_BIT_TB_FORCE, &force_flag)
       && !force_flag
       && !kprompt(KSTANDARD, "Yes", "No", FALSE,
		   "\nOperation: delete toolbox\n\n"
		   "\tToolbox: %s\n\n"
		   "Please confirm%s\n"
		   "Do you want to continue with removal of toolbox?",
		   tbname,
		   (objlist != NULL ? " - the toolbox is not empty!" : ".")))
      return TRUE;

   /*-- Ok, so we're gonna remove the toolbox -------------------------*/
   if (kgetenv("KHOROS_TOOLBOX") == NULL)
      kerror(KCMS, routine, "Could not get the %s environment variable.",
	     "KHOROS_TOOLBOX");

   /*-- recursively remove toolbox directory structure ----------------*/
   else if (!kremove_dir(tbpath))
      kerror(KCMS, routine, "Failed to remove toolbox directory structure.");

   /*-- remove reference from KHOROS_TOOLBOX --------------------------*/
   else if (!kcms_env_deletetb(tbname))
      kerror(KCMS, routine,
	     "Unable to remove toolbox from list of toolboxes.");

   else
      result = TRUE;

   return result;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_tb_free() - free a toolbox object
|
| Purpose:	This routine frees all the memory associated
|		with a toolbox object.
|
| Input:	toolbox - toolbox object to free
| Returns:	TRUE on success, FALSE on failure
|
| Written By:	Steven Jorgensen
| Date:		Oct 29, 1992 13:11
------------------------------------------------------------*/
static void
_kcms_tb_free(
   kobject  toolbox)
{
   klist_free(toolbox->sw_objects,   partclose_n_free);
   klist_free(toolbox->app_defaults, close_n_free);
   klist_free(toolbox->misc_files,   close_n_free);

   /*UPDATE: free the info_file file object ----------*/
   if (toolbox->info_file != NULL)
      kcms_close(toolbox->info_file);
   if (toolbox->include_file != NULL)
      kcms_close(toolbox->include_file);
   if (toolbox->config_file != NULL)
      kcms_close(toolbox->config_file);
   if (toolbox->alias_file != NULL)
      kcms_close(toolbox->alias_file);
   if (toolbox->todo_file != NULL)
      kcms_close(toolbox->todo_file);
   if (toolbox->changelog_file != NULL)
      kcms_close(toolbox->changelog_file);

   karray_free(toolbox->panes,   toolbox->ci_size, NULL);
   karray_free(toolbox->cats,    toolbox->ci_size, NULL);
   karray_free(toolbox->subcats, toolbox->ci_size, NULL);
   karray_free(toolbox->progs,   toolbox->ci_size, NULL);

   /*-- free toolbox string attributes --*/
   kfree(toolbox->path);
   kfree(toolbox->poc_email);
   kfree(toolbox->poc_name);
   kfree(toolbox->title);
   kfree(toolbox->name);
   kfree(toolbox->short_copyright);
   kfree(toolbox->long_copyright);
   kfree(toolbox->dbpath);

   kfree(toolbox);
}

/*-----------------------------------------------------------
| Routine Name:	kcms_tb_initialize - allocate and initialize a toolbox object
|
| Purpose:	This routine will initialize the CMS program
|		and library object.
|
| Returns:	new data object on success, NULL otherwise
|
| Written By:	Tom Sauer, Steven Jorgensen, Neil Bowers
| Date:		13-oct-92
------------------------------------------------------------*/
kobject
kcms_tb_initialize(void)
{
   kstring  routine = "kcms_tb_initialize()";
   kobject  toolbox = NULL;


   /* Malloc the object structure. */
   if ((toolbox = (kobject) kcalloc(1, sizeof(struct _kobject))) == NULL)
   {
      kerror(KCMS, routine,
	     "Unable to kmalloc (%d) bytes for toolbox instance.",
	     sizeof(struct _kobject));
   }
   else
   {
      toolbox->type                    = KOBJ_CMSTB;
      toolbox->attributes              = (AttrTbl *) kcms_tb_attr_table();
      toolbox->attribute_cnt           = KCMS_TB_MAX;

      toolbox->add_sub_obj             = _kcms_tb_add_cmobj;
      toolbox->del_sub_obj             = _kcms_tb_del_cmobj;
      toolbox->mod_sub_obj             = NULL;
      toolbox->parent                  = NULL;
      toolbox->app_defaults            = NULL;
      toolbox->flags                   = (unsigned long)0;
      toolbox->update                  = KCMS_UPDATE_NONE;
      toolbox->path                    = NULL;
      toolbox->name                    = NULL;

      /*-- toolbox file objects ---------------------------------------*/
      toolbox->info_file               = NULL;
      toolbox->config_file             = NULL;
      toolbox->alias_file              = NULL;
      toolbox->todo_file               = NULL;
      toolbox->changelog_file          = NULL;

      toolbox->poc_email               = NULL;
      toolbox->poc_name                = NULL;
      toolbox->title                   = NULL;
      toolbox->short_copyright         = NULL;
      toolbox->long_copyright          = NULL;
      toolbox->default_short_copyright = FALSE;
      toolbox->default_long_copyright  = FALSE;

      toolbox->keywords                = NULL;
      toolbox->progs                   = NULL;
      toolbox->sw_objects              = NULL;

      toolbox->ci_set                  = FALSE;

      toolbox->callbacks               = NULL;
   }

   return toolbox;
}

/*-----------------------------------------------------------
| Routine Name:	kcms_tb_rename_cmobj - update object list in toolbox
| Purpose:	This function is a kludge which is needed when renaming
|		a software object.  We have to update the klist of software
|		objects in the parent toolbox, since the identifier of the
|		klist entry which contains the renamed object is the token
|		of the object name.
|
| Input:	toolbox - The toolbox object containing the renamed object
|		oldname - The old name of the software object
|		newname - The new name of the software object.
|
| Written By:	Neil Bowers
| Date:		14-aug-94
------------------------------------------------------------*/
void
kcms_tb_rename_cmobj(
   kobject  toolbox,
   kstring  newname)
{
   kobject  object;
   kstring  oname;
   int      token = kstring_to_token(newname);
   klist   *list;


   for (list=toolbox->sw_objects; list != NULL; list = klist_next(list))
   {
      object = (kobject)klist_clientdata(list);
      if (object == NULL || !kcms_get_attribute(object, KCMS_NAME, &oname))
	 continue;

      if (!kstrcmp(oname, newname))
	 list->identifier = (kaddr)token;
   }
}
