/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Toolbox creation routines
   >>>>
   >>>>   Static:
   >>>>		_kcms_tb_create_dirs
   >>>>		_kcms_tb_create_files
   >>>>
   >>>>   Private:
   >>>>		kcms_tb_create
   >>>>
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include "toolboxP.h"

static char *tbDirectories[] =
{
   "bin",
   "examples",
   "include",
   "lib",
#if 0
   "manual",
      "manual/hardcopy",
      "manual/contents",
#endif
   "repos",
      "repos/license",
      "repos/config",
         "repos/config/imake_config",
      "repos/db",
   "testsuite"
};

static void _kcms_tb_create_files	PROTO((kobject,char *));

/*-----------------------------------------------------------
| Routine Name:	_kcms_tb_create_dirs - create initial toolbox directories
|
| Purpose:	This routine creates all the directories required for a
|		toolbox.  The configuration file telling this routine
|		which directories to create is $KHOROS/repos/kcms
|
| Input:	tbpath	- full path to top of toolbox
| Returns:	TRUE (1) on success, FALSE (0) otherwise
| Written By:	Neil Bowers
| Date:		25-mar-94
------------------------------------------------------------*/
static int 
_kcms_tb_create_dirs(
   kstring  tbpath)
{
   kstring  routine           = "_kcms_tb_create_dirs()";
   kbool    status            = TRUE;
   char     dirpath[KLENGTH];
   int      i;


   for (i=0; i<knumber(tbDirectories) && status; i++)
   {
      ksprintf(dirpath,"%s/%s",tbpath,tbDirectories[i]);
      if (kmkdir(dirpath,0777) != 0 && errno != EEXIST)
      {
	 kerror(KCMS, routine, "Unable to create the directory: %s\n",
		dirpath);
	 status = FALSE;
      }
   }

   return status;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_tb_create_files - create initial toolbox files
|
| Purpose:	This routine creates the configuration files necessary
|		for a toolbox to function to work correctly.  The
|		list of files to be created is determined by a configuration
|		file stored in $KHOROS/repos/kcms
|
| Input:	tbname	- Name of the toolbox
|		tbpath	- Path to the toplevel of the toolbox
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		25-mar-94
------------------------------------------------------------*/
static void
_kcms_tb_create_files(
   kobject  toolbox,
   kstring  tbpath)
{
   kstring  routine  = "Create Toolbox Files";
   kstring  tbname;
   char	    upper[KLENGTH];
   char	    lower[KLENGTH];
   char	    capitalize[KLENGTH];
   char	    template[KLENGTH];
   char	    filename[KLENGTH];
   int	    status;


   if (!kcms_get_attribute(toolbox, KCMS_NAME, &tbname)
       || kstring_lower(tbname,lower) == NULL
       || kstring_upper(tbname,upper) == NULL
       || kstring_capitalize(tbname,capitalize) == NULL)
   {
      return;
   }

   /*-- create a template <toolbox>.def imake config file --------*/
   kstrcpy(template,
	   "$BOOTSTRAP/objects/library/kcms/templates/toolbox/toolbox.def");
   ksprintf(filename,"$%s/repos/config/imake_config/toolbox.def", upper);
   if (!ksedfile(template,filename,FALSE,KFILE_OVERWRITE,&status,
		 "#toolbox-name#",	lower,
		 "#TOOLBOX-NAME#",	upper,
		 "#Toolbox-Name#",	capitalize,
		 NULL))
      kwarn(KCMS,routine,"Could not copy the imake template into "
	    "the new toolbox's imake template\n\n"
	    "\tTemplate: %s\n"
	    "\tToolbox : %s\n",
	    template, filename);
   else
      kcms_create_fileobj(toolbox, filename, NULL,
			  KCMS_FOBJ_TYPE_CONFIG,
			  KCMS_FOBJ_SUBTYPE_IMAKE_CONFIG,
			  KCMS_FOBJ_GEN_NONE, KCMS_FOBJ_ACCESS_RDWR);

   /*--	 create a template toolbox include file -------------------*/
   kstrcpy(template,
	   "$BOOTSTRAP/objects/library/kcms/templates/toolbox/toolbox.h");
   ksprintf(filename,"$%s/include/%s.h", upper, lower);
   if (!ksedfile(template,filename, FALSE, KFILE_OVERWRITE, &status,
		 "#toolbox-name#",	lower,
		 "#TOOLBOX-NAME#",	upper,
		 "#Toolbox-Name#",	capitalize,
		 NULL))
      kwarn(KCMS,routine,"Could not copy the toolbox include template into "
	    "the new toolbox's include file\n\n"
	    "\tTemplate: %s\n"
	    "\tToolbox : %s\n",
	    template, filename);
   else
      kcms_create_fileobj(toolbox, filename, NULL,
			  KCMS_FOBJ_TYPE_SRC,
			  KCMS_FOBJ_SUBTYPE_INCL,
			  KCMS_FOBJ_GEN_NONE, KCMS_FOBJ_ACCESS_RDWR);
}

/*----------------------------------------------------------
| Routine Name:	kcms_tb_create - create a new toolbox object
|
| Purpose:	This routine adds a toolbox entry to the list of installed
|		toolboxes, and creates all the files and directories
|		necessary to make the toolbox functional
|
|		This function is the private back-end to the public
|		function kcms_create_toolbox().  The public function does
|		all the sanity checking, this does the work.
|
| Input:	tbname	-	name of the toolbox to create
|		tbpath	-	path to new toolbox
| Output:	
| Returns:	a new toolbox object, or NULL on error.
| Written By:	Steven Jorgensen & Neil Bowers
| Date:		4-jan-93
----------------------------------------------------------*/
kobject 
kcms_tb_create(
   kstring  tbname,
   kstring  tbpath)
{
   kstring   routine             = "Create Toolbox";
   kobject   toolbox             = NULL;
   kstring  *file_list;
   int	     nfiles;
   int	     notempty;
   char	     fullpath[KLENGTH];
   char	     cwd[KLENGTH];
   kstring   datestamp;


   if (kstrncmp(tbpath, "./", 2) != 0 && kstrncmp(tbpath, "../", 3) != 0)
   {
      kgetcwd(cwd, KLENGTH);
      kfullpath(tbpath, cwd, fullpath);
   }
   else
      kexpandpath(tbpath,NULL,fullpath);
	
   if (kaccess(fullpath,R_OK) == 0)
   {
      file_list = karray_dirlist(fullpath,NULL,NULL,
				 (KDIR | KFILE),FALSE,&nfiles);
      notempty = (nfiles > 1 || (nfiles==1 &&
				 kstrcmp(file_list[0],"..") != 0));
      karray_free(file_list,nfiles,NULL);

      if (notempty)
      {
	 kerror(KCMS,routine,
		"The specified directory for the new toolbox "
		"is non-empty.\nAborting toolbox creation.");
	 return NULL;
      }
   }

   if (kmkdir(fullpath, 0777) != 0 && errno != EEXIST)
   {
      errno = KINTERNAL;
      kerror(KCMS,routine, "Cannot create top level toolbox directory\n");
      errno = KCMS_TOOLBOXUPDATEFAIL;
   }

   else if (_kcms_tb_create_dirs(fullpath))
   {
      toolbox = kcms_tb_initialize();

      /*-- add toolbox to toolboxes file ------------------------------*/
      kcms_env_addtb(tbname,tbpath);

      /*-- make sure the update flag is gonna result in a sync --------*/
      toolbox->name   = kstring_upper(tbname,NULL);
      toolbox->path   = kstrdup(tbpath);
      toolbox->dbpath = kstring_cat(tbpath,TOOLBOX_DB_PATH,NULL);
      toolbox->status = KCMS_STATUS_DEVELOPMENT;
      toolbox->update = KCMS_UPDATE_SYNC;
      toolbox->version = kstrdup(KCMS_DEFAULT_TB_VERSION);

      /*-- set creation, modification and generation dates ------------*/
      datestamp = kcms_get_date();
      toolbox->times.creation     = kstrdup(datestamp);
      toolbox->times.modification = kstrdup(datestamp);
      toolbox->times.generation   = kstrdup(datestamp);
      kfree(datestamp);

      _kcms_tb_create_files(toolbox,fullpath);

      /*-- let's get this puppy onto disk, just in case ---------------*/
      kcms_tb_sync(toolbox);
   }

   return toolbox;
}
