 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Khoros Man Page Utility Routines
   >>>>
   >>>>  Private:
   >>>>      run_kman()
   >>>>      find_token() 
   >>>>   Static:
   >>>>      _process_so()
   >>>>      _process_include()
   >>>>      _process_help()
   >>>>      _process_code()
   >>>>      _process_whatis()
   >>>>      _process_syntax()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "kman.h"

	/*
	 * This is the Khoros roff tmac macro that must be included
	 * for the documentation to work correctly
	 */
#define MANUAL_CF   "$BOOTSTRAP/repos/config/doc_config/manual.cf"
#define TMAC_MAN    "$BOOTSTRAP/repos/tmac/tmac.an"
#define TMAC_HELP   "$BOOTSTRAP/repos/tmac/tmac.khoros"
#define TMAC_MANUAL "$BOOTSTRAP/repos/tmac/tmac.khoros"


struct token_list {
        char         *key;    
        int          (*function)();  
};

static int _process_title	PROTO((char  *, char  *, kfile *));
static int _process_so		PROTO((char  *, char  *, kfile *));
static int _process_include	PROTO((char  *, char  *, kfile *));
static int _process_help	PROTO((char  *, char  *, kfile *));
static int _process_code	PROTO((char  *, char  *, kfile *));
static int _process_whatis	PROTO((char  *, char  *, kfile *));
static int _process_syntax	PROTO((char  *, char  *, kfile *));
int	   find_token		PROTO((char  *, kfile *));

	/* 
	 * This table contains the search key and function to call
	 * when that search key is found. To add to this table
	 * list the search key before the NULL statement and
	 * create the appropriate function at the bottom of this
	 * file. The key will automatically be search for
	 * when processing the manual files.
	 */

static struct token_list map[] = {
        { "^\\.TH .*$",          _process_title        },
        { "^\\.include .*$",     _process_include      },
        { "^\\.help .*$",        _process_help         },
        { "^\\.so .*$",          _process_so           },
        { "^\\.cI .*$",          _process_so           },
        { "^\\.code .*$",        _process_code         },
        { "^\\.whatis .*$",      _process_whatis       },
        { "^\\.syntax .*$",      _process_syntax       },
        { "^\\.PSPIC .*$",       NULL      },
        { NULL,                  NULL                  },
};

static char *tbname = NULL;
static kobject toolbox = NULL;


/*-----------------------------------------------------------
| Routine Name:	get_roff_cmd - Find the roff cmd
|
| Purpose:	This routine will parse the manual.cf file in the 
|		Khoros toolbox in repos/config/doc_config and search
|		for the roff commands and roff macros.
|
| Input:	mode    - the roff mode (MAN, MANUAL, HELP)
| Output:	command - the formatted command from the manual.cf file
|			   or the default "neqn filename | tbl | nroff -ms"
| Returns:	returns TRUE (1) on success, FALSE (0) on failure
|		 "neqn filename | tbl | nroff -ms"
| Written By:	Tom Sauer
| Date:		Jan 29, 1993
------------------------------------------------------------*/
void
get_roff_cmd(
   int  mode,
   char *command)
{
   kfile *fptr;
   int   count, status;
   char **roff;
   char **tbl;
   char **macro;
   char **eqn;
   char *roffcmd = "groff -Tascii";
   char *roffmacro = "";
   char *roffeqn = "geqn";
   char *rofftbl = "gtbl";


   /*--------------------------------------------------------------------
   |	check to see if we have already set the roff command.
   |	parsing the file each time is too expensive.
   |	build the command up as "eqn | tbl | roff -ms"
   --------------------------------------------------------------------*/
   if (mode == HELP)
   {
      if ((fptr = kfopen(MANUAL_CF, "r")) != NULL)
      {
	 /*-- get the nroff command -----------------------------------*/
	 roff = kparse_file_scan_delimit(fptr, "^#define.*NroffCmd", "$",
					 KIGNORE_CASE, "[ \t]", NULL, NULL,
					 &count, &status);

	 if (roff != NULL && status == KPARSE_OK)
	 {
	    roffcmd = karray_to_string(roff, count, " ");
	    karray_free(roff, count, NULL);
	 }

	 /*-- get the nroff macro -------------------------------------*/
	 krewind(fptr);
	 macro = kparse_file_scan_delimit(fptr, "^#define.*RoffMacro", "$",
					  KIGNORE_CASE, "[ \t]", NULL, NULL,
					  &count, &status);

	 if (macro != NULL && status == KPARSE_OK)
	 {
	    roffmacro = karray_to_string(macro, count, " ");
	    karray_free(macro, count, NULL);
	 }

	 /*-- get the equation command --------------------------------*/
	 krewind(fptr);
	 eqn = kparse_file_scan_delimit(fptr, "^#define.*EqnCmd", "$",
					KIGNORE_CASE, "[ \t]", NULL, NULL,
					&count, &status);

	 if (eqn != NULL && status == KPARSE_OK)
	 {
	    roffeqn = karray_to_string(eqn, count, " ");
	    karray_free(eqn, count, NULL);
	 }

	 /*-- get the table command -----------------------------------*/
	 krewind(fptr);
	 tbl = kparse_file_scan_delimit(fptr, "^#define.*TblCmd", "$",
					KIGNORE_CASE, "[ \t]", NULL, NULL,
					&count, &status);

	 if (tbl != NULL && status == KPARSE_OK)
	 {
	    rofftbl = karray_to_string(tbl, count, " ");
	    karray_free(tbl, count, NULL);
	 }
	 kfclose(fptr);
      }

      ksprintf(command, "%s | %s | %s %s", roffeqn, rofftbl, roffcmd,
	       roffmacro);
   }
   else
      ksprintf(command, "%s | %s %s", rofftbl, roffcmd, roffmacro);
}


/*-----------------------------------------------------------
|
|  Routine Name: get_filename - gets the filename 
|
|       Purpose: This routine takes a key line and parses to
|		 get the filename, then expands the filename.
|
|         Input: filename - the filename to expand
|        Output: None
|       Returns: An expanded filename on success, otherwise NULL on failure
|    Written By: Tom Sauer
|          Date: Dec 08, 1992
| Modifications:
|
------------------------------------------------------------*/

static char *get_filename(
   char *filename)
{
	char *temp, **list;
	char *delimiter = " \t";
	int item_count;
	
	if (filename == NULL)
           return(NULL);

        list = kparse_string_delimit(filename, delimiter, KDELIM_CLEAN,
				     &item_count);
        if (item_count <= 0)
        {
           kerror(NULL, "get_filename",
		    "An error occured trying to find the filename in the line:\n%s\nThe format should be .{command} {filename} {other optional parameters}\n", filename);
           return(FALSE);
        }
	temp = kfullpath(list[1], NULL, NULL);
	karray_free(list, item_count, NULL);
	return(temp);
}


/*-----------------------------------------------------------
|
|  Routine Name: _process_so - routine to process a .so line
|
|       Purpose: This routine will expand the filename for a .so line
|		 and write it to the output file as 
|		   .so {expanded file}
|
|         Input: file_line - a pointer to the .so string
|		 fptr - a pointer to the beginning of the .so string
|                optr - a kfile pointer to the output file
|        Output: updated output file pointer
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Jul 07, 1993
| Modifications:
|
------------------------------------------------------------*/

static int _process_so(
   char  *file_line,
   char  *fptr,
   kfile *optr)
{
	char *new_file;


	if ((new_file = get_filename(file_line)) == NULL)
	    return(FALSE);

	(void) find_token(new_file, optr);
	kfree(new_file);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _process_title - routine to process a .TH line
|
|       Purpose: This routine will expand the filename for a .TH line
|		 and write it to the output file as 
|
|		   .TH "oname" "toolbox" "title" "explanation" "date"
|
|		 Sets the toolbox name.
|
|         Input: file_line - a pointer to the .TH string
|		 fptr - a pointer to the beginning of the .TH string
|                optr - a kfile pointer to the output file
|        Output: 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young & Steve Jorgensen
|          Date: Apr 10, 1995
| Modifications:
|
------------------------------------------------------------*/

static int _process_title(
   char  *file_line,
   char  *fptr,
   kfile *optr)
{
	char **list;
	int  item_count;


	kinfo(KDEBUG, "\nIn _process_title\n");
        list = kparse_string_delimit(file_line, " \t\"", KDELIM_CLEAN,
				     &item_count);
        if (item_count < 3)
        {
           kerror(NULL, "run_kman",
                    "An error occured trying to parse the line:\n%s\nThe format should be .TH \"oname\" \"toolbox\" \"title\" \"explanation\" \"date\".\n");
           return(FALSE);
        }
	kfree(tbname);
	if (toolbox) kcms_close(toolbox);
	toolbox = NULL;

	tbname = list[2]; list[2] = NULL;
	karray_free(list, item_count, NULL);
	kfprintf(optr, "%s\n", file_line);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _process_help - routine to process a .help line
|
|       Purpose: This routine will strip off the .help key and expand 
|		 the filename, then call find_token to process the file.
|		 The on-line help file will be preprocessed to strip off
|		 some of the commands specific to on-line help.
|		 The result is the file passed in and all files it calls
|		 will be literally included in the output file. 
|		 For example, if "file.man" has a .sec {filename}
|		 line, then this routine will call find_token with
|		 the .sec's file. This file will be parsed and included
|		 in the output file.
|
|         Input: file_line - a pointer to the .help string
|		 fptr - a pointer to the beginning of the .help string
|                optr - a kfile pointer to the output file
|        Output: updated output file pointer
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Tom Sauer
|          Date: Dec 08, 1992
| Modifications:
|
------------------------------------------------------------*/

static int _process_help(
   char  *file_line,
   char  *fptr,
   kfile *optr)
{
        char  *file_contents, *key_fmt, *new_file;
	char *search_key = "^\\.onlineHelp.*$";
	char *nh_search_key = "^\\.section.*$";
	char *delimiter = " \t";
	char **list, *sptr, *start_point, *tmp_ptr;
        int   i, status, start_header, item_count, num, done = FALSE;

	kinfo(KDEBUG, "\nIn _process_help\n");

        list = kparse_string_delimit(file_line, delimiter, KDELIM_CLEAN,
				     &item_count);
        if (item_count != 3)
        {
           kerror(NULL, "run_kman",
                    "An error occured trying to parse the line:\n%s\nThe format should be .help starting_section  {filename}\nWhere starting_section is greater than 0 and less than 4\n", file_line);
           return(FALSE);
        }

	new_file = kfullpath(list[2], NULL, NULL);
	start_header = atoi(list[1]);
	if (start_header == 0)
	{
           kerror(NULL, "run_kman",
                    "An error occured trying to parse the line:\n%s\nThe format should be .help starting_section  {filename}\n Where starting_section is greater than 0 and less than 4\n", file_line);
           return(FALSE);
	}
	karray_free(list, item_count, NULL);

         /*
          * open the input file and read it
          */

	if ((file_contents = kreadfile(new_file, NULL)) == NULL)
	{
	   kfree(new_file);
	   return(FALSE);
	}

        sptr = kparse_string_search(file_contents, search_key,
					 KIGNORE_CASE, NULL, &status);
	switch(status)
	{
	   case KPARSE_OK:
	      break;
	   case KPARSE_PARTKEY:
	   case KPARSE_SYNTAXKEY:
	      kerror(NULL, "run_kman",
		       "A file parsing error occurred, check the\n%s\n file for errors\n", new_file);
	      kfree(new_file);
	      return(FALSE);
	   case KPARSE_NOKEY:
	      sptr = file_contents;
	      break;
	}

	while (! done)
	{ 
	   start_point = sptr;
           sptr = kparse_string_search(sptr, nh_search_key,
					 KIGNORE_CASE, &key_fmt, &status);
	   switch(status)
	   {
	      case KPARSE_OK:
	         list = kparse_string_delimit(key_fmt, delimiter, KDELIM_CLEAN,
						&item_count);
		 if (item_count < 3 || (list != NULL && atoi(list[1]) == 0))
        	 {
           	    kerror(NULL, "run_kman",
                    	   "An error occured trying to parse the line:\n%s\nThe format should be .help starting_section  {filename}\nWhere starting_section is greater than 0 and less than 4\n", key_fmt);
		    karray_free(list, item_count, NULL);
	            kfree(new_file);
           	    return(FALSE);
        	 }

	         if ( (sptr - kstrlen(key_fmt) - 1) != start_point)
	         {
	            tmp_ptr = kstring_ncopy(start_point,
				  sptr - kstrlen(key_fmt) - 1 - start_point, 
					    NULL);
		    kfprintf(optr, "%s", tmp_ptr);
	 	    kfree(tmp_ptr);
		 }
	         num = atoi(list[1]); 
		 num = num -1 + start_header;
	         kfprintf(optr, "%s %d ", list[0], num);
	         for (i = 2; i < item_count; i++)
	            kfprintf(optr, "%s ", list[i]);
	         kfprintf(optr, "\n");
		 karray_free(list, item_count, NULL);
		 kfree(key_fmt);
	         break;
	      case KPARSE_PARTKEY:
	      case KPARSE_SYNTAXKEY:
	         kerror(NULL, "run_kman",
		          "A file parsing error occurred, check the\n%s\n file for errors\n", new_file);
	         kfree(new_file);
	         return(FALSE);
	      case KPARSE_NOKEY:
	         kfprintf(optr, "%s", start_point);
		 kfree(key_fmt);
	         done = TRUE;
	         break;
   	   }
	}
	kfree(new_file);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: _process_code - routine to process a .code line
|
|       Purpose: This routine will strip off the .code key and expand 
|		 the filename, then read the code file. Some special
|		 roff commands are placed around the code then the
|		 code is output to the document or ,ms file.
|
|         Input: file_line - a pointer to the .coed string
|		 fptr - a pointer to the beginning of the .coed string
|                optr - a kfile pointer to the output file
|        Output: updated output file pointer
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Tom Sauer
|          Date: Dec 18, 1992
| Modifications:
|
------------------------------------------------------------*/

static int _process_code(
   char  *file_line,
   char  *fptr,
   kfile *optr)
{
        char  *file_contents, *new_file;


	kinfo(KDEBUG, "\nIn _process_code\n");
	if ((new_file = get_filename(file_line)) == NULL)
	    return(FALSE);

        /*
         * open the input file and read it
         */
	if ((file_contents = kreadfile(new_file, NULL)) == NULL)
	{
	   kfree(new_file);
	   return(FALSE);
	}
	kfprintf(optr, ".ps 9\n");
	kfprintf(optr, ".vs 11\n");
	kfprintf(optr, "\\f(CW\n");
	kfprintf(optr, ".nf\n");
	kfprintf(optr, "%s", file_contents);
	kfprintf(optr, ".fi\n");
	kfprintf(optr, "\\fP\n");
	kfprintf(optr, ".LP\n");
	kfree(new_file);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _process_whatis - routine to process a .whatis line
|
|       Purpose: This routine will strip off the .whatis key and expand 
|		 the object name, then replace retrieve the object's
|		 whatis entry. Some special roff commands are placed
|		 around the whatis entry and then the whatis is output
|		 to the document or, ms file.
|
|		 The following is the form of the generated whatis
|		 section for the routine kcadd:
|
|			\(bu
|			\fIkcadd()\fP -  add two complex numbers
|			.br
|
|         Input: file_line - a pointer to the .so string
|                optr - a kfile pointer to the output file
|
|        Output: updated output file pointer
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Jorgensen
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static int _process_whatis(
   char  *file_line,
   char  *fptr,
   kfile *optr)
{
        char **list;
        char *delimiter = " \t:,;";
        int item_count;
        header_info *header;
        kobject object;


        kinfo(KDEBUG, "\nIn _process_function\n");
        if (file_line == NULL)
           return(FALSE);

        list = kparse_string_delimit(file_line, delimiter, KDELIM_CLEAN,
			             &item_count);

        if (item_count < 3)
        {
           kerror(NULL, "run_kman",
                    "A .whatis line in one of the .sec or .help files has an incorrect format, the line in error is:\n%s\nThe format should be .whatis {program_object_name} {function_name}\n", file_line);
           return(FALSE);
        }

	errno = 0;
	if (!toolbox) toolbox = kcms_open_toolbox(tbname);
	if ((object = kcms_open_cmobj(toolbox, list[1])) == NULL)
        {
	   kerror(NULL,"_process_whatis()","Unable to open the program or library object for inclusion of the '%s' whatis.\nThe error occurred on the line:\n%s\nPlease check to make sure the program object '%s' exists in the toolbox\n", list[2], file_line, list[1]);
           karray_free(list, item_count, NULL);
           return(FALSE);
        }

        kcms_set_attribute(object, KCMS_CMOBJ_ROUTINENAME, list[2]);

        if (! kcms_get_attribute(object, KCMS_CMOBJ_HEADERINFO, &header) )
        {
           kerror(NULL, "run_kman",
                "Unable to read the source code header for the whatis '%s'\nThe error occurred on the line:\n%s\nPlease check the source code header for errors and make sure\nit conforms with the Khoros coding conventions", list[1], file_line);
           karray_free(list, item_count, NULL);
           return(FALSE);
        }

	if (header->shortdesc == NULL)
	  kinfo(KFORCE, "The short description for the routine '%s' is empty.  kgenmanual will put this entry in your manual, but you will need to update the header for this routine, and rerun both kgenman and kgenmanual.", header->name);
	kfprintf(optr, "\\(bu\n\\fI%s()\\fP - %s\n.br\n",
		 header->name, (header->shortdesc == NULL) ? "" :
		 header->shortdesc);

        karray_free(list, item_count, NULL);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _process_syntax - routine to process a .syntax line
|
|       Purpose: This routine will call kgen_interpret_clui_usage()
|                when it sees a ".syntax" line.
|
|         Input: file_line - a pointer to the .syntax string
|                optr - a kfile pointer to the output file
|
|        Output: updated output file pointer
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Danielle Argiro
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static int _process_syntax(
   char  *file_line,
   char  *fptr,
   kfile *optr)
{
	kform   *form;
        char    **list;
        kobject program;
	kobject file_object;
	char    *oname, *panepath;
        char    *delimiter = " \t:,;";
	int     item_count, progtype;


        kinfo(KDEBUG, "\nIn _process_syntax\n");

	/*-- man pages don't have the usage statement -----------------*/
	if (clui_info->man_flag
	    || (!clui_info->man_flag
		&& !clui_info->hlp_flag
		&& !clui_info->manual_flag))
	   return TRUE;

        if (file_line == NULL)
           return(FALSE);

        list = kparse_string_delimit(file_line, delimiter, KDELIM_CLEAN,
			             &item_count);
        if (item_count < 3)
        {
           kerror(NULL, "run_kman",
                    "A .syntax line in one of the .1, .3, or .hlp files has an incorrect format, the line in error is:\n%s\nThe format should be .syntax {toolbox_name} {program_object_name}\n", file_line);
           return(FALSE);
        }
	errno = 0;


	tbname = list[1];
	oname  = list[2];
	if (!toolbox) toolbox = kcms_open_toolbox(tbname);
	if (toolbox == NULL) 
	{
	    kerror(NULL, "_process_syntax",
		   "Warning: unable to open toolbox '%s' referenced by .syntax line in man page;  will not be able to print syntax for program '%s'", tbname, oname);
	    return(FALSE);
	}
	program = kcms_open_cmobj(toolbox, oname);
	if (program == NULL)
	{
	    kerror(NULL, "_process_syntax",
		   "Unable to open program object '%s'; will not be able to print syntax",
		   oname);
	    return(FALSE);
	}

	if (!kcms_get_attribute(program, KCMS_CMOBJ_UIS_PANE, &file_object))
	    return(FALSE);
        if (file_object == NULL) return(FALSE);

	if (!kcms_get_attribute(file_object, KCMS_FOBJ_FULLNAME, &panepath))
	    return(FALSE);
	if (file_object == NULL) return(FALSE);

	if (!kcms_get_attribute(program, KCMS_CMOBJ_PROGTYPE, &progtype))
	    return(FALSE);


	if ((form = kvf_create_form(panepath, NONE, NULL, NULL)) == NULL)
	   return(FALSE);

	if (clui_info->hlp_flag)
	{
	    kfprintf(optr, ".nf\n");
	    kgen_interpret_clui_selections(optr, form);
	    kfprintf(optr, ".fi\n");
	}
	else
	{
	    if ((progtype == KCMS_KROUTINE) || (progtype == KCMS_PANE))
	        kvf_append_selections(form, KGEN_KSTDARGS_PANEPATH);
	    else if (progtype == KCMS_XVROUTINE)
	        kvf_append_selections(form, KGEN_XVSTDARGS_PANEPATH);
	    else return(FALSE);

	    kfprintf(optr, ".nf\n");
	    kgen_interpret_clui_usage(optr, form, tbname, oname, NULL);
	    kfprintf(optr, ".fi\n");
	}
	kvf_destroy_form(form);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _process_include - routine to process a .sec file
|
|       Purpose: This routine will strip off the .sec key and expand 
|		 the filename, then call call find_token to process the file.
|		 The result is the file passed in and all files it calls
|		 will be literally included in the output file. 
|		 For example, if "file.man" has a .sec {filename}
|		 line, then this routine will call find_token with
|		 the .sec's file. This file will be parsed and included
|		 in the output file.
|
|         Input: file_line - a pointer to the .sec or .section string
|		 fptr - a pointer to the beginning of the .section string
|                optr - a kfile pointer to the output file
|        Output: updated output file pointer
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Tom Sauer
|          Date: Dec 08, 1992
| Modifications:
|
------------------------------------------------------------*/

static int _process_include(
   char  *file_line,
   char  *fptr,
   kfile *optr)
{
	char *new_file;

	if ((new_file = get_filename(file_line)) == NULL)
	    return(FALSE);

	/*
	 * recursivly call find_token to process the current file
	 */
	if (!find_token(new_file, optr))
	{
	   kfree(new_file);
	   return(FALSE);
	}
	kfree(new_file);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: find_token - find a line in the file that matches a token
|
|       Purpose: 
|
|         Input: argument1 - explanation
|
|        Output: argument4 - explanation
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Tom Sauer
|          Date: Dec 04, 1992 15:46
| Modifications:
|
------------------------------------------------------------*/

int find_token(
   char  *manpage,
   kfile *optr)
{

        char  *file_contents, *file_part, *start, *currentline;
	char  *fptr, *temp, *final;
	char  *line = NULL;
        int   index, tstat, status, done, length;
	int   (*key_function)();

         /*
          * open the input file and read it
          */
	
	if ((file_contents = kreadfile(manpage, NULL)) == NULL)
	{
	   kinfo(KSTANDARD,"\n\nWarning!!  Unable to open file '%s'\n",manpage);
	   return(FALSE);
	}

	done = FALSE;
	fptr = file_contents;
	  /*
	   * start scanning the file search for the keys in the
	   * map structure above. Call the appropriate routine
	   * based on the key found. Terminate when we have hit
	   * the end of the file.
	   */
	while (! done)
        {
	   start = fptr;
	   final = NULL;
	   currentline = NULL;
	   status = KPARSE_NOKEY;
	   index = 0;

	     /* 
	      * since the kparser does not support an "or" function
	      * we must do it ourselves. So, find the first occurance of
	      * one of the keys listed in the map structure above.
	      */
	   while (map[index].key != NULL)
	   {
	       kfree(line);
	       if  ((temp = kparse_string_search(fptr, map[index].key,
					 KIGNORE_CASE, &line, &tstat)) == NULL)
	       {
		  index++;
		  continue;
	       }
	       else if ((temp != NULL && temp < final) || final == NULL)
	       {
		  final = temp;
		  status = tstat;
		  kfree(currentline);
		  currentline = kstring_copy(line, NULL);
		  key_function = map[index].function;
	       }
	       index++;
	   }
	   fptr = final;

		/*
		 * Now check the status from the key search above
		 * and call the appropriate routine for that key if
		 * the status was OK. Otherwise hopefully there was no
		 * error and we hit the end of the file.
		 */
	   switch (status)
	   {
	      case KPARSE_OK:

		   /*
		    * print out the contents of the file we are processing
		    * up to the current point in the file.
		    */
		 length = fptr - (kstrlen(currentline) + 1) - start;
		 if (length > 0)
		 {
		    file_part = kstring_ncopy(start, length, NULL );
	            kfprintf(optr, "%s", file_part);
		    kfree(file_part);
	 	 }

		    /*
		     * call the routine to process the key found
		     */
		 if (key_function != NULL)
		    if ( ! (key_function(currentline, start + length, optr)) &&
			   (key_function != _process_syntax))
                    {
	               kfree(currentline);
		       return(FALSE);
	            }

	         kfree(currentline);
	         break;
	      case KPARSE_PARTKEY:
	      case KPARSE_SYNTAXKEY:
	         kerror(NULL, "run_kman",
		       "A file parsing error occurred, check the '.man', '.sec' and 'help files' for errors\n");
	         return(FALSE);
	      case KPARSE_NOKEY:
		    /* 
	             * probably hit the end of the file or there was not a key 
		     * in the file 
		     * check to see if we hit the end of the file.
		     * if so then write out the remaining part of the
		     * file being processed.
		     */

		 kinfo(KDEBUG, "Processing file `%s`\n", manpage);
	         kfprintf(optr, "%s", start);
	         done = TRUE;
	         break;
	   }
	}
	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: run_kman - routine for printing khoros man pages
|
|       Purpose: This routine will process a manpage.
|         Input: manpages  - the man pages to be printed
|		 manpage_num    - the number of toolboxes
|		 toolboxes - the name of the toolboxes to search
|		 tb_num    - the number of toolboxes
|		 mode      - the mode in which to print the man pages
|
|        Output: None
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Jul 07, 1993
| Modifications:
|
------------------------------------------------------------*/

int run_kman(
   char    **manpages,
   int	   manpage_num,
   char	   **toolboxes,
   int	   tb_num,
   int	   mode)
{
	kdbm   *dbm;
	kdatum key, data;

	kfile  *file;
	int     i, j, found;
	char   *more, *tmacfile; 
	char   command[KLENGTH], filename[KLENGTH], temp[KLENGTH];


	get_roff_cmd(mode, temp);
	more = (!isatty(1) ? "cat" : ((more = kgetenv("PAGER")) == NULL) ?
			"more" : more);
	ksprintf(command, "%s | %s", temp, more);

	if (mode == HELP)
	   tmacfile = TMAC_HELP;
	else if (mode == MAN)
	   tmacfile = TMAC_MAN;
	else if (mode == MANUAL)
	   tmacfile = TMAC_MANUAL;

	if ((file = kpopen(command, "w")) == NULL)
	{
	   kerror(NULL, "run_kman", "Cannot execute the command '%s'\n",
			command);
	   return(FALSE);
	}
	find_token(tmacfile, file);

	for (i = 0; i < manpage_num; i++)
	{
	   if (mode != MAN && kaccess(manpages[i], R_OK) == 0)
	   {
	      find_token(manpages[i], file);
	      continue;
	   }

	   key.dptr  = manpages[i];
	   key.dsize = kstrlen(key.dptr) + 1;
           for (j = 0, found = FALSE; j < tb_num; j++)
           {
	      if (toolboxes[j] == NULL) continue;

	      ksprintf(filename, "$%s/repos/db/manpage", toolboxes[j]);
	      if (toolbox) kcms_close(toolbox);
	      toolbox = NULL;
	      tbname = toolboxes[j];

	      if ((dbm = kdbm_open(filename, O_RDONLY, 0666)) == NULL)
	      {
		 toolboxes[j] = NULL;
	         kinfo(KVERBOSE, "kman:  Failed to open database '%s'", 
		       filename);
	         continue;
	      }
	      data = kdbm_fetch(dbm, key);
	      if (data.dsize > 0)
	      {
	         if (data.dptr)
	         {
		    found = TRUE;
	            if (mode == HELP)
	            {
		         kstring_replace(data.dptr, "/man/", "/help/", temp);
		         kstring_replace(temp, ".1", ".hlp", filename);
	            }
		    else kstring_copy(data.dptr, filename);
	            find_token(filename, file);
	         }
	      }
	      kdbm_close(dbm);
	   }

	   tbname = NULL;
	   if (kaccess(manpages[i], R_OK) == 0)
	   {
	      find_token(manpages[i], file);
	      continue;
	   }


	   if (!found)
	   {
	      kinfo(KSTANDARD, "kman: No manual entry for '%s'.\n"
		       "      Try \"kman -k <keyword>\" if you are looking "
		       "for something.\n", manpages[i]);
	   }
	}
	kpclose(file);
	return(TRUE);
}
