/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            kgenmanual: Utility Routines
   >>>>
   >>>>  Private:
   >>>>		output_string()
   >>>>
   >>>>   Static:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "kgenmanual.h"

static void print_header_section  PROTO((kfile *, kstring, kstring));
static void generate_arguments    PROTO((kfile *, kobject));
static void generate_man_section  PROTO((kfile *, kobject, int, kstring));

static kobject cur_toolbox = NULL;
static kobject cur_object  = NULL;


/*-----------------------------------------------------------
| Routine Name:	output_string - write string to a transport,
|				escaping metacharacters
|
| Purpose:	This routine takes a string and an output file.  Certain
|		substitutions are performed on the string, to escape
|		metacharacters, before printing it to the transport.
|
| Input:	file    - Output transport
|		text    - Ahe text string to output
|		newline - A boolean flag which specifies whether a newline
|			  should be written to the transport after the string.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise.
|
| Written By:	Neil Bowers
| Date:		7-aug-94
------------------------------------------------------------*/
kbool
output_string(
   kfile     *transport,
   kstring    text,
   kbool      newline)
{
   kstring    outstring;


   if ((outstring = kstring_replace(text, "\\n", "\\\\n", NULL)) == NULL)
      return FALSE;

   kfprintf(transport, "%s", outstring);
   if (newline && outstring[kstrlen(outstring)-1] != '\n')
      kfprintf(transport, "\n");

   kfree(outstring);

   return TRUE;
}


/*-----------------------------------------------------------
| Routine Name:	output_header - write out the header in roff format
|
| Purpose:	This routine will take a function header and write out the
|		header information in roff format for the printed manual.
|
| Input:	optr          - The output file pointer
|		header        - A pointer to the header info
|		function      - The name of the function.
|		section_level - The level section to generate.
|
| Returns:	TRUE (1) upon success, FALSE (0) upon failure
|
| Written By:	Neil Bowers
| Date:		4-sep-94
------------------------------------------------------------*/
kbool
output_header(
   kfile        *optr,
   header_info  *header,
   kstring       function,
   int           section_level)
{
   kstring  routine = "output_header()";
   kstring  shortdesc;
   int      i;


   if (header->name == NULL)
   {
      kerror(NULL, routine,
	     "The name of the function '%s' is missing in the header\n"
	     "Please check the source code header and make sure it conforms "
	     "to the Khoros coding standards\n", function);
      return FALSE;
   }

   if (header->purpose == NULL)
   {
      kerror(NULL, routine,
	     "The description for the function '%s' is missing in the header\n"
	     "Please check the source code header and make sure it conforms "
	     "to the Khoros coding standards\n", function);
      return FALSE;
   }

   if (header->declaration == NULL)
   {
      kerror(NULL, routine,
	     "The declaration for the function '%s' is missing in the header\n"
	     "Please check the source code header and make sure it conforms "
	     "to the Khoros coding standards\n",
	     function);
      return FALSE;
   }

   for (i = 0; i < header->in_cnt; i++)
   {
      if (header->input[i][1] == NULL)
      {
	 kerror(NULL, routine,
		"The description for the paramter '%s' in the function '%s' "
	        "is missing\nPlease check the source code header and make "
	        "sure it conforms to the Khoros coding standards\n",
		header->input[i][0], function);
	 return FALSE;
      }
   }
   for (i = 0; i < header->out_cnt; i++)
   {
      if (header->output[i][1] == NULL)
      {
	 kerror(NULL, routine,
		"The description for the paramter '%s' in the function '%s' "
	        "is missing\nPlease check the source code header and make "
	        "sure it conforms to the Khoros coding standards\n",
		header->output[i][0], function);
	 return FALSE;
      }
   }

   /*UPDATE:-------------------------------------------------------------
   |	This is a roaring great kludge, but it seems to work ;-)
   --------------------------------------------------------------------*/
   if (header->shortdesc == NULL)
   {
      kfprintf(kstderr,
	       "kgenmanual: function %s() has no short description!\n",
	       header->name);
      shortdesc = "";
   }
   else
      shortdesc = header->shortdesc;

   kfprintf(optr,
	    ".section %d \"%s() \\(em \\fP\\s-1\\fI%s\\fP\\s+1\" box\n",
	    section_level, header->name, shortdesc);
   kfprintf(optr, ".indexEntry definitive %s()\n", header->name);

   /*-- Synopsis -- function declaration ------------------------------*/
   kfprintf(optr,
	    "\\fB\\s+1Synopsis\\s-1\\fP\n"
	    ".br\n"
	    ".in +0.25in\n"
	    ".nf\n"
	    "\\f(CW\\s-1%s\\s+1\\fP\n"
	    ".fi\n"
	    ".in -0.25in\n"
	    ".paragraph\n", header->declaration);

   /*-- Input Arguments -- description of function arguments ----------*/
   if (header->in_cnt > 0)
   {
      kfprintf(optr, "\\fB\\s+1Input Arguments\\s-1\\fP\n");
      kfprintf(optr, ".br\n");
      kfprintf(optr, ".in +0.25in\n");

      for (i = 0; i < header->in_cnt; i++)
      {
	 kfprintf(optr, "\\f(CW%s\\fP\n", header->input[i][0]);
	 kfprintf(optr, ".br\n");
	 kfprintf(optr, ".in +0.25in\n");
         expand_string(header->input[i][1], "(unknown)", optr);
	 kfprintf(optr, ".in -0.25in\n");
      }

      kfprintf(optr, ".in -0.25in\n");
      kfprintf(optr, ".paragraph\n");
   }

   /*-- Output Arguments -- description of function arguments ---------*/
   if (header->out_cnt > 0)
   {
      kfprintf(optr, "\\fB\\s+1Output Arguments\\s-1\\fP\n");
      kfprintf(optr, ".br\n");
      kfprintf(optr, ".in +0.25in\n");

      for (i = 0; i < header->out_cnt; i++)
      {
	 kfprintf(optr, "\\f(CW%s\\fP\n", header->output[i][0]);
	 kfprintf(optr, ".br\n");
	 kfprintf(optr, ".in +0.25in\n");
	 expand_string(header->output[i][1], "(unknown)", optr);
	 kfprintf(optr, ".in -0.25in\n");
      }

      kfprintf(optr, ".in -0.25in\n");
      kfprintf(optr, ".paragraph\n");
   }

   /*-- Returns -- the return value function arguments ----------------*/
   print_header_section(optr, "Returns", header->returns);

   /*-- Description -- the Purpose: field from the header -------------*/
   print_header_section(optr, "Description", header->purpose);

   /*-- Side Effects -- side effects of the function ------------------*/
   print_header_section(optr, "Side Effects", header->sideeff);

   /*-- Restrictions -- restrictions in use of function ---------------*/
   print_header_section(optr, "Restrictions", header->restrictions);

   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	print_header_section - print section of function header
|
| Purpose:	This routine is used to print a section of a function
|		description generated for the .function pseudo macro.
|
| Input:	file     - The output file pointer
|		section  - The section of the function description being
|			   generated.
|		contents - The string sucked out of the header as the contents
|			   of the field corresponding to this section.
|
| Written By:	Neil Bowers
| Date:		3-sep-94
------------------------------------------------------------*/
static void
print_header_section(
   kfile    *file,
   kstring   section,
   kstring   contents)
{
   if (contents == NULL)
      return;

   kfprintf(file, "\\fB\\s+1%s\\s-1\\fP\n", section);
   kfprintf(file, ".br\n");
   kfprintf(file, ".RS\n");
   /* output_string(file, contents, TRUE); */
   expand_string(contents, "(unknown)", file);
   kfprintf(file, "\n.RE\n");
   kfprintf(file, ".paragraph\n");
}

/*-----------------------------------------------------------
| Routine Name:	generate_man1 - generate man1 style man page information
|
| Purpose:	
|
| Input:	object   - software object
|		section  - The section of the function description being
|			   generated.
|		contents - The string sucked out of the header as the contents
|			   of the field corresponding to this section.
|
| Written By:	Neil Bowers
| Date:		3-sep-94
------------------------------------------------------------*/
int
generate_man1(
   kobject   object,
   kfile    *file,
   kstring   level)
{
   kstring   oname;
   kstring   short_description;
   kstring   category;
   kstring   subcategory;
   kstring   icon_name;
   int       incantata;
   kstring   long_description;
   int       otype;
   kstring   typestring;


   if (!kcms_get_attributes(object,
			    KCMS_NAME,                    &oname,
			    KCMS_CMOBJ_SHORT_DESC,        &short_description,
			    KCMS_CMOBJ_CATEGORY,          &category,
			    KCMS_CMOBJ_SUBCATEGORY,       &subcategory,
			    KCMS_CMOBJ_ICON_NAME,         &icon_name,
			    KCMS_CMOBJ_INCANTATA,         &incantata,
			    KCMS_CMOBJ_PROG_MAN1LONGDESC, &long_description,
			    KCMS_CMOBJ_TYPE,              &otype,
			    KCMS_END))
      return FALSE;
   typestring = kcms_attr_int2string(KCMS_CMOBJ_TYPE, otype);
   if (short_description == NULL)
      short_description = "";
   if (category == NULL)
      category = "";
   if (subcategory == NULL)
      subcategory = "";
   if (long_description == NULL)
      long_description = "";
   if (icon_name == NULL)
      icon_name = "";

   kfprintf(file, ".section %s \"%s \\(em \\s-1\\fI%s\\fP\\s+1\" box\n",
	    level, oname, short_description);
   kfprintf(file, ".indexEntry definitive %s\n", oname);
   kfprintf(file, "\\fB\\s+1Object Information\\s-1\\fP\n");
   kfprintf(file, ".br\n");
   kfprintf(file, ".RS\n");
   kfprintf(file,
	    "Category: \\h'1.0i-\\w'Category:'u'\\fI%s\\fP\n"
	    "\\h'2.0i-\\w'%s'u'Subcategory: \\h'1.0i-\\w'Subcategory:'u'\\fI%s\\fP\n"
	    ".br\n"
	    "Operator: \\h'1.0i-\\w'Operator:'u'\\fI%s\\fP\n"
	    "\\h'2.0i-\\w'%s'u'Object Type: \\h'1.0i-\\w'Object Type:'u'\\fI%s\\fP\n"
	    ".br\n"
	    "In Cantata: \\h'1.0i-\\w'In Cantata:'u'\\fI%s\\fP\n"
	    ".br\n",
	    category, category, subcategory, icon_name, icon_name, typestring,
	    (incantata ? "Yes" : "No"));
   kfprintf(file, ".RE\n");
   kfprintf(file, ".paragraph\n");

   print_header_section(file, "Description", long_description);

   generate_man_section(file, object, KCMS_CMOBJ_PROG_MAN1EXAMPLES,
			"Examples");
   generate_man_section(file, object, KCMS_CMOBJ_PROG_MAN1SEEALSO,
			"See Also");
   generate_man_section(file, object, KCMS_CMOBJ_PROG_MAN1RESTRICT,
			"Restrictions");
   generate_man_section(file, object, KCMS_CMOBJ_PROG_MAN1REFERENCES,
			"References");

   generate_arguments(file, object);

   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	generate_arguments - generate CLUI information for object
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Neil Bowers
|          Date: Sep 23, 1994 16:20
------------------------------------------------------------*/
static void
generate_arguments(
   kfile     *file,
   kobject    object)
{
   kstring   routine      = "generate_arguments()";
   kform    *form        = NULL;
   kobject   file_object;
   kobject   toolbox;
   char     *panepath;
   kstring   oname;
   kstring   tbname;


   if (!kcms_get_attributes(object,
			    KCMS_CMOBJ_UIS_PANE, &file_object,
			    KCMS_NAME,           &oname,
			    KCMS_PARENT,         &toolbox,
			    KCMS_END)
       || !kcms_get_attribute(toolbox, KCMS_NAME, &tbname)
       || file_object == NULL
       || !kcms_get_attribute(file_object, KCMS_FOBJ_FULLNAME, &panepath)
       || panepath == NULL)
   {
      return;
   }

   if ((form = kvf_create_form(panepath, NONE, NULL, NULL)) == NULL)
   {
      kwarn(NULL, routine, "Couldn't get command-line information for "
	    "object %s in toolbox %s\n", oname, tbname);
      return;
   }

   kfprintf(file, "\\fB\\s+1Command Line Arguments\\s-1\\fP\n");
   kfprintf(file, ".br\n");
   kfprintf(file, ".RS\n");
   kfprintf(file, "\\f(CW\\s-2\n");
   kfprintf(file, ".vs -2\n");
   kfprintf(file, ".nf\n");
   kgen_interpret_clui_usage(file, form, tbname, oname, NULL);
   kfprintf(file, ".fi\n");
   kfprintf(file, ".vs +2\n");
   kfprintf(file, "\\fP\\s+2\n");
   kfprintf(file, ".RE\n");
   
   kvf_destroy_form(form);
}

/*-----------------------------------------------------------
| Routine Name: generate_man_section - generate section for a man1
| Purpose:
| Input:
| Written By:	Neil Bowers
| Date:		23-sep-94
------------------------------------------------------------*/
static void
generate_man_section(
   kfile   *file,
   kobject  object,
   int      attribute,
   kstring  heading)
{
   kstring  stringval;


   if (!kcms_get_attribute(object, attribute, &stringval)
       || stringval == NULL)
      
      return;

   kfprintf(file, "\\fB\\s+1%s\\s-1\\fP\n", heading);
   kfprintf(file, ".br\n");
   kfprintf(file, ".RS\n");
   expand_string(stringval, "(unknown)", file);
   kfprintf(file, ".RE\n");
   kfprintf(file, ".paragraph\n");
}

/*-----------------------------------------------------------
| Routine Name:	get_object - open software object
|
| Purpose:	This function is used to open a software object in a
|		specified toolbox.  We try to be smart, and keep an object
|		and/or toolbox open for as long as possible.
|
| Input:	tbname_wanted - The name of a toolbox.
|		oname_wanted  - The name of a software object in the toolbox.
|
| Returns:	A reference for the software object requested,
|		or NULL on failure.
|
| Written By:	Neil Bowers
| Date:		17-jul-94
------------------------------------------------------------*/
kobject
get_object(
   kstring  tbname_wanted,
   kstring  oname_wanted)
{
   kstring  tbname;
   kstring  oname;


   if (tbname_wanted == NULL)
   {
      if (cur_object != NULL)
	 kcms_close(cur_object);
      if (cur_toolbox != NULL)
	 kcms_close(cur_toolbox);
      return NULL;
   }

   if (cur_toolbox != NULL)
   {
      if (!kcms_get_attribute(cur_toolbox, KCMS_TB_NAME, &tbname))
         return NULL;

      if (!kstrcasecmp(tbname, tbname_wanted))
      {
         /*-- same toolbox, what about the object? --*/
         if (cur_object != NULL)
         {
            if (!kcms_get_attribute(cur_object, KCMS_CMOBJ_ONAME, &oname))
               return NULL;

            if (!kstrcasecmp(oname, oname_wanted))
            {
               return cur_object;
            }
            else
            {
               /*-- different object (in same toolbox) --*/
               kcms_close(cur_object);
               cur_object = kcms_open_cmobj(cur_toolbox, oname_wanted);
               return cur_object;
            }
         }
         else
         {
            cur_object = kcms_open_cmobj(cur_toolbox, oname_wanted);
            return cur_object;
         }
      }
      else
      {
         /*-- different toolbox, so close currently opened one --*/
         kcms_close(cur_toolbox);
         if ((cur_toolbox = kcms_open_toolbox(tbname_wanted)) == NULL)
            return NULL;
         cur_object = kcms_open_cmobj(cur_toolbox, oname_wanted);
         return cur_object;
      }
   }
   if ((cur_toolbox = kcms_open_toolbox(tbname_wanted)) == NULL)
      return NULL;
   cur_object = kcms_open_cmobj(cur_toolbox, oname_wanted);
   return cur_object;
}
