/*
 * Khoros: $Id$
 */

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>       Purpose: Contains common defines and structures
   >>>>                used by Khoros link list utilities located
   >>>>		       in the src/Lib/kutils library.
   >>>>
   >>>>    Written By: Mark Young
   >>>>
   >>>>          Date: Sep 11, 1992 22:23
   >>>>
   >>>> Modifications:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#ifndef _kutils_klist_h_
#define _kutils_klist_h_

        /*-------------------------------------*
        |       #include 
        ---------------------------------------*/


        /*-------------------------------------*
        |       #defines 
        ---------------------------------------*/

#define KLIST_HEAD	0
#define KLIST_PREV	1
#define KLIST_NEXT	2
#define KLIST_TAIL	3


        /*-------------------------------------*
        |       typedefs
        ---------------------------------------*/

typedef struct _klist
{
	kaddr identifier;
	kaddr client_data;
	unsigned char head;

	struct _klist *next, *prev;
} klist;


        /*-------------------------------------*
        |       macros
        ---------------------------------------*/


/************************************************************
*
*  Routine Name: klist_next - return the next entry on the list
*
*       Purpose: This routine simply returns the next klist entry
*		 on the link list.  If the current entry is last on the
*		 link list and the list is not a circular link list
*		 then NULL is returned.
*
*         Input: list - the list in which we will be returning the
*			next entry from
*
*        Output: none
*
*       Returns: the next entry on the list or NULL if at the end
*		 and the link list is not circular
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Sep 12, 1992 14:00
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: klist *klist_next(
*		 !   klist *list)
*
*************************************************************/

#define klist_next(list) (list != NULL ? list->next : NULL)


/************************************************************
*
*  Routine Name: klist_prev - return the previous entry on the list
*
*       Purpose: This routine simply returns the previous klist entry
*		 on the link list.  If the current entry is first on the
*		 link list and the list is not a circular link list
*		 then NULL is returned.
*
*         Input: list - the list in which we will be returning the
*			previous entry from
*
*        Output: none
*
*       Returns: the previous entry on the list or NULL if at the
*		 beginning and the link list is not circular
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Sep 12, 1992 14:00
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: klist *klist_prev(
*		 !   klist *list)
*
*************************************************************/

#define klist_prev(list) (list != NULL ? list->prev : NULL)


/************************************************************
*
*  Routine Name: klist_clientdata - return the client data associated with an
*				      entry on the list
*
*       Purpose: This routine simply returns the client data associated with
*		 the klist entry on the link list.  If the current entry is
*		 NULL then NULL is returned.  Otherwise, whatever is stored
*		 in the klist entry is returned.
*
*         Input: list - the list in which we will be returning the
*			client data
*
*        Output: none
*
*       Returns: the client data associated with the klist entry on the list
*
*  Restrictions: 
*    Written By: Mark Young and John Salas
*          Date: Dec 30, 1992 14:22
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: kaddr klist_clientdata(
*		 !   klist *list)
*
*************************************************************/

#define klist_clientdata(list) (list != NULL ? list->client_data : NULL)


/************************************************************
*
*  Routine Name: klist_identifier - return the identifier associated with an
*				    entry on the list
*
*       Purpose: This routine simply returns the identifier associated with
*		 the klist entry on the link list.  If the current entry is
*		 NULL then NULL is returned.  Otherwise, whatever is stored
*		 in the klist entry identifier is returned.
*
*         Input: list - the list in which we will be returning the
*			identifier for
*
*        Output: none
*
*       Returns: the client data associated with the klist entry on the list
*
*  Restrictions: 
*    Written By: Mark Young and John Salas
*          Date: Feb 23, 1993 21:16
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: kaddr klist_identifier(
*		 !   klist *list)
*
*************************************************************/

#define klist_identifier(list) (list != NULL ? list->identifier : NULL)


/************************************************************
*
*  Routine Name: klist_checkhead - check if the current entry
*				   is the head of the list
*
*       Purpose: This routine simply returns whether the current entry
*		 is the head of the link list.  If the entry is the head
*		 then TRUE is returned otherwise FALSE is returned.  This
*		 is helpful since if a link list is circular you cannot
*		 rely on the previous value being NULL.
*		 
*         Input: list - the list in which we will be returning whether
*			it is the head of the list or not
*
*        Output: none
*
*       Returns: return TRUE if the entry is the head of the link list,
*		 otherwise FALSE is returned
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Sep 12, 1992 14:00
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int klist_checkhead(
*		 !   klist *list)
*
*************************************************************/

#define klist_checkhead(list) (list == NULL ? FALSE : \
		       (list->prev == NULL || list->head == TRUE))


/************************************************************
*
*  Routine Name: klist_checktail - check if the current entry
*				   is the tail of the list
*
*       Purpose: This routine simply returns whether the current entry
*		 is the tail of the link list.  If the entry is the tail
*		 then TRUE is returned otherwise FALSE is returned.  This
*		 is helpful since if a link list is circular you cannot
*		 rely on the next value being NULL.
*		 
*         Input: list - the list in which we will be returning whether
*			it is the tail of the list or not
*
*        Output: none
*
*       Returns: return TRUE if the entry is the tail of the link list,
*		 otherwise FALSE is returned
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Sep 12, 1992 14:00
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int klist_checktail(
*		 !   klist *list)
*
*************************************************************/

#define klist_checktail(list) (list == NULL ? FALSE : \
		       (list->next == NULL || list->next->head == TRUE))


/************************************************************
*
*  Routine Name: klist_checkident - check if the identifier
*				    is currently on the link list
*
*       Purpose: This routine simply returns whether the identifier
*		 currently exists on the link list.  If the identifier exists
*		 then TRUE is returned otherwise FALSE is returned.
*		 
*         Input: list - the link list in which we will be checking the
*			identifier against
*		 identifier - the identifier to check for on the link list
*
*        Output: none
*
*       Returns: return TRUE if the identifier is the list,
*		 otherwise FALSE is returned
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Sep 12, 1992 14:00
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int klist_checkident(
*		 !   klist *list)
*
*************************************************************/

#define klist_checkident(list, identifier) \
			(!klist_locate(list, identifier) ? FALSE : TRUE)


/************************************************************
*
*  Routine Name: klist_checkentry - check if the klist entry is
*				    currently on the link list
*
*       Purpose: This routine simply returns whether the klist entry
*		 currently exists on the link list.  If the entry exists
*		 then TRUE is returned otherwise FALSE is returned.
*		 
*         Input: list -  the link list in which we will be checking the
*			 klist entry against
*		 entry - the klist entry to check for on the link list
*
*        Output: none
*
*       Returns: return TRUE if the entry is the list,
*		 otherwise FALSE is returned
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Sep 12, 1992 14:00
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int klist_checkentry(
*		 !   klist *list)
*
*************************************************************/

#define klist_checkentry(list, entry) \
		(klist_tail(list) != klist_tail(entry) ? FALSE : TRUE)


/************************************************************
*
*  Routine Name: klist_makecircular - changes a consecutive or linear
*				      link list into a circular link list
*
*       Purpose: This routine is used to make a consecutive or linear
*		 doubly linked list into a circular linked list.
*		 
*         Input: list -  the link list in which we will be making circular
*
*        Output: none
*
*       Returns: return the head of the circular link list or NULL upon
*		 failure.
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Sep 12, 1992 14:00
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int klist_makecircular(
*		 !   klist *list)
*
*************************************************************/

#define klist_makecircular(list) klist_merge(list, list)


/************************************************************
*
*  Routine Name: klist_makelinear - changes a circular link list into
*				    a consecutive or linear link list
*
*       Purpose: This routine is used to make a circular link list
*		 into a consecutive or linear linked list.
*		 
*         Input: list -  the link list in which we will be making linear
*			 link list.
*
*        Output: none
*
*       Returns: return the head of the circular link list or NULL upon
*		 failure.
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Sep 12, 1992 14:00
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int klist_makelinear(
*		 !   klist *list)
*
*************************************************************/

#define klist_makelinear(list) klist_merge(list, NULL)


        /*-------------------------------------*
        |       routines
        ---------------------------------------*/

/* list.c */
klist *klist_insert   PROTO((klist *, kaddr, kaddr, int, int));
klist *klist_add      PROTO((klist *, kaddr, kaddr));
klist *klist_copy     PROTO((klist *));
klist *klist_delete   PROTO((klist *, kaddr));
klist *klist_locate   PROTO((klist *, kaddr));
klist *klist_head     PROTO((klist *));
klist *klist_tail     PROTO((klist *));
klist *klist_sort     PROTO((klist *, kfunc_int, int));
klist *klist_merge    PROTO((klist *, klist *));
klist *klist_filelist PROTO((char *, char *, int));
klist *klist_dirlist  PROTO((char *, char *, char *, int, int));
klist *klist_locate_clientdata   PROTO((klist *, kaddr));

void klist_free  PROTO((klist *, kfunc_void));
int  klist_split PROTO((klist *, klist **, klist **));
int  klist_size  PROTO((klist *));

char **klist_to_array PROTO((klist *, int *));

#endif /* _kutils_klist_h_ */
/* Don't add after this point. */
