/* xbgsun.c:
 *
 * Jim Frost, Associative Design Technology
 * 12.15.88
 *
 * this reads a sun raster image file and displays it in a window.  it is
 * derived from the 12.14.88 version of xbgsun.
 *
 * 12.30.80 broken apart to use a function library common with xbgsun.
 * 12.15.88 original version
 *
 * May 1989 -- hacked by S. Levy, Geometry Project, University of Minnesota
 *		to act like Pixar "see" for displaying
 *		raw raster images in various formats and sizes
 * Feb 1990 -- hacked by S. Levy to allow loading Nx3-byte color maps
 *		and to properly handle different kinds of displays.
 */

#include <stdio.h>
#include <ctype.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>

extern Display          *disp;
extern Screen		*screen;
extern Visual		*visual;
extern int               width, height, depth;
extern int               bytesperline;
extern int               xorigin, yorigin;
extern int               pwidth, pheight;
extern int               cx, cy;
extern int               cwidth, cheight;
extern int               verbose;
extern int		 raw;
extern long		 offset;
extern char             *fgcolor, *bgcolor;
extern Pixmap            pic;
extern XWindowAttributes wa;
extern unsigned char	*data;


void usage(name)
     char *name;
{
    printf("\
Usage: %s [-display display_name] [-geometry X_geometry] \n\
	[-foreground color] [-background color]\n\
	[-16bw] [-8bw] [-11bw] [-lut colormap.lut] [-seek file_offset]\n\
	[-w xsize ysize] rasterfile\n\
rasterfile may be '-' for stdin.  -16bw &c, -seek, -w are for 'raw' images.\n",
	name);
    exit(1);
}

void nuked(disp)
Display *disp;
{
  exit(0);
}

/* this returns the name of the file with no added junk
 */

char *tail(name)
char *name;
{ int a;

  if (strlen(name) < 2)
    return(name);
  for (a= strlen(name); (a > 0) && (name[a - 1] != '/'); a--)
    ;
  return(&name[a]);
}

main(argc, argv)
int argc;
char *argv[];
{ int           	a, dummy,
                	x = 0, y = 0;      /* where to put the window */
  char                 *dname;    /* display name */
  XGCValues     	gcv;
  GC            	gc;
  Window        	window;    /* window for image */
  int			mydepth;   /* depth of the visual we're creating the window on */
  XSetWindowAttributes	swa;
  XSizeHints    	sh;
  XEvent		ev;
  char			cmapname[256];

  if (argc < 2)
    usage(argv[0]);

  dname= NULL;
  verbose= 1;
  xorigin= yorigin= cx= cy= 0;
  cmapname[0] = '\0';

  /* parse options
   */

  for (a= 1; (a < argc) && argv[a][0] == '-' && argv[a][1] != '\0'; a++) {
    if (isoption("-help", argv[a], 2))
      usage(argv[0]);
    else if (isoption("-display", argv[a], 2))
      dname= argv[++a];
    else if (isoption("-geometry", argv[a], 2))
      XParseGeometry(argv[++a], &x, &y, &dummy, &dummy);
    else if (isoption("-foreground", argv[a], 2))
      fgcolor= argv[++a];
    else if (isoption("-background", argv[a], 2))
      bgcolor= argv[++a];
    else if (isoption("-seek", argv[a], 2)) {
      raw++;
      offset = atoi(argv[++a]);
      if(depth == 0) depth = 8;
    } else if (isoption("-8bw", argv[a], 2)) {
      depth = 8;
    } else if (isoption("-11bw", argv[a], 3)) {
      depth = 11;
    } else if (isoption("-16bw", argv[a], 3)) {
      depth = 16;
    } else if (isoption("-w", argv[a], 2)) {
      int w[4];
      int i;
      for(i=0; i<3 && a<argc-1 && isdigit(argv[a+1][0]); i++)
	w[i] = atoi(argv[++a]);
      if(i == 4) {
	width = w[1]-w[0]+1;
	height = w[3]-w[2]+1;
      } else if(i == 2) {
	width = w[0];
	height = w[1];
      } else {
	printf("Expected -w xmin xmax ymin ymax -or- -w xsize ysize\n");
	exit(1);
      }
      raw++;
      if(depth == 0) depth = 8;
    } else if (isoption("-lut", argv[a], 2)) {
      strcpy(cmapname, argv[++a]);
    }
  }

  if (a != argc - 1)
    usage(argv[0]);

  /* try to load the image
   */

  if (loadImage(argv[a]) < 0)
    exit(1);
  cwidth= pwidth= width;
  cheight= pheight= height;

  /* open display and get its configuration
   */

  if ((disp= XOpenDisplay(dname)) == NULL) {
    printf("Can't open display.\n");
    exit(1);
  }
  mydepth = GetCmapped(disp, &screen, &visual);


  XSynchronize(disp, 1);

  XSetIOErrorHandler(nuked);

  XGetWindowAttributes(disp, RootWindowOfScreen(screen), &wa);
  if(mydepth > 0) {
	/* colormapped display */
	wa.colormap = XCreateColormap(disp, RootWindowOfScreen(screen),
				visual, AllocAll);
	wa.depth = mydepth;
  } else if (wa.colormap == 0) {
	wa.colormap= DefaultColormapOfScreen(screen);
  }

  /* allocate our destination pixmap or window
   */

  pic= XCreatePixmap(disp, RootWindowOfScreen(screen),
		       width, height, wa.depth);

  /* get the image to the server
   */

  if (depth == 1)
    sendMonoImage();
  else {
    if (depth <= wa.depth)
      sendColorImage();
    else {
      translateColors();
      sendColorPixels();
    }
  }

  /* get a window for the image
   */

  swa.event_mask= ButtonPressMask | Button1MotionMask | ExposureMask
		| EnterWindowMask | LeaveWindowMask | ColormapChangeMask;
  swa.colormap = wa.colormap;
  window= XCreateWindow(disp, DefaultRootWindow(disp),
			x, y, width, height, 0, wa.depth,
			InputOutput, visual, CWEventMask|CWColormap, &swa);
  XStoreName(disp, window, tail(argv[a]));
  XSetIconName(disp, window, tail(argv[a]));
  sh.width= width;
  sh.height= height;
  sh.flags= USSize;
  XSetNormalHints(disp, window, &sh);

  gcv.function= GXcopy;
  gc= XCreateGC(disp, window, GCFunction, &gcv);

  XDefineCursor(disp, window, XCreateFontCursor(disp, XC_crosshair));
  XMapWindow(disp, window);

  /* set up the colormap if necessary and set up the window
   */

  if (wa.depth > 1 && visual->class == PseudoColor) {
    readlut(cmapname[0] != '\0' ? cmapname : NULL, disp, &wa.colormap);
    XSetWindowColormap(disp, window, wa.colormap);
  }
  XCopyArea(disp, pic, window, gc, 0, 0, width, height, 0, 0);

  printf("Click on:\n\
	LEFTMOUSE to inquire about image point\n\
	MIDDLEMOUSE to select new color map from .lut file\n\
	RIGHTMOUSE to exit\n");

  for (;;) {
    int k;
    int x, y;
    char newname[256];

    XNextEvent(disp, &ev);
    switch(ev.xany.type) {

    case MotionNotify :		/* We asked (only) for Button1Motion */
	x = ev.xmotion.x;
	y = ev.xmotion.y;
	goto inquire;

    case ButtonPress :
      switch(ev.xbutton.button) {

	case Button1: /* Button 1 -- inquire about a point */
	  x = ev.xbutton.x;
	  y = ev.xbutton.y;
	inquire:
	  k = (data[y * bytesperline + x * depth / 8]
		>> (8-depth - (x*depth % 8))) & ((1 << depth) - 1);
	  printf("x %4d y %4d pixel %3d = 0x%02x  \r",
		  x, y, k, k);
	  fflush(stdout);
	  break;

	case Button2:
	  if(wa.visual->class != PseudoColor) {
	    printf("Can't handle colormaps on this display.\n");
	    break;
	  }
	  printf("Enter new colormap name [%s]: ", cmapname);
	  newname[0] = '\0';
	  fgets(newname, sizeof(newname), stdin);
	  k = strlen(newname);
	  if(k > 1) {
	    newname[k - 1] = '\0';
	    if(readlut(newname, disp, &wa.colormap))
		strcpy(cmapname, newname);
	    XSetWindowColormap(disp, window, wa.colormap);
	  }
	  break;
	  
	case Button3:
	  XFreePixmap(disp, pic);
	  XDestroyWindow(disp, window);
	  XCloseDisplay(disp);
	  exit(0);
      }
      break;

    case Expose :
      XCopyArea(disp, pic, window, gc, ev.xexpose.x, ev.xexpose.y,
		ev.xexpose.width, ev.xexpose.height,
		ev.xexpose.x, ev.xexpose.y);
      break;

    case ColormapNotify:
      XGetWindowAttributes(disp, window, &wa);
      if(wa.colormap == None)	/* Our master must have disappeared */
	wa.colormap = DefaultColormapOfScreen(screen);
      break;

    case EnterNotify :
      if (wa.depth > 1)
	XInstallColormap(disp, wa.colormap);
      break;

    case LeaveNotify :
      if (wa.depth > 1)
	XUninstallColormap(disp, wa.colormap);
      break;
    }
  }
}

readlut(fname, dpy, curmapp)
    char *fname;
    Display *dpy;
    Colormap *curmapp;
{
    unsigned char cmap[256*3];
    XColor colors[256];
    register XColor *c;
    register unsigned char *p;
    int ncolors;
    register int i;
    FILE *f;

    if(fname != NULL) {
	f = fopen(fname, "r");
	if(f == NULL) {
	    fprintf(stderr, "Can't open colormap file %s\n", fname);
	    return 0;
	}
	ncolors = fread(cmap, 3, 256, f);
	fclose(f);
	if(ncolors <= 1) {
	    fprintf(stderr, "Can't read %s -- expected up to 256 three-byte entries\n",
		fname);
	    return 0;
	}
	for(i = 0, c = colors, p = cmap; i < ncolors; c++, i++) {
	    c->pixel = i;
	    c->red = *p++ * 257;
	    c->green = *p++ * 257;
	    c->blue = *p++ * 257;
	    c->flags = DoRed|DoGreen|DoBlue;
	}
    } else {
	ncolors = 256;
	for(i = 0, c = colors; i < ncolors; c++, i++) {
	    c->pixel = i;
	    c->flags = DoRed|DoGreen|DoBlue;
	    if(i < 8) {		/* Basic 8 colors */
		c->red =   i&1 ? 65535 : 0;
		c->green = i&2 ? 65535 : 0;
		c->blue =  i&4 ? 65535 : 0;
	    } else if(i < 31) {	/* Gray ramp */
		c->red = c->green = c->blue = (i-8) * 65535 / 22;
	    } else {		/* 5*9*5 color cube */
		c->red = ((i-31) % 5) * 65535 / 4;
		c->green = (((i-31) / 5) % 9) * 65535 / 8;
		c->blue = ((i-31) / 45) * 65535 / 4;
	    }
	}
    }
    if(*curmapp == DefaultColormapOfScreen(screen)) {
	*curmapp = XCreateColormap(disp, RootWindowOfScreen(screen),
				visual, AllocAll);
    }
    XStoreColors(dpy, *curmapp, colors, ncolors);
    return 1;
}

int
GetCmapped(disp, scrp, visp)
    register Display *disp;
    Screen **scrp;
    Visual **visp;
{
    int nvi;
    register XVisualInfo *vi;
    XVisualInfo viproto;

    viproto.class = PseudoColor;
    viproto.depth = 8;
    nvi = -1;			/* Do I need this? */
    vi = XGetVisualInfo(disp, VisualDepthMask|VisualClassMask,
		&viproto, &nvi);
    if(nvi > 0) {
	*visp = vi[0].visual;
	*scrp = ScreenOfDisplay(disp, vi[0].screen);
	return vi[0].depth;
    }

    /* No luck, use default */
    *scrp = DefaultScreenOfDisplay(disp);
    *visp = DefaultVisualOfScreen(*scrp);
    return 0;
}
