/* setup.c */
/* Copyright 1988 John M. Sullivan.  See main program for details. */
/*** Setup the voronoi diagram topologically ***/

#include "vcs.h"

typedef
struct tetq {
	facet_edge tri;
	site new;
	int cnt;
	facet_edge othertris[3];
} * tetq;

#include "queue.h"
def_q(tetq)

static struct q_of_tetq Queue;

/***** External functions
void voronoi_diagram(): Calculates the voronoi diagram of the Sites
*****/

/***** local functions *****/
static void add_queue(); 	 /* add a new tetrahedron to the queue */
static site get_fourth();	 /* get fourth point in a tetrahedron */
static void make_faces();	 /* call add*faces to construct tetr */
static void add_012_faces();	 /* handle case of < 3 new triangles */
static void add_three_faces();	 /* handle case of 3 new triangles */
static void print_q();		 /* print out a q entry for debugging */
static site intermediate();	 /* help in case of 5 pts on sphere */

/***********
This function calculates the voronoi diagram.
First we enqueue the first three.
Then while the q is non-empty, we grab something off it, construct and
link the appropriate wings, and put more stuff on the q
***********/

tetq
newq(w,s)
facet_edge w;
site s;
{
    register tetq nq;
    nq = mem_alloc(1,tetq);
    nq->tri = w;
    nq->new = s;
    nq->cnt = 0;
    nq->othertris[0] = nq->othertris[1] = nq->othertris[2] = NULL;
    return nq;
}

void
voronoi_diagram()
{
    facet_edge first3;
    site fourth;
    tetq tq;

    first3 = first_three();
    fourth = get_fourth(first3);
    first3->vert[0] = first3->nextfacet->vert[0] = first3->prevfacet->vert[0] =
	inf_vert(first3,1);
	/* backwards since this wing faces in */
    empty(Queue);
    enq(Queue,newq(first3,fourth),tetq);
    fourth->cnst_q = Queue.head;
    /* could call add_queue but we know this is what it will do */

    while non_empty(Queue)
    {
	deq(Queue,tq,tetq);
	tq->new->cnst_q = (Queue.head && Queue.head->data->new == tq->new)?
				Queue.head : NULL;
	DBG(2, print_q(tq); putstr("dequeued\n"))
	make_faces(tq);
    }
}

static void
add_queue(wing)
register facet_edge wing;
{
    register site fourth;
    register tetq_q_entry q;
    corner v;

    if (match(wing,wing->cell[0]) || match(wing,wing->cell[1])
				  || match(wing,wing->cell_2_))
	return; /* same tet was in queue already */

    /* if not found, put at end of list after calling get_fourth */
    fourth = get_fourth(wing);
    if (fourth = get_fourth(wing)
#ifdef CRAY
,no_op(),fourth
#endif
)
    {
	unless (fourth->cnst_q && fourth->cnst_q->data->new==fourth)
	{
	    enq(Queue,newq(wing,fourth),tetq);
	    fourth->cnst_q = Queue.tail;
	    DBG(2, print_q(Queue.tail->data); putstr("enqueued\n"))
	}
	else
	{
	    for (q=fourth->cnst_q;
		 q->next && q->next->data->new==fourth; q=q->next)
		;
	    enq_after(Queue,q,newq(wing,fourth),tetq);
	    DBG(2,print_q(q->next->data); putstr("enqueued\n"))
	}
    }
    else
    {
	v = inf_vert(wing,0);
	for_facets(wing,f)
	    if (!f->vert[0])
		f->vert[0] = v;
	    else
		f->vert[1] = v;
	end_facets(wing,f)
	    DBG(2,printf("triangle on edge not queued\n"))
    }
}

bool
match(w,s)
facet_edge w;
site s;
{
    tetq_q_entry qq;
    tetq q;
    site fourth;

    for(qq=s->cnst_q; qq && qq->data->new == s; qq=qq->next)
    {
	q = qq->data;
	if (    (find(w->cell[0],q->tri) || w->cell[0]==q->new)
	     && (find(w->cell[1],q->tri) || w->cell[1]==q->new)
	     && (find(w->cell_2_,q->tri) || w->cell_2_==q->new))
	{
	    while (find(q->tri->cell[1], w))
		q->tri = q->tri->nextfacet;
	    fourth = q->tri->cell[1];

	    if (volume(w->cell[0]->loc,w->cell[1]->loc,
			w->cell_2_->loc,fourth->loc) < 0.)
	    {
		DBG(1,
		    fprint_wing(stderr,w);
		    fprint_wing(stderr,q->tri);
		    fprint_site(stderr,q->new);
		    if (w->vert[0])
			fprint_corner(stderr,w->vert[0]);
		    else if (w->vert[1])
			fprint_corner(stderr,w->vert[1]);
		)
		error("Found two tetrahedra on same side of triangle %s",
		    "; try some combination of flags r,i");
	    }

	    while (q->new != w->cell[1])
		w = w->nextfacet;	/* now cell[1]'s are distinct cells */
	    q->othertris[q->cnt++] = w;
		DBG(2,putstr("enqueued after "); print_q(q))
	    return TRUE;
	}
    }
    return FALSE;
}

static site
get_fourth(f)
facet_edge f;
{
    site s4;
    point p1,p2,p3, area321;
    point_with_norm2 ccra;
    point oldvc,newvc;
    real vol0,bda,toler,tmp;
    int i1,i2,i3;

    p1 = f->cell[0]->loc;
    p2 = f->cell[1]->loc;
    p3 = f->cell_2_->loc;
    area321 = vor_corner_prep(p3,p2,p1);
    vol0 = dot_q(area321,p1);
    ccra = mem_alloc(1,point_with_norm2);
    cross_x(area321+2,area321,pnt(ccra));
    /*ada*/ nm2(ccra) = norm2_q(area321)/2;
    bda = dot_q(area321+1,area321);
    toler = 1e-12;
loop:
    i1 = f->cell[0] - Sites;
    i2 = f->cell[1] - Sites;
    i3 = f->cell_2_ - Sites;
    s4 = Sites + get4lp(pwn(Sites->loc),&N_Sites,&i1,&i2,&i3,
			area321,&vol0,ccra,&bda,&toler);
    if (s4<Sites) s4 = 0;
    if (s4 && (f->vert[0] || f->vert[1]))
    {
	if (f->vert[0]) oldvc = f->vert[0]->loc;
	else oldvc = f->vert[1]->loc;
	newvc = voronoi_corner(s4->loc,p1,p2,p3);
	tmp = volume(s4->loc,p1,p2,p3);
	/*newvc = do_vor_corner(s4->loc,area321,
		    (tmp=dot_q(s4->loc,area321)-vol0));*/
	if (dot_q(oldvc,area321) > dot_q(newvc,area321))
	{
	    print_point(oldvc); print_point(newvc); print_point(s4->loc);
	    print_wing(f);
	    printf("\n vol %f found bug, going back\n",tmp);
	    toler *= 10;
	    goto loop;
	}
    }
    mem_free(ccra);
    mem_free(area321);
    return s4;
}

static void
make_faces(q)
tetq q;
{
    if (q->cnt==0)
	add_three_faces(q->tri,q->new);
    else if (q->cnt>=4)
	error("bad queue->cnt %d in make_faces",q->cnt);
    else
	add_012_faces(q->tri,q->othertris[0],q->othertris[1],q->othertris[2]);
    mem_free(q);
}

static void
add_012_faces(f0,f1,f2,f3)
facet_edge f0,f1,f2,f3;
{
    corner c;
    bool new2,new3;

    while (find(f0->cell[1],f1))
	f0 = f0->nextfacet;
    c = new_corner(f1->cell[1],f0->cell[1],f0->cell_2_,f0->cell[0]);

    if (new2 = !f2)
	f2 = make_wing(c->cell[1],c->cell[0],c->cell[3]);

    if (new3 = !f3)
    {
	if (find(c->cell[3],f2))
	    f3 = make_wing(c->cell[1],c->cell[2],c->cell[0]);
	else
	    f3 = make_wing(c->cell[1],c->cell[0],c->cell[3]);
    }

    link_wings(f0,f1,c);
    link_wings(f0,f2,c);
    link_wings(f0,f3,c);
    link_wings(f1,f2,c);
    link_wings(f1,f3,c);
    link_wings(f2,f3,c);

    c->one_fe = f0;

    if (new2)
	add_queue(f2);
    if (new3)
	add_queue(f3);
}

static void
add_three_faces(f,s)
facet_edge f;
site s;
{
    facet_edge f1,f2,f3;
    corner c;

    c = new_corner(s,f->cell[1],f->cell_2_,f->cell[0]);

   /*f = make_wing(c->cell[1],c->cell[2],c->cell[3]);*/
    f1 = make_wing(c->cell[2],c->cell[3],c->cell[0]);
    f2 = make_wing(c->cell[1],c->cell[0],c->cell[3]);
    f3 = make_wing(c->cell[0],c->cell[1],c->cell[2]);

    link_wings(f,f1,c);
    link_wings(f,f2,c);
    link_wings(f,f3,c);

    /* could just do link_wings here too, but we know what is happening */
    link_edge_loops(f1->prevfacet,f2->prevfacet,c);
    link_edge_loops(f1,f3,c);
    link_edge_loops(f2->nextfacet,f3->nextfacet,c);

    DBG(3, if (s->one_fe)
	{print_point(s->loc);printf(": already has one_fe set\n");})
    c->cell[0]->one_fe = f1;
    c->one_fe = f;

    add_queue(f1);
    add_queue(f2);
    add_queue(f3);
}

static void
print_q(q)
tetq q;
{
print_num(q->new); printf(" ");
print_num(q->tri->cell[0]); printf(" ");
print_num(q->tri->cell[1]); printf(" ");
print_num(q->tri->cell_2_); printf(" ");
    printf(" [%d]\n",q->cnt);
}
