/* Copyright 1988 John M. Sullivan.  See main program for details. */
#include "point.h"

/* allocate and initialize a new point structure */
/* routines with _x always do the same thing, but use the point P for */
/* the result instead of allocating new space */
point
new_point_x(x,y,z,P)
real x,y,z;
register point P;
{
    P->x = x; P->y = y; P->z = z;
    return P;
}

void
fprint_point(fp,p)  /* print coords of point */
file fp;
register point p;
{
    if (p)
	fprintf(fp,"(%.7f,%.7f,%.7f)",p->x,p->y,p->z);
    else
	fputs("(nowhere)",fp);
}

point
fread_point_x(fp,P)
file fp;
register point P;
{
    char c;

    fscanf(fp," %c",&c);
    if (c == '(')
    {
	if (3 != fscanf(fp," %lf , %lf , %lf )",&P->x,&P->y,&P->z))
	    return(NULL);
    }
    else
    {
	ungetc(c,fp);
	if (3 != fscanf(fp,"%lf %lf %lf",&P->x,&P->y,&P->z))
	    return(NULL);
    }
    return(P);
}

static struct point Origin_Struct = {0.,0.,0.};
point Origin = &Origin_Struct;

real
dot(p,q)
register point p,q;
/* compute dot product of two vectors */
{
    return dot_q(p,q);
}

point
cross(p,q)
register point p,q;
/* compute cross product of two vectors */
{
    return pt(  p->y*q->z - p->z*q->y,
		p->z*q->x - p->x*q->z,
		p->x*q->y - p->y*q->x);
}

point
cross_x(p,q,P)
register point p,q;
register point P;
{
    real x,y; /* so that P can be the same address as p,q */
    x = p->y*q->z - p->z*q->y;
    y = p->z*q->x - p->x*q->z;
    P->z = p->x*q->y - p->y*q->x;
    P->x = x; P->y = y;
    return P;
}

real
norm2(p)
register point p;
/* norm squared of a vector */
{
    return norm2_q(p);
}

point
scalar(a,p)
real a;
register point p;
/* scalar product */
{
    return pt(a*p->x,a*p->y,a*p->z);
}

point
scalar_x(a,p,P)
real a;
register point p;
register point P;
{
    P->x = a * p->x;
    P->y = a * p->y;
    P->z = a * p->z;
    return P;
}

point
diff(p,q)
register point p,q;
/* gives the difference between two points */
{
    return pt(p->x-q->x, p->y-q->y, p->z-q->z);
}

point
sum(p,q)
register point p,q;
/* adds two points */
{
    return pt(p->x+q->x, p->y+q->y, p->z+q->z);
}

point
diff_x(p,q,P)
register point p,q;
register point P;
{
    P->x = p->x - q->x;
    P->y = p->y - q->y;
    P->z = p->z - q->z;
    return P;
}

point
sum_x(p,q,P)
register point p,q;
register point P;
{
    P->x = p->x + q->x;
    P->y = p->y + q->y;
    P->z = p->z + q->z;
    return P;
}

real
dist2(p,q)
register point p,q;
{
    struct point temp;
    return norm2(diff_x(p,q,&temp));
}

real
triple(p,q,r)
register point p,q,r;
/* triple product gives volume of ||opiped, =3!*vol(tertrahedron 0,p,q,r) */
{
    struct point storage;

    return dot(p,cross_x(q,r,&storage));
/*  return (p->x*q->y*r->z + p->y*q->z*r->x + p->z*q->x*r->y  **
**        - p->z*q->y*r->x - p->x*q->z*r->y - p->y*q->x*r->z) */
}

real
volume(p,q,r,s)
register point p,q,r,s;
/* return 6*volume of arbitrary tetrahedron */
{
    struct point P,Q,R;
    return triple(diff_x(p,s,&P),diff_x(q,s,&Q),diff_x(r,s,&R));
/*  return (triple(p,q,r) + triple(q,p,s) + triple(r,s,p) + triple(s,r,q)); */
}

point
area(p,q,r)
register point p,q,r;
/* gives 2* the vector area of the arbitrary triangle p,q,r */
{
    struct point P,Q;
/*  return sum(cross(p,q),sum(cross(q,r),cross(r,p))); */
    return cross(diff_x(p,r,&P),diff_x(q,r,&Q));
}

point
area_x(p,q,r,X)
register point p,q,r;
register point X;
{
    struct point P,Q;
    return cross_x(diff_x(p,r,&P),diff_x(q,r,&Q),X);
}

real
area2(p,q,r)
register point p,q,r;
{
    struct point P;
    return norm2(area_x(p,q,r,&P));
}

point
voronoi_corner_x(p,q,r,s,X)
register point p,q,r,s;
register point X;
/* If p,q,r,s are points which form a delauney tetrahedron, this routine  **
** will calculate the location of the voronoi corner in the middle */
{
    struct point P,Q,R,S;
    return divide_x( sum_x( scalar_x(norm2(p)/2., area_x(q,r,s,&P), &P),
		      sum_x( scalar_x(norm2(q)/2., area_x(p,s,r,&Q), &Q),
		       sum_x( scalar_x(norm2(r)/2., area_x(s,p,q,&R), &R),
			       scalar_x(norm2(s)/2., area_x(r,q,p,&S), &S),
				  &R), &Q), &P),
		   volume(p,q,r,s), X);
}

point
vor_corner_prep(q,r,s)
register point q,r,s;
/* this pulls outside a loop all calculations for voronoi **
** corners of arbitrary p with q,r,s */
{
    struct point temp;
    register point answer;
    real q2,r2,s2;

    answer = mem_alloc(3,point);
    q2 = -norm2_q(q)/2;
    r2 = -norm2_q(r)/2;
    s2 = -norm2_q(s)/2;

    cross_x(q,r,answer);
    scalar_x(s2,answer,answer+1);

    cross_x(r,s,&temp); sum_x(answer,&temp,answer);
    scalar_x(q2,&temp,&temp); sum_x(answer+1,&temp,answer+1);

    cross_x(s,q,&temp); sum_x(answer,&temp,answer);
    scalar_x(r2,&temp,&temp); sum_x(answer+1,&temp,answer+1);

    diff_x(q,r,answer+2); scalar_x(s2,answer+2,answer+2);
    diff_x(r,s,&temp); scalar_x(q2,&temp,&temp);
    sum_x(&temp,answer+2,answer+2);
    diff_x(s,q,&temp); scalar_x(r2,&temp,&temp);
    sum_x(&temp,answer+2,answer+2);
    /* now voronoi corner is (p^2*ans0 + ans1 + p x ans2)/volume */

    return answer;
}

point
do_vor_corner_x(p,ans,vol,vc)
register point p,ans,vc;
real vol;
{
    real p2;

    p2 = norm2_q(p)/2;
    vc = cross_x(p,ans+2,vc);
    vc->x = (vc->x + p2*ans->x + (ans+1)->x) / vol;
    vc->y = (vc->y + p2*ans->y + (ans+1)->y) / vol;
    vc->z = (vc->z + p2*ans->z + (ans+1)->z) / vol;
    return vc;
}
