/* areas.c */
/* Copyright 1988 John M. Sullivan.  See main program for details */
/* Routines for calculating areas and volumes and derivatives */

/***** Externally called routines:
real compute_areas(): fills in area, unit_normal fields of all faces
	returns total interface area between different colors
void derivs(s): calculates all derivatives if s moves
*****/

#include "vcs.h"
/* local functions */
static point face_area();
static void fill_in_areas();
static void add_vol();
static void calc_face_deriv();
static void calc_edge_deriv();

/* compute once and for all the vector area of face f */
static point
face_area(f,a)
facet_edge f;
point a;
{
    struct point p[1];
    set_to_0(a);

    for_edges(f,e)
	if (infinite(e->vert[0]) || infinite(e->vert[1]))
	    return NULL; /* area is infinite */
	cross_x(e->vert[0]->loc, e->vert[1]->loc, p);
	sum_q(a,p);
    end_edges(f,e)

    return a;
}

static face_info Face_infos;

/* fill in area of face and add to total interface area */
static
void
fill_in_areas(face)
facet_edge face;
{
    register face_info info;
    struct point tmp[1];
    int count;
    real ww;
    int w;
    clr k[2];

    info = &Face_infos[N_DEdge++];
    k[0] = face->cell[0]->color;
    k[1] = face->cell[1]->color;
    info->which_wulff = -1;
    info->which_wulff = 0;
    if (face_area(face,&info->fi_un))
    {
	info->area = sqrt(norm2_q(&info->fi_un));
	divide_q(&info->fi_un,info->area);
	info->area /= 2.;
	if (Wulff_Count)
	    for (w=0; w<Wulff_Count; w++)
		if ((ww = info->area*dot_q(&info->fi_un,Wulff_Vect+w)) >
			    info->wulff_energy)
		    info->which_wulff = w, info->wulff_energy=ww;
    }
    else
    {
	unless (ignored(k[0]) && ignored(k[1]))
	    error("unexpected infinite area face %d %d",k[0],k[1]);
	diff_x(face->cell[0]->loc,face->cell[1]->loc,&info->fi_un);
	info->area = sqrt(norm2_q(&info->fi_un));
	divide_q(&info->fi_un,info->area);
	/* may not agree in sign with orientation of vertex loop, as it does
	   in finite case */
	info->area = info->wulff_energy = HUGE;
    }
    info->height = /* sqrt(dist2(face->cell[0]->loc,face->cell[1]->loc)) */
		 fabs(dot(&info->fi_un,
			  diff_x(face->cell[0]->loc,face->cell[1]->loc,tmp)));

    for_edges(face,e)
	e->fi = info;
    end_edges(face,e)

#ifdef VAX
    if (info->area == HUGE)
    {
	face->cell[0]->volume = face->cell[1]->volume = HUGE;
	return;
    }
#endif

    face->cell[0]->volume += info->height*info->area/6.;
    face->cell[1]->volume += info->height*info->area/6.;

    if (k[0] == k[1] || info->area == HUGE)
	return;

    if (Flag_torus && k[0] <= 0 && k[1] <= 0)
	return;
    if (Flag_torus && Torus_Same && k[0]==-k[1])
	return;
    count = 1;
    if (Flag_torus)
    {
	if (k[0]>0 && k[1]>0)
	    count = 2;
    }
    else if (Flag_kelvin)
    {
	unless(ignored(k[0]) || ignored(k[1]))
	    count = 2;
    }
/* if face is between two real cells, counts twice as much as bndry face */
    Total_Area += info->area * count;
    Total_Energy += info->wulff_energy * count;
}

/* fill in area of each face in the diagram, and return total interface area */
real
compute_areas()
{
    site s;
    clr k;
    real ave;

    Face_infos = mem_alloc(N_Sites+N_DTri-N_DTet-1,face_info);
    trace_all_faces(fill_in_areas);
    if (N_DEdge != N_Sites+N_DTri-N_DTet-1)
	error("Euler number didn't come out right (%d faces found, not %d)",
		    N_DEdge,N_Sites+N_DTri-N_DTet-1);

    for_sites(s)
	if (s->color>0)
	    Volumes[s->color] += s->volume;
    for_colors(k)
	Total_Volume += Volumes[k];
    ave = Total_Volume/Max_color; /* average volume */

    for_colors(k)
	Desired_Changes[k] = (Flag_kelvin)? (ave-Volumes[k]) :
				(Targets? Targets[k]-Volumes[k] : 0);

    return Total_Area;
}

/* called once for each site of positive color */
/* lets us determine how to move the site */
void
derivs(s)
site s;
{
    trace_faces_of_cell(s,calc_face_deriv);
    trace_edges_of_cell(s,calc_edge_deriv);
}

static void
calc_face_deriv(fe,s)
facet_edge fe;
site s;
{
    struct point grad[2];
    struct point tmp[1];
    site t;
    real coeff[2];

    if (fe->fi->area == HUGE)
	return;	/* must be between ignored color sites, so doesn't matter */
    t = other_cell(fe,s);
    set_to_0(grad); /* area grad */
    set_to_0(grad+1); /* wulff grad */
    for_edges(fe,e)
	diff_x(e->vert_2_->loc,e->vert[0]->loc, tmp);
	coeff[0] = triple(unit_normal(e), dffrntl(e->vert[1],s), tmp);
	if (Wulff_Count)
	    coeff[1] = triple(wulff_vect(e), dffrntl(e->vert[1],s), tmp);
	diff_x(e->vert[1]->loc,s->loc, tmp);
	scalar_q(coeff[0], tmp);
	sum_q(grad, tmp);
	scalar_q(coeff[1]/coeff[0], tmp);
	sum_q(grad+1, tmp);
    end_edges(fe,e)

    if (s->color != t->color)
	unless (Flag_torus && s->color == -t->color && Torus_Same)
	{
	    sum_q(area_grad(s),grad);
	    if (Wulff_Count)
		sum_q(wulff_grad(s),grad+1);
	}

    scalar_q(fe->fi->height/6.,grad);
    unless(ignored(s->color))
	sum_q(vgrad(s,s),grad);
    if (Flag_kelvin && !t->color)
    {
	if (!Pos_Err)
	    warning("moving site (color %d) next to color 0 site",s->color);
	Pos_Err = TRUE;
	diff_x(s->loc,t->loc, motion(s));
	scalar_q(.1,motion(s));
    }
    unless(ignored(t->color))
	sum_q(vgrad(t,s),grad);
    coeff[0] = fe->fi->area / 6.;
    if (dot(unit_normal(fe),s->loc) < dot(unit_normal(fe),t->loc))
	coeff[0] = -coeff[0]; /* want normal pointing away from t */
    scalar_x(coeff[0],unit_normal(fe), tmp);
    unless(ignored(t->color))
	sum_q(vgrad(t,s),tmp);
}

static void
calc_edge_deriv(fe,s)
facet_edge fe;
site s;
{

/* called once for every edge around s;
we should calculate areagrad for face above this edge (swing_out)
x = v1-v0 cross unit_n
agrad = xdot dff0 (v0-s) + (x dot dff1) (v1-s)
(if it is between different colors)
and then we should add term in to vol grad (1/6 * agrad * height)
for cell on each side of the face */
    struct point grad[2],x[1];
    struct point tmp[2];
    register corner v0,v1;
    clr k[2];

    fe = swing_out(fe,s);
    if (fe->fi->area == HUGE)
	return;	/* must be between ignored volumes */

    v0 = fe->vert[0]; v1 = fe->vert[1];
    k[0] = fe->cell[0]->color;
    k[1] = fe->cell[1]->color;

    cross_x(diff_x(v1->loc,v0->loc, tmp),unit_normal(fe),x);
    sum_x(
      scalar_x(dot(x,dffrntl(v0,s)),diff_x(v0->loc,s->loc,tmp),tmp),
      scalar_x(dot(x,dffrntl(v1,s)),diff_x(v1->loc,s->loc,tmp+1),tmp+1),grad);
    
    if (Wulff_Count)
    {
	cross_x(diff_x(v1->loc,v0->loc, tmp),wulff_vect(fe),x);
	sum_x(
	  scalar_x(dot(x,dffrntl(v0,s)),diff_x(v0->loc,s->loc,tmp),tmp),
	  scalar_x(dot(x,dffrntl(v1,s)),diff_x(v1->loc,s->loc,tmp+1),tmp+1),
	  grad+1);
    }
    
    if (k[0] != k[1])
    {
	unless (Flag_torus && k[0] == -k[1] && Torus_Same)
	{
	    sum_q(area_grad(s),grad);
	    if (Wulff_Count)
		sum_q(wulff_grad(s),grad+1);
	}
    }
    else
	if (Flag_torus && k[0]<0 && !Torus_Same)
	{
/* really should only do this if they are in different unit cells */
/* but for now we will just say that you can't have multi-cell colors */
/* in torus-different mode */
	    sum_q(area_grad(s),grad);
	    if (Wulff_Count)
		sum_q(wulff_grad(s),grad+1);
	}

    scalar_q(fe->fi->height/6.,grad);
    unless (ignored(k[0]))
	sum_q(vgrad(fe->cell[0],s),grad);
    unless (ignored(k[1]))
	sum_q(vgrad(fe->cell[1],s),grad);
}
