/* $Id$
 *
 * hcommand.c
 *
 * This module implements the hypercad command set
 *
 */

/**************************************************************************
 *     Copyright (C) 1990 by Mark B. Phillips and Robert R. Miner	  *
 * 									  *
 * Permission to use, copy, modify, and distribute this software, its	  *
 * documentation, and any images it generates for any purpose and without *
 * fee is hereby granted, provided that					  *
 * 									  *
 * (1) the above copyright notice appear in all copies and that both that *
 *     copyright notice and this permission notice appear in supporting	  *
 *     documentation, and that the names of Mark B.  Phillips, Robert R.  *
 *     Miner, or the University of Maryland not be used in advertising or *
 *     publicity pertaining to distribution of the software without	  *
 *     specific, written prior permission.				  *
 *									  *
 * (2) Explicit written credit be given to the authors Mark B.  Phillips  *
 *     and Robert R. Miner in any publication which uses part or all of	  *
 *     any image produced by this software.				  *
 *									  *
 * This software is provided "as is" without express or implied warranty. *
 **************************************************************************/

#include <stdio.h>		/* for NULL, fopen() */
#include "hcore.h"
#include "hglobals.h"

#define PERROR	output_int(-1)

static int ReturnXY();
static int ReturnObject();
gdb_Entity_type mask;

int Batching=0;
int BatchDeletion=0;

static FILE *psfile;		/* current PostScript file */
static int PSWrite();		/* PostScript file output function */
  

/*-----------------------------------------------------------------------
 * Function:	parse_command
 * Description:	parse a command string, breaking it up into the command
 *		part and its arguments
 * Args  IN:	cmd: the command string
 *      OUT:	*command: the command
 *		*argc: number of arguments given in cmd
 *		args: array of arguments
 * Returns:	(nothing)
 * Author:	mbp
 * Date:	Thu Apr 12 09:59:33 1990
 */
parse_command(cmd, command, argc, args)
     char *cmd;
     int *command, *argc;
     char **args;
{
  static char *seps = " \t";
  static char *delims = "\"";
  static char argtable[512];
  char *argptr;
  char *c;
  extern char *strtokq();

  /* Get the command */
  *command = UNKNOWN;
  if ((c=strtokq(cmd, seps, delims)) == NULL) return;
  *command = fsa_parse(c);

  /* Get the args */
  *argc = 0;
  argptr = argtable;
  while ((c=strtokq(NULL, seps, delims)) != NULL) {
    strcpy(argptr, c);
    *args = argptr;
    argptr += strlen(argptr) + 1;
    ++args;
    ++(*argc);
  }
}

/*-----------------------------------------------------------------------
 * Function:	execute_cmd
 * Description:	execute a command
 * Args  IN:	command: command to execute
 *		argc: number of arguments for command
 *		args: array of argument values (as strings)
 * Returns:	nothing
 * Author:	mbp
 * Date:	Thu Apr 12 10:00:13 1990
 */
execute_cmd(command, argc, args)
     int command, argc;
     char **args;
{
  switch (command) {
  case READ_FILE:
    if (argc != 1) PERROR;
    else output_int(read_file(args[0]));
    break;
  case UNKNOWN:
    PERROR;
    break;
  case SET_MODEL:
    if (argc != 1) PERROR;
    else if (strlen(args[0]) != 1) PERROR;
    else switch (args[0][0]) {
    case 'k':
    case 'K':
      SethModel(KLEIN);
      break;
    case 'p':
    case 'P':
      SethModel(POINCARE);
      break;
    case 'h':
    case 'H':
      SethModel(UHP);
      break;
    default:
      PERROR;
      break;
    }
    break;
  case SET_COORD_SYS:
    if (argc != 3) PERROR;
    else {
      double x,y,s;
      hWindow win;

      x = atof(args[0]);
      y = atof(args[1]);
      s = atof(args[2]);

      win.center[0] = x + s/2;
      win.center[1] = y + s/2;
      win.corner[0][0] = x;
      win.corner[0][1] = y;
      win.corner[1][0] = x+s;
      win.corner[1][1] = y;
      win.corner[2][0] = x+s;
      win.corner[2][1] = y+s;
      win.corner[3][0] = x;
      win.corner[3][1] = y+s;

      SethWindow(&win);

      output_int(0);
    }
    break;
  case DRAW_POINT:
    if (argc != 2) PERROR;
    else {
      kPoint p;

      p[0] = atof(args[0]);
      p[1] = atof(args[1]);
      output_string(AddEntity(POINT, (char*)p));
    }
    break;
  case DRAW_SEGMENT:
    if (argc != 4) PERROR;
    else {
      kSegment s;
      
      s.p[0][0] = atof(args[0]);
      s.p[0][1] = atof(args[1]);
      s.p[1][0] = atof(args[2]);
      s.p[1][1] = atof(args[3]);
      output_string(AddEntity(SEGMENT, (char*)&s));
    }
    break;
  case ERASE_OBJECT:
    if (argc != 1) PERROR;
    else output_int(DeleteEntity(args[0]));
    break;
  case CLEAR_ALL:
    if (argc != 0) PERROR;
    else output_int(DeleteAll());
    break;
  case BEGIN_BATCH:
    if (argc != 0) PERROR;
    else if (!Batching) {
      Batching = 1;
      BatchDeletion = 0;
      output_int(0);
    }
    else PERROR;
    break;
  case END_BATCH:
    if (argc != 0) PERROR;
    else if (Batching) {
      Batching = 0;
      if (BatchDeletion)
	DrawPicture();
      else
	UpdatePicture(LastDrawn);
      output_int(0);
    }
    else PERROR;
    break;
  case GET_POINT:
    if (argc != 1) PERROR;
    else {
      wMessage(args[0]);
      wGetXY(ReturnXY);
      hwait(1);
    }
    break;
  case GET_OBJECT:
    if (argc != 2) PERROR;
    else if (SetMask(args[1]) != 0) {
      wMessage(args[0]);
      wGetXY(ReturnObject);
      hwait(1);
    }
    else PERROR;
    break;
  case POSTSCRIPT:
    if (argc != 1) PERROR;
    else output_int( WritePostScriptFile(args[0]) );
    break;
  case QUIT_PROGRAM:
    if (argc != 0) PERROR;
    else {
      output_int(0);
      exit(0);
    }
  }
}

static int
  output_int(n)
     int n;
{
  printf("%1d\n", n);
  if (log_mode) {
    fprintf(logf,"hypercad out: %1d\n", n);
    fflush(logf);
  }
}

static int
  output_double(x)
double x;
{
  printf("%1f\n", x);
  if (log_mode) {
    fprintf(logf,"hypercad out: %f\n", x);
    fflush(logf);
  }
}

static int
  output_string(s)
char *s;
{
  printf("%s\n", s);
  if (log_mode) {
    fprintf(logf,"hypercad out: %s\n", s);
    fflush(logf);
  }
}

/*-----------------------------------------------------------------------
 * Function:	read_file
 * Description:	read input from a file
 * Args  IN:	fname: the name of the file to read
 * Returns:	success status
 * Author:	mbp
 * Date:	Thu Apr 12 10:00:34 1990
 * Notes:	Reads input from file until EOF, then returns.
 */
static int
  read_file(fname)
char *fname;
{
  FILE *fp;
  char inbuf[INBUF_SIZE];
  int command, argc;
  char *args[MAXARGS];

  if (strlen(fname)<=0) {
    if (log_mode) fprintf(logf,"read_file error: missing filename\n");
    return(-1);
  }

  if ((fp=fopen(fname,"r")) == NULL) {
    if (log_mode) fprintf(logf,"read_file error: can't open %s\n", fname);
    return(-1);
  }

  while (fgets(inbuf, INBUF_SIZE-1, fp) != NULL) {
  inbuf[strlen(inbuf)-1] = '\0';
    if (log_mode) {
      fprintf(logf, "hypercad in: %s\n", inbuf);
      fflush(logf);
    }
    parse_command(inbuf, &command, &argc, args);
    execute_cmd(command, argc, args);
  }
  fclose(fp);

  return(0);
 
}

static int
  ReturnXY(x,y,valid)
double x,y;
int valid;
{
  R2Point p, kp;

  p[0] = x;
  p[1] = y;

  if (model == KLEIN) {
    kp[0] = x;
    kp[1] = y;
  }
  else {
    p[0] = x;
    p[1] = y;
    switch (model) {
    case POINCARE:
      PoincareToKlein(kp, p);
      break;
    case UHP:
      UHPToKlein(kp, p);
      break;
    }
  }
  printf("{%f, %f, %d}\n", kp[0], kp[1], valid);
  hwait(0);
  wMessage(NULL);
  prompt();
}

static int
  SetMask(s)
char *s;
{
  mask = 0;
  if (strchr(s,'p') != NULL) 
    mask = mask | POINT;
  if (strchr(s,'s') != NULL) 
    mask = mask | SEGMENT;
  return(mask);
}

static int
  ReturnObject(x,y,valid)
double x,y;
int valid;
{
  gdb_Distance_spec spec;
  gdb_Entity entity;
  kPoint p,q;
  int error;

  p[0] = x;
  p[1] = y;
  switch(model){
  case KLEIN:
    q[0] = p[0];
    q[1] = p[1];
    break;
  case POINCARE:
    PoincareToKlein(q,p);
    break;
  case UHP:
    UHPToKlein(q,p);
    break;
  }
  spec.seek_data = (char *)q;
  spec.seek_type = POINT;
  spec.tolerance = vps/40;
  spec.sequence_spec = GDB_FIRST;
  entity = NULL;
  error = 1;

  do {
    gdb_retrieve_entity(mask, GDB_DISTANCE, &spec, &entity);
    if (QueryObject(entity)) {
      printf("%s\n",gdb_entity_name(entity));
      error = 0;
      entity = NULL;
    }
  } while (entity != NULL);
  if (error) printf("%d\n",-1);
  hwait(0);
  wMessage(NULL);
  prompt();
}

QueryObject(entity)
gdb_Entity entity;
{
  if (entity == NULL)
    return(0);
  else 
    return(1);
}

/*-----------------------------------------------------------------------
 * Function:	WritePostScriptFile
 * Description:	Write a postscript file of the current picture
 * Args  IN:	filename: name of file to write
 * Returns:     success
 * Author:	mbp
 * Date:	Sat May 12 11:45:18 1990
 */
static int
  WritePostScriptFile(filename)
char *filename;
{
  int success;

  psfile = fopen(filename, "w");
  if (psfile == NULL) return(-1);
  success = PSOutput(PSWrite);
  fclose(psfile);
  return(success);
}

/*-----------------------------------------------------------------------
 * Function:	PSWrite
 * Description:	Write a string to the currently open PostScript file
 * Args  IN:	s: string to write
 * Returns:	success status
 * Author:	mbp
 * Date:	Sat May 12 11:46:02 1990
 * Notes:	The currently open PostScript file is *psfile.
 */
static int
  PSWrite(s)
char *s;
{
  return(fputs(s, psfile));
}
