/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/******************************************************************
*
*  File:   verpopst.c
*
*  Purpose: In string model, pop vertices with 4 or more edges into
*           a configuration of vertices with only 3 edges each.
*           Works in 3-space.  No assumptions about edges bounding
*           facets or not, but does treat facets properly.
*/

#include "include.h"

/*********************************************************************
*
*  Function: verpop_str()
*
*  Purpose:  Assembles list of edges for each vertex, and calls
*            single-vertex routine for vertices with more than 3
*            edges.  List is first made as list of edge-vertex
*            pairs, then sorted on vertex.
*
*  Return:  Number of vertices popped.
*/
/* comparison routine for qsort */
static int vecomp(a,b)
struct veredge *a,*b;
{
  if ( a->v_id < b->v_id ) return -1;
  if ( a->v_id > b->v_id ) return 1;
  return 0;
}

int verpop_str()
{
  struct veredge *velist;
  edge_id e_id;
  int count,maxcount;
  int popcount = 0;
  int spot,dups;

  /* Allocate list, with room for each edge twice */
  maxcount = 2*web.skel[EDGE].count;
  velist = (struct veredge *)temp_calloc(sizeof(struct veredge),maxcount);  

  /* create unsorted list */
  count = 0;
  FOR_ALL_EDGES(e_id)
    {
      velist[count].e_id = e_id;
      velist[count].v_id = get_edge_tailv(e_id);
      count++;
      invert(e_id);             
      velist[count].e_id = e_id;
      velist[count].v_id = get_edge_tailv(e_id);
      count++;
      if ( count > maxcount )
        {
          error("Not enough structures allocated for vertex-edge list.\n",
            RECOVERABLE);
          return 0;
        }
    }

  /* sort by vertex order */
  qsort((char *)velist,count,sizeof(struct veredge),vecomp);

  /* go through list and pop appropriate vertices */
  for ( spot = 0 ; spot < count ; spot += dups )
    {
      /* find how many successive duplicates of current vertex */
      for ( dups = 1 ; spot+dups < count ; dups++ )
        if ( !equal_id(velist[spot+dups].v_id,velist[spot].v_id) ) break;

      /* if too many edges, pop it */
      if ( dups > 3 ) 
          popcount += poponest(velist+spot,dups);
    }
          
  temp_free((char *)velist);
  
  calc_content();
  calc_pressure();
  calc_energy();

  return popcount;
}

/*****************************************************************************
*
*  function: poponest()
*
*  purpose: pop vertex with more that 3 edges
*
*/

int poponest(first,edges)
struct veredge *first; /* place in list */
int edges;  /* number of edges at this vertex */
{
  int i,j;
  vertex_id v_id;       /* vertex being popped */
  vertex_id new_v;
  edge_id new_e;
  facetedge_id fe_id;
  REAL cosine,maxcos;
  REAL *x;
  int si=0,sj=0;
  struct side { edge_id e_id;   /* which edge */
                REAL vec[MAXCOORD];  /* side vector, from vertex */
                REAL norm;    /* length of side           */
              } *side;

  v_id = first->v_id;
  if ( valid_id ( get_vertex_fe(v_id) ) )
     return new_popverst(v_id,edges);
 
  side = (struct side *)temp_calloc(sizeof(struct side),edges);
  
  /* get edge vectors */
  for ( i = 0 ; i < edges ; i++ )
    { 
      side[i].e_id = first[i].e_id;
      get_edge_side(first[i].e_id,side[i].vec);
      side[i].norm = sqrt(dot(side[i].vec,side[i].vec,web.sdim));
    }

  while ( edges > 3 )
    {
      /* find closest pair of edges */
      maxcos = -2.0;
      for ( i = 0 ; i < edges ; i++ )
        {
          for ( j = i+1 ; j < edges ; j++ )
            {
              cosine = dot(side[i].vec,side[j].vec,web.sdim)/
                                             side[i].norm/side[j].norm;
              if ( cosine > maxcos )
                { 
                  /* toss in a little randomness if a choice of ways */
		  /* but big problem if edges not adjacent! */
              /*  if ( (maxcos > -0.5) && (rand() & 0x0100) ) continue; */
                  maxcos = cosine;
                  si = i; 
                  sj = j;
                }
             }
        }

      /* split off edges si and sj */
      new_v = new_vertex(get_coord(v_id));  /* new vertex in same spot */
      set_attr(v_id,get_vattr(v_id));
      if ( get_vattr(v_id) & BOUNDARY )
        {
          REAL *p,*pp;
          set_boundary(new_v,get_boundary(v_id));
          p = get_param(v_id);
          pp = get_param(new_v);
          for ( i = 0 ; i < MAXPARAM ; i++ )
            pp[i] = p[i];
        }
    
      new_e = new_edge(v_id,new_v);
      set_edge_color(new_e,get_edge_color(first[0].e_id));
      set_edge_tailv(side[si].e_id,new_v);
      set_edge_tailv(side[sj].e_id,new_v);
      if ( get_vattr(v_id) & BOUNDARY )
          set_edge_boundary(new_e,get_edge_boundary(v_id));
      
      /* take care of facet incidences, assuming plane configuration */
      do
        {
          generate_edge_fe_init();
          while ( generate_edge_fe(side[si].e_id,&fe_id) )
            {
              edge_id next_e;
              facetedge_id next_fe,new_fe,other_fe;
              facet_id f_id;
              
              /* see if this facet got split */
              next_fe = get_prev_edge(fe_id);
              next_e = get_fe_edge(next_fe);
              if ( equal_id(inverse_id(side[sj].e_id),next_e) ) continue;
        
              /* install new facetedge */
              f_id = get_fe_facet(fe_id);
              new_fe = new_facetedge(f_id,new_e);
	      set_vertex_fe(v_id,new_fe);  /* vertex at tail of its fe */
	      set_vertex_fe(new_v,inverse_id(new_fe));
              set_prev_edge(fe_id,new_fe);
              set_next_edge(next_fe,new_fe);
              set_prev_edge(new_fe,next_fe);
              set_next_edge(new_fe,fe_id);
              other_fe = get_edge_fe(new_e);
              if ( valid_id(other_fe) )
                {
                  set_next_facet(new_fe,other_fe);
                  set_prev_facet(new_fe,other_fe);
                  set_prev_facet(other_fe,new_fe);
                  set_next_facet(other_fe,new_fe);
                }
              else
                {
                  set_edge_fe(new_e,new_fe);
                  set_next_facet(new_fe,new_fe);
                  set_prev_facet(new_fe,new_fe);
                }
             }
          /* swap roles of sj and si */       
          i = sj; sj = si; si = i;
        }
      while ( sj < si ); /* exit when swapped back */

      /* adjust edge list for another round, if necessary */
      /* direction of new edge is average of old two */
      /* Also move new vertex a little ways away to avoid zero edge */
      edges --;
      x = get_coord(new_v);
      for ( i = 0 ; i < web.sdim ; i++ )
        {
          side[si].vec[i] = side[si].vec[i]/side[si].norm 
                           + side[sj].vec[i]/side[si].norm;
          x[i] += 0.000001 * side[si].vec[i];
        }
      side[si].norm = sqrt(dot(side[si].vec,side[si].vec,web.sdim)); 
      side[si].e_id = new_e;
      if ( sj < edges ) /* trim top of list */
        side[sj] = side[edges];

      if ( phase_flag && (web.dimension == STRING) )
          { facetedge_id fe = get_edge_fe(new_e);
            int i = get_tag(get_fe_facet(fe));	
            int j = get_tag(get_fe_facet(get_next_facet(fe)));	
	    set_edge_density(new_e,phase_data[i][j]);
          }
      else	
        set_edge_density(new_e,(get_edge_density(side[si].e_id)
	  + get_edge_density(side[sj].e_id))/2);
    }
  temp_free((char *)side);
  return 1;
}


/*****************************************************************************
*
*  function: new_popverst()
*
*  purpose: pop vertex with more that 3 edges with variable density
*
*/

int new_popverst(v_id,edges)
vertex_id v_id; /* vertex being popped */
int edges;  /* number of edges at this vertex */
{
  int i,j,m;
  int besti;
  vertex_id new_v;
  edge_id new_e;
  facetedge_id fe_id,first_fe;
  REAL cosine,maxcos;
  REAL *x;
  double shortlen;   /* length of edge to insert */
  facet_id f1,f2;
  double ff,f[MAXCOORD],netf[MAXCOORD],ffmax;
  double new_density;
  struct side { edge_id e_id;   /* which edge */
		facetedge_id fe;  /* canonical fe */
                REAL vec[MAXCOORD];  /* side vector, from vertex */
                REAL norm;    /* length of side           */
		double density;
              } *side;

  side = (struct side *)temp_calloc(edges+1,sizeof(struct side));
  
  /* get edge vectors */
  fe_id = first_fe = get_vertex_fe(v_id);
  for ( i = 0 ; i < edges ; i++  )
    { 
      if  ( !valid_id(fe_id) )
	error("Invalid facetedge at vertex.\n",RECOVERABLE);
      side[i].fe = fe_id;
      side[i].e_id = get_fe_edge(fe_id);
      get_edge_side(side[i].e_id,side[i].vec);
      side[i].norm = sqrt(dot(side[i].vec,side[i].vec,web.sdim));
      side[i].density = get_edge_density(side[i].e_id);
      fe_id = get_prev_edge(fe_id);
      fe_id = get_prev_facet(fe_id);
      fe_id = inverse_id(fe_id);
      if ( equal_id(fe_id,first_fe) && ( i < edges-1 ) )
	error("Not expected number of edges around vertex\n",RECOVERABLE);
    }
  side[edges] = side[0];  /* easy wraparound */

      /* find two edges that pull apart the most */
      ffmax = 0.01; /* margin to prevent looping with autopop */
      besti = -1;
      for ( m = 0 ; m < web.sdim ; m++ ) netf[m] = 0.0;
      for ( i = 0 ; i < edges ; i++ )
        for ( m = 0 ; m < web.sdim ; m++ )
          netf[m] += side[i].density*side[i].vec[m]/side[i].norm ;
      for ( i = 0 ; i < edges ; i++ )
	  { 
	    for ( m = 0 ; m < web.sdim ; m++ )
	       f[m] = 2*(side[i].density*side[i].vec[m]/side[i].norm 
		    + side[i+1].density*side[i+1].vec[m]/side[i+1].norm)
		     - netf[m];
            ff = sqrt(dot(f,f,web.sdim));
            if ( phase_flag )
	     {
               f1 = get_fe_facet(get_prev_facet(side[i].fe));
	       f2 = get_fe_facet(side[i+1].fe);
	       new_density = phase_data[get_tag(f1)][get_tag(f2)];
             }
	    else new_density = 1;
	    if ( ff - 2*new_density > ffmax )
	      { ffmax = ff - 2*new_density ; besti = i;  
	      }
          } 
      if ( besti == -1 ) 
        { temp_free((char *)side);
          return 0;
        }

      /* split off edges besti and besti+1 */
      new_v = new_vertex(get_coord(v_id));  /* new vertex in same spot */
      set_attr(v_id,get_vattr(v_id));
      if ( get_vattr(v_id) & BOUNDARY )
        {
          REAL *p,*pp;
          set_boundary(new_v,get_boundary(v_id));
          p = get_param(v_id);
          pp = get_param(new_v);
          for ( i = 0 ; i < MAXPARAM ; i++ )
            pp[i] = p[i];
        }
    
      new_e = new_edge(v_id,new_v);
      set_edge_color(new_e,get_edge_color(side[0].e_id));
      set_edge_tailv(side[besti].e_id,new_v);
      set_edge_tailv(side[besti+1].e_id,new_v);
      if ( get_vattr(v_id) & BOUNDARY )
          set_edge_boundary(new_e,get_edge_boundary(v_id));
      
      /* take care of facet incidences, assuming plane configuration */
            {
              facetedge_id new_fe1,new_fe2,other_fe;
              
            f1 = get_fe_facet(get_prev_facet(side[besti].fe));
	    f2 = get_fe_facet(side[besti+1].fe);
            if ( phase_flag )
	       new_density = phase_data[get_tag(f1)][get_tag(f2)];
            else new_density = 1.0;
	    set_edge_density(new_e,new_density);

              /* install new facetedge */
	      fe_id = get_prev_facet(side[besti].fe);
	      fe_id = inverse_id(fe_id);
	      other_fe = get_next_edge(fe_id);
              new_fe1 = new_facetedge(f1,new_e);
	      set_edge_fe(new_e,new_fe1);
	      set_vertex_fe(v_id,new_fe1);  /* vertex at tail of its fe */
	      set_vertex_fe(new_v,inverse_id(new_fe1));
              set_prev_edge(other_fe,inverse_id(new_fe1));
              set_next_edge(fe_id,inverse_id(new_fe1));
              set_prev_edge(new_fe1,inverse_id(other_fe));
              set_next_edge(new_fe1,inverse_id(fe_id));

	      fe_id = side[besti+1].fe;
	      other_fe = get_prev_edge(fe_id);
              new_fe2 = new_facetedge(f2,new_e);
              set_next_edge(other_fe,new_fe2);
              set_prev_edge(fe_id,new_fe2);
              set_next_edge(new_fe2,fe_id);
              set_prev_edge(new_fe2,other_fe);

	      set_prev_facet(new_fe1,new_fe2);
	      set_next_facet(new_fe1,new_fe2);
	      set_prev_facet(new_fe2,new_fe1);
	      set_next_facet(new_fe2,new_fe1);

            }

      /* move new vertex a little ways away to avoid zero edge */
      x = get_coord(new_v);
      shortlen = 0.05 * (side[besti].norm<side[besti+1].norm?
			 side[besti].norm:side[besti+1].norm);
      for ( i = 0 ; i < web.sdim ; i++ )
        {
          x[i] += shortlen 
 	      * (2*(side[besti].density*side[besti].vec[i]/side[besti].norm 
	     +side[besti+1].density*side[besti+1].vec[i]/side[besti+1].norm)
	      - netf[i]);
        }

  temp_free((char *)side);
  if ( edges > 4 ) return 1 + new_popverst(v_id,edges-1); /* recurse if need */
  return 1;
}

/********************************************************************
*
*  function: autopop_detect()
*
*  purpose: Find which edges will shrink to zero length or  beyond.
*           (half length when runge-kutta in effect)
*           Makes a list, but does not change anything.  Do
*           motion, then call autopop_pop() to eliminate edges.
*           Also detects for autochop.
*
*/

edge_id *autopop_list;  /* list of edges found */
edge_id *autochop_list;  /* list of edges found */

void autopop_detect(scale)
REAL scale;  /* scale factor for motion */
{
  edge_id e_id;
  double minlength = sqrt(2*scale); /* stability critical length */

  if ( autopop_flag )
  {
   autopop_list = (edge_id *)temp_calloc(web.skel[EDGE].count,sizeof(edge_id));
   autopop_count = 0;
  }
  if ( autochop_flag )
  {
   autochop_list = (edge_id *)temp_calloc(web.skel[EDGE].count,sizeof(edge_id));
   autochop_count = 0;
  }
  FOR_ALL_EDGES(e_id)
   {
      vertex_id headv = get_edge_headv(e_id);
      REAL *headf = get_force(headv);
      REAL heada = get_vertex_star(headv)/star_fraction;
      vertex_id tailv = get_edge_tailv(e_id);
      REAL *tailf = get_force(tailv);
      REAL taila = get_vertex_star(tailv)/star_fraction;
      REAL side[MAXCOORD];
      REAL length,dx;

      if ( get_eattr(e_id) & FIXED ) continue;
      get_edge_side(e_id,side);
      length = sqrt(dot(side,side,web.sdim));
      if ( !web.area_norm_flag ) { heada = taila = 1.0; }
      if ( length > 0.0 )
        dx = scale*(dot(headf,side,web.sdim)
  	          - dot(tailf,side,web.sdim))/length;
      else dx = 0.0;
      if ( autopop_flag && (dx < 0.0) )
       { if ( length + dx <= (runge_kutta_flag ? 0.52*length : 0.0) )
	  { /* add to list */
	    autopop_list[autopop_count++] = e_id;
          }
         else if ( length < minlength )
	    autopop_list[autopop_count++] = e_id;
       }
      if ( autochop_flag )
       if ( length + dx > autochop_size)
	{ /* add to list */
	  autochop_list[autochop_count++] = e_id;
        }
   }
}

/***********************************************************************
*
*  function: autopop_pop()
*
*  purpose: After motion, eliminate edges found by autopop_detect()
*           and pop any resulting bad vertices.
*
*/

void autopop_pop()
{
  int k;
  int popped = 0;

  if ( autopop_list == NULL ) return;

  for ( k = 0 ; k < autopop_count ; k++ )
   { /*  printf("popping edge %d\n",ordinal(autopop_list[k])+1);  */
    if ((get_eattr(autopop_list[k])&ALLOCATED)&&eliminate_edge(autopop_list[k]))
      free_element(autopop_list[k]);
  }
  if ( autopop_count )
     popped = verpop_str();

  if ( (autopop_count > 0) || (popped > 0) )
    { sprintf(msg,"Autopopped %d edges, %d vertices.\n",autopop_count,popped);
      outstring(msg);
    }

  temp_free((char *)autopop_list);
  autopop_list = NULL;
  autopop_count = 0;
  free_discards(); /* prevents auto pileup */

}


/***********************************************************************
*
*  function: autochop_chop()
*
*  purpose: After motion, divides edges found by autochop_detect()
*
*/

void autochop_chop()
{
  int k;

  if ( autochop_list == NULL ) return;

  for ( k = 0 ; k < autochop_count ; k++ )
   { /* printf("chopping edge %d\n",ordinal(autochop_list[k])+1); */
     edge_divide(autochop_list[k]);
   }

  if ( autochop_count > 0 )
    { sprintf(msg,"Autochopped %d edges.\n",autochop_count);
      outstring(msg);
    }

  temp_free((char *)autochop_list);
  autochop_list = NULL;
  autochop_count = 0;

}

