/* lib.d file pqueue.c */
/*
This file contains the definition of all functions 
in the |PQUEUE| function table  (together with functions which
they depend upon) which are called by the |list| manipulation functions of
`list.c' whenever the |list| being manipulated is a priority queue.
This file needs to be compiled together with `list.c' whenever functions
from `list.c' 
are being used to manipulate a priority queue.
A priority queue consists of a |heap| (essentially an array, though in fact 
it is set up as a pointer to allow it to have variable length) of datapointers 
which give the addresses of the items in the queue, together with a pair of 
integers . The position in the array of a datapointer  (its |heap index|) 
determines the position of the associated item in the queue; the datapointer
in position 0 is always null, so no item is stored in that position. The two 
integers specify the |heapsize| (number of items in the queue) and the actual 
size (|maxheapsize| +1, since nothing is stored in position 0) of the array 
(some of whose entries are trivial if there are not enough items in the queue 
to fill it). The heap is viewed as a tree, in which the j-th entry has as
its children the 2j-th and 2j+1-st entries (where these exist). The 1st entry is
the root. The heap condition is satisfied at node j if according to the
signature function the j-th entry in the array precedes both of its children in
the ordering.
The functions  in this file which are list function table functions must take
data pointer arguments (since the structure |llist| is unknown to the |list|
manipulation functions), although they will only ever be used to manipulate
priority queues.
The items in any one list (either a linked list or a priority queue), 
necessarily all of the same type, are manipulated 
using standard functions which operate on data structures of that particular 
type and appear in a function table associated with that particular datatype. 
These functions may be called by the |pqueue| functions defined in
this file, but their definitions and descriptions appear in the files
associated with that particular datastructure. Of course the relevant .c
file needs to be compiled together with `pqueue.c' for any data structure
being stored in a priority queue. 

Each data structure function table is an array containing items labelled from 
0 to 7, and called in this file according to their position in that array.
  Since the function table is actually an array in  which all elements must
necessarily have the same type, we declare it as an array of type |PTFNRDP|,
(pointer to function returning datapointer).
This means that all function table functions are declared to return datapointers
and for use may need to be cast to another type.
Since the `llist' routines don't know what kind of structure is being stored
in the list, the data must be accessed via datapointers; thus the arguments
of these function table functions are always datapointers.
The data structure function table always contains the following items.

A signature function. This returns +1 if the first argument is ``earlier''
in the desired ordering of this kind of structure and -1 if the first 
argument is ``later''. (so +1 if the two arguments appear
in the order coherent withn the desired total ordering, -1 if the opposite 
is true). 
It returns zero if the two arguments are ``equal''. this
does not necessarily mean the two arguments are actually equal. For
example, we might want to regard the arguments as equal if they are
equal in several important respects.
A copy function returning void.
A create function, which gets space for exactly one object. It
returns a pointer to type {\bf XX}.
A kill function, which takes back the space obtained.
A print function.
A debugging print function, which prints out more detail of the structure.
A function that returns the heap index for an item.
A function to set the heap index for an item.
*/
#include <stdio.h>
#include "defs.h"
#include "list.h"
#define SIGNATURE 0
#define COPY 1
#define CREATE 2
#define KILL 3
#define PRINT 4
#define DPRINT 5
#define GETHEAPINDEX 6
#define SETHEAPINDEX 7
static boolean pqueue_insert PARMS((dp PQp, dp in));
static boolean pqueue_delete PARMS((dp PQp,dp in));
static boolean pqueue_get_first PARMS((dp PQp, dp out));
static boolean pqueue_delget_first PARMS((dp PQp, dp out));
static boolean pqueue_empty PARMS((dp PQp));
static void pqueue_init PARMS((pqueue*, elt_fntab *fntab ));
static  pqueue * pqueue_create PARMS((elt_fntab *));
static void pqueue_clear PARMS((pqueue * pqp));
static void pqueue_kill PARMS((dp PQp));
static void pqueue_reset PARMS((dp PQp));
static void pqueue_cpy PARMS((dp oldPQp, dp newPQp));
static void pqueue_print PARMS((FILE*,dp PQp));
static void pqueue_traverser_init PARMS((pqueue_traverser *pqtp,
pqueue * pqp));
static pqueue_traverser * pqueue_traverser_create PARMS((dp PQp));
#define pqueue_traverser_clear(pqtp)  (pqtp) = (pqtp)
static void pqueue_traverser_kill PARMS((dp PQtp));
static boolean pqueue_next PARMS((dp PQtp,dp out)); 

static void downheap PARMS ((int, pqueue *)); 
static boolean heap_condition_OK PARMS((pqueue* pqp));
static void upheap PARMS((int , pqueue *)); 
static void pqueue_double PARMS((pqueue * pqp));
static void pqueue_append PARMS((pqueue * pqp, dp in));


/* Initialize a priority queue, giving the address of the appropriate
function table. Initially |maxheapsize| is set at 2, but the |heap| is
empty.
*/
static void 
pqueue_init(pqp,fntab) 
	pqueue *pqp; 
	elt_fntab *fntab; 
{ 
	assert(pqp); 
	pqp->maxheapsize = 2; 
	pqp->heapsize = 0;
	pqp->fntab = fntab; 
	pqp->heap = vzalloc2(dp,(pqp->maxheapsize)+1); 
} 

static pqueue * 
pqueue_create(fntab) 
	elt_fntab *fntab; 
{ 
	pqueue * pqp; 
	pqp = vzalloc1(pqueue); 
	pqueue_init(pqp,fntab); 
	return(pqp);  
} 

/*The clear function clears a priority queue. Afterwards both
|maxheapsize| and |heapsize| are set to zero and the |heap| itself is a
null-pointer.
*/
static void 
pqueue_clear(pqp) 
	pqueue * pqp; 
{ 
	auto void (*killfnp) PARMS ((dp)); 
	assert(pqp);  
	killfnp = pqp->fntab->kill;  
	while (pqp->heapsize){ 
		(*killfnp)((pqp->heap)[pqp->heapsize]);  
		pqp->heap[(pqp->heapsize)--]=0;
	}
	Free_dp((dp)(pqp->heap)); pqp->heap = 0; 
	pqp->maxheapsize = 0; 
} 
	
static void 
pqueue_kill(PQp) 
	dp PQp; 
{ 
	pqueue_clear((pqueue *)PQp); 
	Free_dp(PQp); PQp = 0; 
} 

/* Reset a priority queue, i.e. clear and re-initialise it

*/
static void 
pqueue_reset(PQp)
	dp PQp; 
{ 
	pqueue *pqp; 
	pqp = (pqueue *)PQp; 
	pqueue_clear(pqp); 
	pqueue_init(pqp,pqp->fntab); 
} 
	
	
/* The boolean valued function |pqueue_empty()| returns |TRUE| if the priority
queue is empty, |FALSE| otherwise, without changing the priority queue in any
way.
*/
static boolean 
pqueue_empty(PQp)
	dp PQp; 
{ 
	pqueue *PQpp = (pqueue *)PQp;
	assert(PQpp); 
	if (PQpp->heapsize == 0) 
		return TRUE; 
	else 
		return FALSE; 
} 


/*The function |downheap(k,pqp)| fixes any heap condition violation 
between node k and one of its children
in the heap pointed at by |pqp|.  It moves down the heap, starting at position 
|k|, exchanging the item at position |j| with that one of its children which 
comes earlier in the ordering if currently |j| and that child appear in 
the heap in the wrong order. As items are exchanged the heap indices
associated with them need to be revised.
*/
static void 
downheap(k,pqp) 
	int k; 
	pqueue *pqp; 
{ 
	int j,N; 
	dp *h;
	dp item; 
	auto int (*sgnfnp) PARMS ((dp,dp)); 
	auto void (*setindexfnp) PARMS ((dp,int)); 

	assert(pqp); 
	assert(1<=k&&k<=(pqp->heapsize)); 
	sgnfnp = pqp->fntab->signature; 
	setindexfnp = pqp->fntab->setheapindex; 
	h = pqp->heap;
	N = pqp->heapsize; 
	
	item = h[k]; 
	while (k<=(pqp->heapsize)/2) /* we are not at a leaf of the tree */ { 
		j = 2*k; 
		if (j<N && (*sgnfnp)(h[j+1],h[j])==1)
			/*if |h[j+1]| is nearer to the
					head of the queue than |h[j]| */
			j = j+1; 
/* now j is the position of the earlier ordered of k's children */
		if ((*sgnfnp)(item,h[j]) == -1) {
			/*|h[j]| needs to be moved up the heap*/
			(*setindexfnp)(h[j],k); 
			h[k] = h[j]; 
			k = j; 
		} 
		else break; 
	} 
	(*setindexfnp)(item,k); 
	h[k] = item; 
} 



static boolean heap_condition_OK(pqp)
	pqueue * pqp;
{
	boolean ans = TRUE;
	int i;
	auto int (*sgnfnp) PARMS ((dp,dp));

	sgnfnp = pqp->fntab->signature; 
	for (i=2;i<=pqp->heapsize;++i) 
		if ((*sgnfnp)((pqp->heap)[i/2],(pqp->heap)[i]) < 0)
			ans = FALSE;
	return ans;
}
	


/*
The function |upheap(k,pqp)| fixes any heap condition violation between node 
k and its parent
in the heap pointed at by |pqp|.  It moves up the heap, starting at position 
|k|, exchanging the item at position |j| with its parent if currently they 
appear in the heap in the wrong order.  As items are exchanged the heap indices
associated with them need to be revised.
*/
static void 
upheap(k,pqp) 
	int k; 
	pqueue *pqp; 
{ 
	int N,j; 
	dp *h; 
	dp item; 
	auto int (*sgnfnp) PARMS ((dp,dp)); 
	auto void (*setindexfnp) PARMS ((dp,int)); 

	assert(pqp); 
	N = pqp->heapsize; 
	assert(1<=k&&k<=N); 
	sgnfnp = pqp->fntab->signature; 
	setindexfnp = pqp->fntab->setheapindex; 
	h = pqp->heap; 
	item = h[k]; 
	while (1<k) /* we are not at the root of the tree */ { 
		j=k/2; /* |j| is the parent of |k| in the tree */ 
		if ((*sgnfnp)(h[j],item) == -1) {
	/* |h[j]| needs to be moved down */
			(*setindexfnp)(h[j],k);
			h[k] = h[j]; 
			k = j; 
		} 
		else break; 
	} 
	(*setindexfnp)(item,k); 
	h[k] = item; 
} 


/* |pqueue_insert()| inserts an item into a priority queue in such a way as to 
maintain the heap condition if it is currently satisfied.
  If the |heap| is already full, more space needs to be allocated. Then the
item is added onto the end of the |heap|, and |upheap()| is called to
rectify any heap violation introduced at the end by the insertion.
The function is boolean valued but only ever returns the value |TRUE|.
*/
static boolean
pqueue_insert(PQp,in) 
	dp PQp, in; 
{ 
	pqueue * pqp; 
	DP2I getindexfnp; 

	pqp = (pqueue *)PQp; 
	/*assert(heap_condition_OK(pqp));*/
	getindexfnp = (pqp->fntab->getheapindex); 
	if ((*getindexfnp)(in)!=INVALID_HEAP_INDEX) { 
		DP2V printfnp; 
		printfnp = pqp->fntab->print;
		(*printfnp)(in); 
		fprintf(stderr,"\nbad index is %d\n",(*getindexfnp)(in)); 
	}
	assert((*getindexfnp)(in)==INVALID_HEAP_INDEX); 

	if (pqp->maxheapsize == pqp->heapsize) 
		pqueue_double(pqp);
	pqueue_append(pqp,in);
	upheap(pqp->heapsize,pqp); 
	/*assert(heap_condition_OK(pqp));*/
	return TRUE;
} 
	
		
static void
pqueue_double(pqp)
	pqueue * pqp;
{ 
	dp * temp; 
	int i; 

	pqp->maxheapsize *= 2; 
	temp = vzalloc2(dp,(pqp->maxheapsize)+1); 
	for (i=1;i<=(pqp->heapsize);++i) 
		temp[i] = (pqp->heap)[i]; 
	Free_dp((dp)(pqp->heap)); 
	pqp->heap = temp; 
} 


static void
pqueue_append(pqp,in)
	pqueue *pqp;
	dp in;
{ 
	auto data_ptr (*createfnp)(); 
	DPI2V setindexfnp; 
	DPDP2V copyfnp; 

	createfnp = pqp->fntab->create; 
	setindexfnp = pqp->fntab->setheapindex; 
	copyfnp = pqp->fntab->copy; 

	pqp->heapsize++; 
	assert((pqp->heapsize)<=(pqp->maxheapsize)); 
	(pqp->heap)[pqp->heapsize] = (*createfnp)(); 
	(*copyfnp)(in,(pqp->heap)[pqp->heapsize]); 
	(*setindexfnp)(pqp->heap[pqp->heapsize],pqp->heapsize); 
} 


/*Delete a specified item from the priority queue. The function is boolean
valued to match |list_delete|, but only ever returns the value |TRUE|. It is
not possible to apply this function if the specified function is not
actually in the priority queue.
First the item to be deleted is replaced in the |heap| by the item at the
end of the heap. Then the item at the end is deleted, |heapsize| decremented
and the space at the end of the heap recovered. Finally |downheap()| and
|upheap()| are 
called to rectify any violation of the heap condition at the position which
originally held the deleted item.
*/

static boolean 
pqueue_delete(PQp,in)
	dp PQp, in; 
{ 
	pqueue * pqp; 
	DP2I getindexfnp; 
	DPI2V setindexfnp; 
	DP2V killfnp; 
	int k; 

	pqp = (pqueue *)PQp; 
	/*assert(heap_condition_OK(pqp));*/
	getindexfnp = pqp->fntab->getheapindex; 
	setindexfnp = pqp->fntab->setheapindex; 

	k = (*getindexfnp)(in); 
	assert(k!=INVALID_HEAP_INDEX); 
	(*setindexfnp)(in,INVALID_HEAP_INDEX); 

	killfnp = pqp->fntab->kill; 
	(*killfnp)(pqp->heap[k]); 
	pqp->heap[k]=0;
	(pqp->heapsize)--; 
	if (k<=(pqp->heapsize)) { 
		(pqp->heap)[k] = (pqp->heap)[(pqp->heapsize)+1]; 
		(*setindexfnp)((pqp->heap)[k],k); 
		downheap(k,pqp); 
		upheap(k,pqp); 
/* only one of |upheap| and |downheap| will do anything, depending on
	whether the heap condition violation is above or below k */
	} 
	/*assert(heap_condition_OK(pqp));*/
	return TRUE; 
} 
	
/* Get a copy of the item at the head of the priority queue, and put it into 
|*out|. The priority queue does not change. The procedure returns |FALSE| if 
the priority queue is empty, and |TRUE| otherwise. The data is copied
using the copy function from the relevant function table.
*/
static boolean 
pqueue_get_first(PQp,out) 
	dp PQp, out; 
{ 
	pqueue * pqp; 
	assert(PQp); 
	pqp = (pqueue *)PQp; 
	if ((pqp->heapsize) != 0) { 
		DPDP2V copyfnp; 
		copyfnp = pqp->fntab->copy; 
		(*copyfnp)(pqp->heap[1],out); 
		return TRUE; 
	} 
	else return FALSE; 
} 

/* Get a copy of the item at the head of the priority queue, and put it into 
|*out|.  Remove this item from the queue.
Returns |TRUE| or |FALSE|, in the same way as |pqueue_get_first()|.
*/
static boolean 
pqueue_delget_first(PQp,out) 
	dp PQp, out; 
{ 
	pqueue * pqp; 
	assert(PQp); 
	pqp = (pqueue *)PQp; 
	/*assert(heap_condition_OK(pqp));*/

	if (pqp->heapsize) { 
		DPDP2V copyfnp; 
		DP2V killfnp; 
		DPI2V setindexfnp; 

		setindexfnp = pqp->fntab->setheapindex; 
		copyfnp = pqp->fntab->copy; 
		(*copyfnp)(pqp->heap[1],out); 
		(*setindexfnp)(out,INVALID_HEAP_INDEX); 
		killfnp = pqp->fntab->kill; 
		(*killfnp)(pqp->heap[1]); 
		pqp->heap[1]=0;
		(pqp->heapsize)--; 
		if (pqp->heapsize) { 
			pqp->heap[1] = pqp->heap[(pqp->heapsize)+1]; 
			(*setindexfnp)(pqp->heap[1],1); 
			downheap(1,pqp); 
		} 
		/*assert(heap_condition_OK(pqp));*/
		return TRUE; 
	}
	else 
		return FALSE; 
} 

/* Copy a priority queue. The queue |*oldPQp| is untouched and 
|*newPQp| is a copy of
|*oldPQp|. The original contents of |*newPQp| are destroyed.
The two queues	are assumed to be of the same type, i.e. they contain the
same kind of data and use the same function table.
*/
static void 
pqueue_cpy(oldPQp, newPQp) 
	dp oldPQp, newPQp; 
{ 
	pqueue * oldpqp, *newpqp; 
	V2DP createfnp; 
	DPDP2V copyfnp; 
	int i; 

	oldpqp  = (pqueue *)oldPQp; 
	newpqp = (pqueue *)newPQp; 
	pqueue_clear(newpqp); 
	newpqp->heapsize = oldpqp->heapsize; 
	newpqp->maxheapsize = oldpqp->maxheapsize; 
	newpqp->heap = vzalloc2(dp,(newpqp->maxheapsize)+1); 

	createfnp = newpqp->fntab->create; 
	copyfnp = newpqp->fntab->copy; 

	for (i=1;i<=oldpqp->heapsize;++i) { 
		dp item; 
		DPI2V setindexfnp; 
		item = (oldpqp->heap)[i]; 

		setindexfnp = newpqp->fntab->setheapindex; 
		(newpqp->heap)[i] = (*createfnp)(); 
		(*copyfnp)(item,(newpqp->heap)[i]); 
		(*setindexfnp)((newpqp->heap)[i],i); 
	} 
} 

static void 
pqueue_print(wfile,PQp) 
	FILE * wfile;
	dp PQp; 
{ 
	pqueue * pqp; 
	DP2V printfnp; 
	int i; 

	pqp = (pqueue *)PQp; 
	assert(pqp); 
	printfnp = pqp->fntab->print; 
	if (pqp->heapsize) { 
		for (i=1;i<=pqp->heapsize;++i) { 
			fprintf(wfile,"\t%d\t",i);
			if (pqp->heap[i]) { 
				(*printfnp)(pqp->heap[i]); 
				fprintf(wfile,"\n"); 
			} 
			else fprintf(wfile,"NULL data"); 
		} 
	} 
	fprintf(wfile,"\n"); 
} 


static void 
pqueue_traverser_init(pqtp,pqp) 
	pqueue_traverser * pqtp; 
	pqueue * pqp; 
{ 
	pqtp->queuep = (dp)pqp; 
	pqtp->place = 0; 
} 
	

static pqueue_traverser * 
pqueue_traverser_create(PQp) 
	dp  PQp; 
{ 
	pqueue* pqp; 
	pqueue_traverser* pqtp; 

	pqp = (pqueue *)PQp; 
	pqtp = vzalloc1(pqueue_traverser); 
	pqueue_traverser_init(pqtp,pqp); 
	return(pqtp); 
} 


static void pqueue_traverser_kill(PQtp) 
	dp PQtp; 
{ 
	pqueue_traverser * pqtp; 
	pqtp = (pqueue_traverser *)PQtp; 
	pqueue_traverser_clear(pqtp); 
	Free_dp(PQtp); 
} 

/* |pqueue_next()| is used when traversing a priority queue . It
returns |TRUE| if there is a next element on the queue
and |FALSE| otherwise. If there is a next item, the |pqueue_traverser| is
advanced one item and the next item is put into the
address given by the datapointer |out| (using the copy function in the
relevant function table).  The copy function clears |*out| 
and assigns the more space to it as necessary, so there is no need to
reset it if it has been used before. |*out| should however be
initialized. 

The procedure does not change the queue being traversed.
*/
static boolean 
pqueue_next(PQtp,out) 
        dp  PQtp, out; 
{ 
	pqueue_traverser * pqtp; 
	pqueue pq; 
	pqtp = (pqueue_traverser *)PQtp; 
	pq = * (pqueue *)(pqtp->queuep); 

	if (pqtp->place == pq.heapsize) 
                return FALSE; 
        (pqtp->place)++; /*otherwise move the traverser to point
to the next place */ 
        (pq.fntab->copy)(pq.heap[pqtp->place],out); 
/* copy the data in the next link into the address given by |out| */
        return TRUE; 
} 




list_fntab
PQUEUE_fntab =  
{  
	pqueue_insert, 
	0,  
	pqueue_delete, 
	pqueue_get_first, 
	pqueue_delget_first, 
	pqueue_empty, 
	pqueue_cpy, 
	(E2DP)pqueue_create, 
	pqueue_kill, 
	pqueue_reset, 
	pqueue_print,  
	(DP2DP)pqueue_traverser_create, 
	pqueue_traverser_kill, 
	pqueue_next, 
	0
}; 

