/* lib.d file llist.c */
/*
This file contains the definition of all functions 
in the |ORDERED|, |FIFO| and |LIFO| function tables  (together with functions 
which they depend upon) which are called by the |list| manipulation functions of
`list.c' whenever the |list| being manipulated is of `ordered', `fifo' or
`lifo' type. This file needs to be compiled together with `list.c' whenever 
functions from `list.c' 
are being used to manipulate a list of `ordered', `fifo' or `lifo' type.
\par\noindent
  `ordered', `fifo' and `lifo' lists all have the structure of `linked lists'.
  We specify a linked list  by describing the
first link, the address of the last link, and the address of the table of 
functions that manipulate the list . 
Each link |l| consists of a data-pointer |l.dp| 
pointing to the item stored in |l| together with a pointer |l.next| to the 
next link. We don't store information in the first link because there's 
no previous link
pointing to it; therefore the datapointer in the first link is always null.
In the last link only the datapointer will ever be non-null, because there's
no "next link" to point to.
 All functions in this file can be used for lifo, fifo and ordered linked
lists unless otherwise specified. Those functions which are list function 
table functions must take
data pointer arguments (since the structure |llist| is unknown to the |list|
manipulation functions), although they will only ever be used to manipulate
linked lists.

The items in any one list (either a linked list or a priority queue), 
necessarily all of the same type, are manipulated 
using standard functions which operate on data structures of that particular 
type and appear in a function table associated with that particular datatype. 
These functions may be called by the |llist| functions defined in
this file, but their definitions and descriptions appear in the files
associated with that particular datastructure. Of course the relevant .c
file needs to be compiled together with `llist.c' for any data structure
being stored in a linked list. 

Each data structure function table is an array containing items labelled from 
0 to 7, and called in this file according to their position in that array.
  Since the function table is actually an array in  which all elements must
necessarily have the same type, we declare it as an array of type |PTFNRDP|,
(pointer to function returning datapointer).
This means that all function table functions are declared to return datapointers
and for use may need to be cast to another type.
Since the `llist' routines don't know what kind of structure is being stored
in the list, the data must be accessed via datapointers; thus the arguments
of these function table functions are always datapointers.
The data structure function table always contains the following items.

A signature function. This returns +1 if the first argument is ``earlier''
in the desired ordering of this kind of structure and -1 if the 
first argument is ``later''. (so +1 if the two arguments appear
in the order coherent withn the desired total ordering, -1 if the opposite 
is true). 
It returns zero if the two arguments are ``equal''. this
does not necessarily mean the two arguments are actually equal. For
example, we might want to regard the arguments as equal if they are
equal in several important respects.
A copy function returning void.
A create function, which gets space for exactly one object. It
returns a pointer to type {\bf XX}.
A kill function, which takes back the space obtained.
A print function.
A debugging print function, which prints out more detail of the structure.
The 6th and 7th items in a data structure function table are not used for
structures stored in linked lists, but only for manipulating structures
stored in priority queues). For data structures which will only ever be
stored in linked lists these two entries can either be set to 0 or left out
altogether.
*/

#define SIGNATURE 0
#define COPY 1
#define CREATE 2
#define KILL 3
#define PRINT 4
#define DPRINT 5
#include <stdio.h>
#include "defs.h"
#include "list.h"
static boolean llist_insert PARMS(( dp LLp, dp in));
static boolean llist_put_last PARMS(( dp LLp, dp in));
static boolean llist_put_first PARMS(( dp LLp, dp in));
static boolean llist_find PARMS((dp LLp, dp in, dp out) );
static boolean llist_delete PARMS((dp LLp, dp in) );
static boolean llist_get_first PARMS((dp LLp, dp out));
static boolean llist_delget_first PARMS((dp LLp, dp out));
static boolean llist_empty PARMS(( dp LLp));
static void llist_cpy PARMS(( dp oldLLp, dp newLLp));
static int llist_sgn PARMS((dp LLp1,dp LLp2));
static void llist_init PARMS((llist *, elt_fntab *));
static llist * llist_create PARMS((elt_fntab *));
static void llist_reset PARMS(( dp LLp));
#define llist_clear(llp)    llist_reset((dp)(llp))
static void llist_kill PARMS((dp LLp));
static void llist_print PARMS ((FILE* wfile, dp LLp));
static void llist_traverser_init PARMS((llist_traverser* ltp,llist* llp));
static llist_traverser * llist_traverser_create PARMS((dp LLp)); 
static void llist_traverser_kill  PARMS((dp LLtp));
static boolean llist_next PARMS(( dp LLtp, data_ptr out));




/* Initialize a linked list , giving the address of the appropriate function
table.
The list is initialized so that the first link 
is the only link.  That means that here's no "next link" after the first, so 
both pointers in the first link
are set to be null. The address of the last link must be the address of the
first and only link.
*/
static void 
llist_init(llp,fntab) 
	llist *llp; 
	elt_fntab *fntab; 
{ 
	llp->first.next = NULL; 
	llp->first.dp = NULL; 
	llp->lastp = &(llp->first); /* the first link is also the last */
	llp->fntab = fntab; 
} 


static llist * 
llist_create(fntab) 
	elt_fntab *fntab; 
{ 
	llist * llp; 
	llp = vzalloc1(llist); 
	llist_init(llp,fntab); 
	return(llp);  
} 

/*  The next function clears a linked list and returns
it to the empty state it is in
immediately after initialisation (so that, in fact, the list is all set to
be reused).
The links are removed one at a time, starting with the second, then the
third, until the only link remaining is the first, which now consists of a
pair of null pointers.

*/
static void 
llist_reset(LLp) 
	dp LLp; 
{ 
	llist * llp; 
	link * p; 
	link * q; 
	auto void (*killfnp) PARMS ((dp)); 

	llp = (llist *)LLp; 
	p = llp->first.next;/* |p| is the pointer from the first link to the
						next (i.e. the second) one. */ 
	while(p) { /* while there is still a second link */ 
	
		q = p; /* |q| is the second link */
		killfnp = llp->fntab->kill; 
		p= p->next; /* move |p| to point to the third link */
		(*killfnp)(q->dp);/* free the space occupied by the data  */  
		q->dp=0;
		Free_dp((dp)q); q = 0; /*free the space occupied by the link*/
  /* (p could become a null pointer) */ 
	}
	llp->first.next = NULL ;/* first is the only link now */ 
	llp->lastp = &(llp->first); /*the first link is also the last link*/
} 


static void 
llist_kill(LLp) 
	dp LLp; 
{ 
	llist_clear((llist *)LLp);  
	Free_dp(LLp); LLp = 0;  
} 


/* Initialize a |llist_traverser| to make it traverse a particular linked list, 
whose address is given in the second argument.  A |llist_traverser| attached 
to a list consists of a pointer (|upto|) to a link in
that list together with the function table of the list . At initialization
the pointer is set to point to the first link in the list .
The list traverser doesn't need to be assigned space, since it is always set
equal to something which already exists.
*/
static void llist_traverser_init(lltp,llp) 
	llist_traverser* lltp; 
	llist* llp; 
{ 
	lltp->upto = &(llp->first); 
	lltp->fntab = llp->fntab; 
} 


static llist_traverser * llist_traverser_create(LLp) 
	dp  LLp; 
{ 
	llist_traverser* lltp; 
	llist* llp; 

	llp = (llist *)LLp; 
	lltp = vzalloc1(llist_traverser); 
	llist_traverser_init(lltp,llp); 
	return(lltp); 
} 


static void
llist_traverser_kill(LLtp) 
	dp LLtp; 
{ 
	Free_dp(LLtp); LLtp=0; 
} 

/* |llist_next()| is used when traversing a list . It
returns |TRUE| if there is a next element on the list (i.e. if the
|llist_traverser| points to a link which is not the last link),
and |FALSE| otherwise. If there is a next item, the |llist_traverser| is
advanced one item and the next item is put into the
address given by the datapointer |out| (using the copy function in the
relevant function table).  The copy function clears |*out| 
and assigns the more space to it as necessary, so there is no need to
reset it if it has been used before. |*out| should however be
initialized. 

The procedure does not change the list being traversed.
*/
static boolean 
llist_next(LLtp,out) 
	dp  LLtp, out; 
{ 
	llist_traverser * lltp; 
	lltp = (llist_traverser *)LLtp; 

	if (lltp->upto->next ==  NULL) /*the traverser is set to point at 
the last link */ 
		return FALSE; 
	lltp->upto = lltp->upto->next; /*otherwise move the traverser to point
to the next link */ 
	(*(lltp->fntab->copy))(lltp->upto->dp,out); 
/* copy the data in the next link into the address given by |out| */
	return TRUE; 
} 

/* The Boolean valued function |llist_empty()| returns |TRUE| if the linked list
 is empty, |FALSE| otherwise without changing the list in any way.
*/
static boolean llist_empty(LLp)
	dp LLp;
{ 
	llist * llp; 
	llp = (llist *)LLp; 
	if (llp->first.next == NULL) 
		return TRUE; 
	else 
		return FALSE; 
} 


/* Is there an item on the linked list which is equal (by the lights of the
appropriate signature function) to the item pointed to by |in|? The
procedure returns |TRUE| or |FALSE|. If the answer is |TRUE|, then
|out| is made to point to a copy of the item on the list . |*out| does
not need to be a 100\% faithful copy of |*in|. It may only be a 95\%
faithful copy. Whether it is guaranteed to be
a faithful copy, or is not guaranteed to be faithful, depends on the
signature function and the copy function associated to this particular
list. 
  The list is assumed to be ordered. We run through it with a pointer,
|p|, to point to the last link in the list 
that contains an item smaller than the item pointed to by |in|.
(We point to the first link if no item is smaller than |*in|.) If there
is a next link, it contains the only item that could possibly be a copy
of |*in|. Otherwise there is no copy.
*/
static boolean 
llist_find(LLp,in,out) 
	dp LLp, in, out; 
{ 
	llist * llp; 
	link *p; 
	auto int (*sgnfnp) PARMS ((dp,dp)); 
	auto void (*copyfnp) PARMS ((dp,dp)); 

	llp = (llist *)LLp; 
	assert(in); 
	sgnfnp = llp->fntab->signature; 
	copyfnp = llp->fntab->copy; 
	p = &(llp->first); 
	while ((p->next) &&  (*sgnfnp)(in,p->next->dp) == -1) 
				p = p->next; 
	if (p->next && (*sgnfnp)(in,p->next->dp) == 0) { 
		(*copyfnp)(p->next->dp,out); 
		return TRUE; 
	} 
	else return FALSE; 
} 



/* |llist_insert()| inserts an item into an ordered list . (This will not
work well unless the list is ordered.) If an item accepted to be the
same as |*in| by the signature function in the relevant function table is 
already in the list , it is replaced by |*in|. The function returns true if
the item was not already in the list in some form or another (i.e. the item
was well and truly inserted and the list got longer).
*/
static boolean
llist_insert(LLp,in) 
	dp LLp, in; 
{ 
	llist * llp; 
	link *p; 
	auto int (*sgnfnp) PARMS ((dp,dp)); 
	auto void (*copyfnp) PARMS ((dp,dp)); 
	boolean ans = TRUE;
	assert(in); 
	llp = (llist *)LLp; 
	sgnfnp = llp->fntab->signature; 
	copyfnp = llp->fntab->copy; 
	p = &(llp->first); 
	while ((p->next) &&  (*sgnfnp)(in,p->next->dp) == -1) 
				p = p->next; 
	if ((p->next)&& (*sgnfnp)(in,p->next->dp) == 0) { 
			/* something similar enough to |*in| has been found */
		(*copyfnp)(in,p->next->dp);
		ans = FALSE;
	} 
	else { 
		link * linkp; 
		auto data_ptr (*createfnp) PARMS ((VOID)); 
	
		createfnp = llp->fntab->create; 
		copyfnp = llp->fntab->copy; 
		linkp = valloc1(link); 
		linkp->next = p->next; /* what was the "next link" becomes the
				link after the new link. If we're at the end of
				the list this is a null pointer.*/ 
		linkp->dp = (*createfnp)(); 
		(*copyfnp)(in,linkp->dp);  
		if (p->next == 0) 
			llp->lastp = linkp; /* if we were at the end of the
	list, the new link is now the last link */ 
	p->next = linkp;/*the new link is now the "next link" after |*p|*/
	} 
	return ans;
} 


/* |llist_put_last()| puts an item on the end of a `fifo' list , whether it's
already there or not. 
Although the function must be boolean valued it always returns the value
|TRUE|.
Warning This should ONLY be used for a `fifo' list. . 
*/

static boolean
llist_put_last(LLp,in) 
	dp LLp, in; 
{ 
	llist * llp; 
	auto void (*copyfnp) PARMS ((dp,dp)); 
	link *p; 
	link * linkp; 
	auto data_ptr (*createfnp) PARMS ((VOID)); 
	assert(in); 
	llp = (llist *)LLp; 
	p = llp->lastp; 

	createfnp = llp->fntab->create; 
	copyfnp = llp->fntab->copy; 
	linkp = valloc1(link); 
	linkp->next = p->next; /* what was the "next link" becomes the
			link after the new link. If we're at the end of
			the list this is a null pointer.*/ 
	linkp->dp = (*createfnp)(); 
	(*copyfnp)(in,linkp->dp);  
	if (p->next == 0) 
		llp->lastp = linkp; /* if we were at the end of the
list, the new link is now the last link */ 
	p->next = linkp;/*the new link is now the "next link" after |*p|*/
	return TRUE;
} 

/* |llist_put_first()| puts an item at the beginning of a `lifo' list , whether 
it's already there or not. 
Although the function must be boolean valued it always returns the value
|TRUE|.
Warning This should ONLY be used for a `lifo' list. . 
*/

static boolean
llist_put_first(LLp,in) 
	dp LLp, in; 
{ 
	llist * llp; 
	auto void (*copyfnp) PARMS ((dp,dp)); 
	link *p; 
	link * linkp; 
	auto data_ptr (*createfnp) PARMS ((VOID)); 
	assert(in); 
	llp = (llist *)LLp; 
	p = &(llp->first); 

	createfnp = llp->fntab->create; 
	copyfnp = llp->fntab->copy; 
	linkp = valloc1(link); 
	linkp->next = p->next; /* what was the "next link" becomes the
			link after the new link. If we're at the end of
			the list this is a null pointer.*/ 
	linkp->dp = (*createfnp)(); 
	(*copyfnp)(in,linkp->dp);  
	if (p->next == 0) 
		llp->lastp = linkp; /* if we were at the end of the
list, the new link is now the last link */ 
	p->next = linkp;/*the new link is now the "next link" after |*p|*/
	return TRUE;
} 

/* Delete an item from the list if the signature function says it is
``equal'' to the item pointed to by |in|.
Returns |TRUE| if an item has been deleted, and |FALSE| otherwise.
This procedure should only be used on an ordered list .
*/
static boolean llist_delete(LLp,in) 
	dp LLp, in; 
{ 
	llist * llp; 
	link *p; 
	auto int (*sgnfnp) PARMS ((dp,dp)); 

	assert(in); 
	llp = (llist *)LLp; 
	sgnfnp = llp->fntab->signature;
	p = &(llp->first); 
	while ((p->next) &&  (*sgnfnp)(in,p->next->dp) == -1) 
				p = p->next; 
	if ((p->next)&&(*sgnfnp)(in,p->next->dp) == 0) { 
		/* something similar enough to |*in| has been found */
		link * temp; 
		auto void (*killfnp) PARMS ((dp)); 
		killfnp = llp->fntab->kill; 
		temp = p->next; 
		(*killfnp)(temp->dp);  
		temp->dp=0;
		if (temp->next == 0) /*we're deleting the last link*/ 
			llp->lastp = p; 
		p->next = temp->next;/* what was the link after next becomes
			the next link */ 
		Free_dp((dp)temp); temp = 0; 
		return TRUE; 
	} 
	else return FALSE; 
} 



/* Get a copy of the first item on the linked list,
and put it into |*out|. The list 
does not change. The procedure returns |FALSE| if the list is empty,
and |TRUE| otherwise. Since no data is ever stored in the first link the
listis non-empty if there is a link after the first. The data is copied
using the copy function from the relevant function table.
*/
static boolean 
llist_get_first(LLp,out) 
	dp LLp, out; 
{ 
	llist * llp; 
	llp = (llist *)LLp; 
	if (llp->first.next != NULL){ /* there is a second link */
		auto void (*copyfnp) PARMS ((dp,dp)); 
		copyfnp = llp->fntab->copy; 
		(*copyfnp)(llp->first.next->dp,out); 
		return TRUE; 
	} 
	else return FALSE; 
} 


/* Get a copy of the first item of the list and put it into |*out|.
Remove the first item from the list .
Returns |TRUE| or |FALSE|, in the same way as |llist_get_first()|.
Since the first item in the list is stored in the second link, not the 
first the list is non-empty precisely if there is a second link.
*/
static boolean 
llist_delget_first(LLp,out) 
	dp LLp, out; 
{ 
	llist * llp; 
	link * linkp; 
	llp = (llist *)LLp; 
	if ((linkp = llp->first.next) != NULL){ /* there is a second link */
		auto void (*copyfnp) PARMS ((dp,dp)); 
		auto void (*killfnp) PARMS ((dp)); 
		copyfnp = llp->fntab->copy; 
		(*copyfnp)(linkp->dp,out); 
		killfnp = llp->fntab->kill; 
		(*killfnp)(linkp->dp); 
		linkp->dp=0;
		llp->first.next = linkp->next; 
		Free_dp((dp)linkp); linkp = 0; 
		if (llp->first.next == NULL) /*empty dp*/ 
			llp->lastp = &(llp->first); 
		return TRUE; 
	} 
	else return FALSE; 
 
} 




/* Print for ordinary use.
The data in each link after the first  (where there is no data stored)
is printed out using the 
print function from the relevant function table. If a link contains no
data the message "NULL data" is printed.
*/
static void 
llist_print(wfile,LLp) 
	FILE * wfile;
	dp LLp; 
{ 
	llist * llp; 
	link * p; 
	auto void (*printfnp) PARMS ((dp)); 
 
	llp = (llist *)LLp; 
	assert(llp); 
	printfnp = llp->fntab->print; 
	p = (llp->first).next; 
	while(p) { 
		if (p->dp) {
			(*printfnp)(wfile,p->dp); 
			fprintf(wfile,"\n"); 
		}
		else
			fprintf(wfile," NULL data "); 
		p = p->next; 
	} 
} 

/* Copy a linked list . The list |*oldLLp| is untouched and |*newLLp| is 
a copy of
|*oldLLp|. The original contents of |*newLLp| are destroyed.
The two lists are assumed to be of the same type, i.e. they contain the
same kind of data and use the same function table.
*/
static void 
llist_cpy(oldLLp, newLLp) 
	dp oldLLp, newLLp; 
{ 
	llist * oldllp, *newllp; 
	link * oldp, *newp; 

	oldllp  = (llist *)oldLLp; 
	newllp = (llist *)newLLp; 
	llist_reset(newLLp); 
	oldp = &(oldllp->first); 
	newp = &(newllp->first); 
	while (oldp->next) { 
		link *linkp; 
		auto data_ptr (*createfnp) PARMS ((VOID)); 
		auto void (*copyfnp) PARMS ((dp,dp)); 
		linkp = valloc1(link); 

		createfnp = newllp->fntab->create; 
		copyfnp = oldllp->fntab->copy; 
		linkp->dp = (*createfnp)(); 
		if (oldp->next->dp)
			(*copyfnp)(oldp->next->dp,linkp->dp);  
		linkp->next = 0; 
		newp->next = linkp; 	
		newllp->lastp = linkp; 
		oldp = oldp->next; 
		newp = newp->next; /* move forward in both lists */
	} 
} 

/* We order lists by dictionary order. Returns 1, 0 or -1 according to
whether the two lists are in the correct order, are equal or are in
the wrong order.
*/
static int
llist_sgn(LLp1, LLp2)
	dp LLp1, LLp2;
{
	llist *llp1= (llist *)LLp1;
	llist *llp2= (llist *)LLp2;
	int ans = 0;
	assert(llp1->fntab == llp2->fntab);
	{
		auto int (*sgnfnp) PARMS ((dp,dp)); 
		auto data_ptr (*createfnp) PARMS ((VOID)); 
		auto void (*killfnp) PARMS ((dp)); 
		llist_traverser t1,t2;
		dp d1, d2;
		
		sgnfnp = llp1->fntab->signature;
		assert(sgnfnp!=0);
		createfnp = llp1->fntab->create; 
		killfnp = llp1->fntab->kill;
		d1 = (*createfnp)();
		d2 = (*createfnp)();
		llist_traverser_init(&t1,llp1);
		llist_traverser_init(&t2,llp2);
		while (ans == 0 && llist_next((dp)&t1,d1)) {
			if (llist_next((dp)&t2,d2))
				ans = (*sgnfnp)(d1,d2);
			else /*first list longer and second list is a sublist*/
				ans = -1;
		}
		if (ans == 0 && llist_next((dp)&t2,d2)) { /*second list is longer
					and first list is a sublist*/
			assert(!llist_next((dp)&t1,d1));
			ans = 1;
		}
		(*killfnp)(d1);  d1=0; 
		(*killfnp)(d2);  d2=0; 
	}
	return ans;
}

list_fntab
ORDERED_fntab =  {  
	llist_insert, 
	llist_find, 
	llist_delete, 
	llist_get_first, 
	llist_delget_first, 
	llist_empty, 
	llist_cpy, 
	(E2DP)llist_create, 
	llist_kill, 
	llist_reset, 
	llist_print, 
	(DP2DP)llist_traverser_create, 
	llist_traverser_kill, 
	llist_next,
	llist_sgn
}; 

list_fntab
FIFO_fntab =  {  
	llist_put_last, 
	0, 
	0, 
	llist_get_first, 
	llist_delget_first, 
	llist_empty, 
	llist_cpy, 
	(E2DP)llist_create, 
	llist_kill, 
	llist_reset, 
	llist_print, 
	(DP2DP)llist_traverser_create, 
	llist_traverser_kill, 
	llist_next,
	0
}; 

list_fntab
LIFO_fntab =  {  
	llist_put_first, 
	0, 
	0, 
	llist_get_first, 
	llist_delget_first, 
	llist_empty, 
	llist_cpy, 
	(E2DP)llist_create, 
	llist_kill, 
	llist_reset, 
	llist_print, 
	(DP2DP)llist_traverser_create, 
	llist_traverser_kill, 
	llist_next,
	0
}; 


 
