/* lib.d file list.h */
/*
This is a header file for lists.
A |list| is used to store a collection of items of the same type. There are
various structures we might choose for a list; which structure we choose
depend on what we want to do with the list, e.g. which operations on it we
want to be able to perform particularly quickly. Here a |list| will have one
of the following structures:-
\item{a.} an `ordered' list. The items in the list appear in the list in the
total order defined by the  `signature' function associated to the 
data structures being stored.  No item may appear in the list more
than once. The first ordered item in the list can be retrieved or
deleted. It is also possible to search the list to see whether or not
 a copy of some  particular item (i.e. an item deemed by the signature
function to be identical with some specified one) is in there, and to delete
such an item if it exists.
\item{b.} a `fifo' (first-in-first-out) list, or `queue'. The items in the list
appear in the list in the order in which they were inserted. Items are 
therefore always inserted at one end (the `back' end), and retrieved or
deleted from the other (the `front' end). Many copies of the same item may
exist in the list.
\item{c.} a `lifo' (last-in-first-out) list, or `stack'. Only the most
recently inserted item in the list can be accessed or deleted at any time.
Many copies of the same item may exist in the list.
\item{d.} a priority queue.
The items in the list are partially sorted within the list according to the
`signature' function, i.e. there is a first item in the list, and precedence
within the list between some but not all pairs of items. In fact the items
are stored in an array whose indices appear as the vertices of a rooted
binary tree, the first item as the root. As each item is inserted it is
marked, so that although there may be other items in the queue  which are
copies of a given item according to the signature function, no two items are 
in fact indistinguishable.  The first item can
be quickly retrieved and/or deleted, and an item can quickly be inserted into
a position which is consistent with the total ordering defined by the
`signature' function. It is possible for many `copies' (in the sense of the
word used above) of the same item to be in the queue. A specific
item can easily be deleted but it is hard to search the list for a copy
of a particular item. 

A list is manipulated using standard
functions contained in one of the four list function tables (called
|ORDERED|, |FIFO|, |LIFO|and |PQUEUE|) relevant to that particular list. 

The structure of a |list| (ordered, fifo, lifo or priority queue) is specified 
at the initialization stage. The appropriate list function table is attached and
the list is initialized as a list with that structure containing data of a
specified type . After initialization the structure of the list can be
ignored by the user. All manipulation on the list is done using the
functions described in this header file, which themselves call the appropriate
functions from the list function tables.

The definition of the functions in the |list| function files and other code
which needs to know the actual structures used for the different kinds of
|list|s are in the files |llist.c| (for ordered, fifo and lifo  lists, for which
the structure of linked lists is used) and |pqueue.c| (for priority queue).
Since
the |list.c| code calls code from |llist.c|  or |pqueue.c| (depending on the
type of |list| being manipulated) it is important to compile the relevant
one of these files together with |list.c|. It should not however be necessary
for the ordinary user of the |list| routines to know anything about
the actual contents of any of the `llist'
or  `pqueue' files.
*/

/* The structure of a |list|. We start off with some casts which will
be useful in element function tables.
*/
typedef	int		(*DPDP2I)			PARMS((dp,dp));
typedef	void	(*DPDP2V)			PARMS((dp,dp));
typedef	dp		(*V2DP)				PARMS((VOID));
typedef	void	(*DP2V)				PARMS((dp));
typedef	int		(*DP2I)				PARMS((dp));
typedef	void	(*DPI2V)			PARMS((dp, int));
typedef dp		(*DP2DP)			PARMS((dp));

typedef struct {
	int		(*signature)		PARMS((dp,dp));
	void	(*copy)				PARMS((dp,dp));
	dp		(*create)			PARMS((VOID));
	void	(*kill)				PARMS((dp));
	void	(*print)			PARMS((FILE *,dp));
	int		(*getheapindex)		PARMS((dp));
	void	(*setheapindex)		PARMS((dp, int));
}
elt_fntab; /*function table for manipulating the elements of a list*/
typedef dp		(*E2DP)				PARMS((elt_fntab*));
typedef struct {
	boolean		(*insert)			PARMS ((dp list_address , dp in));
	boolean		(*find)				PARMS ((dp list_address, dp in, dp out));
	boolean		(*deletefn)			PARMS ((dp list_address, dp out));
	boolean		(*get_first)		PARMS ((dp list_address, dp out));
	boolean		(*delget_first)		PARMS ((dp list_address, dp out));
	boolean		(*empty)			PARMS ((dp list_address));
	void		(*copy)				PARMS ((dp old_address,dp new_address));
	dp			(*create)			PARMS ((elt_fntab *));
	void		(*kill)				PARMS ((dp list_address));
	void		(*reset)			PARMS ((dp list_address));
	void		(*print)			PARMS ((FILE * wfile,dp list_address));
	dp			(*trav_create)		PARMS ((dp list_address));
	void		(*trav_kill)		PARMS (( dp list_trav_address));
	boolean		(*next)				PARMS (( dp list_address, dp out));
	int			(*signature)		PARMS ((dp list_address1,dp list_address2));
}
list_fntab;

typedef struct { 
	dp address; /* in reality the datapointer will point to a structure
                                 of type |llist| or |pqueue|.*/ 
	list_fntab * fntab;/*address of table of functions which manipulate 
				lists */ 
	int lock; /* this shows whether or not the list is being traversed,
and prevents changes being made to the list during a traversal. Many
traversals are possible simulaneously; the number of traversals at any
pointin time is recorded */
} list; 

typedef struct { 
	dp address; /* in reality a pointer to a structure of type, say, 
				|llist_traverser|  */ 
	list_fntab * fntab; 
	int * lockaddress;
}
list_traverser; 

/* The structure of linked lists. */

/* The basic building block of a |llist| is a |link|.  */
typedef struct link { 
	struct link * next;
	data_ptr dp;
} link;

/* The structure of a |llist|. */
typedef struct {
	elt_fntab *fntab;/*address of table of functions which manipulate 
		structures stored in list*/
		/*Functions in the table need to be cast to return
			the correct kind of value. */
	link first;
	link * lastp;
} llist;

/* The structure used to traverse a linked list.  */
typedef struct {
	elt_fntab * fntab; /*the function table of the list itself*/
	link * upto; /*space is not specially  allocated for this pointer. It
		is designed to move along the list, pointing at what is there
		already*/
} llist_traverser;

/* The structure of priority queues.  */
typedef struct {
	elt_fntab * fntab;/*address of table of functions which manipulate
			objects in the queue */
	dp *heap;
	int heapsize;
	int maxheapsize;
}
pqueue;

typedef struct {
	dp  queuep;/* a pointer to the queue being traversed*/
	int place;
}
pqueue_traverser;

/* The structure of self adjusting trees.  */
typedef struct treelink { 
	struct treelink * child[2]; 
	data_ptr dp;
} treelink;

typedef struct {
	elt_fntab *fntab;/*address of table of functions which manipulate 
		structures stored in list*/
		/*Functions in the table need to be cast to return
			the correct kind of value. */
	treelink *  root;
} satree;

typedef struct {
	elt_fntab * fntab; /*the function table of the list itself*/
	treelink * current; /*space is not specially  allocated for this pointer. It
		is designed to move along the list, pointing at what is there
		already*/
	list * ancestry;
} satree_traverser;

/* The list functions. */

/* 
Initialize a list, giving the address of the appropriatelist and data function
tables. |listfntab| should be set to either |ORDERED|, |FIFO|, |LIFO| or 
|PQUEUE|, as appropriate.
*/

extern void 
list_init PARMS((list *, elt_fntab *, list_fntab *));

/* The next function deallocates space allocated by |list_init()| */
extern void list_clear PARMS(( list * lp));

/* Reset a list for reuse. */
extern void list_reset PARMS(( list *lp)); 

/* Initialize a |list_traverser| to make it traverse a particular
list, whose address is given in the second argument.
*/
extern void list_traverser_init PARMS((list_traverser * ltp, list * lp)); 

/* Deallocate the space occupied by a |list_traverser|.
*/
extern void list_traverser_clear PARMS(( list_traverser *ltp)); 

/* |list_next()| is used when traversing a list. It
returns |TRUE| if there is a next element on the list.
The procedure does not change the list being traversed.
|ltp| is the address of a list traverser and |out| is the address of
the item.
*/
#define list_next(ltp,out) list_next_dp((ltp),(dp)(out))
extern boolean list_next_dp PARMS ((list_traverser*,dp));

/* The Boolean valued function |list_empty()| returns |TRUE| if the list is
empty, |FALSE| otherwise without changing the list in any way.
*/
extern boolean list_empty PARMS(( list * lp));

/* Is there an item on the list which is equal (by the lights of the
appropriate signature function) to the item pointed to by |in|? The
procedure returns |TRUE| or |FALSE|. If the answer is |TRUE|, then
|out| is made to point to a copy of the item on the list. |*out| does
not need to be a 100\% faithful copy of |*in|. It may only be a 95\%
faithful copy. Whether it is guaranteed to be
a faithful copy, or is not guaranteed to be faithful, depends on the
signature function and the copy function associated to this particular
list.
*/
#define list_find(lp,in,out)	list_find_dp((lp),(dp)(in),(dp)out)
extern boolean list_find_dp PARMS ((list*,dp in, dp out));

/* |list_insert()| inserts an item into a list.
|lp| is the address of a list and |in| is the address of
the item. It returns |TRUE| if an item is inserted.
*/
#define list_insert(lp,in)	list_insert_dp((lp),(dp)(in))
extern boolean list_insert_dp PARMS ((list*,dp));

/* Delete an item from the list if the signature function says it is
``equal'' to the item pointed to by |in|.
Returns |TRUE| if an item has been deleted, and |FALSE| otherwise.
|lp| is the address of a list and |in| is the address of
the item.
*/
#define list_delete(lp,in)	list_delete_dp((lp),(dp)(in))
extern boolean list_delete_dp PARMS ((list*,dp));

/* Get a copy of the first item on the list, and put it into |*out|. The list
does not change. The procedure returns |FALSE| if the list is empty,
and |TRUE| otherwise.
|lp| is the address of a list and |in| is the address of
the item.
*/
#define list_get_first(lp,in)	list_get_first_dp((lp),(dp)(in))

/* Get a copy of the first item of the list and put it into |*out|.
Remove the first item from the list.  Returns |TRUE| or |FALSE|, 
in the same way as |list_get_first()|.  |lp| is the address of a list 
and |in| is the address of the item.  */
#define list_delget_first(lp,in)	list_delget_first_dp((lp),(dp)(in)) 
extern boolean list_delget_first_dp PARMS ((list*,dp out));

/* Print for ordinary use.
*/
extern void list_print PARMS ((FILE * wfile, list * lp));

/*Copy a list. The list |*oldp| is untouched and |*newp| is a copy of
|*oldp|. The original contents of |*newp| are destroyed.
Warning Do not try to copy a list to itself.
*/
extern void list_cpy PARMS(( list* oldp, list* newp));

extern int list_sgn PARMS (( list *lp1, list *lp2)); 




extern list_fntab PQUEUE_fntab;
#define PQUEUE (&PQUEUE_fntab)
extern list_fntab ORDERED_fntab;
extern list_fntab FIFO_fntab;
#define ORDERED (&ORDERED_fntab)
#define FIFO	(&FIFO_fntab)
extern list_fntab LIFO_fntab;
#define LIFO	(&LIFO_fntab)
extern list_fntab SATREE_fntab;
elt_fntab TREELINKP_fntab;
#define SATREE	(&SATREE_fntab)
#define TREELINKP (&TREELINKP_fntab)

