/*  Derek's original version of the Knuth-Bendix process used to
    compute the first word difference machine, with Sarah's
    input functions tagged on.
    After tidying, word differences are computed, and
    those equations that give new word differences are treated earlier.
    The equations are linked by fpt for this purpose
    Word differences are closed under inverses.
    Then, the transition function from
    X x X operating on word differences is computed.
    This is the new untyped version.
    Output of diff tables is as if dollar=num_gens+1
*/
#define tmalloc(D,T,N) {D = (T *) malloc(sizeof(T)*(N)); \
  if (D==0) { fprintf(stderr,"Out of space.\n"); exit(2);}}
#define tfree(D) {if (D) free( (char *) D); D=0;}

#include <stdio.h>
#include <ctype.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/times.h>

#include "defs.h"
#include "list.h"
#include "word.h"
#include "input.h"

#define PROCRELSOP ".procrels"
#define DIFF1OP ".diff1"
#define FSASPACE 400000
#define MRELS 10000
#define MDIFF 2000
#define MLEN  5000
#define MSTLLEN 40001

extern word * user_gen_name;
extern gen * inv_of;
extern int gen_array_size;
extern int num_gens;

char *word1[MRELS],*word2[MRELS],tt1[MLEN+1],tt2[MLEN+1],
     *test1,*test2,*wspe,
     tidyon,*diffword[MDIFF],
     inf[100],outf[100],subsetcf[MDIFF],nw,onintr;
char print_kb_rules=0,print_wdiffs=0,tidset=0,rephalt=0,timehalt;

int nrels,nnewrels,ninvrels,mrels=MRELS,fsasp[FSASPACE],*fsa[128],
 num_gens,mstates,nstates,history[MLEN],tidint,lasttid,fpt[MRELS],stl1[MSTLLEN],
   stl2[MSTLLEN],crelno,lastrel,lastrel2,ondiff,ndiff,lastndiff,
   ***difftab,num_reps,max_reps,maxtime,timegone=0,ntidies=0;
short *cf1,*cf2;
FILE *fopen(),*ip=0,*op=0;
struct tms buffer;
/* word1[i] and word2[i] point to the left and
   right hand sides of the ith relation. There can be at most MRELS relations,
   and the individual strings can have length at most MLEN.
   The FSA will be stored within the array fsasp, as a sequence of vectors,
   one for each generator pointed to by *fsa, where the length of each vector
   is equal to the current number of states. mstates is the maximum number
   of states that we have room for.
*/
interrupt() {if (onintr) tidyon=1; else onintr=1;}

main(argc,argv) int argc; char *argv[];
{ int len,i,cr2,j,k,l,n,onrels,nr2,cr,arg;
  char *ptr,c,d,keyword[30];
   word uw1,uw2,pw2,rel;

  setbuf(stdout,(char*)0);
  setbuf(stderr,(char*)0);

  test1=tt1+1; test2=tt2+1;

  arg=1;
  while (arg<argc && argv[arg][0]=='-')
  { if (argv[arg][1]=='r') print_kb_rules=1;
    else if (argv[arg][1]=='w') print_wdiffs=1;
    else if (argv[arg][1]=='t')
    { tidset=1;
      arg++; if (arg>=argc) badusage();
      tidint=stringtoint(argv[arg]);
    }
    else if (argv[arg][1]=='n')
    { rephalt=1;
      arg++; if (arg>=argc) badusage();
      max_reps=stringtoint(argv[arg]);
    }
    else if (argv[arg][1]=='m')
    { timehalt=1;
      arg++; if (arg>=argc) badusage();
      maxtime=stringtoint(argv[arg]);
    }
    else badusage();
    arg++;
  }
  if (arg>=argc) {ip=stdin; op=stdout;}
  else
  { strcpy(inf,argv[arg]); strcpy(outf,inf);
    strcat(inf,PROCRELSOP); strcat(outf,DIFF1OP);
    if ((ip=fopen(inf,"r"))==0)
    { fprintf(stderr,"Cannot open file %s\n",inf); exit(2);}
    op=fopen(outf,"w");
    setbuf(op,(char*)0);
  }

/* Input generators */
 
  user_gen_name = vzalloc2(word,gen_array_size);
  for (i=0;i<gen_array_size;++i)
    word_init(user_gen_name+i);
     
  n=readstring(keyword);
  if (n==EOF)
  { fprintf(stderr,"Error: Could not find keyword 'words' or 'gens'.\n");
    exit(2);
  }  
  if (strcmp(keyword,"Format")==0){
        n=readstring(keyword);
        fprintf(op,"Format %s\n",keyword);
  }
  while (strcmp(keyword,"words") && strcmp(keyword,"gens"))
  {
    if (n==EOF)
    { fprintf(stderr,"Error: Could not find keyword 'words' or 'gens'.\n");
      exit(2);
    }
    n=readstring(keyword);
  }
  read_gen_name_array(ip);
   if (num_gens>127)
    { fprintf(stderr,"diff1c has a maximum of 127 generators.\n");
      exit(2);
   }
   n=readstring(keyword);
  if (strcmp(keyword,"inverses")==0)
  { read_inverse_array(ip);
    n=readstring(keyword);
    if (strcmp(keyword,"rels"))
    { fprintf(stderr,"Error: Could not find keyword 'rels'.\n");
      exit(2);
    }
  }
  else
  { default_inverse_array();
    if (strcmp(keyword,"rels"))
    { fprintf(stderr,"Error: Could not find keyword 'rels'.\n");
      exit(2);
    }
  }

/* Now insert inverse relations */
  nrels=0;
  for (i=1;i<=num_gens;i++)
  { nrels++;
      tmalloc(word1[nrels],char,3) tmalloc(word2[nrels],char,1)
      word1[nrels][0]=i; word1[nrels][1]=inv_of[i]; word1[nrels][2]=0;
      word2[nrels][0]=0;
  }
  ninvrels=nrels;


/* Now define the pointers for the FSA and initialize it. Initially, each
   generator defines its own state.
*/
  mstates=FSASPACE/(num_gens+1);
  for (i=0;i<=num_gens;i++) fsa[i]=fsasp+i*mstates-1;
  tmalloc(cf1,short,mstates+1) tmalloc(cf2,short,mstates+1)
  nstates=num_gens+1;
  for (i=1;i<=num_gens;i++) for (j=1;j<=nstates;j++) fsa[i][j]=i+1;
  for (i=1;i<=nrels;i++) fsa[*(word1[i]+1)][*word1[i]+1]= -i;
/* looks crazy, but its right! */


/* Now input the defining relations. New equations are always inserted
   initially into the strings test1 and test2.
*/
  while ((n=readchar())!='{')
    if (n==EOF)
    { fprintf(stderr,"Error: Unexpected end of file.\n");
      exit(2);
    }
 
  word_init(&rel);
  while (read_next_rel(&rel,ip)){
/* Now transfer to Derek's mode of storage! */
    ptr=test1;
    for (i=rel.first;i<=rel.last;i++)
      *(ptr++) = (rel.g)[i];
    *ptr='\0';
    *test2='\0';
    word_reset(&rel);
 
    if (strlen(test1)!=2 || test1[0]!=inv_of[test1[1]])
    { nrels++;
      if (insert(nrels)==0) nrels--;
    }
 
  }
  word_clear(&rel);
  while (getc(ip)!='\}');

  for (i=0;i<=nrels;i++) fpt[i]=i+1;
  fpt[nrels]=0; lastrel=nrels;

  tidyon=0;
  num_reps=lastndiff=0;
  if (tidset==0)
  { n=readstring(keyword);
    if (n==EOF) { tidint=30; goto Continue;}
    while (strcmp(keyword,"parameters"))
    { n=readstring(keyword);
      if (n==EOF) { tidint=30; goto Continue;}
    }
    while ((n=readchar())!='{')
    if (n==EOF)
    { fprintf(stderr,"Error: Unexpected end of file.\n");
      exit(2);
    }
again:
    while ((n=readchar())!='K' && n!='}' && n!=EOF);
    if (n==EOF || n=='}') { tidint=30; goto Continue;}
    readstring(keyword);
    if (strcmp(keyword,"Btidyint")) goto again;
    if (fscanf(ip,"%d",&tidint)<=0)
    { fprintf(stderr,"Error: Could not read parameter.\n"); exit(2);}
  }  

Continue:

/* Now we are ready to start the algorithm. The policy is that whenever a new
   relation is found, it is immediately compared with all of the inverse
   relations. This seems to be most efficient. It might also be a good idea
  to immediately compare new relations with any other very short relations.
  Otherwise, a new relation has to wait in the queue before being compared
  with the others.
*/

  tmalloc(difftab,int **,MDIFF+1)
  for (i=0;i<=MDIFF;i++) difftab[i]=0;
  signal(SIGINT,interrupt); signal(SIGKILL,interrupt);
  signal(SIGQUIT,interrupt);
  for (k=ninvrels+1;k<=nrels;k++)
  { for (l=1;l<=ninvrels;l++)
    { consider(k,l); consider(l,k); if (tidyon) break;}
    if (tidyon) break;
  }

  nnewrels=0;
  for (crelno=fpt[ninvrels];crelno!=0;crelno=fpt[crelno])
  { j=0;
     do
    { j=fpt[j];
      if (tidyon) break;
      onrels=nrels; consider(crelno,j); if (crelno!=j) consider(j,crelno);
      if (nrels>onrels)
/* Compare new relations with inverse relations */
      for (k=onrels+1;k<=nrels;k++)
      { for (l=1;l<=ninvrels;l++)
        { if (tidyon) break; consider(k,l); consider(l,k);}
        if (tidyon) break;
      }
      if (timehalt)
      { times(&buffer); timegone=buffer.tms_utime/60;
        if (timegone>maxtime && ntidies<2) tidyon=onintr=1;
      }
      if (tidyon) break;
    } while (j!=crelno);
/* If we have more than tidint new relations since last tidying up, then
   tidy; that is, go through all relations, remove the redundant ones, and
   then redefine the FSA. This will usually reduce its number of states.
*/
    if (tidyon)
    { crelno=tidyup(crelno);
      for (i=0;i<=nrels;i++) if (fpt[i]==crelno) {crelno=i; break;}
      if (finddiff()) exit(2);
      if (closeinv()) exit(2);
      if (rephalt)
      { if (ndiff!=lastndiff) num_reps=0;
        else num_reps++;
        if (num_reps>=max_reps) onintr=1;
        lastndiff=ndiff;
      }
      if (onintr)
      { output();
        if (timegone>maxtime && ntidies<3) exit(1);
        exit(0);
      }
      tidyon=nnewrels=0;
    }
  }

/* Algorithm has now completed. */
  fprintf(op,"\t#System is confluent.\n");
/* Perform final tidying. */
  nrels=tidyup(nrels);
  if (finddiff()) exit(2);
  if (closeinv()) exit(2);
  output();
  exit(0);
}

insert(relno) int relno;
/* Look at the words in test1 and test2, and remove any common prefixes and
   suffixes. If they are not both empty at that stage, then use procedure
   "compare" to see which word comes first in the ordering.
*/
{ int bigger; char *ptr1,*ptr2,*t1,*t2,*ts,*ptre1,*ptre2;
  ptr1=test1; ptr2=test2; ptre1=ptr1+strlen(test1); ptre2=ptr2+strlen(test2);
  while (ptr1<ptre1)
  if (*ptr1==*ptr2) *(ptr1++)= *(ptr2++)=0; else break;
  if (ptr1==ptre1 && ptr2==ptre2) return(0);
  if (strlen(test1) && strlen(test2))
  while (*(ptre1-1)==*(ptre2-1)) {ptre1--; ptre2--;}
  t1=ptr1; t2=ptr2; *ptre1= *ptre2= '\0';
/* Common prefixes and suffixes have now been stripped.
   transfer inverses to other side if possible
*/
  bigger=compare(t1,t2);
  if (bigger==2)
  { ts=t1; t1=t2; t2=ts; ts=ptre1; ptre1=ptre2; ptre2=ts;}
  while (strlen(t1)-strlen(t2)>2)
  { ptre1--; *ptre2=inv_of[*ptre1]; *ptre1='\0'; *(++ptre2)='\0';}
  if (strlen(t1)-strlen(t2)==2 && strlen(t2)>0)
  { if (*t1>*t2)
    { ptre1--; *ptre2=inv_of[*ptre1]; *ptre1='\0'; *(++ptre2)='\0';}
    else if (*(t1+1)>inv_of[*t1])
    { *(--t2)=inv_of[*t1]; t1++;}
  }
/* t1 is biggest according to the ordering, so it becomes the LHS of the new
   relation.
*/
  tmalloc(word1[relno],char,strlen(t1)+1)
  ptr1=t1; ptr2=word1[relno];
  while (ptr1<ptre1)  *(ptr2++)= *(ptr1++);
  *(ptr2++)='\0';
  tmalloc(word2[relno],char,strlen(t2)+1)
  ptr1=t2; ptr2=word2[relno];
  while (ptr1<ptre2)  *(ptr2++)= *(ptr1++);
  *(ptr2++)='\0';


  modifyfsa(relno);
  return(1);
}

compare(w1,w2) char *w1,*w2;
/* Compare words w1 and w2 to see which is bigger according to the ordering.
   The ordering used here is longer words are bigger, and amongst equal
   length words, lexicographical ordering according to the order of the
   generators as input is used. (I use 'bigger' to mean greater in the
   ordering.) This procedure could be altered to use a different ordering
   if desired.
*/
{ int len1,len2,n1,n2,bigger; char *we1,*we2;
  len1=strlen(w1); len2=strlen(w2);
  we1=w1+len1; we2=w2+len2;
  if (len1>len2) bigger=1; else if (len2>len1) bigger=2;
  else while (w1<we1)
  { n1= *w1; n2= *w2;
    if (n1>n2) {bigger=1; break;} else if (n2>n1) {bigger=2; break;}
    w1++; w2++;
  }
  return(bigger);
}

consider(k,l) int k,l;
/* The left hand sides of the relations k and l are considered for common
   parts, according to the Knuth-Bendix procedure, and any new relations
   produced are processed.
*/
{ char ok,*midwd,*ptr1,*ptr2,*ptr,*testwd1,*testwd2;
   midwd=testwd1=word1[k]; testwd2=word1[l];
   while (*midwd!='\0')
   { ptr1=midwd; ptr2=testwd2; ok=1;
     while (*ptr1!='\0' && *ptr2!='\0')
     { if (*ptr1!= *ptr2) {ok=0; break;} ptr1++; ptr2++;}
     if (ok)
/* A common segment has been found. The new equation for consideration is
   inserted into words test1 and test2.
*/
     { if (*ptr1=='\0')
       { ptr1=testwd1-1; ptr=test1-1;
         while (++ptr1<midwd) *(++ptr)= *ptr1;
         ptr1=word2[l]-1;
         while (*(++ptr1)!='\0') *(++ptr)= *ptr1;
         *(++ptr)='\0';
         ptr1=word2[k]-1; ptr=test2-1;
         while (*(++ptr1)!='\0') *(++ptr)= *ptr1;
         ptr1=ptr2-1;
         while (*(++ptr1)!='\0') *(++ptr)= *ptr1;
         *(++ptr)='\0';
       }
       else
       { ptr2=testwd1-1; ptr=test1-1;
         while (++ptr2<midwd) *(++ptr)= *ptr2;
         ptr2=word2[l]-1;
         while (*(++ptr2)!='\0') *(++ptr)= *ptr2;
         ptr2=ptr1-1;
         while (*(++ptr2)!='\0') *(++ptr)= *ptr2;
         *(++ptr)='\0';
         ptr2=word2[k]-1; ptr=test2-1;
         while (*(++ptr2)!='\0') *(++ptr)= *ptr2;
         *(++ptr)='\0';
       }
/* Now reduce test1 and test2, using the current relations.
*/
       reduce(test1,0); reduce(test2,0);
       if (insert(nrels+1))
       { nrels++;
         if (nrels>=mrels)
         { fprintf(stderr,"Too many rels.\n"); exit(1);}
         fpt[lastrel]=nrels; lastrel=nrels; fpt[nrels]=0;
/* Decide if it is time to tidy up. */
         if (++nnewrels==tidint) tidyon=1;
       }
     }
     midwd++;
   }
}

reduce(w,i) char *w; int i;
/* Reduce "w", by replacing any occurrences of the LHS of the current
   relations by their LHS. The FSA is used for this. The complete sequence
   of states that it goes through on reading "w" is remembered in the
   array "history". If i!=0, do not use equation <= i for reduction.
*/
{ int l,len,st,longer; char *midwd,*ptr1,*ptr2,*ptr;
  midwd=w; len=0; history[0]=1; st=1;
  while (*midwd!='\0')
  { st=history[++len]=fsa[*midwd][st];
    if (st<0 && st != -i)
/* st= -n means that we have just read the LHS or relation  n. Replace it by
   RHS, and go back to the beginning of that subword.
*/
    { st= -st; ptr1=midwd; ptr2=word2[st]-1;
      if ((longer = (strlen(word2[st])-strlen(word1[st])))>0)
/* With the current version of "compare" this cannot happen, but it has been
   put in in case it is ever desired to replace a segment by a longer one.
*/
      { ptr=w+strlen(w); while (ptr>midwd) {*(ptr+longer)= *ptr; ptr--;}
        ptr1+=longer;
      }
      len-=strlen(word1[st]); midwd=w+len-1; ptr=midwd;
      while (*(++ptr2)!='\0') *(++ptr)= *ptr2;
      if (ptr!=ptr1)
      { while (*(++ptr1)!='\0') *(++ptr)= *ptr1;
        *(++ptr)='\0';
      }
      st=history[len];
    }
    midwd++;
  }
}

tidyup(done) int done;
/* Remove redundant relations. "done" is the current relation being processed.
*/
{ int i,inew,j,k,nelim,nnrels,ret,l1,l2;
  char c,moving,retain,changed,*w1,*w2,*t1,*t2,*ptr1,*ptr2,*ptre1,*ptre2,*elim,
       **word1n,**word2n,*ch;
  ntidies++;
  lasttid=nrels;
  tmalloc(elim,char,nrels+1)
  tmalloc(word1n,char *,nrels+1)
  tmalloc(word2n,char *,nrels+1)
  tmalloc(ch,char,nrels+1)
  for (i=1;i<=nrels;i++) elim[i]=0;
  for (i=1;i<=nrels;i++) ch[i]=0;
  nelim=nnrels=0; moving=0; ret=done; changed=0;
  for (i=1;i<=nrels;i++) 
  { strcpy(test1,word1[i]);strcpy(test2,word2[i]);
    k=checkreduce(test1,i);
    if (k==1)
    { retain=1;
      reduce(test2,i);
      if (strcmp(test2,word2[i]))
      { t1=test1; t2=test2;
        changed=ch[i]=1;
      }
    }
    else
    { reduce(test1,i); reduce(test2,i);
      retain=strcmp(test1,test2);
      if (retain)
      { if (compare(test1,test2)==1) { t1=test1; t2=test2;}
        else {t1=test2; t2=test1;}
        changed=ch[i]=1;
      }
    }
    if (ch[i])
    { while((l1=strlen(t1))-(l2=strlen(t2)) > 2)
      { t2[l2]=inv_of[t1[l1-1]]; t1[l1-1]=t2[l2+1]='\0';}
      tmalloc(word1n[i],char,strlen(t1)+1)
      tmalloc(word2n[i],char,strlen(t2)+1)
      strcpy(word1n[i],t1); strcpy(word2n[i],t2);
    }

    if (retain==0) { moving=1; elim[i]=1;}
  }
/* If any relations have been eliminated, then we rearrange the remainder,
   and completely reconstruct the FSA from scratch. This is not too time
   consuming provided that we do not tidy up too often.
*/
  if (moving || changed)
  { moving=0;
    for (i=1;i<=nrels;i++) if (elim[i]==0)
    { nnrels++;
      if (ch[i])
      { tfree(word1[i]) tfree(word2[i])
        word1[i]=word1n[i]; word2[i]=word2n[i];
      }
      if (moving)
/* "moving" means that some previous relations have not been retained, and so
   we have to renumber words.
*/
      { word1[nnrels]=word1[i]; word2[nnrels]=word2[i];
        word1[i]=word2[i]=0;
      }
    }
    else
    { moving=1;
      tfree(word1[i]) tfree(word2[i])
      inew=i-nelim; nelim++;
      if (inew<=ninvrels) ninvrels--;
      j=0; while (1)
      { k=fpt[j];
        if (k==inew)
        { if (ret==inew) ret=j;
          if (lastrel==inew) lastrel=j;
          fpt[j]=fpt[k]; break;
        }
        j=k;
      }
      if (ret>inew) ret--; if (lastrel>inew) lastrel--;
      for (j=inew;j<=nrels-nelim;j++) fpt[j]=fpt[j+1];
      for (j=0;j<=nrels-nelim;j++) if (fpt[j]>inew) fpt[j]--;
    }
    nrels-=nelim;
  }
  if (moving || changed)
  { nstates=num_gens+1;
    for (i=1;i<=num_gens;i++) for (j=1;j<=nstates;j++) fsa[i][j]=i+1;
    for (i=1;i<=nrels;i++)  modifyfsa(i);
  }
  if (print_kb_rules)
  { fprintf(op,"\n\t# Knuth-Bendix equations after tidying:\n");
    for (i=1;i<=nrels;i++)
    { fprintf(op,"\t#%d.  ",i);
      wordprint(word1[i]);
      fprintf(op," > ");
      wordprint(word2[i]);
      fprintf(op,"\n");
    }
  }
  tfree(elim)
  tfree(word1n) tfree(word2n)
  tfree(ch)
  return(ret);
}

checkreduce(w,i) char *w; int i;
{ int st;  st=1;
  while (*w!='\0')
  { st=fsa[*(w++)][st];
    if (st<0 && st != -i) return(0);
  }
  return(1);
}

finddiff()
/* Computes word differences. */
{ int i,j,k,l,l1,l2,inspt,lasti,dno,lastdno,g1,g2;
  char c,new,*ch,*w1,*w2,changeorder,newdiff,pastcrel,*e1,*e2;
  ndiff=0;
  tfree(diffword[0])
  tmalloc(diffword[0],char,1)
  diffword[0][0]='\0';
  if (difftab[0]==0)
  { tmalloc(difftab[0],int *,num_gens+1);
    for (k=1;k<=num_gens;k++) tmalloc(difftab[0][k],int,num_gens+1)
  }
  for (k=1;k<=num_gens;k++) for (l=0;l<=num_gens;l++) difftab[0][k][l]= -1;
/* We definitely want transitions of the form (g,$) on the empty word,
since these are used by wa
*/
  for (j=1;j<=num_gens;j++)
  { test1[0]=inv_of[j]; test1[1]='\0';
    reduce(test1,0);
    dno=diffno();
    if (dno== -1)
    { ndiff++;
      tfree(diffword[ndiff])
      tmalloc(diffword[ndiff],char,strlen(test1)+1)
      strcpy(diffword[ndiff],test1);
      if (difftab[ndiff]==0)
      { tmalloc(difftab[ndiff],int *,num_gens+1);
        for (k=1;k<=num_gens;k++) tmalloc(difftab[ndiff][k],int,num_gens+1)
      }
      for (k=1;k<=num_gens;k++) for (l=0;l<=num_gens;l++)
           difftab[ndiff][k][l]= -1;
      dno=ndiff;
    }
    difftab[0][j][0]=dno;
  }

  changeorder=pastcrel=0;

  i=1;
  if (nrels>0) do
  { newdiff=0;
    w1=word1[i]; w2=word2[i]; l1=strlen(w1); l2=strlen(w2);
    lastdno=0;
    for (j=0;j<l1;j++)
    { g1=w1[j];
      g2= (j<l2) ? w2[j] : 0;
      if ((k=difftab[lastdno][g1][g2])!= -1) lastdno=k;
      else
      { *test1=inv_of[g1]; test1[1]='\0';
        strcat(test1,diffword[lastdno]);
        l=strlen(test1);
        if (j<l2) {test1[l]=g2; test1[l+1]='\0';}
        reduce(test1,0); 
        dno=diffno();
        if (dno== -1)
        { if (++ndiff>=MDIFF)
          { fprintf(stderr,"Too many differences.\n"); exit(2); }
          tfree(diffword[ndiff])
          tmalloc(diffword[ndiff],char,strlen(test1)+1)
          strcpy(diffword[ndiff],test1);
          newdiff=1;
          if (difftab[ndiff]==0)
          { tmalloc(difftab[ndiff],int *,num_gens+1);
            for (k=1;k<=num_gens;k++) tmalloc(difftab[ndiff][k],int,num_gens+1)
          }
          for (k=1;k<=num_gens;k++) for (l=0;l<=num_gens;l++)
               difftab[ndiff][k][l]= -1;
          dno=ndiff;
        }
        difftab[lastdno][g1][g2]=dno;
        lastdno=dno;
      }
    }
    if (newdiff==0 && changeorder==0 && pastcrel)
    { changeorder=1; inspt=lasti;}
    if (newdiff && changeorder)
    { 
/* printf("ins: %d %d %d\n",inspt,i,fpt[inspt]); */
      fpt[lasti]=fpt[i]; fpt[i]=fpt[inspt]; fpt[inspt]=i; inspt=i;
      if (i==lastrel) lastrel=lasti;
    }
    else lasti=i;
    if (i==crelno) pastcrel=1;
    i=fpt[i];
  } while (i!=0);

  return(0);
}
        
closeinv()
{ int i,j,k; char *ch,c,new;
  ondiff=ndiff;
  for (i=1;i<=ondiff;i++)
  { ch=test1; j=strlen(diffword[i]);
    for (k=j-1;k>=0;k--)
    { c=inv_of[diffword[i][k]];
      if (c==0)
      { printf("No inverse for %c.\n",diffword[i][k]); return(1);}
      *(ch++)=c;
    }
    *(ch++)=0;
    reduce(test1,0); 
    new=1;
    for (k=0;k<=ndiff;k++) if (strcmp(test1,diffword[k])==0) {new=0; break;}
    if (new)
    { if (++ndiff>=MDIFF)
      { fprintf(stderr,"Too many differences.\n"); return(1); }
      tfree(diffword[ndiff])
      tmalloc(diffword[ndiff],char,strlen(test1)+1)
      strcpy(diffword[ndiff],test1);
    }
  }

  fprintf(op,
          "\t#After tidying, there are %d equations and %d word differences,\n",
            nrels,ndiff);
  fprintf(op,"\t#and the fsa has %d states.\n",nstates);
  if (print_wdiffs)
  { fprintf(op,
         "\n\t#Word differences after tidying and closing under inversion:\n");
    for (i=1;i<=ndiff;i++)
    { fprintf(op,"\t#%d.  ",i);
      wordprint(diffword[i]);
      fprintf(op,"\n");
    }
  }
    
  return(0);
}

diffno()
{ int n,k;
  n= -1;
  for (k=0;k<=ndiff;k++) if (strcmp(test1,diffword[k])==0) {n=k; break;}
  return(n);
}

modifyfsa(relno) int relno;
/* Modify the FSA so that it returns the failure state -relno  whenever it
   encounters the LHS of the new relation with number relno as a substring.
   This is the trickiest part of the program.
*/
{ int genno,*cg,*cgp,*cge,*list1,*list2,*swlist,nnstates,x,y,z,i,*ip,*lp,*lpe;
  char *ptr;
  short *cvp,*cve,*swcf;
  ptr=word1[relno]; genno= *(ptr++);
  cg=cgp=fsa[genno]; cge=cgp+nstates;
  if (*ptr=='\0') while (++cgp<=cge) *cgp= -relno;
  else
  { list1=stl1; list2=stl2; *list1=0;
    cvp=cf1; cve=cf1+nstates; while (++cvp<=cve) *cvp=0;
    while (++cgp<=cge) if (*cgp>0 && cf1[*cgp]==0)
    { cf1[*cgp]=1; (*list1)++;
      if (*list1>=MSTLLEN)
      { fprintf(stderr,"Too many states in list.\n"); exit(1);}
      list1[*list1]= *cgp;
    }
    genno= *(ptr++);
    while (*ptr)
    { cg=cgp=fsa[genno]; cge=cgp+nstates; *list2=0; nnstates=0;
      cvp=cf2; cve=cf2+nstates; while (++cvp<=cve) *cvp=0;
      while (++cgp<=cge) if ((y= *cgp)>0)
      { x=cgp-cg;
        if (cf2[y]==0)
        { if (cf1[x]==1)
          { cf2[y]=1; (*list2)++;
            if (*list2>=MSTLLEN)
            { fprintf(stderr,"Too many states in list.\n"); exit(1);}
            list2[*list2]=y;
          }
          else cf2[y]= -1;
        }
        else if (cf2[y]==-1 && cf1[x]==1)
        { nnstates++; 
          if (nnstates>=127)
          { fprintf(stderr,"Too many new states.\n"); exit(1);}
          z=nstates+nnstates; if (z>mstates)
          { fprintf(stderr,"Too many states.\n"); exit(1);}
          for (i=1;i<=num_gens;i++) fsa[i][z]=fsa[i][y];
          cf2[z]=1; cf2[y]= nnstates+1; (*list2)++;
          if (*list2>=MSTLLEN)
          { fprintf(stderr,"Too many states in list.\n"); exit(1);}
          list2[*list2]=z;
        }
        else if (cf2[y]==1 && cf1[x]!=1)
        { nnstates++; 
          if (nnstates>=127)
          { fprintf(stderr,"Too many new states.\n"); exit(1);}
          z=nstates+nnstates; if (z>mstates)
          { fprintf(stderr,"Too many states.\n"); exit(1);}
          for (i=1;i<=num_gens;i++) fsa[i][z]=fsa[i][y];
          cf2[z]=1; cf2[y]= nnstates+1;
          for (i=1;i<= *list2;i++) if (list2[i]==y) { list2[i]=z; break;}
        }
      }
      lp=list1; lpe=lp+ *lp;
      while (++lp<=lpe)
      { ip=cg+ *lp;
        if (*ip>0 && (x=cf2[*ip]-1)>0) 
        {*ip=nstates+x;
          if ((y=cf2[*lp]-1)>0) cg[nstates+y]=nstates+x;
        }
      }
      nstates+=nnstates;
      swlist=list1; list1=list2; list2=swlist; swcf=cf1; cf1=cf2; cf2=swcf;
      genno= *(ptr++);
    }
    cg=fsa[genno];
    for (i=1;i<= *list1;i++) cg[list1[i]]= -relno;
  }
  return(1);
}


wordprint(w) char *w;
{ int i,l;
  l=strlen(w);
  for (i=0;i<=l-2;i++) {gen_print(op, (int)w[i]); putc('*',op);}
  gen_print(op, (int)w[l-1]);
}

stringtoint(w) char *w;
{ char *p; int n;
  p=w; n=0;
  while (*p!='\0')
  { if (isdigit(*p)==0) badusage();
    n=10*n+(*p-'0');
    p++;
  }
  return(n);
}

badusage()
{ printf("Usage:  diff1c [-r] [-w] [-t KBtidyint] [-n numreps]  [groupname]\n");
   exit(2);
}

readchar()
{ int n;
  n=getc(ip);
  if (n=='#') while ((n=getc(ip))!='\n');
  if (n==' ' || n=='\t' || n=='\n')
  { while ((n=getc(ip))==' ' || n=='\t' || n=='\n' || n=='#')
    { if (n=='#') while ((n=getc(ip))!='\n');}
    ungetc( (char) n,ip);
    n='\n';
  }
  return(n);
}

readstring(s) char *s;
{ int n;
  while ((n=readchar()) == '\n');
  while (n != '\n')
  { if (n== -1) {*s='\0'; return(-1);}
    *(s++)=n;
    n=readchar();
  }
  *s='\0';
  return(0);
}

output()
{ int i,j,k,l,m,ct,*ptr;
  fprintf(op,"fsa {\n");
  fprintf(op,"\tstates %d\n\tsymbols %d\n",ondiff+1,(num_gens+1)*(num_gens+1));
  fprintf(op,"\tvariables 2\n");
  fprintf(op,"\tbase_alphabet {");
  for (j=1;j<=num_gens;j++)
  { fprintf(op," %d = ",j); gen_print(op,j);}
  fprintf(op," %d = $",num_gens+1);
  fprintf(op,"}\n");
  fprintf(op,"\talphabet {");
  ct=0;
  for (j=1;j<=num_gens;j++)
  { for (k=1;k<=num_gens;k++)
    { ct++;
      fprintf(op,"%2d = (",ct); gen_print(op,j);
      fprintf(op,","); gen_print(op,k); fprintf(op,") ");
    }
    ct++;
    fprintf(op,"%2d = (",ct); gen_print(op,j);
    fprintf(op,",$)\n");
  }
  for (k=1;k<=num_gens;k++)
  { ct++;
    fprintf(op,"%2d = ($,",ct); gen_print(op,k);
    fprintf(op,") ");
  }
  ct++;
  fprintf(op,"%2d = ($,$)}\n",ct);
  fprintf(op,"\tstart { 1 }\n\n%%\nctable\n");

  for (i=0;i<=ondiff;i++)
  { char first=1;
    fprintf(op,"  %2d N",i+1);
    if (i<=ondiff) for (j=1;j<=num_gens;j++)
    { for (k=1;k<=num_gens;k++) if ((m=difftab[i][j][k])!= -1)
      { if (first) first=0; else fprintf(op,",");
        fprintf(op," %d > %d",(j-1)*(num_gens+1)+k,m+1);
      }
      if ((m=difftab[i][j][0])!= -1)
      { if (first) first=0; else fprintf(op,",");
        fprintf(op," %d > %d",j*(num_gens+1),m+1);
      }
    }
    fprintf(op,";\n");
  }
  fprintf(op,"}\n");

  fprintf(op,"inverses {\n");
  for (i=1;i<=num_gens;i++)
  { fprintf(op,"inv("); gen_print(op,i);
    fprintf(op,")="); gen_print(op,inv_of[i]); fprintf(op," ");
  }
  fprintf(op,"}\n");
  fprintf(op,"words {\nepsilon\n");
  for (i=1;i<=ndiff;i++) { wordprint(diffword[i]); fprintf(op,"\n");}
  fprintf(op,"}\n");
}
