#ident "%W%" %G%
 
/**************************************************************************
# Copyright (C) 1994 Kubota Graphics Corp.
# 
# Permission to use, copy, modify, and distribute this material for
# any purpose and without fee is hereby granted, provided that the
# above copyright notice and this permission notice appear in all
# copies, and that the name of Kubota Graphics not be used in
# advertising or publicity pertaining to this material.  Kubota
# Graphics Corporation MAKES NO REPRESENTATIONS ABOUT THE ACCURACY
# OR SUITABILITY OF THIS MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED
# "AS IS", WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
# PURPOSE AND KUBOTA GRAPHICS CORPORATION DISCLAIMS ALL WARRANTIES,
# EXPRESS OR IMPLIED.
**************************************************************************/
 
#include <dore.h>
#include <stdio.h>
#include <fcntl.h>
#include <poll.h>
#include <sys/stropts.h>

extern DtInt debug;

static int fd;

/*******************************************************************************
 **************************  o p e n _ d i a l s  ******************************
 *******************************************************************************
 * Module Name: open_dials
 * Module Type: c
 *     Purpose: Open the dial box device line (for reading and writing) and
 *		initialize the dial box attached to that line.
 *   Arguments: None
 * ReturnValue: File descriptor associated with the dial box or -1 if there
 *		doesn't appear to be a dial box attached to this machine.
 ******************************************************************************/
int open_dials()
{
/*******************************************************************************
 * Declare variables.
 ******************************************************************************/
	char c;
	static unsigned char c_reset[3] = { 0x1f, 0, 0 };
	static unsigned char c_srate[3] = { 0x90, 0, 0x1e };
	struct pollfd tmp_fds[1];

/*******************************************************************************
 * Open the device line attached to the dial box.
 ******************************************************************************/
	if ((fd = open("/dev/dials", O_RDWR))<0) {
		if ((fd = open("/dev/ttya", O_RDWR))<0) {
			fprintf(stderr,"open_dials: No /dev/dials or ");
			fprintf(stderr,"/dev/ttya\n");	
			return(-1);
		}
	}

/*******************************************************************************
 * Try to initialize the dial box.  If unable to communicate properly with 
 *  the device, assume it is not available and return.
 ******************************************************************************/
	if (write(fd,c_reset,3)!=3) {
		fprintf(stderr,"open_dials: Can't write to dials device.\n");
		return(-1);
	}


	/* Setup for polling the dialbox for input -- timeout after 2 seconds if
	 * no input is received from the device */
	tmp_fds[0].fd = fd;
	tmp_fds[0].events = POLLIN;
	poll(tmp_fds,1,2000);

	if (tmp_fds[0].revents != POLLIN) {
		fprintf(stderr,"No response from dial box -- assuming there ");
		fprintf(stderr,"isn't one.\n");
		return(-1);
	}

	if (read(fd,&c,1)!=1 || c != 0x5a) {
		fprintf(stderr,"open_dials: Wrong reply from dials after ");
		fprintf(stderr,"reset.\n");
		return(-1);
	}

	if (write(fd,c_srate,3)!=3) {
		fprintf(stderr,"Dialbox.c/open_dials: Can't initialize dials ");
		fprintf(stderr,"sample rate.\n");
		return(-1);
	}

/*******************************************************************************
 * Return the file descriptor associated with the dial box device.
 ******************************************************************************/
	return(fd);

}  /* End of open_dials function */



/*******************************************************************************
 **************************  r e a d _ d i a l s  ******************************
 *******************************************************************************
 * Module Name: read_dials
 * Module Type: c
 *     Purpose: Read from the dial box device until all of the queued up 
 *		data has been read.  Then return the results of the read.
 *   Arguments: offsets ==> An array that returns the delta movements (in
 *		   ticks) for all of the 8 knobs.
 * ReturnValue: The number of knobs that have moved.
 ******************************************************************************/
int read_dials(offsets)
int *offsets;
{
/*******************************************************************************
 * Declare and initialize variables.
 ******************************************************************************/
	int dummy,off,tokens;
	unsigned char dial;

	for (dial=0; dial<8; dial++)
		offsets[dial] = 0;

	tokens = 0;

/*******************************************************************************
 * Loop, reading the dial box device as long as there are messages waiting on
 *  the device queue.  Sum each of the 8 dial movements until there is no more
 *  data queued.
 ******************************************************************************/
	while(ioctl(fd, I_NREAD, &dummy) > 0) {
		if((off = read_dial(&dial))==0)
			break;
		if (dial>7)
			continue;
		offsets[dial] += off;
		tokens++;
	}

/*******************************************************************************
 * Return the number of dials that have moved. This value is primarily used
 *  by calling routines to check for a zero value -- no knob movement.
 ******************************************************************************/
	return(tokens);

}  /* End of read_dials function */



/*******************************************************************************
 ***************************  r e a d _ d i a l  *******************************
 *******************************************************************************
 * Module Name: read_dial
 * Module Type: c
 *     Purpose: Read the dial device line to check for movement of any one of 
 *		the 8 dials.
 *   Arguments: dial ==> Which dial moved.
 * ReturnValue: The amount (number of ticks) the dial has moved.
 ******************************************************************************/
int read_dial(dial)
unsigned char *dial;
{
/*******************************************************************************
 * Declare variables.
 ******************************************************************************/
	short delta;
	unsigned char ch;
	int dummy;

/*******************************************************************************
 * Try to read one byte from the dial device.  The first byte should be the
 *  dial number that moved.
 ******************************************************************************/
	read(fd, dial, 1);
	if ((*dial & 0xf0) != 0x80) {
		if (debug)
			printf("Dialbox.c/read_dial: Alignment error\n");
		while(ioctl(fd, I_NREAD, &dummy) > 0) {
			read(fd, dial, 1);
			if((*dial & 0xf0) == 0x80) goto ok;
		}
		return 0;

	}  /* end of if((*dial... */

/*******************************************************************************
 * Got the dial number byte, now read the two bytes indicating magnitude and
 *  direction of the dial movement.
 ******************************************************************************/
ok:
	read(fd, &ch, 1);
	delta = ch << 8;
	read(fd, &ch, 1);
	delta |= ch;
	*dial &= 7;
	return(delta);

}  /* End of read_dial function */

