/*ScianObjWindow.c
  Object window stuff
  Eric Pepke
  March 28, 1990
*/
#include "Scian.h"
#include "ScianTypes.h"
#include "ScianLists.h"
#include "ScianWindows.h"
#include "ScianColors.h"
#include "ScianIDs.h"
#include "ScianObjWindows.h"
#include "ScianErrors.h"
#include "ScianDraw.h"
#include "ScianControls.h"
#include "ScianSpaces.h"
#include "ScianArrays.h"
#include "ScianScripts.h"
#include "ScianVisWindows.h"
#include "ScianIcons.h"
#include "ScianEvents.h"

#ifdef PROTO
void ReshapeList(ThingListPtr, int, int, int, int, int, int, int, int);
#else
void ReshapeList();
#endif

int pauseDrawing = 0;
char tempStr[TEMPSTRSIZE + 1];
ObjPtr objWindowClass;			/*Object window class*/
ObjPtr dragBuffer = NULLOBJ;		/*Objects held*/

Bool Get2DIntBounds(object, l, r, b, t)
ObjPtr object;
int *l, *r, *b, *t;
/*Gets the 2D integer bounds of a screen object*/
{
    ObjPtr boundsArray;
    boundsArray = GetFixedArrayVar("Get2DIntBounds", object, BOUNDS, 1, 4L);
    if (boundsArray)
    {
	real temp[4];
	Array2CArray(temp, boundsArray);
	*l = temp[0];
	*r = temp[1];
	*b = temp[2];
	*t = temp[3];
	return true;
    }
    else
    {
	return false;
    }
}

Bool Set2DIntBounds(object, l, r, b, t)
ObjPtr object;
int l, r, b, t;
/*Sets the 2D integer bounds of a screen object to l, r, b, t*/
{
    ObjPtr boundsArray;
    boundsArray = NewRealArray(1, 4L);
    if (boundsArray)
    {
	real temp[4];
	temp[0] = l;
	temp[1] = r;
	temp[2] = b;
	temp[3] = t;
	CArray2Array(boundsArray, temp);
	SetVar(object, BOUNDS, boundsArray);
	return true;
    }
    else
    {
	return false;
    }
}

ObjPtr FindPanelContents(win)
WinInfoPtr win;
/*Finds the contents of a grey panel in win*/
{
    ObjPtr panel = 0;
    ObjPtr var;
    ThingListPtr contents;

    var = GetListVar("FindPanelContents", (ObjPtr) win, CONTENTS);
    if (!var) return NULLOBJ;
    contents = LISTOF(var);
    while (contents)
    {
	if (InClass(contents -> thing, greyPanelClass))
	{
	    panel = contents -> thing;
	    break;
	}
	contents = contents -> next;
    }

    if (panel)
    {
	return GetListVar("FindPanelContents", panel, CONTENTS);
    }
    else
    {
	return NULLOBJ;
    }
}

static ObjPtr CloseObjWindow(theInfo)
WinInfoPtr theInfo;
/*Closes obj window theInfo*/
{  
    if (theInfo -> mainMenu)
    {
	DeleteMenus(theInfo);
    }
    theInfo -> mainMenu = 0;
    return ObjTrue;
}

static ObjPtr DrawObjWindow(theInfo)
WinInfoPtr theInfo;
/*Draws the obj window theInfo*/
{
    ObjPtr contents;

    InitClipRect();

    /*Draw the objects within the window*/
    contents = GetVar((ObjPtr) theInfo, CONTENTS);
    if (contents)
    {
	if (IsList(contents))
	{
	    DrawList(contents);
	}
    }

    ResetClipRect();
    return ObjTrue;
}

Bool NeedsSBDrawing(object)
ObjPtr object;
/*Returns true iff an object needs to be drawn in single buffer mode*/
{
    ObjPtr picVar;

    /*See if this object itself needs to be drawn*/
    picVar = GetVar(object, PICSTATE);
    if (picVar)
    {
	int picState;
	picState = GetInt(picVar);

	/*See if this object needs to be drawn*/
	return picState & FG ? false : true;
    }
    return false;
}


Bool NeedsDBDrawing(object)
ObjPtr object;
/*Returns true iff an object needs to be drawn in double buffer mode*/
{
    ObjPtr picVar;
    
    /*See if this object itself needs to be drawn*/
    picVar = GetVar(object, PICSTATE);
    if (picVar)
    {
	int picState;
	picState = GetInt(picVar);

	/*See if this object needs to be drawn*/
	return picState & BG ? false : true;
    }
    return false;
}

ObjPtr MakeObjWindowDrawn(curWindow)
WinInfoPtr curWindow;
/*Method to make an obj window drawn.  Returns true iff drawing done*/
{
    ObjPtr retVal = ObjFalse;
    int picState;
    ObjPtr picVar;

    picVar = GetVar((ObjPtr) curWindow, PICSTATE);
    if (picVar)
    {
	picState = GetInt(picVar);
    }
    else
    {
	picState = FBBB;
    }    

    if (curWindow -> flags & WINDBUF)
    {
	/*Double buffer*/
	if (!(picState & BG))
	{
	    /*Back buffer is bad, have to redraw it*/
	    SetVar((ObjPtr) curWindow, PICSTATE, 
		   NewInt(picState & FG ? FGBG : FGBB));
	    SelectWindow(curWindow -> id);	/*MUST DO!*/
	    DrawWindowInfo(curWindow);
	    retVal = ObjTrue;
	}
	else
	{
	    /*There's still a chance that a subobject needs drawing*/
	    Bool drawAnything;
	    ObjPtr contents;

	    drawAnything = false;
	    
	    contents = GetVar((ObjPtr) curWindow, CONTENTS);
	    if (contents && IsList(contents))
	    {
		ThingListPtr runner;
		runner = LISTOF(contents);

		/*Check all contents for need to draw*/
		while (runner)
		{
		    ObjPtr drawCover;
		    if (NeedsDBDrawing(runner -> thing))
		    {
			drawAnything = true;
			SetVar(runner -> thing, DRAWME, NewInt(1));
			drawCover = runner -> thing;
			/*Draw its covers*/
			while(drawCover = GetVar(drawCover, DRAWCOVER))
			{
			    SetVar(drawCover, DRAWME, NewInt(1));
			}
		    }
		    drawCover = GetVar(runner -> thing, DRAWCOVER);
		    if (drawCover)
		    {
			if (NeedsDBDrawing(drawCover))
			{
			    drawAnything = true;
			    SetVar(runner -> thing, DRAWME, NewInt(1));
			}
		    }
		    runner = runner -> next;
		}
		if (drawAnything)
		{
		    /*Set everything that's not DRAWME to INHIBITDRAWING*/
		    runner = LISTOF(contents);
		    while (runner)
		    {
			if (GetPredicate(runner -> thing, DRAWME))
			{
			    SetVar(runner -> thing, DRAWME, NULLOBJ);
			}
			else
			{
			    SetVar(runner -> thing, INHIBITDRAWING, NewInt(1));
			}
			runner = runner -> next;
		    }

		    /*Draw what's not inhibited*/
		    SelectWindow(curWindow -> id);
		    DrawWindowInfo(curWindow);

		    /*Uninhibit drawing*/
		    runner = LISTOF(contents);
		    while (runner)
		    {
			SetVar(runner -> thing, INHIBITDRAWING, NULLOBJ);
			runner = runner -> next;
		    }
		    retVal = ObjTrue;
		}
	    }
	}
    }
    else
    {
	/*Single buffer*/
	if (!(picState & FG))
	{
	    SetVar((ObjPtr) curWindow, PICSTATE, NewInt(FG));
	    SelectWindow(curWindow -> id);
	    DrawWindowInfo(curWindow);
	    retVal = ObjTrue;
	}
	else
	{
	    /*There's still a chance that a subobject needs drawing*/
	    Bool drawAnything;
	    ObjPtr contents;

	    drawAnything = false;
	    
	    contents = GetVar((ObjPtr) curWindow, CONTENTS);
	    if (contents && IsList(contents))
	    {
		ThingListPtr runner;
		runner = LISTOF(contents);

		/*Check all contents for need to draw*/
		while (runner)
		{
		    if (NeedsSBDrawing(runner -> thing))
		    {
			drawAnything = true;
			SetVar(runner -> thing, DRAWME, NewInt(1));
		    }
		    runner = runner -> next;
		}
		if (drawAnything)
		{
		    /*Set everything that's not DRAWME to INHIBITDRAWING*/
		    runner = LISTOF(contents);
		    while (runner)
		    {
			if (GetPredicate(runner -> thing, DRAWME))
			{
			    SetVar(runner -> thing, DRAWME, NULLOBJ);
			}
			else
			{
			    SetVar(runner -> thing, INHIBITDRAWING, NewInt(1));
			}
			runner = runner -> next;
		    }

		    /*Draw what's not inhibited*/
		    SelectWindow(curWindow -> id);
		    DrawWindowInfo(curWindow);

		    /*Uninhibit drawing*/
		    runner = LISTOF(contents);
		    while (runner)
		    {
			SetVar(runner -> thing, INHIBITDRAWING, NULLOBJ);
			runner = runner -> next;
		    }
		    retVal = ObjTrue;
		}
	    }
	}
    }
    return retVal;
}

ObjPtr PressObjWindow(theInfo, flags)
WinInfoPtr theInfo;
long flags;
/*Does a press in object window theInfo*/
{
    ObjPtr contents;

    /*Press in the objects within the window*/
    contents = GetVar((ObjPtr) theInfo, CONTENTS);
    if (contents)
    {
	if (IsList(contents))
	{
	    int x, y;
	    Mouse(&x, &y);
	    return PressObject(contents, x, y, flags);
	}
    }
    return ObjTrue;
}

static ObjPtr DropObjWindow(window, object, x, y)
ObjPtr window, object;
int x, y;
/*Does a drop in object window theInfo*/
{
    ObjPtr contents;

    /*Press in the objects within the window*/
    contents = GetVar(window, CONTENTS);
    if (contents)
    {
	if (IsList(contents))
	{
	    return DropList(contents, object, x, y);
	}
    }
    return ObjTrue;
}


static ObjPtr KeyDownObjWindow(theInfo, theKey, flags)
WinInfoPtr theInfo;
short theKey;
int flags;
/*Does a key down in object window theInfo*/
{
    ObjPtr contents;

    /*Press in the objects within the window*/
    contents = GetVar((ObjPtr) theInfo, CONTENTS);
    if (contents)
    {
	if (IsList(contents))
	{
	    KeyDownList(contents, theKey, flags);
	}
    }
    return ObjTrue;
}

void ReshapeObject(object, ol, or, ob, ot, left, right, bottom, top)
ObjPtr object;
int ol, or, ob, ot;
int left, right, bottom, top;
/*Reshapes object, which used to exist within owner with edges ol, or, ob, ot
  to one which exists within owner with edges left, right, bottom, top.*/
{
    FuncTyp method;
    method = GetMethod(object, RESHAPE);
    if (method)
    {
	(*method)(object, ol, or, ob, ot, left, right, bottom, top);
    }
    else
    {
	ObjPtr boundsArray;
	ObjPtr stickyInt;
	real bounds[4];
	real oldWidth, oldHeight;	/*Old width and height*/
	Bool sideLocked[4];		/*True iff side is locked*/
	Bool xStretch, yStretch;	/*Stretchiness in x and y*/
	int stickiness;			/*Side stickiness of the object*/
	real oldBounds[4];		/*Old bounds of the object*/
	ObjPtr contents;		/*Contents of the object, if any*/
	real wr, hr;			/*Width and height ratios*/

	wr = ((real) (right - left)) / ((real) (or - ol));
	hr = ((real) (top - bottom)) / ((real) (ot - ob));

	boundsArray = GetVar(object, BOUNDS);
	if (!boundsArray || !IsArray(boundsArray) || RANK(boundsArray) != 1 ||
	    DIMS(boundsArray)[0] != 4)
	{
	    return;
	}
	Array2CArray(bounds, boundsArray);
	Array2CArray(oldBounds, boundsArray);
	oldWidth = bounds[1] - bounds[0];
	oldHeight = bounds[3] - bounds[2];

	/*Get the object's stickiness*/
	stickyInt = GetVar(object, STICKINESS);
	if (stickyInt && IsInt(stickyInt))
	{
	    stickiness = GetInt(stickyInt);
	}
	else
	{
	    stickiness = 0;
	}

	if ((stickiness & STICKYLEFT) || (stickiness & FLOATINGLEFT))
	{
	    if (stickiness & FLOATINGLEFT)
	    {
		bounds[0] = (bounds[0] - ol) * wr + left;
	    }
	    else
	    {
		bounds[0] += left - ol;
	    }
	    if (!((stickiness & STICKYRIGHT) || (stickiness & FLOATINGRIGHT)))
	    {
		bounds[1] = bounds[0] + oldWidth;
	    }
	}
	if ((stickiness & STICKYRIGHT) || (stickiness & FLOATINGRIGHT))
	{
	    if (stickiness & FLOATINGRIGHT)
	    {
		bounds[1] = (bounds[1] - ol) * wr + left;
	    }
	    else
	    {
		bounds[1] += right - or;
	    }
	    if (!((stickiness & STICKYLEFT) || (stickiness & FLOATINGLEFT)))
	    {
		bounds[0] = bounds[1] - oldWidth;
	    }
	}

	if ((stickiness & STICKYBOTTOM) || (stickiness & FLOATINGBOTTOM))
	{
	    if (stickiness & FLOATINGBOTTOM)
	    {
		bounds[2] = (bounds[2] - ob) * hr + bottom;
	    }
	    else
	    {
		bounds[2] += bottom - ob;
	    }
	    if (!((stickiness & STICKYTOP) || (stickiness & FLOATINGTOP)))
	    {
		bounds[3] = bounds[2] + oldHeight;
	    }
	}
	if ((stickiness & STICKYTOP) || (stickiness & FLOATINGTOP))
	{
	    if (stickiness & FLOATINGTOP)
	    {
		bounds[3] = (bounds[3] - ob) * hr + bottom;
	    }
	    else
	    {
		bounds[3] += top - ot;
	    }
	    if (!((stickiness & STICKYBOTTOM) || (stickiness & FLOATINGBOTTOM)))
	    {
		bounds[2] = bounds[3] - oldHeight;
	    }
	}

	/*We've got a new bounds, put it back*/
	boundsArray = NewRealArray(1, 4L);
	CArray2Array(boundsArray, bounds);
	SetVar(object, BOUNDS, boundsArray);

	/*If there are some contents to this, do the reshape recursively*/
	contents = GetVar(object, CONTENTS);
	if (contents && IsList(contents))
	{
	    ReshapeList(LISTOF(contents),
		    0, (int) (oldBounds[1] - oldBounds[0]),
		    0, (int) (oldBounds[3] - oldBounds[2]),		0, (int) (bounds[1] - bounds[0]),
		    0, (int) (bounds[3] - bounds[2]));
	}
    }
}

void ReshapeList(contents, ol, or, ob, ot, left, right, bottom, top)
ThingListPtr contents;
int ol, or, ob, ot;
int left, right, bottom, top;
/*Reshapes a list of objects using ReshapeObject*/
{
    while (contents)
    {
	ObjPtr object;

	object = contents -> thing;
	ReshapeObject(object, ol, or, ob, ot, left, right, bottom, top);
	    
	contents = contents -> next;
    }
}

ObjPtr ReshapeObjWindow(window, ol, or, ob, ot)
ObjPtr window;
int ol, or, ob, ot;
/*Reshapes object window given that ol, or, ob, and ot was the old
  viewport.  Does not redraw anything.*/
{
    ObjPtr contents;
    Screencoord l, r, b, t;

    getviewport(&l, &r, &b, &t);
    contents = GetVar(window, CONTENTS);
    if (contents && IsList(contents))
    {
	ReshapeList(LISTOF(contents),
		ol, or, ob, ot, l, r, b, t);
    }   
    return ObjTrue;
}

void ImInvalid(object)
ObjPtr object;
/*Alerts the window system that an object is no longer valid*/
{
    ObjPtr testObj;
    if (IsWindow(object))
    {
	/*It's already a window, just set its pic state*/
	SetVar(object, PICSTATE, NewInt(FBBB));
	return;
    }
    testObj = object;
    /*Go through parents until you hit an object at top level*/
    while (testObj)
    {
	ObjPtr list;
	object = testObj;
	list = GetVar(testObj, PARENTS);
	if (list)
	{
	    ThingListPtr runner;
	    runner = LISTOF(list);
	    while (runner)
	    {
		ImInvalid(runner -> thing);
		runner = runner -> next;
	    }
	    return;
	}
	testObj = GetVar(testObj, PARENT);
	if (testObj && IsWindow(testObj))
	{
	    /*Space or panel, change it.*/
	    SetVar(object, PICSTATE, NewInt(FBBB));
	    return;
	}
    }
/*
    Error("ImInvalid", NOPARENT, object);
*/
}

#ifdef PROTO
void PauseDrawing(Bool whether)
#else
void PauseDrawing(whether)
Bool whether;
#endif
/*Pauses drawing for a while*/
{
    if (whether)
    {
	++pauseDrawing;
    }
    else
    {
	--pauseDrawing;
    }
}

void DrawMe(object)
ObjPtr object;
/*Redraws object in the current window undergoing interaction*/
{
	ImInvalid(object);

	if (!pauseDrawing)
	{
	    drawMouse = true;
	}
}

void DoPickUpObjects()
/*Picks up the objects in the icon in the selected window*/
{
    ObjPtr corral, contents;
    ThingListPtr runner;

    if (!selWinInfo)
    {
	return;
    }
    corral = FindMainCorral(selWinInfo);
    if (!corral)
    {
	return;
    }
    contents = GetListVar("DoPickUpObjects", corral, CONTENTS);
    if (!contents)
    {
	return;
    }

    if (dragBuffer)
    {
	DeleteThing(dragBuffer);
	dragBuffer = NULLOBJ;
    }

    Log("drag");
    runner = LISTOF(contents);
    while (runner)
    {
	ObjPtr icon, locArray;
	real loc[2];
	icon = runner -> thing;
	if (GetPredicate(icon, SELECTED))
	{
	    char s[256];
	    if (!dragBuffer)
	    {
		iconXOff = iconYOff = 0;
		dragBuffer = NewList();
		AddToReferenceList(dragBuffer);
		MySetCursor(ICONCURSOR);
	    }
	    s[0] = ' ';
	    MakeObjectName(s + 1, icon);
	    Log(s);
	    PostfixList(dragBuffer, icon);
	}
	runner = runner -> next;
    }
    Log("\n");
}

#if 0
void DoDropObjects()
/*Drops the objects in the drag buffer*/
{
    FuncType method;
    if (!selWinInfo)
    {
	return;
    }
    if (!dragBuffer)
    {
	return;
    }
    method = GetMethod((ObjPtr) selWinInfo, DROPOBJECTS);
    if (method)
    {
	iconXOff = 0;
	iconYOff = 0;
	if (logging)
	{
	    char cmd[256];
	    sprintf(cmd, "drop %d %d\n", dropX, dropY);
	    Log(cmd);
	    InhibitLogging(true);
	}
	(*method)(selWinInfo, dragBuffer, dropX, dropY);
	DeleteThing(dragBuffer);
	dragBuffer = NULLOBJ;
	if (logging)
	{
	    InhibitLogging(false);
	}
    }
}
#endif

WinInfoPtr NewObjWindow(superClass, title, flags, minWidth, minHeight, maxWidth, maxHeight)
ObjPtr superClass;
char *title;
long flags;
int minWidth, minHeight, maxWidth, maxHeight;
/*Creates a new object window with title title and flags flags.  Window
  constraints must be set up correctly before this is called.  superClass is
  the class of the window, if NULLOBJ objWindowClass will be used*/
{
    WinInfoPtr retVal;
    WindowID theWindow;

    theWindow = NewOpenedWindow(title, minWidth, minHeight, maxWidth, maxHeight,flags);

    retVal = NewWinInfo(superClass ? superClass : objWindowClass, theWindow, 
		flags, title,  minWidth, minHeight, maxWidth, maxHeight);
    SetVar((ObjPtr) retVal, CONTENTS, NewList());

    DefineMenuItem((ObjPtr) retVal, OBJECTMENU, "Pick Up", DoPickUpObjects);

    return retVal;
}

static ObjPtr FindObjectObjClass(object, name)
ObjPtr object;
char *name;
/*Searches an object and its contents for an object with name*/
{
    ObjPtr retVal = NULLOBJ;
    ObjPtr objName, contents;

    /*First check to see if I am the object*/
    objName = GetVar(object, NAME);
    if (objName && IsString(objName) && 0 == strcmp2(GetString(objName), name))
    {
	if (!retVal)
	{
	    retVal = NewList();
	}
	PostfixList(retVal, object);
    }

    /*Now check my CONTENTS*/
    contents = GetVar(object, CONTENTS);
    if (contents)
    {
	ObjPtr foundObjects;
	foundObjects = FindNamedObject(contents, name);
	if (foundObjects)
	{
	    if (!retVal)
	    {
		retVal = NewList();
	    }
	    AppendList(retVal, foundObjects);
	}
    }

    return retVal;
}

extern ObjPtr objClass;

void InitObjWindows()
/*Initializes the obj window system*/
{
    objWindowClass = NewObject(windowClass, 0);
    AddToReferenceList(objWindowClass);
    SetMethod(objWindowClass, CLOSE, CloseObjWindow);
    SetMethod(objWindowClass, DRAW, DrawObjWindow);
    SetMethod(objWindowClass, PRESS, PressObjWindow);
    SetMethod(objWindowClass, DROPOBJECTS, DropObjWindow);
    SetMethod(objWindowClass, KEYDOWN, KeyDownObjWindow);
    SetMethod(objWindowClass, RESHAPE, ReshapeObjWindow);
    SetMethod(objWindowClass, MAKEDRAWN, MakeObjWindowDrawn);

    /*KLUDGE*/
    SetMethod(objClass, FINDOBJECT, FindObjectObjClass);
}

void DoDelete()
/*Deletes objects*/
{
    ObjPtr corral;
    Bool didit = false;

    if (logging)
    {
	InhibitLogging(true);
    }
    if (selWindow)
    {
	corral = FindMainCorral(selWinInfo);
	if (corral)
	{
	    ThingListPtr contents;
	    ObjPtr contentsList;
	    contentsList = GetVar(corral, CONTENTS);
	    contents = LISTOF(contentsList);
	    while (contents)
	    {
		ObjPtr icon;
		icon = contents -> thing;
		if (IsObject(icon) &&
		    InClass(icon, iconClass))
		{
		    if (GetPredicate(icon, SELECTED))
		    {
			    FuncTyp method;
			    method = GetMethod(icon, DELETEICON);
			    if (method)
			    {
				if (IsTrue((*method)(icon)))
				{
				    contents = contents -> next;
				    DeleteFromList(contentsList, icon);
				    ImInvalid(icon);
				    didit = true;
				    continue;
				}
			    }
		    }
		}
		contents = contents -> next;
	    }
	}
	if (!didit)
	{
	    /*Search for something else to delete*/
	    ObjPtr current, parent, contents;
	    FuncTyp method;
	    current = GetVar((ObjPtr) selWinInfo, CURRENT);
	    if (current)
	    {
		parent = GetVar(current, PARENT);
		if (parent && InClass(parent, panelClass) &&
			(method = GetMethod(current, DELETEICON)))
		{
		    if (IsTrue((*method)(current)))
		    {
			contents = GetListVar("DoDelete", parent, CONTENTS);
			if (contents)
			{
			    MakeMeCurrent(NULLOBJ);
			    DeleteFromList(contents, current);
			    ImInvalid(parent);
			}
		    }		    
		}
	    }
	}
	else
	{
	    MakeMeCurrent(NULLOBJ);
	}
    }
    if (logging)
    {
	InhibitLogging(false);
	if (didit)
	{
	    Log("delete\n");
	}
    }
}

void KillObjWindows()
/*Kills the obj window system*/
{
    DeleteThing(objWindowClass);
}

ObjPtr FindNamedObject(object, name)
ObjPtr object;
char *name;
/*Finds object with name within object.  Returns a list of matching objects*/
{
    FuncTyp method;
    method = GetMethod(object, FINDOBJECT);
    if (method)
    {
	return (*method)(object, name);
    }
    else
    {
	return NULLOBJ;
    }
}

void MakeMeCurrent(object)
ObjPtr object;
/*Makes object the current object in the local window*/
{
    if (selWinInfo)
    {
	ObjPtr oldCurrent;
	oldCurrent = GetVar((ObjPtr) selWinInfo, CURRENT);
        if (oldCurrent == object) return;
	FlushKeystrokes();
	SetVar((ObjPtr) selWinInfo, CURRENT, object);
	if (oldCurrent)
	{
	    FuncTyp method;
	    method = GetMethod(oldCurrent, YOURENOTCURRENT);
	    if (method)
	    {
		Log("#You're not current\n");
		(*method)(oldCurrent);
		Log("#After you're not current\n");
	    }
	    ImInvalid(oldCurrent);
	}
	if (object) ImInvalid(object);
    }
}

Bool AmICurrent(object)
ObjPtr object;
/*Returns true iff object is the current object in the current window*/
{
    if (selWinInfo)
    {
	return GetVar((ObjPtr) selWinInfo, CURRENT) == object ? true : false;
    }
}
