/*ScianHelp.c
  Eric Pepke
  3 September 1991
  On-line help system for SciAn
*/

#include "Scian.h"
#include "ScianTypes.h"
#include "ScianMethods.h"
#include "ScianLists.h"
#include "ScianIDs.h"
#include "ScianStyle.h"
#include "ScianWindows.h"
#include "ScianObjWindows.h"
#include "ScianEvents.h"
#include "ScianDialogs.h"
#include "ScianControls.h"
#include "ScianButtons.h"
#include "ScianArrays.h"
#include "ScianTextBoxes.h"
#include "ScianErrors.h"
#include "ScianPreferences.h"
#include "ScianScripts.h"
#include "ScianColors.h"

/*Topics for help*/
#define TOPIC_INTRO		0
#define TOPIC_COPYRIGHT		1
#define TOPIC_CREDITS		2
#define TOPIC_BUGS		3
#define TOPIC_REVISION		4
#define TOPIC_MOUSE		5
#define TOPIC_MODIFIERS		6
#define TOPIC_MENUS		7
#define TOPIC_CONTEXT		8
#define NTOPICS			9

#define ISVOWEL(c) ((c) == 'a' || (c) == 'A' || (c) == 'e' || (c) == 'E' || (c) == 'i' || (c) == 'I' || (c) == 'o' || (c) == 'O' || (c) == 'u' || (c) == 'U')

typedef struct
    {
	Bool indent;		/*True iff indent*/
	char *buttonName;	/*Name of the button*/
	char *topicName;	/*Name of the topic*/
	char *topicString;	/*String of the topic*/
    } HelpInfo;

HelpInfo helpInfo[NTOPICS] =
    {
	{false, "Introduction", "Intro", 
	 "Welcome to SciAn, a scientific visualization and animation package \
developed at the Supercomputer Computations Research Institute (SCRI) at Florida \
State University.\n\nThis on-line help system is intended to help you get started \
using SciAn.  To see help on a particular topic, click with the left mouse \
button on the radio button on the right showing the name of the topic.\n\
\n\
If there is more help text than can be shown on the screen, you can move up and \
down through the text using the scroll bar on the right."},
	{false, "Copyright", "Copy",
	 "SciAn is Copyright \201 1991, 1992 by the Florida State University.  \
All rights reserved.\n\
\n\
You may distribute the source and binary of this program provided that\n\
1) All distributions retain this entire copyright notice\n\
2) All documentation and advertising mentioning features or the use of this \
software include the acknowlegement, \"This product includes software developed \
by the Supercomputer Computations Research Institute at Florida State University \
and its contributors.\"\n\
\n\
You may not use the name of the University, the Institute, or any of its contributors \
to endorse or promote products derived from this software without specific prior written permission.\n\
\n\
If you use any images produced by SciAn in papers, conference proceedings, etc. we \
also ask you to acknowledge the Supercomputer Computations Research Institute \
and Florida State University.  We have worked very hard on SciAn and will work \
very hard to keep it free to benefit the entire research community, and we think \
we should get some credit."},
	{false, "Credits", "Cred",
#ifdef HDFDEF
	 "SciAn was written by Eric Pepke, John Murray, Jim Lyons, and Doug Lee.  \
The SciAn logo and T-shirt were designed by Dave Poindexter.\n\
\n\
SciAn is dedicated to Mike Mermikides, long advocate of interactive computer\n\
graphics and pioneer in the use of socket communication.\n\
\n\
Many people have contributed in many ways to the SciAn project.  Among them are \
Ken Johnson, Dennis Duke, Hwa Lim, Andy Hasenfeld, Saul Youssef, Jerry Magnan, \
P.K. Jayakumar, Sergiu Sanielevici, Gregor Bali, Dave Kopriva, Taekyu Reu, Susan X. \
Ying, Jim Carr, Mike Mermikides, Sam Adams, Dave Poindexter, Jim O'Brien, Mark Luther, \
John McCalpin, and Al Davis.\n\
\n\
Partially funded by the U.S. Department of Energy under contract DE-FC05-85ER250000.\n\
Supported by the State of Florida.\n\
\n\
Portions developed at the National Center for Supercomputing Applications at \
the University of Illinois at Urbana-Champaign."},
#else
	 "SciAn was written by Eric Pepke, John Murray, Jim Lyons, and Doug Lee.  \
The SciAn logo and T-shirt were designed by Dave Poindexter.\n\
\n\
SciAn is dedicated to Mike Mermikides, long advocate of interactive computer\n\
graphics and pioneer in the use of socket communication.\n\
\n\
Many people have contributed in many ways to the SciAn project.  Among them are \
Ken Johnson, Dennis Duke, Hwa Lim, Andy Hasenfeld, Saul Youssef, Jerry Magnan, \
P.K. Jayakumar, Sergiu Sanielevici, Gregor Bali, Dave Kopriva, Taekyu Reu, Susan X. \
Ying, Jim Carr, Mike Mermikides, Sam Adams, Dave Poindexter, Jim O'Brien, Mark Luther, \
John McCalpin, and Al Davis.\n\
\n\
Partially funded by the U.S. Department of Energy under contract DE-FC05-85ER250000.\n\
Supported by the State of Florida"},
#endif
	{false, "Revision History", "Revision",
	 "0.42\t\t1/25/92\t\t\tFixed some script problems, problem with time-dependent files and min/max, removed file extensions from dataset names\n\
0.41\t\t1/5/92\t\t\tImproved color editor, fixed geometric object and NFF bugs\n\
0.40\t\t11/15/91\t\t\tFirst beta release\n\
"},
	{false, "Reporting Bugs", "Bugs",
	 "We appreciate your help in beta testing SciAn.  Unfortunately, you will \
most likely find some bugs in the program.\n\n\
If you find a bug, first check to see if it is listed in the list of known bugs (if you have been \
given such a list, that is).  If it is not on the list, try to replicate the bug while \
logging to a file from within SciAn (using the -l filename switch on the command line).  Some bugs will \
produce error messages to the standard error device, and these error messages will also \
appear in the log.  If the bug results in a core dump, keep the core file too.\n\n\
Report the bug via electronic mail to scian-bugs@scri.fsu.edu.  You don't \
need to include the core file when you first report a bug, but we may ask you to send it \
along later.\n\
\n\
If you are not on the network or do not have access to electronic mail, call us and we \
will find some other way for you to report bugs."},
	{false, "Using the Mouse", "Mouse",
	 "Most of the interaction with SciAn uses the three-button mouse directly \
to manipulate objects, such as icons and controls.  Some actions are performed through menus.\n\
\n\
The left mouse button is used to select and drag objects, press and slide controls, and select text.  Double-clicking \
on an object such as an icon will usually perform the most common operation on that object.  \
See Help In Context to determine the double-click action for any given object.\n\
\n\
The center mouse button is used to rotate within visualization spaces.  Double-clicking \
with the center mouse button will snap the space to the closest orthogonal orientation.\n\
\n\
The right mouse button is used to bring up the main menu.  Move the mouse to any SciAn window \
and press and hold the right button."
	},
	{false, "Modifier Keys", "Modifiers",
	 "Some keys on the keyboard modify the meaning of mouse actions if you \
hold them down while performing the action.\n\
\n\
The Shift key usually constrains motion in convenient ways.  Try it when dragging \
on spaces, controls, and text box handles.  When used on icons in an icon corral, the shift \
key allows a selection to be extended.\n\
\n\
The Alt key or Option key provides an optional way of doing some actions.  For \
example, if you hold down the Alt key when you click on the Visualize button, \
the visualizations will come up turned off instead of turned on.\n\
\n\
See Help In Context to determine what each modifier key will do with a specific \
control."},
	{false, "Menus", "Menu",
	 "Pressing and holding the right mouse button within a window \
brings up the main menu for that window, which contains individual items and submenus.  The main \
menu and its submenus are specific to the window and may vary depending on which objects \
within the window are selected.\n\
\n\
The Help item brings up this help window.  It is always active.\n\
\n\
The Windows submenu contains actions that can be performed on a window, such as resizing \
and tiling.  It also contains actions to show specific windows and bring them to the \
front.\n\
\n\
The Object submenu contains common actions that can be performed on objects within a window, \
such as Delete and Duplicate.\n\
\n\
The Text submenu contains commands to change the font and size of text boxes as well \
as commands to create new text annotations in space panels.\n\
\n\
The Quit SciAn item causes SciAn to quit."
	},
	{false, "Help In Context", "CSH",
#ifdef CURSORS4D
"Notice that your mouse pointer has changed to look like a question mark.  \
Click now within any control, icon, or window to find out information \
about it.  If you decide that you don't really want information about any \
particular control, you can click on this text box to return your mouse pointer \
to normal."
#else
"Click now within any control, icon, or window to find out information \
about it.  If you decide that you don't really want information about any \
particular control, you can click on this text box to return your mouse pointer \
to normal."
#endif
}
    };

static ObjPtr ChangeHelpRadio(radio)
ObjPtr radio;
/*ChangeValue as a result of a click in the help radio*/
{
    ObjPtr repObj;
    ObjPtr value;
    ObjPtr scrollbar;
    ObjPtr contents;
    int k;

    repObj = GetObjectVar("ChangeHelpRadio", radio, REPOBJ);
    if (!repObj)
    {
	return ObjFalse;
    }

    value = GetValue(radio);
    if (!value)
    {
	return ObjFalse;
    }

    k = GetInt(value);

    if (k == TOPIC_CONTEXT)
    {
	SetContextSensitiveHelp();
    }

    contents = GetVar(radio, HELPCONTENTS);
    if (!contents)
    {
	return ObjFalse;
    }

    if (k >= 0 || k < DIMS(contents)[0])
    {
	/*There's a predefined string*/
	ObjPtr *elements;
	elements = ELEMENTS(contents);
	SetVar(repObj, CONTENTS, elements[k]);
    }
    
    scrollbar = GetVar(repObj, VSCROLL);
    if (scrollbar)
    {
	SetSliderValue(scrollbar, 0.0);
	RecalcScroll(repObj);
    }
    ImInvalid(repObj);
    return ObjTrue;
}

void DoShowHelp()
/*Shows the help window*/
{
    WinInfoPtr dialogExists, newDialog;
    ObjPtr whichDialog;

    Log("show help");
    InhibitLogging(true);

    whichDialog = NewString("Help");

    dialogExists = DialogExists(0, whichDialog);
    newDialog = GetDialog(0, whichDialog, "Help",
	HWWIDTH,
	HWHEIGHT,
	HWWIDTH,
	HWHEIGHT,
	WINDBUF + WINFIXEDSIZE);

    if (!dialogExists)
    {
	/*Fill the dialog with stuff*/
	ObjPtr contents, panel, field, button, radio, textBox, helpContents;
	int titleHeight;
	ObjPtr *elements;
	int k, top;
	long dim;

	SetVar((ObjPtr) newDialog, HELPSTRING,
	    NewString("This window shows the on-line help for SciAn.  For more \
information on a topic, click on its name."));

	contents = GetListVar("DoShowHelp", (ObjPtr) newDialog, CONTENTS);
	if (!contents) return;
	panel = NewPanel(greyPanelClass, 
	    0, HWWIDTH, 
	    0, HWHEIGHT);
	if (!panel)
	{
	    return;
	}
	PrefixList(contents, panel);
	SetVar(panel, PARENT, (ObjPtr) newDialog);

	contents = GetListVar("DoShowHelp", panel, CONTENTS);
	if (!contents) return;

	/*Make all the radio buttons*/
	radio = NewRadioButtonGroup("Topics");
	SetVar(radio, HELPSTRING,
	    NewString("These radio buttons show the names of topics on which you can get help.")); 

	dim = NTOPICS;
	helpContents = NewArray(AT_OBJECT, 1, &dim);
	if (!helpContents)
	{
	    return;
	}
	elements = ELEMENTS(helpContents);
	SetVar(radio, HELPCONTENTS, helpContents);

	field = NewControlField(MINORBORDER, HWWIDTH - 2 * MAJORBORDER - HWRADIOWIDTH,
			 MINORBORDER, HWHEIGHT - MINORBORDER,
			 "Help text", OBJECTSFROMTOP | BARRIGHT);	
	if (!field)
	{
	    return;
	}
	PrefixList(contents, field);
	SetVar(field, PARENT, panel);
	SetVar(field, BACKGROUND, NewInt(UIGRAY75));
	SetVar(field, BORDERTYPE, NewInt(1));

	top = HWHEIGHT - MAJORBORDER;
	for (k = 0; k < NTOPICS; ++k)
	{
	    elements[k] = NewList();
	    if (!elements[k])
	    {
		return;
	    }
	    switch(k)
	    {
		default:
		    /*Do a title text box*/
		    textBox = NewTextBox(MINORBORDER, HWWIDTH - MINORBORDER - 3 * MAJORBORDER - HWRADIOWIDTH - BARWIDTH,
					 -MINORBORDER - HWWIDTH,
					 -MINORBORDER,
					 0, "Title", helpInfo[k] . buttonName);
		    SetVar(textBox, TEXTFONT, NewString(HELPFONT));
		    SetVar(textBox, TEXTSIZE, NewInt(HELPTITLESIZE));
		    SetVar(textBox, PARENT, field);
		    titleHeight = TextHeight(textBox);
		    Set2DIntBounds(textBox,
			MINORBORDER, HWWIDTH - MINORBORDER - 3 * MAJORBORDER - HWRADIOWIDTH - BARWIDTH,
			-MINORBORDER - titleHeight,
			-MINORBORDER);
		    PrefixList(elements[k], textBox);
		
		    textBox = NewTextBox(MINORBORDER, HWWIDTH - MINORBORDER - 3 * MAJORBORDER - HWRADIOWIDTH - BARWIDTH,
					 -MINORBORDER - HWWIDTH - titleHeight,
					 -MINORBORDER - titleHeight,
					 0, helpInfo[k] . topicName, helpInfo[k] . topicString);
		    SetVar(textBox, TEXTFONT, NewString(HELPFONT));
		    SetVar(textBox, TEXTSIZE, NewInt(HELPSIZE));
		    SetVar(textBox, PARENT, field);
		    Set2DIntBounds(textBox,
			MINORBORDER, HWWIDTH - MINORBORDER - 3 * MAJORBORDER - HWRADIOWIDTH - BARWIDTH,
			-MINORBORDER - titleHeight - TextHeight(textBox),
			-MINORBORDER - titleHeight);
		    PrefixList(elements[k], textBox);
	    }
	    button = NewRadioButton(HWWIDTH - MAJORBORDER - HWRADIOWIDTH,
				    HWWIDTH - MAJORBORDER,
				    top - CHECKBOXHEIGHT,
				    top,
				    helpInfo[k] . buttonName);
	    AddRadioButton(radio, button);
	    top -= CHECKBOXHEIGHT + CHECKBOXSPACING;
	}

	SetVar(radio, PARENT, panel);
	SetMethod(radio, CHANGEDVALUE, ChangeHelpRadio);
	PrefixList(contents, radio);


	SetVar(radio, REPOBJ, field);
	SetVar(radio, HALTHELP, ObjTrue);

	/*Finally, set the radio button*/
	SetValue(radio, NewInt(0));
	SetVar((ObjPtr) newDialog, HELPRADIO, radio);
    }
    InhibitLogging(false);
}

static ObjPtr helpObject;

void DoContextHelp()
/*Gives context sensitive help for screen object helpObject*/
{
    WinInfoPtr theDialog;
    ObjPtr whichDialog;

    DoShowHelp();
    whichDialog = NewString("Help");

    theDialog = DialogExists(0, whichDialog);
    if (theDialog)
    {
	FuncTyp method;
	ObjPtr radio, repObj, scrollbar;
	ObjPtr textBox, contents;
	int top, left, right, bottom;
	char *oldType = 0;
	char objectString[400];
	char *s;

	SelectWindow(theDialog -> id);

	radio = GetObjectVar("ContextHelp", (ObjPtr) theDialog, HELPRADIO);
	if (!radio) return;

	method = GetMethod(radio, CHANGEDVALUE);
	SetMethod(radio, CHANGEDVALUE, (FuncTyp) 0);
	SetValue(radio, NewInt(TOPIC_CONTEXT));
	SetMethod(radio, CHANGEDVALUE, method);

	repObj = GetObjectVar("ContextHelp", radio, REPOBJ);
	if (!repObj) return;

 	scrollbar = GetVar(repObj, VSCROLL);
	if (scrollbar)
	{
	    SetSliderValue(scrollbar, 0.0);
	    RecalcScroll(repObj);
	}

	top = -MINORBORDER;
	left = MINORBORDER;
	right = HWWIDTH - MINORBORDER - 3 * MAJORBORDER - HWRADIOWIDTH - BARWIDTH;

	contents = NewList();

	/*Now build up the helpObject*/
	if (helpObject == repObj)
	{
	    /*It's a click in the text box*/
	    s = "Your mouse pointer has been returned to normal.  If you want to get \
help in context for any other object, click on \"Help In Context\" once more.";
	    bottom = top -
		(HELPSIZE + DEFLINESPACE) * (1 + LineCount(s, HWWIDTH - 2 * MINORBORDER - 3 * MAJORBORDER - HWRADIOWIDTH - BARWIDTH));

	    textBox = NewTextBox(left, right, bottom, top, 0, "info", s);
	    SetVar(textBox, TEXTFONT, NewString(HELPFONT));
	    SetVar(textBox, TEXTSIZE, NewInt(HELPSIZE));
	    PrefixList(contents, textBox);
	}
	else while (helpObject)
	{
	    ObjPtr var;
	    ObjPtr class;
	    Bool firstTime = true;

	    /*Get help for all the classes*/
	    class = helpObject;
	    while (class)
	    {
		FuncTyp method;
		method = Get1Method(class, MAKE1HELPSTRING);
		if (method)
		{
		    (*method)(helpObject, class);
		}
		var = Get1Var(class, HELPSTRING);
		if (var)
		{
		    if (firstTime)
		    {
			ObjPtr var;
			char *name, *type;

			/*Print out the introduction*/
			firstTime = false;

			/*Get the type and name strings*/
			var = GetVar(helpObject, NAME);
			if (var)
			{
			    name = GetString(var);
			}
			else
			{
			    name = 0;
			}
			
			var = GetVar(helpObject, TYPESTRING);
			if (var)
			{
			    type = GetString(var);
			}
			else
			{
			    type = 0;
			}
			if (name || type)
			{
			    /*Print out help for this object*/
			    if (!type)
			    {
				type = "object";
			    }
			    if (name && (0 == strcmp2(type, name)))
			    {
				name = 0;
			    }

			    /*Generate the object string*/
			    s = objectString;
			    if (oldType)
			    {
				sprintf(s,
				    "The %s is located within a%s %s",
				    oldType,
				    ISVOWEL(*type) ? "n" : "",
				    type);
				while (*s) ++s;
				if (name)
				{
				    sprintf(s, " named \"%s.\"", name);
				}
				else
				{
				    sprintf(s, ".");
				}
				bottom = top - 37;

				textBox = NewTextBox(left, right, bottom, top, 0, "info", objectString);
				SetVar(textBox, TEXTFONT, NewString(HELPFONT));
				SetVar(textBox, TEXTSIZE, NewInt(HELPSIZE));
				bottom = top - TextHeight(textBox);
				Set2DIntBounds(textBox, left, right, bottom, top);
				PrefixList(contents, textBox);
			    }
			    else
			    {
				sprintf(s,
				    "Help for a%s %s",
				    ISVOWEL(*type) ? "n" : "",
				    type);
				while (*s) ++s;
				    if (name)
				{
				    sprintf(s, " named \"%s.\"", name);
				}
				else
				{
				    sprintf(s, ".");
				}

				textBox = NewTextBox(left, right, bottom, top, 0, "info", objectString);
				SetVar(textBox, TEXTFONT, NewString(HELPFONT));
				SetVar(textBox, TEXTSIZE, NewInt(HELPTITLESIZE));
				bottom = top - TextHeight(textBox);
				Set2DIntBounds(textBox, left, right, bottom, top);
				PrefixList(contents, textBox);
			    }

			    oldType = type;
			    top = bottom;
			}

		    }

		    s = GetString(var);
		    bottom = top - 37;

		    textBox = NewTextBox(left, right, bottom, top, 0, "info", s);
		    SetVar(textBox, TEXTFONT, NewString(HELPFONT));
		    SetVar(textBox, TEXTSIZE, NewInt(HELPSIZE));
		    bottom = top - TextHeight(textBox);
		    Set2DIntBounds(textBox, left, right, bottom, top);
		    PrefixList(contents, textBox);

		    top = bottom;

		}
		class = ClassOf(class);
	    }

	    if (GetPredicate(helpObject, HALTHELP))
	    {
		helpObject = NULLOBJ;
	    }
	    else
	    {
		helpObject = GetVar(helpObject, PARENT);
	    }
	}

	SetVar(repObj, CONTENTS, contents);
	RecalcScroll(repObj);
	ImInvalid(repObj);
    }
}

static char *ObjTypeStr(object)
ObjPtr object;
/*Returns a temporary type string from flags*/
{
    if (!object)
    {
	return "NULLOBJ";
    }
    switch(object -> flags)
    {
    	case ISOBJECT:
	    return "Object";
	    break;
	case REALARRAY:
	    return "Real array";
	    break;
	case STRING:
	    return "String";
	    break;
	case INTEGER:
	    return "Integer";
	    break;
	case REAL:
	    return "Real number";
	    break;
	case LIST:
	    return "List";
	    break;
	case PALETTE:
	    return "Palette";
	    break;
	case MATRIX:
	    return "Matrix";
	    break;
	case WINDOW:
	    return "Window";
	    break;
	case PICTURE:
	    return "Picture";
	    break;
	case OBJECTARRAY:
	    return "Object array";
	    break;
    }
}

static char *ObjValueStr(object)
ObjPtr object;
/*Returns a temporary value string from flags*/
{
    if (!object)
    {
	return "";
    }
    switch(object -> flags)
    {
	case STRING:
	    if (strlen(GetString(object)) < 20)
	    {
		sprintf(tempStr, "\"%s\"", GetString(object));
	    }
	    else
	    {
		tempStr[0] = '"';
		strncpy(&(tempStr[1]), GetString(object), 20);
		strcpy(&(tempStr[50]), "\"...");
	    }
	    return tempStr;
	    break;
	case INTEGER:
	    sprintf(tempStr, "%d", GetInt(object));
	    return tempStr;
	    break;
	case REAL:
	    sprintf(tempStr, "%g", GetReal(object));
	    return tempStr;
	    break;
	default:
	    sprintf(tempStr, "%lx", object);
	    return tempStr;
	    break;
    }
}

static ObjPtr objectToSniff;
void SniffObject(void);

static ObjPtr SniffRepobj(button)
ObjPtr button;
/*Sniffs a button's repobj*/
{
    ObjPtr repObj;
    repObj = GetVar(button, REPOBJ);
    if (repObj)
    {
	objectToSniff = repObj;
	DoUniqueTask(SniffObject);
    }
    return ObjTrue;
}

static int StuffVarControls(vars, contents, top)
VarsPtr vars;
ObjPtr contents;
int top;
/*Stuffs controls for var in contents from top, returns new top*/
{
    ObjPtr object;
    ObjPtr textBox, button;
    int mid, bottom, left, right;
    if (!vars) return top;

    top = StuffVarControls(vars -> left, contents, top);

    bottom = top - BUTTONHEIGHT;
    mid = (bottom + top) / 2;

    /*Create the ID text box*/
    left = MINORBORDER;
    right = left + SNIFFIDWIDTH;
    textBox = NewTextBox(left, right,
			  mid - TEXTBOXHEIGHT / 2, mid + TEXTBOXHEIGHT / 2,
			  0, "ha!", IDName(vars -> name));
    SetVar(textBox, PARENT, contents);
    PrefixList(contents, textBox);
    SetVar(textBox, TEXTFONT, NewString("Helvetica-Bold"));
    left = right + MINORBORDER;

    object = vars -> value;

    /*Create the type box*/
    right = left + SNIFFTYPEWIDTH;
    textBox = NewTextBox(left, right,
			  mid - TEXTBOXHEIGHT / 2, mid + TEXTBOXHEIGHT / 2,
			  0, "ha!", ObjTypeStr(object));
    SetVar(textBox, PARENT, contents);
    PrefixList(contents, textBox);
    SetVar(textBox, TEXTFONT, NewString("Helvetica-Bold"));
    left = right + MINORBORDER;

    /*Create the value box*/
    right = left + SNIFFVALWIDTH;
    textBox = NewTextBox(left, right,
			  mid - TEXTBOXHEIGHT / 2, mid + TEXTBOXHEIGHT / 2,
			  0, "ha!", ObjValueStr(object));
    SetVar(textBox, PARENT, contents);
    PrefixList(contents, textBox);
    SetVar(textBox, TEXTFONT, NewString("Helvetica-Bold"));
    SetVar(textBox, REPOBJ, object);
    SetMethod(textBox, DOUBLECLICK, SniffRepobj);
    left = right + MINORBORDER;

    /*Create the show button box*/
    if (object)
    {
	right = left + SNIFFSHOWWIDTH;
	button = NewButton(left, right,
			bottom, top, "Show");
	SetVar(button, REPOBJ, object);
	SetVar(button, PARENT, contents);
	PrefixList(contents, button);
	SetMethod(button, CHANGEDVALUE, SniffRepobj);
	left = right + MINORBORDER;
    }

    top = bottom - MINORBORDER;

    return StuffVarControls(vars -> right, contents, top);
}

static void SniffObject()
/*Sniffs objectToSniff*/
{
    WinInfoPtr objWin;
    ObjPtr panel, contents, field, textBox, button;
    int left, right, bottom, top, mid;
    char title[256];
    ObjPtr object;

    if (!objectToSniff)
    {
	return;
    }

    strcpy(title, ObjTypeStr(objectToSniff));
    strcat(title, " ");
    strcat(title, ObjValueStr(objectToSniff));

    objWin = NewObjWindow(NULLOBJ, title, WINRGB + WINDBUF + WINFIXEDSIZE, SNIFFWINWIDTH, SNIFFWINHEIGHT, SNIFFWINWIDTH, SNIFFWINHEIGHT);
    contents = GetVar((ObjPtr) objWin, CONTENTS);

    /*Create a panel*/
    panel = NewPanel(greyPanelClass, 0, SNIFFWINWIDTH, 0, SNIFFWINHEIGHT);
    SetVar(panel, PARENT, (ObjPtr) objWin);
    PrefixList(contents, panel);

    contents = GetListVar("SniffObject", panel, CONTENTS);
    if (!contents) return;

    field = NewControlField(MINORBORDER, SNIFFWINWIDTH - MINORBORDER,
			 MINORBORDER, SNIFFWINHEIGHT - MINORBORDER,
			 "Sniff box", OBJECTSFROMTOP | BARRIGHT);	
    if (!field)
    {
	return;
    }
    PrefixList(contents, field);
    SetVar(field, PARENT, panel);
    SetVar(field, BORDERTYPE, NewInt(1));
    SetVar(field, BACKGROUND, NewInt(UIGRAY62));

    contents = GetListVar("SniffObject", field, CONTENTS);
    if (!contents)
    {
	return;
    }
    SetVar(contents, PARENT, field);

    top = -MINORBORDER;

    /*Put in an entry for the class*/

    bottom = top - BUTTONHEIGHT;
    mid = (bottom + top) / 2;

    /*Create the class text box*/
    left = MINORBORDER;
    right = left + SNIFFIDWIDTH;
    textBox = NewTextBox(left, right,
			  mid - TEXTBOXHEIGHT / 2, mid + TEXTBOXHEIGHT / 2,
			  0, "ha!", "Superclass");
    SetVar(textBox, PARENT, contents);
    PrefixList(contents, textBox);
    SetVar(textBox, TEXTFONT, NewString("Helvetica-Bold"));
    left = right + MINORBORDER;

    object = ClassOf(objectToSniff);

    /*Create the type box*/
    right = left + SNIFFTYPEWIDTH;
    textBox = NewTextBox(left, right,
			  mid - TEXTBOXHEIGHT / 2, mid + TEXTBOXHEIGHT / 2,
			  0, "ha!", ObjTypeStr(object));
    SetVar(textBox, PARENT, contents);
    PrefixList(contents, textBox);
    SetVar(textBox, TEXTFONT, NewString("Helvetica-Bold"));
    left = right + MINORBORDER;

    /*Create the value box*/
    right = left + SNIFFVALWIDTH;
    textBox = NewTextBox(left, right,
			  mid - TEXTBOXHEIGHT / 2, mid + TEXTBOXHEIGHT / 2,
			  0, "ha!", ObjValueStr(object));
    SetVar(textBox, PARENT, contents);
    PrefixList(contents, textBox);
    SetVar(textBox, TEXTFONT, NewString("Helvetica-Bold"));
    SetVar(textBox, REPOBJ, object);
    SetMethod(textBox, DOUBLECLICK, SniffRepobj);
    left = right + MINORBORDER;

    /*Create the show button box*/
    if (object)
    {
	right = left + SNIFFSHOWWIDTH;
	button = NewButton(left, right,
			bottom, top, "Show");
	SetVar(button, REPOBJ, object);
	SetVar(button, PARENT, contents);
	PrefixList(contents, button);
	SetMethod(button, CHANGEDVALUE, SniffRepobj);
	left = right + MINORBORDER;
    }

    top = bottom - MINORBORDER;

    /*Go through the variables*/
    top = StuffVarControls(objectToSniff -> vars, contents, top);

    RecalcScroll(field);
}

void ContextHelp(object)
ObjPtr object;
/*Does in-context help for object*/
{
    helpObject = object;
#ifdef GODLIKE
    if (getbutton(LEFTALTKEY) || getbutton(RIGHTALTKEY))
    {
	objectToSniff = object;
	DoUniqueTask(SniffObject);
    }
    else
#endif
    DoUniqueTask(DoContextHelp);
}

