/*ScianControls.c
  Controls, control panel stuff, etc. for scian
  Eric Pepke
  March 28, 1990
*/

#include "Scian.h"
#include "ScianTypes.h"
#include "ScianLists.h"
#include "ScianControls.h"
#include "ScianButtons.h"
#include "ScianTitleBoxes.h"
#include "ScianColors.h"
#include "ScianDraw.h"
#include "ScianIDs.h"
#include "ScianSliders.h"
#include "ScianTextBoxes.h"
#include "ScianEvents.h"
#include "ScianArrays.h"
#include "ScianErrors.h"
#include "ScianWindows.h"
#include "ScianObjWindows.h"
#include "ScianVisWindows.h"
#include "ScianIcons.h"
#include "ScianMethods.h"
#include "ScianScripts.h"
#include "ScianStyle.h"
#include "ScianHelp.h"
#include "ScianPreferences.h"

ObjPtr controlClass = 0;	/*Great mother of all controls*/
ObjPtr fieldClass = 0;
ObjPtr panelClass = 0;
ObjPtr greyPanelClass = 0;	/*A panel with a grey background*/
ObjPtr corralClass = 0;
ObjPtr switchClass = 0;
ObjPtr controlFieldClass = 0;	/*Text field class*/

#define CELLHEIGHT	25	/*Height of a cell*/
#define CELLFONTSIZE	12.0	/*Size of font in a cell*/
#define CELLTEXTDOWN	18	/*Offset of text from top of a cell*/
#define CELLTEXTRIGHT	6	/*Offset from left*/
#define CELLNAMEWIDTH	150	/*Width of a name*/

#define VB_ICON		1	/*View by icon*/
#define VB_NAME		2	/*View by name*/

#define LABELFONTSIZE   12.0    /* size in points of label type */
#define LABELFONTFUDGE  1       /* fudge strings up (or <0 = down) to center */

#define SWITCHRADIUS	15	/*Radius of a turn on a switch*/
#define SWITCHSLOPLEFT	10	/*Pixels to slop the center of a switch left*/
#define ARROWLENGTH	12	/*Length of an arrow*/
#define ARROWHEIGHT	6	/*Height of an arrow*/

#define FIELDDEPTH	3	/* Depth of a field*/

real xGrid = 16.0, yGrid = 16.0;		/*x and y grid steps*/

#define GETSCROLL(object, xOff, yOff)					\
	{								\
	    ObjPtr var;							\
	    var = GetVar(object, VSCROLL);				\
	    if (var) yOff = -(int) GetSliderValue(var);			\
		else yOff = 0;						\
	    var = GetVar(object, HSCROLL);				\
	    if (var) xOff = -(int) GetSliderValue(var);			\
		else xOff = 0;						\
	}
	   
#ifdef GRAPHICS
void StartPanel(left, right, bottom, top)
int left, right, bottom, top;
{
    pushviewport();
    viewport((Screencoord) left, (Screencoord) right, (Screencoord) bottom, (Screencoord) top);
    /*Set up a projection that works correctly with window*/
    InitClipRect();
    SetOrigin(0, 0);

    ortho2(0.0, (Coord) right - left, 0.0, (Coord) top - bottom);
    RegisterInSpace(false);
}

void StopPanel()
{
    RestoreOrigin();
    popviewport();
    ResetClipRect();
    EndRegister();
}
#endif

#ifdef GRAPHICS
static ObjPtr DrawPanel(object)
ObjPtr object;
/*Draws a control panel*/
{
    int l, r, b, t;
    ObjPtr backColor, borderType;

    Get2DIntBounds(object, &l, &r, &b, &t);

    /*Setup the new viewport*/
    StartPanel(l, r, b, t);

    /*Get the color, if any, and draw it*/
    backColor = GetVar(object, BACKGROUND);
    if (backColor)
    {
	SetObjectColor(backColor);
	clear();
    }

    borderType = GetVar(object, BORDERTYPE);
    if (borderType)
    {
	FrameUIRect(0, r - l - 1, 0, t - b - 1, UIBLACK);
    }

    /*Draw the contents of the panel*/
    DrawList(GetVar(object, CONTENTS));
    StopPanel();

    return NULLOBJ;
}
#endif

static ObjPtr FindObjectField(object, name)
ObjPtr object;
char *name;
/*Searches a field and its contents for an object with name*/
{
    ObjPtr retVal = NULLOBJ;
    ObjPtr objName, contents;
    ObjPtr scrollBar;
    /*First check to see if I am the object*/
    objName = GetVar(object, NAME);
    if (objName && IsString(objName) && 0 == strcmp2(GetString(objName), name))
    {
	if (!retVal)
	{
	    retVal = NewList();
	}
	PostfixList(retVal, object);
    }

    /*Now check my CONTENTS*/
    contents = GetVar(object, CONTENTS);
    if (contents)
    {
	ObjPtr foundObjects;
	foundObjects = FindNamedObject(contents, name);
	if (foundObjects)
	{
	    if (!retVal)
	    {
		retVal = NewList();
	    }
	    AppendList(retVal, foundObjects);
	}
    }

    /*Now check the scroll bars*/
    scrollBar = GetVar(object, HSCROLL);
    if (scrollBar)
    {
	objName = GetVar(scrollBar, NAME);
	if (objName && IsString(objName) && 0 == strcmp2(GetString(objName), name))
	{
	    if (!retVal)
	    {
		retVal = NewList();
	    }
	    PostfixList(retVal, scrollBar);
	}
    }
    scrollBar = GetVar(object, VSCROLL);
    if (scrollBar)
    {
	objName = GetVar(scrollBar, NAME);
	if (objName && IsString(objName) && 0 == strcmp2(GetString(objName), name))
	{
	    if (!retVal)
	    {
		retVal = NewList();
	    }
	    PostfixList(retVal, scrollBar);
	}
    }

    return retVal;
}

static Bool IconConflict(icons, x, y)
ObjPtr icons;
int x, y;
/*See if the position conflicts with any of the icons*/ 
{
    ObjPtr locArray;
    real loc[2];

    if (IsList(icons))
    {
	ThingListPtr list;

	list = LISTOF(icons);
	while (list)
	{
	    locArray = GetFixedArrayVar("IconConflict", list -> thing, ICONLOC, 1, 2L);
	    if (!locArray)
	    {
		return false;
	    }
	    Array2CArray(loc, locArray);

	    if (ABS(x - loc[0]) < ICONXMINSPACE && ABS(y - loc[1]) < ICONYMINSPACE)
	    {
		return true;
	    }
	    list = list -> next;
	}
    }
    return false;
}

#ifdef PROTO
void DropIconInCorral(ObjPtr corral, ObjPtr icon)
#else
void DropIconInCorral(corral, icon)
ObjPtr corral, icon;
#endif
/*Drops icon in corral, if ICONLOC is null, trying to find a place for it to 
  fit.*/
{
    int x, y;			/*Trial icon spacing*/
    int left, right, bottom, top;
    int xmin, xmax;
    ObjPtr contents;
    ObjPtr locArray;
    real loc[2];
    char newName[256];		/*Trial new name*/
    int k;
    ObjPtr name;
    Bool stagger;
    Bool fromTop;

    fromTop = GetPredicate(corral, TOPDOWN);

    name = GetStringVar("DropIconInCorral", icon, NAME);
    if (name)
    {
	/*Create a new name*/
	for (k = 1; ; ++k)
	{
	    strncpy(newName, GetString(name), 255);
	    newName[255] = 0;
	    if (k > 1)
	    {
		sprintf(tempStr, " (%d)", k);
		strcat(newName, tempStr);
	    }
	    if (FindNamedObject(corral, newName))
	    {
		continue;
	    }
	    break;
	}
	SetVar(icon, NAME, NewString(newName));
    }

    Get2DIntBounds(corral, &left, &right, &bottom, &top);

    contents = GetListVar("DropIconInCorral", corral, CONTENTS);
    if (!contents)
    {
	return;
    }

    if (GetVar(icon, ICONLOC) == NULLOBJ)
    {
	/*Calculate a location*/

	if (GetPredicate(corral, SINGLECORRAL))
	{
	    x = (right - left) / 2;
	    y = GetPredicate(corral, TOPDOWN) ?
		(bottom - top) / 2 + 25 :
		(top - bottom) / 2 + 25;
	}
	else
	{
	stagger = GetPrefTruth(PREF_STAGGERICONS);
	xmin = ICONLEFTBORDER;
	xmax = right - left - ICONLEFTBORDER;
	if (xmax < xmin) xmax = xmin;
	for (y = fromTop ? - ICONTOPBORDER : ICONBOTBORDER; ; 
		    y += fromTop ? -ICONYSPACE : ICONYSPACE)
	{
	    k = 0;
	    for (x = xmin; x <= xmax; x += ICONXSPACE)
	    {
		int tempY;
		
		tempY = ((k & 1) && stagger) ? (y + (fromTop ? -ICONYSPACE / 2 : ICONYSPACE / 2)) : y;
		if (!IconConflict(contents, x, tempY))
		{
			y = tempY;
		    goto getoutahere;
		}
		++k;
	    }
	}
	}
getoutahere:
	loc[0] = x;
	loc[1] = y;
	locArray = NewRealArray(1, (long) 2);
	CArray2Array(locArray, loc);
	SetVar(icon, ICONLOC, locArray);
    }
    PrefixList(contents, icon);
    SetVar(icon, PARENT, corral);
    RecalcScroll(corral);
    ImInvalid(icon);
    return;
}

#ifdef INTERACTIVE
void SetScreenGrid(object)
ObjPtr object;
/*Sets the screen grid to a grid within object*/
{
    int l, r, b, t;
    Get2DIntBounds(object, &l, &r, &b, &t);
    xGrid = ((real) (l - r)) / ((real) NGRIDSTEPS);
    yGrid = ((real) (t - b)) / ((real) NGRIDSTEPS);
}
#endif

#ifdef INTERACTIVE
static ObjPtr PressPanel(object, x, y, flags)
ObjPtr object;
int x, y;
int flags;
/*Does a press in a panel beginning at x and y.  Returns
  true iff the press really was in the panel.*/
{
    int left, right, bottom, top;

    /*See if it's a dragBuffer click*/
    if (dragBuffer)
    {
	/*Yes it is.  Move dragBuffer and exit*/
	dropObject = dragBuffer;
	dragBuffer = NULLOBJ;
	return ObjTrue;
    }

    Get2DIntBounds(object, &left, &right, &bottom, &top);

    SetScreenGrid(object);

    if (x >= left && x <= right && y >= bottom && y <= top)
    {
	/*Hey!  It really was a click in the panel*/
	ObjPtr contents;
	ObjPtr retVal;

    	/*Setup the new viewport*/
	StartPanel(left, right, bottom, top);

        x -= left;
        y -= bottom;
	
        contents = GetVar(object, CONTENTS);
	if (contents && IsList(contents))
	{
#ifdef GODLIKE
	    if (flags & F_COMMANDDOWN)
	    {
		ThingListPtr list;
		real ob[4];
		ObjPtr boundsArray;
		/*It's a magic drag*/

		retVal = ObjFalse;
		list = LISTOF(contents);
		while (list)
		{
		    boundsArray = GetVar(list -> thing, BOUNDS);
		    if (boundsArray)
		    {
			Array2CArray(ob, boundsArray);
			if (x >= ob[0] && x <= ob[1] &&
			    y >= ob[2] && y <= ob[3])
			{
			    /*Click here!*/
			    int initX, initY;
			    Bool ml, mr, mb, mt;
			    real newBounds[4];
			    real oldNewBounds[4];
			    newBounds[0] = oldNewBounds[0] = ob[0];
			    newBounds[1] = oldNewBounds[1] = ob[1];
			    newBounds[2] = oldNewBounds[2] = ob[2];
			    newBounds[3] = oldNewBounds[3] = ob[3];

			    initX = x;
			    initY = y;

			    /*Determine bits to move based on position*/
			    mr = x >= ob[0] + 0.75 * (ob[1] - ob[0]);
			    ml = x <= ob[0] + 0.25 * (ob[1] - ob[0]);
			    mt = y >= ob[2] + 0.75 * (ob[3] - ob[2]);
			    mb = y <= ob[2] + 0.25 * (ob[3] - ob[2]);

			    if (!(mr | ml | mb | mt))
			    {
				mr = ml = mb = mt = true;
			    }
			    while (Mouse(&x, &y))
			    {
				if (x != initX || y != initY)
				{
				    /*The mouse has moved*/

				    if (ml) newBounds[0] = ob[0] + x - initX;
				    if (mr) newBounds[1] = ob[1] + x - initX;
				    if (mb) newBounds[2] = ob[2] + y - initY;
				    if (mt) newBounds[3] = ob[3] + y - initY;

				    if (flags & F_OPTIONDOWN)
				    {
					/*Grid drag*/
					if (ml && mr && mb && mt)
					{
					    /*Special case--whole object gridded
					      Only grid top left*/
					    real width, height;
					    width = newBounds[1] - newBounds[0];
					    height = newBounds[3] - newBounds[2];
					    newBounds[0] = GRIDX(newBounds[0]);
					    newBounds[1] = newBounds[0] + width;
					    newBounds[3] = ob[3] - 
							(GRIDY(ob[3] - newBounds[3]));
					    newBounds[2] = newBounds[3] - height;
					}
					else
					{
					    /*Normal case*/
					    if (ml) newBounds[0] = GRIDX(newBounds[0]);
					    if (mr) newBounds[1] = ob[1] - GRIDX(ob[1] - newBounds[1]);
					    if (mb) newBounds[2] = GRIDY(newBounds[2]);
					    if (mt) newBounds[3] = ob[3] - GRIDY(ob[3] - newBounds[3]);
					}
				    }

				    if ((newBounds[0] != oldNewBounds[0] ||
					 newBounds[1] != oldNewBounds[1] ||
					 newBounds[2] != oldNewBounds[2] ||
					 newBounds[3] != oldNewBounds[3]) &&
					 newBounds[0] < newBounds[1] &&
					 newBounds[2] < newBounds[3])
				    {
					CArray2Array(boundsArray, newBounds);
					oldNewBounds[0] = newBounds[0];
					oldNewBounds[1] = newBounds[1];
					oldNewBounds[2] = newBounds[2];
					oldNewBounds[3] = newBounds[3];
					DrawMe(list -> thing);
				    }
				}
			    }
			    retVal = ObjTrue;

			    MakeMeCurrent(list -> thing);
			    if (logging)
			    {
				char cmd[256];
				MakeObjectName(tempStr, list -> thing);
				sprintf(cmd, "set bounds %s [%g %g %g %g]\n",
					tempStr, newBounds[0], newBounds[1],
					newBounds[2], newBounds[3]);
				Log(cmd);
			    }
			    break;
			}
		    }
		    list = list -> next;
		}
	    }
	    else
#endif
	    {
		/*It's a normal click*/
		retVal = PressObject(contents, x, y, flags);
	    }
	    if (TOOL(flags) == T_HELP && !IsTrue(retVal))
	    {
		ContextHelp(object);
	    }
	}
	else
	{
	    retVal = ObjFalse;
	}

	StopPanel();
	return retVal;
    }
    else
    {
	return ObjFalse;
    }
}
#endif

#ifdef INTERACTIVE
static ObjPtr DropInPanel(object, dropObj, x, y)
ObjPtr object, dropObj;
int x, y;
/*Drops object in a panel beginning at x and y.  Returns
  true iff the drop really was in the panel.*/
{
    int left, right, bottom, top;

    Get2DIntBounds(object, &left, &right, &bottom, &top);

    if (x >= left && x <= right && y >= bottom && y <= top)
    {
	/*Hey!  It really was a drop in the panel*/
	ObjPtr contents;

    	/*Setup the new viewport*/
	StartPanel(left, right, bottom, top);

        x -= left;
        y -= bottom;
	contents = GetVar(object, CONTENTS);
	if (contents && IsList(contents))
	{
	    DropList(contents, dropObj, x, y);
	}

	StopPanel();
	return ObjTrue;
    }
    else
    {
	return ObjFalse;
    }
}
#endif

#ifdef INTERACTIVE
static ObjPtr KeyDownContents(object, key, flags)
ObjPtr object;
short key;
int flags;
/*Does a keydown in something with a CONTENTS*/
{
    ObjPtr contents;
    contents = GetVar(object, CONTENTS);
    if (contents)
    {
	return KeyDownList(contents, key, flags);
    }
    else
    {
	return ObjFalse;
    }
}
#endif

#ifdef INTERACTIVE
static ObjPtr PressFieldContents(object, x, y, flags)
ObjPtr object;
int x, y;
int flags;
/*Presses the contents of a field*/
{
    ObjPtr contents;

    contents = GetListVar("PressFieldContents", object, CONTENTS);
    if (contents)
    {
	return PressObject(contents, x, y, flags);
    }
    else
    {
	return ObjFalse;
    }
}
#endif

void ScrollHome(field)
ObjPtr field;
/*Scrolls any field to home.*/
{
    ObjPtr scrollbar;

    scrollbar = GetVar(field, HSCROLL);
    if (scrollbar)
    {
	SetSliderValue(scrollbar, 0.0);
    }

    scrollbar = GetVar(field, VSCROLL);
    if (scrollbar)
    {
	SetSliderValue(scrollbar, 0.0);
    }
}

#ifdef INTERACTIVE
static ObjPtr PressField(object, x, y, flags)
ObjPtr object;
int x, y;
int flags;
/*Does a press in a field beginning at x and y.  Returns
  true iff the press really was in the field.*/
{
    int left, right, bottom, top;
    FuncTyp pressContents;		/*Routine to press the contents*/
    ObjPtr scrollBar;

    Get2DIntBounds(object, &left, &right, &bottom, &top);

    /*See if it's a press in a scrollbar*/
    scrollBar = GetVar(object, HSCROLL);
    if (scrollBar)
    {
	if (IsTrue(PressObject(scrollBar, x, y, flags)))
	{
	    return ObjTrue;
	}
    }
    scrollBar = GetVar(object, VSCROLL);
    if (scrollBar)
    {
	if (IsTrue(PressObject(scrollBar, x, y, flags)))
	{
	    return ObjTrue;
	}
    }

    if (x >= left && x <= right && y >= bottom && y <= top)
    {
	/*Hey!  It really was a click in the field*/
	int xOff, yOff;
	ObjPtr contentsPressed = ObjFalse;
	int fieldDepth;
	if (GetPredicate(object, BORDERTYPE))
	{
	    fieldDepth = 1;
	}
	else
	{
	    fieldDepth = FIELDDEPTH;
	}

	GETSCROLL(object, xOff, yOff);

	/*This doesn't really count as interactive drawing*/
	interactiveMoving = false;

	SetClipRect(left + fieldDepth, right - fieldDepth, bottom + fieldDepth, top - FIELDDEPTH);
	if (GetPredicate(object, TOPDOWN))
	{
	    SetOrigin(left + fieldDepth + xOff, top - fieldDepth + yOff);
            x -= left + fieldDepth + xOff;
            y -= top - fieldDepth + yOff;
	}
	else
	{
	    SetOrigin(left + fieldDepth + xOff, bottom + fieldDepth + yOff);
            x -= left + fieldDepth + xOff;
            y -= bottom + fieldDepth + yOff;
	}
	
	pressContents = GetMethod(object, PRESSCONTENTS);
	if (pressContents)
	{
	    contentsPressed = (*pressContents)(object, x, y, flags);
	}
	if (!IsTrue(contentsPressed))
	{
	    if (TOOL(flags) == T_HELP)
	    {
		ContextHelp(object);
		return ObjTrue;
	    }
	}

	RestoreOrigin();
	RestoreClipRect();
	return ObjTrue;
    }
    else
    {
	return ObjFalse;
    }
}
#endif

#ifdef INTERACTIVE
static ObjPtr DropInCorral(object, dropObj, x, y)
ObjPtr object, dropObj;
int x, y;
/*Drops dropObj in corral object at x and y.  Returns
  true iff the drop really was in the corral.*/
{
    int left, right, bottom, top;

    MakeMeCurrent(object);

    Get2DIntBounds(object, &left, &right, &bottom, &top);

    if (x >= left && x <= right && y >= bottom && y <= top)
    {
	/*Hey!  It really was a drop in the corral*/
	ObjPtr contents;
	int xOff, yOff;
	ObjPtr firstIcon;
	ThingListPtr restIcons;
	ThingListPtr runner;
	ObjPtr iconLoc;
	real loc[2];
	int xDisp, yDisp;
	int fieldDepth;

	if (GetPredicate(object, BORDERTYPE))
	{
	    fieldDepth = 1;
	}
	else
	{
	    fieldDepth = FIELDDEPTH;
	}

	GETSCROLL(object, xOff, yOff);

	SetClipRect(left + fieldDepth, right - fieldDepth, bottom + fieldDepth, top - FIELDDEPTH);
	if (GetPredicate(object, TOPDOWN))
	{
	    SetOrigin(left + fieldDepth + xOff, top - fieldDepth + yOff);
	    y -= top - fieldDepth + yOff;
	}
	else
	{
	    SetOrigin(left + fieldDepth + xOff, bottom + fieldDepth + yOff);
	    y -= bottom + fieldDepth + yOff;
	}
        x -= left + fieldDepth + xOff;
	x += iconXOff;
	y += iconYOff;

	/*Get the first icon and the rest of the icons*/
	if (IsList(dropObj))
	{
	    restIcons = LISTOF(dropObj);
	    firstIcon = restIcons -> thing;
	    restIcons = restIcons -> next;
	}
	else if (InClass(dropObj, iconClass))
	{
	    firstIcon = dropObj;
	    restIcons = 0;
	}
	else
	{
	    ReportError("DropInCorral", "An object not an icon was dropped");
	    return ObjFalse;
	}

	/*Get the location of the first icon*/
	iconLoc = GetFixedArrayVar("DropInCorral", firstIcon, ICONLOC, 1, 2L);
	if (!iconLoc)
	{
	    return ObjFalse;
	}
	Array2CArray(loc, iconLoc);
	xDisp = x - loc[0];
	yDisp = y - loc[1];

	/*See if the first icon is already in the contents*/
	contents = GetVar(object, CONTENTS);
	if (contents && IsList(contents))
	{
	    ThingListPtr list;

	    list = LISTOF(contents);
	    while (list)
	    {
		if (list -> thing == firstIcon)
		{
		    /*Hey, it's just a move*/
		    real loc[2];
		    ObjPtr array;
		    loc[0] = x;
		    loc[1] = y;

		    array = NewRealArray(1, 2L);
		    CArray2Array(array, loc);
		    SetVar(firstIcon, ICONLOC, array);

		    runner = restIcons;
		    while (runner)
		    {
			array = GetFixedArrayVar("DropInCorral", runner -> thing, ICONLOC, 1, 2L);
			if (!array) return ObjFalse;
			Array2CArray(loc, array);
			loc[0] += xDisp;
			loc[1] += yDisp;
			array = NewRealArray(1, 2L);
			CArray2Array(array, loc);
			SetVar(runner -> thing, ICONLOC, array);
			runner = runner -> next;
		    }

		    RecalcScroll(object);
		    ImInvalid(object);

		    break;
		}
		list = list -> next;
	    }
	    if (!list)
	    {
		FuncTyp method;
		ObjPtr array;
		real loc[2];
		ObjPtr hScroll, vScroll;
		
		/*It must not have been already in the corral.  Give the
		  corral a chance to act on it*/
		hScroll = GetVar(object, HSCROLL);
		vScroll = GetVar(object, VSCROLL);
		method = GetMethod(object, DROPINCONTENTS);
		if (method)
		{
		    (*method)(object, firstIcon, x, y);
		    runner = restIcons;
		    while (runner)
		    {
			array = GetFixedArrayVar("DropInCorral", runner -> thing, ICONLOC, 1, 2L);
			if (!array) return ObjFalse;
			Array2CArray(loc, array);
			loc[0] += xDisp;
			loc[1] += yDisp;

			/*Trim location to be inside corral*/
			if (!hScroll)
			{
			    if (loc[0] < left - FIELDDEPTH) loc[0] = left - FIELDDEPTH;
			    else if (loc[0] > right - FIELDDEPTH) loc[0] = right - FIELDDEPTH;
			}
			if (!vScroll)
			{
			    if (loc[1] < bottom - FIELDDEPTH) loc[1] = bottom - FIELDDEPTH;
			    else if (loc[1] > top - FIELDDEPTH) loc[1] = top - FIELDDEPTH;
			}

			(*method)(object, runner -> thing, (int) loc[0], (int) loc[1]);
			runner = runner -> next;
		    }
		    ImInvalid(object);
		    RecalcScroll(object);
		}
	    }
	}
	RestoreOrigin();
	RestoreClipRect();

	return ObjTrue;
    }
    else
    {
	return ObjFalse;
    }
}
#endif

#ifdef INTERACTIVE
Bool DragIcons(objects, x, y)
ObjPtr objects;
int x, y;
/*Drags an icon starting at x and y.  Also drags a box around l, r, b, t.
  Returns true if it succeeded*/
{
    Bool dragging;		/*True iff dragging*/
    int xDisp, yDisp;		/*X and Y displacements*/
    int xOff, yOff;		/*X and y offsets*/
    int newX, newY;
    int lastX, lastY;
    long ox, oy;
    ThingListPtr runner;	/*A runner for the list*/
    ObjPtr firstIcon;		/*The first icon in the list*/
    ThingListPtr restIcons;	/*The list of the rest of the icons*/
    ObjPtr locArray;
    real loc[2];		/*Location of the first icon*/

    if (IsList(objects))
    {
	/*It's a list of icons.*/
	restIcons = LISTOF(objects);
	firstIcon = restIcons -> thing;
	restIcons = restIcons -> next;
    }
    else
    {
	/*It's a single icon, kludge it up so that it will continue to work*/
	firstIcon = objects;
	restIcons = 0;
    }

    getorigin(&ox, &oy);

    CurOffset(&xDisp, &yDisp);

    locArray = GetFixedArrayVar("DragIcons", firstIcon, ICONLOC, 1, 2L);
    if (!locArray)
    {
	return;
    }
    Array2CArray(loc, locArray);

    iconXOff = loc[0] - x;
    iconYOff = loc[1] - y;

#ifdef INTERWINDRAG
    xOff = xDisp - x;
    yOff = yDisp - y;
#else
    xOff = -x;
    yOff = -y;
    {
	short sl, sr, sb, st;
	getviewport(&sl, &sr, &sb, &st);
	scrmask(sl, sr, sb, st);
    }
#endif

    dragging = false;
    
    lastX = x;
    lastY = y;
    while (Mouse(&newX, &newY))
    {
	if (ABS(newX - x) > 2 || ABS(newY - y) > 2)
	{
	    /*Start to drag*/
	    dragging = true;
	    pushattributes();
#ifdef INTERWINDRAG
	    fullscrn();
#endif
	    drawmode(OVERDRAW);
	    overlay(2);
	    mapcolor(0, 0, 0, 0);
	    mapcolor(1, 255, 0, 0);
	    mapcolor(2, 255, 0, 0);
	    gconfig();
	    color(1);
	    DrawIconGhost(firstIcon, newX + xOff, newY + yOff);
	    runner = restIcons;
	    while (runner)
	    {
		DrawIconGhost(runner -> thing, newX + xOff, newY + yOff);
		runner = runner -> next;
	    }
	    lastX = newX;
	    lastY = newY;
	    break;
	}
    }

    if (!dragging) return false;
    
    if (logging)
    {
	char cmd[400];
	char *s;

	sprintf(cmd, "drag ");
	s = &(cmd[0]);
	while (*s) ++s;
	MakeObjectName(s, firstIcon);
	while (*s) ++s;
	runner = restIcons;
	while (runner)
	{
	    *s++ = ' ';
	    MakeObjectName(s, runner -> thing);
	    while (*s) ++s;
	    runner = runner -> next;
	}
	*s++ = '\n';
	*s = 0;
	Log(cmd);
    }

    while (Mouse(&newX, &newY))
    {
	if (newX != lastX || newY != lastY)
	{
	    /*Erase, move, and redraw*/
	    color(0);
	    DrawIconGhost(firstIcon, lastX + xOff, lastY + yOff);
	    runner = restIcons;
	    while (runner)
	    {
		DrawIconGhost(runner -> thing, lastX + xOff, lastY + yOff);
		runner = runner -> next;
	    }
	    color(1);
	    DrawIconGhost(firstIcon, newX + xOff, newY + yOff);
	    runner = restIcons;
	    while (runner)
	    {
		DrawIconGhost(runner -> thing, newX + xOff, newY + yOff);
		runner = runner -> next;
	    }

	    lastX = newX;
	    lastY = newY;
	}
    }
    color(0);
    DrawIconGhost(firstIcon, lastX + xOff, lastY + yOff);
    runner = restIcons;
    while (runner)
    {
	DrawIconGhost(runner -> thing, lastX + xOff, lastY + yOff);
	runner = runner -> next;
    }
    drawmode(NORMALDRAW);
    popattributes();
#ifdef INTERWINDRAG
    endfullscrn();
#endif
    gconfig();
    Mouse(&dropX, &dropY);
    dropX += ox + xDisp;
    dropY += oy + yDisp;
    return true;
}
#endif

void SelectAllIcons(window)
ObjPtr window;
/*Selects all icons in the window*/
{
    ObjPtr corral, contents;
    ThingListPtr runner;

    corral = FindMainCorral((WinInfoPtr) window);
    if (!corral)
    {
	return;
    }
    contents = GetListVar("SelectAllIcons", corral, CONTENTS);
    if (!contents)
    {
	return;
    }
    if (logging)
    {
	Log("selectall\n");
	InhibitLogging(true);
    }
    runner = LISTOF(contents);
    while (runner)
    {
	if (!GetPredicate(runner -> thing, SELECTED))
	{
	    Select(runner -> thing, true);
	}
	runner = runner -> next;
    }
    if (logging)
    {
	InhibitLogging(false);
    }
}

void DeselectAllIcons(window)
ObjPtr window;
/*Selects all icons in the window*/
{
    ObjPtr corral, contents;
    ThingListPtr runner;

    corral = FindMainCorral((WinInfoPtr) window);
    if (!corral)
    {
	return;
    }
    contents = GetListVar("DeselectAllIcons", corral, CONTENTS);
    if (!contents)
    {
	return;
    }
    if (logging)
    {
	Log("deselectall\n");
	InhibitLogging(true);
    }
    runner = LISTOF(contents);
    while (runner)
    {
	if (GetPredicate(runner -> thing, SELECTED))
	{
	    Select(runner -> thing, false);
	}
	runner = runner -> next;
    }
    if (logging)
    {
	InhibitLogging(false);
    }
}

ObjPtr CorralNotCurrent(corral)
ObjPtr corral;
/*Alerts a corral that it's not current*/
{
    ThingListPtr runner;
    ObjPtr contents;
    ObjPtr scroll;

    scroll = GetVar(corral, HSCROLL);
    if (scroll && AmICurrent(scroll)) return ObjTrue;
    
    scroll = GetVar(corral, VSCROLL);
    if (scroll && AmICurrent(scroll)) return ObjTrue;

    if (GetPredicate(corral, ONEICON))
    {
	return ObjTrue;
    }
    
    contents = GetListVar("CorralNotCurrent", corral, CONTENTS);
    if (!contents)
    {
	return;
    }
    if (logging)
    {
	Log("deselectall\n");
	InhibitLogging(true);
    }
    runner = LISTOF(contents);
    while (runner)
    {
	if (GetPredicate(runner -> thing, SELECTED))
	{
	    Select(runner -> thing, false);
	}
	runner = runner -> next;
    }
    if (logging)
    {
	InhibitLogging(false);
    }
}

void DoSelectAllIcons()
/*Selects all icons in the current window*/
{
    if (!selWinInfo)
    {
	return;
    }
    SelectAllIcons((ObjPtr) selWinInfo);
}

void DoDeselectAllIcons()
/*Selects all icons in the current window*/
{
    if (!selWinInfo)
    {
	return;
    }
    DeselectAllIcons((ObjPtr) selWinInfo);
}

#ifdef INTERACTIVE
static ObjPtr PressCorralContents(corral, x, y, flags)
ObjPtr corral;
int x, y;
int flags;
/*Does a press in the contents of a corral*/
{
    ObjPtr contents;
    ObjPtr pressedIcon = 0;	/*The icon pressed in*/
    ThingListPtr runner;
    FuncTyp method;

    MakeMeCurrent(corral);

    contents = GetListVar("PressCorralContents", corral, CONTENTS);
    if (!contents) return ObjFalse;

    runner = LISTOF(contents);
    while (runner)
    {
    	ObjPtr icon;
	ObjPtr theLoc;
	real loc[2];

	icon = runner -> thing;
	theLoc = GetFixedArrayVar("PressCorralContents", icon, ICONLOC, 1, 2L);
	Array2CArray(loc, theLoc);

	if (x >= ((int) loc[0]) - 16 && x <= ((int) loc[0]) + 16 && 
	    y >= ((int) loc[1]) - 32 && y <= ((int) loc[1]) + 16)
	{
	    pressedIcon = icon;
	}

	runner = runner -> next;
    }

    if (pressedIcon)
    {
	Bool wasSelected;

	if (TOOL(flags) == T_HELP)
	{
	    ContextHelp(pressedIcon);
	    return ObjTrue;
	}

	/*Select or deselect the icon*/
	if (GetPredicate(pressedIcon, SELECTED))
        {
	    /*Already selected.  Only deselect if shift down*/
	    wasSelected = true;
	    if ((flags & F_SHIFTDOWN) || (TOOL(flags) == T_ROTATE))
	    {
		method = GetMethod(pressedIcon, SELECT);
		if (method)
		{
		    (*method)(pressedIcon, false);
		    DrawMe(pressedIcon);
		}
	    }
	}
	else
	{
	    wasSelected = false;
	    method = GetMethod(pressedIcon, SELECT);
	    if (method)
	    {
		(*method)(pressedIcon, true);
		DrawMe(pressedIcon);
	    }
	}

	/*Something has been pressed.  
	  Deselect all other icons IF
	  1) It's a oneIcon field, or
	  2) It's not a oneIcon field and
	     the shift key is up and
	     the icon was not previously selected*/

	if (GetPredicate(corral, ONEICON) ? 1 :
			((0 == (flags & F_SHIFTDOWN)) && TOOL(flags) != T_ROTATE && !wasSelected))
	{
	    runner = LISTOF(contents);
	    while (runner)
	    {
    		ObjPtr oneObject;

		oneObject = runner -> thing;
		if (oneObject != pressedIcon)
		{
		    FuncTyp method;
		    method = GetMethod(oneObject, SELECT);
		    if (method)
		    {
			InhibitLogging(true);
			(*method)(oneObject, false);
			InhibitLogging(false);
		    }
		}
		runner = runner -> next;
	    }
	    DrawMe(corral);
	}
    }
    else if (!GetPredicate(corral, ONEICON))
    {
	/*Start a marquee drag*/
	int newX, newY;
	int curX, curY;	
	ObjPtr marquee;
	real marqueeBounds[4];
	Bool yesMarquee = false;

	curX = x;
	curY = y;

	if (TOOL(flags) == T_HELP)
	{
	    ContextHelp(corral);
	    return ObjTrue;
	}

	/*Deselect all other icons if the shift key is not down*/
	if (0 == (flags & F_SHIFTDOWN) && TOOL(flags) != T_ROTATE)
	{
	    runner = LISTOF(contents);
	    while (runner)
	    {
    		ObjPtr oneObject;
		FuncTyp method;

		oneObject = runner -> thing;
		method = GetMethod(oneObject, SELECT);
		if (method)
		{
		    (*method)(oneObject, false);
		}
		runner = runner -> next;
	    }
	    DrawMe(corral);
	}

	while (Mouse(&newX, &newY))
	{
	    if (newX != curX || newY != curY)
	    {
		yesMarquee = true;
		curX = newX;
		curY = newY;
		if (curX > x)
		{
		    marqueeBounds[0] = x;
		    marqueeBounds[1] = curX;
		}
		else if (x > curX)
		{
		    marqueeBounds[0] = curX;
		    marqueeBounds[1] = x;
		}
		else
		{
		    yesMarquee = false;
		}

		if (curY > y)
		{
		    marqueeBounds[2] = y;
		    marqueeBounds[3] = curY;
		}
		else if (y > curY)
		{
		    marqueeBounds[2] = curY;
		    marqueeBounds[3] = y;
		}
		else
		{
		    yesMarquee = false;
		}
		
		if (yesMarquee)
		{
		    marquee = NewRealArray(1, 4L);
		    CArray2Array(marquee, marqueeBounds);
		}
		else
		{
		    marquee = NULLOBJ;
		}

		SetVar(corral, MARQUEE, marquee);
		DrawMe(corral);
	    }
	}
	SetVar(corral, MARQUEE, NULLOBJ);

	if (yesMarquee)
	{
	    /*Go throught list of icons to determine which ones should be pressed*/
	    runner = LISTOF(contents);
	    while (runner)
	    {
		ObjPtr icon;
		ObjPtr theLoc;
		real loc[2];

		icon = runner -> thing;
		theLoc = GetFixedArrayVar("PressCorralContents", icon, ICONLOC, 1, 2L);
		Array2CArray(loc, theLoc);

		if (loc[0] > marqueeBounds[0] - 16 &&
		    loc[0] < marqueeBounds[1] + 16 &&
		    loc[1] > marqueeBounds[2] - 8 &&
		    loc[1] < marqueeBounds[3] + 32)
		{
		    /*Select or deselect the icon*/
		    if (GetPredicate(icon, SELECTED))
        	    {
			/*Already selected.  Only deselect if shift down*/
			if ((flags & F_SHIFTDOWN) || (TOOL(flags) == T_ROTATE))
			{
			    method = GetMethod(icon, SELECT);
			    if (method)
			    {
				(*method)(icon, false);
			    }
			}
		    }
		    else
		    {
			method = GetMethod(icon, SELECT);
			if (method)
			{
			    (*method)(icon, true);
			}
		    }
		}

		runner = runner -> next;
	    }
	}

	DrawMe(corral);
    }
    else
    {
	return ObjFalse;
    }

    UpdateDrawing();

    if (pressedIcon)
    {
	if ((flags & F_DOUBLECLICK) && GetPredicate(pressedIcon, SELECTED))
	{
	    /*Do it's doubleclick method*/
	    FuncTyp method;
	    method = GetMethod(pressedIcon, DOUBLECLICK);
	    if (method)
	    {
		(*method)(pressedIcon);
	    }
	}
	else
	{
	    /*Construct a list of icons and drag it*/
	    ObjPtr iconList;
	    iconList = NewList();

	    runner = LISTOF(contents);
	    while (runner)
	    {
		if (GetPredicate(runner -> thing, SELECTED) &&
		    runner -> thing != pressedIcon)
		{
		    PrefixList(iconList, runner -> thing);
		}
		runner = runner -> next;
	    }
	    PrefixList(iconList, pressedIcon);
	    if (DragIcons(iconList, x, y))
	    {
		dropObject = iconList;
		AddToReferenceList(iconList);
	    }
	}
    }

    return ObjTrue;
}
#endif

#ifdef GRAPHICS
ObjPtr DrawField(object)
ObjPtr object;
/*Draws a field*/
{
    int left, right, bottom, top;
    ObjPtr backColor;			/*Color of the background*/
    FuncTyp drawContents;		/*Routine to draw the contents*/
    ObjPtr scrollBar;			/*Temporary place for the scroll bar*/

    Get2DIntBounds(object, &left, &right, &bottom, &top);

    /*Draw the background*/
    DrawPitFrame(left, right, bottom, top, FIELDDEPTH);

    SetClipRect(left + FIELDDEPTH, right - FIELDDEPTH, bottom + FIELDDEPTH, top - FIELDDEPTH);
    backColor = (ObjPtr) GetVar(object, BACKGROUND);
    if (backColor)
    {
	SetObjectColor(backColor);
    }
    else
    {
	SetUIColor(UIICONPIT);
    }
    clear();
    
    drawContents = GetMethod(object, DRAWCONTENTS);
    if (drawContents)
    {
	int xOff, yOff;
	int fieldDepth;
	GETSCROLL(object, xOff, yOff);
	if (GetPredicate(object, BORDERTYPE))
	{
	    fieldDepth = 1;
	}
	else
	{
	    fieldDepth = FIELDDEPTH;
	}

	if (GetPredicate(object, TOPDOWN))
	{
	    SetOrigin(left + fieldDepth + xOff, top - fieldDepth + yOff);
	}
	else
	{
	    SetOrigin(left + fieldDepth + xOff, bottom + fieldDepth + yOff);
	}
	(*drawContents)(object);
	RestoreOrigin();
    }

    RestoreClipRect();

    scrollBar = GetVar(object, HSCROLL);
    if (scrollBar)
    {
	DrawObject(scrollBar);
    }

    scrollBar = GetVar(object, VSCROLL);
    if (scrollBar)
    {
	DrawObject(scrollBar);
    }
    return ObjTrue;
}
#endif

#ifdef GRAPHICS
ObjPtr DrawControlField(object)
ObjPtr object;
/*Draws a control field*/
{
    int left, right, bottom, top;
    ObjPtr backColor;			/*Color of the background*/
    FuncTyp drawContents;		/*Routine to draw the contents*/
    ObjPtr scrollBar;			/*Temporary place for the scroll bar*/
    ObjPtr borderType;			/*Type of border*/
    int fieldDepth;

    Get2DIntBounds(object, &left, &right, &bottom, &top);

    /*Draw the border*/
    borderType = GetVar(object, BORDERTYPE);
    if (borderType)
    {
	FrameUIRect(left, right, bottom, top, UIBLACK);
    }

    if (borderType)
    {
	fieldDepth = 1;
    }
    else
    {
	fieldDepth = FIELDDEPTH;
    }

    SetClipRect(left + fieldDepth, right - fieldDepth,
		bottom + fieldDepth, top - fieldDepth);

    /*Get the color, if any, and draw it*/
    backColor = (ObjPtr) GetVar(object, BACKGROUND);
    if (backColor)
    {
	SetObjectColor(backColor);
    }
    else
    {
	SetUIColor(UIICONPIT);
    }
    clear();

    /*Draw the contents of the panel*/
    drawContents = GetMethod(object, DRAWCONTENTS);
    if (drawContents)
    {
	int xOff, yOff;

	GETSCROLL(object, xOff, yOff);

	if (GetPredicate(object, TOPDOWN))
	{
	    SetOrigin(left + fieldDepth + xOff, top - fieldDepth + yOff);
	}
	else
	{
	    SetOrigin(left + fieldDepth + xOff, bottom + fieldDepth + yOff);
	}

	(*drawContents)(object);
	RestoreOrigin();
    }

    RestoreClipRect();

    scrollBar = GetVar(object, HSCROLL);
    if (scrollBar)
    {
	DrawObject(scrollBar);
    }

    scrollBar = GetVar(object, VSCROLL);
    if (scrollBar)
    {
	DrawObject(scrollBar);
    }
    return ObjTrue;
}
#endif

#ifdef GRAPHICS 
static ObjPtr DrawControlFieldContents(object)
ObjPtr object;
/*Draws a control field's contents*/
{
    ObjPtr contents;
    contents = GetVar(object, CONTENTS);
    if (contents)
    {
	DrawList(contents);
    }
    return ObjTrue;
}
#endif

#ifdef GRAPHICS
static ObjPtr DrawCorralContents(object)
ObjPtr object;
/*Draws the contents of an icon corral*/
{
    ObjPtr marquee;
    ObjPtr var;
    int viewBy;

    var = GetVar(object, VIEWBYWHAT);
    if (var)
    {
	viewBy = GetInt(var);
    }
    else
    {
	viewBy = VB_ICON;
    }

    if (viewBy == VB_ICON)
    {
	/*View by icon.  Just let the icons draw themselves*/
	real marqueeBounds[4];
	marquee = GetVar(object, MARQUEE);
	if (marquee)
	{
	    Array2CArray(marqueeBounds, marquee);
	    FrameUIRect(marqueeBounds[0] + 1.0,
			marqueeBounds[1] + 1.0,
			marqueeBounds[2] - 1.0,
			marqueeBounds[3] - 1.0, UIBLACK);
	    
	    FrameUIRect(marqueeBounds[0],
			marqueeBounds[1],
			marqueeBounds[2],
			marqueeBounds[3], UIWHITE);
	}

	/*Now draw the contents*/
	DrawList(GetVar(object, CONTENTS));
    }
    return ObjTrue;
}
#endif

#ifdef GRAPHICS
static ObjPtr DrawSwitch(object)
ObjPtr object;
/*Draws a switch*/
{
    int left, right, bottom, top;
    ObjPtr var;
    int nCells;
    int outCell;
    int value;
    int k, x, y, cl, cr, c, y2;
    
    Get2DIntBounds(object, &left, &right, &bottom, &top);

    /*Draw the background*/
    FillUIRect(left, right, bottom, top, UIBACKGROUND);

    /*Get the parameters*/
    var = GetVar(object, NCELLS);
    if (!var || !IsInt(var))
    {
	return;
    }
    nCells = GetInt(var);

    var = GetVar(object, OUTCELL);
    if (!var || !IsInt(var))
    {
	return;
    }
    outCell = GetInt(var);

    var = GetVar(object, VALUE);
    if (!var || !IsInt(var))
    {
	return;
    }
    value = GetInt(var);

    /*Set up temporary vars to make drawing easier*/
    c = (left + right) / 2 - SWITCHSLOPLEFT;
    if (c < left + SWITCHRADIUS) c = left + SWITCHRADIUS;
    cl = c - SWITCHRADIUS;
    cr = c + SWITCHRADIUS;

    /*Draw the background*/
    SetUIColor(UIBLACK);
    for (k = 0; k < nCells; ++k)
    {
	y = top - (top - bottom) * k / nCells - (top - bottom) / nCells / 2;

	move2i(left, y);
	draw2i(cl, y);
	if (k < outCell)
	{
	    /*Draw a curve downward*/
	    arci(cl, y - SWITCHRADIUS, SWITCHRADIUS, 0, 900);
	}
	else if (k > outCell)
	{
	    /*Draw a curve upward*/
	    arci(cl, y + SWITCHRADIUS, SWITCHRADIUS, -900, 0);
	}
	else
	{
	    /*Draw a junction*/
	    if (k > 0)
	    {
		/*Something coming in from above*/
		arci(cr, y + SWITCHRADIUS, SWITCHRADIUS, 1800, 2700);
	    }
	    move2i(cl, y);
	    draw2i(right, y);
	    if (k < nCells - 1)
	    {
		/*Something coming in from below*/
		arci(cr, y - SWITCHRADIUS, SWITCHRADIUS, 900, 1800);
	    }
	}
    }

    /*Draw the vertical lines*/
    if (outCell > 0)
    {
	/*There's a top part*/
	y = top - (top - bottom) / nCells / 2;
	y2 = top - (top - bottom) * outCell / nCells - (top - bottom) / nCells / 2;
	move2i(c, y - SWITCHRADIUS);
	draw2i(c, y2 + SWITCHRADIUS);
    }

    if (outCell < nCells - 1)
    {
	/*There's a bottom part*/
	y = top - (top - bottom) * outCell / nCells - (top - bottom) / nCells / 2;
	y2 = top - (top - bottom) * (nCells - 1) / nCells - (top - bottom) / nCells / 2;
	move2i(c, y - SWITCHRADIUS);
	draw2i(c, y2 + SWITCHRADIUS);
    }

    /*Draw the highlighted path*/
    if (value >= 0)
    {
	long vertices[3][2];
	linewidth(3);

	if (value < outCell)
	{
	    /*Over, down, over*/
	    y = top - (top - bottom) * value / nCells - (top - bottom) / nCells / 2;
	    y2 = top - (top - bottom) * outCell / nCells - (top - bottom) / nCells / 2;
	    move2i(left, y);
	    draw2i(cl, y);
	    arci(cl, y - SWITCHRADIUS, SWITCHRADIUS, 0, 900);
	    move2i(c, y - SWITCHRADIUS);
	    draw2i(c, y2 + SWITCHRADIUS);
	    arci(cr, y2 + SWITCHRADIUS, SWITCHRADIUS, 1800, 2700);
	    move2i(cr, y2);
	    draw2i(right - 2, y2);
	}
	else if (value > outCell)
	{
	    /*Over, up, over*/
	    y = top - (top - bottom) * value / nCells - (top - bottom) / nCells / 2;
	    y2 = top - (top - bottom) * outCell / nCells - (top - bottom) / nCells / 2;
	    move2i(left, y);
	    draw2i(cl, y);
	    arci(cl, y + SWITCHRADIUS, SWITCHRADIUS, -900, 0);
	    move2i(c, y + SWITCHRADIUS);
	    draw2i(c, y2 - SWITCHRADIUS);
	    arci(cr, y2 - SWITCHRADIUS, SWITCHRADIUS, 900, 1800);
	    move2i(cr, y2);
	    draw2i(right - 2, y2);
	}
	else
	{
	    /*Straight over*/
	    y2 = top - (top - bottom) * value / nCells - (top - bottom) / nCells / 2;
	    move2i(left, y2);
	    draw2i(right - 2, y2);
	}
	linewidth(1);

	/*Draw half the arrow head*/
	vertices[0][0] = right - ARROWLENGTH;
	vertices[0][1] = y2;
	vertices[1][0] = right - ARROWLENGTH;
	vertices[1][1] = y2 - ARROWHEIGHT;
	vertices[2][0] = right;
	vertices[2][1] = y2;
	polf2i(3, vertices);

	/*Draw half the arrow head*/
	vertices[0][0] = right - ARROWLENGTH;
	vertices[0][1] = y2;
	vertices[1][0] = right - ARROWLENGTH;
	vertices[1][1] = y2 + ARROWHEIGHT;
	vertices[2][0] = right;
	vertices[2][1] = y2;
	polf2i(3, vertices);
    }
    else
    {
	/*Draw a single completion outcell*/
	y2 = top - (top - bottom) * outCell / nCells - (top - bottom) / nCells / 2;
	move2i(cr, y2);
	draw2i(right, y2);
    }
    return ObjTrue;
}
#endif

#ifdef INTERACTIVE
static ObjPtr PressSwitch(object, mouseX, mouseY, flags)
ObjPtr object;
int mouseX, mouseY;
int flags;
/*Press in a switch*/
{
    int left, right, bottom, top;
    ObjPtr backColor;			/*Color of the background*/
    FuncTyp drawContents;		/*Routine to draw the contents*/
    ObjPtr var;
    int value;
    int trackCell;			/*The current cell to track*/
    int lastCell;			/*The last cell tracked in*/
    int nCells;				/*Number of cells in the switch*/

    Get2DIntBounds(object, &left, &right, &bottom, &top);

    if (mouseX < left || mouseX > right || mouseY < bottom || mouseY > top)
    {
	return ObjFalse;
    }

    if (TOOL(flags) == T_HELP)
    {
	ContextHelp(object);
	return ObjTrue;
    }

    /*Get the current value of the control*/
    var = GetIntVar("PressSwitch", object, VALUE);
    if (!var)
    { 
	return ObjFalse;
    }
    value = GetInt(var);

    /*Get ncells*/
    var = GetIntVar("PressSwitch", object, NCELLS);
    if (!var)
    { 
	return ObjFalse;
    }
    nCells = GetInt(var);

    /*Get lastCell*/
    trackCell = lastCell = nCells - 1 - (mouseY - bottom) * nCells / (top - bottom);
    if (lastCell >= nCells) lastCell = nCells - 1;
    if (lastCell < 0) lastCell = 0;
    SetVar(object, VALUE, NewInt(lastCell));
    DrawMe(object);

    InhibitLogging(true);
    while (Mouse(&mouseX, &mouseY))
    {
	if (mouseX < left || mouseX > right || mouseY < bottom || mouseY > top)
	{
	    trackCell = value;
	}
	else
	{
	    trackCell = nCells - 1 - (mouseY - bottom) * nCells / (top - bottom);
	    if (trackCell >= nCells) trackCell = nCells - 1;
	    if (trackCell < 0) trackCell = 0;
	}
	if (trackCell != lastCell)
	{
	    lastCell = trackCell;
	    SetVar(object, VALUE, NewInt(trackCell));
	    DrawMe(object);
	}
    }
    InhibitLogging(false);
    if (logging)
    {
	LogControl(object);
    }
    if (logging || (trackCell != value))
    {
	ChangedValue(object);
    }
    return ObjTrue;
}
#endif

ObjPtr RecalcCorralScroll(corral)
ObjPtr corral;
/*Recalculates the scroll parameters in an icon corral*/
{
    real minx, maxx, miny, maxy;
    real iconLoc[2], bounds[4];
    int left, right, bottom, top;
    ObjPtr var, vScroll, hScroll;
    ObjPtr contents;
    ThingListPtr list;
    
    /*Get bounds*/
    Get2DIntBounds(corral, &left, &right, &bottom, &top);
    bounds[0] = left;
    bounds[1] = right;
    bounds[2] = bottom;
    bounds[3] = top;

    /*Look through icons*/
    contents = GetListVar("RecalcCorralScroll", corral, CONTENTS);
    if (!contents) return ObjFalse;
    list = LISTOF(contents);

    /*Get first icon*/
    if (list)
    {
	var = GetFixedArrayVar("RecalcCorralScroll", list -> thing, ICONLOC, 1, 2L);
	if (!var) return ObjFalse;
	Array2CArray(iconLoc, var);
	list = list -> next;
    }
    else
    {
	iconLoc[0] = 0.0;
	iconLoc[1] = 1.0;
    }
    minx = iconLoc[0] - ICONLEFTBORDER;
    maxx = iconLoc[0] + ICONRIGHTBORDER;
    miny = iconLoc[1] - ICONBOTBORDER;
    maxy = iconLoc[1] + ICONTOPBORDER;

    /*Go through the rest of the icons*/
    while (list)
    {
	var = GetFixedArrayVar("RecalcCorralScroll", list -> thing, ICONLOC, 1, 2L);
	if (!var) return ObjFalse;
	Array2CArray(iconLoc, var);

	if (iconLoc[0] - ICONLEFTBORDER < minx)
	{
	    minx = iconLoc[0] - ICONLEFTBORDER;
	}
	if (iconLoc[0] + ICONRIGHTBORDER > maxx)
	{
	    maxx = iconLoc[0] + ICONRIGHTBORDER;
	}
	if (iconLoc[1] - ICONBOTBORDER < miny)
	{
	    miny = iconLoc[1] - ICONBOTBORDER;
	}
	if (iconLoc[1] + ICONTOPBORDER > maxy)
	{
	    maxy = iconLoc[1] + ICONTOPBORDER;
	}
	list = list -> next;
    }

    /*Adjust hscroll*/
    hScroll = GetVar(corral, HSCROLL);
    if (hScroll)
    {
	SetPortionShown(hScroll, bounds[1] - bounds[0]);
	SetSliderRange(hScroll, maxx - (bounds[1] - bounds[0]), minx, 20.0);
    }

    /*Adjust vscroll*/
    vScroll = GetVar(corral, VSCROLL);
    if (vScroll)
    {
	SetPortionShown(vScroll, bounds[3] - bounds[2]);
	if (GetPredicate(corral, TOPDOWN))
	{
	    SetSliderRange(vScroll, maxy, miny + (bounds[3] - bounds[2]), 20.0);
	}
	else
	{
	    SetSliderRange(vScroll, maxy - (bounds[3] - bounds[2]), miny, 20.0);
	}
    }

    /*Set the virtual bounds*/
    if (hScroll) bounds[1] -= BARWIDTH + CORRALBARBORDER;
    if (vScroll) bounds[3] -= BARWIDTH + CORRALBARBORDER;
    if (minx < bounds[0]) bounds[0] = minx;
    if (maxx > bounds[1]) bounds[1] = maxx;
    if (miny < bounds[2]) bounds[2] = miny;
    if (maxy > bounds[3]) bounds[3] = maxy;
    var = NewRealArray(1, 4L);
    CArray2Array(var, bounds);
    SetVar(corral, VIRTUALBOUNDS, var);

    return ObjTrue;
}

ObjPtr RecalcControlFieldScroll(field)
ObjPtr field;
/*Recalculates the scroll parameters in a control field*/
{
    int minx, maxx, miny, maxy;
    real bounds[4];
    int left, right, bottom, top;
    ObjPtr var, vScroll, hScroll;
    ObjPtr contents;
    ThingListPtr list;
    
    /*Get bounds*/
    Get2DIntBounds(field, &left, &right, &bottom, &top);
    bounds[0] = left;
    bounds[1] = right;
    bounds[2] = bottom;
    bounds[3] = top;

    /*Look through objects*/
    contents = GetListVar("RecalcControlFieldScroll", field, CONTENTS);
    if (!contents) return ObjFalse;
    list = LISTOF(contents);

    /*Get first object*/
    if (list)
    {
	Get2DIntBounds(list -> thing, &left, &right, &bottom, &top);
	list = list -> next;
    }
    else
    {
	left = right = 0;
	bottom = top = 0;
    }
    minx = left - MINORBORDER;
    maxx = right + MINORBORDER;
    miny = bottom - MINORBORDER;
    maxy = top + MINORBORDER;

    /*Go through the rest of the objects*/
    while (list)
    {
	Get2DIntBounds(list -> thing, &left, &right, &bottom, &top);

	minx = MIN(minx, left - MINORBORDER);
	maxx = MAX(maxx, right + MINORBORDER);
	miny = MIN(miny, bottom - MINORBORDER);
	maxy = MAX(maxy, top + MINORBORDER);
	list = list -> next;
    }

    /*Adjust hscroll*/
    hScroll = GetVar(field, HSCROLL);
    if (hScroll)
    {
	SetPortionShown(hScroll, bounds[1] - bounds[0]);
	SetSliderRange(hScroll, maxx - (bounds[1] - bounds[0]), minx, 20.0);
    }

    /*Adjust vscroll*/
    vScroll = GetVar(field, VSCROLL);
    if (vScroll)
    {
	SetPortionShown(vScroll, bounds[3] - bounds[2]);
	SetSliderRange(vScroll, maxy, miny + (bounds[3] - bounds[2]), 20.0);
    }

    /*Set the virtual bounds*/
    if (hScroll) bounds[1] -= BARWIDTH + CORRALBARBORDER;
    if (vScroll) bounds[3] -= BARWIDTH + CORRALBARBORDER;

    if (minx < bounds[0]) bounds[0] = minx;
    if (maxx > bounds[1]) bounds[1] = maxx;
    if (miny < bounds[2]) bounds[2] = miny;
    if (maxy > bounds[3]) bounds[3] = maxy;
    var = NewRealArray(1, 4L);
    CArray2Array(var, bounds);
    SetVar(field, VIRTUALBOUNDS, var);

    return ObjTrue;
}

ObjPtr SetSwitchVal(object, value)
ObjPtr object;
ObjPtr value;
/*Sets the value of the object to value*/
{
    if (IsInt(value))
    {
	int newValue, oldValue;
	ObjPtr var;
	newValue = GetInt(value);
	var = GetIntVar("SetSwitchVal", object, VALUE);
	if (!var) return false;
	oldValue = GetInt(var);
	if (newValue != oldValue)
	{
	    SetVar(object, VALUE, NewInt(newValue));
	    ImInvalid(object);
	    ChangedValue(object);
	}
	if (logging)
	{
	    LogControl(object);
	}
	return ObjTrue;
    }
    else if (IsReal(value))
    {
	int newValue, oldValue;
	ObjPtr var;
	newValue = (int) GetReal(value);
	var = GetIntVar("SetSwitchVal", object, VALUE);
	if (!var) return false;
	oldValue = GetInt(var);
	if (newValue != oldValue)
	{
	    SetVar(object, VALUE, NewInt(newValue));
	    ImInvalid(object);
	    ChangedValue(object);
	}
	if (logging)
	{
	    LogControl(object);
	}
	return ObjTrue;
    }
    else
    {
	return ObjFalse;
    }
}

ObjPtr ReshapeCorral(object, ol, or, ob, ot, left, right, bottom, top)
ObjPtr object;
int ol, or, ob, ot;
int left, right, bottom, top;
/*Reshapes object, which used to exist within owner with edges ol, or, ob, ot
  to one which exists within owner with edges left, right, bottom, top.*/
{
	ObjPtr boundsArray;
	ObjPtr stickyInt;
	real bounds[4];
	real oldWidth, oldHeight;	/*Old width and height*/
	Bool sideLocked[4];		/*True iff side is locked*/
	Bool xStretch, yStretch;	/*Stretchiness in x and y*/
	int stickiness;		/*Side stickiness of the object*/
	real oldBounds[4];		/*Old bounds of the object*/
	ObjPtr contents;		/*Contents of the object, if any*/
	ObjPtr scrollBar;		/*Scroll bar to reshape*/
	real scrollBounds[4];		/*Bounds of scroll bar*/
	real wr, hr;			/*Width and height ratios*/

	wr = ((real) (right - left)) / ((real) (or - ol));
	hr = ((real) (top - bottom)) / ((real) (ot - ob));

	boundsArray = GetVar(object, BOUNDS);
	if (!boundsArray || !IsArray(boundsArray) || RANK(boundsArray) != 1 ||
	    DIMS(boundsArray)[0] != 4)
	{
	    return;
	}
	Array2CArray(bounds, boundsArray);
	Array2CArray(oldBounds, boundsArray);
	oldWidth = bounds[1] - bounds[0];
	oldHeight = bounds[3] - bounds[2];

	/*Get the object's stickiness*/
	stickyInt = GetVar(object, STICKINESS);
	if (stickyInt && IsInt(stickyInt))
	{
	    stickiness = GetInt(stickyInt);
	}
	else
	{
	    stickiness = 0;
	}

	if ((stickiness & STICKYLEFT) || (stickiness & FLOATINGLEFT))
	{
	    if (stickiness & FLOATINGLEFT)
	    {
		bounds[0] = (bounds[0] - ol) * wr + left;
	    }
	    else
	    {
		bounds[0] += left - ol;
	    }
	    if (!((stickiness & STICKYRIGHT) || (stickiness & FLOATINGRIGHT)))
	    {
		bounds[1] = bounds[0] + oldWidth;
	    }
	}
	if ((stickiness & STICKYRIGHT) || (stickiness & FLOATINGRIGHT))
	{
	    if (stickiness & FLOATINGRIGHT)
	    {
		bounds[1] = (bounds[1] - ol) * wr + left;
	    }
	    else
	    {
		bounds[1] += right - or;
	    }
	    if (!((stickiness & STICKYLEFT) || (stickiness & FLOATINGLEFT)))
	    {
		bounds[0] = bounds[1] - oldWidth;
	    }
	}

	if ((stickiness & STICKYBOTTOM) || (stickiness & FLOATINGBOTTOM))
	{
	    if (stickiness & FLOATINGBOTTOM)
	    {
		bounds[2] = (bounds[2] - ob) * hr + bottom;
	    }
	    else
	    {
		bounds[2] += bottom - ob;
	    }
	    if (!((stickiness & STICKYTOP) || (stickiness & FLOATINGTOP)))
	    {
		bounds[3] = bounds[2] + oldHeight;
	    }
	}
	if ((stickiness & STICKYTOP) || (stickiness & FLOATINGTOP))
	{
	    if (stickiness & FLOATINGTOP)
	    {
		bounds[3] = (bounds[3] - ob) * hr + bottom;
	    }
	    else
	    {
		bounds[3] += top - ot;
	    }
	    if (!((stickiness & STICKYBOTTOM) || (stickiness & FLOATINGBOTTOM)))
	    {
		bounds[2] = bounds[3] - oldHeight;
	    }
	}

	/*We've got a new bounds, put it back*/
	boundsArray = NewRealArray(1, 4L);
	CArray2Array(boundsArray, bounds);
	SetVar(object, BOUNDS, boundsArray);

	/*Reshape the scroll bars*/
	scrollBar = GetVar(object, VSCROLL);
	if (scrollBar)
	{
	    /*Get the bounds*/
	    real width;
	    boundsArray = GetFixedArrayVar("ReshapeCorral", scrollBar, BOUNDS, 1, 4L);
	    if (!boundsArray) return ObjFalse;

	    Array2CArray(scrollBounds, boundsArray);
	    width = scrollBounds[1] - scrollBounds[0];
	    if (scrollBounds[0] > oldBounds[1])
	    {
		/*It was on the right*/
		scrollBounds[0] = bounds[1] + (scrollBounds[0] - oldBounds[1]);
		scrollBounds[1] = scrollBounds[0] + width;
		scrollBounds[2] = bounds[2];
		scrollBounds[3] = bounds[3];
		
		boundsArray = NewRealArray(1, 4L);
		CArray2Array(boundsArray, scrollBounds);
		SetVar(scrollBar, BOUNDS, boundsArray);
	    }
	    else
	    {
		/*It was on the left*/
		scrollBounds[1] = bounds[0] + (scrollBounds[1] - oldBounds[0]);
		scrollBounds[0] = scrollBounds[1] - width;
		scrollBounds[2] = bounds[2];
		scrollBounds[3] = bounds[3];
		
		boundsArray = NewRealArray(1, 4L);
		CArray2Array(boundsArray, scrollBounds);
		SetVar(scrollBar, BOUNDS, boundsArray);
	    }
	}
	scrollBar = GetVar(object, HSCROLL);
	if (scrollBar)
	{
	    /*Get the bounds*/
	    real height;
	    boundsArray = GetFixedArrayVar("ReshapeCorral", scrollBar, BOUNDS, 1, 4L);
	    if (!boundsArray) return ObjFalse;

	    Array2CArray(scrollBounds, boundsArray);
	    height = scrollBounds[3] - scrollBounds[2];
	    if (scrollBounds[2] > oldBounds[3])
	    {
		/*It was on the top*/
		scrollBounds[0] = bounds[0];
		scrollBounds[1] = bounds[1];
		scrollBounds[2] = bounds[3] + (scrollBounds[2] - oldBounds[3]);
		scrollBounds[3] = scrollBounds[2] + height;
		
		boundsArray = NewRealArray(1, 4L);
		CArray2Array(boundsArray, scrollBounds);
		SetVar(scrollBar, BOUNDS, boundsArray);
	    }
	    else
	    {
		/*It was on the bottom*/
		scrollBounds[0] = bounds[0];
		scrollBounds[1] = bounds[1];
		scrollBounds[3] = bounds[2] + (scrollBounds[3] - oldBounds[2]);
		scrollBounds[2] = scrollBounds[3] - height;
		
		boundsArray = NewRealArray(1, 4L);
		CArray2Array(boundsArray, scrollBounds);
		SetVar(scrollBar, BOUNDS, boundsArray);
	    }
	}
	RecalcScroll(object);
    return ObjTrue;
}

ObjPtr NewPanel(superClass, left, right, bottom, top)
ObjPtr superClass;
int left, right, bottom, top;
/*Makes a new panel with bounds left, right, bottom, top*/
{
    ObjPtr retVal;
    ObjPtr contents;

    retVal = NewObject(superClass ? superClass : panelClass, 0);
    Set2DIntBounds(retVal, left, right, bottom, top);
    SetVar(retVal, CONTENTS, contents = NewList());
    SetVar(contents, PARENT, retVal);
    return retVal;
}

ObjPtr NewControlField(left, right, bottom, top, name, scrollBars)
int left, right, bottom, top;
char *name;
int scrollBars;
/*Makes a new control field*/
{
    ObjPtr retVal;
    ObjPtr contents;
    ObjPtr hScrollBar = NULLOBJ, vScrollBar = NULLOBJ;

    retVal = NewObject(controlFieldClass, 0);
    SetVar(retVal, CONTENTS, contents = NewList());
    SetVar(contents, PARENT, retVal);
    SetVar(retVal, NAME, NewString(name));

    if (scrollBars & BARRIGHT)
    {
	right -= BARWIDTH + CORRALBARBORDER;
    }
    if (scrollBars & BARLEFT)
    {
	left += BARWIDTH + CORRALBARBORDER;
    }

    if (scrollBars & BARBOTTOM)
    {
	bottom += BARWIDTH + CORRALBARBORDER;
    }

    if (scrollBars & BARRIGHT)
    {
	vScrollBar = NewScrollbar(right + CORRALBARBORDER,
		    right + CORRALBARBORDER + BARWIDTH,
		    bottom,
		    top, "VScroll");
	SetVar(retVal, VSCROLL, vScrollBar);
	SetVar(vScrollBar, PARENT, retVal);
    }
    else if (scrollBars & BARLEFT)
    {
	vScrollBar = NewScrollbar(left - CORRALBARBORDER - BARWIDTH,
		    left - CORRALBARBORDER,
		    bottom,
		    top, "VScroll");
	SetVar(retVal, VSCROLL, vScrollBar);
	SetVar(vScrollBar, PARENT, retVal);
    }

    if (scrollBars & BARBOTTOM)
    {
	hScrollBar = NewScrollbar(left,
		    right,
		    bottom - CORRALBARBORDER - BARWIDTH,
		    bottom - CORRALBARBORDER,
		    "HScroll");
	SetVar(retVal, HSCROLL, hScrollBar);
	SetVar(hScrollBar, PARENT, retVal);
    }

    if (scrollBars & OBJECTSFROMTOP)
    {
	SetVar(retVal, TOPDOWN, ObjTrue);
	if (vScrollBar)
	{
	    SetSliderRange(vScrollBar, 0.0, -10.0, 0.0);
	}
    }

    Set2DIntBounds(retVal, left, right, bottom, top);
    return retVal;
}

ObjPtr NewIconCorral(superClass, left, right, bottom, top, scrollBars)
ObjPtr superClass;
int left, right, bottom, top;
int scrollBars;
/*Makes a new panel with bounds left, right, bottom, top
  if superClass non-nil, uses that as the superclass of the object
  scrollBars is a bunch of flags for scroll bars AND OTHER STUFF*/
{
    ObjPtr retVal;
    ObjPtr hScrollBar = NULLOBJ, vScrollBar = NULLOBJ;
    ObjPtr contents;

    retVal = NewObject(superClass ? superClass : corralClass, 0);
    SetVar(retVal, CONTENTS, contents = NewList());
    SetVar(contents, PARENT, retVal);

    if (scrollBars & BARRIGHT)
    {
	right -= BARWIDTH + CORRALBARBORDER;
    }
    if (scrollBars & BARLEFT)
    {
	left += BARWIDTH + CORRALBARBORDER;
    }

    if (scrollBars & BARBOTTOM)
    {
	bottom += BARWIDTH + CORRALBARBORDER;
    }
    
    if (scrollBars & BARRIGHT)
    {
	vScrollBar = NewScrollbar(right + CORRALBARBORDER,
		    right + CORRALBARBORDER + BARWIDTH,
		    bottom,
		    top, "VScroll");
	SetVar(retVal, VSCROLL, vScrollBar);
	SetVar(vScrollBar, PARENT, retVal);
    }
    else if (scrollBars & BARLEFT)
    {
	vScrollBar = NewScrollbar(left - CORRALBARBORDER - BARWIDTH,
		    left - CORRALBARBORDER,
		    bottom,
		    top, "VScroll");
	SetVar(retVal, VSCROLL, vScrollBar);
	SetVar(vScrollBar, PARENT, retVal);
    }

    if (scrollBars & BARBOTTOM)
    {
	hScrollBar = NewScrollbar(left,
		    right,
		    bottom - CORRALBARBORDER - BARWIDTH,
		    bottom - CORRALBARBORDER,
		    "HScroll");
	SetVar(retVal, HSCROLL, hScrollBar);
	SetVar(hScrollBar, PARENT, retVal);
    }

    if (scrollBars & OBJECTSFROMTOP)
    {
	SetVar(retVal, TOPDOWN, ObjTrue);
	if (vScrollBar)
	{
	    SetSliderRange(vScrollBar, 0.0, -10.0, 0.0);
	}
    }

    Set2DIntBounds(retVal, left, right, bottom, top);

    return retVal;
}

ObjPtr NewSwitch(left, right, bottom, top, nCells, outCell, initValue, name)
int left, right, bottom, top;
int nCells, outCell, initValue;
char *name;
/*Makes a new switch with bounds left, right, bottom, top
  nCells is the number of cells, outCell is which cell is the output,
  initValue is the initial switch*/
{
    ObjPtr retVal;

	retVal = NewObject(switchClass, 0);
	Set2DIntBounds(retVal, left, right, bottom, top);
	SetVar(retVal, NCELLS, NewInt(nCells));
	SetVar(retVal, OUTCELL, NewInt(outCell));
	SetVar(retVal, VALUE, NewInt(initValue));
	SetVar(retVal, NAME, NewString(name));
	return retVal;
}

#ifdef GRAPHICS
#ifdef PROTO
void DrawPitFrame(int left, int right, int bottom, int top, int depth)
#else
void DrawPitFrame(left, right, bottom, top, depth)
int left, right, bottom, top;
int depth;
#endif
/*Draws the frame of a pit*/
{
    Coord v[4][2];

    /*Draw top edge*/
    v[0][0] = left - 0.5;
    v[0][1] = top + 0.5;
    v[1][0] = right - 0.5;
    v[1][1] = top + 0.5;
    v[2][0] = right - depth + 0.5;
    v[2][1] = top - depth + 0.5;
    v[3][0] = left + depth + 0.5;
    v[3][1] = top - depth + 0.5;
    SetUIColor(UIBOTTOMEDGE);
    polf2(4, v);

    /*Draw left edge*/
    v[0][0] = left - 0.5;
    v[0][1] = top + 0.5;
    v[1][0] = left + depth - 0.5;
    v[1][1] = top - depth + 0.5;
    v[2][0] = left + depth - 0.5;
    v[2][1] = bottom + depth - 0.5;
    v[3][0] = left - 0.5;
    v[3][1] = bottom - 0.5;
    SetUIColor(UIRIGHTEDGE);
    polf2(4, v);

    /*Draw right edge*/
    v[0][0] = right - depth + 0.5;
    v[0][1] = top - depth + 0.5;
    v[1][0] = right + 0.5;
    v[1][1] = top + 0.5;
    v[2][0] = right + 0.5;
    v[2][1] = bottom - 0.5;
    v[3][0] = right - depth + 0.5;
    v[3][1] = bottom + depth - 0.5;
    SetUIColor(UILEFTEDGE);
    polf2(4, v);

    /*Draw bottom edge*/
    v[0][0] = left + depth - 0.5;
    v[0][1] = bottom + depth - 0.5;
    v[1][0] = right - depth + 0.5;
    v[1][1] = bottom + depth - 0.5;
    v[2][0] = right + 0.5;
    v[2][1] = bottom - 0.5;
    v[3][0] = left - 0.5;
    v[3][1] = bottom - 0.5;
    SetUIColor(UITOPEDGE);
    polf2(4, v);

}
#endif

#ifdef PROTO
real ChooseGoodStep(real min, real max)
#else
real ChooseGoodStep(min, max)
real min;
real max;
#endif
/*Choose a good step to get between min and max*/
{
    real diff;
    int iexp;

    diff = max - min;
    iexp = ((int) (1000.0 + log10((double) diff))) - 1000;
    return (real) pow((double) 10.0, (double) iexp);
}

#ifdef PROTO
void ChooseGoodSliderParams(real *min, real *max, real *bigStep, real *littleStep, real *anchor)
#else
void ChooseGoodSliderParams(min, max, bigStep, littleStep, anchor)
real *min, *max, *bigStep, *littleStep, *anchor;
#endif
/*Chooses good slider params for data in min and max.  Sets all four params*/
{
    real diff;

    if (*max <= *min)
    {
	*max = *min + 1.0;
    }

    if (*min >= 0.0 && *max >= 0.0)
    {
	real powerMin, powerMax;
	int iexpMin, iexpMax;
	int multiplier;

	iexpMax = ((int) (1000.0 + log10((double) *max))) - 1000;
	if (*min == 0.0)
	{
	    iexpMin = iexpMax - 1;
	}
	else
	{
	    iexpMin = ((int) (1000.0 + log10((double) *min))) - 1000;
	}
	powerMin = (real) pow((double) 10.0, (double) iexpMin);
	powerMax = (real) pow((double) 10.0, (double) iexpMax);

	/*They're on the high side*/
	if (iexpMin < iexpMax)
	{
	    /*Lower bound might as well be zero*/
	    *min = 0.0;
	}
	else
	{
	    /*Lower min to nearest lower power of ten*/
	    multiplier = *min / powerMin;
	    *min = multiplier * powerMin;
	}
	/*Raise max to the next power of ten*/
	multiplier = *max / powerMax;
	if (*max > multiplier * powerMax)
	{
	    *max = (multiplier + 1) * powerMax;
	}
	else
	{
	    *max = multiplier * powerMax;
	}
	
	*bigStep = powerMax;
	*littleStep = *bigStep * 0.1;
	*anchor = *min;
    }
    else if (*min <= 0.0 && *max <= 0.0)
    {
	real powerMin, powerMax;
	int iexpMin, iexpMax;
	int multiplier;
	real temp;

	/*They're on the low side*/
	temp = *max;
	*max = -*min;
	*min = -temp;

	iexpMax = ((int) (1000.0 + log10((double) *max))) - 1000;
	if (*min == 0.0)
	{
	    iexpMin = iexpMax - 1;
	}
	else
	{
	    iexpMin = ((int) (1000.0 + log10((double) *min))) - 1000;
	}
	powerMin = (real) pow((double) 10.0, (double) iexpMin);
	powerMax = (real) pow((double) 10.0, (double) iexpMax);

	if (iexpMin < iexpMax)
	{
	    /*Lower bound might as well be zero*/
	    *min = 0.0;
	}
	else
	{
	    /*Lower min to nearest lower power of ten*/
	    multiplier = *min / powerMin;
	    *min = multiplier * powerMin;
	}
	/*Raise max to the next power of ten*/
	multiplier = *max / powerMax;
	if (*max > multiplier * powerMax)
	{
	    *max = (multiplier + 1) * powerMax;
	}
	else
	{
	    *max = multiplier * powerMax;
	}

	temp = *max;
	*max = -*min;
	*min = -temp;

	*bigStep = powerMax;
	*littleStep = *bigStep * 0.1;
	*anchor = *min;
    }
    else if (*min <= 0.0 && *max >= 0.0)
    {
	/*They span zero*/
	int multiplier;
	int iexp;
	real power;

	iexp = ((int) (1000.0 + log10((double) (*max - *min)))) - 1000;
	power = (real) pow((double) 10.0, (double) iexp);

	*min = -*min;

	/*Raise max to the next power of ten*/
	multiplier = *max / power;
	if (*max > multiplier * power)
	{
	    *max = (multiplier + 1) * power;
	}
	else
	{
	    *max = multiplier * power;
	}

	/*Raise min to the next power of ten*/
	if (*min > multiplier * power)
	{
	    *min = (multiplier + 1) * power;
	}
	else
	{
	    *min = multiplier * power;
	}

	*bigStep = power;
	*littleStep = *bigStep * 0.1;
	*min = -*min;
	*anchor = 0.0;
    }
}

static ObjPtr MakeCorralHelpString(object, class)
ObjPtr object, class;
/*Makes a corral help string for object in class*/
{
    if (GetPredicate(object, ONEICON))
    {
	SetVar(class, HELPSTRING, NewString("You can select a single icon within this corral by clicking on it with the left mouse button.  \
Only one icon can be selected at any given time."));
    }
    else if (GetPredicate(object, SINGLECORRAL))
    {
	SetVar(class, HELPSTRING, NewString("This corral can only hold a single icon.  You can select the icon by clicking it \
with the left mouse button \
or deselect it by clicking on the background.  If you drag another icon into this corral, \
it will replace any icon that is already there.  The icon can be deleted with the Delete \
item in the Object menu."));
    }
    else
    {
	SetVar(class, HELPSTRING, NewString("You can select a single icon within this corral by clicking it \
with the left mouse button.  \
You can select more than one icon by holding the Shift key down while you click on additional icons.  \
You can also click outside an icon and drag a rectangle around icons you would like to select.\n\
\n\
Items in the Object menu and some action buttons in the window \
will work on all selected objects."));
    }
    return ObjTrue;
}

void InitControls()
/*Initializes controls*/
{
    controlClass = NewObject(NULLOBJ, 0);
    SetVar(controlClass, TYPESTRING, NewString("control"));
    SetVar(controlClass, ACTIVATED, ObjTrue);
    AddToReferenceList(controlClass);

    panelClass = NewObject(controlClass, 0);
    AddToReferenceList(panelClass);
    SetVar(panelClass, NAME, NewString("Panel"));
#ifdef GRAPHICS
    SetMethod(panelClass, DRAW, DrawPanel);
#endif
#ifdef INTERACTIVE
    SetMethod(panelClass, PRESS, PressPanel);
    SetMethod(panelClass, DROPOBJECTS, DropInPanel);
    SetMethod(panelClass, KEYDOWN, KeyDownContents);
#endif

    greyPanelClass = NewObject(panelClass, 0);
    AddToReferenceList(greyPanelClass);
    SetVar(greyPanelClass, BACKGROUND, NewInt(UIBACKGROUND));

    fieldClass = NewObject(controlClass, 0);
    AddToReferenceList(fieldClass);
#ifdef GRAPHICS
    SetMethod(fieldClass, DRAW, DrawField);
#endif
#ifdef INTERACTIVE
    SetMethod(fieldClass, PRESS, PressField);
#endif
    SetVar(fieldClass, NAME, NewString("Field"));
    SetMethod(fieldClass, FINDOBJECT, FindObjectField);
    SetVar(fieldClass, TYPESTRING, NewString("field"));
#ifdef INTERACTIVE
    SetMethod(fieldClass, PRESSCONTENTS, PressFieldContents);
    SetMethod(fieldClass, KEYDOWN, KeyDownContents);
#endif

    corralClass = NewObject(fieldClass, 0);
    AddToReferenceList(corralClass);
    SetVar(corralClass, NAME, NewString("Corral"));
    SetVar(corralClass, VIEWBYWHAT, NewInt(VB_ICON));
#ifdef GRAPHICS
    SetMethod(corralClass, DRAWCONTENTS, DrawCorralContents);
#endif
#ifdef INTERACTIVE 
    SetMethod(corralClass, PRESSCONTENTS, PressCorralContents);
    SetMethod(corralClass, DROPOBJECTS, DropInCorral);
#endif
    SetMethod(corralClass, RECALCSCROLL, RecalcCorralScroll);
    SetMethod(corralClass, RESHAPE, ReshapeCorral);
    SetVar(corralClass, TYPESTRING, NewString("icon corral"));
    SetMethod(corralClass, MAKE1HELPSTRING, MakeCorralHelpString);
    SetMethod(corralClass, YOURENOTCURRENT, CorralNotCurrent);

    switchClass = NewObject(controlClass, 0);
    AddToReferenceList(switchClass);
    SetVar(switchClass, NAME, NewString("Switch"));
#ifdef GRAPHICS
    SetMethod(switchClass, DRAW, DrawSwitch);
#endif
#ifdef INTERACTIVE
    SetMethod(switchClass, PRESS, PressSwitch);
#endif
    SetMethod(switchClass, SETVAL, SetSwitchVal);
    SetVar(switchClass, TYPESTRING, NewString("data switch"));
    SetVar(switchClass, HELPSTRING, NewString("To change the data path within a switch, click on the data path you desire."));

    controlFieldClass = NewObject(fieldClass, 0);
    AddToReferenceList(controlFieldClass);
#ifdef GRAPHICS
    SetMethod(controlFieldClass, DRAW, DrawControlField);
    SetMethod(controlFieldClass, DRAWCONTENTS, DrawControlFieldContents);
#endif
    SetMethod(controlFieldClass, RECALCSCROLL, RecalcControlFieldScroll);
    SetVar(controlFieldClass, TYPESTRING, NewString("field"));

    InitButtons();
    InitSliders();
    InitTextBoxes();
    InitPerspecControls();
    InitTitleBoxes();
}
 
void KillControls()
/*Kills the controls*/
{
    KillTitleBoxes();
    KillPerspecControls();
    KillTextBoxes();
    KillSliders();
    KillButtons();
    DeleteThing(controlFieldClass);
    DeleteThing(switchClass);
    DeleteThing(corralClass);
    DeleteThing(fieldClass);
    DeleteThing(greyPanelClass);
    DeleteThing(panelClass);
    DeleteThing(controlClass);
}

