/*ScianDatasets.c
  Eric Pepke
  August 17, 1990
  Stuff for Datasets
*/

#include "Scian.h"
#include "ScianTypes.h"
#include "ScianLists.h"
#include "ScianIcons.h"
#include "ScianWindows.h"
#include "ScianObjWindows.h"
#include "ScianVisWindows.h"
#include "ScianButtons.h"
#include "ScianTextBoxes.h"
#include "ScianTitleBoxes.h"
#include "ScianControls.h"
#include "ScianDatasets.h"
#include "ScianErrors.h"
#include "ScianColors.h"
#include "ScianDialogs.h"
#include "ScianVisObjects.h"
#include "ScianStyle.h"
#include "ScianSpaces.h"
#include "ScianIDs.h"
#include "ScianArrays.h"
#include "ScianMethods.h"
#include "ScianTimers.h"
#include "ScianDepend.h"
#include "ScianFiles.h"
#include "ScianFileSystem.h"

#define VISMENU
#define SHOWCTLMENU

ObjPtr geometryClass;			/*Class for geometry object*/
ObjPtr datasetClass, data3DScalar, data2DScalar, data1DVector, data3DUnstructSurface;
ObjPtr iconDataset;
ObjPtr icon1DVector, icon2DVector, icon3DVector, icon4DVector;
ObjPtr icon1DScalar, icon2DScalar, icon3DScalar, icon4DScalar;
ObjPtr iconColorPalette;

Bool timedDatasets = true;		/*True iff reading timed datasets*/
ObjPtr dataFormClass;
ObjPtr allDatasets = NULLOBJ;		/*All datasets*/
static ObjPtr dsLocArray = NULLOBJ;	/*Location of icon for dataset*/

Bool onePalette = false;
ObjPtr commonPalette;

DatasetBuffer curFields[MAXNCURFIELDS];

ObjPtr GetLongName(dataset)
ObjPtr dataset;
/*Gets the long name of dataset*/
{
    FuncTyp method;
    method = GetMethod(dataset, GETLONGNAME);
    if (method)
    {
	return (*method)(dataset);
    }
    else
    {
	return NULLOBJ;
    }
}

ObjPtr GetPlainDatasetLongName(dataset)
ObjPtr dataset;
/*Gets the long name of a plain dataset*/
{
    return GetVar(dataset, NAME);
}

static void CleanCurField(k)
int k;
/*Cleans the current field number k*/
{
    if (curFields[k] . components)
    {
	int i;
	for (i = 0; i < curFields[k] . nComponents; ++i)
	{
	    if (curFields[k] . components[i] . indices)
	    {
		free(curFields[k] . components[i] . indices);
		curFields[k] . components[i] . indices = 0;
	    }
	    if (curFields[k] . components[i] . dimensions)
	    {
		free(curFields[k] . components[i] . dimensions);
		curFields[k] . components[i] . dimensions = 0;
	    }
	    if (curFields[k] . components[i] . steps)
	    {
		free(curFields[k] . components[i] . steps);
		curFields[k] . components[i] . steps = 0;
	    }
	}
	free(curFields[k] . components);
    }
    curFields[k] . nComponents = 0;
    curFields[k] . components = (Component *) 0;
}

ObjPtr MakeDatasetCurData(dataset)
ObjPtr dataset;
/*Makes the dataset's CURDATA.  Returns ObjTrue if it had to make it*/
{
    ObjPtr data = NULLOBJ, curData = NULLOBJ;
    ObjPtr retVal;

    retVal = ObjFalse;

    data = GetVar(dataset, DATA);
    if (data)
    {
	curData = GetVar(data, CURDATA);
	if (!curData)
	{
	    /*For old-fashioned objects that don't know about CURDATA*/
	    curData = data;
	}
    }
    else
    {
	curData = data;
    }
    if (!curData)
    {
	ReportError("MakeDatasetCurData", "CURDATA of DATA is missing!");
	retVal = ObjTrue;
	return retVal;
    }
    SetVar(dataset, CURDATA, curData);
    return ObjTrue;
}

ObjPtr MakeDatasetTimesteps(dataset)
ObjPtr dataset;
/*Makes the dataset's TIMESTEPS.  Returns ObjTrue if it had to make it*/
{
    ObjPtr data = NULLOBJ, timesteps = NULLOBJ;

    data = GetVar(dataset, DATA);
    if (data)
    {
	MakeVar(data, TIMESTEPS);
	timesteps = GetVar(data, TIMESTEPS);
    }
    else
    {
	SetVar(dataset, TIMESTEPS, NULLOBJ);
	return ObjTrue;
    }
    SetVar(dataset, TIMESTEPS, timesteps);
    return ObjTrue;
}

#ifdef PROTO
Bool SampleSpatField(int dataField, int formField,
	int nResultComp, real resultComp[], int nSpatIndices, 
	real spatIndices[], Bool interpolate)
#else
Bool SampleSpatField(dataField, formField, nResultComp, resultComp, nSpatIndices, spatIndices, interpolate)
int dataField, formField;
int nResultComp;
real resultComp[];
int nSpatIndices;
real spatIndices[];
Bool interpolate;
#endif
/*Samples a field dataField defined over formField using nSpatIndices spatIndices and puts
  the result in resultComp components*/
{
    long *topIndices;
    real *rTopIndices;
    long indexBuffer[10];
    real rIndexBuffer[10];
    int nTopIndices;
    int k;

    nTopIndices = curFields[formField] . topDim;
    if (nTopIndices != curFields[dataField] . topDim)
    {
	ReportError("SampleSpatField", "Topological dimension mismatch");
	return false;
    }

    if (nResultComp != curFields[dataField] . nComponents)
    {
	ReportError("SampleSpatField", "Wrong number of result components");
	return false;
    }

    if (nTopIndices <= 10)
    {
	topIndices = &(indexBuffer[0]);
	rTopIndices = &(rIndexBuffer[0]);
    }
    else
    {
	topIndices = malloc(nTopIndices * sizeof(long));
	rTopIndices = malloc(nTopIndices * sizeof(real));
    }
    SampleToTopIndex(formField, nTopIndices, rTopIndices, nSpatIndices, spatIndices);
    for (k = 0; k < nTopIndices; ++k)
    {
	topIndices[k] = rTopIndices[k] + 0.5;
    }

    for (k = 0; k < nResultComp; ++k)
    {
	if (interpolate)
	{
	    resultComp[k] = InterpolateFieldComponent(dataField, k, rTopIndices);
	}
	else
	{
	    resultComp[k] = SelectFieldComponent(dataField, k, topIndices);
	}
    }

    if (nTopIndices > 10)
    {
	free(topIndices);
	free(rTopIndices);
    }

    return true;
}

Bool SampleToTopIndex(whichField, nIndices, indices, nComponents, sample)
int whichField;
int nIndices;
real indices[];
int nComponents;
real sample[];
/*Finds and returns in indices the closest index in whichField to sample.
  The arrays had better be big enough*/
{
    int i, comp, k, index, myIndex;
    int indicesAssigned;
    Bool succeed;
    Bool inHere, otherFree, others;
    long *longIndices;

    /*Find the number of components, should be == size of sample*/
    if (nComponents < curFields[whichField] . nComponents)
    {
	ReportError("FindClosestSample", "Not enough components passed");
	return false;
    }

    longIndices = malloc(nIndices * sizeof(long));

    /*Make all indices free*/
    for (k = 0; k < nIndices; ++k)
    {
	indices[k] = -1.0;
	longIndices[k] = -1;
    }

    /*Number of indices which have been assigned, start at 0*/
    indicesAssigned = 0;

    do
    {
	succeed = false;
	/*Search for an index and component such that
	  1) The index is free
	  2) The index appears in the component
	  3) No other component is free*/

	for (i = 0; i < nIndices; ++i)
	{
	    /*See if it's free*/
	    if (longIndices[i] < 0)
	    {
		/*See if it appears in any components with free index*/
		for (comp = 0; comp < curFields[whichField] . nComponents; ++comp)
		{
		    inHere = others = otherFree = false;
		    for (k = 0;
			 k < curFields[whichField] . components[comp] . nIndices;
			 ++k)
		    {
			index = curFields[whichField] . components[comp] . indices[k];
			if (index == i)
			{
			    myIndex = k;
			    inHere = true;
			}
			else if (index >= 0)
			{
			    if (longIndices[index] < 0)
			    {
				otherFree = true;
			    }
			    else
			    {
				others = true;
			    }
			}
		    }

		    if (inHere && !otherFree)
		    {
			/*Found one which is free*/

			if (!others &&
			    0 == (curFields[whichField] . components[comp] . flags & (CF_MONOTONIC | CF_NOTMONOTONIC)))
			{
			    int testo;
			    real next, last;

			    testo = 0;
			    
			    /*Might be monotonic.  Let's check*/
			    longIndices[i] = 0;
			    last = SelectFieldComponent(whichField, comp, longIndices);
			    for (longIndices[i] = 1;
			         longIndices[i] < curFields[whichField] . components[comp] . dimensions[myIndex];
			         ++(longIndices[i]))
			    {
				next = SelectFieldComponent(whichField, comp, longIndices);
				if (next > last)
				{
				    testo |= 1;
				}
				else if (last > next)
				{
				    testo |= 2;
				}
				last = next;
			    }
			    if (testo < 3)
			    {
				curFields[whichField] . components[comp] . flags |= CF_MONOTONIC;
			    }
			    else
			    {
				curFields[whichField] . components[comp] . flags |= CF_NOTMONOTONIC;
			    }
			}

			/*Hey!  Found it!*/

			if (curFields[whichField] . components[comp] . flags & CF_MONOTONIC)
			{
			    long beg, end, mid;
			    real begFit, endFit, midFit, diff1, diff2;

			    beg = 0;
			    end = curFields[whichField] . components[comp] . dimensions[myIndex] - 1;
			    longIndices[i] = beg;
			    begFit = SelectFieldComponent(whichField, comp, longIndices);
			    longIndices[i] = end;
			    endFit = SelectFieldComponent(whichField, comp, longIndices);

			    if (begFit < endFit)
			    {
				/*Positive monotonic*/
				if (sample[comp] <= begFit)
				{
				    longIndices[i] = beg;
				    indices[i] = beg;
				}
				else if (sample[comp] >= endFit)
				{
				    longIndices[i] = end;
				    indices[i] = end;
				}
				else
				{
				    while (beg + 1 < end)
				    {
					mid = (beg + end) / 2;
					if (mid <= beg) mid = beg + 1;
					else if (mid >= end) mid = end - 1;
					longIndices[i] = mid;
					midFit = SelectFieldComponent(whichField, comp, longIndices);
					if (midFit > sample[comp])
					{
					    endFit = midFit;
					    end = mid;
					}
					else
					{
					    begFit = midFit;
					    beg = mid;
					}
				    }
				    diff1 = begFit - sample[comp];
				    diff1 = ABS(diff1);
				    diff2 = endFit - sample[comp];
				    diff2 = ABS(diff2);
				    longIndices[i] = (diff1 > diff2) ? end : beg;
				    indices[i] = ((beg * diff2) + (end * diff1)) / (diff1 + diff2);
				}
				succeed = true;
				++indicesAssigned;
			    }
			    else
			    {
				/*Negative monotonic*/
				if (sample[comp] >= begFit)
				{
				    longIndices[i] = beg;
				    indices[i] = beg;
				}
				else if (sample[comp] <= endFit)
				{
				    longIndices[i] = end;
				    indices[i] = end;
				}
				else
				{
				    while (beg + 1 < end)
				    {
					mid = (beg + end) / 2;
					if (mid >= beg) mid = beg + 1;
					else if (mid <= end) mid = end - 1;
					longIndices[i] = mid;
					midFit = SelectFieldComponent(whichField, comp, longIndices);
					if (midFit < sample[comp])
					{
					    endFit = midFit;
					    end = mid;
					}
					else
					{
					    begFit = midFit;
					    beg = mid;
					}
				    }
				    diff1 = begFit - sample[comp];
				    diff1 = ABS(diff1);
				    diff2 = endFit - sample[comp];
				    diff2 = ABS(diff2);
				    longIndices[i] = (diff1 > diff2) ? end : beg;
				    indices[i] = ((beg * diff2) + (end * diff1)) / (diff1 + diff2);
				}
				succeed = true;
				++indicesAssigned;
			    }
			}
			else
			{
			    /*It's not monotonic.  Do a linear search*/
			    long bestFit;
			    real bestDist;
			    real diff;

			    succeed = true;
			    ++indicesAssigned;

			    longIndices[i] = 0;
			    bestFit = 0;
			    diff = SelectFieldComponent(whichField, comp, longIndices) -
				sample[comp];
			    bestDist = ABS(diff);

			    for (longIndices[i] = 1;
				 longIndices[i] < curFields[whichField] . components[comp] . dimensions[myIndex];
				 ++(longIndices[i]))
			    {
				diff = SelectFieldComponent(whichField, comp, longIndices) -
				    sample[comp];
				diff = ABS(diff);
				if (diff < bestDist)
				{
				    bestDist = diff;
				    bestFit = longIndices[i];
				}
			    }
			    longIndices[i] = bestFit;
			    if (longIndices[i] <= 0)
			    {
				real sCenter, sPlus;
				/*At left, see if it's between 0 and 1*/

				sCenter = SelectFieldComponent(whichField, comp, longIndices);
				++longIndices[i];
				sPlus = SelectFieldComponent(whichField, comp, longIndices);
				--longIndices[i];
				if (sCenter <= sample[comp] <= sPlus)
				{
				    indices[i] =
					(sPlus - sample[comp]) * longIndices[i] +
					(sample[comp] - sCenter) * (longIndices[i] + 1);
				}
				else if (sPlus <= sample[comp] <= sCenter)
				{
				    indices[i] =
					(sCenter - sample[comp]) * longIndices[i] +
					(sample[comp] - sPlus) * (longIndices[i] + 1);
				}
				else
				{
				    indices[i] = longIndices[i];
				}
			    }
			    else if (longIndices[i] >= curFields[whichField] . components[comp] . dimensions[myIndex])
			    {
				real sCenter, sMinus;
				/*At right, see if it's between right and right - 1*/

				sCenter = SelectFieldComponent(whichField, comp, longIndices);
				--longIndices[i];
				sMinus = SelectFieldComponent(whichField, comp, longIndices);
				++longIndices[i];
				if (sCenter <= sample[comp] <= sMinus)
				{
				    indices[i] =
					(sMinus - sample[comp]) * longIndices[i] +
					(sample[comp] - sCenter) * (longIndices[i] - 1);
				}
				else if (sMinus <= sample[comp] <= sCenter)
				{
				    indices[i] =
					(sCenter - sample[comp]) * longIndices[i] +
					(sample[comp] - sMinus) * (longIndices[i] - 1);
				}
				else
				{
				    indices[i] = longIndices[i];
				}
			    }
			    else
			    {
				real sCenter, sMinus, sPlus;
				/*It's in the center*/
				sCenter = SelectFieldComponent(whichField, comp, longIndices);
				--longIndices[i];
				sMinus = SelectFieldComponent(whichField, comp, longIndices);
				++longIndices[i];
				++longIndices[i];
				sPlus = SelectFieldComponent(whichField, comp, longIndices);
				--longIndices[i];
				if (sCenter <= sample[comp] <= sPlus)
				{
				    indices[i] =
					(sPlus - sample[comp]) * longIndices[i] +
					(sample[comp] - sCenter) * (longIndices[i] + 1);
				}
				else if (sPlus <= sample[comp] <= sCenter)
				{
				    indices[i] =
					(sCenter - sample[comp]) * longIndices[i] +
					(sample[comp] - sPlus) * (longIndices[i] + 1);
				}
				else if (sCenter <= sample[comp] <= sMinus)
				{
				    indices[i] =
					(sMinus - sample[comp]) * longIndices[i] +
					(sample[comp] - sCenter) * (longIndices[i] - 1);
				}
				else if (sMinus <= sample[comp] <= sCenter)
				{
				    indices[i] =
					(sCenter - sample[comp]) * longIndices[i] +
					(sample[comp] - sMinus) * (longIndices[i] - 1);
				}
				else
				{
				    indices[i] = longIndices[i];
				}
			    }
			}
		    }
		}
	    }
	}
    } while (succeed && indicesAssigned < nIndices);

    free(longIndices);

    if (indicesAssigned < nIndices)
    {
	ReportError("FindClosestSample", "Failed");
	return false;
    }

    return true;
}

real SelectFieldComponent(whichField, whichComponent, indices)
int whichField;
int whichComponent;
long indices[];
/*Selects a field component given indices into it.  There had better be enough
  indices*/
{
    register int k;
    register long offset = 0;		/*Offset into the data*/
    register Component *component;	/*The current component*/

    /*Get the current component*/
    if (whichComponent < 0 || 
	whichComponent >= curFields[whichField] . nComponents)
    {
	return missingData;
    }
    component = &(curFields[whichField] . components[whichComponent]);

    /*Calculate the offset*/
    for (k = 0; k < component -> nIndices; ++k)
    {
	register int whichIndex;
	register long index;
	
	whichIndex = component -> indices[k];

	if (whichIndex >= 0)
	{
	    index = indices[whichIndex];
	    if (index < 0 || index > component -> dimensions[k])
	    {
		return missingData;
	    }

	    offset += component -> steps[k] * index;
	}
    }
    return component -> data[offset];
}

real InterpolateFieldComponent(whichField, whichComponent, indices)
int whichField;
int whichComponent;
real indices[];
/*Interpolates a component at a set of real topological indices*/
{
    register int i, k;
    register long test;
    register Component *component;	/*The current component*/
    real retVal;
    register int whichIndex;
    register real index;

    component = &(curFields[whichField] . components[whichComponent]);

    for (k = 0; k < component -> nIndices; ++k)
    {
	whichIndex = component -> indices[k];

	if (whichIndex >= 0)
	{
	    index = indices[whichIndex];

	    test = index;
	    if (index != (real) test)
	    {
		/*It's non integral*/

		if (index < -0.5 ||
		    index > ((real) component -> dimensions[whichIndex]) - 0.5)
		{
		    /*It's way outside*/
		    return missingData;
		}
		else
		{
		    /*It's between two vertices*/
		    long i1, i2;
		    real v1, v2;
		    real *tempIndices;
		    long copyIndex;

		    i1 = test;
		    i2 = i1 + 1;

		    tempIndices = (real *) malloc(component -> nIndices * sizeof(real));

		    /*Copy all the indices*/
		    for (i = 0; i < component -> nIndices; ++i)
		    {
			copyIndex = component -> indices[i];
			tempIndices[copyIndex] = indices[copyIndex];
		    }

		    /*Now get the two values for the two indices*/
		    tempIndices[whichIndex] = i1;
		    v1 = InterpolateFieldComponent(whichField, whichComponent, tempIndices);
		    tempIndices[whichIndex] = i2;
		    v2 = InterpolateFieldComponent(whichField, whichComponent, tempIndices);

		    free(tempIndices);

		    /*Now return an interpolation*/
		    if (v1 == missingData)
		    {
			if (v2 == missingData)
			{
			    return missingData;
			}
			else
			{
			    if (i2 - index <= index - i1)
			    {
				return v2;
			    }
			    else
			    {
				return missingData;
			    }
			}
		    }
		    else if (v2 == missingData)
		    {
			if (index - i1 <= i2 - index)
			{
			    return v1;
			}
			else
			{
			    return missingData;
			}
		    }
		    else
		    {
			return (index - i1) * v2 + (i2 - index) * v1;
		    }
		}
	    }
	}
    }

    /*If we fell through here, it must be smack dab on a vertex.  Just get it*/
    {
	long *tempIndices;
	tempIndices = (long *) malloc(component -> nIndices * sizeof(long));
for (k = 0; k < 3; ++k) printf("%g ", indices[k]);
	for (k = 0; k < component -> nIndices; ++k)
	{
	    whichIndex = component -> indices[k];
	    printf("%d ", whichIndex);
	    if (whichIndex >= 0)
	    {
	        tempIndices[whichIndex] = indices[whichIndex];
	    }
	}
	printf("\n");
	retVal = SelectFieldComponent(whichField, whichComponent, tempIndices);
	free(tempIndices);
	return retVal;
    }
}

long GetComponentOffset(whichField, whichComponent, indices)
int whichField;
int whichComponent;
long indices[];
/*Get the offset into a field component given indices into it.  There had better be enough
  indices*/
{
    register int k;
    register long offset = 0;		/*Offset into the data*/
    register Component *component;	/*The current component*/

    /*Get the current component*/
    if (whichComponent < 0 || 
	whichComponent > curFields[whichField] . nComponents)
    {
	return 0;
    }
    component = &(curFields[whichField] . components[whichComponent]);

    /*Calculate the offset*/
    for (k = 0; k < component -> nIndices; ++k)
    {
	register int whichIndex;
	register long index;
	
	whichIndex = component -> indices[k];

	if (whichIndex >= 0)
	{
	    index = indices[whichIndex];
	    if (index < 0 || index > component -> dimensions[k])
	    {
		return 0;
	    }
	}

	offset += component -> steps[k] * index;
    }
    return offset;
}


#ifdef PROTO
void PutFieldComponent(int whichField, int whichComponent, long indices[], real data)
#else
void PutFieldComponent(whichField, whichComponent, indices, data)
int whichField;
int whichComponent;
long indices[];
real data;
#endif
/*Puts data into a field component given indices into it.  There had better be enough
  indices*/
{
    register int k;
    register long offset = 0;		/*Offset into the data*/
    register Component *component;	/*The current component*/

    /*Get the current component*/
    if (whichComponent < 0 || 
	whichComponent > curFields[whichField] . nComponents)
    {
	return;
    }
    component = &(curFields[whichField] . components[whichComponent]);

    /*Calculate the offset*/
    for (k = 0; k < component -> nIndices; ++k)
    {
	register int whichIndex;
	register long index;
	
	whichIndex = component -> indices[k];

	if (whichIndex >= 0)
	{
	    index = indices[whichIndex];
	    if (index < 0 || index > component -> dimensions[k])
	    {
		return;
	    }

	    offset += component -> steps[k] * index;
	}
    }
    component -> data[offset] = data;
}

void StuffScalarIJSquare(dest, whichField, indices)
real dest[2][2];
int whichField;
long indices[];
/*Stuffs the square in the ij plane with its lower left corner defined by
  indices into dest.  Takes data from the specified field, which had better
  be scalar.  There had better be enough indices.  This does not check
  validity of the indices*/
{
    int k;
    register long offset = 0;		/*Offset into the data*/
    register long iStep = 0, jStep = 0;	/*Step in the i and j directions*/
    register Component *component;	/*The current component*/
    register real *data;

    /*Get the current component*/
    component = &(curFields[whichField] . components[0]);

    /*Calculate the offset*/
    for (k = 0; k < component -> nIndices; ++k)
    {
	register int whichIndex;
	register long index;
	
	whichIndex = component -> indices[k];

	if (whichIndex >= 0)
	{
	    index = indices[whichIndex];
	    if (whichIndex == 0)
	    {
		/*I component*/
		iStep = component -> steps[k];
		offset += iStep * index;
	    }
	    else if (whichIndex == 1)
	    {
		/*J component*/
		jStep = component -> steps[k];
		offset += jStep * index;
	    }
	    else
	    {
		offset += component -> steps[k] * index;
	    }
	}
    }
    dest[0][0] = component -> data[offset];
    dest[1][0] = component -> data[offset + iStep];
    dest[0][1] = component -> data[offset + jStep];
    dest[1][1] = component -> data[offset + iStep + jStep];
}

void StuffIJSquare(dest, whichField, whichComponent, indices)
real dest[2][2];
int whichField;
int whichComponent;
long indices[];
/*Stuffs the square in the ij plane with its lower left corner defined by
  indices into dest.  Takes data from the specified field, which had better
  have component.  There had better be enough indices.  This does not check
  validity of the indices*/
{
    int k;
    register long offset = 0;		/*Offset into the data*/
    register long iStep = 0, jStep = 0;	/*Step in the i and j directions*/
    register Component *component;	/*The current component*/
    register real *data;

    /*Get the current component*/
    component = &(curFields[whichField] . components[whichComponent]);

    /*Calculate the offset*/
    for (k = 0; k < component -> nIndices; ++k)
    {
	register int whichIndex;
	register long index;
	
	whichIndex = component -> indices[k];

	if (whichIndex >= 0)
	{
	    index = indices[whichIndex];
	    if (whichIndex == 0)
	    {
		/*I component*/
		iStep = component -> steps[k];
		offset += iStep * index;
	    }
	    else if (whichIndex == 1)
	    {
		/*J component*/
		jStep = component -> steps[k];
		offset += jStep * index;
	    }
	    else
	    {
		offset += component -> steps[k] * index;
	    }
	}
    }
    dest[0][0] = component -> data[offset];
    dest[1][0] = component -> data[offset + iStep];
    dest[0][1] = component -> data[offset + jStep];
    dest[1][1] = component -> data[offset + iStep + jStep];
}

Bool SetCurField(whichField, field)
int whichField;
ObjPtr field;
/*Sets current field whichField to field.  Returns true iff it did*/
{
    FuncTyp method;

    /*Clean the field before setting it*/
    CleanCurField(whichField);

    curFields[whichField] . topDim = GetTopDim(field);

    method = GetMethodSurely("SetCurField", field, REGISTERFIELD);
    if (method)
    {
	ObjPtr result;
	result = (*method)(field, whichField);
	return IsTrue(result) ? true : false;
    }
    return false;
}

Bool SetCurForm(whichField, field)
int whichField;
ObjPtr field;
/*Sets current field whichField to the data form of field*/
{
    FuncTyp method;

    CleanCurField(whichField);

    curFields[whichField] . topDim = GetTopDim(field);

    method = GetMethodSurely("SetCurField", field, REGISTERFORM);
    if (method)
    {
	ObjPtr result;
	result = (*method)(field, whichField);
	return IsTrue(result) ? true : false;
    }
    return false;
}

ObjPtr RegisterDatasetForm(dataset, whichField)
int whichField;
ObjPtr dataset;
/*Registers the form of a dataset in whichField*/
{
    ObjPtr dataForm;

    dataForm = GetVar(dataset, DATAFORM);
    if (dataForm)
    {
	return SetCurField(whichField, dataForm) ? ObjTrue : ObjFalse;
    }
    ReportError("RegisterDatasetForm", "No DATAFORM");
    return ObjFalse;
}

Bool RegisterComponent(whichField, whichComponent, field)
int whichField, whichComponent;
ObjPtr field;
/*Registers field as whichComponent of whichField*/
{
    FuncTyp method;

    if (whichComponent < 0 || whichComponent >= curFields[whichField] . nComponents)
    {
	ReportError("RegisterComponent", "Component number out of range");
	return false;
    }

    method = GetMethodSurely("RegisterComponent", field, REGISTERCOMP);
    if (method)
    {
	ObjPtr result;
	result = (*method)(field, whichField, whichComponent);
	return IsTrue(result) ? true : false;
    }
    return false;
}

static ObjPtr globalDataset;		/*Global dataser for ForAllVisWindows*/
static void ChangeWindowForDataset(window)
WinInfoPtr window;
/*Changes a given window upon the knowledge that the dataset has changed*/
{
    ObjPtr space;
    space = FindSpace(window);
    if (space)
    {
	ObjPtr clock;
	ImInvalid((ObjPtr) window);
	clock = GetVar(space, CLOCK);
	if (clock)
	{
	    WinInfoPtr dialogExists;
	    dialogExists = DialogExists((WinInfoPtr) clock, NewString("Clock"));
	    if (dialogExists)
	    {
		ObjPtr timeControl;
		timeControl = GetVar((ObjPtr) dialogExists, TIMECONTROL);
		if (timeControl)
		{
		    RecalcScroll(timeControl);
		}
		ImInvalid((ObjPtr) dialogExists);
	    }
	}
    }
}

void DatasetChanged(dataset)
ObjPtr dataset;
/*Alerts the system that a dataset has changed*/
{
    globalDataset = dataset;
    ForAllVisWindows(ChangeWindowForDataset);
}

static ObjPtr ChangeDatasetInterpolate(object)
ObjPtr object;
/*Changes, according to the value of object, whether a dataset is interpolated*/
{
    ObjPtr repObj, value, dataObj;

    repObj = GetObjectVar("ChangeDatasetInterpolate", object, REPOBJ);
    if (!repObj)
    {
	return ObjFalse;
    }

    value = GetValue(object);

    SetVar(repObj, INTERPOLATEP, IsTrue(value) ? ObjTrue : ObjFalse);

    dataObj = GetVar(repObj, DATA);
    if (dataObj)
    {
	SetVar(dataObj, INTERPOLATEP, IsTrue(value) ? ObjTrue : ObjFalse);
    }

    DatasetChanged(repObj);
    return ObjTrue;
}

static ObjPtr HideDatasetWindow(window)
ObjPtr window;
/*Hide a dataset window, just return OK*/
{
    return ObjTrue;
}

static void NotPaletteAlert()
{
    WinInfoPtr errWindow;
    errWindow = AlertUser(UIRED, (WinInfoPtr) 0, "Only palettes can be dropped here.", 0, 0, "");
    SetVar((ObjPtr) errWindow, HELPSTRING,
	NewString("You have tried to drop an object which is not a palette into the palette \
box of a dataset.  Try again using a real palette."));
}

static ObjPtr DropInPaletteCorral(corral, object, x, y)
ObjPtr corral, object;
int x, y;
/*Drops an icon in a palette corral*/
{
    ObjPtr dataset;
    ObjPtr palette;
    ObjPtr icon;
    ObjPtr name;
    ObjPtr defaultIcon;
    ObjPtr contents;
    ObjPtr button;

    /*Find the visualization object*/
    dataset = GetObjectVar("DropInPaletteCorral", corral, REPOBJ);
    if (!dataset)
    {
	return ObjFalse;
    }

    /*Get the field object*/
    palette = GetVar(object, REPOBJ);
    if (!palette)
    {
	return ObjFalse;
    }
    if (!IsPalette(palette))
    {
	DoTask(NotPaletteAlert);
	return ObjFalse;
    }

    /*Create an icon for it*/
    name = GetStringVar("DropInPaletteCorral", palette, NAME);
    if (!name)
    {
	return ObjFalse;
    }

    defaultIcon = GetVar(palette, DEFAULTICON);
    if (defaultIcon)
    {
	ObjPtr locArray;
	real loc[2];
	icon = NewObject(defaultIcon, 0);
	SetVar(icon, NAME, name);
	loc[0] = x;
	loc[1] = y;
	locArray = NewRealArray(1, 2L);
	CArray2Array(locArray, loc);
	SetVar(icon, ICONLOC, locArray);
    }
    else
    {
	icon = NewIcon(x, y, ICONQUESTION, GetString(name));
    }
    
    /*Make the icon point to the palette*/
    SetVar(icon, REPOBJ, palette);

    /*Make it the only icon in the corral*/
    contents = NewList();
    PrefixList(contents, icon);
    SetVar(corral, CONTENTS, contents);
    SetVar(contents, PARENT, corral);
    SetVar(icon, CORRAL, corral);
    SetMethod(icon, CHANGEDVALUE, ChangeIconButtons);

    SetVar(icon, PARENT, corral);
    RecalcScroll(corral);

    /*Make this object the colored object*/
    SetVar(dataset, CPALETTE, palette);

    ImInvalid(corral);
    DatasetChanged(dataset);
}


static ObjPtr ShowDatasetControls(object, ownerWindow, windowName)
ObjPtr object;
WinInfoPtr ownerWindow;
char *windowName;
/*Makes a new control window to control a dataset*/
{
    WinInfoPtr controlWindow;
    ObjPtr var;
    ObjPtr panel;
    ObjPtr corral;
    ObjPtr contents;
    WinInfoPtr dialogExists;

    dialogExists = DialogExists((WinInfoPtr) object, NewString("Controls"));
    controlWindow = GetDialog((WinInfoPtr) object, NewString("Controls"), windowName, 
	DCWINWIDTH, DCWINHEIGHT, DCWINWIDTH, DCWINHEIGHT, WINDBUF + WINFIXEDSIZE);
    
    if (!dialogExists)
    {
	long info;
	ObjPtr textBox, checkBox, icon, palette, name, button;
	int left, right, bottom, top;
	char *s;
	SetVar((ObjPtr) controlWindow, REPOBJ, object);

	DefineMenuItem((ObjPtr) controlWindow, OBJECTMENU, "Show Controls", DoShowControls);
	DefineMenuItem((ObjPtr) controlWindow, OBJECTMENU, "Make Local Copy", DoMakeLocalCopy);

	/*Set help string*/
	SetVar((ObjPtr) controlWindow, HELPSTRING, NewString("This window \
shows controls for a dataset.  This portion of the program is unfinished, so the \
window is not quite as it should be.\n"));

	/*Add in a panel*/
	panel = NewPanel(greyPanelClass, 0, DCWINWIDTH, 0, DCWINHEIGHT);
	if (!panel)
	{
	    return ObjFalse;
	}
	contents = GetVar((ObjPtr) controlWindow, CONTENTS);
	PrefixList(contents, panel);
	SetVar(panel, PARENT, (ObjPtr) controlWindow);

	contents = GetVar(panel, CONTENTS);
	
	/*Add the info description*/
	left = MAJORBORDER;
	right = DCWINWIDTH - MAJORBORDER;
	top = DCWINHEIGHT - MAJORBORDER;

	info = GetDatasetInfo(object);

	s = tempStr;
	sprintf(s, "Data: ");
	while (*s) ++s;
	if (info & DS_TIMEDEPENDENT)
	{
	    sprintf(s, "Time-dependent ");
	}
	else
	{
	    sprintf(s, "Static ");
	}

	while (*s) ++s;
	if (info & DS_HASGEOMETRY)
	{
	    sprintf(s, "geometric data\n");
	}
	else
	{
	    SetCurField(FIELD1, object);
	    if (info & DS_VECTOR)
	    {
		long nComponents;
		nComponents = GetNComponents(FIELD1);
		sprintf(s, "%d-vector %s\n", nComponents,
		    info & DS_HASFORM ? "field" : "data");
	    }
	    else
	    {
		sprintf(s, "scalar %s\n", info & DS_HASFORM ? "field" : "data");
	    }
	}
	while (*s) ++s;
	sprintf(s, "\nGrid: ");
	while (*s) ++s;
	if (info & DS_HASFORM)
	{
	    sprintf(s, "%d-dimensional %s grid\n", 
		GetTopDim(object),
		info & DS_UNSTRUCTURED ? "unstructured" : "structured");
	}
	else
	{
	    sprintf(s, "none");
	}
	while (*s) ++s;
	bottom = top - DCINFOHEIGHT;
	textBox = NewTextBox(left, right, 
			 bottom, top,
			 0, "Info Text", tempStr);
	PrefixList(contents, textBox);
	SetVar(textBox, PARENT, panel);
	SetVar(textBox, TEXTFONT, NewString(DCFONT));
	SetVar(textBox, TEXTSIZE, NewInt(DCFONTSIZE));
	top = bottom - MAJORBORDER;

	/*Add the interpolate check box*/
	if (info & DS_TIMEDEPENDENT)
	{
	    bottom = top - CHECKBOXHEIGHT;
	    /*Create the Display Light check box*/
	    checkBox = NewCheckBox(left, right, bottom, top,
				   "Interpolate in time",
				   GetPredicate(object, INTERPOLATEP));
	    SetVar(checkBox, HELPSTRING,
		NewString("If this box is checked, the field will automatically be \
interpolated in time.  If it is not checked, the time step nearest the current \
time will always be used.\n"));
	    PrefixList(contents, checkBox);
	    SetVar(checkBox, PARENT, panel);
	    SetVar(checkBox, REPOBJ, object);
	    SetMethod(checkBox, CHANGEDVALUE, ChangeDatasetInterpolate);
	}

	/*Add in the colors corral*/
	bottom = MAJORBORDER + BUTTONHEIGHT + MINORBORDER + TEXTBOXHEIGHT + TEXTBOXSEP;
	top = bottom + ONECORRALHEIGHT;
	right = left + ONECORRALWIDTH;
	corral = NewIconCorral(NULLOBJ,
			   left, right, bottom, top,
			   0);
	SetVar(corral, SINGLECORRAL, ObjTrue);
	SetVar(corral, TOPDOWN, ObjTrue);
	SetVar(corral, NAME, NewString("Color Palette"));
	PrefixList(contents, corral);
	SetVar(corral, HELPSTRING,
	NewString("This corral shows the color palette of the data in the dataset.  This \
may be another dataset.  Right now, you cannot drag any icons \
into the corral."));
	SetVar(corral, PARENT, panel);
	SetVar(corral, REPOBJ, object);
	SetMethod(corral, DROPINCONTENTS, DropInPaletteCorral);

	/*Create the source text box*/
	textBox = NewTextBox(left, right, 
				bottom - TEXTBOXSEP - TEXTBOXHEIGHT, bottom - TEXTBOXSEP,
				0, "Colors Text", "Color Palette");
	PrefixList(contents, textBox);
	SetVar(textBox, PARENT, panel);
	SetTextAlign(textBox, CENTERALIGN);
	left += ONECORRALWIDTH + 2 * MAJORBORDER;

	/*Make a palette icon*/
	MakeVar(object, CPALETTE);
	palette = GetVar(object, CPALETTE);
	if (palette)
	{
	    MakeVar(palette, NAME);
	    name = GetVar(palette, NAME);
	    icon = GetVar(palette, DEFAULTICON);
	    if (icon)
	    {
		icon = NewObject(icon, 0);
		SetVar(icon, NAME, name);
	    }
	    else
	    {
		icon = NewIcon(0, 0, ICONQUESTION, GetString(name));
	    }
	    SetVar(icon, ICONLOC, NULLOBJ);
	    SetVar(icon, REPOBJ, palette);
	    SetVar(icon, CORRAL, corral);
	    SetMethod(icon, CHANGEDVALUE, ChangeIconButtons);
	    DropIconInCorral(corral, icon);
	}

	left = MAJORBORDER;
	right = DCWINWIDTH - MAJORBORDER;
	bottom = MAJORBORDER;
	top = bottom + BUTTONHEIGHT;
	/*Make a show info button*/
	button = NewButton(left, left + (right - left) / 3, 
		bottom, top, "Show Controls");
	SetVar(button, PARENT, panel);
	SetVar(button, STICKINESS, NewInt(STICKYBOTTOM + FLOATINGLEFT + STICKYRIGHT));
	SetVar(corral, SHOWCNTBUTTON, button);
	PrefixList(contents, button);
	SetMethod(button, CHANGEDVALUE, ShowControlsButton);
	SetVar(corral, SHOWCNTBUTTON, button);
	ActivateButton(button, false);
	SetVar(button, HELPSTRING, 
	NewString("Pressing this button will bring up a window that shows \
controls for the \
selected objects."));

    }

    return (ObjPtr) controlWindow;
}

ObjPtr FindDatasetByName(name)
ObjPtr name;
/*Finds a dataset by its name, returns it or nothing*/
{
    ThingListPtr runner;
    runner = LISTOF(allDatasets);
    while (runner)
    {
	if (Eql(GetVar(runner -> thing, NAME), name))
	{
	    return runner -> thing;
	}
	runner = runner -> next;
    }
    return NULLOBJ;
}

static ObjPtr globalVisButton, globalVisAsButton, globalShowCntButton, globalFilterButton;
static ObjPtr globalIconList;

static void DoSetIconButtons()
/*Goes through the list of icons and sets buttons accordingly*/
{
    ThingListPtr runner;

    if (!globalIconList) return;
    runner = LISTOF(globalIconList);
    while (runner)
    {
	if (GetPredicate(runner -> thing, SELECTED))
	{
	    if (globalVisButton)
	    {
		ActivateButton(globalVisButton, true);
		globalVisButton = NULLOBJ;
	    }
	    if (globalVisAsButton)
	    {
		ActivateButton(globalVisAsButton, true);
		globalVisButton = NULLOBJ;
	    }
	    if (globalShowCntButton)
	    {
		ActivateButton(globalShowCntButton, true);
		globalShowCntButton = NULLOBJ;
	    }
	    if (globalFilterButton)
	    {
		ActivateButton(globalFilterButton, true);
		globalFilterButton = NULLOBJ;
	    }
	    return;
	}
	runner = runner -> next;
    }
    if (globalVisButton)
    {
	ActivateButton(globalVisButton, false);
	globalVisButton = NULLOBJ;
    }
    if (globalVisAsButton)
    {
	ActivateButton(globalVisAsButton, false);
	globalVisAsButton = NULLOBJ;
    }
    if (globalShowCntButton)
    {
	ActivateButton(globalShowCntButton, false);
	globalShowCntButton = NULLOBJ;
    }
    if (globalFilterButton)
    {
	ActivateButton(globalFilterButton, false);
	globalFilterButton = NULLOBJ;
    }
    return;
}

ObjPtr ChangeIconButtons(icon)
ObjPtr icon;
/*Changedvalue for a dataset icon, changes the buttons*/
{
    ObjPtr corral;

    corral = GetObjectVar("ChangeIconButtons", icon, CORRAL);
    if (!corral)
    {
	return ObjFalse;
    }
    globalVisButton = GetVar(corral, VISBUTTON);
    globalVisAsButton = GetVar(corral, VISASBUTTON);
    globalShowCntButton = GetVar(corral, SHOWCNTBUTTON);
    globalFilterButton = GetVar(corral, FILTERBUTTON);
    globalIconList = GetListVar("ChangeIconButtons", corral, CONTENTS);
    if (!globalIconList)
    {
	return ObjFalse;
    }
    DoUniqueTask(DoSetIconButtons);
    return ObjTrue;
}

Bool RegisterDataset(dataset)
ObjPtr dataset;
/*Registers dataset as a dataset*/
{
    ObjPtr corral, contents;
    ThingListPtr runner;
    ObjPtr var;
    char *name;
    ObjPtr icon;
    ObjPtr defaultIcon;
    char *sPtr;
    WinInfoPtr datasetsWindow;
    int k;

    datasetsWindow = DatasetsWindow();
    PopWindow(datasetsWindow);

    SelectWindow(datasetsWindow -> id);
    LongOperation();
    IdleAllWindows();

    corral = FindMainCorral(datasetsWindow);
    var = GetStringVar("RegisterDataset", dataset, NAME);
    if (!var) return false;
    name = GetString(var);

    k = 1;
    while (FindDatasetByName(var))
    {
	++k;
	sprintf(tempStr, "%s (%d)", name, k);
	var = NewString(tempStr); 
    }
    if (k > 1)
    {
	name = GetString(var);
	SetVar(dataset, NAME, var);
    }


    if (timedDatasets)
    {
    /*See if there is a time name*/
    sPtr = name;
    while (*sPtr)
    {
	if (*sPtr == '@')
	{
	    /*Yes, it's a time slice!*/
	    strncpy(tempStr, name, TEMPSTRSIZE);
	    if (sPtr - name < TEMPSTRSIZE)
	    {
		tempStr[sPtr - name] = 0;
	    }
	    else
	    {
		tempStr[TEMPSTRSIZE] = 0;
	    }

	    /*Edit the dataset's name*/
	    var = NewString(tempStr);
	    SetVar(dataset, NAME, var);

	    /*Copy the time*/
	    strncpy(tempStr, sPtr + 1, TEMPSTRSIZE);
	    tempStr[TEMPSTRSIZE] = 0;

	    /*Get the edited name back into name*/
	    name = GetString(var);
	    break;
	}
	++sPtr;
    }

    if (*sPtr == '@')
    {
	/*It must be a time slice*/
	ObjPtr priorDataset;
	real time;
	int format;

	/*Go through and parse the time*/
	if (ParseTime(&time, &format, tempStr) <= 0)
	{
	    ReportError("RegisterDataset","Bad time format");
	}

	/*Find out if there already exists a time object like this*/

	var = GetVar(dataset, NAME);
	priorDataset = FindDatasetByName(var);
	if (priorDataset)
	{
	    /*Add on to an existing dataset*/
	    InsertTimeSlice(GetVar(priorDataset, DATA), NewReal(time), GetVar(dataset, DATA));

	    /*Register the dataset as changed*/
	    DatasetChanged(priorDataset);

	    /*Make the prior dataset the dataset in use for the icon test*/
	    dataset = priorDataset;
	}
	else
	{
	    ObjPtr timeSteps, timeData;
	    /*Make a new dataset*/

	    timeSteps = NewList();
	    timeData = NewList();
	    PrefixList(timeSteps, NewReal(time));
	    PrefixList(timeData, GetVar(dataset, DATA));
	    SetVar(dataset, DATA, NewTimedObject(timeSteps, timeData));
	    SetVar(dataset, TIMEFORMAT, NewInt(format));
	    PrefixList(allDatasets, dataset);
	}
    }
    else
    {
	PrefixList(allDatasets, dataset);
    }
    }
    else
    {
	PrefixList(allDatasets, dataset);
    }

    /*Search for an icon*/
    contents = GetListVar("RegisterDataset", corral, CONTENTS);
    if (!contents) return false;
    runner = LISTOF(contents);
    while (runner)
    {
	ObjPtr repObj;
	repObj = GetVar(runner -> thing, REPOBJ);
	if (repObj == dataset)
	{
	    break;
	}
	runner = runner -> next;
    }

    if (!runner)
    {
	/*Must create a new icon*/
	defaultIcon = GetVar(dataset, DEFAULTICON);
	if (defaultIcon)
	{
	    icon = NewObject(defaultIcon, 0);
	    SetVar(icon, NAME, NewString(name));
	}
	else
	{
	    icon = NewIcon(0, 0, ICONQUESTION, name);
	    SetVar(icon, HELPSTRING,
		NewString("This icon represents a dataset.  You can select it and use the controls \
in this window to modify or visualize it, or you can drag it into the \
corral of an existing visualization window to visualize it there."));

	}
	SetVar(icon, REPOBJ, dataset);
	SetVar(icon, CORRAL, corral);
	SetMethod(icon, CHANGEDVALUE, ChangeIconButtons);
	SetMethod(icon, DOUBLECLICK, ShowIconControls); 
	SetVar(icon, ICONLOC, dsLocArray);
	dsLocArray = NULLOBJ;
	DropIconInCorral(corral, icon);
	IdleAllWindows();
    }
    return true;
}

void ReportFileFormatError()
/*Task to report a bad read file error*/
{
    WinInfoPtr errWindow;
    errWindow = AlertUser(UIRED, DatasetsWindow(),
	"SciAn can only read data files with defined formats.  Set the file formats with Set Format and try again.",
	0, 0, "Yeah, right.");
    SetVar((ObjPtr) errWindow, HELPSTRING,
	NewString("You have tried to open a file cannot be opened because it \
is not a SciAn data file or its file format has not been set.  If you know what \
the proper format is, you can select the icon representing the file and press the \
Set Format button.  You will be presented with a series of choices.  Select the \
proper format, and try this operation again."));
}

void ReportDirError()
/*Task to report a directory tried to drag into datasets*/
{
    WinInfoPtr errWindow;
    errWindow = AlertUser(UIRED, DatasetsWindow(),
	"Directories cannot be opened as datasets.",
	0, 0, "Yeah, right.");
    SetVar((ObjPtr) errWindow, HELPSTRING,
	NewString("You have tried to open an entire directory as a dataset.  \
This does not work.  If you want to open all the files contained in the \
directory, open the directory, select all the files, and open them."));
}

static ObjPtr DropInDatasetsCorral(corral, icon, x, y)
ObjPtr corral, icon;
int x, y;
/*Drops an icon in a vis corral*/
{
    ObjPtr repObj;
    repObj = GetVar(icon, REPOBJ);
    if (repObj && InClass(repObj, fileClass))
    {
	ObjPtr fileName;		/*Name of the file*/
	ObjPtr fileType;		/*Type of the file*/
	ObjPtr fileFormat;		/*Format of the file*/
	real loc[2];
	loc[0] = x;
	loc[1] = y;
	dsLocArray = NewRealArray(1, 2L);
	CArray2Array(dsLocArray, loc);

	fileName = GetVar(repObj, NAME);
	fileType = GetVar(repObj, FILETYPE);
	fileFormat = GetVar(repObj, FORMAT);

	if (!fileName || !fileType)
	{
	    DoUniqueTask(ReportFileFormatError);
	    return ObjFalse;
	}
	if (GetInt(fileType) == DIRFILE)
	{
	    DoUniqueTask(ReportDirError);
	    return ObjFalse;
	}
	if (!fileFormat)
	{
	    DoUniqueTask(ReportFileFormatError);
	    return ObjFalse;
	}
	LongOperation();
	ReadFile(GetString(fileName), GetString(fileFormat));
	dsLocArray = NULLOBJ;
	return ObjTrue;
    }
    else
    {
	return ObjFalse;
    }
}

static void VOTask()
/*Task to visualize objects*/
{
    InhibitLogging(true);
    VisObjects();
    InhibitLogging(false);
}

ObjPtr VisualizeButton(button)
ObjPtr button;
/*Does a DoVisualize*/
{
    DoTask(VOTask);
    return ObjTrue;
}

static void VATask()
/*Task to visualize objects as*/
{
    InhibitLogging(true);
    VisObjectsAs();
    InhibitLogging(false);
}

ObjPtr VisualizeAsButton(button)
ObjPtr button;
/*Does a DoVisualizeAs*/
{
    DoTask(VATask);
    return ObjTrue;
}

static void SCTask()
/*Task for show controls*/
{
    InhibitLogging(true);
    DoShowControls();
    InhibitLogging(false);
}

ObjPtr ShowControlsButton(button)
ObjPtr button;
/*Does a DoShowControls*/
{
    DoTask(SCTask);
    return ObjTrue;
}

static void ModTask()
/*Task for modify datasets*/
{
    InhibitLogging(true);
    ModifyDatasets();
    InhibitLogging(false);
}

ObjPtr ModifyButton(button)
ObjPtr button;
/*Does a DoShowControls*/
{
    DoTask(ModTask);
    return ObjTrue;
}

WinInfoPtr NewDatasetsWindow(void)
/*Create a new Datasets window*/
{
    WinInfoPtr objWin;
    ObjPtr panel, contents, corral, button;
    int l, r, b, t;
    int bw;

    /*Create the window*/
    objWin = NewObjWindow(NULLOBJ, "Datasets", WINDBUF + WINRGB, DSWINWIDTH, DSWINHEIGHT, SCRWIDTH, SCRHEIGHT);
    GetWindowBounds(&l, &r, &b, &t);

    /*Set a null but successful HIDE routine*/
    SetMethod((ObjPtr) objWin, HIDE, HideDatasetWindow);

    /*Add a help string*/
    SetVar((ObjPtr) objWin, HELPSTRING, 
	NewString("This window shows all the datasets that have been opened in \
Scian."));

    /*Add a visualize entry*/
    DefineMenuItem((ObjPtr) objWin, OBJECTMENU, "Visualize", VisObjects);
    DefineMenuItem((ObjPtr) objWin, OBJECTMENU, "Visualize As...", VisObjectsAs);
    DefineMenuItem((ObjPtr) objWin, OBJECTMENU, "Show Controls", DoShowControls);
#if 0
    DefineMenuItem((ObjPtr) objWin, OBJECTMENU, "Unify Palettes", DoUnifyPalettes);
#endif

    /*Put in a panel*/
    panel = NewPanel(greyPanelClass, l, r, b, t);
    SetVar(panel, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT +
				     STICKYBOTTOM + STICKYTOP));

    contents = GetVar((ObjPtr) objWin, CONTENTS);
    PrefixList(contents, panel);
    SetVar(panel, PARENT, (ObjPtr) objWin);

    /*Put in buttons and an icon corral*/
    contents = GetListVar("NewDatasetsWindow", panel, CONTENTS);
    if (!contents)
    {
	return 0;
    }
    /*Make an icon corral*/
    corral = NewIconCorral(NULLOBJ, l + MINORBORDER, r - MINORBORDER, b + 3 * MINORBORDER + 2 * BUTTONHEIGHT, t - MINORBORDER, BARRIGHT + BARBOTTOM);
    SetVar(corral, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT +
				     STICKYBOTTOM + STICKYTOP));
    SetVar(corral, TOPDOWN, ObjTrue);
    SetVar(corral, NAME, NewString("Datasets Corral"));
    SetVar(corral, HELPSTRING,
	NewString("This corral contains icons for all the datasets that have \
been read into SciAn.  You can visualize, show the controls of, or modify the datasets by selecting \
some of them and pressing the buttons at the bottom of the window.  You can delete \
datasets by choosing Delete from the Object menu.  You can also open new datasets by dragging icons from \
the file window into this corral."));
    SetMethod(corral, DROPINCONTENTS, DropInDatasetsCorral);
    PrefixList(contents, corral);
    SetVar(corral, PARENT, panel);

    l += MINORBORDER;
    r -= MINORBORDER;
    b += 2 * MINORBORDER + BUTTONHEIGHT;
    t = b + BUTTONHEIGHT;
    bw = (r - l - MINORBORDER) / 2;

    /*Make a visualize button*/
    button = NewButton(l, l + bw,
		b, b + BUTTONHEIGHT, "Visualize");
    SetVar(button, PARENT, panel);
    SetVar(button, STICKINESS, NewInt(STICKYBOTTOM + STICKYLEFT + FLOATINGRIGHT));
    PrefixList(contents, button);
    SetMethod(button, CHANGEDVALUE, VisualizeButton);
    SetVar(corral, VISBUTTON, button);
    ActivateButton(button, false);
    SetVar(button, HELPSTRING, 
	NewString("Pressing this button will visualize the selected datasets \
together in a new window using their default visualization techniques."));

    /*Make a choose visualization button*/
    button = NewButton(r - bw, r,
		b, b + BUTTONHEIGHT, "Visualize As..."); 
    SetVar(button, PARENT, panel);
    SetVar(button, STICKINESS, NewInt(STICKYBOTTOM + FLOATINGLEFT + FLOATINGRIGHT));
    PrefixList(contents, button);
    SetMethod(button, CHANGEDVALUE, VisualizeAsButton);
    SetVar(corral, VISASBUTTON, button);
    ActivateButton(button, false);
    SetVar(button, HELPSTRING, 
	NewString("Pressing this button will bring up a dialog box that will \
show several ways of visualizing the selected datasets."));

    t = b - MINORBORDER;
    b = t - BUTTONHEIGHT;
    /*Make a show info button*/
    button = NewButton(l, l + bw, 
		b, b + BUTTONHEIGHT, "Show Controls");
    SetVar(button, PARENT, panel);
    SetVar(button, STICKINESS, NewInt(STICKYBOTTOM + STICKYLEFT + FLOATINGRIGHT));
    PrefixList(contents, button);
    SetMethod(button, CHANGEDVALUE, ShowControlsButton);
    SetVar(corral, SHOWCNTBUTTON, button);
    ActivateButton(button, false);
    SetVar(button, HELPSTRING, 
	NewString("Pressing this button will bring up a window that shows \
controls for the \
selected datasets."));

    /*Make a modify button*/
    button = NewButton(r - bw, 
		r,
		b, b + BUTTONHEIGHT, "Modify...");
    SetVar(button, PARENT, panel);
    SetVar(button, STICKINESS, NewInt(STICKYBOTTOM + FLOATINGLEFT + STICKYRIGHT));
    PrefixList(contents, button);
    SetMethod(button, CHANGEDVALUE, ModifyButton);

    SetVar(corral, FILTERBUTTON, button);
    ActivateButton(button, false);
    SetVar(button, HELPSTRING, 
	NewString("Pressing this button will bring up a window containing \
ways to modify the selected datasets to produce new derived datasets."));

    DefineMenuItem((ObjPtr) objWin, OBJECTMENU, "Select All", DoSelectAllIcons);
    DefineMenuItem((ObjPtr) objWin, OBJECTMENU, "Deselect All", DoDeselectAllIcons);
    DefineMenuItem((ObjPtr) objWin, OBJECTMENU, "Delete", DoDelete);

    return objWin;
}

WinInfoPtr DatasetsWindow()
/*Returns or creates a datasets window*/
{
    WinInfoPtr retVal;

    retVal = GetWinFromTitle("Datasets");
    if (!retVal)
    {
	retVal = NewDatasetsWindow();
    }
    return retVal;
}

ObjPtr NewDataset()
/*Returns a new dataset*/
{
    return NewObject(datasetClass, 0L);
}

ObjPtr NewDataForm()
/*Returns a new dataset*/
{
    return NewObject(dataFormClass, 0L);
}

static ObjPtr GetDataMinMax(object)
ObjPtr object;
/*Gets and returns the min and max of the data in object for an object
  with DATA that points to an array*/
{
    ObjPtr retVal;
    retVal = GetVar(object, MINMAX);
    if (retVal && IsArray(retVal) && RANK(retVal) == 1 && DIMS(retVal)[0] == 2)
    {
	return retVal;
    }
    else
    {
	ObjPtr data;
	data = GetArrayVar("GetDataMinMax", object, DATA);
	if (!data) return NULLOBJ;
	retVal = NewRealArray(1, 2L);
	ArrayMinMax(ArrayMeat(retVal), ArrayMeat(retVal) + 1, data);
	SetVar(object, MINMAX, retVal);
	return retVal;
    }
}

ObjPtr GetMinMax(object)
ObjPtr object;
/*Front end for GETMINMAX method*/
{
    FuncTyp method;
    method = GetMethodSurely("GetMinMax", object, GETMINMAX);
    if (method)
    {
	return (*method)(object);
    }
    else
    {
	return NULLOBJ;
    }
}

static ObjPtr MakeDatasetPalette(object)
ObjPtr object;
/*Gets a palette for a dataset object*/
{
    ObjPtr palette;
    palette = GetVar(object, CPALETTE);
    if (!palette)
    {
	ObjPtr minMax;
	real min, max, *elements;

	minMax = GetMinMax(object);
	if (minMax)
	{
	    elements = ELEMENTS(minMax);
	    min = elements[0];
	    max = elements[1];
	}
	else
	{
	    min = 0.0;
	    max = 1.0;
	}

	if (onePalette)
	{
	    if (commonPalette)
	    {
		min = MIN(min, ((PPtr) commonPalette) -> min);
		max = MAX(max, ((PPtr) commonPalette) -> max);
		palette = commonPalette;
	    }
	}
	if (!palette)
	{
	    palette = NewPalette(DEFPALSIZE);
	    BlueToRedPalette(palette);
	}

	SetVar(object, CPALETTE, palette);
	if (onePalette)
	{
	    commonPalette = palette;
	}
	SetPaletteMinMax(palette, min, max);
    }
    return ObjTrue;
}

real *GetScalarDataPtr(dataset)
ObjPtr dataset;
/*Gets a pointer to the scalar data in a dataset.  Returns 0 if not valid*/
{
    ObjPtr data;
    MakeVar(dataset, CURDATA);
    data = GetArrayVar("GetScalarDataPtr", dataset, CURDATA);
    if (data)
    {
	return ArrayMeat(data);
    }
    else
    {
	return (real *) 0;
    }
}

static ObjPtr GetPlainDatasetInfo(dataset)
ObjPtr dataset;
/*Returns a sum of flags defined in ScianDatasets.h giving info on the dataset*/
{
    long retVal;
    ObjPtr data, form, var;

    retVal = 0;
    if (form = GetVar(dataset, DATAFORM))
    {
	retVal |= DS_HASFORM;
	if (GetPredicate(form, UNSTRUCTURED))
	{
	    retVal |= DS_UNSTRUCTURED;
	}
    }
    if (data = GetVar(dataset, DATA))
    {
	if (InClass(data, timedObjClass))
	{
	    ObjPtr timeData;

	    retVal |= DS_TIMEDEPENDENT;
	    timeData = GetVar(data, TIMEDATA);
	    if (timeData)
	    {
		data = ((ObjPtr *) ELEMENTS(timeData))[0];
		if (IsPicture(data))
		{
		    retVal |= DS_HASGEOMETRY;
		}
		else
		{
		    retVal |= DS_HASFIELD;
		}
	    }
	    else
	    {
		return 0;
	    }
	}
	else
	{
	    if (IsPicture(data))
	    {
		retVal |= DS_HASGEOMETRY;
	    }
	    else
	    {
		retVal |= DS_HASFIELD;
	    }
	}
    }
    if (var = GetVar(dataset, NCOMPONENTS))
    {
	retVal |= DS_VECTOR;
    }
    return NewInt(retVal);
}

int GetTopDim(dataset)
ObjPtr dataset;
/*Returns the topological or computational dimensionality of the dataset
  Returns 0 if it has no topological dimensions*/
{
    FuncTyp method;
    method = GetMethodSurely("GetTopDim", dataset, GETTOPDIM);
    if (method)
    {
	ObjPtr result;
	result = (*method)(dataset);
	if (result)
	{
	    return GetInt(result);
	}
	else
	{
	    ReportError("GetTopDim", "GETTOPDIM did not return dimension");
	    return 0;
	}
    }
    else
    {
	return 0;
    }
}

int GetSpatialDim(dataset)
ObjPtr dataset;
/*Returns the spatial dimensionality of the dataset
  Returns 0 if it has no topological dimensions*/
{
    FuncTyp method;
    method = GetMethodSurely("GetSpatialDim", dataset, GETSPATIALDIM);
    if (method)
    {
	ObjPtr result;
	result = (*method)(dataset);
	if (result)
	{
	    return GetInt(result);
	}
	else
	{
	    return 1;
	}
    }
    else
    {
	return 1;
    }
}

long GetDatasetInfo(dataset)
ObjPtr dataset;
/*Returns the spatial dimensionality of the dataset
  Returns 0 if it has no topological dimensions*/
{
    FuncTyp method;
    method = GetMethodSurely("GetDatasetInfo", dataset, GETDATASETINFO);
    if (method)
    {
	ObjPtr result;
	result = (*method)(dataset);
	if (result)
	{
	    return GetInt(result);
	}
	else
	{
	    ReportError("GetDatasetInfo", "No info returned from GETDATASETINFO");
	    return 0;
	}
    }
    else
    {
	return 0;
    }
}

ObjPtr GetDataFormTopDim(dataForm)
ObjPtr dataForm;
/*Returns the topological dimension of dataForm*/
{
    ObjPtr dims;
    if (GetPredicate(dataForm, UNSTRUCTURED))
    {
	/*Unstructured, ***UPDATE*** assume 2*/
	return NewInt(2);
    }

    dims = GetArrayVar("GetDataFormTopDim", dataForm, DIMENSIONS);
    if (dims)
    {
	return NewInt(DIMS(dims)[0]);
    }
    else
    {
	return NewInt(0);
    }
}

static ObjPtr GetDatasetTopDim(dataset)
ObjPtr dataset;
/*Returns the topological or computational dimensionality of the dataset
  Returns 0 if it has no topological dimensions*/
{
    ObjPtr dataForm;
    FuncTyp method;

    dataForm = GetVar(dataset, DATAFORM);

    if (dataForm)
    {
	method = GetMethod(dataForm, GETTOPDIM);
	if (method)
	{
	    return (*method)(dataForm);
	}
    }
    else
    {
	ObjPtr data;
	/*No data form, must get the top dim of its DATA*/
	data = GetVar(dataset, DATA);
	if (data)
	{
	    method = GetMethod(data, GETTOPDIM);
	    if (method)
	    {
		return (*method)(data);
	    }
	}
    }
	
    return NewInt(0);
}

static ObjPtr GetDatasetSpatialDim(dataset)
ObjPtr dataset;
/*Returns the spatial dimensionality of the dataset*/
{
    ObjPtr dataForm;
    ObjPtr nComponents, bounds;

    dataForm = GetVar(dataset, DATAFORM);
    if (!dataForm)
    {
	return NewInt(0);
    }

    if (GetPredicate(dataForm, UNSTRUCTURED))
    {
	/*Unstructured, ***UPDATE*** assume 3*/
	return NewInt(3);
    }

    nComponents = GetVar(dataForm, NCOMPONENTS);
    if (nComponents)
    {
	return nComponents;
    }

    bounds = GetVar(dataForm, BOUNDS);
    if (bounds)
    {
	return NewInt(DIMS(bounds)[0] / 2);
    }
}

ObjPtr GetDatasetFormBounds(dataset)
ObjPtr dataset;
/*Returns the bounds of the dataset's data form*/
{
    ObjPtr dataForm;
    ObjPtr dims;
    
    dataForm = GetVar(dataset, DATAFORM);
    if (!dataForm)
    {
	return 0;
    }

    dims = GetArrayVar("GetDatasetFormBounds", dataForm, BOUNDS);
    return dims;
}

ObjPtr GetDatasetFormDims(dataset)
ObjPtr dataset;
/*Returns the dimensions of the dataset's data form*/
{
    FuncTyp method;
    method = GetMethodSurely("GetDatasetFormDims", dataset, GETFORMDIMS);

    if (method)
    {
	return (*method)(dataset);
    }
    else
    {
	return NULLOBJ;
    }
}

ObjPtr GetPlainDatasetFormDims(dataset)
ObjPtr dataset;
/*Returns the dimensions of the dataset's data form*/
{
    ObjPtr dataForm;
    ObjPtr dims;
    
    dataForm = GetVar(dataset, DATAFORM);
    if (!dataForm)
    {
	return 0;
    }

    dims = GetArrayVar("GetPlainDatasetFormDims", dataForm, DIMENSIONS);
    return dims;
}

ObjPtr GetDatasetNodes(dataset)
ObjPtr dataset;
/*Returns the nodes of the dataset.*/
{
    ObjPtr dataForm;
    dataForm = GetVar(dataset, DATAFORM);
    if (!dataForm)
    {
	return NULLOBJ;
    }
    return GetVar(dataForm, NODES);
}

ObjPtr GetDatasetKEdges(dataset, k)
ObjPtr dataset;
int k;
/*Returns a list of k-edges of the dataset.  Only works for unstructured 
  data forms.
*/
{
    ObjPtr dataForm;
    dataForm = GetVar(dataset, DATAFORM);
    if (!dataForm)
    {
	return NULLOBJ;
    }
    if (k == 1)
    {
	return GetVar(dataForm, EDGES);
    }
    else if (k == 2)
    {
	return GetVar(dataForm, CELLS);
    }
    else
    {
	ReportError("GetDatasetKEdges", "Edges beyond 2 are not defined");
	return NULLOBJ;
    }
}

static ObjPtr RegisterDatasetField(dataset, whichField)
ObjPtr dataset;
int whichField;
/*Registers datastet's data as whichField.*/
{
    ObjPtr data;
    FuncTyp method;

    MakeVar(dataset, CURDATA);
    data = GetVar(dataset, CURDATA);

    if (!data)
    {
	ReportError("RegisterDatasetField", "No CURDATA");
	return ObjFalse;
    }

    /*It's an array or vector.  Stuff it recursively*/
    method = GetMethodSurely("RegisterDatasetField", data, REGISTERFIELD);
    if (method)
    {
	return (*method)(data, whichField);
    }

    return ObjFalse;
}

int GetNComponents(whichField)
int whichField;
/*Returns the number of components of whichField*/
{
    return curFields[whichField] . nComponents;
}

int CountComponentDims(whichField, whichComponent)
int whichField, whichComponent;
/*Returns a count of whichField's component dimensions*/
{
    return curFields[whichField] . components[whichComponent] . nIndices;
}

int *GetComponentDims(whichField, whichComponent)
int whichField, whichComponent;
/*Returns a count of whichField's component dimensions*/
{
    return curFields[whichField] . components[whichComponent] . dimensions;
}

static ObjPtr RegisterDataFormField(form, whichField)
ObjPtr form;
int whichField;
/*Registers data form form in field whichField*/
{
    ObjPtr dataset;			/*Dataset that may define this dataform*/

    MakeVar(form, DATA);
    dataset = GetVar(form, DATA);
    if (dataset)
    {
	Bool test;
	/*It's based on a dataset*/

	MakeVar(dataset, CURDATA);
	test = SetCurField(whichField, dataset);
	curFields[whichField] . topDim = GetTopDim(form);
	return test ? ObjTrue : ObjFalse;
    }

    /*Assume structured grid*/
    {
	ObjPtr dimsArray, boundsArray;
	real *dimsPtr, *boundsPtr;
	Component *components;
	int whichComponent;
	long topDim;
	Bool omErr = false;		/*True if om error occurred*/

	dimsArray = GetArrayVar("RegisterDataFormField", form, DIMENSIONS);
	boundsArray = GetArrayVar("RegisterDataFormField", form, BOUNDS);
	if (!dimsArray || !boundsArray)
	{
	    return ObjFalse;
	}

	topDim = DIMS(dimsArray)[0];

	/*Topological dimension is now topDim  Create that many components*/
	components = (Component *) malloc(sizeof(Component) * topDim);

	dimsPtr = ELEMENTS(dimsArray);
	boundsPtr = ELEMENTS(boundsArray);

	for (whichComponent = 0; whichComponent < topDim; ++whichComponent)
	{
	    int *indices = 0;
	    long *dimensions = 0;
	    long *steps = 0;
	    ObjPtr dataArray;

	    /*Set up one index, the current component*/
	    components[whichComponent] . nIndices = 1;
	    indices = malloc(sizeof(int));
	    if (indices)
	    {
		*indices = whichComponent;
	    }
	    else
	    {
		omErr = true;
		OMErr();
	    }

	    /*One set of dimensions*/
	    dimensions = malloc(sizeof(int));
	    if (dimensions)
	    {
		*dimensions = *dimsPtr;
	    }
	    else
	    {
		omErr = true;
		OMErr();
	    }

	    /*One step, unity*/
	    steps = malloc(sizeof(int));
	    if (steps)
	    {
		*steps = 1;
	    }
	    else
	    {
		omErr = true;
		OMErr();
	    }

	    /*Fill in a data array*/
	    dataArray = NewRealArray(1, (long) *dimsPtr);
	    if (dataArray)
	    {
		real min, max, *dp;
		long k;
		dp = ELEMENTS(dataArray);
		min = *boundsPtr;
		max = *(boundsPtr + 1);

		for (k = 0; k < (long) *dimsPtr; ++k)
		{
		    *dp = min + (max - min) * (k / (*dimsPtr - 1));
		    ++dp;
		}
	    }
	    else
	    {
		omErr = true;
		OMErr();
	    }
	    components[whichComponent] . flags = 0;
	    components[whichComponent] . data = ELEMENTS(dataArray);
	    components[whichComponent] . indices = indices;
	    components[whichComponent] . dimensions = dimensions;
	    components[whichComponent] . steps = steps;

	    ++dimsPtr;
	    boundsPtr += 2;
	}
	curFields[whichField] . components = components;
	curFields[whichField] . nComponents = topDim;
    }
    return ObjTrue;
}

static ObjPtr DeleteDatasetIcon(icon)
ObjPtr icon;
/*Delete repObj from dataset icon within a corral.  Always OK.*/
{
    ChangeIconButtons(icon);
    return ObjTrue;
}

static ObjPtr CleanupDataset(object)
ObjPtr object;
/*Cleans up a dataset*/
{
    DeleteFromList(allDatasets, object);
    return ObjTrue;
}

static ObjPtr MakeDatasetIconHelp(object, class)
ObjPtr object;
ObjPtr class;
/*Makes a dataset icon help string*/
{
    long info;
    long topDim;
    ObjPtr repObj;
    char *s;

    repObj = GetVar(object, REPOBJ);
    if (!repObj)
    {
	SetVar(class, HELPSTRING,
	   NewString("This icon is supposed to represent a dataset.  Unfortunately, \
it does not appear to be connected to anything right now.  See the section on Reporting Bugs \
to find out how to report this bug."));
	return ObjTrue;
    }
    info = GetDatasetInfo(repObj);
    if (info & DS_HASFORM)
    {
	topDim = GetTopDim(repObj);
    }
    else
    {
	topDim = 0;
    }
    s = tempStr;
    sprintf(s,
	"This icon represents a dataset, which contains");
    while (*s) ++s;
    if (info & DS_HASGEOMETRY)
    {
	sprintf(s, "%s geometry.  ",
	    info & DS_TIMEDEPENDENT ? " time-dependent" : "");
    }
    else
    {
	if (topDim)
	{
	    sprintf(s, " a %d-D%s %s field.  ",
		topDim,
		info & DS_TIMEDEPENDENT ? " time-dependent" : "",
		info & DS_VECTOR ? "vector" : "scalar");
 	}
	else
	{
	    sprintf(s, " a%s %s field.  ",
		info & DS_TIMEDEPENDENT ? " time-dependent" : "",
		info & DS_VECTOR ? "vector" : "scalar");
	}
    }
    while (*s) ++s;
    strcpy(s, "You can select it and use the controls \
in this window to modify or visualize it, or you can drag it into the \
corral of an existing visualization window to visualize it there.");

    SetVar(class, HELPSTRING, NewString(tempStr));
    return ObjTrue;
}

ObjPtr DatasetPaletteSet(dataset)
ObjPtr dataset;
/*Sets PALETTESET on dataset*/
{
    SetVar(dataset, PALETTESET, ObjTrue);
    return ObjTrue;
}

void InitDatasets()
/*Initializes all the kinds of data sets*/
{
    int k;
    for (k = 0; k < MAXNCURFIELDS; ++k)
    {
	curFields[k] . nComponents = 0;
	curFields[k] . components = (Component *) 0;
    }

    iconColorPalette = NewIcon(0, 0, ICONCTABLE, "Color Palette");
    AddToReferenceList(iconColorPalette);
    SetMethod(iconColorPalette, DOUBLECLICK, ShowIconControls);

    iconDataset = NewIcon(0, 0, ICONQUESTION, "Dataset");
    AddToReferenceList(iconDataset);
    SetMethod(iconDataset, MAKE1HELPSTRING, MakeDatasetIconHelp);
    SetMethod(iconDataset, DOUBLECLICK, ShowIconControls);
    SetMethod(iconDataset, DELETEICON, DeleteDatasetIcon);

    icon4DScalar = NewObject(iconDataset, 0);
    SetVar(icon4DScalar, WHICHICON, NewInt(ICON4DSCALAR));
    AddToReferenceList(icon4DScalar);

    icon3DScalar = NewObject(iconDataset, 0);
    SetVar(icon3DScalar, WHICHICON, NewInt(ICON3DSCALAR));
    AddToReferenceList(icon3DScalar);

    icon2DScalar = NewObject(iconDataset, 0);
    SetVar(icon2DScalar, WHICHICON, NewInt(ICON2DSCALAR));
    AddToReferenceList(icon2DScalar);

    icon1DScalar = NewObject(iconDataset, 0);
    SetVar(icon1DScalar, WHICHICON, NewInt(ICON1DSCALAR));
    AddToReferenceList(icon1DScalar);


    icon4DVector = NewObject(iconDataset, 0);
    SetVar(icon4DVector, WHICHICON, NewInt(ICON4DVECTOR));
    AddToReferenceList(icon4DVector);

    icon3DVector = NewObject(iconDataset, 0);
    SetVar(icon3DVector, WHICHICON, NewInt(ICON3DVECTOR));
    AddToReferenceList(icon3DVector);

    icon2DVector = NewObject(iconDataset, 0);
    SetVar(icon2DVector, WHICHICON, NewInt(ICON2DVECTOR));
    AddToReferenceList(icon2DVector);

    icon1DVector = NewObject(iconDataset, 0);
    SetVar(icon1DVector, WHICHICON, NewInt(ICONVECTORS));
    AddToReferenceList(icon1DVector);

    datasetClass = NewObject(NULLOBJ, 0);
    AddToReferenceList(datasetClass);
    SetMethod(datasetClass, GETLONGNAME, GetPlainDatasetLongName);
    SetVar(datasetClass, DEFAULTICON, iconDataset);
    SetVar(datasetClass, NAME, NewString("Dataset"));
    SetMethod(datasetClass, CLEANUP, CleanupDataset);
    SetMethod(datasetClass, GETDATASETINFO, GetPlainDatasetInfo);
    SetMethod(datasetClass, GETMINMAX, GetDataMinMax);
    SetMethod(datasetClass, NEWCTLWINDOW, ShowDatasetControls);
    SetMethod(datasetClass, REGISTERFIELD, RegisterDatasetField);
    SetMethod(datasetClass, REGISTERFORM, RegisterDatasetForm);
    SetMethod(datasetClass, GETTOPDIM, GetDatasetTopDim);
    SetMethod(datasetClass, GETSPATIALDIM, GetDatasetSpatialDim);
    SetMethod(datasetClass, GETFORMDIMS, GetPlainDatasetFormDims);
    DeclareIndirectDependency(datasetClass, PALETTESET, CPALETTE, CHANGED);
    DeclareIndirectDependency(datasetClass, PALETTESET, CPALETTE, JUSTCOLORCHANGE);
    SetMethod(datasetClass, CPALETTE, MakeDatasetPalette);
    SetMethod(datasetClass, PALETTESET, DatasetPaletteSet);
    DeclareDependency(datasetClass, CURDATA, DATA);
    DeclareIndirectDependency(datasetClass, CURDATA, DATA, CURDATA);
    SetMethod(datasetClass, CURDATA, MakeDatasetCurData);
    DeclareDependency(datasetClass, TIMESTEPS, DATA);
    DeclareIndirectDependency(datasetClass, TIMESTEPS, DATA, TIMESTEPS);
    SetMethod(datasetClass, TIMESTEPS, MakeDatasetTimesteps);
    DeclareDependency(datasetClass, CURDATA, TIME);

    dataFormClass = NewObject(NULLOBJ, 0);
    SetVar(dataFormClass, NCOMPONENTS, NewInt(3));
    SetMethod(dataFormClass, REGISTERFIELD, RegisterDataFormField);
    SetMethod(dataFormClass, GETTOPDIM, GetDataFormTopDim);
    /*SetMethod(dataFormClass, NORMALS, MakeDataFormNormals);*/
    AddToReferenceList(dataFormClass);

    data3DScalar = NewObject(datasetClass, 0);
    SetVar(data3DScalar, DEFAULTICON, icon3DScalar);
    AddToReferenceList(data3DScalar);

    data2DScalar = NewObject(datasetClass, 0);
    SetVar(data2DScalar, DEFAULTICON, icon2DScalar);
    AddToReferenceList(data2DScalar);

    data1DVector = NewObject(datasetClass, 0);
    SetVar(data1DVector, DEFAULTICON, icon1DVector);
    AddToReferenceList(data1DVector);

    data3DUnstructSurface = NewObject(datasetClass, 0);
    SetVar(data3DUnstructSurface, DEFAULTICON, icon2DScalar);
    AddToReferenceList(data3DUnstructSurface);

    /*Class for a geometry file*/
    geometryClass = NewObject(datasetClass, 0);
    AddToReferenceList(geometryClass);
    SetVar(geometryClass, NAME, NewString("Geometry"));
    SetVar(geometryClass, DEFAULTICON, NewIcon(0, 0, ICONGEOMETRY, "Geometry"));

    allDatasets = NewList();
    SetMethod(allDatasets, MARK, (FuncTyp) 0);
    AddToReferenceList(allDatasets);
}

void KillDatasets()
/*Kills the data sets*/
{
    int k;
    for (k = 0; k < MAXNCURFIELDS; ++k)
    {
	CleanCurField(k);
    }
    DeleteThing(allDatasets);
    DeleteThing(geometryClass);
    DeleteThing(data3DUnstructSurface);
    DeleteThing(data1DVector);
    DeleteThing(data3DScalar);
    DeleteThing(data2DScalar);
    DeleteThing(dataFormClass);
    DeleteThing(datasetClass);

    DeleteThing(icon1DVector);
    DeleteThing(icon2DVector);
    DeleteThing(icon3DVector);
    DeleteThing(icon4DVector);

    DeleteThing(icon1DScalar);
    DeleteThing(icon2DScalar);
    DeleteThing(icon3DScalar);
    DeleteThing(icon4DScalar);
    DeleteThing(iconDataset);
    DeleteThing(iconColorPalette);
}
