/****************************************************************************
 * vertex.c
 * Author Joel Welling
 * Copyright 1992, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/* 
This module implements vertexs as a user data type.
*/

#include <stdio.h>
#include "ge_error.h"
#include "alisp.h"
#include "p3d.h"
#include "ren.h"

/* Notes-
*/

#define DEFAULT_X 0.0
#define DEFAULT_Y 0.0
#define DEFAULT_Z 0.0
#define DEFAULT_INDEX 0
#define INITIAL_VERTICES 5000;

/* The USERDATA struct referenced by all vertexs */
static USERDATA vertexdef;

/* The list of free vertex data structures, and total number allocated. */
static Vertex_body *free_list= (Vertex_body *)0;
static int total_vertices= 0;

static void vertex_print( data )
char *data;
/* This is the print method for the vertex */
{
  Vertex_body *vtx;
  Color clr;
  Vector norm;

  vtx= (Vertex_body *)data;
  clr= vtx->d.color;
  norm= vtx->d.normal;

  ger_debug("vertex_print");
  fprintf(stderr,"*vertex* ");
#ifdef never
  fprintf(stderr,"vertex: XYZ= (%f, %f, %f)\n",vtx->d.x, vtx->d.y, vtx->d.z);
  if (!null(clr))
    fprintf(stderr,"      color: RGBA= (%f, %f, %f, %f)\n",
	    color_red(clr),color_green(clr),color_blue(clr),
	    color_alpha(clr));
  if (!null(norm))
    fprintf(stderr,"      normal: XYZ= (%f, %f, %f)\n",
	    vector_x(norm),vector_y(norm),vector_z(norm));
#endif
}

void vertex_destroy( data )
char *data;
/* This routine destroys a vertex instance */
{
  Vertex_body *thisvertexdata;
  Color clr;
  Vector vec;

  ger_debug("vertex_destroy");

  thisvertexdata= (Vertex_body *)data;
  clr= thisvertexdata->d.color;
  vec= thisvertexdata->d.normal;

  /* Free the parts */
  free_color(clr);
  free_vector(vec);

  /* Pop the cell onto the free list */
  thisvertexdata->next= free_list;
  free_list= thisvertexdata;
}

static void vertex_setf( place, location, value )
NODE *place, *location, *value;
/* This routine provides setf methods for vertex 'structs'. */
{
  char *fname, *cname;

  ger_debug("vertex_setf");

  fname= getname(place);
  cname= callout_name(place);

  if (!strcmp(fname,"vertex-x") || !strcmp(cname,"~vertex-x"))
    set_vertex_x((Vertex)location,get_number(value));
  else if (!strcmp(fname,"vertex-y") || !strcmp(cname,"~vertex-y"))
    set_vertex_y((Vertex)location,get_number(value));
  else if (!strcmp(fname,"vertex-z") || !strcmp(cname,"~vertex-z"))
    set_vertex_z((Vertex)location,get_number(value));
  else if (!strcmp(fname,"vertex-clr") || !strcmp(cname,"~vertex-clr"))
    set_vertex_color((Vertex)location,(Color)value);
  else if (!strcmp(fname,"vertex-normal") || !strcmp(cname,"~vertex-normal"))
    set_vertex_normal((Vertex)location,(Vector)value);
  else if (!strcmp(fname,"vertex-index") || !strcmp(cname,"~vertex-index"))
    set_vertex_index((Vertex)location,(int)get_number(value));
  else ger_error("vertex_setf: %s is not a setf-able field of a vertex.",
		 fname);
}

static void init_vertex()
/* This routine initializes this module and plugs it into Alisp. */
{
  ger_debug("init_vertex");
  vertexdef.type= "vertex";
  vertexdef.destroy= vertex_destroy;
  vertexdef.print= vertex_print;
  vertexdef.setf_method= vertex_setf;
}

static void allocate_vertices()
/* This routine adds some new vertices to the free list */
{
  int num_to_alloc;
  Vertex_body *new_block;
  int i;

  if (total_vertices) num_to_alloc= total_vertices/2;
  else num_to_alloc= INITIAL_VERTICES;

  ger_debug("allocate_vertices: current total %d; allocating %d more.",
	    total_vertices, num_to_alloc);

  if ( !(new_block= (Vertex_body *)malloc(num_to_alloc*sizeof(Vertex_body))) )
    ger_fatal("allocate_vertices: unable to allocate %d bytes!",
	      num_to_alloc*sizeof(Vertex_body));

  /* String them together and add them to the free list */
  new_block[0].next= free_list;
  for (i=1; i<num_to_alloc; i++) new_block[i].next= &(new_block[i-1]);
  free_list= &(new_block[num_to_alloc - 1]);

  total_vertices += num_to_alloc;
}

static Vertex new_vertex()
/*
This function creates a vertex.  Its x, y, and z components are initially
set to 0.0, its color and normal are NIL, and its index is 0.
*/
{
  Vertex_body *thisvertexdata;
  Vertex thisvertex;
  static int initialized= 0;

  ger_debug("new_vertex:");

  if (!initialized) {
    init_vertex();
    initialized= 1;
  }

  /* Pop a cell off the free list */
  if (!free_list) allocate_vertices();
  thisvertexdata= free_list;
  free_list= free_list->next;

  thisvertexdata->d.x= DEFAULT_X;
  thisvertexdata->d.y= DEFAULT_Y;
  thisvertexdata->d.z= DEFAULT_Z;
  thisvertexdata->d.color= (Color)NIL;
  incr_ref(thisvertexdata->d.color);
  thisvertexdata->d.normal= (Vector)NIL;
  incr_ref(thisvertexdata->d.normal);
  thisvertexdata->d.index= DEFAULT_INDEX;
  
  /* Build the node */
  thisvertex= new_node(N_USERDATA);
  ((NODE *)thisvertex)->val.userdef.data= (char *)thisvertexdata;
  ((NODE *)thisvertex)->val.userdef.methods= &vertexdef;

  return( thisvertex );
}

NODE *vertex_create( arglist )
NODE *arglist;
/* This routine creates a new vertex instance */
{
  Vertex thisvertex;
  Vertex_body *thisvertexdata;

  ger_debug("vertex_create:");

  thisvertex= new_vertex();
  thisvertexdata= (Vertex_body *)((NODE *)thisvertex)->val.userdef.data;

  /* Decode the NODE * here.  It is a list containing the data
   * entries in order.
   */
  if (length_list(arglist) < 6) {
    ger_error("vertex_create: need at least 6 arguments!");
    return( NIL );
  }

  if ( realp(safe_car(arglist)) )
    thisvertexdata->d.x= get_number(safe_car(arglist));
  else {
    ger_error("vertex_create: argument for x not int or real!");
    thisvertexdata->d.x= DEFAULT_X;
  }
  arglist= safe_cdr(arglist);

  if ( realp(safe_car(arglist)) )
    thisvertexdata->d.y= get_number(safe_car(arglist));
  else {
    ger_error("vertex_create: argument for y not int or real!");
    thisvertexdata->d.y= DEFAULT_Y;
  }
  arglist= safe_cdr(arglist);

  if ( realp(safe_car(arglist)) )
    thisvertexdata->d.z= get_number(safe_car(arglist));
  else {
    ger_error("vertex_create: argument for z not int or real!");
    thisvertexdata->d.z= DEFAULT_Z;
  }
  arglist= safe_cdr(arglist);

  /* Color and normal entries should really be tested to see if
   * they are a Color (or NIL) or a normal (or NIL), but this is
   * almost as good and uses only available info.
   */
  if (Type_Of(car(arglist)) == N_USERDATA) {
    decr_elem(thisvertexdata->d.color);
    thisvertexdata->d.color= safe_car(arglist);
    incr_ref(thisvertexdata->d.color);
  }
  arglist= safe_cdr(arglist);

  if (Type_Of(car(arglist)) == N_USERDATA) {
    decr_elem(thisvertexdata->d.normal);
    thisvertexdata->d.normal= safe_car(arglist);
    incr_ref(thisvertexdata->d.normal);
  }
  arglist= safe_cdr(arglist);

  if ( realp(safe_car(arglist)) )
    thisvertexdata->d.index= get_number(safe_car(arglist));
  else {
    ger_error("vertex_create: argument for index not int or real!");
    thisvertexdata->d.z= DEFAULT_INDEX;
  }
  arglist= safe_cdr(arglist);

  return(thisvertex);
}

Vertex create_vertex()
/*
This function creates a vertex.  It's x, y, and z components are initially
set to 0.0.
*/
{
  Vertex thisvertex;
  
  ger_debug("vertex_create:");

  thisvertex= new_vertex();
  incr_ref( (NODE *)thisvertex );
  return( thisvertex );
}

void free_vertex( thisvertex )
Vertex thisvertex;
/*
This routine frees the given vertex.  If this was the last reference to
this vertex, it will be deleted.
*/
{
  ger_debug("free_vertex:");
  decr_elem( (NODE *)thisvertex );
}

Vertex set_vertex_x( thisvertex, value )
Vertex thisvertex;
float value;
/* This routine sets the x value of a vertex */
{
  Vertex_body *thisvertexdata;

  ger_debug("set_vertex_x:");

  if ((Type_Of((NODE *)thisvertex) != N_USERDATA) 
      || (((NODE *)thisvertex)->val.userdef.methods != &vertexdef))
    ger_error("set_vertex_x: passed a non-vertex!");
  else {
    Vertex_body *thisvertexdata;
    thisvertexdata= (Vertex_body *)((NODE *)thisvertex)->val.userdef.data;
    thisvertexdata->d.x= value;
  }

  return(thisvertex);
}

Vertex set_vertex_y( thisvertex, value )
Vertex thisvertex;
float value;
/* This routine sets the y value of a vertex */
{
  Vertex_body *thisvertexdata;

  ger_debug("set_vertex_y:");

  if ((Type_Of((NODE *)thisvertex) != N_USERDATA) 
      || (((NODE *)thisvertex)->val.userdef.methods != &vertexdef))
    ger_error("set_vertex_y: passed a non-vertex!");
  else {
    Vertex_body *thisvertexdata;
    thisvertexdata= (Vertex_body *)((NODE *)thisvertex)->val.userdef.data;
    thisvertexdata->d.y= value;
  }

  return(thisvertex);
}

Vertex set_vertex_z( thisvertex, value )
Vertex thisvertex;
float value;
/* This routine sets the z value of a vertex */
{
  Vertex_body *thisvertexdata;

  ger_debug("set_vertex_z:");

  if ((Type_Of((NODE *)thisvertex) != N_USERDATA) 
      || (((NODE *)thisvertex)->val.userdef.methods != &vertexdef))
    ger_error("set_vertex_z: passed a non-vertex!");
  else {
    Vertex_body *thisvertexdata;
    thisvertexdata= (Vertex_body *)((NODE *)thisvertex)->val.userdef.data;
    thisvertexdata->d.z= value;
  }

  return(thisvertex);
}

Vertex set_vertex_color( thisvertex, value )
Vertex thisvertex;
Color value;
/* This routine sets the color value of a vertex */
{
  Vertex_body *thisvertexdata;

  ger_debug("set_vertex_color:");

  if ((Type_Of((NODE *)thisvertex) != N_USERDATA) 
      || (((NODE *)thisvertex)->val.userdef.methods != &vertexdef))
    ger_error("set_vertex_z: passed a non-vertex!");
  else {
    Vertex_body *thisvertexdata;
    thisvertexdata= (Vertex_body *)((NODE *)thisvertex)->val.userdef.data;
    thisvertexdata->d.color= value;
    incr_ref(value);
  }

  return(thisvertex);
}

Vertex set_vertex_normal( thisvertex, value )
Vertex thisvertex;
Vector value;
/* This routine sets the normal value of a vertex */
{
  Vertex_body *thisvertexdata;

  ger_debug("set_vertex_normal:");

  if ((Type_Of((NODE *)thisvertex) != N_USERDATA) 
      || (((NODE *)thisvertex)->val.userdef.methods != &vertexdef))
    ger_error("set_vertex_normal: passed a non-vertex!");
  else {
    Vertex_body *thisvertexdata;
    thisvertexdata= (Vertex_body *)((NODE *)thisvertex)->val.userdef.data;
    thisvertexdata->d.normal= value;
    incr_ref(value);
  }

  return(thisvertex);
}

Vertex set_vertex_index( thisvertex, value )
Vertex thisvertex;
int value;
/* This routine sets the index value of a vertex */
{
  Vertex_body *thisvertexdata;

  ger_debug("set_vertex_z:");

  if ((Type_Of((NODE *)thisvertex) != N_USERDATA) 
      || (((NODE *)thisvertex)->val.userdef.methods != &vertexdef))
    ger_error("set_vertex_index: passed a non-vertex!");
  else {
    Vertex_body *thisvertexdata;
    thisvertexdata= (Vertex_body *)((NODE *)thisvertex)->val.userdef.data;
    thisvertexdata->d.index= value;
  }

  return(thisvertex);
}

NODE *lisp_vertex_x( arglist )
NODE *arglist;
/* This routine provides the (vertex-x thisvec) function in lisp */
{
  Vertex thisvertex;
  float val;
  NODE *result;

  ger_debug("lisp_vertex_x:");

  if (length_list(arglist) < 1) {
    ger_error("lisp_vertex_x: passed empty list!");
    return(NIL);
  }

  thisvertex= (Vertex)safe_car(arglist);
  val= vertex_x(thisvertex);
  result= get_floatrep(val);
  return(result);
}

NODE *lisp_vertex_y( arglist )
NODE *arglist;
/* This routine provides the (vertex-y thisvec) function in lisp */
{
  Vertex thisvertex;
  float val;
  NODE *result;

  ger_debug("lisp_vertex_y:");

  if (length_list(arglist) < 1) {
    ger_error("lisp_vertex_y: passed empty list!");
    return(NIL);
  }

  thisvertex= (Vertex)safe_car(arglist);
  val= vertex_y(thisvertex);
  result= get_floatrep(val);
  return(result);
}

NODE *lisp_vertex_z( arglist )
NODE *arglist;
/* This routine provides the (vertex-z thisvec) function in lisp */
{
  Vertex thisvertex;
  float val;
  NODE *result;

  ger_debug("lisp_vertex_z:");

  if (length_list(arglist) < 1) {
    ger_error("lisp_vertex_z: passed empty list!");
    return(NIL);
  }

  thisvertex= (Vertex)safe_car(arglist);
  val= vertex_z(thisvertex);
  result= get_floatrep(val);
  return(result);
}

NODE *lisp_vertex_color( arglist )
NODE *arglist;
/* This routine provides the (vertex-clr thisvec) function in lisp */
{
  Vertex thisvertex;
  NODE *result;

  ger_debug("lisp_vertex_color:");

  if (length_list(arglist) < 1) {
    ger_error("lisp_vertex_color: passed empty list!");
    return(NIL);
  }

  thisvertex= (Vertex)safe_car(arglist);
  result= (NODE *)vertex_color(thisvertex);
  return(result);
}

NODE *lisp_vertex_normal( arglist )
NODE *arglist;
/* This routine provides the (vertex-normal thisvec) function in lisp */
{
  Vertex thisvertex;
  NODE *result;

  ger_debug("lisp_vertex_normal:");

  if (length_list(arglist) < 1) {
    ger_error("lisp_vertex_normal: passed empty list!");
    return(NIL);
  }

  thisvertex= (Vertex)safe_car(arglist);
  result= (NODE *)vertex_normal(thisvertex);
  return(result);
}

NODE *lisp_vertex_index( arglist )
NODE *arglist;
/* This routine provides the (vertex-index thisvec) function in lisp */
{
  Vertex thisvertex;
  int val;
  NODE *result;

  ger_debug("lisp_vertex_index:");

  if (length_list(arglist) < 1) {
    ger_error("lisp_vertex_index: passed empty list!");
    return(NIL);
  }

  thisvertex= (Vertex)safe_car(arglist);
  val= vertex_index(thisvertex);
  result= get_integerrep(val);
  return(result);
}

