/****************************************************************************
 * sgigl_ren.c
 * Author Joel Welling
 * Copyright 1990, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
  This renderer interfaces to the SGI GL library.  Traversal is done on the
  C side, in this module rather than within GL.  
  */
#include <stdio.h>
#include <math.h>
#include "alisp.h"
#include "p3d.h"
#include "ge_error.h"
#include "matrix_ops.h"
#include "assist.h"
#include "ren.h"

#ifndef AVOID_X
#include <X11/Xlib.h>
#ifndef _IBMR2
#include <gl/glws.h>
#else /* ifndef _IBMR2 */
#include <sys/types.h>
#include <X11/Intrinsic.h>
#include <gl.h>
#include <device.h>
#endif /* ifndef _IBMR2 */
#else /* ifndef AVOID_X */
#include <gl.h>
#endif /* ifndef AVOID_X */

/*
  Notes: 
  */

/* Set the following switch to avoid using NURBS to draw Bezier patches */
#ifdef never
#define AVOID_NURBS
#endif

/* Global to hook into the X Windows drawing environment.  This external
 * is a 'back door' used to let the renderer control the configuration
 * of the Motif widget into which it will draw.
 */
#ifndef AVOID_X
#ifndef _IBMR2
static GLXconfig glxconfig[]= {
  { GLX_NORMAL, GLX_DOUBLE, TRUE },
  { GLX_NORMAL, GLX_RGB, TRUE },
  { GLX_NORMAL, GLX_ZSIZE, GLX_NOCONFIG },
  { 0, 0, 0 } /* add more entries before this line */
};
GLXconfig *ren_gl_glx_config= glxconfig;
#else /* ifndef _IBMR2 */
int ren_gl_aix_ready= 0;
#endif /* ifndef _IBMR2 */
#endif
     
/* NURBS pixel tolerance */
#define NURBS_PIXEL_TOL 6.0
     
/* IBM AIX misses some definitions */
#ifndef AVOID_NURBS
#ifndef N_V3D
#define N_V3D 0x4c
#endif
#ifndef N_C4D
#define N_C4D 0xd0
#endif
#endif
     
/* Needed external definitions */
#ifndef NOMALLOCDEF
     extern char *malloc(), *realloc();
#endif
     
/* Constants */
#define NORMAL_EPS 0.01
#define OPAQUE_PATTERN_ID 0
#define TRANSPARENT_PATTERN_ID 1
#define ORTHO_STD_FOVEA 10.0 /* degrees, used for contracting eye dist. 
				in ortho projection */
     
/* Symbols defined by ren_setup and used in parsing attribute lists */
static Symbol color_symbol, backcull_symbol, text_height_symbol,
  lighting_model_symbol, material_symbol;

/* Space for the current background color */
static float backgroundvec[3]= {0.0, 0.0, 0.0};

/* An enumerated type and storage to indicate whether a traversal is
 * for rendering or lighting.
 */
enum traversal_type { RENDER, LIGHTING, INVALID };
static enum traversal_type current_traversal= INVALID;

/* Primitive currently being executed, set by fast_render for the benefit
 * of primitive funcions which need data about themselves.
 */
static Primitive current_primitive;

/* Flag to indicate whether or not lighting model should be active */
static int use_lighting_model= 1; /* default on */

/* Array of lights and current binding number.
 * Lights, once defined, are never undefined, so if the metafile
 * keeps creating new light primitives memory usage will accumulate.
 */
static short lights[MAXLIGHTS]=  /* GL defines MAXLIGHTS */
   { LIGHT0, LIGHT1, LIGHT2, LIGHT3, LIGHT4, LIGHT5, LIGHT6, LIGHT7 };
static int current_bind= 0;

/* Current color mode */
static long current_color_mode= -100; /* defaults to non-existent mode */

/* Current material */
static Material current_material= (Material)0;
     
/* Lighting model */
static float lighting_model[]= {
  AMBIENT, 0.0, 0.0, 0.0,
  LMNULL 
  };
     
/* Identity matrix, and rotation component of current camera matrix */
static Matrix idmat= { 1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0,
			 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0};
static float *current_camera_rot= (float *)idmat;
     
/* The following generic chunk of code implements the hash table used to
 * translate prim id's into light id's.  Things get added to the table
 * by ren_light or ren_ambient the first time those routines are called for
 * a given primitive, and accessed by the same routine on subsequent calls.
 */
/* here the error codes are the same since 0 is an acceptable invalid index */
typedef short Hash_result;
#define BAD_HASH_LOOKUP 0 
#define BAD_HASH_ADD 0
/********** Hash table implementation follows **********************/
/* What the int is being hashed to, and failure return code.
   failed lookups return (Hash_result)BAD_HASH_LOOKUP 
   multiple defines of same value return (Hash_result)BAD_HASH_ADD
   */
     
/* Hash table size */
#define hashsize 4099 /* a prime number */
     
typedef struct hash_cell_struct {
  int val;
  Hash_result result;
  struct hash_cell_struct *next, *last;
} Hash_cell;

/* Hash table itself- static allocation assures all the elements will start
 * out null.
 */
static Hash_cell *hashtable[ hashsize ];

static Hash_result hash_lookup(val)
int val;
/* Hash table lookup */
{
  int id;
  Hash_cell *thiscell;
  
  id= abs( val % hashsize );
  for (thiscell= hashtable[id]; thiscell; thiscell= thiscell->next)
    if ( thiscell->val== val ) return(thiscell->result);
  return( (Hash_result) BAD_HASH_LOOKUP );
}

static Hash_result hash_add( val, result )
int val;
Hash_result result;
/* Add a value-result pair to hash table */
{
  int id;
  Hash_cell *thiscell;
  
  if ( hash_lookup(val) != (Hash_result)BAD_HASH_LOOKUP )
    return( (Hash_result)BAD_HASH_ADD );
  
  id= abs( val % hashsize );
  if (!(thiscell= (Hash_cell *)malloc( sizeof(Hash_cell) )))
    ger_fatal("hash_add: unable to allocate %d bytes!\n",
	      sizeof(Hash_cell));
  thiscell->val= val;
  thiscell->result= result;
  thiscell->next= hashtable[id];
  if (hashtable[id]) hashtable[id]->last= thiscell;
  thiscell->last= NULL;
  hashtable[id]= thiscell;
  
  return( result );
}

static void hash_free(val)
int val;
/* Free a value-result pair from the hash table.  Does nothing if the
 * value is not found.
 */
{
  int id;
  Hash_cell *thiscell;
  
  id= abs( val % hashsize );
  for (thiscell= hashtable[id]; thiscell; thiscell= thiscell->next)
    if ( thiscell->val==val ) {
      if (thiscell->next) thiscell->next->last= thiscell->last;
      if (thiscell->last) thiscell->last->next= thiscell->next;
      else hashtable[id]= thiscell->next;
      free( (char *)thiscell );
    }
}
/****************End of hash table implementation*******************/

/*
  This routine dumps the current transformation matrix.  It is a debugging
  aid.
  */
static void dump_current_matrix()
{
  Matrix currentmatrix;
  int i;
  
  ger_debug("dump_current_matrix:");
  
  getmatrix(currentmatrix);
  fprintf(stderr,"Current transformation matrix:\n");
  for (i=0; i<4; i++) 
    fprintf(stderr,"      %f %f %f %f\n",
	    currentmatrix[i][0], currentmatrix[i][1],
	    currentmatrix[i][2], currentmatrix[i][3]);
  fprintf(stderr,"\n");
}

static void create_transparency_pattern()
     /* This routine creates a half-full pattern, to be used to simulate 
      * transparency.  Unfortunately it's not possible to tell if this
      * machine has alpha planes, so we use 'screen dooring' to make
      * objects with sufficiently low opacity look transparent.
      */
{
  static unsigned short mask[16]= { 
    0xAAAA, 0x5555, 0xAAAA, 0x5555, 0xAAAA, 0x5555,
    0xAAAA, 0x5555, 0xAAAA, 0x5555, 0xAAAA, 0x5555,
    0xAAAA, 0x5555, 0xAAAA, 0x5555 };
  
  defpattern( TRANSPARENT_PATTERN_ID, 16, mask );
}

/* 
  This routine should dump the renderer's internal definition
  of an object. 
  */
void ren_dump(thisgob)
Gob thisgob;
{
  int thisid;
  
  thisid= gob_idnum(thisgob);
  ger_debug("ren_dump: dump requested for gob number %d",thisid);
  fprintf(stderr,
	  "ren_dump not implemented for this renderer; can't dump gob %d\n",
	  thisid);
}

static void clear_screen()
/* This routine just clears the window. */
{
  ger_debug("clear_screen");
  
  if (current_color_mode!=LMC_COLOR) {
    current_color_mode= LMC_COLOR;
    lmcolor(current_color_mode);
  }
  c3f(backgroundvec);
  setpattern( OPAQUE_PATTERN_ID );
  clear();
}

static void inherit_color()
/* This routine installs the current color attribute.  Note that it
 * expects inherit_material to have been called first.
 */
{
  Color color;
  float clr[3];
  float kd;
  
  ger_debug("inherit_color");
  color= ast_color_attribute(color_symbol);
  if (use_lighting_model) kd= material_kd(current_material);
  else kd= 1.0;
  clr[0]= kd*color_red(color);
  clr[1]= kd*color_green(color);
  clr[2]= kd*color_blue(color);
  c3f(clr);
  if ( color_alpha(color) <= 0.5 ) setpattern( TRANSPARENT_PATTERN_ID );
  else setpattern( OPAQUE_PATTERN_ID );
}

static void inherit_backcull()
/* This routine turns backface culling on and off as appropriate. */
{
  backface( ast_bool_attribute(backcull_symbol) );
}

static void inherit_material()
/* This routine sets the material appropriately. */
{
  static Material oldmaterial = (Material)0;
  short material_id;
  
  ger_debug("inherit_material");
  if ( use_lighting_model ) {
    current_material= ast_material_attribute(material_symbol);
    if ( current_material != oldmaterial ) {
      material_id= *((short *)material_ren_data(current_material));
      lmbind(MATERIAL,material_id);
      oldmaterial= current_material;
    }
  }
}

static void do_vertex( vertex )
Vertex vertex;
/* 
  This routine passes a vertex to the geometry pipeline.
*/
{
  float coords[3], norm[3], clr[3], lengthsqr, length, kd;
  Vector normal;
  Color color;
  
  normal= vertex_normal(vertex);
  if ( !null(normal) && use_lighting_model ) {
    if (current_color_mode != LMC_AD) {
      current_color_mode= LMC_AD;
      lmcolor(current_color_mode);
      inherit_color(); /* call in primitive routine gets lost */
    }
  }
  else /* no normals */
    if (current_color_mode!=LMC_COLOR) {
      current_color_mode= LMC_COLOR;
      lmcolor(current_color_mode);
      inherit_color(); /* call in primitive routine gets lost */
    }
  if ( !null( color= vertex_color(vertex) ) ) {
    if (use_lighting_model) kd= material_kd(current_material);
    else kd= 1.0;
    clr[0]= kd * color_red( color );
    clr[1]= kd * color_green( color );
    clr[2]= kd * color_blue( color );
    c3f( clr );
    if ( color_alpha(color) <= 0.5 ) setpattern( TRANSPARENT_PATTERN_ID );
    else setpattern( OPAQUE_PATTERN_ID );
  }
  if ( !null(normal) && use_lighting_model ) {
    norm[0]= vector_x( normal );
    norm[1]= vector_y( normal );
    norm[2]= vector_z( normal );
    lengthsqr= norm[0]*norm[0]+norm[1]*norm[1]+norm[2]*norm[2];
    if ( ( fabs(lengthsqr-1.0) > NORMAL_EPS )&&(lengthsqr != 0.0) ) {
      length= sqrt(lengthsqr);
      norm[0]= norm[0]/length;
      norm[1]= norm[1]/length;
      norm[2]= norm[2]/length;
    }
    n3f( norm );
  }
  coords[0]= vertex_x( vertex );
  coords[1]= vertex_y( vertex );
  coords[2]= vertex_z( vertex );
  v3f( coords );
}

void ren_sphere()
/*
  This routine draws a sphere primitive object.
*/
{
  ger_debug("ren_sphere");
  if (current_traversal != RENDER) return;
  ast_sphere(ren_mesh);
}

void ren_cylinder()
/*
  This routine draws a cylinder primitive object.
  */
{
  ger_debug("ren_cylinder");
  if (current_traversal != RENDER) return;
  ast_cylinder(ren_mesh);
}

void ren_torus(bigradius, smallradius)
float bigradius, smallradius;
/*
  This routine draws a torus primitive object.
 */
{
  ger_debug("ren_torus");
  if (current_traversal != RENDER) return;
  ast_torus( bigradius, smallradius, ren_mesh );
}

void ren_polyline( vlist, count )
Vertex_list vlist;
int count;
/*
  This routine draws a polyline primitive object.
*/
{
  Color color;
  float clr[3];
  int shortcount;
  Vertex thisvertex= (Vertex)NIL;
  
  ger_debug("ren_polyline");
  
  if (current_traversal != RENDER) return;
  
  if (count<2) {
    ger_error("ren_polyline:  not enough vertices!"); 
    return;
  }
  inherit_material();
  inherit_color();
  
  /* Because we can only do 256 vertices at a time, we take the points
   * in 255 vertex groups, always saving the most recently used vertex
   * so that it can be used to connect the next group to the last.
   */
  while (count) {
    shortcount= ( count>255 ) ? 255 : count;
    bgnline();
    if ( !null(thisvertex) ) do_vertex(thisvertex); /* for continuations */
    while (shortcount--) {
      thisvertex= first_vertex( vlist );
      do_vertex( thisvertex );
      vlist= rest_of_vertices( vlist );
      count--;
    }
    endline();
  }
}

void ren_polymarker( vlist, count )
Vertex_list vlist;
int count;
/*
  This routine draws a polymarker primitive object.
*/
{
  int shortcount;
  
  ger_debug("ren_polymarker");
  
  if (current_traversal != RENDER) return;
  
  if (count<1) {
    ger_error("ren_polymarker:  not enough vertices!"); 
    return;
  }
  inherit_material();
  inherit_color();
  while (count) {
    shortcount= ( count>256 ) ? 256 : count;
    bgnpoint();
    while (shortcount--) {
      do_vertex( first_vertex( vlist ) );
      vlist= rest_of_vertices( vlist );
      count--;
    }
    endpoint();
  }
}

void ren_polygon( vlist, count )
Vertex_list vlist;
int count;
/*
  This routine draws a polygon primitive object.
*/
{
  ger_debug("ren_polygon");
  
  if (current_traversal != RENDER) return;
  
  if (count<3) {
    ger_error("ren_polygon:  not enough vertices!"); 
    return;
  }
  
  /* Enforce 256 vertex limit */
  if (count>256) {
    ger_error("ren_polygon: too many vertices; found %d, using first %d",
	      count, 256);
    count= 256;
  }
  
  /* Permanently add normals to vertices if needed */
  if ( null( vertex_normal( first_vertex(vlist) ) ) ) 
    ast_polygon_normals( vlist, count );
  
  inherit_backcull();
  inherit_material();
  inherit_color();
  bgnpolygon();
  while (count--) {
    do_vertex( first_vertex( vlist ) );
    vlist= rest_of_vertices( vlist );
  }
  endpolygon();
}

void ren_triangle( vlist, count )
Vertex_list vlist;
int count;
/*
  This routine draws a triangle strip primitive object.
*/
{
  ger_debug("ren_triangle");
  
  if (current_traversal != RENDER) return;
  
  if (count<3) {
    ger_error("ren_triangle:  not enough vertices!"); 
    return;
  }
  
  /* Permanently add normals to vertices if needed */
  if ( null( vertex_normal( first_vertex(vlist) ) ) ) 
    ast_triangle_normals( vlist, count );
  
  inherit_backcull();
  inherit_material();
  inherit_color();
  bgntmesh();
  while (count--) {
    do_vertex( first_vertex( vlist ) );
    vlist= rest_of_vertices( vlist );
  }
  endtmesh();
}

void ren_mesh( vlist, vcount, flist, fcount )
Vertex_list vlist;
Facet_list flist;
int vcount, fcount;
/*
  This routine draws a general mesh object from the args passed it.
*/
{
  Vertex_list thisfacet;
  int facet_vertices;
  
  ger_debug("ren_mesh");
  
  if (current_traversal != RENDER) return;
  
  if (fcount<0) {
    ger_error("ren_mesh: invalid facet count!");
    return;
  }
  
  /* Permanently add normals to vertices if needed */
  if ( null( vertex_normal( first_vertex(vlist) ) ) ) 
    ast_mesh_normals( vlist, vcount, flist, fcount );
  
  while (fcount--) {
    thisfacet= first_facet( flist );
    flist= rest_of_facets( flist );
    facet_vertices= list_length( thisfacet );
    ren_polygon( thisfacet, facet_vertices );
  };
}

static void impl_trans(trans)
Transformation trans;
/* This function is used by ast_text to do transformations between letters. */
{
  float *transmatrix;
  
  transmatrix= transpose( array2d_to_c( trans ) );
  multmatrix( *(Matrix *)transmatrix );
  free( (char *)transmatrix );
}

void ren_text(txtpoint,uvec,vvec,txtstring)
Point txtpoint;
Vector uvec, vvec;
char *txtstring;
/*
 * This routine draws a text primitive object.
 */
{
  ger_debug("ren_text");
  if (current_traversal != RENDER) return;
  pushmatrix();
  ast_text( txtpoint, uvec, vvec, txtstring, impl_trans, ren_polyline );
  popmatrix();
}

void ren_bezier( vlist, count )
Vertex_list vlist;
int count;
/*
  This routine draws a Bezier patch primitive object.
*/
{
  static double bez_knots[8]= {0.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0, 1.0};
  double coords[3*16];
  double colors[4*16];
  double *coordptr, *clrptr;
  Color thiscolor;
  int i;
  int color_flag= 0;
  
  ger_debug("ren_bezier");
  
  if (current_traversal != RENDER) return;
  
#ifdef AVOID_NURBS
  ast_bezier( vlist, ren_mesh );
#else
  
  if ( !null( vertex_color( first_vertex(vlist) ) ) ) color_flag= 1;
  
  coordptr= coords;
  clrptr= colors;
  for (i=0; i<16; i++) {
    *coordptr++= vertex_x( first_vertex( vlist ) );
    *coordptr++= vertex_y( first_vertex( vlist ) );
    *coordptr++= vertex_z( first_vertex( vlist ) );
    if (color_flag) {
      thiscolor= vertex_color( first_vertex(vlist) );
      if ( !null( thiscolor ) ) {
	*clrptr++= color_red( thiscolor );
	*clrptr++= color_green( thiscolor );
	*clrptr++= color_blue( thiscolor );
      }
      else {
	ger_error("ren_bezier: expected vertex color info not found!");
	color_flag= 0;
      }
    }
    vlist= rest_of_vertices( vlist );
  }
  
  bgnsurface();
  
  if (color_flag) nurbssurface( 8, bez_knots, 8, bez_knots, 
			       4*sizeof(double), 4*4*sizeof(double), colors,
			       4, 4, N_C4D );
  
  nurbssurface( 8, bez_knots, 8, bez_knots, 
	       3*sizeof(double), 4*3*sizeof(double), coords,
	       4, 4, N_V3D );
  
  endsurface();
#endif /* else of AVOID_NURBS */
}

void ren_gob(current_gob, trans, attr, primitive, children )
int current_gob;
Transformation trans;
Attribute_list attr;
Primitive primitive;
Child_list children;
/*
  This routine sees all gobs as they are defined.  Since traversal is done
  from the lisp-side DAG, there is no need to do anything here.
*/
{
  ger_debug("ren_gob: Defining gob %d.", current_gob);
}

static int def_light(propcount, props)
int propcount;
float *props;
/* This routine defines a light, returning its index. */
{
  static int lightid= 1;
  
  lmdef( DEFLIGHT, lightid++, propcount, props );
  return( lightid-1 );
}

static void bind_light(lightindex)
short lightindex;
/* This routine binds the given light into the lighting environment, if
 * there are slots left.
 */
{
  ger_debug("bind_light: adding light index %d as light %d.",
	    lightindex, current_bind);
  
  if (current_bind<MAXLIGHTS) lmbind( lights[current_bind++], lightindex );
  else ger_error("bind_light: tried to bind %d lights; %d allowed.",
		 ++current_bind, MAXLIGHTS);
}

static void unbind_lights()
/* This routine unbinds all lights. */
{
  int i;
  
  ger_debug("unbind_lights: unbinding lights.");
  
  if ( MAXLIGHTS < current_bind ) current_bind= MAXLIGHTS;
  for (i=0; i<current_bind; i++) lmbind( lights[i], 0 );
  current_bind= 0;
}

void ren_light( location, lightcolor )
Point location;
Color lightcolor;
/*                                             
  This routine creates a light primitive object.
*/
{
  short lightindex;
  int primid;
  static float directional_light[]= {
    LCOLOR, 0.7, 0.7, 0.7,
    POSITION, 0.0, 0.0, 1.0, 0.0,
    LMNULL };
  
  ger_debug("ren_light");
  
  if (current_traversal != LIGHTING) return;
  
  /* Check the light hash table to see if the light exists.  If not,
   * it must be defined.
   */
  lightindex= hash_lookup( primitive_id(current_primitive) );
  if (!lightindex) {
    /* Build a light with the given color at the given position */
    directional_light[1]= color_red( lightcolor );
    directional_light[2]= color_green( lightcolor );
    directional_light[3]= color_blue( lightcolor );
    directional_light[5]= point_x( location );
    directional_light[6]= point_y( location );
    directional_light[7]= point_z( location );
    lightindex= def_light( 10, directional_light );
    if ( !hash_add( primitive_id(current_primitive), lightindex ) )
      ger_error("ren_light: attempted to redefine existing light primitive %d",
		primitive_id(current_primitive));
  }
  
  /* Bind the light into the lighting environment. */
  bind_light( lightindex );
}

void ren_ambient(lightcolor)
Color lightcolor;
/*                                             
  This routine creates an ambient light primitive object.
*/
{
  static float ambient_light[]= {
    AMBIENT, 0.0, 0.0, 0.0,
    LCOLOR, 0.0, 0.0, 0.0,
    POSITION, 0.0, 0.0, 0.0, 0.0,
    LMNULL };
  int lightindex;
  
  ger_debug("ren_ambient");
  
  if (current_traversal != LIGHTING) return;
  
  /* Check the light hash table to see if the light exists.  If not,
   * it must be defined.
   */
  lightindex= hash_lookup( primitive_id(current_primitive) );
  if (!lightindex) {
    /* Build a light with the given color at the given position */
    ambient_light[1]= color_red( lightcolor );
    ambient_light[2]= color_green( lightcolor );
    ambient_light[3]= color_blue( lightcolor );
    lightindex= def_light( 14, ambient_light );
    if ( !hash_add( primitive_id(current_primitive), lightindex ) )
      ger_error(
		"ren_ambient: attempted to redefine existing light primitive %d",
		primitive_id(current_primitive));
  }
  
  /* Bind the light into the lighting environment. */
  bind_light( lightindex );
}

static float *get_normal_component( v1x, v1y, v1z, v2x, v2y, v2z )
float v1x, v1y, v1z, v2x, v2y, v2z;
/* This routine returns a 4-vector holding the perpendicular component
 * of v1 with respect to v2, or (0, 1, 0, 1) if the two vectors are
 * parallel.
 */
{
  float *perpvec;
  float dot, normsqr;
  
  if ( !(perpvec= (float *)malloc( 4*sizeof(float) ) ) )
    ger_fatal("get_normal_component: unable to allocate 4 floats!\n");
  
  dot= v1x*v2x + v1y*v2y + v1z*v2z;
  normsqr= v2x*v2x + v2y*v2y + v2z*v2z;
  
  perpvec[0]= v1x - v2x*dot/normsqr;
  perpvec[1]= v1y - v2y*dot/normsqr;
  perpvec[2]= v1z - v2z*dot/normsqr;
  perpvec[3]= 1.0;
  
  if ( (perpvec[0]==0.0) && (perpvec[1]==0.0) && (perpvec[2]==0.0) ) {
    ger_error("get_normal_component: up and viewing vectors align; using Y.");
    perpvec[1]= 1.0;
  }
  
  return( perpvec );
}

static void calc_ortho_screen( width, height, dx, dy, dz,
			      fov, xsize, ysize )
float *width, *height;
float dx, dy, dz, fov, xsize, ysize;
/* This routine calculates equivalent orthographic projection screen
 * dimensions for a given fovea and view distance.
 */
{
  float range, min_dim;
  
  ger_debug("calc_ortho_screen:");
  
  range= sqrt( dx*dx + dy*dy + dz*dz );
  
  if (fov==0.0) fov= 0.5; /* to avoid overflow */
  min_dim= 2.0*range*tan( DegtoRad*fov/2.0 );
  if (xsize<ysize) {
    *width= min_dim;
    *height= (ysize/xsize) * min_dim;
  }
  else {
    *height= min_dim;
    *width= (xsize/ysize) * min_dim;
  }
}

void ren_camera( lookatpt, lookfrom, lookup, fov, hither, yon, background )
Point lookatpt, lookfrom;
Vector lookup;
float fov, hither, yon;
Color background;
/*
  This routine sets the current camera.
*/
{
  float *trans, *rot, *urot, *temp, *cameramatrix, *upvec, *rup;
  float fx, fy, fz, dx, dy, dz, upx, upy, upz;
  Screencoord left, right, bottom, top;
  float xsize, ysize, width, height;
  
  ger_debug("ren_camera");
  
  /* Free old camera rotation, if it exists */
  if (current_camera_rot != (float *)idmat ) 
    free( (char *)current_camera_rot );
  
  /* Extract components */
  fx= point_x(lookfrom);
  fy= point_y(lookfrom);
  fz= point_z(lookfrom);
  dx= point_x(lookatpt) - point_x(lookfrom);
  dy= point_y(lookatpt) - point_y(lookfrom);
  dz= point_z(lookatpt) - point_z(lookfrom);
  upx= vector_x(lookup);
  upy= vector_y(lookup);
  upz= vector_z(lookup);
  
  /* Install the new projection matrix.  We use orthographic
   * projection if the fovea is less than 1.0 degree.
   */
  getviewport(&left,&right,&bottom,&top);
  xsize= (float)(right-left);
  ysize= (float)(top-bottom);
  if (fov<1.0) { /* orthographic projection */
    float scale;
    
    /* Calculate view dimensions */
    calc_ortho_screen( &width, &height, dx, dy, dz, fov, xsize, ysize );
    
    /* Contract the view distance to something reasonable */
    scale= tan( DegtoRad * fov ) / tan( DegtoRad * ORTHO_STD_FOVEA );
    fx= fx + (1.0-scale)*dx;
    fy= fy + (1.0-scale)*dy;
    fz= fz + (1.0-scale)*dz;
    dx *= scale;
    dy *= scale;
    dz *= scale;
    hither *= scale;
    yon *= scale;
    
    /* Set up projection */
    ortho( -width/2.0, width/2.0, -height/2.0, height/2.0, -hither, -yon );
  }
  else perspective( (int)(10.0*fov), xsize/ysize, -hither, -yon );
  
  /* Construct matrix to implement camera position */
  /* Translate the lookfrom point to the origin */
  trans= make_translate_c( -fx, -fy, -fz );
  
  /* Rotate the lookat point onto the -z axis */
  rot= make_aligning_rotation( dx, dy, dz, 0.0, 0.0, -1.0 );
  
  /* Align the up vector with the y axis */
  upvec= get_normal_component( upx, upy, upz, dx, dy, dz ); 
  rup= matrix_vector_c( rot, upvec );
  urot= make_aligning_rotation(rup[0], rup[1], rup[2], 0.0, 1.0, 0.0);
  
  /* Construct and save the matrix */
  temp= matrix_mult_c( rot, trans );
  cameramatrix= matrix_mult_c( urot, temp );
  current_camera_rot= transpose(cameramatrix);
  
  /* Set background color, used when screen is cleared. */
  backgroundvec[0]= color_red( background );
  backgroundvec[1]= color_green( background );
  backgroundvec[2]= color_blue( background );
  
  /* clean up */
  free( (char *)trans );
  free( (char *)rot );
  free( (char *)urot );
  free( (char *)temp );
  free( (char *)upvec );
  free( (char *)rup );
  
}

static void fast_render(thisgob)
Gob thisgob;
/* This routine renders a gob, by handling its transformation and attributes
 * and either triggering execution of its primitive or rendering its
 * children.  It is used to traverse both the lighting and geometry
 * gobs.
 */
{
  int thisid;
  Attribute_list newattr;
  Transformation newtrans;
  float *transmatrix;
  Primitive newprim;
  Child_list kidlist;
  
  thisid= gob_idnum( thisgob );
  ger_debug("fast-render: rendering object given by gob %d", thisid);
  
  /* If there is a transformation, add it to the transformation stack */
  if ( !null( newtrans=gob_trans(thisgob) ) ) {
    transmatrix= transpose( array2d_to_c( newtrans ) );
    pushmatrix();
    multmatrix( *(Matrix *)transmatrix );
  }
  /* Check for and if necessary handle new attributes */
  if ( !null( newattr=gob_attr(thisgob) ) )
    ast_push_attributes( newattr );
  
  /* Either execute the primitive, or render the children */
  if ( !null( newprim= gob_primitive(thisgob) ) ) {
    current_primitive= newprim;
    eval_function( primitive_op(newprim) );
  }
  else {
    kidlist= gob_children( thisgob );
    while ( !null(kidlist) ) {
      fast_render( first_child( kidlist ) );
      kidlist= rest_of_children( kidlist );
    }
  }
  
  /* Pop attributes and transformations, and clean up. */
  if ( !null(newattr) ) ast_pop_attributes( newattr );
  if ( !null(newtrans) ) {
    popmatrix();
    free( (char *)transmatrix );
  }
}

void ren_render(thisgob, thistrans, thisattr)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattr;
/* 
  This routine initiates rendering, by handling the default attribute
  list and transformation and rendering the gob.
*/
{
  int thisid; 
  float *transmatrix;
  
  thisid= gob_idnum( thisgob );
  ger_debug("render: rendering object given by gob %d", thisid);
  
  /* Handle top-level transformation and attributes */
  loadmatrix(*(Matrix *)current_camera_rot);
  if ( !null(thistrans) ) {
    transmatrix= transpose( array2d_to_c(thistrans) );
    pushmatrix();
    multmatrix( *(Matrix *)transmatrix );
  }
  if ( !null(thisattr) ) ast_push_attributes( thisattr );
  
  /* Clear the display, and set and zero the z-buffer */
  clear_screen();
  lsetdepth(0,0x7fffff);
  zbuffer(TRUE);
  zclear();
  
  /* Set the traversal type indicator, and render the gob */
  current_color_mode= LMC_COLOR;
  lmcolor(current_color_mode);
  current_traversal= RENDER;
  fast_render( thisgob );
  
  /* Flush DGL pipeline if it is in use, and swap buffers */
#ifndef _IBMR2
  gflush();
#endif
  swapbuffers();
  
  /* Clean up */
  if ( !null(thisattr) ) ast_pop_attributes( thisattr );
  if ( !null(thistrans) ) {
    popmatrix();
    free( (char *)transmatrix );
  }
}

void ren_traverselights(thisgob, thistrans, thisattr)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattr;
/*
  This routine traverses (recursively) a gob, looking for light sources
  and defining them as they are found.
*/
{
  int thisid; 
  float *transmatrix;
  static int most_recent_gob= -1; /* impossible value */
  
  /* Skip this whole business if the lighting model is off. */
  thisid= gob_idnum( thisgob );
  if (!use_lighting_model) {
    ger_debug("traverselights: lighting model not active, ignoring gob %d",
	      thisid);
    return;
  }
  else ger_debug("traverselights: traversing light object given by gob %d", 
		 thisid);
  
  /* Do not reset the lights if they are already set up. */
  if (thisid==most_recent_gob) {
    ger_debug("               this light gob already set;  doing nothing.");
    return;
  }
  most_recent_gob= thisid;
  
  /* Handle top-level transformation and attributes */
  loadmatrix(*(Matrix *)current_camera_rot);
  if ( !null(thistrans) ) {
    transmatrix= transpose( array2d_to_c(thistrans) );
    pushmatrix();
    multmatrix( *(Matrix *)transmatrix );
  }
  if ( !null(thisattr) ) ast_push_attributes( thisattr );
  
  /* Set the traversal type indicator, free existing lights, 
   * and traverse the gob. 
   */
  current_traversal= LIGHTING;
  unbind_lights();
  fast_render( thisgob );
  
  /* Clean up */
  if ( !null(thisattr) ) ast_pop_attributes( thisattr );
  if ( !null(thistrans) ) {
    popmatrix();
    free( (char *)transmatrix );
  }
}

void ren_free(thisgob)
Gob thisgob;
/* 
  This routine frees memory associated with the given gob.  Since traversal
  takes place on the lisp side, there is no memory associated with the gobs
  in the renderer, so this routine does nothing.
*/
{
  int thisid;
  
  thisid= gob_idnum( thisgob );
  ger_debug("ren_free:  freeing gob %d",thisid);
}

#ifndef AVOID_X
#ifdef _IBMR2
/* This routine handles GL window initialization in an IBM AIX 
 * Motif environment.
*/
void ren_gl_aix_init(w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
  /* Do all the GL initialization here */
  RGBmode();
  doublebuffer();
  gconfig();
  ren_gl_aix_ready= 1;
}
#endif /* ifdef _IBMR2 */
#endif /* ifndef AVOID_X */

void ren_setup(renderer,device,open_device,outfile,hints)
char *renderer, *device, *outfile;
int open_device;
Attribute_list hints;
/* This routine initializes the renderer, if it is not already initialized. */
{
  static int initialized=0; /* to hold initialization state */
  Pair thispair;
  
  ger_debug("ren_setup: initializing renderer; renderer %s, device %s",
	    renderer, device);
  
  /* Ultimately renderer may be useful;  for the moment it is ignored.  */
  
  if (!initialized) {
    
    /* Generate some symbols to be used later in attribute list parsing. */
    color_symbol= create_symbol("color");
    backcull_symbol= create_symbol("backcull");
    text_height_symbol= create_symbol("text-height");
    lighting_model_symbol= create_symbol("lighting-model");
    material_symbol= create_symbol("material");

    /* If open_device is false, someone else (presumably the UI) has 
     * initialized GL.  If not, do so now.
     */
    if (open_device) {
#ifndef _IBMR2 /* IBM R6000 series doesn't support this */
      foreground();
#endif
      noborder();
      prefposition(0,639,0,479);
#ifdef never
      prefposition(0,1280,0,1024);
#endif
      winopen("P3D");
      RGBmode();
      
      doublebuffer();
      gconfig();
    }
        
    create_transparency_pattern();
    
    lsetdepth(0,0x7fffff);
    zbuffer(TRUE);
    clear_screen();
    swapbuffers();
    mmode(MVIEWING);
    
    /* NURBS properties, potentially used in ren_bezier() */
#ifndef AVOID_NURBS
    setnurbsproperty(N_PIXEL_TOLERANCE, NURBS_PIXEL_TOL);
#endif
    
    /* If the hints list contains information on the lighting model
     * switch, use it.  If not, the value is left with it's default,
     * which is set where the global variable is defined.
     */
    if ( !null( thispair=symbol_attr(lighting_model_symbol,hints) ) )
      use_lighting_model= pair_boolean( thispair );
    
    if (use_lighting_model) {
      
      /* Define lighting model */
      lmdef(DEFLMODEL, 1, 5, lighting_model);
      
      /* Bind lighting model */
      lmbind(MATERIAL, 0); /* new material will get bound before drawing */
      lmbind(LMODEL, 1);
    }
    
    initialized= 1;
  }
  else ger_error("ren_setup: called twice, this call ignored");
  
}

char *ren_def_material(thismaterial)
Material thismaterial;
/* This routine defines a material */
{
  static int material_id= 2;
  short *id_ptr;
  static float material[]= {
    SPECULAR, 0.0, 0.0, 0.0,
    DIFFUSE,  0.0, 0.0, 0.0,
    AMBIENT,  0.0, 0.0, 0.0,
    SHININESS, 0.0,
    LMNULL
    };
  
  ger_debug("ren_def_material: defining material %d",material_id);
  
  if ( !(id_ptr= (short *)malloc(sizeof(short))) )
    ger_fatal("ren_def_material: unable to allocate %d bytes!\n",
	      sizeof(short));
  
  if (use_lighting_model) {
    *id_ptr= material_id++;
    material[1] = material[2] = material[3] = material_ks(thismaterial);
    if ( material_kd(thismaterial) < 0.35 )
      material[5] = material[6] = material[7] = 0.35;
    else
      material[5] = material[6] = material[7] = material_kd(thismaterial);
    material[13]= material_exp(thismaterial);
    if ( material[13]>100.0 ) material[13]= 100.0;
    
    lmdef(DEFMATERIAL, *id_ptr, 15, material);
  }
  else *id_ptr= 0;
  
  return((char *)id_ptr);
}

void ren_free_material(thismat)
Material thismat;
/* This routine frees memory allocated for the material. */
{
  ger_debug("ren_free_material:");
  free( (char *)material_ren_data(thismat) );
}

void ren_reset( hard )
int hard;
/* This routine resets the renderer */
{
  ger_debug("ren_reset: resetting renderer; hard= %d", hard);
  
  /* Hard resets require recreation of lisp-side variables */
  if (hard) {
    color_symbol= create_symbol("color");
    backcull_symbol= create_symbol("backcull");
    text_height_symbol= create_symbol("text-height");
    lighting_model_symbol= create_symbol("lighting-model");
    material_symbol= create_symbol("material");
  }
}

void ren_shutdown()
/* This routine shuts down the renderer */
{
  /* At the moment, this routine does nothing. */
  ger_debug("ren_shutdown: doing nothing");
}
