/****************************************************************************
 * sgigl_mi.c
 * Author Joel Welling
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This module joins with the mouse user interface module to provide a point-
and-click interface for workstations under Silicon Graphics GL.
*/

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <gl.h>
#include <device.h>
#include "ge_error.h"
#include "mi.h"
#include "sgigl_utils/knobs.h"

/* Size of graphics area in pixels, and control panel height */
#define VIEWPORT_SIZE 512
#define PANEL_HEIGHT 89

/* File name dialog panel information */
#define DLG_X_MIN 30
#define DLG_Y_MIN ((3*VIEWPORT_SIZE)/8 + PANEL_HEIGHT)
#define DLG_X_SIZE VIEWPORT_SIZE-60
#define DLG_Y_SIZE VIEWPORT_SIZE/4

/* Help information */
#define HELP_START_X 30
#define HELP_START_Y (460 + PANEL_HEIGHT)
#define HELP_STEP_Y 20
static char help_text[]= "\
To rotate the model, drag across the model with the\n\
left mouse button.  It's like rolling a ball in\n\
which the model is embedded.\n\
\n\
To slide the model, drag across the model with the\n\
middle mouse button.\n\
\n\
To move toward or away from the model, click in the\n\
slider in the control panel.\n\
\n\
The 'load' button loads a new P3D file.\n\
\n\
The 'next' button moves to the next model in the\n\
current P3D file.\n\
\n\
The 'help' button gets this message.\n\
\n\
The 'quit' button exits the program.\n\
\n\
To exit help, click 'OK'.";

/* Name of P3D file to be loaded */
#define MAX_NAME_LENGTH 128
static char default_filename[]= "dummy_file_name";
static char *filename= default_filename;

/* Controls */
static ControlType file_dialog= { DIALOG, 0,
				    "Enter the name of the file to load:",
				    DLG_X_MIN, DLG_Y_MIN, 
				    DLG_X_SIZE, DLG_Y_SIZE,
				    0, NULL, NULL, 0, 0, 0, 0 };
static ControlType help_ok_button= { BUTTON, 0, "OK", 10, PANEL_HEIGHT+10,
                            0, 0, 0, NULL, NULL, 0, 0, 0};
static ControlType load_button={ BUTTON, 0, "load", 10, 47, 0, 0,
                            0, NULL, NULL, 0, 0, 0, 0};
static ControlType next_button={ BUTTON, 0, "next", 10, 10, 0, 0,
                            0, NULL, NULL, 0, 0, 0, 0};
static ControlType help_button={ BUTTON, 0, "help", 95, 47, 0, 0,
                            0, NULL, NULL, 0, 0, 0, 0};
static ControlType quit_button={ BUTTON, 0, "quit", 95, 10, 0, 0,
                            0, NULL, NULL, 0, 0, 0, 0};
static ControlType inout_slider={ SLIDER, 0, "in - Viewing Distance - out", 
                            190, 10, 0, 0,
                            0, NULL, NULL, 0, 0, 0, 0, NULL, NULL, 1.0, 0.0,
                            0.0, 0.0, 1.0, 1.0, 0.0, 1};

/* display mode */
typedef enum mode_enum {MODEL, HELP, FILE_DIALOG} Mode;
static Mode current_mode= MODEL;

/* Current projection transformation (in GL format) */
static Matrix currenttrans;

static void draw_text(start_x, start_y, step_y, textstr)
int start_x, start_y, step_y;
char *textstr;
/* This routine draws text, with returns at line ends. */
{
  int current_x, current_y, char_width;
  char string[2];

  ger_debug("draw_text");

  current_x= start_x;
  current_y= start_y;
  char_width= strwidth("M");
  string[1]= '\0';

  while (*textstr) {
    if ( *textstr=='\n' ) {
      current_x= start_x;
      current_y= current_y - step_y;
    }
    else {
      cmov2( (Coord)current_x, (Coord)current_y );
      string[0]= *textstr;
      charstr( string );
      current_x += char_width;
    }
    textstr++;
  }
}

static void control_drawing_mode()
/* This routine sets things up for drawing the panel */
{
  mmode(MPROJECTION);
  getmatrix(currenttrans);
  mmode(MSINGLE);
  viewport( 0, VIEWPORT_SIZE-1, 0, VIEWPORT_SIZE+PANEL_HEIGHT-1 );
  ortho2(0.0, (float)(VIEWPORT_SIZE-1),
         0.0, (float)(VIEWPORT_SIZE+PANEL_HEIGHT-1));
  frontbuffer(1);
  lmcolor(LMC_COLOR);
  zbuffer (FALSE);
}

static void model_drawing_mode()
/* This resets things for drawing the model (undoes control_drawing_mode) */
{
  zbuffer (TRUE);
  frontbuffer(0);
  viewport( 0, VIEWPORT_SIZE-1, PANEL_HEIGHT, VIEWPORT_SIZE+PANEL_HEIGHT-1 );
  mmode(MPROJECTION);
  loadmatrix(currenttrans);
  mmode(MVIEWING);
}

static void init_help()
/* 
This routine does whatever is necessary (maybe nothing) to initialize
the help display.
*/
{
  ger_debug("init_help");
}

static void init_panel()
/* This routine sets up the control panel */
{
  ger_debug("init_panel");

  control_drawing_mode();

  KnobInit(0);
  KnobPanel(0,0,VIEWPORT_SIZE,PANEL_HEIGHT);
  button_draw(&load_button,1);
  button_draw(&next_button,1);
  button_draw(&help_button,1);
  button_draw(&quit_button,1);
  slider_draw(&inout_slider,1);

  model_drawing_mode();
}

static void redraw_file_dialog()
/* This routine redraws the file name dialog display */
{
  ger_debug("redraw_file_dialog");

  control_drawing_mode();
  dialog_draw(&file_dialog);
  model_drawing_mode();
}

static void redraw_help()
/* This routine redraws the help display */
{
  ger_debug("redraw_help");

  control_drawing_mode();

  KnobPanel(0,PANEL_HEIGHT,VIEWPORT_SIZE,VIEWPORT_SIZE);
  draw_text(HELP_START_X,HELP_START_Y,HELP_STEP_Y,help_text);
  button_draw(&help_ok_button,1);

  model_drawing_mode();
}

static void redraw_panel()
/* This routine redraws the panel */
{
  ger_debug("redraw_panel");

  control_drawing_mode();

  KnobPanel(0,0,VIEWPORT_SIZE,PANEL_HEIGHT);
  button_draw(&load_button,1);
  button_draw(&next_button,1);
  button_draw(&help_button,1);
  button_draw(&quit_button,1);
  inout_slider.new= inout_slider.old= miu_viewing_distance();
  inout_slider.scale= inout_slider.max= 2.0*inout_slider.old;
  slider_draw(&inout_slider,1);

  model_drawing_mode();
}

static void handle_file_dialog(x,y, pblock,dev,val)
long x, y;
int *pblock;
Device dev;
short val;
/* This routine handles file name dialog interaction */
{
  ger_debug("handle_file_dialog");

  control_drawing_mode();

  mouse_x= (int)x; /* global variables, in knobs.h */
  mouse_y= (int)y;
  mouse_state= getbutton(LEFTMOUSE);
  mouse_transition= 1;

  if ( dialog_service(&file_dialog,dev,val) ) { /* Done with dialog */
    current_mode= MODEL;
    miu_redraw(); /* happens after return from event handler */
    *pblock= 1;

    if (file_dialog.flag) { /* load option accepted */
      filename= file_dialog.new_string;
      miu_load(filename); /* delayed effect */
    }
  }

  model_drawing_mode();
}

static void handle_help(x,y, pblock)
long x, y;
int *pblock;
/* This routine handles help display interaction */
{
  ger_debug("handle_help");

  control_drawing_mode();

  mouse_x= (int)x; /* global variables, in knobs.h */
  mouse_y= (int)y;
  mouse_state= getbutton(LEFTMOUSE);
  mouse_transition= 1;

  if ( (x>=0) && (y>=PANEL_HEIGHT) && (x<VIEWPORT_SIZE)
      && (y<PANEL_HEIGHT+VIEWPORT_SIZE) ) { /* click in window */
    if ( button_service(&help_ok_button) ) {
      current_mode= MODEL;
      miu_redraw(); /* happens after return from event handler */
      *pblock= 1;
    }
  } else if (mouse_state) ringbell();

  model_drawing_mode();
}

static void handle_panel(x, y, predraw, pblock)
long x, y;
int *predraw, *pblock;
/* This routine handles panel interaction */
{
  float scale;

  ger_debug("handle_panel");

  control_drawing_mode();

  mouse_x= (int)x; /* global variables, in knobs.h */
  mouse_y= (int)y;
  mouse_state= getbutton(LEFTMOUSE);
  mouse_transition= 1;
  if (current_mode==MODEL) { /* panel clicks in other modes are invalid */
    if ( button_service(&load_button) ) {
     current_mode= FILE_DIALOG;
     redraw_file_dialog();
     *pblock= 1;
    }
    else if ( button_service(&next_button) ) 
      miu_done();  /* delayed effect */
    else if ( button_service(&help_button) ) {
      current_mode= HELP;
      redraw_help();
      *pblock= 1;
    }
    else if ( button_service(&quit_button) ) miu_exit(); /* delayed effect */
    else if ( slider_service(&inout_slider,1) ) { 
      scale= inout_slider.new/inout_slider.old;
      miu_approach_model( scale );
      inout_slider.old= miu_viewing_distance();
      inout_slider.scale= inout_slider.max= 2.0*inout_slider.old;
      *predraw= 1;
      model_drawing_mode();
      miu_prompt_redraw();
    }
  } else if (mouse_state) ringbell();

  model_drawing_mode();
}

void mil_update()
/* This routine updates the controls */
{
  ger_debug("mil_update");

  redraw_panel(); /* since slider depends on viewing distance */
  if (current_mode==HELP) redraw_help();
  else if (current_mode==FILE_DIALOG) redraw_file_dialog();
}

static void handle_mouse_motion(dev, val, predraw)
Device dev;
short val;
int *predraw;
/* This routine interprets mouse motion and click events */
{
  static MousePosition mousedown, mouseup;
  static int block_next_event= 0, mouse_ready= 0;
  long x, y, window_x, window_y;

  ger_debug("handle_mouse_motion:");

  getorigin( &window_x, &window_y );
  x= getvaluator(MOUSEX) - window_x;
  y= getvaluator(MOUSEY) - window_y;

  if (block_next_event) {
    block_next_event= 0;
    return;
  }

  if (current_mode==HELP) 
    handle_help(x, y, &block_next_event);
  else if (current_mode==FILE_DIALOG)
    handle_file_dialog(x, y, &block_next_event, dev, val);
  else if ( y >= PANEL_HEIGHT ) {  /* event in display window */
    if (val) { /* mouse down case */
      mousedown.x= (int)x;
      mousedown.y= (int)y - PANEL_HEIGHT;
      mousedown.maxx= VIEWPORT_SIZE-1;
      mousedown.maxy= VIEWPORT_SIZE-1;
      mouse_ready= 1;
    }
    else { /* mouse up case */
      mouseup.x= (int)x;
      mouseup.y= (int)y - PANEL_HEIGHT;
      mouseup.maxx= VIEWPORT_SIZE-1;
      mouseup.maxy= VIEWPORT_SIZE-1;
      if ( mouse_ready && dev==LEFTMOUSE ) {
        miu_rotate_model( mousedown, mouseup );
      }
      else if ( mouse_ready && dev==MIDDLEMOUSE ) {
        miu_translate_model( mousedown, mouseup );
        miu_redraw();
        *predraw= 1;
      }
      mouse_ready= 0;
    }
  }
  else handle_panel(x,y,predraw,&block_next_event);  /* event in panel */
}

static void handle_text_input(dev, val, predraw)
Device dev;
short val;
int *predraw;
/* This routine interprets mouse motion and click events */
{
  static MousePosition mousedown, mouseup;
  static int block_next_event= 0, mouse_ready= 0;
  long x, y, window_x, window_y;

  ger_debug("handle_text_input:");

  getorigin( &window_x, &window_y );
  x= getvaluator(MOUSEX) - window_x;
  y= getvaluator(MOUSEY) - window_y;

  if (block_next_event) {
    block_next_event= 0;
    return;
  }

  if (current_mode==FILE_DIALOG)
    handle_file_dialog(x, y, &block_next_event, dev, val);
}

void mil_event()
/* This routine carries out one pass through the event loop */
{
  int redraw= 0;
  Device dev;
  short val;

  /* No debugging;  called too often */

  dev= qread(&val);
  switch (dev) {
  case INPUTCHANGE: /* window focus */
    /* do nothing */
    break;
  case WINSHUT: /* exit program */
    exit(0);
    break;
  case REDRAW:
    miu_prompt_redraw();
    redraw= 1;
    break;
  case LEFTMOUSE:
  case MIDDLEMOUSE:
    handle_mouse_motion(dev,val,&redraw);
    break;
  case KEYBD:
    handle_text_input(dev,val,&redraw);
    break;
  }

  if (redraw) {
    redraw_panel();
    if (current_mode==HELP) redraw_help();
    else if (current_mode==FILE_DIALOG) redraw_file_dialog();
  }
}

void mil_setup( id_string, argc, argv )
char *id_string;
int argc;
char *argv[];
/* 
This routine initializes this user interface module.  It is guaranteed
to be called only once. 
*/
{
  ger_debug("mil_setup");

  /* Initialize GL */
  prefsize(VIEWPORT_SIZE,VIEWPORT_SIZE+PANEL_HEIGHT);
  winopen(id_string);
  viewport( 0, VIEWPORT_SIZE-1, PANEL_HEIGHT, VIEWPORT_SIZE+PANEL_HEIGHT-1 );
  RGBmode();
  
  lsetdepth(0,0x7fffff);
  zbuffer(TRUE);
  doublebuffer();
  gconfig();
  
  /* Set up event handling */
  /* INPUTCHANGE and REDRAW automatically queued */
  qdevice(WINSHUT);
  qdevice(LEFTMOUSE);
  qdevice(MIDDLEMOUSE);
  qdevice(KEYBD);
  
  /* Enqueue a redraw event */
  qenter(REDRAW,1);

  /* Set up the control panel and tools */
  init_panel();
  init_help();
  dialog_init(&file_dialog);

  /* Initialize transformation matrix */
  getmatrix(currenttrans); 

  /* Initialize dialog box default */
  file_dialog.old_string= default_filename;
}

void mil_shutdown()
/* This routine shuts down this user interface module */
{
  ger_debug("mil_shutdown");
}
