/****************************************************************************
 * painter_ren.c
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 * Author Chris Nuuja
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
#include <stdio.h>
#include <math.h>
#include "ge_error.h"
#include "alisp.h"
#include "p3d.h"
#include "ren.h"
#include "painter_vars.h"
#include "painter.h"
#include "matrix_ops.h"
#include "assist.h"

/* 
The following is designed to figure out what machine this is running
on, and include appropriate definition files.
*/
#ifdef unix
#define USE_UNIX
#endif
#ifdef CRAY
#undef USE_UNIX
#endif
#ifdef ardent
#undef USE_UNIX
#endif
#ifdef CRAY
#include "unicos_defs.h"
#endif
#ifdef ardent
#include "unicos_defs.h"
#endif
#ifdef USE_UNIX
#include "unix_defs.h"
#endif

/*			Global Variable Declarations   			*/


int Black,White,Background;    /* the indices of the color records 
				  for black, white and background */

/*  
   The following global variables hold the values  of the transformation
   matrix EyeToImage, the eye coordinates to 2d (virtual) screen coordinate
   transform.  The value at EI[3][3] is held in (3+3*4 =) EI15
*/
float EI0,   EI1,   EI2,   EI3,   EI4,   EI5,  EI6,  EI7,  EI8,  EI9,  EI10,  
      EI11,  EI12,  EI13,  EI14,  EI15;


static int REN_TEXT_FLAG=0;
float *RecentTrans;
int RecentObject;
Attribute_list RecentAttr;


/*  Global values relating to the part of the object currently being rendered */
float TextHeight;             /* Height, in pixels, of text 		      */


/*  Primitive table and fill-pointer */
ren_objecttype *PrimTable;      /* the table itself      */
ren_objecttype *NewPrimObj;   /* the fill pointer      */
int PrimTableSize=0;            /* the size of the table */


/*  
   Buffer of depth records, and fill index.  Each record holds a polyrecord
   index and that polyrecord's Z depth.  This buffer is sorted by Z depth,
   and then the polyrecords drawn in (sorted) order.  Thats why its called
   painter
*/
depthtable_rec *DepthBuffer;      /* the table itself	*/
int DepthCount=0;		  /* Fill index */

/* 
   Buffer of object records, the current size of the buffer, and the fill 
   index. 
*/
ren_objecttype *ObjectBuffer;
int MaxObjects=0,ObjectCount=0;

/*  
   Buffer of polyrecords, current size of buffer, and fill index.
   These records hold the 3d world space versions of the primitives
   One records per primitive (multiple instances refer to the sxfgoame record)
*/
ren_polytype *PolyBuffer;
int MaxPolyCount=0;	       
int PolyCount=0;	      

/*  
   Buffer of disposable polyrecords, current size of buffer, and fill index.
   These records hold the 2d transformed and clipped versions of all 
   primitive instances.  This buffer holds the primitives that are sorted by
   zdepth.  Its indices are stored in DepthBuffer
*/
ren_polytype *DPolyBuffer;
int DepthPolyCount=0;
int MaxDepthPoly=0;

/*
   Buffer of color records, current size of buffer, and fill index.
   One record per primitive (only filled if its a polygon with color
   specified per vertex, in which case the average of all vertex colors
   is saved as that polygon's color).
*/
extern ren_colortype  *ColorBuffer;
int ColorCount=0;
int MaxColorCount=0;

/*
   Buffer of disposable color records, current size of buffer, and fill index.
   One record per primitive instance .  Each poly record has a color at this
   point, either its own or an inherited color.
*/
extern ren_colortype  *DColorBuffer;
int DColorCount=0;
int MaxDColorCount=0;

/*
   Buffer of light sources, current size of buffer, and fill index.
*/
ren_pointtype  *LightPosBuffer,*OldLightBuffer;
ren_colortype *LightColorBuffer;
int LightCount=0;
int MaxLightCount=0;
int AmbientColor;

/*  Symbols defined by ren_setup and used in parsing attribute lists */
Symbol color_symbol, backcull_symbol, text_height_symbol, material_symbol;

/*			Function Declarations   		*/

/*
   Global Data Used: All Global Buffers
   Expl:  Initializes all global buffers
*/
void setup_Buffers()
{   

	ObjectCount =0;
	ObjectBuffer = (ren_objecttype *)
		      malloc( MaxObjects*sizeof(ren_objecttype));
	if (!ObjectBuffer)
		{
		fprintf(stderr,"ERROR: COULD NOT ALLOCATE %d OBJECT RECORDS\n",
			MaxObjects);
		exit(1);
		}
	DepthPolyCount =0;
	DPolyBuffer = (ren_polytype *)
		      malloc( MaxDepthPoly*sizeof(ren_polytype));
	if (!DPolyBuffer)
		{
		fprintf(stderr,"ERROR: COULD NOT ALLOCATE %d POLYGONS\n",
			MaxDepthPoly);
		exit(1);
		}
	PolyCount = 0;
	PolyBuffer = (ren_polytype *)
		      malloc( MaxPolyCount*sizeof(ren_polytype));
	if (!PolyBuffer)
		{
		fprintf(stderr,"ERROR: COULD NOT ALLOCATE %d POLYGONS\n",
			MaxPolyCount);
		exit(1);
		}
	DepthCount = 0;
	DepthBuffer = (depthtable_rec *)
		       malloc( MaxPolyCount*sizeof(depthtable_rec) );
	if (!DepthBuffer)
		{
		fprintf(stderr,"ERROR: COULD NOT ALLOCATE %d POLYGON RECORDS\n",
			MaxPolyCount);
		exit(1);
		}
	LightCount = 0;
	LightPosBuffer = (ren_pointtype *)
	                malloc( MaxLightCount*sizeof(ren_pointtype));
	OldLightBuffer = (ren_pointtype *)
	                malloc( MaxLightCount*sizeof(ren_pointtype));
	if (!LightPosBuffer)
		{
		fprintf(stderr,"ERROR: COULD NOT ALLOCATE %d LIGHT POSITION RECORDS\n",
			MaxLightCount);
		exit(1);
		}
	LightColorBuffer = (ren_colortype *)
	                malloc( MaxLightCount*sizeof(ren_colortype));
	if (!LightColorBuffer)
		{
		fprintf(stderr,"ERROR: COULD NOT ALLOCATE %d LIGHT COLOR RECORDS\n",
			MaxLightCount);
		exit(1);
		}
}

/*
   Global Data Used: PolyCount, DepthBuffer
   Expl:  Draws the polyrecords in the now sorted DepthBuffer from back to 
	  front
*/
void draw_DepthBuffer()
{
	register int i;

	if (DepthCount >= MaxDepthPoly)
		{
		fprintf(stderr,"ERROR, was about to overflow!\n");
		exit(0);
		}
	for (i=0;i<DepthCount;i++)
		render_polyrec(i);
	DepthCount=0;			/* reset DepthBuffer */
}

/*
   Global Data Used: NONE
   Expl: fills the slot for material definition routine
*/
char *ren_def_material(material)
Material material;
/* This routine fills the slot for a material definition routine */
{
  ger_debug("ren_def_material: doing nothing");
  return((char *)0);
}

/*
   Global Data Used: NONE
   Expl: fills the slot for material deletion routine
*/
void ren_free_material(thismat)
Material thismat;
/* This routine fills the slot for a material deletion routine */
{
  ger_debug("ren_free_material: doing nothing");
}

/*
   Global Data Used: NONE
   Expl:  Not implemented yet
*/
void ren_dump(agob)
Gob agob;
{
}

/*
   Global Data Used: NONE
   Expl:  returns an index into the ColorBuffer containing <color>'s
     	  information
*/
int convert_color( color )
Color color;
/* This routine converts a (lisp) color into an internal c-vector-color */
{
	int newcolor;

	ger_debug("convert_color");
	newcolor = new_ColorIndex();
	ColorBuffer[newcolor].r = color_red( color );
	ColorBuffer[newcolor].g = color_green( color );
	ColorBuffer[newcolor].b = color_blue( color );
	ColorBuffer[newcolor].a = color_alpha( color );
	return(newcolor);
}

void ren_sphere()
{
	ger_debug("ren_sphere");
	ast_sphere( ren_mesh );
}

void ren_cylinder()
{
	ger_debug("ren_cylinder");
	ast_cylinder( ren_mesh );
}

void ren_torus(bigradius, smallradius)
float bigradius, smallradius;
{
	ger_debug("ren_torus");
	ast_torus( bigradius, smallradius, ren_mesh );
}

/*  
    This routine is called by ast_text to effectuate a change in the
    current transformation matrix.  The function name gets passed to
    ast_text, and ast_text calls it.  RecentTrans is set to the current
    transformation matrix inside of internal_render.
*/
void text_trans(trans)
Transformation trans;
{
	float *atrans;

	atrans = array2d_to_c( trans );
	transpose(atrans); 
	RecentTrans = mult3dMatrices( atrans, RecentTrans );
	free(atrans);
}

/* 
   This routine is called by ast_text to draw a line that makes up part
   of a character in a text-string.  This routine's name is passed to
   ast_text, and it calls this routine for every stroke in every character.
   This routine is different from ren_polyline in that it doesn't store 
   the polyline for later drawing.  It renders it now, then cleans up the
   space.  This is necessary, since the lines that make up a text string 
   can change from (gob) traversal to traversal.
*/
void text_polyline(vlist,count)
Vertex_list vlist;
int count;
{
   int polyline;
   void ren_primitive();

   if (count<2) 
      {
      fprintf(stderr,"***Error***: not enough vertices for polyline %d",count); 
      return;
      }
   polyline = get_polyrec(vlist,count,POLYLINE);
   NewPrimObj->poly_index = polyline;
   NewPrimObj->num_polygons = 1;
   ren_primitive( (NewPrimObj-ObjectBuffer) , RecentTrans);
   CoordBuffIndex -= 3*count;
   PolyCount--;
}

/* 
   This routine writes a string of text as polyline segments. ast_text
   will call "text_trans" and "text_polyline" as needed to draw the text.
   We set the REN_TEXT_FLAG flag here so that the next call to ren_primitive
   will not attempt to draw this primitive (it was already drawn by
   ast_text.  "Drawn" in the sense that it has been added to the DepthBuffer,
   for later sorting and displaying.  The flag also signals add_primitive 
   to free the object record for this primitive.
*/
void ren_text(txtpoint,uvec,vvec,txtstring)
Point txtpoint;
Vector uvec, vvec;
char *txtstring;
{
	ger_debug("ren_text");
	ast_text(txtpoint,uvec,vvec,txtstring,text_trans,text_polyline);
	REN_TEXT_FLAG=1;
}

/*
   This routine fills the polyrecord at index <poly> with the vertex and
   color information held in <vlist>.  <count> is the number of vertices
   read from vlist.  Since painter can not interpolate color across 
   vertices, the sum of the colors at each vertex is stored as the color for the
   whole polyrecord at index <poly>
*/
void get_clrpoints(vlist,poly,count)
Vertex_list vlist;
int poly;
int count;
{
	Color color;
	float r=0.0,g=0.0,b=0.0;
	int x_index,y_index,z_index,newcolor,i;

	if (count == 0)
		{
		fprintf(stderr,"ERROR: BAD CLR POINTS\n");
		return;
		}
	x_index = poly_Xindex(poly);
	y_index = poly_Yindex(poly);
	z_index = poly_Zindex(poly);
	for (i=0;i<count;i++)
		{
		CoordBuffer[x_index+i] = vertex_x(first_vertex(vlist));
		CoordBuffer[y_index+i] = vertex_y(first_vertex(vlist));
		CoordBuffer[z_index+i] = vertex_z(first_vertex(vlist));
		if (!null(color = vertex_color(first_vertex(vlist))))
			{
			r += (float) color_red( color );
			g += (float) color_green( color );
			b += (float) color_blue( color );
			};
		vlist= rest_of_vertices( vlist );
		};
/*  
    Note that we assume each vertex has color information, even though we don't require it
    (i.e. it won't cause an error, just a different shade than you might expect)
*/
	r /= (float) count;  
	g /= (float) count;
	b /= (float) count;
	newcolor = new_ColorIndex();
	ColorBuffer[newcolor].r = r;
	ColorBuffer[newcolor].g = g;
	ColorBuffer[newcolor].b = b;
	ColorBuffer[newcolor].a = 1.0;	/*  Currently, <a> is ignored  */
	poly_ColorIndex(poly) = newcolor;

}

/* 
   Same as above, but does not check for color information
   No routine checks for normal information, since painter can't interpolate
   color across polygon points
*/
void get_normpoints(vlist,poly,count)
Vertex_list vlist;
int poly;
int count;
{
	register int i;
	int x_index,y_index,z_index;

	x_index = poly_Xindex(poly);
	y_index = poly_Yindex(poly);
	z_index = poly_Zindex(poly);
	for (i=0;i<count;i++)
		{
		CoordBuffer[x_index+i] = vertex_x(first_vertex(vlist));
		CoordBuffer[y_index+i] = vertex_y(first_vertex(vlist));
		CoordBuffer[z_index+i] = vertex_z(first_vertex(vlist));
		vlist= rest_of_vertices( vlist );
		};
	poly_ColorIndex(poly) = 0;
}

/*
    returns an index into the polyrecord buffer  for the polyrecord that
    is filled with the coordinate values found in <vlist>.  It will ignore 
    normal information.
*/
int get_polyrec( vlist, count,type )
Vertex_list vlist;
int count;
primtype type;
{
	int newpoly;

	newpoly = new_PolyIndex();
	fillpoly_rec(newpoly,count,type);
	if (!null(vertex_color(first_vertex(vlist))))
		get_clrpoints(vlist,newpoly,count);
	else
		get_normpoints(vlist,newpoly,count);
	return(newpoly);
}

/*
   Global Data Used:NewPrimObj
   Expl:   This routine creates a polyline primitive.
	   calls get_polyrec to get a ren_polyrec structure with <vlist>'s
	   vertices, and then adds it to the ObjectBuffer at NewPrimObj;
*/
void ren_polyline( vlist, count )
Vertex_list vlist;
int count;
{
	int polyline;

	ger_debug("ren_polyline");
	if (count<2) 
	   {
	   fprintf(stderr,
	      "***Error***: not enough vertices for polyline %d",count); 
	   return;
	   }
	polyline = get_polyrec(vlist,count,POLYLINE);
	NewPrimObj->poly_index = polyline;
	NewPrimObj->num_polygons = 1;
}

/*
   Global Data Used:NewPrimObj
   Expl:   This routine creates a polymarker primitive.
	   calls get_polyrec to get a ren_polyrec structure with <vlist>'s
	   vertices, and then adds it to the ObjectBuffer at NewPrimObj;
*/
void ren_polymarker( vlist, count )
Vertex_list vlist;
int count;
{
	int polymarkers,newmarker,newcolor, i;
	Vertex vertex;
	Color color;

	if (count<1) {
		fprintf(stderr,"***Error***: not enough vertices"); 
		return;
		}
	polymarkers = new_PolyIndicies(count);
	newmarker = polymarkers;
	for (i=0;i<count;i++)
		{
		vertex = first_vertex( vlist );
		vlist = rest_of_vertices( vlist );
		fillpoly_rec(newmarker,1,POLYMARKER);
		CoordBuffer[poly_Xindex(newmarker)] = vertex_x(vertex);
		CoordBuffer[poly_Yindex(newmarker)] = vertex_y(vertex);
		CoordBuffer[poly_Zindex(newmarker)] = vertex_z(vertex);
		if (!null(color = vertex_color(vertex)))
		   {
	           newcolor = new_ColorIndex();
		   ColorBuffer[newcolor].r = (float) color_red( color );
		   ColorBuffer[newcolor].g = (float) color_green( color );
		   ColorBuffer[newcolor].b = (float) color_blue( color );
	           ColorBuffer[newcolor].a = 1.0;	
		   poly_ColorIndex(newmarker) = newcolor;
		   }
	        else
		   poly_ColorIndex(newmarker) = 0;
	        newmarker++;
		}
	NewPrimObj->poly_index = polymarkers;
	NewPrimObj->num_polygons = count;
}

/*
   Global Data Used:NewPrimObj
   Expl:   This routine creates a polygon primitive.
	   calls get_polyrec to get a ren_polyrec structure with <vlist>'s
	   vertices, and then adds it to the PrimTable at PrimFillPtr;
*/
void ren_polygon( vlist, count )
Vertex_list vlist;
int count;
{
	int polygon;

	ger_debug("ren_polygon, %d coords",count);
	if (count<3) {
		fprintf(stderr,"***Error***: not enough vertices"); 
		return;
		}
	polygon = get_polyrec(vlist,count,POLYGON);
	NewPrimObj->poly_index = polygon;
	NewPrimObj->num_polygons = 1;
}
 
/*
   Global Data Used:NewPrimObj
   Expl:   This routine creates a triangle strip primitive.  It creates many triangle shaped 
           polygon primitives.  It is necessary to reverse the order of the first and second
	   coordinate every other polygon so that all the triangle's normals will be computed
	   in the same direction.
*/
void ren_triangle( vlist, count )
Vertex_list vlist;
int count;
{
	int triangles,new_triangle;
	int i,even;
	float tempX,tempY,tempZ;

 	ger_debug("ren_triangle");
	if (count<3) {
		fprintf(stderr,"***Error***: not enough vertices"); 
		return;
		}
	count -= 2;
	triangles = new_PolyIndicies(count);
	new_triangle = triangles;
	even = 0;
	for (i=0;i<count;i++)
		{
		fillpoly_rec(new_triangle,3,POLYGON);
		if (!null(vertex_color(first_vertex(vlist))))
			get_clrpoints(vlist,new_triangle,3);
		else
			get_normpoints(vlist,new_triangle,3);

		if (even)
			{
			tempX = CoordBuffer[poly_Xindex(new_triangle)];
			tempY = CoordBuffer[poly_Yindex(new_triangle)];
			tempZ = CoordBuffer[poly_Zindex(new_triangle)];

			CoordBuffer[poly_Xindex(new_triangle)] = 
			CoordBuffer[poly_Xindex(new_triangle) + 1];

			CoordBuffer[poly_Yindex(new_triangle)] =
			CoordBuffer[poly_Yindex(new_triangle) + 1];

			CoordBuffer[poly_Zindex(new_triangle)] = 
			CoordBuffer[poly_Zindex(new_triangle) + 1];


			CoordBuffer[poly_Xindex(new_triangle) + 1] = tempX;
			CoordBuffer[poly_Yindex(new_triangle) + 1] = tempY;
			CoordBuffer[poly_Zindex(new_triangle) + 1] = tempZ;

			even = 0;
			}
		else
			{
			even = 1;
			}
		vlist = rest_of_vertices(vlist);
		new_triangle++;
		}
	NewPrimObj->poly_index = triangles;
	NewPrimObj->num_polygons = count;
	
}
/*  Creates a surface from a mesh of bezier curve definitions */
void ren_bezier( vlist, count )
Vertex_list vlist;
int count;
{
  ger_debug("ren_bezier");
  ast_bezier(vlist, ren_mesh);
}

void ren_mesh( vlist, vcount, flist, fcount)
Vertex_list vlist;
Facet_list flist;
int vcount, fcount;
{
	Vertex_list vertexlist;
	int newpoly, polylist;
	int numverts,i;

	polylist = new_PolyIndicies(fcount);
	newpoly = polylist;
	for (i=0;i<fcount;i++)
		{
		vertexlist = first_facet( flist );
		flist = rest_of_facets( flist );
		numverts = length_list( vertexlist );
		fillpoly_rec(newpoly,numverts,POLYGON);
		if (!null(vertex_color(first_vertex(vertexlist))))
			get_clrpoints(vertexlist,newpoly,numverts);
		else
			get_normpoints(vertexlist,newpoly,numverts);
		newpoly++;
		}
	NewPrimObj->poly_index = polylist;
	NewPrimObj->num_polygons = fcount;
}
 

/*
   Global Data Used:NONE
   Expl:  Called when defining a gob.  Does nothing.  Non primitive gobs are effectuated 
          during traversal at render time, and primitive gobs are added to the object
	  table (if they aren't already there), then rendered (also at traversal time)
*/
void ren_gob(current_gob, trans, attr, prim, children )
int current_gob;
Transformation trans;
Attribute_list attr;
Primitive prim;
Child_list children;
{
	ger_debug("ren_gob: Defining gob %d.", current_gob);
}

/*
   Global Data Used: LightPosBuffer, LightColorBuffer
   Expl:  Places a light source in the Light Buffer.
*/
void ren_light( location, lightcolor )
Point location;
Color lightcolor;
{
  int newlight;
  float px, py, pz;
  register float *rtrans= RecentTrans;

  ger_debug("ren_light");

  newlight = new_LightIndex();
  px= point_x(location);
  py= point_y(location);
  pz= point_z(location);
  LightPosBuffer[newlight].x= 
    rtrans[0]*px + rtrans[4]*py + rtrans[8]*pz + rtrans[12];
  LightPosBuffer[newlight].y= 
    rtrans[1]*px + rtrans[5]*py + rtrans[9]*pz + rtrans[13];
  LightPosBuffer[newlight].z= 
    rtrans[2]*px + rtrans[6]*py + rtrans[10]*pz + rtrans[14];
  LightColorBuffer[newlight].r = color_red(lightcolor);
  LightColorBuffer[newlight].g = color_green(lightcolor);
  LightColorBuffer[newlight].b = color_blue(lightcolor);
  LightColorBuffer[newlight].a = color_alpha(lightcolor);
}

/*
   Global Data Used: AmbientColor
   Expl:  Sets the ambient light color/intensity.
*/
void ren_ambient(lightcolor)
Color lightcolor;
{
	ger_debug("ren_ambient");

	if ( !null(lightcolor) )
		AmbientColor = convert_color( lightcolor );
}

/*
   Global Data Used: EyetoImage, ViewMatrix, Zmin, and Zmax
   Expl:  This routine defines the camera attributes.  It sets ViewMatrix
	  so that the camera is at <lookfrom> looking at <lookat> with
	  up-vector <lookup>.  It sets Zmin to <yon> and Zmax to <hither>.
	  It sets the variables representing the indices of the 3d-to-2d transformation
	  matrix EyetoImage (the variables EI0 - EI15) so that the camera has aperature 
	 <fov>, and the viewing fustrum is normalized for <hither> and <yon>
*/
void ren_camera( lookat, lookfrom, lookup, fov, hither, yon, background )
Point lookat, lookfrom;
Vector lookup;
float fov, hither, yon;
Color background;
{
	float *trans_camera, *Rotx, *Roty, *Rotz;
	float a, b, c, d, a_prime, b_prime, d_prime,length;
	ren_vectortype *up, *forward, *rotated_upvector;
	ren_pointtype *origin,*camera;
  	float deg_to_rad= 3.14159265/180;
	int i;

	ger_debug("ren_camera");

	if (yon == 0.0)
		yon = 0.00001;
	origin = makepoint_rec( point_x(lookat), point_y(lookat), 
				point_z(lookat) );
	camera = makepoint_rec( point_x(lookfrom), point_y(lookfrom), 
				point_z(lookfrom) );
	/* Background color record */
	Background = makecolor_rec(color_red(background),
				   color_green(background),
				   color_blue(background),
				   color_alpha(background));

	/* what if Camera_Up is not a unit vector ? */
	/* and what if Camera_Up is not perpendicular to Forward */

	up = makevector_rec( vector_x(lookup), vector_y(lookup), 
	        	     vector_z(lookup) );
	forward = make_directionvector(camera,origin);

	ViewMatrix = make3dScale(1.0,1.0,1.0);
	trans_camera = make3dTrans(
	     0.0 - camera->x, 0.0 - camera->y, 0.0 - camera->z);
	append3dMatrices(ViewMatrix,trans_camera);

	a = forward->x;
	b = forward->y;
	c = forward->z;
	d = (float) sqrt( (b*b + c*c) );

	Rotx = make3dScale(1.0,1.0,1.0);
	Roty = make3dScale(1.0,1.0,1.0);
	Rotz = make3dScale(1.0,1.0,1.0);

	/*  Need check for d=0 */
	if (d != 0.0)
		{
		Rotx[5] = c/d;
		Rotx[6] = b/d;
		Rotx[9] = 0.0-b/d;
		Rotx[10] = c/d;
		}
	else
		{
		Rotx[5] = 1.0;
		Rotx[6] = 0.0;
		Rotx[9] = 0.0;
		Rotx[10] = 1.0;
		}

	Roty[0] =  d;
	Roty[2] =  a;
	Roty[8] =  0.0 - a;
	Roty[10] = d;

	append3dMatrices(ViewMatrix,Rotx);
	append3dMatrices(ViewMatrix,Roty);

	rotated_upvector = vector_matrix_mult3d(up,ViewMatrix);
	a_prime = rotated_upvector->x;
	b_prime = rotated_upvector->y;
	d_prime = (float) sqrt(a_prime*a_prime + b_prime*b_prime);
	
	if (d_prime != 0.0)
		{
		Rotz[0] = b_prime/d_prime;
		Rotz[1] = a_prime/d_prime;
		Rotz[4] = (0.0 - a_prime)/d_prime;
		Rotz[5] = b_prime/d_prime;
		}
	else
		{
		Rotz[0] = 1.0;
		Rotz[1] = 0.0;
		Rotz[4] = 0.0;
		Rotz[5] = 1.0;
		}
	append3dMatrices(ViewMatrix,Rotz);
	
	fov = fabs(fov) * deg_to_rad;
	EI0  = 1.0 / tan((double) fov/2.0);   
	EI5  = 1.0 / tan((double) fov/2.0);   

	EI8  = -1.0;		/*  Offset origin to middle of screen */
	EI9  = -1.0;		/*  Screen dimensions are 2.0  by 2.0 */
	EI10 = 1.0 / (1.0 - hither/yon);
	EI11 = -1.0;
	EI14 = (0.0 + hither)/ (1.0 - hither/yon);
	Zmax = hither;
	Zmin = yon;

	free(rotated_upvector);
	free(up);
	free(forward);
}

/*
   Global Data Used: REN_TEXT_FLAG
   Expl: This routine will transform the input object's composite polyrecs from (3d) 
         world space to screen space, then add them to the Depth buffer for later sorting
	 and display.  It calls the painter function ren_primitive with the current 
	 attributes and transforms.  If REN_TEXT_FLAG is true, all the composite lines of the text
         have already been transformed and added to the Depth buffer.  It has already been
	 "rendered"
*/
void ren_primitive(object,transform)
int object;
float *transform;
{
	int color,back_cull;
	Material material;
	
	if (REN_TEXT_FLAG) 
	   {
	   REN_TEXT_FLAG=0;
	   return;
	   }
	color = convert_color(ast_color_attribute(color_symbol));
	back_cull = ast_bool_attribute(backcull_symbol);
	material = ast_material_attribute(material_symbol);
	ColorBuffer[color].r *= material_kd(material);
	ColorBuffer[color].g *= material_kd(material);
	ColorBuffer[color].b *= material_kd(material);
	/* 
	   There is no need to get TextHeight, since ast_text does that for
	   us 
	*/
	render_primitive(object,transform,back_cull,color);
        ColorCount--;  /*  Free color  */
}

/*
   Global Data Used: NewPrimObj, PrimTable
   Expl:  This routine adds a primitive to the Object buffer.  It sets NewPrimObj
	  to the newly allocated space, evals the lisp-function that will create the 
	  primitive (the lisp function should be a call-out which will call a function like
	  ren_sphere, ren_polygon, etc.)  This should only be called if that primitive
	  with <primid> has not yet been added to the Object buffer.  This routine only
	  adds the primitive to the database.  It does not render it (ren_primitive does 
          that).
*/
int add_primitive(primitive,primid)
Function primitive;
int primid;
{
	int newobj;

	newobj = new_ObjectIndex();
	hash_add(primid,newobj);
	NewPrimObj = ObjectBuffer+newobj;
	eval_function(primitive);
	if (REN_TEXT_FLAG)
		{
		hash_free(primid);
		ObjectCount--;
		}
	return(newobj);
}
/*
   Global Data Used:None
   Expl:  walks down <thisgob>, adding all the polyrecs of each primitive it
	  encounters along the walk to the Object buffer (if they aren't already
          there) and then renderers them (.i.e. transforms them from world space
	  to screen space, and stores them in Depth Buffer for later sorting).
	  It alters the current gob-related transformation matrix and attributes to
	  include all new contributions of the current gob, so they will affect any
	  primitive objects "rendered" at this level of recursion. 
	  Note that matricies are converted to their transposes here.
	  Note also that since we are travelling from the top of the gob to the
	  bottom, we pre-concatenate new gob transforms to the existing
	  transform.
	  This procedure could be altered to work with an explicit stack instead of
	  recursion, but you wouldn't gain more than a 1% speedup.
*/
void internal_render(thisgob,thistrans)
Gob thisgob;
float *thistrans;
{
   int thisid,object; 
   Attribute_list newattrlist;
   Primitive newprim;
   float *newtrans, *atrans;
   Child_list kidlist;

   thisid= gob_idnum( thisgob );
   ger_debug("render:rendering object given by gob %d",gob_idnum(thisgob));

   if ( !null( gob_trans(thisgob) ) )
	{
	atrans = transpose( array2d_to_c( gob_trans(thisgob) ) );
	newtrans = mult3dMatrices( atrans, thistrans );
	free(atrans);
	}
   else
	newtrans  = thistrans;
   if ( !null( newattrlist= gob_attr(thisgob) ) ) 
	ast_push_attributes(newattrlist);
   if ( !null( newprim = gob_primitive(thisgob) ) )
	{
	int object, primid;

	RecentTrans = newtrans;
	primid = primitive_id(newprim);
	if ((object=hash_lkup(primid)) == BAD_HASH_LOOKUP)
   		object = add_primitive(primitive_op(newprim),primid);
	ren_primitive( object, newtrans );
	}
   else {
	kidlist= gob_children( thisgob );
		while ( !null(kidlist) ) {
		internal_render( first_child( kidlist ), newtrans);
		kidlist= rest_of_children( kidlist );
		}
	};
   if (newtrans != thistrans)
	free(newtrans);
   if ( !null( newattrlist) ) 
	ast_pop_attributes(newattrlist);
}

/*
   Global Data Used:None
   Expl:  used by qsort, the standard quik-sort algorithim of (at least) this
	  c compiler.  It returns 1, -1 or 0 depending on whether a is greater
	  than, less than, or equal b
*/
int compare(a,b)
depthtable_rec *a,*b;
{
	if (a->key > b->key)
		return(1);
	if (a->key < b->key)
		return(-1);
	return(0);
}

/*
   Global Data Used: MaxPolyCount, DepthBuffer, Background
   Expl:  Draws the object <thisgob> with the current attributes and transforms.
          The traversal of the gob <thisgob> occurs in internal_render.  This
          routine sets up all the preliminaries, then calls internal_render to actually
	  transform <thisgob> to  screen space and place in the Depth Buffer.  Depth
	  Buffer is then sorted by depth, then displayed back to front.  This routine tells
	  drawcgm to close the previous picture (we don't want to erase it until a new one is
	  ready to be drawn), if there is one, and open a new picture.
*/
void ren_render(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
{
	int thisid,error=0,color_mode=1; 
	float *newtrans;
	
	thisid= gob_idnum( thisgob );
	ger_debug("render: rendering object given by gob %d", thisid);
	newtrans = array2d_to_c(thistrans);
	transpose(newtrans);

	fprintf(stderr,"Reading in object\n");
	if (!null(thisattrlist))        ast_push_attributes(thisattrlist);
	internal_render(thisgob,newtrans);
	if (!null(thisattrlist))	ast_pop_attributes(thisattrlist);
/*	quiksort(0,PolyCount-1,DepthBuffer);  */
	qsort((char *)DepthBuffer,(unsigned) DepthCount
	      ,sizeof(depthtable_rec),compare);
	fprintf(stderr,"Done reading in object\n\n");
	
	wrbegp( &error );		/* begin picture  */
	wrtcsm( &color_mode,&error );	/* use direct color */
					/* background color */
	wrbgdc( &(ColorBuffer[Background].r), &(ColorBuffer[Background].g), 
		&(ColorBuffer[Background].b) );
	wrbgpb( &error );		/* begin picture body  */
	if (error) {
	   fprintf(stderr,"ERROR: Bad Picture Initialization\n");
	   exit(1);
	   }
	draw_DepthBuffer();
	wrendp(&error);
/*  Reset fill indices for disposable records  */
	DepthCount = 0;
	DepthPolyCount = 0;
	DColorCount = 0;
	DCoordIndex = 0;
	if (error)
		{
		fprintf(stderr,"ERROR: Bad Picture End\n");
		exit(1);
		}
}

/*
   Global Data Used: NONE
   Expl: This routine dumps a gob, by dumping any attributes or transforms it may
	 carry and by evaluating its primop (if present) or alternatively by
	 recursively dumping its children.
*/
void internal_traverselights(thisgob, thistrans)
Gob thisgob;
float *thistrans;
{
   int thisid; 
   Primitive newprim;
   float *newtrans, *atrans;
   Child_list kidlist;

   thisid= gob_idnum( thisgob );
   ger_debug("internal_traverselights: rendering light given by gob %d", thisid);

   if ( !null( gob_trans(thisgob) ) )
	{
	atrans = transpose( array2d_to_c( gob_trans(thisgob) ) );
	newtrans = mult3dMatrices( atrans, thistrans );
	free(atrans);
	}
   else
	newtrans  = thistrans;

   if ( !null( newprim = gob_primitive(thisgob) ) )
	{
	int object, primid;

	RecentTrans = newtrans;

	eval_function(primitive_op(newprim));
	}
   else {
	kidlist= gob_children( thisgob );
		while ( !null(kidlist) ) {
		internal_traverselights( first_child( kidlist ), newtrans);
		kidlist= rest_of_children( kidlist );
		}
	};
   if (newtrans != thistrans)
	free(newtrans);
}

void ren_traverselights(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
{
	int thisid;
	float *newtrans;

	thisid= gob_idnum( thisgob );
	ger_debug("ren_traverselights: rendering light given by gob %d", thisid);

	LightCount   = 0;
	AmbientColor = Black;
	newtrans = array2d_to_c(thistrans);
	transpose(newtrans);
	internal_traverselights(thisgob,newtrans); 
}

/*
   Global Data Used: NONE
   Expl:  This routine frees memory associated with the given gob.  Note that any
	  future call referencing this gob or its children will be impossible 
	  unless it is somehow redefined first. 
*/
void ren_free(thisgob)
Gob thisgob;
{
	int thisid;
	int x;
        Hash_cell *thiscell,*nextcell;
	
	thisid= gob_idnum( thisgob );
	ger_debug("ren_free_gob:  freeing gob %d",thisid);

	/* Reset all memory buffers to the start */
	PolyCount = 0;
	ColorCount = 2;         /* save colors for black and white */
	ObjectCount = 0;
	CoordBuffIndex = 0;
	for (x= 0; x < HASHSIZE; x++)
           for (thiscell= hashtable[x]; thiscell; thiscell= nextcell)
	      {
	      nextcell = thiscell->next;
	      free( (char *)thiscell);
	      }
	free( (char *)hashtable);	/* Erase all records of primitives */
	/* Make a new, clean hash table */
	hashtable = (Hash_cell **) malloc(HASHSIZE*sizeof(Hash_cell *));
	for (x=0;x<HASHSIZE;x++)
		hashtable[x] = (Hash_cell *) NULL;
}

/*
   Global Data Used: color_symbol, backcull_symbol, text_height_symbol
   Expl:  Resets the renderer.  If this is a hard reset, this requires
          recreation of lisp-side variables.
*/
void ren_reset( hard )
int hard;
{
  ger_debug("ren_reset: resetting renderer; hard= %d", hard);
  
  /* Hard resets require recreation of lisp-side variables */
  if (hard) {
    int x;

    backcull_symbol= create_symbol("backcull");
    text_height_symbol= create_symbol("text-height");
    color_symbol= create_symbol("color");
    material_symbol= create_symbol("material");
    PolyCount = 0;
    ColorCount = 2;         /* save colors for black and white */
    ObjectCount = 0;
    CoordBuffIndex = 0;
    LightCount = 0;
    AmbientColor = Black;
    free(hashtable);	/* Erase all records of primitives */
    /* Make a new, clean hash table */
    hashtable = (Hash_cell **) malloc(HASHSIZE*sizeof(Hash_cell *));
    for (x=0;x<HASHSIZE;x++)
      hashtable[x] = (Hash_cell *) NULL;
  }
}

/*
   Global Data Used: White, Black, 
        TextHeight, DeviceName, color_symbol, backcull_symbol,
		     text_height_symbol
   Expl: Initializes the renderer.  It calls init_renderer, which, among other
	 things, will initialize the gplot routines.  Note that DeviceName is 
	 set here from the input <device> string.
*/
void ren_setup( renderer, device, open_device, outfile, hints )
char *renderer,*device,*outfile;
int open_device;
Attribute_list hints;
{
	int length,x;
	static int initialized=0; /* to hold initialization state */
 
 	/* 
	Generate some symbols to be used later in attribute list
	parsing.
	*/
	if (!initialized) {
		hashtable = (Hash_cell **) malloc(HASHSIZE*sizeof(Hash_cell *));
		for (x=0;x<HASHSIZE;x++)
			hashtable[x] = (Hash_cell *) NULL;

		backcull_symbol= create_symbol("backcull");
		text_height_symbol= create_symbol("text-height");
		color_symbol= create_symbol("color");
		material_symbol= create_symbol("material");

 		ger_debug("ren_setup: initializing renderer");

		initialized= 1;
		length = 1 + strlen(device);
		DeviceName = malloc(length);
		strcpy(DeviceName,device);

		CoordBuffIndex=0;
		DCoordIndex=0;
		MaxCoordIndex=1000;
		MaxDCoordIndex = 1000;
		CoordBuffer  = (float *) malloc(MaxCoordIndex*sizeof(float));
		DCoordBuffer = (float *) malloc(MaxCoordIndex*sizeof(float));

		ColorCount = 0;
		DColorCount = 0;
		MaxColorCount = 1000;
		MaxDColorCount = 1000;
		ColorBuffer = (ren_colortype *)
			       malloc( MaxColorCount*sizeof(ren_colortype) );
		DColorBuffer = (ren_colortype *)
			       malloc( MaxColorCount*sizeof(ren_colortype) );
		White = makecolor_rec(1.0,1.0,1.0,1.0);
		Black = makecolor_rec(0.0,0.0,0.0,1.0);

		init_renderer(outfile);
		MaxPolyCount = 2500;
		MaxObjects = 2;
		MaxDepthPoly = MaxPolyCount;

		MaxLightCount = 10;
		AmbientColor = Black;  /*  Default ambient is nothing */

		setup_Buffers();
		PrimTable = (ren_objecttype *)0;
	}
	else ger_error("ren_setup: called twice, this call ignored");

}

void ren_shutdown()
/* This routine shuts down the renderer */
{
	int error=0;
	/* At the moment, this routine does nothing. */
	ger_debug("ren_shutdown: shutting down renderer");

	shutdown_renderer();
}










