/****************************************************************************
 *                                                                          *
 *	painter_clip.c                                                      *
 *      Copyright 1989, Pittsburgh Supercomputing Center                    *
 *                      All Rights Reserved                                 *
 *			Author Chris Nuuja                                  *
 *                                                                          *
 ****************************************************************************/

#include <stdio.h>
#include <math.h>
#include "ge_error.h"
#include "painter_vars.h"
#include "painter.h"


/*
   Global Data Used: NONE
   Expl:  Called by clip_Zline/poly to determine if a z-coordinate value is 
	  within a boundry, either HITHER  or YON;
*/
int insideZbound(z1,edgenum)
float z1;
int edgenum;
{
	switch(edgenum)
		{
		case HITHER_PLANE:
			if (z1>Zmin)		return(1);
			else 			return(0);
		case YON_PLANE:
			if (z1<Zmax) 		return(1);
			else 			return(0);
		default:
			fprintf(stderr,"zbound-val error \n");
			break;
		}
	return(0);
}

/*
   Global Data Used:  none
   Expl:  used by clip_Zline/poly to find the intersection of
	  (<x1>,<y1>,<z1>) and (<x2>,<y2>,<z2>) with the plane
	  <edgeNum> (valued HITHER_PLANE or YON_PLANE)
*/
void Zintersect(x1,y1,z1,x2,y2,z2,edgeNum,interX,interY,interZ)
float x1,y1,z1,x2,y2,z2,*interX,*interY,*interZ;
int edgeNum;
{
	float dx,dy,dz,m;
	
	dx =  x2-x1;
	dy =  y2-y1;
	dz =  z2-z1;
	    
	switch(edgeNum)
		{
		case YON_PLANE:
			if (dz == 0.0)
	    			m = 0.0;
			else
	    			m = (Zmax - z1)/dz;
			*interX = x1 + m*dx;
			*interY = y1 + m*dy;
			*interZ = Zmax;
			break;
		case HITHER_PLANE:
			if (dz == 0.0)
	    			m = 0.0;
			else
	    			m = (Zmin-z1)/dz;
			*interZ = Zmin;
			*interX = x1 + m*dx;
			*interY = y1 + m*dy;
			break;
		default:
			fprintf(stderr,"Unknown plane in Zintersect\n");
			break;
		}
}

/*
   Global Data Used: Xcoord_buffer, Ycoord_buffer, Zcoord_buffer,
		     Xclip_buffer, Yclip_buffer, Zclip_buffer
   Expl: Clips every consecutive pair of vertices in a polyrecord
	 against the plane <edgeNum> (either HITHER_PLANE or YON_PLANE).  
	 The polyrecord's vertices are located in Xcoord_buffer, Ycoord_buffer,
	 and Zcoord_buffer if <edgenum> is HITHER_PLANE. If <edgenum> is
	 YON_PLANE, then they are in Xclip_buffer, Yclip_buffer, and 
	 Zclip_buffer.  The clipped points are stored in the coordinate buffers
	 that did not originally contain the points.  It is important, 
	 therefore, that the HITHER_PLANE is always clipped before the 
	 YON_PLANE is. The average z value for this polyrecord is 
	 computed in this procedure, and the pointer <zdepth> saves it.
*/
void clip_Zpolygon(edgeNum,numcoords,newnumcoords,zdepth)
int edgeNum,numcoords,*newnumcoords;
float *zdepth;
{
	int old_points=0,new_points=0;
	float newx,newy,newz;
	float *xbuff_old,*ybuff_old,*zbuff_old,
	      *xbuff_new,*ybuff_new,*zbuff_new;

	*zdepth = 0.0;	/* clear out any old value of zdepth  */
	ger_debug("Clip_Zpolygon with %d points",numcoords);
	if (numcoords < 2)  
		{	   
		*newnumcoords = 0;
		return;
		};
/* 
   Figure out which buffer holds the old values, and which gets the new values 
*/
	if (edgeNum == HITHER_PLANE)
		{
		xbuff_old = Xcoord_buffer;
		ybuff_old = Ycoord_buffer;
		zbuff_old = Zcoord_buffer;
		xbuff_new = Xclip_buffer;
		ybuff_new = Yclip_buffer;
		zbuff_new = Zclip_buffer;
		}
	else
		{
		xbuff_old = Xclip_buffer;
		ybuff_old = Yclip_buffer;
		zbuff_old = Zclip_buffer;
		xbuff_new = Xcoord_buffer;
		ybuff_new = Ycoord_buffer;
		zbuff_new = Zcoord_buffer;
		}
	/*  loop over all points in polyrecord */
	while (old_points < numcoords)
		{
		/* Points inside */
		while ( (old_points < numcoords) && 
			(insideZbound(zbuff_old[old_points],edgeNum)))
			{
			xbuff_new[new_points]=xbuff_old[old_points];
			ybuff_new[new_points]=ybuff_old[old_points];
			*zdepth += zbuff_old[old_points];
			zbuff_new[new_points++]=zbuff_old[old_points++];
			}

		if (old_points < numcoords)
			{
			/*	Intersection point   	*/
			if (old_points > 0)
				{
				Zintersect(xbuff_old[old_points-1],
				   	   ybuff_old[old_points-1], 
				   	   zbuff_old[old_points-1],
				   	   xbuff_old[old_points], 
				   	   ybuff_old[old_points], 
				   	   zbuff_old[old_points],edgeNum,
				   	   &newx,&newy,&newz);
				xbuff_new[new_points]=newx;
				ybuff_new[new_points]=newy;
				*zdepth += newz;
				zbuff_new[new_points++]=newz;
			        }

			/*	Points outside		*/
			while ( (old_points < numcoords) && 
			        (!(insideZbound(zbuff_old[old_points],
				   edgeNum))))
				{
				old_points++;
				}

			/*	Intersection point	*/
			if (old_points < numcoords) 
				{
				Zintersect(xbuff_old[old_points],
				   	   ybuff_old[old_points], 
				 	   zbuff_old[old_points], 
			   		   xbuff_old[old_points-1], 
				   	   ybuff_old[old_points-1], 
				  	   zbuff_old[old_points-1],edgeNum,
				   	   &newx,&newy,&newz);
				xbuff_new[new_points]=newx;
				ybuff_new[new_points]=newy;
				*zdepth += newz;
				zbuff_new[new_points++]=newz;
				}
			else
				{
				/* first point inside, last point outside ? */
				if (insideZbound(zbuff_old[0],edgeNum))
					{
					Zintersect(
					   xbuff_old[numcoords-1],
				   	   ybuff_old[numcoords-1], 
				   	   zbuff_old[numcoords-1], 
				   	   xbuff_old[0],
					   ybuff_old[0], 
				   	   zbuff_old[0],edgeNum, 
					   &newx,&newy,&newz
						  );
					xbuff_new[new_points]=newx;
					ybuff_new[new_points]=newy;
					*zdepth += newz;
					zbuff_new[new_points++]=newz;
					}
				}
			}
		}
	/* check if first point outside, last point inside */
	if ( (insideZbound(zbuff_old[numcoords-1],edgeNum)) &&
	     (!insideZbound(zbuff_old[0],edgeNum)) )
			{
			Zintersect(
			   xbuff_old[numcoords-1],
			   ybuff_old[numcoords-1], 
			   zbuff_old[numcoords-1], 
			   xbuff_old[0],
			   ybuff_old[0], 
			   zbuff_old[0],edgeNum, &newx,&newy,&newz);
			xbuff_new[new_points]=newx;
			ybuff_new[new_points]=newy;
			*zdepth += newz;
			zbuff_new[new_points++]=newz;
			}
	*newnumcoords = new_points;
	if (new_points != 0)
		*zdepth = *zdepth / (float) new_points;

}

void clip_Zline(edgeNum,numcoords,newnumcoords,zdepth)
int edgeNum,numcoords,*newnumcoords;
float *zdepth;
{
	int old_points=0,new_points=0;
	float newx,newy,newz,*xbuff_old,*ybuff_old,*zbuff_old,
	      *xbuff_new,*ybuff_new,*zbuff_new;

	*zdepth = 0.0;
	ger_debug("Clip_Zline with %d points",numcoords);
	ger_debug("Zmax:%f, Zmin:%f \n",Zmax,Zmin);
	if (numcoords < 2)
		{
		*newnumcoords = 0;
		return;
		};
	if (edgeNum == HITHER_PLANE)
		{
		xbuff_old = Xcoord_buffer;
		ybuff_old = Ycoord_buffer;
		zbuff_old = Zcoord_buffer;
		xbuff_new = Xclip_buffer;
		ybuff_new = Yclip_buffer;
		zbuff_new = Zclip_buffer;
		}
	else
		{
		xbuff_old = Xclip_buffer;
		ybuff_old = Yclip_buffer;
		zbuff_old = Zclip_buffer;
		xbuff_new = Xcoord_buffer;
		ybuff_new = Ycoord_buffer;
		zbuff_new = Zcoord_buffer;
		}
	while (old_points < numcoords)
		{
		/* Points inside */
		while ( (old_points < numcoords) && 
			(insideZbound(zbuff_old[old_points],edgeNum)))
			{
			xbuff_new[new_points]=xbuff_old[old_points];
			ybuff_new[new_points]=ybuff_old[old_points];
			*zdepth += zbuff_old[old_points];
			zbuff_new[new_points++]=zbuff_old[old_points++];
			}

		if (old_points < numcoords)
			{
			/*	Intersection point   	*/
			if (old_points > 0)
				{
				Zintersect(xbuff_old[old_points-1],
				   	   ybuff_old[old_points-1], 
				   	   zbuff_old[old_points-1],
				   	   xbuff_old[old_points], 
				   	   ybuff_old[old_points], 
				   	   zbuff_old[old_points],edgeNum,
				   	   &newx,&newy,&newz);
				xbuff_new[new_points]=newx;
				ybuff_new[new_points]=newy;
				*zdepth += newz;
				zbuff_new[new_points++]=newz;
			        }

			/*	Points outside		*/
			while ( (old_points < numcoords) && 
			        (!(insideZbound(zbuff_old[old_points],
				   edgeNum))))
				{
				old_points++;
				}

			/*	Intersection point	*/
			if (old_points < numcoords) 
				{
				Zintersect(xbuff_old[old_points],
				   	   ybuff_old[old_points], 
				 	   zbuff_old[old_points], 
			   		   xbuff_old[old_points-1], 
				   	   ybuff_old[old_points-1], 
				  	   zbuff_old[old_points-1],edgeNum,
				   	   &newx,&newy,&newz);
				xbuff_new[new_points]=newx;
				ybuff_new[new_points]=newy;
				*zdepth += newz;
				zbuff_new[new_points++]=newz;
				}
			}
		}
	*newnumcoords = new_points;
	if (new_points != 0)
		*zdepth /= (float) new_points;

}
/*   OLD 2-D clipping routines		*/

/*
void intersect(x1,y1,z1,x2,y2,z2,edgeNum,windNum,interX,interY,interZ)
float x1,y1,z1,x2,y2,z2,*interX,*interY,*interZ;
int edgeNum,windNum;
{
	 float dx,dy,dz,m,bminc,bminr,sminc,sminr,bmaxc,bmaxr,smaxc,smaxr;
	
	
	dx =  x2-x1;
	dy =  y1-y2;
	dz =  z1-z2;
	if (dx == 0.0)
	    m = HUGE;
	else
	    m = dy/dx;
	bminc = (float) bpicw_minc + 1.0;
	sminc = (float) spicw_minc + 1.0;
	bmaxc = (float) bpicw_maxc - 1.0;
	smaxc = (float) spicw_maxc - 1.0;
	bminr = (float) bpicw_minr + 1.0;
	sminr = (float) spicw_minr + 1.0;
	bmaxr = (float) bpicw_maxr - 1.0;
	smaxr = (float) spicw_maxr - 1.0;
	switch(edgeNum)
		{
		case 0:
			if (windNum==1) 
				{
				*interX = bminc;
				*interY = y1 - 
				    m*(bminc - x1);
				*interZ = z1 - 
				    m*(bminc - x1);
				}
			else
				{
				*interX = sminc;
				*interY = y1 - 
				   m*(sminc - x1);
				*interZ = z1 - 
				   m*(sminc - x1);
				}
			break;
		case 1:
			if (windNum==1) 
				{
				*interX = bmaxc;
				*interY = y1 - 
				   m*(bmaxc - x1);
				*interZ = z1 - 
				   m*(bmaxc - x1);
				}
			else
				{
				*interX = smaxc;
				*interY = y1 - 
				   m*(smaxc - x1);
				*interZ = z1 - 
				   m*(smaxc - x1);
				}
			break;
		case 2:
			if (windNum==1) 
				{
				*interX = x1 + 
				   (y1-bminr)/m;	
				*interZ = z1 + 
				   (y1-bminr)/m;	
				*interY = bminr;
				}
			else
				{
				*interX = x1 + 
				   (y1-sminr)/m;	
				*interZ = z1 + 
				   (y1-sminr)/m;	
				*interY = sminr;
				}
			break;
		case 3:
			if (windNum==1) 
				{
				*interX = x1 + 
				   (y1-bmaxr)/m;	
				*interZ = z1 + 
				   (y1-bmaxr)/m;	
				*interY = bmaxr;
				}
			else
				{
				*interX = x1 + 
				   (y1-smaxr)/m;	
				*interZ = z1 + 
				   (y1-smaxr)/m;	
				*interY = smaxr;
				}
			break;
		default:
			printf("ERROR \n");
			break;
		}
}


int inside(x1,y1,edgenum,windNum)
float x1,y1;
int edgenum,windNum;
{
	int x,y;

	x= (int) x1;
	y= (int) y1;
	switch(edgenum)
		{
			if (windNum == 1)
				if (x>bpicw_minc) 	return(1);
				else 			return(0);
			else 
				if (x>spicw_minc) 	return(1);
				else 			return(0);
			if (windNum == 1)
				if (x<bpicw_maxc) 	return(1);
				else 			return(0);
			else 
				if (x<spicw_maxc) 	return(1);
				else 			return(0);

		case 2:	
			if (windNum == 1)
				if (y>bpicw_minr) 	return(1);
				else 			return(0);
			else 
				if (y>spicw_minr) 	return(1);
				else 			return(0);
		case 3:	
			if (windNum == 1)
				if (y<bpicw_maxr) 	return(1);
				else 			return(0);
			else 
				if (y<spicw_maxr) 	return(1);
				else 			return(0);
		default:
			printf("ERROR2 \n");
			break;
		}
}

clip_edge(poly,edgeNum,new_poly,windNum)
ren_polytype *poly,*new_poly;
int edgeNum,windNum;
{
	int old_points,totPoints,new_points=0;
	float newx,newy,newz;

	old_points = 0;
	totPoints = poly->numcoords2d;
	while (old_points < totPoints)
		{
		while ( (old_points < totPoints) && (inside(
		   poly->xcoord2d[old_points],poly->ycoord2d[old_points],
		   edgeNum,windNum)))
			{
			new_poly->xcoord2d[new_points] = poly->xcoord2d[old_points];
			new_poly->ycoord2d[new_points] = poly->ycoord2d[old_points];
			new_poly->zcoord2d[new_points++] = poly->zcoord2d[old_points];
			old_points++;
			}
		if (old_points < totPoints)
			{
			if (old_points > 0)
				{
				intersect(poly->xcoord2d[old_points-1],
				   poly->ycoord2d[old_points-1], 
				   poly->zcoord2d[old_points-1],
				   poly->xcoord2d[old_points], 
				   poly->ycoord2d[old_points], 
				   poly->zcoord2d[old_points],
				   edgeNum,windNum, &newx,&newy,&newz);
				new_poly->xcoord2d[new_points]=newx;
				new_poly->ycoord2d[new_points]=newy;
				new_poly->zcoord2d[new_points++] =newz;
			        }
			while ( (old_points < totPoints) && (!(inside(
			   poly->xcoord2d[old_points],poly->ycoord2d[old_points],
			   edgeNum,windNum))))
				{
				old_points++;
				}
			if (old_points < totPoints) 
				{
				intersect(poly->xcoord2d[old_points],
				   poly->ycoord2d[old_points], 
				   poly->zcoord2d[old_points], 
				   poly->xcoord2d[old_points-1], 
				   poly->ycoord2d[old_points-1], 
				   poly->zcoord2d[old_points-1], 
			           edgeNum,windNum, &newx,&newy,&newz);
				new_poly->xcoord2d[new_points]=newx;
				new_poly->ycoord2d[new_points]=newy;
				new_poly->zcoord2d[new_points++] = newz;
				}
			else
				{
				if (inside(poly->xcoord2d[0], poly->ycoord2d[0],
			    	    edgeNum,windNum))
					{
					intersect(poly->xcoord2d[totPoints-1],
				   	   poly->ycoord2d[totPoints-1], 
				   	   poly->zcoord2d[totPoints-1], 
				   	   poly->xcoord2d[0],
					   poly->ycoord2d[0], 
					   poly->zcoord2d[0], 
				   	   edgeNum,windNum, &newx,&newy,&newz);
					new_poly->xcoord2d[new_points]=newx;
					new_poly->ycoord2d[new_points]=newy;
					new_poly->zcoord2d[new_points++] = newz;
					}
				}
			}
		}
	if ( (inside(poly->xcoord2d[totPoints-1],
	            poly->ycoord2d[totPoints-1],edgeNum,windNum)) &&
	     (!inside(poly->xcoord2d[0],poly->ycoord2d[0],edgeNum,
		     windNum)) )
			{
			intersect(poly->xcoord2d[totPoints-1],
			   poly->ycoord2d[totPoints-1], 
			   poly->zcoord2d[totPoints-1], 
			   poly->xcoord2d[0],
			   poly->ycoord2d[0], 
			   poly->zcoord2d[0], 
			   edgeNum,windNum, &newx,&newy,&newz);
			new_poly->xcoord2d[new_points]=newx;
			new_poly->ycoord2d[new_points]=newy;
			new_poly->zcoord2d[new_points++] = newz;
			}
	new_poly->numcoords2d=new_points;
	new_poly->colorind = poly->colorind;
}
*/
