/****************************************************************************
 * painter.c
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 * Author Chris Nuuja
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/

#include <ctype.h>
#include <math.h>
#include "ge_error.h"
#include "alisp.h"
#include "painter_vars.h"
#include "painter.h"

/* 
The following is designed to figure out what machine this is running
on, and include appropriate definition files.
*/
#ifdef unix
#define USE_UNIX
#endif
#ifdef CRAY
#undef USE_UNIX
#endif
#ifdef ardent
#undef USE_UNIX
#endif
#ifdef CRAY
#include "unicos_defs.h"
#endif
#ifdef ardent
#include "unicos_defs.h"
#endif
#ifdef USE_UNIX
#include "unix_defs.h"
#endif

/* 			Global Variable Defintions			*/

/*    Global strings to be used as input to gplot initialization routines    */
char FileName[2] = "-"; /* If a filename where put here, output would become */
			/* a cgm file of that name, which could be           */
			/* interpreted by gplot.  This will changed to an    */
			/* input option later    			     */
char *DeviceName;       /* Name of device as a string, filled in ren_setup   */


/*	Hither and Yon Clipping Boundries   */
float Zmin, Zmax;


/*  Tranformation Matricies   */ 
float *Identity;      /* the identity tranform   */
float *ObjectTrans;   /* the current polygon's world-coordinate trans */
		      /* as determined by its position in the gob   */
float *ViewMatrix;    /* defines the world-to-camera-view transform */

/*  Coordinate buffers used for transformations and clipping  */
float *Xcoord_buffer,*Ycoord_buffer,*Zcoord_buffer;
float *Xclip_buffer, *Yclip_buffer, *Zclip_buffer;
int TempCoordBuffSz=256;


/*  
	The following variables define the 3d-to-2d-perspective transform.
	Since this is not set very often, and never multiplied,
	it is quicker to maintain seperate matrix values than to access
	seperate indices into the matrix.  EIx represents the value of
	the transformation matrix at position x.
*/  
extern float EI0, EI1,  EI2,  EI3,  EI4,  EI5,  EI6,  EI7,  EI8,  
	     EI9, EI10, EI11, EI12, EI13, EI14, EI15;


/*			Function Declarations			*/

/* 
   Utility function to enlarge the size of the temporary coordinate buffers if
   needed
*/
void check_coordbuffsz(numcoords)
int numcoords;
{
	if (numcoords >= TempCoordBuffSz)
		{
		while(numcoords >= TempCoordBuffSz)
			TempCoordBuffSz  *= 2;
		Xcoord_buffer = (float *) 
		   realloc(Xcoord_buffer,TempCoordBuffSz*sizeof(float));
		Ycoord_buffer = (float *) 
		   realloc(Ycoord_buffer,TempCoordBuffSz*sizeof(float));
		Zcoord_buffer = (float *) 
		   realloc(Zcoord_buffer,TempCoordBuffSz*sizeof(float));
		Xclip_buffer = (float *) 
		   realloc(Xclip_buffer,TempCoordBuffSz*sizeof(float));
		Yclip_buffer = (float *) 
		   realloc(Yclip_buffer,TempCoordBuffSz*sizeof(float));
		Zclip_buffer = (float *) 
		   realloc(Zclip_buffer,TempCoordBuffSz*sizeof(float));
		}
}

/*
   Global Data Used: Xcoord_buffer, Ycoord_buffer, and Zcoord_buffer,
		     DepthBuffer, and DepthCount;
   Expl:  This routine adds a record to the depth buffer, which is later
	  sorted to achieve painters algorithm for hidden surface handeling.
	  It creates a polygon record using the coordinates in the
	  global coordinate buffers and the input parameters.  This polygon
	  is then added to DepthBuffer at index DepthCount.
*/
void insert_depthBuff(numcoords,zdepth,type,color)
int numcoords;
float zdepth;
primtype type;
int color;
{
	int i,newpoly,x_index,y_index,z_index;

	newpoly = new_DepthPoly();
	fillDpoly_rec(newpoly,numcoords,type);

	x_index = DPolyBuffer[newpoly].x_index;
	y_index = DPolyBuffer[newpoly].y_index;
	z_index = DPolyBuffer[newpoly].z_index;
	for (i=0;i<numcoords;i++)
		{
		DCoordBuffer[x_index++] = Xcoord_buffer[i];
		DCoordBuffer[y_index++] = Ycoord_buffer[i];
		DCoordBuffer[z_index++] = Zcoord_buffer[i];
		};
	DepthBuffer[DepthCount].poly = newpoly;
	DepthBuffer[DepthCount].key = zdepth;
	DepthCount++;
	DPolyBuffer[newpoly].color = color;
	if (DepthCount > MaxDepthPoly)
		printf("ERROR: Depth Buffer Overflow: %d > %d\n",
			DepthCount, MaxDepthPoly);

}

/*
   Global Data Used: CoordBuffer, LightCount, LightPosBuffer,
          LightColorBuffer, ColorBuffer

   Expl:  This routine alters the intensity of <poly>'s color record
	  depending on its orientation to the lighting source(s).  
*/
int calc_intensity(oldcolor, polygon, trans, normal)
int oldcolor;
int polygon;
float *trans;
ren_vectortype *normal;
{
  ren_vectortype light;
  float red, green, blue, mx, my, mz, x, y, z, distance, percent,
  source_red, source_green, source_blue;
  int newcolor, lightnum;
  register float *rtrans= trans;
  
  red   = ColorBuffer[oldcolor].r;
  green = ColorBuffer[oldcolor].g;
  blue  = ColorBuffer[oldcolor].b;
  
  source_red = 0.0;
  source_green = 0.0;
  source_blue = 0.0;
  
  /* Translate vertex coords from model to world coordinate system */
  mx= CoordBuffer[ PolyBuffer[polygon].x_index ];
  my= CoordBuffer[ PolyBuffer[polygon].y_index ];
  mz= CoordBuffer[ PolyBuffer[polygon].z_index ];
  x = rtrans[0]*mx + rtrans[4]*my + rtrans[8]*mz + rtrans[12];
  y = rtrans[1]*mx + rtrans[5]*my + rtrans[9]*mz + rtrans[13];
  z = rtrans[2]*mx + rtrans[6]*my + rtrans[10]*mz + rtrans[14];
  
  for (lightnum = 0; lightnum < LightCount; lightnum++)
    {
      light.x = LightPosBuffer[lightnum].x - x;
      light.y = LightPosBuffer[lightnum].y - y;
      light.z = LightPosBuffer[lightnum].z - z;
      distance = sqrt( (light.x * light.x) + (light.y * light.y)
		      + (light.z * light.z) );
      if (distance == 0.0) percent= 1.0;
      else {
	light.x /= distance; light.y /= distance; light.z /= distance;
	percent = dotproduct( normal, &light );
      }
      if (percent < 0.0)
	percent = 0.0;
      source_red += (percent * LightColorBuffer[lightnum].r);
      source_green += (percent * LightColorBuffer[lightnum].g);
      source_blue += (percent * LightColorBuffer[lightnum].b);
    }
  
  if ((red= red*ColorBuffer[AmbientColor].r  +  red * source_red) > 1.0)
    red = 1.0;
  if ((green= green*ColorBuffer[AmbientColor].g + green * source_green) > 1.0)
    green = 1.0;
  if ((blue= blue*ColorBuffer[AmbientColor].b  + blue * source_blue) > 1.0)
    blue = 1.0;
  /*	the <a> in a color rec is not used yet 		*/
  newcolor = makeDcolor_rec(red,green,blue,1.0);
  return(newcolor);
}


/*
   Global Data Used: NONE (The wire-frame model switch will go here so that
			   the interior style will be hollow)
   Expl:  Draws a polygon,polyline, or polymarker with it's recorded color 
	  wristl is a cgmgen routine to set the interior style;
	  wpgndc is a cgmgen routine to set the polygon fill color;
	  wrtpgn is a cgmgen routine to draw a polygon;
*/
void render_polyrec(depth_index)
int depth_index;
{
	int i,error=0,poly_index,numcoords,x_index,y_index,color;
	float red,green,blue;

	poly_index = DepthBuffer[depth_index].poly;
		
	i=1;
	wristl(&i,&error);
	if (error)
		fprintf(stderr,"ERROR: Bad interior style:%d\n",error);
	x_index = DPolyBuffer[poly_index].x_index;
	y_index = DPolyBuffer[poly_index].y_index;
	numcoords = DPolyBuffer[poly_index].numcoords;

	color = DPolyBuffer[poly_index].color;
	red   = DColorBuffer[color].r;
	green = DColorBuffer[color].g;
	blue  = DColorBuffer[color].b;

	switch(DPolyBuffer[poly_index].type)
	   {
	   case POLYGON:
		wpgndc(&red,&green,&blue,&error);
		wrtpgn(DCoordBuffer+x_index,DCoordBuffer+y_index, &numcoords,&error);
		break;
	   case POLYLINE:
	        wplndc(&red,&green,&blue,&error);
		wrplin(DCoordBuffer+x_index,DCoordBuffer+y_index, &numcoords,&error);
		break;
	   case POLYMARKER:
		wpmkdc(&red,&green,&blue,&error);
		wrtpmk(DCoordBuffer+x_index,DCoordBuffer+y_index,&numcoords, &error);
		break;
	   default:
		fprintf(stderr,"ERROR: Unknown primitive:%d\n",error);
	   }
}


/*
   Global Data Used:  Xcoord_buffer, Ycoord_buffer, Zcoord_buffer
   Expl:  This routine transforms the polygon cooridnates in the
	  global coordinate buffers, which are positions in 3 dimensional
	  space, to their corresponding positions in 2-dimensional space.
	  The perspective transformation happens here, as well as normalization
	  of the viewing fustrum.  All these transformations are contained
	  in the transformation matrix represented by the global variables
	  EIx where x is between 0 and 15.  Note that only the EIx values with
   	  non-zero values are used in the calculations.
*/
void trans3Dto2D(numcoords)
int numcoords;
{ 

    float x,y;
    float z,w;
    int i;

    for (i=0;i<numcoords;i++)
	{
	x = Xcoord_buffer[i];
	y = Ycoord_buffer[i];
	z = Zcoord_buffer[i];

        Xcoord_buffer[i] = EI0*x + EI8*z;
        Ycoord_buffer[i] = EI5*y + EI9*z;
        Zcoord_buffer[i] = EI10*z + EI14;
        w =  EI11*z;
        if (w == 0.0) w= HUGE;

/* must normalize the coordinates */
        Xcoord_buffer[i] /=  w;
        Ycoord_buffer[i] /=  w;
        Zcoord_buffer[i] /=  w;
	}
}

/*
   Global Data Used: Xcoord_buffer, Ycoord_buffer, Zcoord_buffer,
		     ViewMatrix.
   Expl:  This routine transforms <polygon>'s coordinates from their position in
	  the world coordinate system to their positions with respect to the 
	  camera.  Also, any transformations of gob-children relative to gobs 
	  have been pre-concatinated onto ViewMatrix, so these transformations 
	  happen here as well.  The transformed coordinates are placed in the
	  global coordinate buffers.
*/
void translateWtoI(poly,numcoords)
int poly;
int numcoords;
{
	int x_index,y_index,z_index;
	register float *ViewM=ViewMatrix,x,y,z;
	int i;

	if (PolyBuffer[poly].numcoords == -1)
		fprintf(stderr,"ERROR, Bad Polygon In WtoI\n");
	x_index = PolyBuffer[poly].x_index;
	y_index = PolyBuffer[poly].y_index;
	z_index = PolyBuffer[poly].z_index;
	for (i=0; i<numcoords;i++)
	    {
	    x = CoordBuffer[x_index++];
	    y = CoordBuffer[y_index++];
	    z = CoordBuffer[z_index++];

	    Xcoord_buffer[i] = ViewM[0]*x + ViewM[4]*y + ViewM[8]*z + ViewM[12];
	    Ycoord_buffer[i] = ViewM[1]*x + ViewM[5]*y + ViewM[9]*z + ViewM[13];
	    Zcoord_buffer[i] = ViewM[2]*x + ViewM[6]*y + ViewM[10]*z + ViewM[14];
	    }
}


/*
   Global Data Used:NONE
   Expl:  This routine returns 1 if the polygon defined by normal vector
	  <normal> is facing the camera, and 0 if it is facing away.
*/
int forward_facing(normal)
ren_vectortype *normal;
{
  /* All we have to do is calculate the z component of the normal in
   * the camera's frame.  If it is less than zero, the polygon is
   * facing backwards.
   */
  register float *ViewM=ViewMatrix;
  float zcomp;

  zcomp= ViewM[2]*normal->x + ViewM[6]*normal->y 
    + ViewM[10]*normal->z;

  return( (zcomp>=0.0) );
}


/*
   Global Data Used:ColorBuffer
   Expl:  This routine transforms the polymarker at index <polymarker> from
	  world coordinates to (virtual)screen coordinates, and inserts it
	  into the depth buffer for later sorting and displaying.  The original
	  marker record is unaltered.
*/
void render_polymarker(polymarker,numcoords,color)
int polymarker;
int numcoords;
int color;
{
	int newcolor;
	float z_depth,red,green,blue;

	if (numcoords != 1)
		{
		fprintf(stderr,"ERROR: Marker with more than one coordinate\n");
		return;
		}
	check_coordbuffsz(numcoords);
	/*  
	  translateWtoI places the translated coordinates of the polygon in
	  the buffers Xcoord_buffer, Ycoord_buffer, Zcoord_buffer
	*/
	translateWtoI( polymarker,numcoords );
	z_depth = Zcoord_buffer[0];

	if (!insideZbound(z_depth,HITHER_PLANE))
		return;
	if (!insideZbound(z_depth,YON_PLANE))
		return;
	trans3Dto2D(numcoords);
	red   = ColorBuffer[color].r;
	green = ColorBuffer[color].g;
	blue  = ColorBuffer[color].b;
	newcolor = makeDcolor_rec(red,green,blue,1.0);
	/*  Add polygon to global DepthBuffer, increment DepthCount  */
	insert_depthBuff(numcoords,z_depth,POLYMARKER,newcolor);
}

/*
   Global Data Used:ColorBuffer
   Expl:  This routine transforms the polyline at index <polyline> from
	  world coordinates to (virtual) screen coordinates, clips it against
	  the hither and yon planes, and inserts it into the depth buffer for 
	  later sorting and displaying.  The original line record is unaltered.
*/
void render_polyline(polyline,numcoords,color)
int polyline;
int numcoords;
int color;
{
	int new_numcoords=0,newcolor;
	float z_depth,red,green,blue;

	/*  
	  translateWtoI places the translated coordinates of the polygon in
	  the buffers Xcoord_buffer, Ycoord_buffer, Zcoord_buffer
	*/
	check_coordbuffsz(numcoords);
	translateWtoI( polyline,numcoords );

	/*  Clip on Far and near Z */
	/*  Clipped coordinates may contain more points than before 
	    clipping */
	z_depth=0.0;
	clip_Zline(HITHER_PLANE,numcoords,&new_numcoords,&z_depth);
	clip_Zline(YON_PLANE,new_numcoords,&numcoords,&z_depth);
	if (numcoords < 2)
		return;
	trans3Dto2D(numcoords);
	red   = ColorBuffer[color].r;
	green = ColorBuffer[color].g;
	blue  = ColorBuffer[color].b;
	newcolor = makeDcolor_rec(red,green,blue,1.0);
	/*  Add polyline to global DepthBuffer, increment DepthCount  */
	insert_depthBuff(numcoords,z_depth,POLYLINE,newcolor);
}

/*
   Global Data Used: Xcoord_buffer,Ycoord_buffer,Zcoord_buffer;
   Expl:  This routine transforms the polygon at index <polygon> from
	  world coordinates to (virtual)screen coordinates, clips it against
	  the hither and yon planes, calculates its color with respect to
	  its orientation to the light, and inserts it into the depth buffer 
	  for later sorting and displaying.  If backface culling is on and a 
	  polygon is facing away from the camera, it is not added to the depth 
	  buffer.  The original polygon record is unaltered.
*/
void render_polygon(polygon,trans,numcoords,backcull,color)
int polygon;
float *trans;
int numcoords;
int backcull;
int color;
{
	int new_numcoords=0,newcolor;
	float length,z_depth;
	ren_vectortype normal;

	/*  
	  translateWtoI places the translated coordinates of the polygon in
	  the buffers Xcoord_buffer, Ycoord_buffer, Zcoord_buffer
	*/
	check_coordbuffsz(numcoords);
	translateWtoI( polygon,numcoords );

	/* calculate the polygon's normal */
	calc_normal(polygon,trans,&normal);
	length = vector_length(&normal);
	if (length == 0.0)
		length=1.0;

	/*  do backface culling, given surface normal and a point on surface */
	if ( backcull && (!forward_facing(&normal)) ) {
	  return;
	}
	else {
	  normal.x /= length;
	  normal.y /= length;
	  normal.z /= length;
	  newcolor = calc_intensity(color,polygon,trans,&normal);
	  /*  Clip on Far and near Z */
	  /*  Clipped coordinates may contain more points than before 
	      clipping */
	  /*  clip_Zpolygon assumes Hither plane is clipped before Yon */
	  z_depth=0.0;
	  clip_Zpolygon(HITHER_PLANE,numcoords,&new_numcoords,&z_depth);
	  clip_Zpolygon(YON_PLANE,new_numcoords,&numcoords,&z_depth);
	  /* Make sure we still have a polygon */
	  if (numcoords < 3)
	    return;
	  trans3Dto2D(numcoords);
	  /*  Add polygon to global DepthBuffer, increment DepthCount  */
	  insert_depthBuff(numcoords,z_depth,POLYGON,newcolor);
	};
}


/*
   Global Data Used: ViewMatrix,ObjectBuffer,PolyBuffer,
   Expl:  Transforms each polygon at index <object> from world coordinates to
	  (virtual)screen coordinates and stores them in DepthBuffer, which is 
	  later sorted and drawn in order.  <ViewMatrix> is altered to include
	  <object_trans> for drawing all polygons at index <object>, and
	  then returned to its original value.
*/
void render_primitive(object,object_trans,back_cull,color)
int object;
float *object_trans;
int back_cull;
int color;
{
    register int poly,poly_index;
    int numcoords,numpolys,newcolor,i;
    float *oldM;
    
    oldM = ViewMatrix;
    /*  
	Add gob-related transformations into ViewMatrix.  These occur BEFORE
	the translation from World to Eye coordinates (the ViewMatrix)
    */
    ViewMatrix = mult3dMatrices(object_trans, ViewMatrix);
    poly = ObjectBuffer[object].poly_index;
    numpolys = ObjectBuffer[object].num_polygons;
    for (poly_index=0;poly_index<numpolys;poly_index++)
	{
	numcoords = PolyBuffer[poly].numcoords;
	if (PolyBuffer[poly].color)
		newcolor = PolyBuffer[poly].color;
	else
		newcolor = color;   /* if no color of its own, inherit one */
	
	switch(PolyBuffer[poly].type)
		{
		case POLYGON:	
			render_polygon(poly,object_trans,numcoords,
				       back_cull,newcolor);
			break;
		case POLYLINE:
			render_polyline(poly,numcoords,newcolor);
			break;
		case POLYMARKER:
			render_polymarker(poly,numcoords,newcolor);
			break;
		default:
			fprintf(stderr,"ERROR: Undefined poly-type:%d\n",
			       PolyBuffer[poly].type);
			break;
		}
	poly++;
	}
   free(ViewMatrix);
   ViewMatrix = oldM;
}

	
/*
   Global Data Used: DeviceName, FileName
   Expl: initialized the cgmgen routines.  It tells gplot that
	 the device is DeviceName, to output to file FileName
	 (which, if it is '-', will cause output to go to the screen)
	 and that all coordinates are normalized to between xmin,ymin
	 and xmax ymax.  If xmax and ymax are not equal, distortion
	 in size results.  If they are made smaller, magnification of
	 the image will occur.  They are set to 1.0 so that a 1x1 square
	 located at the camera position will just barely cover the entire
	 screen.  They have been set to values that produce results close to
	 Dore's renderer
*/
void init_cgmgen(outfile)
char *outfile;
{

	int error=0;
	float xmin=0.0,ymin=0.0,xmax=2.0,ymax=2.0;

	csetdev(DeviceName,&error);
	wrcopn(outfile,&error);
	setwcd(&xmin,&ymin,&xmax,&ymax,&error);
	if (error)
		fprintf(stderr,"ERROR: Cgmgen initialization error:%d\n",error);
}

/*
   Global Data Used: none
   Expl: shuts down the cgmgen routines by calling wrtend.
*/
void shutdown_cgmgen()
{
        int error=0;

	wrtend(&error);
	if (error)
	        fprintf(stderr,"ERROR: Cgmgen shutdown error:%d\n",error);
}

/*
   Global Data Used: Identity, LightLocation,LightIntensity,AmbientIntensity,
		     ViewMatrix, Zmin, Zmax,
		     Xcoord_buffer, Ycoord_buffer, Zcoord_buffer
   Expl:  calls init_cgmgen, and initializes all global variables.
	   Zmin, Zmax, and EyeMatrix will all be set to appropriate values
	   in ren_camera (located in painter_ren.c).  Light characteristics are
	   set, but are not (currently) used.
*/
void init_renderer(outfile)
char *outfile;
{
	int i;

	Identity =  make3dScale(1.0,1.0,1.0);
	
	/*  
	    ViewMatrix is the matrix that translates  coordinates from
	    their position in the world to their position relative to the
	    Eye/Camera.
	*/
	ViewMatrix = (float *) malloc(16*sizeof(float));
	for (i=0;i<16;i++)
		ViewMatrix[i]=0.0;
	ViewMatrix[0] = 1.0;
	ViewMatrix[5] = 1.0;
	ViewMatrix[10] = 1.0;
	ViewMatrix[15] = 1.0;

/*	Set (default) Z clipping boundries,  Zmin is hither, Zmax is yon */
	Zmin = -0.01;
	Zmax = -900.0;

	/*
	   These buffers hold coordinates of polygons, polylines, markers, etc
	   as they are transformed from World coordinates to screen coordinates
	   The Xclip_buffer,Zclip_buffer,Yclip_buffer are used to
	   hold coordinates between clippings agianst the hither and
	   yon planes (Zmin and Zmax).
	*/
	Xcoord_buffer = (float *) malloc(TempCoordBuffSz*sizeof(float));
	Ycoord_buffer = (float *) malloc(TempCoordBuffSz*sizeof(float));
	Zcoord_buffer = (float *) malloc(TempCoordBuffSz*sizeof(float));
	Xclip_buffer = (float *) malloc(TempCoordBuffSz*sizeof(float));
	Yclip_buffer = (float *) malloc(TempCoordBuffSz*sizeof(float));
	Zclip_buffer = (float *) malloc(TempCoordBuffSz*sizeof(float));
	init_cgmgen(outfile);

}

/*
   Global Data Used: none
   Expl:  calls shutdown_cgmgen.
*/
void shutdown_renderer()
{
        shutdown_cgmgen();
}



