/****************************************************************************
 * kui.c
 * Author Joel Welling and Doug Wagner
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This module provides a 'keyboard user interface' which lets the user rotate
and examine models with only a keyboard.
*/

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "alisp.h"
#include "p3d.h"
#include "ge_error.h"
#include "matrix_ops.h"
#include "control.h"
#include "ren.h"
#include "ui.h"
#include "assist.h"

/* 
Default angle to rotate camera position (degrees) and default fractional
distance to translate camera position
*/
#define default_angle 5.0
#define default_fraction 0.1

/* Help and entry messages */
static char entry_message[]= "Ready for commands (type 'h' for help)\n";
static char help_message[]= "\
The following character inputs are accepted:\n\
\n\
  u	move the camera up\n\
  d	move the camera down\n\
  l	move the camera left\n\
  r	move the camera right\n\
  c	rotate the camera clockwise\n\
  a	rotate the camera anticlockwise\n\
  i	move the camera in\n\
  o	move the camera out\n\
  t	start trace file\n\
  U	move the focal point up\n\
  D	move the focal point down\n\
  L	move the focal point left\n\
  R	move the focal point right\n\
  I	move the focal point in\n\
  O	move the focal point out\n\
  .     redraw the current model\n\
  q     quit moving and go on to next model\n\
  e	exit from the program\n\
  h,?	print this message\n\
\n\
Angular moves are 5 degrees; movement in and out and movement of\n\
the focal point is 5% of the current distance to the origin.  Any\n\
letter can be prefixed by an integer, which then multiplies the\n\
amount moved or rotated.\n\
"; /* End of help message */

/* Name of file to hold trace, file pointer for file which holds the
   trace and flag to indicate that tracing is enabled. */
#define TRACE_FILE_NAME "kui_trace.dat"
static FILE *tracefile;
static int trace_on= 0;

/* Current camera data*/
static Point lookfrom, lookatpt, orig_lookat;
static Vector up;
static float fovea, hither, yon;
static Color background;

/* Current light data */
static Gob lightgob;
static Attribute_list lightattr;

/* Variable to hold initialization state */
static int initialized= 0;

static void align_up()
/* 
This routine removes components of the up vector which are parallel to
the viewing direction, and renormalizes the up vector.
*/
{
  float vx, vy, vz, ux, uy, uz, unorm, vnorm, dot;

  ger_debug("align_up:");

  vx= point_x(lookfrom) - point_x(lookatpt);
  vy= point_y(lookfrom) - point_y(lookatpt);
  vz= point_z(lookfrom) - point_z(lookatpt);
  vnorm= vx*vx + vy*vy + vz*vz;
  if (vnorm==0.0) {
    ger_error("align_up: viewing distance is zero! continueing.");
    return;
  }

  ux= vector_x( up );
  uy= vector_y( up );
  uz= vector_z( up );

  dot= ux*vx + uy*vy + uz*vz;

  ux= ux - dot * (vx/vnorm);
  uy= uy - dot * (vy/vnorm);
  uz= uz - dot * (vz/vnorm);
  unorm= ux*ux + uy*uy + uz*uz;
  if (unorm==0.0) {
    ger_error("align_up: up vector vanishes, using y direction");
    ux= 0.0; uy= 1.0; uz= 0.0; unorm= 1.0;
  }

  unorm= sqrt(unorm);
  ux= ux/unorm;
  uy= uy/unorm;
  uz= uz/unorm;

  set_vector_x( up, ux );
  set_vector_y( up, uy );
  set_vector_z( up, uz );

  return;
}

static void rotate_up( angle )
float angle;
/* This routine rotates the camera up by 'angle' degrees. */
{
  float xview, yview, zview, xcomp, ycomp, zcomp, *matrix, *new_from,
        from_vec[3];

  ger_debug("rotate_up: angle= %f",angle);

  from_vec[0]= point_x(lookfrom);
  from_vec[1]= point_y(lookfrom);
  from_vec[2]= point_z(lookfrom);

  xview= -point_x(lookfrom);
  yview= -point_y(lookfrom);
  zview= -point_z(lookfrom);

  xcomp= vector_y(up)*zview - vector_z(up)*yview;
  ycomp= vector_z(up)*xview - vector_x(up)*zview;
  zcomp= vector_x(up)*yview - vector_y(up)*xview;

  matrix= make_rotate_c( angle, xcomp, ycomp, zcomp );
  new_from= matrix_vector_c( matrix, from_vec );

  set_point_x( lookfrom, new_from[0] );
  set_point_y( lookfrom, new_from[1] );
  set_point_z( lookfrom, new_from[2] );

  free( (char *)new_from );
  free( (char *)matrix );
}

static void rotate_right( angle )
float angle;
/* This routine rotates the camera right by 'angle' degrees. */
{
  float *matrix, *new_from, from_vec[3];

  ger_debug("rotate_right: angle= %f",angle);

  from_vec[0]= point_x(lookfrom);
  from_vec[1]= point_y(lookfrom);
  from_vec[2]= point_z(lookfrom);

  matrix= make_rotate_c( angle, vector_x(up), vector_y(up), vector_z(up) );
  new_from= matrix_vector_c( matrix, from_vec );

  set_point_x( lookfrom, new_from[0] );
  set_point_y( lookfrom, new_from[1] );
  set_point_z( lookfrom, new_from[2] );

  free( (char *)new_from );
  free( (char *)matrix );
}

static void rotate_clk( angle )
float angle;
/* This routine rotates the camera clockwise by 'angle' degrees. */
{
  float *matrix, *new_up, up_vec[3];

  ger_debug("rotate_clk: angle= %f",angle);

  up_vec[0]= vector_x(up);
  up_vec[1]= vector_y(up);
  up_vec[2]= vector_z(up);

  matrix= make_rotate_c( -angle, point_x(lookfrom), point_y(lookfrom),
			point_z(lookfrom) );
  new_up= matrix_vector_c( matrix, up_vec );

  set_vector_x( up, new_up[0] );
  set_vector_y( up, new_up[1] );
  set_vector_z( up, new_up[2] );

  free( (char *)new_up );
  free( (char *)matrix );
}

static void translate_in( fraction )
float fraction;
/* This routine translates the camera in by 'distance' units. */
{

  ger_debug("translate_in: fraction= %f",fraction);

  set_point_x( lookfrom, (1.0-fraction)*point_x(lookfrom) );
  set_point_y( lookfrom, (1.0-fraction)*point_y(lookfrom) );
  set_point_z( lookfrom, (1.0-fraction)*point_z(lookfrom) );
}

static void fpoint_up( dist )
float dist;
/* This routine translates the focal point up by 'distance' units. */
{
  ger_debug("fpoint_y: distance= %f",dist);

  set_point_x( lookatpt, (point_x(lookatpt) + (dist * vector_x(up))) ); 
  set_point_y( lookatpt, (point_y(lookatpt) + (dist * vector_y(up))) );
  set_point_z( lookatpt, (point_z(lookatpt) + (dist * vector_z(up))) );
}

static void fpoint_right( dist )
float dist;
/* This routine translates the focal point right by 'distance' units. */
{
  float vx, vy, vz, cross_x, cross_y, cross_z, len;

  ger_debug("fpoint_x: distance= %f",dist);
 
  vx= point_x(lookfrom) - point_x(lookatpt);
  vy= point_y(lookfrom) - point_y(lookatpt);
  vz= point_z(lookfrom) - point_z(lookatpt);

  cross_x= (vector_y(up) * vz) - (vector_z(up) * vy);
  cross_y= (vector_z(up) * vx) - (vector_x(up) * vz);
  cross_z= (vector_x(up) * vy) - (vector_y(up) * vx);
 
  len= sqrt( cross_x*cross_x + cross_y*cross_y + cross_z+cross_z );

  cross_x= cross_x / len;
  cross_y= cross_y / len;
  cross_z= cross_z / len;

  set_point_x( lookatpt, (point_x(lookatpt) + (dist * cross_x)) );
  set_point_y( lookatpt, (point_y(lookatpt) + (dist * cross_y)) );
  set_point_z( lookatpt, (point_z(lookatpt) + (dist * cross_z)) );
}

static void fpoint_in( dist )
float dist;
/* This routine translates the focal point up by 'distance' units. */
{
  float vx, vy, vz, len;

  ger_debug("fpoint_z: distance= %f",dist);

  vx= point_x(lookfrom) - point_x(lookatpt);
  vy= point_y(lookfrom) - point_y(lookatpt);
  vz= point_z(lookfrom) - point_z(lookatpt);
 
  len= sqrt(vx*vx + vy*vy + vz*vz);

  vx= vx / len;
  vy= vy / len;
  vz= vz / len;

  set_point_x( lookatpt, (point_x(lookatpt) + (dist * vx)) );
  set_point_y( lookatpt, (point_y(lookatpt) + (dist * vy)) );
  set_point_z( lookatpt, (point_z(lookatpt) + (dist * vz)) );
}
 
static Transformation recenter_model(oldtrans)
Transformation oldtrans;
/* 
This routine retranslates the lookatpt point to the origin.  It modifies
the camera information accordingly, and returns a transformation to move
the model and lights appropriately.
*/
{
  float *center;
  Transformation change, centering_trans;

  ger_debug("recenter_model:");

  center= make_translate_c( -point_x(lookatpt), -point_y(lookatpt), 
	-point_z(lookatpt) );
  change= array2d_to_lisp( center );
  grab(change); /* grab followed by free assures garbage collection */
  centering_trans= matrix_mult( change, oldtrans );
  free_transformation( oldtrans );
  free_transformation( change );
  grab(centering_trans);
  free( (char *)center );

  set_point_x( lookfrom, point_x(lookfrom) - point_x(lookatpt) );
  set_point_y( lookfrom, point_y(lookfrom) - point_y(lookatpt) );
  set_point_z( lookfrom, point_z(lookfrom) - point_z(lookatpt) );
  set_point_x( lookatpt,0.0 );
  set_point_y( lookatpt,0.0 );
  set_point_z( lookatpt,0.0 );

  return( centering_trans );
}

static Transformation center_model()
/* 
This routine translates the lookatpt point to the origin.  It modifies
the camera information accordingly, and returns a transformation to move
the model and lights appropriately.
*/
{
  float *center;
  Transformation centering_trans;

  ger_debug("center_model:");

  center= make_translate_c( -point_x(lookatpt), -point_y(lookatpt), 
	-point_z(lookatpt) );
  centering_trans= array2d_to_lisp( center );
  grab(centering_trans);
  free( (char *)center );

  set_point_x( lookfrom, point_x(lookfrom) - point_x(lookatpt) );
  set_point_y( lookfrom, point_y(lookfrom) - point_y(lookatpt) );
  set_point_z( lookfrom, point_z(lookfrom) - point_z(lookatpt) );
  set_point_x( lookatpt,0.0 );
  set_point_y( lookatpt,0.0 );
  set_point_z( lookatpt,0.0 );

  return( centering_trans );
}

static void report()
/* This routine enters the current info in the trace file   */
{
  static int stepcount= 0;
  float fromx, fromy, fromz, tox, toy, toz;

  if (trace_on) {
    fromx= point_x(lookfrom) + point_x(orig_lookat);
    fromy= point_y(lookfrom) + point_y(orig_lookat);
    fromz= point_z(lookfrom) + point_z(orig_lookat);

    tox = point_x(orig_lookat);
    toy = point_y(orig_lookat);
    toz = point_z(orig_lookat);

    fprintf(tracefile,
	    "----------------------------------------------------\n");
    fprintf(tracefile,"%d\n",stepcount++);
    fprintf(tracefile,"lookfrom= %f %f %f;", fromx, fromy, fromz);
    fprintf(tracefile," lookat= %f %f %f\n", tox, toy, toz);
    fprintf(tracefile,"up= %f %f %f\n", point_x(up), point_y(up), point_z(up));
  }
}

static void start_trace()
/* This routine indicates the recording of a trace file */
{
  ger_debug("start_trace: tracing enabled");

  /* Open the trace file */
  tracefile= fopen(TRACE_FILE_NAME,"w");
  if (!tracefile) ger_error("Unable to open trace file %s!", TRACE_FILE_NAME);

  trace_on= 1;

  fprintf(tracefile,"fovea= %f, hither= %f, yon=%f\n", fovea, hither, yon);
  fprintf(stderr,"Trace begun.\n");
}

static void ask_and_get(string,pval)
char *string;
float *pval;
/* This routine interactively requests an input value (a float). */
{
  int num_got= 0;
  char input_buf[81];

  while (num_got != 1) {
    fprintf(stderr,string);
    fgets(input_buf,81,stdin);
    num_got= sscanf(input_buf,"%f",pval);
    if (num_got != 1) {
      clearerr(stdin);
      fprintf(stderr,"Couldn't parse that;  please try again.\n");
    }
  }
}

static void load_position()
/* 
This routine interactively loads position information.
*/
{
  char junk[81];
  
  float x, y, z, upx, upy, upz;

  ger_debug("load_position: loading position information");

  /* Get rid of possible junk record in stdin */
  fgets(junk,81,stdin);

  fprintf(stderr,
    "Ready to load position and motion parameters, as from a trace file\n");

  ask_and_get("x lookfrom coordinate: ",&x);
  ask_and_get("y lookfrom coordinate: ",&y);
  ask_and_get("z lookfrom coordinate: ",&z);

  set_point_x( lookfrom, x );
  set_point_y( lookfrom, y );
  set_point_z( lookfrom, z );

  ask_and_get("x up component: ",&upx);
  ask_and_get("y up component: ",&upy);
  ask_and_get("z up component: ",&upz);

  set_point_x( up, upx );
  set_point_y( up, upy );
  set_point_z( up, upz );

  fprintf(stderr,"Position and motion parameters loaded.\n");
}

main(argc, argv)
int argc;
char *argv[];
{
	alisp(argc, argv);
}

void ui_setup( renderer, device, outfile, hints, controller, startframe, 
	      framecopies )
char *renderer, *device, *controller, *outfile;
int startframe, framecopies;
Attribute_list hints;
/* This routine initializes everything. */
{
  ger_debug(
	    "ui_setup: initializing renderer %s, device %s and controller %s",
	    renderer, device, controller);
  
  /* Initialize the controller and renderer */
  if (!initialized) {
    ctrl_setup( controller, startframe, framecopies );
    ren_setup( renderer, device, 1, outfile, hints );
    lookatpt= create_point();
    lookfrom= create_point();
    orig_lookat= create_point();
    up= create_vector();
    background= create_color();
    initialized= 1;
  }
  else {
    ger_error(
	   "ui_setup: called twice, this call interpreted as a hard reset.");
    ui_reset( 1, startframe, framecopies );
  }
}

void ui_reset( hard, startframe, framecopies )
int hard, startframe, framecopies;
/* This routine resets everything */
{
        ger_debug("ui_reset: resetting renderer and controller;  hard= %d",
		  hard);

	/* hard resets require recreation of lisp-side variables */
	if (hard) {
	  lookatpt= create_point();
	  lookfrom= create_point();
	  orig_lookat= create_point();
	  up= create_vector();	  
	  background= create_color();
	  lightgob= (Gob)0;
	  lightattr= (Attribute_list)0;;
	  initialized= 1;
	}

	ast_text_reset( hard );
	ast_prim_reset( hard );
	ast_spln_reset( hard );
	ctrl_reset( hard, startframe, framecopies );
	ren_reset( hard );
}

void ui_shutdown()
/* This routine shuts everything down */
{
      ger_debug("ui_shutdown: shutting everything down.\n");

      ren_shutdown();
      ctrl_end();
}

void ui_camera(cam_lookat, cam_lookfrom, cam_up, cam_fovea, cam_hither, 
	       cam_yon, cam_background)
Point cam_lookat, cam_lookfrom;
Vector cam_up;
float cam_fovea, cam_hither, cam_yon;
Color cam_background;
/* This routine handles camera setting */
{
  ger_debug("ui_camera: fovea= %f, hither= %f, yon= %f", cam_fovea,
	    cam_hither, cam_yon);

  /* Save camera values */
  set_point_x( lookatpt, point_x(cam_lookat) );
  set_point_y( lookatpt, point_y(cam_lookat) );
  set_point_z( lookatpt, point_z(cam_lookat) );
  set_point_x( orig_lookat, point_x(cam_lookat) );
  set_point_y( orig_lookat, point_y(cam_lookat) );
  set_point_z( orig_lookat, point_z(cam_lookat) );
  set_point_x( lookfrom, point_x(cam_lookfrom) );
  set_point_y( lookfrom, point_y(cam_lookfrom) );
  set_point_z( lookfrom, point_z(cam_lookfrom) );
  set_vector_x( up, vector_x(cam_up) );
  set_vector_y( up, vector_y(cam_up) );
  set_vector_z( up, vector_z(cam_up) );
  fovea= cam_fovea;
  hither= cam_hither;
  yon= cam_yon;
  set_color_red( background, color_red(cam_background) );
  set_color_green( background, color_green(cam_background) );
  set_color_blue( background, color_blue(cam_background) );
  set_color_alpha( background, color_alpha(cam_background) );

  /* remove parallel component of up vector */
  align_up();

  /* Set camera */
  ren_camera( cam_lookat, cam_lookfrom, cam_up, cam_fovea, 
	     cam_hither, cam_yon, cam_background );
}

void ui_render(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* 
This routine handles actual rendering of the gob. It implements the loop
that watches for keyboard input.
*/
{
  int multiplier= 0, redraw= 0, recenter= 0, thischar;
  Transformation centering_trans;
  float dist, xdiff, ydiff, zdiff;

  ger_debug("ui_render:");

  ren_render( thisgob, thistrans, thisattrlist );

  /* Calculate the distance from the lookatpt to the lookfrom point */
  xdiff = point_x(lookfrom) - point_x(lookatpt);
  ydiff = point_y(lookfrom) - point_x(lookatpt);
  zdiff = point_z(lookfrom) - point_z(lookatpt);
  dist = sqrt( (xdiff * xdiff) + (ydiff * ydiff) + (zdiff * zdiff) );

  /* Calculate and implement centering transformation */
  centering_trans= center_model();
  ren_traverselights( lightgob, centering_trans, lightattr );      

  /* The following loop handles user interaction */
  align_up(); /* align up vector */
  fputs(entry_message,stderr);
  while (!feof(stdin)) {
    thischar= getchar();
    if (isdigit(thischar)) multiplier= 10*multiplier + (thischar - 48);
    else {
      if (multiplier==0) multiplier= 1;
      switch (thischar) {
      case (int)'h':
      case (int)'?':
	fputs(help_message,stderr); break; /* 'help' */
      case (int)'u': /* 'up' */
	rotate_up( multiplier*default_angle ); redraw= 1; break;
      case (int)'d': /* 'down' */
	rotate_up( -multiplier*default_angle ); redraw= 1; break;
      case (int)'r': /* 'right' */
	rotate_right( multiplier*default_angle ); redraw= 1; break;
      case (int)'l': /* 'left' */
	rotate_right( -multiplier*default_angle ); redraw= 1; break;
      case (int)'c': /* 'clockwise' */
	rotate_clk( multiplier*default_angle ); redraw= 1; break;
      case (int)'a': /* 'anticlockwise' */
	rotate_clk( -multiplier*default_angle ); redraw= 1; break;
      case (int)'i': /* 'in' */
	translate_in( multiplier*default_fraction ); redraw= 1; break;
      case (int)'o': /* 'out' */
	translate_in( -multiplier*default_fraction ); redraw= 1; break;
      case (int)'U': /* 'focal point up' */
        fpoint_up( multiplier*default_fraction*dist ); redraw= 1; break;
      case (int)'D': /* 'focal point down' */
        fpoint_up( -multiplier*default_fraction*dist ); 
	recenter= 1; redraw= 1; break;
      case (int)'L': /* 'focal point left' */
        fpoint_right( -multiplier*default_fraction*dist ); 
	recenter= 1; redraw= 1; break;
      case (int)'R': /* 'focal point right' */
        fpoint_right( multiplier*default_fraction*dist ); 
	recenter= 1; redraw= 1; break;
      case (int)'I': /* 'focal point in' */
        fpoint_in( multiplier*default_fraction*dist );
	recenter= 1; redraw= 1; break;
      case (int)'O': /* 'focal point out' */
        fpoint_in( -multiplier*default_fraction*dist ); 
	recenter= 1; redraw= 1; break;
      case (int)'.': /* 'redraw current model' */
	redraw= 1; break;
      case (int)'t': /* 'start trace' */
        start_trace(); break;
      case (int)'p': /* 'load position' */
        load_position(); redraw=1; break;
      case (int)'q': return; /* 'quit' */
      case (int)'e': ui_shutdown(); exit(0); /* 'exit' */
      }

      if (recenter) {
	centering_trans= recenter_model(centering_trans);
	ren_traverselights( lightgob, centering_trans, lightattr );      
      }

      if (redraw) {
        align_up(); /* align up vector */
        ren_camera( lookatpt, lookfrom, up, fovea, hither, yon, background );
        ren_render( thisgob, centering_trans, thisattrlist );
        report();
      }
      multiplier= 0;
      recenter= 0;
      redraw= 0;
    }
  };

  free_transformation(centering_trans);

  /* The controller's doframe never gets called in this user interface */
}

void ui_traverselights(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* This routine handles actual rendering of the gob. */
{
  ger_debug("ui_traverselights");

  if ( lightgob && !null(lightgob) ) ungrab(lightgob);
  grab( lightgob= thisgob );
  if ( lightattr && !null(lightattr) ) ungrab(lightattr);
  grab( lightattr= thisattrlist );
  ren_traverselights( thisgob, thistrans, thisattrlist );
}

void ui_free(thisgob)
Gob thisgob;
/* This routine causes the renderer to free storage associated with the gob. */
{
  ger_debug("ui_free:");

  ren_free(thisgob);
}


