/****************************************************************************
 * fui.c
 * Author Joel Welling
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/

/*
This module provides a 'flying user interface' which lets the user fly around
and examine models with only a keyboard.  A trace file called fui_trace.dat
is maintained;  it contains needed data to continue a flight or construct
a series of P3D 'snap' commands from the points of the flight path.
*/

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "alisp.h"
#include "p3d.h"
#include "ge_error.h"
#include "matrix_ops.h"
#include "control.h"
#include "ren.h"
#include "ui.h"
#include "assist.h"

/* 
Default angle to rotate plane (degrees) and default acceleration
*/
#define default_angle 1.0
#define default_accel 1.0

/*
Dynamical constants:
	initial_velocity_multiplier: initial vel= this * view distance
	delta_t: time step (a time)
	lift_scale: linearly increases lift
	torque_scale: linearly increases torque
	vel_torque_scale: linearly increases torque due to velocity above
		mach_1 (used to stabilize the plane against pitch)
	mach_1: used to scale speeds; a velocity
	rot_damp: damping constant for rotations of plane; plane loses
		this fraction of angular velocity every time step (used
		to stabilize against yaw)
	gravity: acceleration of gravity
*/
#define initial_velocity_multiplier 0.020661157
#define delta_t 1.0
#define lift_scale 1.0
#define thrust_scale 1.0
#define torque_scale 1.0
#define vel_torque_scale 0.003
#define mach_1 300.0
#define rot_damp 0.75
#define gravity 50.0

/* 
   Dynamical variables of 'plane' - it will initially be placed at
   the lookfrom point heading toward the lookat point.
*/
static float x, y, z;     /* position */
static float px, py, pz;  /* nose pointing direction */
static float vx, vy, vz;  /* velocity components */
static float ax, ay, az;  /* acceleration components */
static float upx, upy, upz; /* components of up vector */
static float wx, wy, wz;  /* components of angular velocity */
static float length_scale; /* overall length scale of geometry (a distance) */
static float thrust;  /* magnitude of thrust */
static float lift;  /* lift magnitude */
static float drag;  /* drag magnitude */
static float elev_torque_scale; /* torque about pitch axis from elevators */
static float gravx, gravy, gravz; /* gravity components */

/*
Name of file to hold flight path trace, file pointer for file which holds 
flight path trace, and flag to indicate that tracing is enabled. 
*/
#define TRACE_FILE_NAME "fui_trace.dat"
static FILE *tracefile;
static int trace_on= 0;

/* Help and entry messages */
static char entry_message[]= "Ready for commands (type 'h' for help)\n";
static char help_message[]= "\
The following character inputs are accepted:\n\
\n\
  u	nose up\n\
  d	nose down\n\
  l	roll left (and start turning)\n\
  r	roll right (and start turning)\n\
  a	accelerate (increase thrust)\n\
  b     brake (reduce thrust)\n\
  space continue along current path\n\
  T     start saving a trace file of the flight\n\
  L     load position information\n\
  q	quit flying and go on to next model\n\
  e	exit from the program\n\
  h,?	print this message\n\
\n\
Any letter can be prefixed by an integer, which then multiplies \n\
the amount moved or rotated.\n\
"; /* End of help message */

/* Current camera data*/
static Point lookfrom, lookat;
static Vector up;
static float fovea, hither, yon;
static Color background;

/* Current light data */
static Gob lightgob;
static Attribute_list lightattr;

/* Variable to hold initialization state */
static int initialized= 0;

static void align_up()
/* 
This routine removes components of the up vector which are parallel to
the viewing direction, and renormalizes the up vector.
*/
{
  float unorm, dot;

  ger_debug("align_up:");

  dot= upx*px + upy*py + upz*pz;

  upx= upx - dot * px;
  upy= upy - dot * py;
  upz= upz - dot * pz;
  unorm= upx*upx + upy*upy + upz*upz;
  if (unorm==0.0) {
    ger_error("align_up: up vector vanishes, using z direction");
    upx= 0.0; upy= 0.0; upz= 1.0; unorm= 1.0;
  }

  unorm= sqrt(unorm);
  upx= upx/unorm;
  upy= upy/unorm;
  upz= upz/unorm;

  set_vector_x( up, upx );
  set_vector_y( up, upy );
  set_vector_z( up, upz );

  return;
}

static void nose_up( angle )
float angle;
/* This routine rotates the plane's nose up by 'angle' degrees. */
{
  float radians_per_degree= 3.14159265/180.0;

  ger_debug("nose_up: angle= %f",angle);

  angle= radians_per_degree*angle;
  elev_torque_scale= -2.0*torque_scale*cos(angle)*sin(angle/2.0);
}

static void roll_right( angle )
float angle;
/* This routine rolls the plane right by 'angle' degrees. */
{
  float *matrix, *new_up, up_vec[3];

  ger_debug("roll_right: angle= %f",angle);

  up_vec[0]= upx;
  up_vec[1]= upy;
  up_vec[2]= upz;

  matrix= make_rotate_c( angle, px, py, pz );
  new_up= matrix_vector_c( matrix, up_vec );

  upx= new_up[0];
  upy= new_up[1];
  upz= new_up[2];

  free( (char *)new_up );
  free( (char *)matrix );
}

static void accelerate( accel )
float accel;
/* This routine accelerates the plane. */
{
  ger_debug("accelerate: accel= %f",accel);

  thrust += accel*thrust_scale*length_scale;
}

static void setup_dynamics()
/* This routine sets dynamical variables appropriately */
{
  float vmag;

  upx= point_x(up);
  upy= point_y(up);
  upz= point_z(up);
  
  /* Set the position and velocity information for the plane appropriately */  
  x= point_x(lookfrom);
  y= point_y(lookfrom);
  z= point_z(lookfrom);
  vx= initial_velocity_multiplier * ( point_x(lookat) - x );
  vy= initial_velocity_multiplier * ( point_y(lookat) - y );
  vz= initial_velocity_multiplier * ( point_z(lookat) - z );
  vmag= sqrt( vx*vx + vy*vy + vz*vz );
  px= vx/vmag;
  py= vy/vmag;
  pz= vz/vmag;
  length_scale= vmag*delta_t/mach_1;

  wx= 0.0;
  wy= 0.0;
  wz= 0.0;

  gravx= 0.0;
  gravy= 0.0;
  gravz= -length_scale*gravity;

  thrust= thrust_scale*length_scale;
  lift= 0.0;
  drag= 0.0;

  elev_torque_scale= 0.0;

  /* remove parallel component of up vector */
  align_up();
  ger_debug("setup_dynamics: length_scale= %f",length_scale);
}

static void step_position()
/* 
This routine integrates the plane's position and velocity through one time step 
*/
{
  float vmag, pmag, umag;
  float vel_torque; /* small stabilizing torque */
  float elev_torque; /* torque due to elevators */
  float pitchx, pitchy, pitchz; /* components of the pitch axis */
  float tx, ty, tz; /* components of torque */

  ger_debug("step_position:");

  /* Calculate the pitch axis */
  pitchx= py*upz - pz*upy;
  pitchy= pz*upx - px*upz;
  pitchz= px*upy - py*upx;

  /* Do the force calculations */
  vmag= sqrt( vx*vx + vy*vy + vz*vz );
  lift= lift_scale * length_scale * gravity * (vmag/mach_1);
  drag= thrust_scale*length_scale/mach_1;
  vel_torque= vel_torque_scale*(vmag-mach_1);
  elev_torque= elev_torque_scale*(vmag/mach_1)*(vmag/mach_1);

  /* Sum forces */
  ax= thrust*px + lift*upx - drag*vx + gravx;
  ay= thrust*py + lift*upy - drag*vy + gravy;
  az= thrust*pz + lift*upz - drag*vz + gravz;

  /* Calculate torque due to difference between vel. and pointing vec. */
  tx= torque_scale/mach_1 * (vmag/mach_1)
    * (py*(vz - vmag*pz) - pz*(vy - vmag*py)) 
    + (elev_torque+vel_torque)*pitchx;
  ty= torque_scale/mach_1 * (vmag/mach_1)
    * (pz*(vx - vmag*px) - px*(vz - vmag*pz))
    + (elev_torque+vel_torque)*pitchy;
  tz= torque_scale/mach_1 * (vmag/mach_1)
    * (px*(vy - vmag*py) - py*(vx - vmag*px))
    + (elev_torque+vel_torque)*pitchz;

  /* Integrate velocity and position */
  vx= vx + ax*delta_t;
  vy= vy + ay*delta_t;
  vz= vz + az*delta_t;
  x= x + vx*delta_t;
  y= y + vy*delta_t;
  z= z + vz*delta_t;

  /* Integrate angular velocity and position */
  wx= wx + tx*delta_t - rot_damp*wx;
  wy= wy + ty*delta_t - rot_damp*wy;
  wz= wz + tz*delta_t - rot_damp*wz;
  px= px + (wy*pz - wz*py)*delta_t;
  py= py + (wz*px - wx*pz)*delta_t;
  pz= pz + (wx*py - wy*px)*delta_t;
  upx= upx + (wy*upz - wz*upy)*delta_t;
  upy= upy + (wz*upx - wx*upz)*delta_t;
  upz= upz + (wx*upy - wy*upx)*delta_t;

  /* Load the new viewing info into the camera */
  set_point_x(lookfrom, x);
  set_point_y(lookfrom, y);
  set_point_z(lookfrom, z);
  vmag= sqrt( vx*vx + vy*vy + vz*vz );
  if (vmag != 0.0) {
    set_point_x(lookat, x + vx/vmag);
    set_point_y(lookat, y + vy/vmag);
    set_point_z(lookat, z + vz/vmag);
  }

  /* Make sure the pointing vector remains normalized */
  pmag= sqrt( px*px + py*py + pz*pz );
  px= px/pmag;
  py= py/pmag;
  pz= pz/pmag;

  /* Make sure the up vector remains normalized */
  umag= sqrt( upx*upx + upy*upy + upz*upz );
  upx= upx/umag;
  upy= upy/umag;
  upz= upz/umag;
}

static void report()
/* 
This routine prints relevant data and enters the current info in
the trace file.
*/
{
  float vmag, pitch, roll;
  static float degrees_per_radian= 180.0/3.14159265;
  static float eps= 0.0001;
  static int stepcount= 0;

  vmag= sqrt( vx*vx + vy*vy + vz*vz );
  pitch= degrees_per_radian *
    asin( (vx/vmag-px)*upx + (vy/vmag-py)*upy + (vz/vmag-pz)*upz );
  if (abs(pz-1.0)<eps) roll= 0.0;
  else roll= degrees_per_radian*acos( upz/sqrt(1.0-pz*pz) );
  if ( (px*upy-py*upx)>0.0 ) roll= -roll;
  fprintf(stderr,"thrust %f, lift %f, drag %f, gravity %f\n",
	  thrust,lift,drag*vmag,gravz);
  fprintf(stderr,"vel= %f, pitch= %f, roll= %f\n",vmag, pitch, roll);
  fprintf(stderr,"position= %f %f %f\n",x,y,z);

  if (trace_on) {
    fprintf(tracefile,
	    "----------------------------------------------------\n");
    fprintf(tracefile,"%d\n",stepcount++);
    fprintf(tracefile,"lookfrom= %f %f %f; lookat= %f %f %f\n",
	    x,y,z,point_x(lookat),point_y(lookat),point_z(lookat));
    fprintf(tracefile,"point= %f %f %f; up= %f %f %f\n",px,py,pz,upx,upy,upz);
    fprintf(tracefile,"vel= %f %f %f; acc= %f %f %f\n",vx,vy,vz,ax,ay,az);
    fprintf(tracefile,"omega= %f %f %f\n",wx,wy,wz);
    fprintf(tracefile,"length_scale= %f\n",
	    length_scale);
    fprintf(tracefile,"thrust= %f, gravz= %f\n",
	    thrust,gravz);
  }
}

static void start_trace()
/* This routine initiates the recording of a trace file of the flight. */
{
  ger_debug("start_trace: flight tracing enabled");

  /* Open the flight path trace file */
  tracefile= fopen(TRACE_FILE_NAME,"w");
  if (!tracefile) ger_error("Unable to open trace file %s!",TRACE_FILE_NAME);

  trace_on= 1;

  fprintf(tracefile,"fovea= %f, hither= %f, yon= %f\n",fovea,hither, yon);
  fprintf(stderr,"Trace begun.\n");
}

static void ask_and_get(string,pval)
char *string;
float *pval;
/* This routine interactively requests an input value (a float). */
{
  int num_got= 0;
  char input_buf[81];

  while (num_got != 1) {
    fprintf(stderr,string);
    fgets(input_buf,81,stdin);
    num_got= sscanf(input_buf,"%f",pval);
    if (num_got != 1) {
      clearerr(stdin);
      fprintf(stderr,"Couldn't parse that;  please try again.\n");
    }
  }
}

static void load_position()
/* 
This routine interactively loads plane position and velocity information,
allowing the plane to be instantaneously moved.
*/
{
  char junk[81];

  ger_debug("load_position: loading position information");

  /* Get rid of possible junk record in stdin */
  fgets(junk,81,stdin);

  fprintf(stderr,
    "Ready to load position and motion parameters, as from a trace file\n");

  ask_and_get("length scale: ",&length_scale);
  ask_and_get("z component of gravity: ",&gravz);

  ask_and_get("x lookfrom coordinate: ",&x);
  ask_and_get("y lookfrom coordinate: ",&y);
  ask_and_get("z lookfrom coordinate: ",&z);

  ask_and_get("x pointing component: ",&px);
  ask_and_get("y pointing component: ",&py);
  ask_and_get("z pointing component: ",&pz);

  ask_and_get("x up component: ",&upx);
  ask_and_get("y up component: ",&upy);
  ask_and_get("z up component: ",&upz);

  ask_and_get("x velocity component: ",&vx);
  ask_and_get("y velocity component: ",&vy);
  ask_and_get("z velocity component: ",&vz);

  ask_and_get("x acceleration component: ",&ax);
  ask_and_get("y acceleration component: ",&ay);
  ask_and_get("z acceleration component: ",&az);

  ask_and_get("x angular vel (omega) component: ",&wx);
  ask_and_get("y angular vel (omega) component: ",&wy);
  ask_and_get("z angular vel (omega) component: ",&wz);

  ask_and_get("thrust: ",&thrust);

  fprintf(stderr,"Position and motion parameters loaded.\n");
}

main(argc, argv)
int argc;
char *argv[];
{
	alisp(argc, argv);
}

void ui_setup( renderer, device, outfile, hints, controller, startframe, 
	      framecopies )
char *renderer, *device, *controller, *outfile;
int startframe, framecopies;
Attribute_list hints;
/* This routine initializes everything. */
{
  ger_debug(
	    "ui_setup: initializing renderer %s, device %s and controller %s",
	    renderer, device, controller);
  
  /* Initialize the controller and renderer */
  if (!initialized) {
    ctrl_setup( controller, startframe, framecopies );
    ren_setup( renderer, device, 1, outfile, hints );
    lookat= create_point();
    lookfrom= create_point();
    up= create_vector();
    background= create_color();
    initialized= 1;
  }
  else {
    ger_debug(
	   "ui_setup: called twice, this call interpreted as a hard reset.");
    ui_reset( 1, startframe, framecopies );
  }
}

void ui_reset( hard, startframe, framecopies )
int hard, startframe, framecopies;
/* This routine resets everything */
{
        ger_debug("ui_reset: resetting renderer and controller;  hard= %d",
		  hard);

	/* hard resets require recreation of lisp-side variables */
	if (hard) {
	  lookat= create_point();
	  lookfrom= create_point();
	  up= create_vector();	  
	  background= create_color();
	  lightgob= (Gob)0;
	  lightattr= (Attribute_list)0;;
	  initialized= 1;
	}

	ast_text_reset( hard );
	ast_prim_reset( hard );
	ast_spln_reset( hard );
	ctrl_reset( hard, startframe, framecopies );
	ren_reset( hard );
}

void ui_shutdown()
/* This routine shuts everything down */
{
      ger_debug("ui_shutdown: shutting everything down.\n");

      ren_shutdown();
      ctrl_end();
}

void ui_camera(cam_lookat, cam_lookfrom, cam_up, cam_fovea, cam_hither, 
	       cam_yon, cam_background)
Point cam_lookat, cam_lookfrom;
Vector cam_up;
float cam_fovea, cam_hither, cam_yon;
Color cam_background;
/* This routine handles camera setting */
{
  ger_debug("ui_camera: fovea= %f, hither= %f, yon= %f", cam_fovea,
	    cam_hither, cam_yon);

  /* Save camera values */
  set_point_x( lookat, point_x(cam_lookat) );
  set_point_y( lookat, point_y(cam_lookat) );
  set_point_z( lookat, point_z(cam_lookat) );
  set_point_x( lookfrom, point_x(cam_lookfrom) );
  set_point_y( lookfrom, point_y(cam_lookfrom) );
  set_point_z( lookfrom, point_z(cam_lookfrom) );
  set_vector_x( up, vector_x(cam_up) );
  set_vector_y( up, vector_y(cam_up) );
  set_vector_z( up, vector_z(cam_up) );
  fovea= cam_fovea;
  hither= cam_hither;
  yon= cam_yon;
  set_color_red( background, color_red(cam_background) );
  set_color_green( background, color_green(cam_background) );
  set_color_blue( background, color_blue(cam_background) );
  set_color_alpha( background, color_alpha(cam_background) );

  /* Set all dynamical variables appropriately */
  setup_dynamics();

  /* Set camera */
  ren_camera( cam_lookat, cam_lookfrom, cam_up, cam_fovea, 
	     cam_hither, cam_yon, cam_background );
}

void ui_render(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* 
This routine handles actual rendering of the gob. It implements the loop
that watches for keyboard input.
*/
{
  int multiplier= 0, mult_set= 0, redraw= 0, thischar;

  ger_debug("ui_render:");

  ren_render( thisgob, thistrans, thisattrlist );

  /* The following loop handles user interaction */
  align_up(); /* align up vector */
  fputs(entry_message,stderr);
  while (!feof(stdin)) {
    thischar= getchar();
    if (isdigit(thischar)) {
      multiplier= 10*multiplier + (thischar-48);
      mult_set= 1;
    }
    else {
      if (!mult_set) multiplier= 1;
      switch (thischar) {
      case (int)'h': 
      case (int)'?': 
	fputs(help_message,stderr); break; /* 'help' */
      case (int)'u': /* 'up' */
	nose_up( multiplier*default_angle ); redraw= 1; break;
      case (int)'d': /* 'down' */
	nose_up( -multiplier*default_angle ); redraw= 1; break;
      case (int)'r': /* 'right' */
	roll_right( multiplier*default_angle ); redraw= 1; break; 
      case (int)'l': /* 'left' */
	roll_right( -multiplier*default_angle ); redraw= 1; break; 
      case (int)'a': /* 'accelerate' */
	accelerate( multiplier*default_accel ); redraw= 1; break;
      case (int)'b': /* 'brake' */
	accelerate( -multiplier*default_accel ); redraw= 1; break; 
      case (int)' ': /* continue along existing path */
	redraw= 1; break;
      case (int)'T': start_trace(); break;
      case (int)'L': load_position(); redraw= 1; break;
      case (int)'q': return; /* 'quit' */
      case (int)'e': ui_shutdown(); exit(0); /* 'exit' */
      }

      if (redraw) {
        step_position();
        align_up(); /* align up vector */
        ren_camera( lookat, lookfrom, up, fovea, hither, yon, background );
        ren_render( thisgob, thistrans, thisattrlist );
	report();
      }
      multiplier= 0;
      mult_set= 0;
      redraw= 0;
    }
  };

  /* The controller's doframe never gets called in this user interface */
}

void ui_traverselights(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* This routine handles actual rendering of the gob. */
{
  ger_debug("ui_traverselights");

  if ( lightgob && !null(lightgob) ) ungrab(lightgob);
  grab( lightgob= thisgob );
  if ( lightattr && !null(lightattr) ) ungrab(lightattr);
  grab( lightattr= thisattrlist );
  ren_traverselights( thisgob, thistrans, thisattrlist );
}

void ui_free(thisgob)
Gob thisgob;
/* This routine causes the renderer to free storage associated with the gob. */
{
  ger_debug("ui_free:");

  ren_free(thisgob);
}
