/****************************************************************************
 * env.c
 * Author Chris Nuuja
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/

      /*
      This file defines all the routines used to add definitions to the
      environment and to remove those definitions once they no longer are
      valid
      */

#include "alisp.h"
#include <stdio.h>
#include <string.h>


/*	
	These hash tables hold the unique representations of symbols, ints,
	and floats
*/
NODE *SYMBOLTABLE[PRIME];
NODE *FLOATTABLE = (NODE *)NULL;
NODE *INTEGERTABLE = (NODE *)NULL;
NODE *memory_start;
NODE *DECR_START;


/* 
    called by alisp, this sets up all the necessary buffers and default 
    symbols and values.
*/

void  setup_env()
{
	int i;
	long val;
        extern NODE *node_buff;
	
   /* 
    Initialize the memory record keepers
    <memory_record> is a global that points to the last block of memory alloced 
    <memory_start> points to the first block alloced
   */
    memory_record = (NODE *) malloc(sizeof(NODE));
    memory_start  = (NODE *) malloc(sizeof(NODE));
    Type_Of(memory_record) = N_LIST;
    Type_Of(memory_start)  = N_LIST;
    cdr(memory_start) = memory_record;
    node_buff = (NODE *)0;

    /*
     Create default values and constants
    */
    NIL = new_node(N_NIL);
    TRUE_NODE = new_node(N_TRUE);
    DECR_START  = new_node(N_LIST_R);
    /*
     Initialize the global environment and result flags to nil
    */
    EVAL_ENV = NIL;
    RESULT = NIL;


    /*
     Initialize the symbol, integer and (currently not used) float valued
     hash tables
    */
    for (i=0;i<PRIME;i++)
	SYMBOLTABLE[i] = NIL;

    LAMBDA_KEY = get_symbolrep("lambda",K_LAMBDA,7);
    INTEGERTABLE = (NODE *) malloc(32768*sizeof(NODE));
    FLOATTABLE = (NODE *) malloc(32768*sizeof(NODE));
    val = -16384;
    for (i=0;i<32768;i++)
	{
	getfixnum(INTEGERTABLE+i) = val;
	Type_Of(INTEGERTABLE+i) = N_INT;
	incr_ref(INTEGERTABLE+i);

	getflonum(FLOATTABLE+i) = (float) val++;
	Type_Of(FLOATTABLE+i) = N_REAL;
	incr_ref(FLOATTABLE+i);
	}
}


/*
    called at the end of a lisp session, this frees all the blocks that were
    allocated during the session
*/
void cleanup_env()
{
    extern NODE *node_buff;
    extern void clean_memory();
    /* 
     set clean flag at end of memory-record chain before cleaning out memory 
    */
    Type_Of(memory_record) = N_FREE;
    clean_memory(memory_start,0);
    node_buff = (NODE *)0;
    free(memory_start);
    free(INTEGERTABLE);
    free(FLOATTABLE);
}


/*
    If a symbol with print-name <name> has been added to the symbol table,
    this will find it, else return nil
*/
NODE *lookup_symbol(name)
char *name;
{
	NODE *symlist;
        int id;

	id = hash_string(name);
	for (symlist=SYMBOLTABLE[id];!null(symlist); symlist = cdr(symlist))
		{
		if ( strcmp( (char *)name, (char *)symbol_name(car(car(symlist))) ) == 0 )
			return(car(car(symlist)));
		}
	return(NIL);
}

/*
    given a symbol node <struct_name> (which is currently assumed to be the 
    name of a structure accessor created by defstruct) this returns its
    p-list.  currently the plist is a cons cell where the car holds the
    symbol and the cdr holds the integer index that represents where in
    an array that represents a structure the accessor <struct_name> looks.
    Currently, only the names of structure accessors have a useful plist.
*/
NODE *get_plist(struct_name)
NODE *struct_name;
{
	char *name;
	int id;
	NODE *symlist;

	name = symbol_name(struct_name);
	id = hash_string(name);
	for (symlist=SYMBOLTABLE[id];!null(symlist);symlist=cdr(symlist))
		{
		if (strcmp(name,symbol_name(car(car(symlist))))==0)
			return(car(symlist));
		}
	fprintf(stderr,"Error, no plist for %s\n",name);
	exit(0);
}

/*
    puts the appropriate integer index that represents where the 
    structure-accessor function <structure> would look in the array
    that represents a structure.  Only valid for structure accessors
*/
void add_setf_method(structure,place)
NODE *structure,*place;
{
	NODE *plist;

	plist = get_plist(structure);
	cdr(plist) = place;
}

/*
    gets the integer index into the array that represents the slot
    <name> is supposed to access in a structure.
*/
NODE *get_setf_method(name)
NODE *name;
{
	NODE *plist;

	plist = get_plist(name);
	return(cdr(plist));
}

/*
	given a floating point <value>, returns a NODE of type N_FLOAT with that
	value
*/
NODE *get_floatrep(value)
float value;
{
	NODE *newfloat;

	if ( (((float) ((int) value)) == value) &&
	     (value < 16384.0) && (value > -16385.0) )
	{
	   return(FLOATTABLE+( ((int)value) + 16384));
	}
    	newfloat = new_node(N_REAL);
	getflonum(newfloat) = value;

	return(newfloat);
}

/*
	given a integer <value>, returns a NODE of type N_INT with that
	value.  If it did not already exist in the int table, it adds it,
	else it returns the one from the table
*/
NODE *get_integerrep(value)
int value;
{
	NODE *newinteger;

	if ( (value < 16384) && (value > -16385) )
	{
	   return(INTEGERTABLE+(value + 16384));
	}

    	newinteger = new_node(N_INT);
	getfixnum(newinteger) = (long)value;
	return(newinteger);
}

/*
   Same as above, only for symbols.  <type> is K_NOOP unless this symbol
   matches a lisp keyword
*/
NODE *get_symbolrep(name,type,char_num)
char *name;
KEY_TYPE type;
int char_num;
{
	int id;
	NODE *symlist,*newsymbol;
	

	id = hash_string(name);
	newsymbol = lookup_symbol(name);
	if (!null(newsymbol))
		return(newsymbol);
	newsymbol = new_node(N_SYMBOL);
	symbol_type(newsymbol) = type;
	symbol_name(newsymbol) = Alloc(char, char_num+2);
	strncpy( symbol_name(newsymbol), name, char_num+1 );
	symlist = SYMBOLTABLE[id];
	symlist = cons(cons(newsymbol,NIL),symlist);
	incr_ref(symlist);
	decr_ref(cdr(symlist));
	SYMBOLTABLE[id] = symlist;
	return(newsymbol);
}

/*
    This function gets the non-keyword version of a keyword variable
    (i.e. if <name> is ":variable" then this returns the symbol with
    print-name "variable").  This is needed when making bindings of arguments
    to keyword variables
*/
NODE *get_nonkey(name)
char *name;
{
	NODE *result;
	
	name++;			      /*   remove the ':' character  */
	result = lookup_symbol(name);
	return(result);
}

	
/*
    Adds a binding to the environment
*/
void Bind(x,y)
NODE *x,*y;
{
	EVAL_ENV = cons( cons(x,y), EVAL_ENV);
	incr_ref(EVAL_ENV);
	decr_ref(cdr(EVAL_ENV));
}

/*
   Removes the top binding from the environment
*/
void unBind()
{
	NODE *old;
	old = EVAL_ENV;
	EVAL_ENV = cdr(EVAL_ENV);
	if (old->ptr_cnt != 1)
		{
		fprintf(stderr,"Bad Unbind %d \n",old->ptr_cnt);
		print_out(old);
		fprintf(stderr,"\n");
		}
	rplacd(old,NIL);
	decr_elem(old);
}


/* 
    This function returns the value bound to <var> in the global environment
    <EVAL_ENV> if <var> occurs in it, and NIL otherwise
*/
NODE *lookup(var)
NODE *var;
{
	NODE *result,*result_pair;

	result_pair = assoc(var,EVAL_ENV);
	if (null(result_pair)) 
		{
		fprintf(stderr,"ERROR, Unbound Variable: \n");
		print_out(var);
		fprintf(stderr,"\n");
		result = result_pair;
		}
	else
		result = cdr(result_pair);
	return(result);
}


/*
    destructively puts <list> at the top of the environment
*/
void add_Bindlist(list)
NODE *list;
{
	NODE *list_ptr;

	for (list_ptr=list;!null(cdr(list_ptr));list_ptr=cdr(list_ptr));

	rplacd(list_ptr,EVAL_ENV);
	EVAL_ENV = list;
}

/*
    Used by  eval_do when stepping the values of loop variables.  It finds
    where in the environment the list of variables beginning with 
    <first_var> starts, and replaces those variable's bindings with bindings
    from the list of <newvals>
*/
void add_vallist(first_var,newvals)
NODE *first_var,*newvals;
{
	NODE *env_ptr,*list_ptr;

	for (env_ptr=EVAL_ENV;car(car(env_ptr)) != first_var; 
	     env_ptr = cdr(env_ptr));
	for (list_ptr=newvals;!null(list_ptr);list_ptr=cdr(list_ptr))
		{
		decr_elem(cdr(car(env_ptr)));
		rplacd(car(env_ptr),car(list_ptr));
		incr_ref(cdr(car(env_ptr)));
		env_ptr = cdr(env_ptr);
		}
	
}

/*
   Removes the binding for the variable <var>
*/
void remove_Binding(var)
NODE *var;
{
	NODE *env_ptr,*prev_ptr;

	if (car(car(EVAL_ENV)) == var)  
		{
		unBind();
		}
	else
		{
		for (env_ptr=EVAL_ENV;car(car(env_ptr)) != var; env_ptr=cdr(env_ptr))
			prev_ptr = env_ptr;
		
		cdr(prev_ptr) = cdr(env_ptr);
		cdr(env_ptr) = NIL;
		decr_elem(env_ptr);
		}
}
	
/*
    Removes the list of <var_count> bindings beggining with <first_var> from
    the environment
*/
void remove_Bindings(first_var,var_count)
NODE *first_var;
int var_count;
{
	NODE *find_ptr,*holder,*prev_ptr;
	int index,new_Bindings=0;

	if (var_count == 0)	return;
	if (car(car(EVAL_ENV)) != first_var)
		{
		holder = EVAL_ENV;
		new_Bindings = 1;
		for (find_ptr=EVAL_ENV;car(car(find_ptr)) != first_var;
	   	   find_ptr=cdr(find_ptr))
			{
			prev_ptr = find_ptr;
	
			if (null(cdr(find_ptr)))
				{
				fprintf(stderr,"Never found it\n");
				break;
				}
			}
		rplacd(prev_ptr,NIL);
		EVAL_ENV = find_ptr;
		}

	for (index=var_count;index != 0;index--)
		unBind();

	if (new_Bindings)
		{
		add_Bindlist(holder);
		}

}

/*
   Unlinks the <list> of <lngth_list> from the environment without
   decr_elem'ing the list
*/
void unlink_Bindlist(list,lngth_list)
NODE *list;
int lngth_list;
{
	NODE *find_ptr,*holder,*prev_ptr;
	int index,new_Bindings=0;

	if (lngth_list == 0)	return;
	if (EVAL_ENV != list)
		{
		holder = EVAL_ENV;
		new_Bindings = 1;
		for (find_ptr=EVAL_ENV;find_ptr != list; find_ptr=cdr(find_ptr))
			{
			prev_ptr = find_ptr;
	
			if null(find_ptr)
				{
				fprintf(stderr,"Never found it 2\n");
				break;
				}
			}
		rplacd(prev_ptr,NIL);
		EVAL_ENV = find_ptr;
		}

	for (index=lngth_list; index != 1 ;index--)
		EVAL_ENV = cdr(EVAL_ENV);

	prev_ptr= EVAL_ENV;		/* the end of <list> */
	EVAL_ENV = cdr(EVAL_ENV);
	rplacd(prev_ptr,NIL);

	if (new_Bindings)
		{
		add_Bindlist(holder);
		}
}

