/****************************************************************************
 * color.c
 * Author Joel Welling
 * Copyright 1992, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/* 
This module implements colors as a user data type.
*/

#include <stdio.h>
#include "ge_error.h"
#include "alisp.h"
#include "p3d.h"
#include "ren.h"

/* Notes-
*/

#define DEFAULT_R 0.8
#define DEFAULT_G 0.8
#define DEFAULT_B 0.8
#define DEFAULT_A 0.8
#define INITIAL_COLORS 5000;

/* The USERDATA struct referenced by all colors */
static USERDATA colordef;

/* The list of free color data structures, and total number allocated. */
static Color_body *free_list= (Color_body *)0;
static int total_colors= 0;

static void color_print( data )
char *data;
/* This is the print method for the color */
{
  Color_body *clr;
  clr= (Color_body *)data;
  ger_debug("color_print");
  fprintf(stderr,"*color* ");
#ifdef never
  fprintf(stderr,"color: RGBA= (%f, %f, %f, %f)\n",
	  clr->d.r, clr->d.g, clr->d.b, clr->d.a);
#endif
}

static void color_destroy( data )
char *data;
/* This routine destroys a color instance */
{
  Color_body *thiscolordata;

  ger_debug("color_destroy");

  thiscolordata= (Color_body *)data;
  thiscolordata->next= free_list;
  free_list= thiscolordata;
}

static void color_setf( place, location, value )
NODE *place, *location, *value;
/* This routine provides setf methods for color 'structs'. */
{
  float val;
  char *fname, *cname;

  ger_debug("color_setf");

  if ( realp(value) ) val= get_number(value);
  else {
    ger_error("color_setf: argument value not int or real!");
    return;
  }

  fname= getname(place);
  cname= callout_name(place);

  if (!strcmp(fname,"color-r") || !strcmp(cname,"~color-r"))
    set_color_red((Color)location,val);
  else if (!strcmp(fname,"color-g") || !strcmp(cname,"~color-g"))
    set_color_green((Color)location,val);
  else if (!strcmp(fname,"color-b") || !strcmp(cname,"~color-b"))
    set_color_blue((Color)location,val);
  else if (!strcmp(fname,"color-a") || !strcmp(cname,"~color-a"))
    set_color_alpha((Color)location,val);
  else ger_error("color_setf: %s is not a setf-able field of a color.",
		 fname);
}

static void init_color()
/* This routine initializes this module and plugs it into Alisp. */
{
  ger_debug("init_color");
  colordef.type= "color";
  colordef.destroy= color_destroy;
  colordef.print= color_print;
  colordef.setf_method= color_setf;
}

static void allocate_colors()
/* This routine adds some new colors to the free list */
{
  int num_to_alloc;
  Color_body *new_block;
  int i;

  if (total_colors) num_to_alloc= total_colors/2;
  else num_to_alloc= INITIAL_COLORS;

  ger_debug("allocate_colors: current total %d; allocating %d more.",
	    total_colors, num_to_alloc);

  if ( !(new_block= (Color_body *)malloc(num_to_alloc*sizeof(Color_body))) )
    ger_fatal("allocate_colors: unable to allocate %d bytes!",
	      num_to_alloc*sizeof(Color_body));

  /* String them together and add them to the free list */
  new_block[0].next= free_list;
  for (i=1; i<num_to_alloc; i++) new_block[i].next= &(new_block[i-1]);
  free_list= &(new_block[num_to_alloc - 1]);

  total_colors += num_to_alloc;
}

static Color new_color()
/*
This function creates a color.  Its r, g, and b components are initially
set to 0.8, and it's a component is set to 1.0 .
*/
{
  Color_body *thiscolordata;
  Color thiscolor;
  static int initialized= 0;
  
  ger_debug("new_color:");

  if (!initialized) {
    init_color();
    initialized= 1;
  }

  /* Pop a cell off the free list */
  if (!free_list) allocate_colors();
  thiscolordata= free_list;
  free_list= free_list->next;

  thiscolordata->d.r= DEFAULT_R;
  thiscolordata->d.g= DEFAULT_G;
  thiscolordata->d.b= DEFAULT_B;
  thiscolordata->d.a= DEFAULT_A;
  
  /* Build the node */
  thiscolor= new_node(N_USERDATA);
  ((NODE *)thiscolor)->val.userdef.data= (char *)thiscolordata;
  ((NODE *)thiscolor)->val.userdef.methods= &colordef;

  return( thiscolor );
}

NODE *color_create( arglist )
NODE *arglist;
/* This routine creates a new color instance */
{
  Color thiscolor;
  Color_body *thiscolordata;

  ger_debug("color_create:");

  thiscolor= new_color();
  thiscolordata= (Color_body *)((NODE *)thiscolor)->val.userdef.data;

  /* Decode the NODE * here.  It is a list containing the data
   * entries in order.
   */
  if (length_list(arglist) < 4) {
    ger_error("color_create: need at least 4 arguments!");
    return( NIL );
  }

  if ( realp(safe_car(arglist)) )
    thiscolordata->d.r= get_number(safe_car(arglist));
  else {
    ger_error("color_create: argument for r not int or real!");
    thiscolordata->d.r= DEFAULT_R;
  }
  arglist= safe_cdr(arglist);

  if ( realp(safe_car(arglist)) )
    thiscolordata->d.g= get_number(safe_car(arglist));
  else {
    ger_error("color_create: argument for g not int or real!");
    thiscolordata->d.g= DEFAULT_G;
  }
  arglist= safe_cdr(arglist);

  if ( realp(safe_car(arglist)) )
    thiscolordata->d.b= get_number(safe_car(arglist));
  else {
    ger_error("color_create: argument for b not int or real!");
    thiscolordata->d.b= DEFAULT_B;
  }
  arglist= safe_cdr(arglist);

  if ( realp(safe_car(arglist)) )
    thiscolordata->d.a= get_number(safe_car(arglist));
  else {
    ger_error("color_create: argument for a not int or real!");
    thiscolordata->d.a= DEFAULT_A;
  }

  return(thiscolor);
}

Color create_color()
/*
This function creates a color.  It's r, g, and b components are initially
set to 0.8, and it's a component is set to 1.0 .
*/
{
  Color thiscolor;
  
  ger_debug("color_create:");

  thiscolor= new_color();
  incr_ref( (NODE *)thiscolor );
  return( thiscolor );
}

void free_color( thiscolor )
Color thiscolor;
/*
This routine frees the given color.  If this was the last reference to
this color, it will be deleted.
*/
{
  ger_debug("free_color:");
  decr_elem( (NODE *)thiscolor );
}

Color set_color_red( thiscolor, value )
Color thiscolor;
float value;
/* This routine sets the red value of a color */
{
  Color_body *thiscolordata;

  ger_debug("set_color_red:");

  if ((Type_Of((NODE *)thiscolor) != N_USERDATA) 
      || (((NODE *)thiscolor)->val.userdef.methods != &colordef))
    ger_error("set_color_red: passed a non-color!");
  else {
    Color_body *thiscolordata;
    thiscolordata= (Color_body *)((NODE *)thiscolor)->val.userdef.data;
    thiscolordata->d.r= value;
  }

  return(thiscolor);
}

Color set_color_green( thiscolor, value )
Color thiscolor;
float value;
/* This routine sets the green value of a color */
{
  Color_body *thiscolordata;

  ger_debug("set_color_green:");

  if ((Type_Of((NODE *)thiscolor) != N_USERDATA) 
      || (((NODE *)thiscolor)->val.userdef.methods != &colordef))
    ger_error("set_color_green: passed a non-color!");
  else {
    Color_body *thiscolordata;
    thiscolordata= (Color_body *)((NODE *)thiscolor)->val.userdef.data;
    thiscolordata->d.g= value;
  }

  return(thiscolor);
}

Color set_color_blue( thiscolor, value )
Color thiscolor;
float value;
/* This routine sets the blue value of a color */
{
  Color_body *thiscolordata;

  ger_debug("set_color_blue:");

  if ((Type_Of((NODE *)thiscolor) != N_USERDATA) 
      || (((NODE *)thiscolor)->val.userdef.methods != &colordef))
    ger_error("set_color_blue: passed a non-color!");
  else {
    Color_body *thiscolordata;
    thiscolordata= (Color_body *)((NODE *)thiscolor)->val.userdef.data;
    thiscolordata->d.b= value;
  }

  return(thiscolor);
}

Color set_color_alpha( thiscolor, value )
Color thiscolor;
float value;
/* This routine sets the alpha value of a color */
{
  Color_body *thiscolordata;

  ger_debug("set_color_alpha:");

  if ((Type_Of((NODE *)thiscolor) != N_USERDATA) 
      || (((NODE *)thiscolor)->val.userdef.methods != &colordef))
    ger_error("set_color_alpha: passed a non-color!");
  else {
    Color_body *thiscolordata;
    thiscolordata= (Color_body *)((NODE *)thiscolor)->val.userdef.data;
    thiscolordata->d.a= value;
  }

  return(thiscolor);
}

NODE *lisp_color_r( arglist )
NODE *arglist;
/* This routine provides the (color-r thisclr) function in lisp */
{
  Color thiscolor;
  float val;
  NODE *result;

  ger_debug("lisp_color_r:");

  if (length_list(arglist) < 1) {
    ger_error("lisp_color_r: passed empty list!");
    return(NIL);
  }

  thiscolor= (Color)safe_car(arglist);
  val= color_red(thiscolor);
  result= get_floatrep(val);
  return(result);
}

NODE *lisp_color_g( arglist )
NODE *arglist;
/* This routine provides the (color-g thisclr) function in lisp */
{
  Color thiscolor;
  float val;
  NODE *result;

  ger_debug("lisp_color_g:");

  if (length_list(arglist) < 1) {
    ger_error("lisp_color_g: passed empty list!");
    return(NIL);
  }

  thiscolor= (Color)safe_car(arglist);
  val= color_green(thiscolor);
  result= get_floatrep(val);
  return(result);
}

NODE *lisp_color_b( arglist )
NODE *arglist;
/* This routine provides the (color-b thisclr) function in lisp */
{
  Color thiscolor;
  float val;
  NODE *result;

  ger_debug("lisp_color_b:");

  if (length_list(arglist) < 1) {
    ger_error("lisp_color_b: passed empty list!");
    return(NIL);
  }

  thiscolor= (Color)safe_car(arglist);
  val= color_blue(thiscolor);
  result= get_floatrep(val);
  return(result);
}

NODE *lisp_color_a( arglist )
NODE *arglist;
/* This routine provides the (color-a thisclr) function in lisp */
{
  Color thiscolor;
  float val;
  NODE *result;

  ger_debug("lisp_color_a:");

  if (length_list(arglist) < 1) {
    ger_error("lisp_color_a: passed empty list!");
    return(arglist); /* It's nil; might as well use it */
  }

  thiscolor= (Color)safe_car(arglist);
  val= color_alpha(thiscolor);
  result= get_floatrep(val);
  return(result);
}

