/****************************************************************************
 * assist_text.c
 * Author Joel Welling
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
The 'assist'  module provides support for renderers which do not have the
capability to do certain operations.  The operations are simulated using
a facility which the renderer (hopefully) does have.  This module provides
text facilities using the renderer's polyline function and transformation
function.  Because the text depends on the current attributes, this
module can only be used to render text if the renderer uses the attribute
assistance module to maintain its attribute state.
*/

#include <math.h>
#include <ctype.h>
#include "alisp.h"
#include "ge_error.h"
#include "p3d.h"
#include "matrix_ops.h"
#include "assist.h"
#include "hershey.h"  /* hershey.h provides a set of Hershey fonts */

#define MAX_FONT_CHARS 128
#define DEFAULT_FONT_ID 0
#define DEFAULT_HEIGHT 1.0

/* Flag to indicate that needed symbols have been generated. */
static int symbols_ready= 0;

/* Index of the current font (currently not resettable), and text height */
static int current_font_id= DEFAULT_FONT_ID;
static float current_height= DEFAULT_HEIGHT;

/*
The following cache holds the current font.  As characters are prepared
by setup_char, they are added to the cache and the 'ready' value for that
character is set to 1.  release_cache releases all cached characters and
resets the 'ready' values to zero.  Characters are readied and cached one
at a time, as they are needed.
*/
static struct cache_struct { int ready; Facet_list flist; } 
	font_cache[ MAX_FONT_CHARS ];

/* The following macros simplify access to the Hershey font data structures */
#define this_char( character ) \
	char_table[ font_table[current_font_id].first_char + character ]
#define this_stroke( character, strokeid ) \
	stroke_table[ this_char( character ).first_stroke + strokeid ]
#define this_vertex( character, strokeid, vtxid ) \
	vertex_table[ this_stroke(character,strokeid).start + vtxid ] 
#define this_vertex_x( character, strokeid, vtxid ) \
	( (int)this_vertex( character, strokeid, vtxid )[0] - COORD_BIAS )
#define this_vertex_y( character, strokeid, vtxid ) \
	( (int)this_vertex( character, strokeid, vtxid )[1] - COORD_BIAS )

/* This routine releases all cached characters */
static void release_cache()
{
	int ichar;

	ger_debug("release_cache: resetting character cache");

	for (ichar=0; ichar<MAX_FONT_CHARS; ichar++)
		if (font_cache[ichar].ready) {
			free_facet_list( font_cache[ichar].flist );
			font_cache[ichar].flist= NIL;
			font_cache[ichar].ready= 0;
			};
}

/* 
This routine is needed to reset the font cache in the event of a
hard reset of the P3D software.  It need never be called by a renderer;
the user interface should call it when carrying out the hard reset.
*/
void ast_text_reset( hard )
int hard;
{
	int ichar;

	ger_debug("ast_text_reset: resetting character cache; hard= %d",
		  hard);

	if (hard) {
	  symbols_ready= 0;
	  for (ichar=0; ichar<MAX_FONT_CHARS; ichar++)
	    if (font_cache[ichar].ready) {
	      font_cache[ichar].flist= NIL;
	      font_cache[ichar].ready= 0;
	    }
        }
}

/* 
This routine checks for the presence of a character in the cache, and
if it is not present adds it.  The Vertex_list for the character is returned.
*/
static Facet_list setup_char( txtchar, ux, uy, uz, vx, vy, vz )
char txtchar;
float ux, uy, uz, vx, vy, vz;
{
	Facet_list flist, fcopy;
	Vertex_list vlist, vcopy;
	Vertex thisvertex;
	int vcount, vloop, fcount, floop;
	float x, y, z;


	/* If the character is not cached, cache it */
	if (!font_cache[txtchar].ready) {
	    ger_debug("setup_char: caching '%c'",txtchar);

	    fcount= this_char( txtchar ).stroke_count;
	    fcopy= flist= create_facet_list( fcount );
	    for (floop=0; floop<fcount; floop++) {
		vcount= this_stroke( txtchar, floop ).count;
		vcopy= vlist= create_vertex_list( vcount );
		for (vloop=0; vloop<vcount; vloop++ ) {
			x= FONT_SCALE * current_height * (
				this_vertex_x( txtchar, floop, vloop ) * ux +
				this_vertex_y( txtchar, floop, vloop ) * vx );
			y= FONT_SCALE * current_height * (
				this_vertex_x( txtchar, floop, vloop ) * uy +
				this_vertex_y( txtchar, floop, vloop ) * vy );
			z= FONT_SCALE * current_height * (
				this_vertex_x( txtchar, floop, vloop ) * uz +
				this_vertex_y( txtchar, floop, vloop ) * vz );
			thisvertex= create_vertex();
			(void)set_vertex_x( thisvertex, x );
			(void)set_vertex_y( thisvertex, y );
			(void)set_vertex_z( thisvertex, z );
			(void)set_first_vertex( vcopy, thisvertex );
			vcopy= rest_of_vertices(vcopy);
			free_vertex( thisvertex ); /* won't get deallocated
						   because of other refs */
			};
		(void)set_first_facet( fcopy, vlist );
		free_vertex_list( vlist ); /* won't get deallocated because of
						other refs */
		fcopy= rest_of_facets( fcopy );
		};
	    font_cache[txtchar].flist= flist;
	    font_cache[txtchar].ready= 1;
	    }
	else ger_debug("setup_char: '%c' ready",txtchar);

	/* Return the facet list for the character */
	return( font_cache[txtchar].flist );
}

/* This function actually emits the character */
static void do_char( txtchar, ux, uy, uz, vx, vy, vz, pline_fun )
char txtchar;
float ux, uy, uz, vx, vy, vz;
void (*pline_fun)();
{
	Facet_list flist;
	int floop, vcount;

	ger_debug("do_char: '%c'", txtchar);

	/* If the character is not in the font table, substitute for it */
	if ( this_char(txtchar).first_stroke == -1 ) txtchar= '?';
	if ( this_char(txtchar).first_stroke == -1 ) return;

	flist= setup_char( txtchar, ux, uy, uz, vx, vy, vz );
	floop= 0;
	while ( !null(flist) ) {
		vcount= this_stroke(txtchar,floop).count;
		if (vcount) (*pline_fun)( first_facet(flist), vcount );
		flist= rest_of_facets(flist);
		floop++;
		};
}

/* This routine generates a translation transformation */
static Transformation shift( x, y, z )
float x, y, z;
{
	float *matrix;
	Transformation result;

	ger_debug("shift: making translation of ( %f, %f, %f )", x, y, z);

	matrix= make_translate_c(x, y, z);
	result= array2d_to_lisp( matrix );
	grab(result);
	free( (char *)matrix );
	return( result );
}

/* 
This routine looks up the current font name in the font table, returning
the appropriate font id.  If the name is not found, an error message
is emitted and id 0 is returned.
*/
static int font_lookup( font )
char *font;
{
  int iloop;
  char *thischar,*fontp;
  int found;

  ger_debug("font_lookup: looking for font <%s>",font);

  for (iloop=0; iloop<NO_FONTS; iloop++) {
    thischar= font_table[iloop].name;
    fontp= font;
    found= 1;
    while (*thischar != ':') thischar++; /* pass up "HERSHEY" */
    thischar++; /* pass up ":" */
    while ( *thischar 
	   && ((thischar - font_table[iloop].name) < FONT_NM_LENGTH) )
      if (*thischar++ != toupper(*fontp++)) {
	found= 0;
	break;
      }
    if (found) return(iloop);
  }
  ger_error("font_lookup: font <%s> not found; using <%s>",
	    font, font_table[0].name);
  return(0);
}

/* 
This routine extracts and implements text attributes from the given
attribute list.
*/
static void check_textattr()
{
	static Symbol text_height_symbol;
	static Symbol text_font_symbol;
	static char current_font[FONT_NM_LENGTH]= "";
	int clear_now= 0;
	float new_height;
	char *new_font;

	ger_debug("check_textattr");

	if (!symbols_ready) {
		text_height_symbol= create_symbol("text-height");
		text_font_symbol= create_symbol("text-font");
		symbols_ready= 1;
		};

	new_height= ast_float_attribute( text_height_symbol );
        ger_debug("           Text height= %f", new_height);
	if ( new_height != current_height ) {
                current_height= new_height;
		clear_now= 1;
                };

	new_font= ast_string_attribute( text_font_symbol );
        ger_debug("           Text font= <%f>", new_font);
	if ( strncmp(new_font,current_font,FONT_NM_LENGTH) ) {
	        strncpy(current_font,new_font,FONT_NM_LENGTH);
		current_font_id= font_lookup(current_font);
		clear_now= 1;
                };

	if (clear_now) release_cache();
}

void ast_text( txtpoint, uvec, vvec, txtstring, trans_fun, pline_fun )
Point txtpoint;
Vector uvec, vvec;
char *txtstring;
void (*trans_fun)(), (*pline_fun)();
/* 
This routine produces text, using the renderer's polyline function.
The renderer must also provide a function to do the translations
between letters.  It is passed in as trans_fun, and should be
defined as follows:

	static void implement_transformation( trans )
	Transformation trans;

and used as follows:

	ast_text(... , implement_transformation, ...);
*/
{
	float x0, y0, z0, ux, uy, uz, vx, vy, vz, unorm, vnorm;
	static float old_ux= 0.0, old_uy= 0.0, old_uz= 0.0,
		old_vx= 0.0, old_vy= 0.0, old_vz= 0.0;
	Transformation shifttrans;

	ger_debug("ast_text: <%s>",txtstring);

	/* Handle text attributes, if they are present. */
	check_textattr();

	/* 
	Extract u and v vector components, and reset cache if they've
	changed.
	*/
	ux= vector_x( uvec ); uy= vector_y( uvec ); uz= vector_z( uvec );
	vx= vector_x( vvec ); vy= vector_y( vvec ); vz= vector_z( vvec );
	unorm= sqrt( ux*ux + uy*uy + uz*uz );
	vnorm= sqrt( vx*vx + vy*vy + vz*vz );
	ux= ux/unorm; uy= uy/unorm; uz= uz/unorm;
	vx= vx/vnorm; vy= vy/vnorm; vz= vz/vnorm;

	/* Extract text coordinates */
	x0= point_x( txtpoint ); 
	y0= point_y( txtpoint );
	z0= point_z( txtpoint );

	if ( (ux != old_ux) || (uy != old_uy) || (uz != old_uz)
		|| (vx != old_vx) || (vy != old_vy) || (vz != old_vz) ) {
		release_cache();
		old_ux= ux;
		old_uy= uy;
		old_uz= uz;
		old_vx= vx;
		old_vy= vy;
		old_vz= vz;
		};

	for (; *txtstring; txtstring++) {
		/* Translate to the starting point */
		shifttrans= shift(
		  x0 + FONT_SCALE * current_height * 
			(float)( ux*this_char(*txtstring).xcenter ),
		  y0 + FONT_SCALE * current_height * 
			(float)( uy*this_char(*txtstring).xcenter ),
		  z0 + FONT_SCALE * current_height * 
			(float)( uz*this_char(*txtstring).xcenter )
		  );
		(*trans_fun)( shifttrans );
		free_transformation( shifttrans );

		/* Draw the character */
		do_char( *txtstring, ux, uy, uz, vx, vy, vz, pline_fun );

		/* advance point to next character starting position */
		x0 = FONT_SCALE * current_height *
			(float)(ux * this_char(*txtstring).xshift);
		y0 = FONT_SCALE * current_height *
			(float)(uy * this_char(*txtstring).xshift);
		z0 = FONT_SCALE * current_height *
			(float)(uz * this_char(*txtstring).xshift);
		};
}
