/****************************************************************************
 * assist_nrml.c
 * Author Joel Welling
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This module provides support for renderers which do not have the
capability to do certain operations.  This module provides utilities
to add vertex normals to polygons and meshes which do not have them.
The normals are added permanently, in the database representation
of the primitives, so the utilities need be called only once per
primitive.  It is the calling program's responsibility to check the
primitives to see if they already include normals.
*/

#include <math.h>
#include "alisp.h"
#include "ge_error.h"
#include "p3d.h"
#include "assist.h"

static void normalize( thisvector )
Vector thisvector;
/* This routine normalized the three element array passed it. */
{
  float v0, v1, v2, lengthsqr, length;

  v0= vector_x( thisvector );
  v1= vector_y( thisvector );
  v2= vector_z( thisvector );

  lengthsqr= v0*v0 + v1*v1 + v2*v2;
  if (lengthsqr != 0.0) {
    length= sqrt( lengthsqr );
    set_vector_x( thisvector, v0/length);
    set_vector_y( thisvector, v1/length);
    set_vector_z( thisvector, v2/length);
  }
}

static int calc_normal(vlist, count, result)
Vertex_list vlist;
int count;
float *result;
/* This routine attempts to calculate a normal from the first three
 * non-colinear vertices in the given vertex list.  If it fails,
 * it returns 0; otherwise it returns 1.  It will fail if there are
 * less than three vertices in the vertex list, or if all the vertices
 * are colinear.  count contains the number of vertices in the list;
 * if it is zero on entry the list will be walked to find the length.
 * result must point to three consecutive floats to recieve the calculated
 * vector; that vector is not normalized.
 */
{
  Vertex thisvertex;
  Vertex_list vlcopy, vlstart;
  float vcoords[3][3], vectors[2][3];
  int i, numtries=0, success_flag=0;

  /* Count the list if necessary */
  if (!count) count= list_length(vlist);

  /* Iterate down the list of vertices, looking for 3 which are not colinear */
  vlstart= vlist;
  while ( !success_flag && numtries<count-2 ) {
    
    vlcopy= vlist;
    for (i=0; i<3; i++) {
      thisvertex= first_vertex( vlcopy );
      vcoords[i][0]= vertex_x( thisvertex );
      vcoords[i][1]= vertex_y( thisvertex );
      vcoords[i][2]= vertex_z( thisvertex );
      vlcopy= rest_of_vertices( vlcopy );
    }
    
    for (i=0; i<2; i++) {
      vectors[i][0]= vcoords[i+1][0] - vcoords[i][0];
      vectors[i][1]= vcoords[i+1][1] - vcoords[i][1];
      vectors[i][2]= vcoords[i+1][2] - vcoords[i][2];
    }
    
    *result= vectors[0][1]*vectors[1][2] - vectors[0][2]*vectors[1][1];
    *(result+1)= vectors[0][2]*vectors[1][0] - vectors[0][0]*vectors[1][2];
    *(result+2)= vectors[0][0]*vectors[1][1] - vectors[0][1]*vectors[1][0];
    
    if ( (*result != 0.0) || (*(result+1) != 0.0) || (*(result+2) != 0.0) )
      success_flag= 1;
    else {
      vlstart= rest_of_vertices(vlstart); /* try next three vertices */
      numtries++;
    }
  }

  return(success_flag);
}


void ast_polygon_normals(vlist, count)
Vertex_list vlist;
int count;
/* This routine adds a (single) normal vector to the vertices of a
 * polygon.  The polygon is assumed to be coplanar.  If all the points
 * in the polygon are colinear, an error message is given and no normals
 * are added.
 */
{
  Vector normal;
  Vertex_list vlcopy;
  float norm[3];

  ger_debug("ast_polygon_normal: count= %d\n",count);

  if ( calc_normal( vlist, count, norm ) ) {
    
    normal= create_vector();
    (void)set_vector_x( normal, norm[0] );
    (void)set_vector_y( normal, norm[1] );
    (void)set_vector_z( normal, norm[2] );
    normalize( normal );
    
    vlcopy= vlist;
    while (!null(vlcopy)) {
      (void)set_vertex_normal( first_vertex(vlcopy), normal );
      vlcopy= rest_of_vertices( vlcopy );
    }
    free_vector( normal ); /* won't get deallocated because of other refs */
  }
  else ger_error("ast_polygon_normals: polygon has colinear vertices.");
}

void ast_mesh_normals(vlist, vcount, flist, fcount)
Vertex_list vlist;
Facet_list flist;
int vcount, fcount;
/* This routine adds normal vectors to the vertices of a mesh, based on
 * the facing directions of the mesh facets.
 */
{
  Vector thisnormal;
  Vertex_list thisfacet, vlcopy;
  Facet_list flcopy;
  float norm[3];

  ger_debug("ast_mesh_normal: vcount= %d, fcount= %d\n",vcount,fcount);

  /* Add (zero length) normals to all vertices */
  vlcopy= vlist;
  while ( !null(vlcopy) ) {
    set_vertex_normal( first_vertex(vlcopy), create_vector() );
    vlcopy= rest_of_vertices(vlcopy);
  };

  /* Iterate down the facets, adding the normal for each facet to
   * the (accumulating) normal for each of its vertices.
   */
  flcopy= flist;
  while ( !null(flcopy) ) {
    thisfacet= first_facet(flcopy);
    (void)calc_normal( thisfacet, 0, norm );
    while ( !null(thisfacet) ) {
      thisnormal= vertex_normal( first_vertex(thisfacet) );
      (void)set_vector_x( thisnormal, vector_x(thisnormal)+norm[0] );
      (void)set_vector_y( thisnormal, vector_y(thisnormal)+norm[1] );
      (void)set_vector_z( thisnormal, vector_z(thisnormal)+norm[2] );
      thisfacet= rest_of_vertices(thisfacet);
    }
    flcopy= rest_of_facets(flcopy);
  }

  /* Renormalize the vertex normals */
  vlcopy= vlist;
  while ( !null(vlcopy) ) {
    normalize( vertex_normal( first_vertex(vlcopy) ) );
    vlcopy= rest_of_vertices(vlcopy);
  }
}

void ast_triangle_normals(vlist, count)
Vertex_list vlist;
int count;
/* This routine adds normal vectors to the vertices of a triangle strip,
 * based on the facing directions of the strip facets.
 */
{
  Vector thisnormal;
  Vertex_list thisfacet, vlcopy;
  Facet_list flcopy;
  float norm[3][3], accum[3];
  int i, j, k, flipflag= 0;

  ger_debug("ast_triangle_normal: count= %d\n",count);

  /* Initialize the collection of normal components */
  for (i=0; i<3; i++) norm[i][0]= norm[i][1]= norm[i][2]= 0.0;

  /* Add (zero length) normals to all vertices */
  vlcopy= vlist;
  while ( !null(vlcopy) ) {
    set_vertex_normal( first_vertex(vlcopy), create_vector() );
    vlcopy= rest_of_vertices(vlcopy);
  };

  /* Iterate down the vertices, adding the normal for each triangle to
   * the (accumulating) normal for each of its vertices.  We save the
   * most recent three vertices worth of normals, and add them all
   * to produce the normal of a given vertex.
   */
  vlcopy= vlist;
  i= 0;
  while ( count-- > 2 ) { /* loop over all but the last triangle */
    (void)calc_normal( vlcopy, 3, norm[i] );
    if (flipflag) /* this triangle uses left hand rule normals */
      for (j=0; j<3; j++) norm[i][j]= -norm[i][j];
    thisnormal= vertex_normal( first_vertex(vlcopy) );
    for (j=0; j<3; j++) {
      accum[j]= 0.0;
      for (k=0; k<3; k++) accum[j] += norm[k][j];
    }
    (void)set_vector_x( thisnormal, accum[0] );
    (void)set_vector_y( thisnormal, accum[1] );
    (void)set_vector_z( thisnormal, accum[2] );
    if (count > 2) i= (i+1) % 3;
    flipflag= !flipflag;
    vlcopy= rest_of_vertices(vlcopy);
  }

  /* Set normals in the last two vertices */
  thisnormal= vertex_normal( first_vertex(vlcopy) );
  for (j=0; j<3; j++) accum[j]= norm[i][j] + norm[ (i+2)%3 ][j];
  (void)set_vector_x( thisnormal, accum[0] );
  (void)set_vector_y( thisnormal, accum[1] );
  (void)set_vector_z( thisnormal, accum[2] );
  vlcopy= rest_of_vertices(vlcopy);
  thisnormal= vertex_normal( first_vertex(vlcopy) );
  (void)set_vector_x( thisnormal, norm[i][0] );
  (void)set_vector_y( thisnormal, norm[i][1] );
  (void)set_vector_z( thisnormal, norm[i][2] );

  /* Renormalize the vertex normals */
  vlcopy= vlist;
  while ( !null(vlcopy) ) {
    normalize( vertex_normal( first_vertex(vlcopy) ) );
    vlcopy= rest_of_vertices(vlcopy);
  }
}

