/****************************************************************************
 * assist.c
 * Author Joel Welling
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This module provides support for renderers which do not have the
capability to do certain operations.  The operations are simulated using
a facility which the renderer (hopefully) does have.
*/

#include <math.h>
#include "alisp.h"
#include "ge_error.h"
#include "p3d.h"
#include "assist.h"
#include "sphere.h"
#include "cylinder.h"

/* These values signal that the appropriate meshes need to be regenerated */
static float old_major_radius= -1.0, old_minor_radius= -1.0;
                                    /* default to impossible values */
static int sphere_initialized=0, cylinder_initialized= 0;

static void build_mesh( vcount, varray, narray, fcount, vert_per_facet, 
		       connect, vlist, flist )
int vcount, fcount;
float varray[][3];
float narray[][3];
int *connect, *vert_per_facet;
Vertex_list *vlist;
Facet_list *flist;
/* This routine produces mesh data from coordinate and connectivity lists */
{
	Vertex thisvertex, *vertex_array;
	Vector thisnorm;
	Vertex_list vcopy, facet;
	Facet_list fcopy;
	int vloop, floop;

	ger_debug("build_mesh:");

	/* Allocate temporary vertex array */
	if (!(vertex_array= 
		(Vertex *)malloc(vcount*sizeof(Vertex))))
		ger_fatal(
		    "build_mesh: Unable to allocate space for mesh vertices");

	/* Assemble the vertex list */
	vcopy= *vlist= create_vertex_list( vcount );
	for (vloop=0; vloop<vcount; vloop++ ) {
		thisvertex= create_vertex();
		thisnorm= create_vector();
		(void)set_vertex_x( thisvertex, varray[vloop][0] );
		(void)set_vertex_y( thisvertex, varray[vloop][1] );
		(void)set_vertex_z( thisvertex, varray[vloop][2] );
		(void)set_vertex_normal( thisvertex, thisnorm );
		(void)set_vector_x( thisnorm, narray[vloop][0] );
		(void)set_vector_y( thisnorm, narray[vloop][1] );
		(void)set_vector_z( thisnorm, narray[vloop][2] );
		vertex_array[vloop]= thisvertex;
		(void)set_first_vertex( vcopy, thisvertex );
		vcopy= rest_of_vertices(vcopy);
		free_vertex( thisvertex ); /* won't get deallocated
					   because of other refs */
		free_vector( thisnorm );   /* won't get deallocated
					   because of other refs */
		};

	/* Assemble the facet list, additional pointers to same vertices */
	fcopy= *flist= create_facet_list( fcount );
	for (floop=0; floop<fcount; floop++ ) {
		vcopy= facet= create_vertex_list( *vert_per_facet );
		(void)set_first_facet( fcopy, facet );
		free_vertex_list( facet ); /* won't get deallocated because
						of other refs */
		fcopy= rest_of_facets(fcopy);
		for (vloop=0; vloop<*vert_per_facet; vloop++) 
			{
			thisvertex= vertex_array[ *connect++ ];
			(void)set_first_vertex( vcopy, thisvertex );
			vcopy= rest_of_vertices(vcopy);
			};
		vert_per_facet++;
		};

	free( (char *)vertex_array );
}

void ast_sphere( mesh_fun )
void (*mesh_fun)();
/* This routine produces a sphere, using the renderer's mesh function. */
{
	static Vertex_list vlist;
	static Facet_list flist;

	ger_debug("ast_sphere:");

	/* If not initialized, generate the sphere data structures */
	if (!sphere_initialized) {
		build_mesh( sphere_vertices, sphere_coords, sphere_coords,
			sphere_facets, sphere_v_counts, sphere_connect,
			&vlist, &flist );
		sphere_initialized= 1;
		}; /* End of initialization */

	/* Make the call to draw the mesh */
	(*mesh_fun)( vlist, sphere_vertices, flist, sphere_facets );
}

void ast_cylinder( mesh_fun )
void (*mesh_fun)();
/* This routine produces a cylinder, using the renderer's mesh function. */
{
	static Vertex_list vlist;
	static Facet_list flist;

	ger_debug("ast_cylinder:");

	/* If not initialized, generate the cylinder data structures */
	if (!cylinder_initialized) {
	  build_mesh( cylinder_vertices, cylinder_coords, cylinder_normals,
		     cylinder_facets, cylinder_v_counts, cylinder_connect,
		     &vlist, &flist );
	  cylinder_initialized= 1;
	}; /* End of initialization */

	/* Make the call to draw the mesh */
	(*mesh_fun)( vlist, cylinder_vertices, flist, cylinder_facets );
}

void ast_torus( major_radius, minor_radius, mesh_fun )
float major_radius, minor_radius;
void (*mesh_fun)();
/* This routine produces a torus, using the renderer's mesh function. */
{
#define major_divisions 8
#define minor_divisions 8
#define pi 3.14159265
#define torus_vertices major_divisions*minor_divisions
#define torus_facets major_divisions*minor_divisions

	static Vertex_list vlist;
	static Facet_list flist;
	float coords[torus_vertices][3];
	float norms[torus_vertices][3];
	int iloop, jloop, ip, jp, vcount;
	float theta= 0.0, dt= 2.0*pi/minor_divisions, 
		phi=0.0, dp= 2.0*pi/major_divisions;
	int vertex_counts[torus_facets];
	int connect[4*torus_facets], *connect_copy;

	/* 
	If the major and minor radii match the last ones used, use
	the existing mesh.  Otherwise...
	*/
	if ((major_radius!=old_major_radius)||(minor_radius!=old_minor_radius))
		{
		old_major_radius= major_radius;
		old_minor_radius= minor_radius;

		/* Fill the vertex_counts array */
		for (iloop=0; iloop<torus_facets; iloop++) 
			vertex_counts[iloop]= 4;

		/* Calculate vertex positions */
		vcount= 0;
		for (iloop=0; iloop<major_divisions; iloop++) {
			for (jloop=0; jloop<minor_divisions; jloop++) {
				coords[ vcount ][0]= 
				  ( major_radius + minor_radius*cos(theta) )
				  * cos(phi);
				coords[ vcount ][1]= 
				  ( major_radius + minor_radius*cos(theta) )
				  * sin(phi);
				coords[ vcount ][2]= 
				  minor_radius * sin(theta);
				norms[ vcount ][0]= cos(theta)*cos(phi);
				norms[ vcount ][1]= cos(theta)*sin(phi);
				norms[ vcount ][2]= sin(theta);
				vcount++;
				theta += dt;
				}
			phi += dp;
			}

		/* Generate the connectivity array */
		connect_copy= connect;
		for (iloop=0; iloop<major_divisions; iloop++)
			for (jloop=0; jloop<minor_divisions; jloop++) {
				ip= (iloop+1) % major_divisions;
				jp= (jloop+1) % minor_divisions;
				*connect_copy++= jp + iloop*minor_divisions;
				*connect_copy++= jloop + iloop*minor_divisions;
				*connect_copy++= jloop + ip*minor_divisions;
				*connect_copy++= jp + ip*minor_divisions;
				}

		/* Generate the mesh from the torus data and draw it */
		build_mesh( torus_vertices, coords, norms, torus_facets, 
			   vertex_counts, connect, &vlist, &flist );
		}

	(*mesh_fun)( vlist, torus_vertices, flist, torus_facets );
}

/* 
This routine is needed to reset the sphere, cylinder, and torus emulation
meshes in the event of a hard reset.  It need never be called by a renderer;
the user interface should call it when carrying out the hard reset.
*/
void ast_prim_reset( hard )
int hard;
{
	ger_debug("ast_prim_reset: resetting primitive meshes; hard= %d",
		  hard);

	if (hard) {
	  sphere_initialized= 0;
	  cylinder_initialized= 0;
	  old_major_radius= old_minor_radius= -1.0;
        }
}
