/****************************************************************************
 * trace_to_snaps.c
 * Author Joel Welling
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
  This program converts a trace file of the sort produced by fui.c into
  a series of P3D 'snap' functions.
*/

#include <stdio.h>

/* input and output files */
static FILE *infile, *outfile;

/* Number of interpolants desired */
static int ninterp;

/* Name of P3D object to be flown by */
static char *objname= "object";

/* 
Total number of control points that can be read in, and number actually
in use.
*/
#define max_points 4000
static int npoints;

/* 
Storage for needed info from points read in, and second derivatives
as used by spline routines.
*/
float frmx[max_points], *frmx2;
float frmy[max_points], *frmy2;
float frmz[max_points], *frmz2;
float atx[max_points], *atx2;
float aty[max_points], *aty2;
float atz[max_points], *atz2;
float upx[max_points], *upx2;
float upy[max_points], *upy2;
float upz[max_points], *upz2;
float clocktime[max_points];

/* P3D preamble text */
static char preamble[]= "\
(defun flysnap (fmx fmy fmz atx aty atz upx upy upz) \n\
  (snap %s \n\
     (def-gob :children (list (ambient (make-color :r 0.3 :g 0.3 :b 0.3)) \n\
           (light (make-point :x (+ fmx upx) :y (+ fmy upy) :z (+ fmz upz)) \n\
                  (make-color :r 0.7 :g 0.7 :b 0.7)))) \n\
     (make-camera :lookfrom (make-point :x fmx :y fmy :z fmz) \n\
		     :lookat (make-point :x atx :y aty :z atz) \n\
		     :up (make-vector :x upx :y upy :z upz) \n\
		     :fovea %f \n\
		     :hither %f \n\
		     :yon %f))) \n\
"; /* end of preamble */

/* 
The following two routines are based on 'spline' and 'splint' from 
Numerical Recipes in C, The Art of Scientific Computing, by Press, 
Flannery, Teukolsky, and Vetterling, Cambridge University Press 1988.
*/
static float *make_spline_table(x, y, n)
float *x, *y;
int n;
/* 
This routine generates and returns a natural bicubic spline second derivative
table for the given data.  It is the equivalent of the Numerical Recipes
routine 'spline'.
*/
{
  float *y2, *u;
  register int i;
  register float sig,p;

  if (!(y2= (float *)malloc( n*sizeof(float) ))) {
    fprintf(stderr,
	    "make_spline_table: could not allocate %d floats for results!\n",
	    n);
    exit(2);
  }

  if (!(u= (float *)malloc( n*sizeof(float) ))) {
    fprintf(stderr,
	    "make_spline_table: could not allocate %d floats temp space!\n",
	    n);
    exit(2);
  }

  y2[0]= u[0]= 0.0;

  for (i=1; i<n-1; i++) {
    sig= (x[i]-x[i-1])/(x[i+1]-x[i]);
    p= sig*y2[i-1] + 2.0;
    y2[i]= (sig-1.0)/p;
    u[i]= (y[i+1]-y[i])/(x[i+1]-x[i]) - (y[i]-y[i-1])/(x[i]-x[i-1]);
    u[i]= (6.0*u[i]/(x[i+1]-x[i-1])-sig*u[i-1])/p;
  }

  y2[n-1]= 0.0;

  for (i=n-2; i>=0; i--) y2[i]= y2[i]*y2[i+1] + u[i];

  free((char *)u);

  return(y2);
}

static float splint( xa, ya, y2a, n, x)
float *xa, *ya, *y2a, x;
int n;
/*
This routine produces a natural spline interpolation at the point x, given
that the values of the functions at the xa points are given in ya, and
that y2a is a valid result from make_spline_table.  This is equivalent to
the Numerical Recipes function 'splint'.  We assume spatially correlated
calls to the routine, rather than random calls, and thus do not use
bisection to pick the appropriate spline segment.
*/
{
  static int low= 0, high= 0;
  float delta, a, b, result;

  /* Make sure the range of xa's includes the value of interest. */
  if (xa[0] > x) {
    fprintf(stderr,"splint: x value %f less than table range.\n",x);
    exit(2);
  }
  if (xa[n-1] < x) {
    fprintf(stderr,"splint: x value %f greater than table range.\n",x);
    exit(2);
  }

  /* Find appropriate low and high values */
  while ( xa[low]>x ) low--;
  while ( (xa[low+1]<x) && (low<n-2) ) low++;
  high= low+1;

  delta= xa[high]-xa[low];
  if (delta==0.0) {
    fprintf(stderr,"splint: bad xa intput\n");
    exit(2);
  }
  a= (xa[high]-x)/delta;
  b= (x-xa[low])/delta;
  result= a*ya[low] + b*ya[high] 
    + ((a*a*a - a)*y2a[low] + (b*b*b - b)*y2a[high])*(delta*delta)/6.0;

  return(result);
}

static void parse_args(argc,argv)
int argc;
char *argv[];
/* This routine parses command line arguments */
{
  if (argc != 4) {
    fprintf(stderr,"\
usage: %s objname tracefile count\n\
               objname        name of P3D object to be flown by\n\
               tracefile      input trace file (- for stdin)\n\
               count          desired number of snaps\n",
	    argv[0]);
    exit(2);
  };

  objname= argv[1];

  if (!strcmp(argv[2],"-")) infile= stdin;
  else if ( !(infile= fopen( argv[2],"r" ) ) ) {
    fprintf(stderr,"Unable to open %s for reading\n",argv[1]);
    exit(2);
  };

  outfile= stdout;

  if ( sscanf(argv[3],"%d",&ninterp) != 1 ) {
    fprintf(stderr,"Unable to parse %s as an integer\n",argv[2]);
    exit(2);
  }

  if (ninterp<=1) {
    fprintf(stderr,"%d is not enough interpolants!\n",ninterp);
    exit(2);
  }
}

static void search_string(file,string)
FILE *file;
char *string;
/* 
This routine scans forward through the file, looking for the first
occurance of the given string, and leaves the file pointer at the
character immediately following that occurrance.
*/
{
  char *thischar;

  thischar= string;

  while (!feof(file)) {
    if ( getc(file) == *thischar ) {
      thischar++;
      if (!(*thischar)) break;
    }
    else thischar= string;
  }
}

static void load_positions()
/* This routine loads positions to be splined over */
{
  int offset= 0;

  search_string(infile,
		"----------------------------------------------------");
  while (!feof(infile)) {
    search_string(infile,"lookfrom=");
    if (3 != fscanf(infile,"%f %f %f",frmx+offset,frmy+offset,frmz+offset)) {
      fprintf(stderr,"Bad input file; ends too soon\n");
      exit(2);
    }
    search_string(infile,"lookat=");
    if (3 != fscanf(infile,"%f %f %f",atx+offset,aty+offset,atz+offset)) {
      fprintf(stderr,"Bad input file; ends too soon\n");
      exit(2);
    }
    search_string(infile,"up=");
    if (3 != fscanf(infile,"%f %f %f",upx+offset,upy+offset,upz+offset) ) {
      fprintf(stderr,"Bad input file; ends too soon\n");
      exit(2);
    }

    *(clocktime+offset)= (float)offset;

    if (offset >= max_points-1) {
      fprintf(stderr,"Path too long;  using first %d points\n",offset+1);
      break;
    }
    offset++;
    search_string(infile,
		  "----------------------------------------------------");
}
  npoints= offset;
  fprintf(stderr,"%d data points found\n",npoints);

  if (npoints<=1) {
    fprintf(stderr,"%d is not enough data points!\n",npoints);
    exit(2);
  };
}

static void emit_snaps()
/* This routine emits the needed snaps. */
{
  float dt, time;
  int i;

  time= 0.0;
  dt= ((float)(npoints-1))/(ninterp-1); 
        /* checked for ninterp==1 in parse_args */

  for (i=0; i<ninterp; i++) {
    fprintf(outfile,"(flysnap %f %f %f\n   %f %f %f %f %f %f)\n",
	    splint( clocktime, frmx, frmx2, npoints, time ),
	    splint( clocktime, frmy, frmy2, npoints, time ),
	    splint( clocktime, frmz, frmz2, npoints, time ),
	    splint( clocktime, atx, atx2, npoints, time ),
	    splint( clocktime, aty, aty2, npoints, time ),
	    splint( clocktime, atz, atz2, npoints, time ),
	    splint( clocktime, upx, upx2, npoints, time ),
	    splint( clocktime, upy, upy2, npoints, time ),
	    splint( clocktime, upz, upz2, npoints, time ));
    time += dt;
    if (time>(float)(npoints-1)) 
      time= (float)(npoints-1); /* to avoid rounding error */
  }
}

main(argc,argv)
int argc;
char *argv[];
{
  float fovea, hither, yon;

  /* Parse command line arguments */
  parse_args(argc,argv);

  /* Load the camera data and emit the preamble */
  if (3 != fscanf(infile,"fovea= %f, hither= %f, yon= %f\n",
		  &fovea,&hither,&yon)) {
    fprintf(stderr,"Bad input format; fovea, hither, or yon not found\n");
    exit(2);
  };
  fprintf(outfile,preamble,objname,fovea,hither,yon);

  /* Load the positions to be splined over */
  load_positions();

  /* Calculate the needed second derivatives */
  frmx2= make_spline_table( clocktime, frmx, npoints );
  frmy2= make_spline_table( clocktime, frmy, npoints );
  frmz2= make_spline_table( clocktime, frmz, npoints );
  atx2= make_spline_table( clocktime, atx, npoints );
  aty2= make_spline_table( clocktime, aty, npoints );
  atz2= make_spline_table( clocktime, atz, npoints );
  upx2= make_spline_table( clocktime, upx, npoints );
  upy2= make_spline_table( clocktime, upy, npoints );
  upz2= make_spline_table( clocktime, upz, npoints );

  /* Emit the results of the interpolation */
  emit_snaps();
}











