/*
** dxftop3d is a utility that convert a valid DXF file to P3D file. 
**
** dxftop3d.c
** version 1.0
** 
** Algorithm:
**
** 1. open the DXF input file, it should be of the form filename.dxf
** 2. print  the header of the P3D file
** 
** 3.  print the color-array of the P3D file
** 
** 4.  print the P3D function definitions
** 
** 5.  reads through the DXF entities data:LINEs, 3DFACEs, 
** and POLYLINEs and print these data in two different methods:
** 
** DXFTOP3D_LINES.
** prints DXF lines as P3D plines
** prints DXF 3dfaces as P3D plines
** prints DXF 3d polyline as a P3D pline
** prints a DXF 3dpolygon mesh as P3D plines
** 
** DXFTOP3D_FACES
** prints DXF lines as P3D pline
** (if the line has been extruded, it prints it as a P3D 3dface)
** prints DXf 3dfaces as P3D 3dface
** prints closed DXF 3dpolyline as P3D pline
** (if the 3d polyline has been extruded, it prints it as a P3D 3dface)
** prints open 3dpolygon mesh as P3D mesh	
** 
** 6. calculates the viewing information:
** - lookat and lookfrom points
** - hither and yon distances
** Finally prints all viewing information.
**
** Configuration options:
** - By default, meshes are drawn using polygons.  This gives a closed
**   surface, but may not be what you want.  The compile-time switch
**   LINE_MESHES can be defined to cause meshes to be drawn with lines
**   instead.
**
** Known Problems:
** - The DXF format doesn't specify which face of a polygon or mesh
**   facet is the 'front' face;  P3D uses a right hand rule to make
**   that distinction.  This means that some of the polygons or meshes
**   in converted models are likely to have 'back' faces showing;
**   we haven't been able to figure out anything useful to do about
**   this.
** 
** 
*/


#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <string.h>


/* Max num of character lines*/
/* for 80 character lines, with two extra for newline and null*/
#define	MAXLINE	82
#define lower 1
/* Buffer size*/
#define BUFSIZE 100000 
/* a pointer to  the input file */     
FILE *DataFile;	
/* a pointer to the current char */	    
char *w;
 /* buffer for ungetch */		    
char buf[BUFSIZE];
/* next free position in buf */	   
int bufp=0;		     

/* max number of points for DXF mesh in M direction */
# define max_m_points 1000   
/* max number of points for DXF mesh in N direction */
# define max_n_points 1000
/* max number of layers in a DXF file*/
# define max_layers 1500
 /* number of coord. (default is 3) */ 
#define	coord_num  3	
     
/* the x viewing factor (default is 3)*/
# define x_view_factor 3  
/* the y viewing factor (default is 3)*/  
# define y_view_factor 3  
/* the z viewing factor (default is 3)*/  
# define z_view_factor 3    

/* bit-coded flags for DXF POLYLINE */
#define b 1
#define b1 (b<<1)
#define b8 (b<<4)
#define b9 ((b<<4)|(b<<1))
#define b16 (b<<5)
#define b17 ((b<<5)|(b<<1))
#define b49 ((b<<6)|(b<<5)|(b<<1))

/*current word that is being read */		                     
char word[MAXLINE];
/* current layer name */
char layername[MAXLINE];
	
/*Number of colors in the color-array*/
int number_of_colors = 256;
/* current used color number */
int current_color_number = 9999;

/* number of vertices in a p3d mesh */
int mesh_vertices_num = 0;           

/* bit-coded DXF polyline flag */
int polyline_flag = 0;
/* the Z elevation of an entity*/
double z_flatland = 0.0;
 /* the Z thickness of an entity*/		  
double z_thickness = 0.0; 
/* DXF 3D Polygon mesh M count */               
int m_count =0;     
/* DXF 3D Polygon mesh N count */                       
int n_count =0;				
int vertices_count =0;
int num_of_layers =0;	

/* total number of DXF 3dfaces */
int DXF_3dfaces_num= 0;	  
/* total number of DXF lines */
int DXF_lines_num = 0;	
/* total number of DXF polylines */	
int DXF_polylines_num= 0;	
/*  number of vertices in a 3dface*/
int vertices_num = 4;	


/************ Viewing variables information **************/
/* the x coord of the "lookfrom"  point */
double x_lookfrom=0.0;	
/* the y coord of the "lookfrom"  point */    
double y_lookfrom=0.0;	 
/* the z coord of the "lookfrom"  point */   
double z_lookfrom=0.0;	 
/* the x coord of the "lookat"  point */   
double x_lookat=0.0;
/* the y coord of the "lookat"  point */	    
double y_lookat=0.0;
/* the z coord of the "lookat"  point */	    
double z_lookat=0.0;	    

 /* min x value of the bounding box */
double x_min= 99999.0;
/* min y value of the bounding box */	   
double y_min= 99999.0;
/* min z value of the bounding box */	    
double z_min= 99999.0;
/* max x value of the bounding box */	    
double x_max= -99999.0;
 /* max y value of the bounding box */	    
double y_max= -99999.0;	
/* max z value of the bounding box */   
double z_max= -99999.0;	    
/* first clipping distance */
double hither=0.0;  
/* second clipping distance */         
double yon=0.0;              


/* data dtructure to store vertices information*/
typedef struct dxf_vertices  dxf_vertices;
struct dxf_vertices {
    double x;
    double y;
    double z;
};
dxf_vertices pline_points[max_m_points][max_n_points];
dxf_vertices line_points[max_m_points];
dxf_vertices faces_points[max_m_points];

/* data dtructure to store layers information*/
typedef struct dxf_layer dxf_layer;
struct dxf_layer{
    char layer_name[MAXLINE];
    int color_num;
};
dxf_layer layer_color[max_layers];


/*----------------------------------------------------------------------*/
void InitRead(char *data_file) 
/* Open the DXF File */
{
    if ((DataFile = fopen(data_file, "r")) == NULL) {
	fprintf(stderr,"could not open %s\n", data_file);
	exit(-1);
    }
} 
                                                             
/*----------------------------------------------------------------------*/
double power (double base, int n)
/* power : raise base to n-th power; n >= 0 */
{
    int i;
    double p;

    p = 1;
    for (i =1; i <= n; ++i){
	p = p * base;
    }
    return p;
}

/*----------------------------------------------------------------------*/
double calc_distance(double xf,double yf, double zf,double xs,double ys,double zs)
/* 
** calculate the distance between two points:
**p_first(xf,yf,zf) and p_second(xs,ys,zs)
*/
{
  double distance;

  distance = sqrt((power ((xf - xs),2))+power((yf - ys),2)+power((zf - zs),2));
  return distance;
}

/*----------------------------------------------------------------------*/
int calc_color (int color_num)
/* 
** calculate the color number
** this function is useful if the input file contains color numbers
** greater than that of the color-array.
*/
{
    return (color_num % number_of_colors);
}

/*----------------------------------------------------------------------*/

void ungetch (int c)
/* push character back on input */
{
    if (bufp >= BUFSIZE){
	printf("ungetch: too many characters\n");
    }
    else 
      buf[bufp++] = c;
}

/*----------------------------------------------------------------------*/

int getword(char *word2,int lim)
/*
** get next word or character from input 
** skip all spaces, newline characters
*/
{

    int c;
    char *w= word2;
    enum State { ST_BEGIN, ST_IN_WORD, ST_END } state;

    /* Check for EOF */
    if (feof(DataFile)) {
      *w++= EOF;
      *w= '\0';
      return word2[0];
    }

    state= ST_BEGIN;
    while (state != ST_END) {
      c= getc(DataFile);
      switch ((int)state) {
      case (int)ST_BEGIN:
	if (c == EOF) { /* quit, with EOF in word */
	  *w++= EOF;
	  state= ST_END;
	}
	else if (isspace(c)) { /* do nothing */ }
	else { /* it's a printable character */
	  *w++= c;
	  state= ST_IN_WORD;
	}
	break;
      case (int)ST_IN_WORD:
	if (c == EOF || isspace(c)) { /* quit */
	  state= ST_END;
	}
	else { /* it's a printable character */
	  *w++= c;
	}	
	break;
      }
    }

    *w= '\0';
    return word2[0];
}

/*----------------------------------------------------------------------*/ 

void get_min_and_max_coord (double x_tmp, double y_tmp, double z_tmp)
/*
** get the min and max coordinates to specify the bounding box
** compare the current coord to the global variables that hold 
** the max and min x, y, and z.
** update these variables if neccessary.
*/
{   
    if (x_tmp > x_max){
	x_max = x_tmp;
    }
    if (x_tmp < x_min){
	x_min = x_tmp;
    }

    if (y_tmp > y_max){
	y_max = y_tmp;
    }
    if (y_tmp < y_min){
	y_min = y_tmp;
    }
    if (z_tmp > z_max){
	z_max = z_tmp;
    }
    if (z_tmp < z_min){
	z_min = z_tmp;
    }
}

/*----------------------------------------------------------------------*/ 
   
void print_header()
/* print the header of a  "P3D" file */
{
    printf(";P3D 2.1\n");
    printf("(load \"p3d.lsp\")\n");
    printf("(defun vtx (xv yv zv) (make-vertex :x xv :y yv :z zv))\n");
    printf("(defun 3dface (args) \n");
    printf("  (let ((clr (car args))\n");
    printf("     (vxs (cdr args)))\n");
    printf("    (def-gob :attr (list (cons 'color (aref color-array clr)))\n");
    printf("      :children (list\n");
    printf("		 (apply 'polygon\n");		
    printf("			(do ((vl vxs (cdr vl)))\n");
    printf("			    ((null vl) vxs)\n");
    printf("			    (setf (car vl)(apply 'vtx (car vl)))))))))\n");
}

/*----------------------------------------------------------------------*/ 
void print_definitions()
/* print the definitions a  "P3D" file */
{
    printf("(defun pline (args)\n");
    printf(" (let ((clr (car args))\n");
    printf("  (vxs (cdr args)))\n");
    printf("   (def-gob :attr (list (cons 'color (aref color-array clr)))\n");
    printf("  :children (list\n");
    printf("		 (apply 'polyline\n");
    printf("			(do ((vl vxs (cdr vl)))\n");
    printf("			    ((null vl) vxs)\n");
    printf("			    (setf (car vl)(apply 'vtx (car vl)))))))))\n");
    printf("(defun vl (vlist) (do ((vls vlist (cdr vls))) ((null vls) vlist)\n");
    printf("		(rplaca vls (apply 'vtx (car vls)))))\n");
    printf("(defun msh (vcount vlist ilist)\n");
    printf("(pprint vcount)\n");
    printf("(pprint vlist)\n");
    printf("(pprint ilist)\n");
    printf(" (let ((varray (do ((result (make-array vcount) result) (i 0 (+ i 1))\n");
    printf("	    (vls (vl vlist) (cdr vls))) ((null vls) result)\n");
    printf("        (setf (aref result i) (car vls))))) \n");
    printf(" (mesh vlist (do ((ils ilist (cdr ils))) ((null ils) ilist) \n");
    printf("	    (do ((vls (car ils) (cdr vls))) ((null vls) T)\n");
    printf("		(rplaca vls (aref varray (car vls))))))))\n");
    printf("(defun dxfmesh (args)\n");
    printf("   (def-gob :attr (list (cons 'color (aref color-array (car args))))\n");
    printf("   :children (list (apply 'msh (cdr args)))))\n");
    printf("(setq model (def-gob :children(list \n");
}

/*----------------------------------------------------------------------*/ 	    
void print_colors()
/* 
** Print the color-array of a P3D file
*/
{
    printf( "(setq color-array\n "); 
    printf( "	(make-array 256 :initial-contents\n "); 
    printf( "		(list\n "); 
    printf("  		    (make-color :r 1.000000 :g 1.000000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 1.000000 :g 0.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 1.000000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 1.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 1.000000 :g 0.000000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 1.000000 :g 1.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 1.000000 :g 1.000000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 1.000000 :g 0.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 1.000000 :g 0.400000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.666667 :g 0.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.400000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.466667 :g 0.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.200000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.266667 :g 0.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.133333 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.066667 :g 0.066667 :b 0.066667)\n "); 
    printf("  		    (make-color :r 1.000000 :g 0.200000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 1.000000 :g 0.600000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.200000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.400000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.466667 :g 0.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.200000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.266667 :g 0.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.133333 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.066667 :g 0.066667 :b 0.066667)\n "); 
    printf("  		    (make-color :r 1.000000 :g 0.400000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 1.000000 :g 0.800000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.400000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.400000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.200000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.400000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.200000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.200000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.066667 :g 0.066667 :b 0.066667)\n "); 
    printf("  		    (make-color :r 1.000000 :g 0.800000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 1.000000 :g 0.800000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.400000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.600000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.400000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.400000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.200000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.266667 :g 0.266667 :b 0.266667)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.133333 :b 0.133333)\n "); 
    printf("  		    (make-color :r 1.000000 :g 1.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 1.000000 :g 1.000000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.600000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.600000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.400000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.400000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.200000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.266667 :g 0.266667 :b 0.266667)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.200000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.133333 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.800000 :g 1.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.800000 :g 1.000000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.600000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.600000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.400000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.400000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.200000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.266667 :g 0.266667 :b 0.266667)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.133333 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.133333 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.400000 :g 1.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.800000 :g 1.000000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.600000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.600000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.400000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.400000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.200000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.200000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.133333 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.066667 :g 0.066667 :b 0.066667)\n "); 
    printf("  		    (make-color :r 0.200000 :g 1.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 1.000000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 1.000000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.600000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.466667 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.400000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.266667 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.133333 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.133333 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.066667 :g 0.066667 :b 0.066667)\n "); 
    printf("  		    (make-color :r 0.000000 :g 1.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 1.000000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.666667 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.600000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.466667 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.400000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.266667 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.133333 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.133333 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.066667 :g 0.066667 :b 0.066667)\n "); 
    printf("  		    (make-color :r 0.000000 :g 1.000000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 1.000000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.600000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.600000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.466667 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.400000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.266667 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.133333 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.133333 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.066667 :g 0.066667 :b 0.066667)\n "); 
    printf("  		    (make-color :r 0.000000 :g 1.000000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 1.000000 :b 0.800000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.600000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.600000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.400000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.400000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.200000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.200000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.133333 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.047059 :g 0.047059 :b 0.066667)\n "); 
    printf("  		    (make-color :r 0.000000 :g 1.000000 :b 0.800000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 1.000000 :b 0.800000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.600000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.600000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.400000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.400000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.200000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.266667 :g 0.266667 :b 0.266667)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.133333 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.133333 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.000000 :g 1.000000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 1.000000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.600000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.600000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.400000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.400000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.200000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.266667 :g 0.266667 :b 0.266667)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.200000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.133333 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.800000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.800000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.400000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.600000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.400000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.400000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.200000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.266667 :g 0.266667 :b 0.266667)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.133333 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.400000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.800000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.400000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.400000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.200000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.400000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.200000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.200000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.066667 :g 0.066667 :b 0.066667)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.200000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.600000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.200000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.400000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 0.466667)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.200000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 0.266667)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.133333 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.066667 :g 0.066667 :b 0.066667)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.400000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 0.666667)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.400000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 0.466667)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.200000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 0.266667)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.133333 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.066667 :g 0.066667 :b 0.066667)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.000000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.400000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.000000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.400000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 0.466667)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.200000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 0.266667)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.133333 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.066667 :g 0.066667 :b 0.066667)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.000000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.800000 :g 0.400000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.000000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.400000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.000000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.200000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.000000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.200000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.066667 :g 0.066667 :b 0.066667)\n "); 
    printf("  		    (make-color :r 0.800000 :g 0.000000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.800000 :g 0.400000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.000000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.400000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.000000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.200000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.000000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.266667 :g 0.266667 :b 0.266667)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.133333 :b 0.133333)\n "); 
    printf("  		    (make-color :r 1.000000 :g 0.000000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 1.000000 :g 0.400000 :b 1.000000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.000000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.400000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.000000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.200000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.000000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.266667 :g 0.266667 :b 0.266667)\n "); 
    printf("  		    (make-color :r 0.000000 :g 0.000000 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.133333 :b 0.133333)\n "); 
    printf("  		    (make-color :r 1.000000 :g 0.000000 :b 0.800000)\n "); 
    printf("  		    (make-color :r 1.000000 :g 0.400000 :b 0.800000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.000000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.400000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.000000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.200000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.000000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.266667 :g 0.266667 :b 0.266667)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.133333 :b 0.133333)\n "); 
    printf("  		    (make-color :r 1.000000 :g 0.000000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 1.000000 :g 0.400000 :b 0.800000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.000000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.400000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.000000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.200000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.000000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.200000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.066667 :g 0.066667 :b 0.066667)\n "); 
    printf("  		    (make-color :r 1.000000 :g 0.000000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 1.000000 :g 0.400000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.000000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.400000 :b 0.400000)\n "); 
    printf("  		    (make-color :r 0.466667 :g 0.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.400000 :g 0.200000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.266667 :g 0.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.133333 :b 0.133333)\n "); 
    printf("  		    (make-color :r 0.133333 :g 0.000000 :b 0.000000)\n "); 
    printf("  		    (make-color :r 0.066667 :g 0.066667 :b 0.066667)\n "); 
    printf("  		    (make-color :r 0.200000 :g 0.200000 :b 0.200000)\n "); 
    printf("  		    (make-color :r 0.333333 :g 0.333333 :b 0.333333)\n "); 
    printf("  		    (make-color :r 0.466667 :g 0.466667 :b 0.466667)\n "); 
    printf("  		    (make-color :r 0.600000 :g 0.600000 :b 0.600000)\n "); 
    printf("  		    (make-color :r 0.733333 :g 0.733333 :b 0.733333)\n "); 
    printf("  		    (make-color :r 0.866667 :g 0.866667 :b 0.866667)\n "); 
    printf( "					)))\n ");

}

/*----------------------------------------------------------------------*/ 

void get_layers_color()
/* 
** Store the color of each layer
** This is useful when a the color of a DXF entity 
** is not defined.
** In case case, it's color will be the color of it's layer
*/
{
    char name_of_layer[MAXLINE];
    int color_of_layer;
    int index = 0;

    for(;;){
	getword(word,MAXLINE);
	if(strcmp(word,"ENDTAB")==0){ 
	    break;
	}
	if(strcmp(word,"LAYER")==0){
	    for(;;){
		getword(word,MAXLINE);
		if(strcmp(word,"2")==0){
		    getword(word,MAXLINE);
		    strcpy(name_of_layer, word);
		}
		/*read layer flag and do nothing with it*/
		if(strcmp(word,"70")==0){
		    getword(word,MAXLINE);
		}
		/* read layer color number and store it.*/
		if(strcmp(word,"62")==0){
		    getword(word,MAXLINE);
		    color_of_layer = atoi(word);
		    break;
		}
		
	    }

	    strcpy(layer_color[index].layer_name, name_of_layer);
	    layer_color[index].color_num = color_of_layer;
	    index++;
	}
    }	
    num_of_layers = (index + 1);
}

/*----------------------------------------------------------------------*/     

void print_current_color()
/* 
** get the current color
** It's either the DXf entity color
** or the color of that entity's color
*/

{
    int n;
    /* if the DXF entity doesn't has a color*/
    /* get the color of its layer */
    if (current_color_number == 9999){
	for(n=0; n<= (num_of_layers - 1); n++){
	    if(strcmp(layer_color[n].layer_name ,layername)==0){ 
		current_color_number= calc_color(layer_color[n].color_num);
	    }
	}
    }
    printf ("%d ",current_color_number);
}

/*----------------------------------------------------------------------*/ 

void read_DXF_3dface()
/*
** read a DXF 3dface
** store its  later name,color, and coordibates
** skip  unnecessary information (at the moment)such as line type.
*/
{
    int old_color_number;
     int m;

    for(;;){  
	getword(word,MAXLINE);

	/* read the layer name and store it it */
	if(strcmp(word,"8")==0){ 
	    getword(word,MAXLINE);
	    strcpy(layername, word);
	}

	/* read the line type and do nothing with it */
	if(strcmp(word,"6")==0){ 
	    getword(word,MAXLINE);	    
	}

	/* read the Z elevation and store it in z_flatland */
	if(strcmp(word,"38")==0){ 
	    getword(word,MAXLINE);	    
	    z_flatland = atof(word);  
	}

	/* read the color number and store it */
	if(strcmp(word,"62")==0){ 
	    getword(word,MAXLINE);	    
	    old_color_number = atoi(word);
	    current_color_number = calc_color (old_color_number);
	}

	/* read the "Paper Space" entity record and do nothing with it */
	if(strcmp(word,"67")==0){ 
	    getword(word,MAXLINE);	    
	}

	/* read the coordinates of the 3DFACE vertices */
	if(strcmp(word,"10")==0){
	    for (m=0; m<= (vertices_num - 1); m++){
		getword(word,MAXLINE);		    /* read x coord information */
		faces_points[m].x= atof(word);
		getword(word,MAXLINE);		    /* read y coord flag */
		getword(word,MAXLINE);		    /* read y coord information */
		faces_points[m].y= atof(word);
		/* read next flag, which might be the Z coord flag,*/
		/* or the flag of the x coord of next vertex */
		getword(word,MAXLINE);
		if((strcmp(word,"30")==0)|| (strcmp(word,"31")==0)|| (strcmp(word,"32")==0)|| (strcmp(word,"33")==0)){
		    getword(word,MAXLINE);
		    faces_points[m].z= atof(word);
		    getword(word,MAXLINE);
		}
		else {
		    faces_points[m].z= z_flatland;
		} 
		get_min_and_max_coord (faces_points[m].x,faces_points[m].y, faces_points[m].z);
	    }	
	    break;
	}
	
    }
}

/*----------------------------------------------------------------------*/ 

void print_DXF_3dface_as_P3D_pline()
/* 
** print a DXF 3dface as P3D pline
*/
{
    int m;
  
    printf ("%s", "(");
    printf ("%s", "pline");
    printf ("%s", " '(");
    print_current_color();
    for (m=0; m<=(vertices_num-1); m++){
	printf("(%f %f %f)",faces_points[m].x, faces_points[m].y, faces_points[m].z); 
    }
    printf ("%s\n", "))");
    z_flatland = 0.0;
    current_color_number = 9999;
}

/*----------------------------------------------------------------------*/ 

void print_DXF_3dface_as_P3D_3dface()
/* 
** print DXf 3dfaces as P3D 3dface
*/
{
    int m;

    printf ("%s", "(");
    printf ("%s", "3dface");
    printf ("%s", " '(");

    print_current_color();
    for (m=0; m<=(vertices_num-1); m++){
	printf("(%f %f %f)",faces_points[m].x, faces_points[m].y, faces_points[m].z); 
    }
    printf ("%s\n", "))");

    z_flatland = 0.0;
    z_thickness = 0.0;
    current_color_number = 9999;
}

/*----------------------------------------------------------------------*/ 

void read_DXF_line()
/*
** read a DXF line
** store its layer name, color and coordinates
** ignore all other data, such as line type
** if the Z coord of line endpoints is not represented,
** then it will be replaced by Z flatland.
*/
{
    int m =0;
    int old_color_number;

    for(;;){  
	getword(word,MAXLINE);

	/* read the layer name and store it */
	if(strcmp(word,"8")==0){ 
	    getword(word,MAXLINE);	
	    strcpy(layername, word);
	}

	/* read the line type and do nothing with it */
	if(strcmp(word,"6")==0){ 
	    getword(word,MAXLINE);	    
	}

	/* read the Z elevation and store it in z_flatland */
	if(strcmp(word,"38")==0){ 
	    getword(word,MAXLINE);	    
	    z_flatland = atof(word);  
	}

	/* read the Z thickness and do nothing with it (at the moment)*/
	if(strcmp(word,"39")==0){ 
	    getword(word,MAXLINE);
	    z_thickness = atof(word); 
	}

	/* read the color number and store it */
	if(strcmp(word,"62")==0){ 
	    getword(word,MAXLINE);	    
	    old_color_number = atoi(word);
	    current_color_number = calc_color (old_color_number);
	}

	/* read the "Paper Space entity record and do nothing with it */
	if(strcmp(word,"67")==0){ 
	    getword(word,MAXLINE);	    
	}

	/* read the coordinates of the line endpoints */
	if(strcmp(word,"10")==0){
	    for (m=0; m<=lower; m++){
		getword(word,MAXLINE);		    /* read x coord information */
		line_points[m].x= atof(word);
		getword(word,MAXLINE);		    /* read y coord flag */
		getword(word,MAXLINE);		    /* read y coord information */
		line_points[m].y= atof(word);
		/* read next flag, which might be the Z coord flag,*/
		/* or the flag of the x coord of next vertex */
		getword(word,MAXLINE);
		if((strcmp(word,"30")==0)|| (strcmp(word,"31")==0)){
		    getword(word,MAXLINE);
		    line_points[m].z= atof(word);
		    getword(word,MAXLINE);
		}
		else {
		    line_points[m].z= z_flatland;
		} 
		get_min_and_max_coord (line_points[m].x,line_points[m].y, line_points[m].z);
	    }
	    break;
	}
    }
}

/*----------------------------------------------------------------------*/ 

void print_DXF_line_as_P3D_pline()
/* 
** print a a DXF line as a P3D pline
*/
{
    int m;
  
    printf ("%s", "(");
    printf ("%s", "pline");
    printf ("%s", " '(");
    print_current_color();

    /* if the DXF line has not been extruded */
    if (z_thickness == 0){
	for (m=0; m<=lower; m++){
	    printf("(%f %f %f)",line_points[m].x, line_points[m].y, line_points[m].z); 
	}
    }
    /* if the DXF line has been extruded */
    if (z_thickness != 0){
	if(z_thickness > 0){
	    printf("(%f %f %f)",line_points[0].x, line_points[0].y, line_points[0].z); 
	    printf("(%f %f %f)",line_points[1].x, line_points[1].y, line_points[1].z);
	    printf("(%f %f %f)",line_points[1].x, line_points[1].y, (line_points[1].z + z_thickness));
	    printf("(%f %f %f)",line_points[0].x, line_points[0].y, (line_points[0].z + z_thickness));
	    printf("(%f %f %f)",line_points[0].x, line_points[0].y, line_points[0].z);
	}
	if(z_thickness < 0){
	    printf("(%f %f %f)",line_points[0].x, line_points[0].y, line_points[0].z); 
	    printf("(%f %f %f)",line_points[0].x, line_points[0].y, (line_points[0].z + z_thickness));
	    printf("(%f %f %f)",line_points[1].x, line_points[1].y, (line_points[1].z + z_thickness));
	    printf("(%f %f %f)",line_points[1].x, line_points[1].y, line_points[1].z);
	    printf("(%f %f %f)",line_points[0].x, line_points[0].y, line_points[0].z);
	}
    }
    printf ("%s\n", "))");
    z_flatland = 0.0;
    z_thickness = 0.0;
    current_color_number = 9999;
}
/*----------------------------------------------------------------------*/ 

void print_extruded_DXF_line_as_P3D_3faces()
/*
**  print a line 
** if the line has been extruded, print it as a 3dface
*/
{
    int m;
   
    /* if the line has not been extruded */
    if (z_thickness == 0){
	printf ("%s", "(");
	printf ("%s", "pline");
	printf ("%s", " '(");
	print_current_color();
	for (m=0; m<=lower; m++){
	    printf("(%f %f %f)",line_points[m].x, line_points[m].y, line_points[m].z); 
	}
	printf ("%s\n", "))");
    }
    /* if the line has been extruded */
    if (z_thickness != 0){
	printf ("%s", "(");
	printf ("%s", "3dface");
	printf ("%s", " '(");
	print_current_color();
	if(z_thickness < 0){
	    printf("(%f %f %f)",line_points[0].x, line_points[0].y, line_points[0].z); 
	    printf("(%f %f %f)",line_points[1].x, line_points[1].y, line_points[1].z);
	    printf("(%f %f %f)",line_points[1].x, line_points[1].y, (line_points[1].z + z_thickness));
	    printf("(%f %f %f)",line_points[0].x, line_points[0].y, (line_points[0].z + z_thickness));
	}
	if(z_thickness > 0){
	    printf("(%f %f %f)",line_points[0].x, line_points[0].y, line_points[0].z); 
	    printf("(%f %f %f)",line_points[0].x, line_points[0].y, (line_points[0].z + z_thickness));
	    printf("(%f %f %f)",line_points[1].x, line_points[1].y, (line_points[1].z + z_thickness));
	    printf("(%f %f %f)",line_points[1].x, line_points[1].y, line_points[1].z);
	}
	printf ("%s\n", "))");
    }
    z_flatland = 0.0;
    z_thickness = 0.0;
    current_color_number = 9999;
}

/*----------------------------------------------------------------------*/ 

void read_DXF_3dpolyline()
/*
** read a DXF 3dpolyline
** store it's layer name, color, and coordinates
** ignore other data such as line type
*/
{
   int m =0;
     for(;;){
	 
	 if(strcmp(word,"SEQEND")==0){ 
	     break;
	 }
	 if(strcmp(word,"VERTEX")==0){ 
	     for(;;){  
		 getword(word,MAXLINE);
		 /* read the layer name and do nothing with it */
		 if(strcmp(word,"8")==0){ 
		     getword(word,MAXLINE);	    
		 }
		 /* read the line type and do nothing with it */
		 if(strcmp(word,"6")==0){ 
		     getword(word,MAXLINE);	    
		 }
		 /* read the Z elevation and store it in z_flatland */
		 if(strcmp(word,"38")==0){ 
		     getword(word,MAXLINE);	
		 }
		 /* read the Z thickness and do nothing with it*/
		 if(strcmp(word,"39")==0){ 
		     getword(word,MAXLINE);	     
		 }
		 /* read the color information and do nothing with it*/
		 if(strcmp(word,"62")==0){ 
		     getword(word,MAXLINE);	    
		 }
		 if(strcmp(word,"10")==0){ 
		     getword(word,MAXLINE);
		     pline_points[m][0].x= atof(word);
		     getword(word,MAXLINE);
		     getword(word,MAXLINE);
		     pline_points[m][0].y= atof(word);
		     getword(word,MAXLINE);
		     if((atoi(word))== 30){
			 getword(word,MAXLINE);
			 pline_points[m][0].z= atof(word);
		     }
		     else {
			 pline_points[m][0].z= z_flatland;
		     } 
		     get_min_and_max_coord (pline_points[m][0].x,pline_points[m][0].y, pline_points[m][0].z);
		     m++;
		     break;
		 }
	     }
	 }
	 getword(word,MAXLINE);
     }
     vertices_count = m;
}

/*----------------------------------------------------------------------*/ 

void print_open_DXF_3dpolyline_as_P3D_pline()
/* 
** print open DXF 3d polyline as a P3D pline
*/
{
    int m;

    printf ("%s", "(");
    printf ("%s", "pline");
    printf ("%s", " '(");
    print_current_color();
    for (m=0; m<=(vertices_count - 1); m++){
	printf("(%f %f %f)",pline_points[m][0].x, pline_points[m][0].y, pline_points[m][0].z); 
    }
    printf ("%s\n", "))");
    z_thickness = 0.0;
    current_color_number = 9999;
}

/*----------------------------------------------------------------------*/ 

void print_closed_DXF_3dpolyline_as_P3D_pline()
/*
** print closed DXF 3dpolyline as P3D pline
*/
{
    int m;
    int k = vertices_count;

    printf ("%s", "(");
    printf ("%s", "pline");
    printf ("%s", " '(");
    print_current_color();
    if(z_thickness == 0){
	for (m=0; m<=vertices_count; m++){
	    printf("(%f %f %f)",pline_points[m % k][0].x, pline_points[m % k][0].y, pline_points[m % k][0].z); 
	}
    }
    if(z_thickness != 0){
	for (m=0; m<=(vertices_count-1); m++){
	    printf("(%f %f %f)",pline_points[m % k][0].x, pline_points[m % k][0].y, pline_points[m % k][0].z);
	    printf("(%f %f %f)",pline_points[(m+1) % k][0].x, pline_points[(m +1)% k][0].y, pline_points[(m+1) % k][0].z);
	    printf("(%f %f %f)",pline_points[(m+1) % k][0].x, pline_points[(m +1)% k][0].y, (pline_points[(m+1) % k][0].z + z_thickness));
	     printf("(%f %f %f)",pline_points[m % k][0].x, pline_points[m % k][0].y, (pline_points[m % k][0].z+ z_thickness));
	     printf("(%f %f %f)",pline_points[m % k][0].x, pline_points[m % k][0].y, pline_points[m % k][0].z);
	}
    }
    printf ("%s\n", "))");
    z_thickness = 0.0;
    current_color_number = 9999;
}

/*----------------------------------------------------------------------*/ 

void print_closed_DXF_3dpolyline_as_P3D_3dface()
/*
** print closed DXF 3dpolyline as P3D 3dface
*/
{
    int m;
    int k = vertices_count;

    /* if the 3dpolyline has not been extruded */
    if(z_thickness == 0){
	printf ("%s", "(");
	printf ("%s", "pline");
	printf ("%s", " '(");
	print_current_color();
	for (m=0; m<=vertices_count; m++){
	    printf("(%f %f %f)",pline_points[m % k][0].x, pline_points[m % k][0].y, pline_points[m % k][0].z); 
	}
	 printf ("%s\n", "))");
    }

    /* if the 3dpolyline has been extruded */
    if(z_thickness != 0){
	printf ("%s", "(");
	printf ("%s", "3dface");
	printf ("%s", " '(");
	print_current_color();
	for (m=0; m<=(vertices_count-1); m++){
	    printf("(%f %f %f)",pline_points[m % k][0].x, pline_points[m % k][0].y, pline_points[m % k][0].z);
	    printf("(%f %f %f)",pline_points[(m+1) % k][0].x, pline_points[(m +1)% k][0].y, pline_points[(m+1) % k][0].z);
	    printf("(%f %f %f)",pline_points[(m+1) % k][0].x, pline_points[(m +1)% k][0].y, (pline_points[(m+1) % k][0].z + z_thickness));
	     printf("(%f %f %f)",pline_points[m % k][0].x, pline_points[m % k][0].y, (pline_points[m % k][0].z+ z_thickness));
	}
	 printf ("%s\n", "))");  
    }
    z_thickness = 0.0;
     current_color_number = 9999;
}
	   
/*----------------------------------------------------------------------*/ 

void read_DXF_3dpolygon_mesh()
/*
** read and store a DXF 3dpolygon mesh vertices.
*/
{
    int m , n;

    for (m=0; m<=(m_count - 1); m++){
	for (n=0; n<=(n_count - 1); n++){

	    for(;;){
		if(strcmp(word,"VERTEX")==0){ 
		    break;
		}
		getword(word,MAXLINE);
	    }
	    for(;;){
		getword(word,MAXLINE);
		if(strcmp(word,"10")==0){ 
		    break;
		}

	    }
	    getword(word,MAXLINE);
	    pline_points[m][n].x= atof(word);
	    getword(word,MAXLINE);
	    getword(word,MAXLINE);
	    pline_points[m][n].y= atof(word);
	    getword(word,MAXLINE);
	    if((atoi(word))== 30){
		getword(word,MAXLINE);
		pline_points[m][n].z= atof(word);
	    }
	    else {
		pline_points[m][n].z= z_flatland;
	    } 
	    get_min_and_max_coord (pline_points[m][n].x,pline_points[m][n].y, pline_points[m][n].z);
	}
    }
}

/*----------------------------------------------------------------------*/ 

void print_open_3dpolygon_mesh_as_P3D_mesh()	
/*
** print open 3dpolygon mesh as P3D mesh
*/
{    
    int m , n, k;

    k = mesh_vertices_num;
    printf ("%s", "(");
    printf ("%s", "dxfmesh");
    printf ("%s", " '(");
    /* print 3d mesh color number */
    print_current_color();
    /* print the number of vertices in the 3D mesh */
    printf ("%d\n ", mesh_vertices_num);
    /* print the vertex coordinates of the 3D mesh */
    printf ("%s", "(");
    for (m=0; m<=(m_count - 1); m++){
	for (n=0; n<=(n_count - 1); n++){
	    printf("(%f %f %f)",pline_points[m][n].x, pline_points[m][n].y, pline_points[m][n].z); 
	}
	printf("\n");
    }
    printf ("%s\n", ")");

    /* print the polygon vertices of the 3D mesh */
    printf ("%s", "(");
    for (m=0; m<=(m_count - 2); m++){
	for (n=0; n<=(n_count - 2); n++){
	    printf("(%d %d %d)",(((n + 1) + (n_count * m )) % k),((n + (n_count * (m + 1))) % k),((n + (n_count * m)) % k));
	    printf("(%d %d %d)\n",(((n + 1) + (n_count * m )) % k),(((n + 1) + (n_count * (m + 1))) % k),((n + (n_count * (m + 1))) % k));
	}
    }
    printf ("%s\n", ")))");
     current_color_number = 9999;
}

/*----------------------------------------------------------------------*/ 

void print_closed_M_DXF_3dpolygon_mesh_as_P3D_mesh()	
/*
** print a closed DXF 3dpolygon mesh (in M direction) as P3D mesh
*/
{    
    int m , n, k;

    k = mesh_vertices_num;
    printf ("%s", "(");
    printf ("%s", "dxfmesh");
    printf ("%s", " '(");
    /* print 3d mesh color number */
    print_current_color();
    /* print the number of vertices in the 3D mesh */
    printf ("%d\n ", mesh_vertices_num);
    /* print the vertex coordinates of the 3D mesh */
    printf ("%s", "(");
    for (m=0; m<=(m_count - 1); m++){
	for (n=0; n<=(n_count - 1); n++){
	    printf("(%f %f %f)",pline_points[m][n].x, pline_points[m][n].y, pline_points[m][n].z); 
	}
	printf("\n");
    }
    printf ("%s\n", ")");
    /* print the polygon vertices of the 3D mesh */
    printf ("%s", "(");
    for (m=0; m<=(m_count - 1); m++){
	for (n=0; n<=(n_count - 2); n++){
	    printf("(%d %d %d)",(((n + 1) + (n_count * m )) % k),((n + (n_count * (m + 1))) % k),((n + (n_count * m)) % k));
	    printf("(%d %d %d)\n",(((n + 1) + (n_count * m )) % k),(((n + 1) + (n_count * (m + 1))) % k),((n + (n_count * (m + 1))) % k));
	}
    }
    printf ("%s\n", ")))");
     current_color_number = 9999;
}

/*----------------------------------------------------------------------*/ 

void print_closed_M_N_DXF_3dpolygon_mesh_as_P3D_mesh()	
/*
** print a closed DXF 3dpolygon mesh (in M and N directions) as P3D mesh
*/
{    
    int m , n, k;

    k = mesh_vertices_num;
	printf ("%s", "(");
	printf ("%s", "dxfmesh");
	printf ("%s", " '(");
	/* print 3d mesh color number */
	print_current_color();
	/* print the number of vertices in the 3D mesh */
	printf ("%d\n ", mesh_vertices_num);
	/* print the vertex coordinates of the 3D mesh */
	printf ("%s", "(");
	for (m=0; m<=(m_count - 1); m++){
	    for (n=0; n<=(n_count - 1); n++){
		printf("(%f %f %f)",pline_points[m][n].x, pline_points[m][n].y, pline_points[m][n].z); 
	    }
	    printf("\n");
	}
	printf ("%s\n", ")");
	/* print the polygon vertices of the 3D mesh */
	printf ("%s", "(");
	for (m=0; m<=(m_count - 1); m++){
	    for (n=0; n<=(n_count - 1); n++){
		printf("(%d %d %d)",(((n + 1) + (n_count * m )) % k),((n + (n_count * (m + 1))) % k),((n + (n_count * m)) % k));
		printf("(%d %d %d)\n",(((n + 1) + (n_count * m )) % k),(((n + 1) + (n_count * (m + 1))) % k),((n + (n_count * (m + 1))) % k));
	    }
	}
	printf ("%s\n", ")))");
	 current_color_number = 9999;
}

/*----------------------------------------------------------------------*/ 

void print_open_DXF_3dpolygon_mesh_as_P3D_plines()	
/*
** print an open DXF 3dpolygon mesh as P3D plines
*/
{                                                                                                                                                                                       
    int m , n;
    for (m=0; m<=(m_count - 1); m++){
	printf ("%s", "(");
	printf ("%s", "pline");
	printf ("%s", " '(");
	
	print_current_color();
	for (n=0; n<=(n_count - 1); n++){
	    printf("(%f %f %f)",pline_points[m][n].x, pline_points[m][n].y, pline_points[m][n].z); 
	}
	printf ("%s\n", "))");
    }
    
    for (n=0; n<=(n_count - 1); n++){
	printf ("%s", "(");
	printf ("%s", "pline");
	printf ("%s", " '(");
	print_current_color();
	for (m=0; m<=(m_count - 1); m++){
	    printf("(%f %f %f)",pline_points[m][n].x, pline_points[m][n].y, pline_points[m][n].z); 
	}
	printf ("%s\n", "))");
    }  
     current_color_number = 9999;
}

/*----------------------------------------------------------------------*/ 

void print_closed_M_DXF_3d_polygon_mesh_as_P3D_plines()	
/*
** print a  DXF 3d polygon mesh (closed in M Direction) as P3D plines
*/
{    
    int m , n;
    for (m=0; m<=(m_count - 1); m++){
	printf ("%s", "(");
	printf ("%s", "pline");
	printf ("%s", " '(");

	print_current_color();
	for (n=0; n<=(n_count -1); n++){
	    printf("(%f %f %f)",pline_points[m][n].x, pline_points[m][n].y, pline_points[m][n].z); 
	}
	printf ("%s\n", "))");
    }

    for (n=0; n<=(n_count - 1); n++){
	printf ("%s", "(");
	printf ("%s", "pline");
	printf ("%s", " '(");
	print_current_color();
	for (m=0; m<=m_count; m++){
	    printf("(%f %f %f)",pline_points[m % m_count][n].x, pline_points[m % m_count][n].y, pline_points[m % m_count][n].z); 
	}
	printf ("%s\n", "))");
    }
     current_color_number = 9999;
}

/*----------------------------------------------------------------------*/ 

void print_closed_M_N_DXF_3dpolygon_mesh_as_P3D_plines()	
/*
**  print a  DXF 3d polygon mesh (closed in M and Directions) as P3D plines
*/
{    
    int m , n;
    for (m=0; m<=(m_count - 1); m++){
	printf ("%s", "(");
	printf ("%s", "pline");
	printf ("%s", " '(");

	print_current_color();
	for (n=0; n<=n_count; n++){
	    printf("(%f %f %f)",pline_points[m][n % n_count].x, pline_points[m][n % n_count].y, pline_points[m][n % n_count].z); 
	}
	printf ("%s\n", "))");
    }

    for (n=0; n<=(n_count - 1); n++){
	printf ("%s", "(");
	printf ("%s", "pline");
	printf ("%s", " '(");
	print_current_color();
	for (m=0; m<=m_count; m++){
	    printf("(%f %f %f)",pline_points[m % m_count][n].x, pline_points[m % m_count][n].y, pline_points[m % m_count][n].z); 
	}
	printf ("%s\n", "))");
    }
    current_color_number = 9999;
   
}

/*----------------------------------------------------------------------*/ 

void read_polyline()
/*
** read a DXF POLYLINE data
** store its layer name, color, bit-coded flag
** and the number of vertices in M and N directions
** ignore all other data, such as line type.
*/
{

    int old_color_number;
   

    /*  the new color number.  They could be the same number if both the input file and */
    /* color file use the same number of colors. */

    for (;;){
	getword(word,MAXLINE);
	/* break the loop at the first vertex */
	if(strcmp(word,"VERTEX")==0){ 
	    break;
	}
	/* read the layer name and store it */
	if(strcmp(word,"8")==0){ 
	    getword(word,MAXLINE);
	    strcpy(layername, word);
	}

	/* read the line type and do nothing with it */
	if(strcmp(word,"6")==0){ 
	    getword(word,MAXLINE);	    
	}

	/* read the Z elevation and store it in z_flatland */
	if(strcmp(word,"38")==0){ 
	    getword(word,MAXLINE);		
	    z_flatland= atof(word);
	}

	/* read the Z thickness and do nothing with it*/
	if(strcmp(word,"39")==0){ 
	    getword(word,MAXLINE);
	    z_thickness=atof(word);
	}

	if(strcmp(word,"62")==0){ 
	    getword(word,MAXLINE);	    /* read color information */
	    old_color_number = atoi(word);
	    current_color_number = calc_color (old_color_number);
	}

	/* read the flag and do nothing with it */
	if(strcmp(word,"66")==0){ 
	    getword(word,MAXLINE);	    
	}
	/* read the "Paper Space entity record and do nothing with it */
	if(strcmp(word,"67")==0){ 
	    getword(word,MAXLINE);	    
	}

	/* get the polyline flag */
	if(strcmp(word,"70")==0){ 
	    getword(word,MAXLINE);             
	    polyline_flag = atoi(word);
	}
	/* Polygon mesh M count */
	if(strcmp(word,"71")==0){ 
	    getword(word,MAXLINE);		
	    m_count = atoi(word);
	}
	/* Polygon mesh N count */
	if(strcmp(word,"72")==0){ 
	    getword(word,MAXLINE);		
	    n_count = atoi(word);
	}

	/* read the Polygon mesh smooth surface M density*/
	/* and do nothing about it */
	if(strcmp(word,"73")==0){ 
	    getword(word,MAXLINE);		
	}

	/* read the Polygon mesh smooth surface N density*/
	/* and do nothing about it */
	if(strcmp(word,"74")==0){ 
	    getword(word,MAXLINE);		
	}

	/* read the smooth surface type flag*/
	/* and do nothing about it */
	if(strcmp(word,"75")==0){ 
	    getword(word,MAXLINE);		
	}

	/* read the offset Z extrusion X coordinate and do nothing with it */
	if(strcmp(word,"210")==0){ 
	    getword(word,MAXLINE);	    
	}
	/* read the offset Z extrusion Y coordinate and do nothing with it */
	if(strcmp(word,"220")==0){ 
	    getword(word,MAXLINE);	    
	}
	/* read the offset Z extrusion Z coordinate and do nothing with it */
	if(strcmp(word,"230")==0){ 
	    getword(word,MAXLINE);	    
	}
    }
    /* calculate the number of vertices in a 3D mesh */
    mesh_vertices_num = (m_count * n_count);
}

/*----------------------------------------------------------------------*/ 

void print_polyline_as_lines()
/*
** print a DXF POLYLINE
** call the appropriate function acoording
** to its flag.
** print the Polyline in P3d plines.
*/
{
    /* a closed polyline (polyline_flag of value "1")*/
    if(polyline_flag ==((b<<1)/2)){
	read_DXF_3dpolyline();
	print_closed_DXF_3dpolyline_as_P3D_pline();
    }
    /* an open 3D polyline (polyline_flag of value "8")*/
    if (polyline_flag ==((b<<4)/2)){
	read_DXF_3dpolyline();
	print_open_DXF_3dpolyline_as_P3D_pline();
    }
    /* a closed 3D polyline (polyline_flag of value "9")*/
    if (polyline_flag ==(((b<<4)|(b<<1))/2)){
	read_DXF_3dpolyline();
	print_closed_DXF_3dpolyline_as_P3D_pline();
    }
    /* an open 3D polygon mesh (polyline_flag of value "16") */
    if (polyline_flag == ((b<<5)/2)){
	read_DXF_3dpolygon_mesh();
	print_open_DXF_3dpolygon_mesh_as_P3D_plines();
    } 
    /* a closed 3D polygon mesh in m direction (polyline_flag of value "17") */
    if (polyline_flag == (((b<<5)|(b<<1))/2)){
	read_DXF_3dpolygon_mesh();
	print_closed_M_DXF_3d_polygon_mesh_as_P3D_plines();
    } 
    /* a closed 3D polygon mesh in both m and n directions */
    /*(polyline_flag of value "49") */
    if (polyline_flag ==(((b<<6)|(b<<5)|(b<<1))/2)){
	read_DXF_3dpolygon_mesh();
	print_closed_M_N_DXF_3dpolygon_mesh_as_P3D_plines();
    } 
   
    z_flatland = 0.0;
    current_color_number = 9999;
}

/*----------------------------------------------------------------------*/
	    
void print_polyline_as_meshes()
/*
** print a DXF POLYLINE
** call the appropriate function acoording
** to its flag.
** print the Polyline as P3d mesh
*/
{
    /* a closed polyline (polyline_flag of value "1")*/
    if(polyline_flag ==((b<<1)/2)){
	read_DXF_3dpolyline();
	print_closed_DXF_3dpolyline_as_P3D_3dface();
    }
    /* an open 3D polyline (polyline_flag of value "8")*/
    if (polyline_flag ==((b<<4)/2)){
	read_DXF_3dpolyline();
	print_closed_DXF_3dpolyline_as_P3D_3dface();
    }
    /* a closed 3D polyline (polyline_flag of value "9")*/
    if (polyline_flag ==(((b<<4)|(b<<1))/2)){
	read_DXF_3dpolyline();
	print_closed_DXF_3dpolyline_as_P3D_3dface();
    }
    /* an open 3D polygon mesh (polyline_flag of value "16") */
    if (polyline_flag == ((b<<5)/2)){
	read_DXF_3dpolygon_mesh();
	print_open_3dpolygon_mesh_as_P3D_mesh();
    } 

    /* a closed 3D polygon mesh in m direction (polyline_flag of value "17") */
    if (polyline_flag == (((b<<5)|(b<<1))/2)){
	read_DXF_3dpolygon_mesh();
	print_closed_M_DXF_3dpolygon_mesh_as_P3D_mesh();

    } 
    /* a closed 3D polygon mesh in both m and n directions */
    /*(polyline_flag of value "49") */
    if (polyline_flag ==(((b<<6)|(b<<5)|(b<<1))/2)){
	read_DXF_3dpolygon_mesh();
	print_closed_M_N_DXF_3dpolygon_mesh_as_P3D_mesh();
    } 
   
    z_flatland = 0.0;
    current_color_number = 9999;
}
	    
/*----------------------------------------------------------------------*/

void print_termination()
/* print the termination of a "P3D" file */
{
       printf("))) ; to close (setq model)\n");    
}

/*----------------------------------------------------------------------*/
	  
void get_lookat_point()
/* get the "lookat" point which is the center of the bounding box */ 
{
    x_lookat = (x_max - x_min)/2;
    y_lookat = (y_max - y_min)/2;
    z_lookat = (z_max - z_min)/2;
}

/*----------------------------------------------------------------------*/

void get_lookfrom_point()
/* 
** get the "lookfrom" point:
** multiply the length of each dim of the bounding box by a  factor (default is 3.0) .
** add the result to the values of the center of the bounding box.
*/
{ 
    double x_length;
    double y_length;
    double z_length;
 
    x_length = x_max - x_min;                /* the x length of the bounding box */
    y_length = y_max - y_min;		     /* the y length of the bounding box */
    z_length = z_max - z_min;		     /* the z length of the bounding box */
    x_lookfrom = x_lookat + (x_view_factor *  x_length);
    y_lookfrom = y_lookat + (y_view_factor *  y_length);
    z_lookfrom = z_lookat + (z_view_factor *  z_length);
}

/*----------------------------------------------------------------------*/

void get_hither()
/* 
**.get the "hither" clipping distance 
** the hither distance is the negative of half the distance from the "lookfrom"
** point to the bounding box.
*/
{
    double dist;

    dist = calc_distance(x_lookfrom,y_lookfrom,z_lookfrom,x_max,y_max,z_max);
    hither = -0.25 * dist;
}
 
/*----------------------------------------------------------------------*/

void get_yon()
/*
** get the "yon" clipping distance.
** the yon distance is the negative of 1.5 times the distance from the "lookfrom"
** point to the far side of the bounding box.
*/
{
    double distance;

    distance = calc_distance(x_lookfrom,y_lookfrom,z_lookfrom, x_min,y_min,(z_min/2));
    yon = -1.5 * distance;
}

/*----------------------------------------------------------------------*/  

void print_viewing_points()
/* print all viewing points*/
{
    printf("(setq lights\n");						
    printf("	(def-gob\n");
    printf("		:children (list\n");
    printf("		(light (make-point :x 10000.0 :y 10000.0 :z 100000.0)\n");
    printf("		      (make-color :r 0.8 :g 0.8 :b 0.8))\n");
    printf("		(ambient (make-color :r 0.5 :g 0.5 :b 0.5)) )))\n");
    printf("(setq this-camera (make-camera\n");
    printf("	:lookfrom (make-point :x %f :y %f :z %f)\n", x_lookfrom, y_lookfrom, z_lookfrom);
    printf("	:lookat (make-point :x %f :y %f :z %f)\n", x_lookat, y_lookat, z_lookat);
    printf(":fovea 45.0\n");
    printf(":up z-vec\n");
    printf(":hither %f\n", hither);
    printf(":yon %f))\n", yon);
    printf("(snap model lights this-camera)\n");
    printf (";Total number of DXF 3DFACEs is %d\n", DXF_3dfaces_num);
    printf (";Total number of DXF LINEs is %d\n", DXF_lines_num);
    printf (";Total number of DXF POLYLINEs is %d\n", DXF_polylines_num );
}

/*----------------------------------------------------------------------*/

void read_input_file_print_lines()
/*
**  read through the DXF input file.
** call the appropriate functions according
** to the main target entities:
** 3DFACE, LINE, and POLYLINE
** get layers data also when reading first "LAYER".
** print all entities in P3D plines
*/	
{
    while ( !feof(DataFile) ) {
        getword(word,MAXLINE);
	if(strcmp(word,"LAYER")==0){ 
	    get_layers_color();
	}
	if(strcmp(word,"3DFACE")==0){ 
	    read_DXF_3dface();
	    print_DXF_3dface_as_P3D_pline();
	    DXF_3dfaces_num++;
	}
	if(strcmp(word,"LINE")==0){ 
	    read_DXF_line();
	    print_DXF_line_as_P3D_pline();
	    DXF_lines_num++;
	}
	if(strcmp(word,"POLYLINE")==0){
	    read_polyline();
	    print_polyline_as_lines();
	    DXF_polylines_num++;
	}
	
    }
}

/*----------------------------------------------------------------------*/

void read_input_file_print_meshes()
/*
**  read through the DXF input file.
** call the appropriate functions according
** to the main target entities:
** 3DFACE, LINE, and POLYLINE
** get layers data also when reading first "LAYER".
** print all entities in P3D meshes and 3dfaces
** (as possible as non extruded DXf lines
** will still be printed in P3D Plines
*/	
{
    while ( !feof(DataFile) ) {
        getword(word,MAXLINE);
	if(strcmp(word,"LAYER")==0){ 
	    get_layers_color();
	}
	if(strcmp(word,"3DFACE")==0){ 
	    read_DXF_3dface();
	    print_DXF_3dface_as_P3D_3dface();
	    DXF_3dfaces_num++;
	}
	if(strcmp(word,"LINE")==0){ 
	    read_DXF_line();
	    print_extruded_DXF_line_as_P3D_3faces();
	    DXF_lines_num++;
	}
	if(strcmp(word,"POLYLINE")==0){
	    read_polyline();
	    print_polyline_as_meshes();
	    DXF_polylines_num++;
	}
	
    }
}

/*----------------------------------------------------------------------*/

void main(int argc, char **argv) 

/* 
** Open the DXF input file
** print the header of the P3D file
** print the color-array of the P3D file
** print the P3D function definitions
** print the 3D model:
** choose "read_input_file_print_lines" to
** print all DXF entities in P3D plines
** choose "read_input_file_print_meshes" to
** print all (almost) entities in P3D meshes and 3dfaces
** (non extruded DXf lines
** will still be printed in P3D Plines)
** print the termination of the P3D file
** calculate the viewing information:
** lookat and lookfrom points
** hither and yon distances
** finally print all viewing information
*/

{   
    if (argc != 2) {
	printf("Usage: %s file.dxf\n", argv[0]);
	exit(-2);
    }
    InitRead(argv[1]); 
    print_header();
    print_colors();
    print_definitions();

    /* switch the next two lines if neccessary */
#ifdef LINE_MESHES
    read_input_file_print_lines();
#else
    read_input_file_print_meshes();
#endif

    print_termination ();
    get_lookat_point();
    get_lookfrom_point();
    get_hither();
    get_yon();
    print_viewing_points(); 
}
	
	


