/*
*	interpolate.c
*
*	Tim McClarren				Dec. 1991
*
*/

#include <stdlib.h>
#include <string.h>
#include "portable.h"

/**********************************************************************
*  Function :   interpolate_float
*  Purpose  :   Interpolate an image by independant X and Y magnification
*                   factors.  Note that the coordinates which specify
*                   the area in the "from" image are _inclusive_, i.e.
*                   if you would like to magnify the entire image, the
*                   "from" coordinates should be specified as:
*                   (0,0) and ((width-1),(height-1)), where "width"
*                   and "height" are the width and height of the image
*                   (respectively) in pixels.
*  Parameters   :
*           from_buffer - pointer to the buffer which contains the image
*                           to interpolate
*           to_buffer - pointer to the buffer in which to place the
*                           interpolated image
*           from_x0, from_y0 - Upper Left Corner (ULC) of the rectangular
*                           area in the image to interpolate
*           from_x1, from_y1 - Lower Right Corner (LRC) of the rectangular
*                           area in the image to interpolate
*           from_width, from_height - the width and height of the image
*                           to interpolate (or "from" image)
*           to_width, to_height - the width and height of the interpolated
*                           image (or "to" image)
*  Returns  :   (TRUE) for success, (FALSE) for error
*  Calls    :
*  Called by    :
**********************************************************************/
#ifdef __STDC__
boolean interpolate_float(uint8 *from_buffer,uint8 *to_buffer,uint16 from_x0,
    uint16 from_y0,uint16 from_x1,uint16 from_y1,uint16 from_width,
    uint16 from_height,uint16 to_width,uint16 to_height)
#else
boolean interpolate_float(from_buffer,to_buffer,from_x0,from_y0,from_x1,from_y1,
    from_width,from_height,to_width,to_height)
uint8 *from_buffer,
    *to_buffer;
uint16 from_x0,from_y0,from_x1,from_y1,
    from_width,from_height,
    to_width,to_height;
#endif
{
     uint8 	*fromOff,			/* offset into data block */
 			*bufOff,			/* current offset into interpolated data */
			*cellOff,			/* offset to cell in interpolated data */
			*lineOff;			/* offset to line in interpolated data */
    uint16 	windWidth,			/* the width of the window to magnify */
        	windHeight;			/* the height of the window to magnify */
	uint16 	*xSizes,			/* array of widths of cells */
			*ySizes,			/* array of heights of cells */
			total;
	uint8 	ulc,				/* upper left byte for cell */
			llc;				/* lower left byte for cell */
	float32	xSize,
			ySize,
			t1,t2,t3,			/* temp float values */
			c1,c2,c3;			/* precalculated floats */
    uint16 	u,v,w,x;			/* counting variables */

	/* make sure everything is valid */
    if (from_width == 0 || from_height == 0) return(FALSE);
    if (to_width == 0 || to_height == 0) return(FALSE);
    if (from_x0 > from_x1 || from_y0 > from_y1) return(FALSE);
	windWidth = (from_x1 - from_x0) + 1;  
	windHeight = (from_y1 - from_y0) + 1;
	xSizes = (uint16*) vmalloc( (uint16)(windWidth * sizeof(uint16)) );
	EXCHECK(xSizes == NULL,xSizesFailed);
	ySizes = (uint16*) vmalloc( (uint16)(windHeight * sizeof(uint16)) );
	EXCHECK(ySizes == NULL,ySizesFailed);
	/* 	calculate heights and widths of cells into xSizes and ySizes...
		these arrays hold scales for each pixel, so that, fer instance, 
		interpolating a 10 x 20 pixel wide source image to 60 x 40 pixels 
		will produce an xSizes array that has 10 entries (each equal to 6) 
		and ySizes array that has 20 entries (each equal to 2)...
		xSizes is as long as the width of the source image,
		ySizes is as long as the height of the source image */
	xSize = (float32)(to_width - 1) / (float32)(windWidth - 1);
	for (u = 0, total = 0; u < (windWidth - 1); u++) {
		xSizes[u] = ( (uint16)(((float32)(u+1) * xSize) + 0.5) - total);
		total += xSizes[u];
	}
	ySize = (float32)(to_height - 1) / (float32)(windHeight - 1);
	for (u = 0, total = 0; u < (windHeight - 1); u++) {
		ySizes[u] = ( (uint16)(((float32)(u+1) * ySize) + 0.5) - total);
		total += ySizes[u];
	}
	/* 	fromOff = first pixel in the source image
		bufOff = first pixel in the destination image */
	fromOff = from_buffer + (from_width * from_y0) + from_x0;
	bufOff = to_buffer;
	/*	start doing the interpolation...iterate over each 
		pixel in the source image */
	for (u = 0; u < (windHeight - 1); u++) {
		for (v = 0; v < (windWidth - 1); v++) {
			/*	the corner values for this cell will be the 
				four pixels in the source image */
			ulc = *fromOff;
			llc = *(fromOff + from_width);
			ySize = (float32)ySizes[u];
			xSize = (float32)xSizes[v];		
			c1 = (float32)((int)*(fromOff+1) - (int)ulc) / xSize;
			c2 = (((float32)((int)*(fromOff+from_width+1) - (int)llc) / xSize) - c1) / ySize;
			c3 = (float32)((int)llc - (int)ulc) / ySize;
			/*	save the pointer to this cell so we can move to the next
				one easily, save the pointer to this line so we can move
				down the cell easily */
			cellOff = bufOff;
			lineOff = bufOff;
			/* 	the following part calculates the Bytes across the first
				row of this cell...it's here for efficiency because we
				know that for the first row in the cell our vertical distance
				down the cell is zero and so we don't need to do bilinear, 
				so this avoids some unnecessary multiplies */
			t1 = ulc;
			for (x = 0; x < (uint16)xSize; x++) {
				*bufOff++ = (uint8)t1;
				t1 += c1;
			}
			/* reset pointer to next row */
			bufOff = lineOff + to_width;
			/* do the interpolation for the remaining part of the cell */
			for (w = 1; w < (uint16)ySize; w++) {
				lineOff = bufOff;
				t1 = (float32)ulc + (c3 * (float32)w);
				t2 = (c2 * (float32)w) + c1;
				t3 = t1;
				for (x = 0; x < (uint16)xSize; x++) {
					*bufOff++ = (uint8)t3;
					t3 += t2;
				}
				/* reset pointer to next row */
				bufOff = lineOff + to_width;
			}
			/* reset the pointer to the next cell */
			bufOff = cellOff + (uint32)xSize;
			/* one cell = one pixel in source image, so move to the next one */
			fromOff++;
		}
		/* we've reach the end of this row of cells, so move down to the next */
		bufOff += (((uint32)ySize - 1) * to_width) + 1;
		/* move source pointer down a row in the source image */
		fromOff += (from_width - (windWidth - 1));
	}
	/* fill in the last column along the right hand side */
	bufOff = to_buffer + (to_width - 1);
	fromOff = from_buffer + (from_width * from_y0) + from_x0 + (windWidth - 1);
	for (u = 0; u < (windHeight - 1); u++) {
		ulc = *fromOff;
		ySize = ySizes[u];
		c1 = (float32)((int)*(fromOff + from_width) - (int)ulc) / ySize;			
		t1 = ulc;
		for (w = 0; w < ySize; w++)	{
			*bufOff = (uint8)t1;
			bufOff += to_width;
			t1 += c1;
		}
		fromOff += from_width;
	}
	/* fill in the last row along the bottom */
	bufOff -= (to_width - 1);
	fromOff -= (windWidth - 1);
	for (v = 0; v < (windWidth - 1); v++) {
		ulc = *fromOff;
		xSize = xSizes[v];
		c1 = (float32)((int)*(fromOff + 1) - (int)ulc) / xSize;
		t1 = ulc;
		for (x = 0; x < xSize; x++) {
			*bufOff++ = (uint8)t1;
			t1 += c1;
			}
		fromOff++;
	}
	/* set the last element in the last row in the destination */
	*bufOff = *fromOff;
	vfree((char*)ySizes);
	vfree((char*)xSizes);
	return (TRUE);
	
	/* if we've failed allocation of ySizes, free up xSizes and return */
ySizesFailed:
	vfree((char*)xSizes);
	/* if we've failed allocation of xSizes, nothing to clean up */
xSizesFailed:
	return (FALSE);
}


#if 0				/* don't use this part for collage */

struct PortPoint {
	uint16				h;
	uint16				v;
};
typedef struct PortPoint PortPoint;

struct Bresenham {
	PortPoint			currPoint;
	PortPoint			fromPoint;
	PortPoint			toPoint;
	uint16				p;
	uint16				yIncrement;
	uint16				twoDeltaY;
	uint16				twoDeltaYminusDeltaX;
};
typedef struct Bresenham Bresenham;

#define SetPortPoint(point,hp,vp) \
	{ (point).h = (hp); (point).v = (vp); }

#define SetBresenham(bres,fromPointP,toPointP) \
	{	uint16 dx; uint16 dy; \
\
		dx = abs((fromPointP).h - (toPointP).h); \
		dy = abs((fromPointP).v - (toPointP).v); \
\
		SetPortPoint((bres).fromPoint,(fromPointP).h,(fromPointP).v); \
		SetPortPoint((bres).toPoint,(toPointP).h,(toPointP).v); \
		SetPortPoint((bres).currPoint,(fromPointP).h,(fromPointP).v); \
		(bres).twoDeltaY = 2 * dy; \
		(bres).twoDeltaYminusDeltaX = 2 * (dy - dx); \
		(bres).p = (bres).twoDeltaY - dx; \
		(bres).yIncrement = ((fromPointP).v > (toPointP).v) ? -1 : 1; \
	}

#define AdvanceBresenham(bres) \
	{ \
		((bres).currPoint.h)++; \
		if ((bres).p < 0) \
			(bres).p += (bres).twoDeltaY; \
		else \
			{ \
				(bres).currPoint.v += (bres).yIncrement; \
				(bres).p += (bres).twoDeltaYminusDeltaX; \
			} \
	}
	
/* can you say 'macro'? */

/**********************************************************************
*  Function :   interpolate_int
*  Purpose  :   Interpolate an image by independant X and Y magnification
*                   factors.  Note that the coordinates which specify
*                   the area in the "from" image are _inclusive_, i.e.
*                   if you would like to magnify the entire image, the
*                   "from" coordinates should be specified as:
*                   (0,0) and ((width-1),(height-1)), where "width"
*                   and "height" are the width and height of the image
*                   (respectively) in pixels.
*  Parameters   :
*           from_buffer - pointer to the buffer which contains the image
*                           to interpolate
*           to_buffer - pointer to the buffer in which to place the
*                           interpolated image
*           from_x0, from_y0 - Upper Left Corner (ULC) of the rectangular
*                           area in the image to interpolate
*           from_x1, from_y1 - Lower Right Corner (LRC) of the rectangular
*                           area in the image to interpolate
*           from_width, from_height - the width and height of the image
*                           to interpolate (or "from" image)
*           to_width, to_height - the width and height of the interpolated
*                           image (or "to" image)
*  Returns  :   (TRUE) for success, (FALSE) for error
*  Calls    :
*  Called by    :
**********************************************************************/
#ifdef __STDC__
boolean interpolate_int(uint8 *from_buffer,uint8 *to_buffer,uint16 from_x0,
    uint16 from_y0,uint16 from_x1,uint16 from_y1,uint16 from_width,
    uint16 from_height,uint16 to_width,uint16 to_height)
#else
boolean interpolate_int(from_buffer,to_buffer,from_x0,from_y0,from_x1,from_y1,
    from_width,from_height,to_width,to_height)
uint8 *from_buffer,
    *to_buffer;
uint16 from_x0,from_y0,from_x1,from_y1,
    from_width,from_height,
    to_width,to_height;
#endif
{
    uint8 		*fromOff,			/* offset into data block */
 				*bufOff,			/* current offset into interpolated data */
				*cellOff,			/* offset to cell in interpolated data */
				*lineOff;			/* offset to line in interpolated data */
    uint16 		windWidth,			/* the width of the window to magnify */
        		windHeight;			/* the height of the window to magnify */
	uint16 		*xSizes,			/* array of widths of cells */
				*ySizes,			/* array of heights of cells */
				dx,					/* x distance for cell */
				dy;					/* y distance for cell */
	uint8 		ulc,				/* upper left byte for cell */
				llc,				/* lower left byte for cell */
				urc,				/* upper right byte for cell */
				lrc;				/* lower right byte for cell */
	Bresenham 	v1Bres,				/* Bresenham along the left col of cell */
				v2Bres,				/* Bresenham along the right col of cell */
				h1Bres;				/* Bresenham between v1Bres and v2Bres */
	PortPoint	firstPoint,			/* temp points to set up Bresenham structures */
				secPoint;
    uint16 		u,v;				/* counting variables */
	float		xSize,
				ySize;

	/* make sure everything is valid */
    if (from_width == 0 || from_height == 0) return(FALSE);
    if (to_width == 0 || to_height == 0) return(FALSE);
    if (from_x0 > from_x1 || from_y0 > from_y1) return(FALSE);
	windWidth = (from_x1 - from_x0) + 1;  
	windHeight = (from_y1 - from_y0) + 1;
	xSizes = (uint16*) vmalloc( (uint16)(windWidth * sizeof(uint16)) );
	EXCHECK(xSizes == NULL,xSizesFailed);
	ySizes = (uint16*) vmalloc( (uint16)(windHeight * sizeof(uint16)) );
	EXCHECK(ySizes == NULL,ySizesFailed);
	/* 	calculate heights and widths of cells into xSizes and ySizes...
		these arrays hold scales for each pixel, so that, fer instance, 
		interpolating a 10 x 20 pixel wide source image to 60 x 40 pixels 
		will produce an xSizes array that has 10 entries (each equal to 6) 
		and ySizes array that has 20 entries (each equal to 2)...
		xSizes is as long as the width of the source image,
		ySizes is as long as the height of the source image */
	xSize = (float32)(to_width - 1) / (float32)(windWidth - 1);
	for (u = 0, dx = 0; u < (windWidth - 1); u++) {
		xSizes[u] = ( (uint16)(((float32)(u+1) * xSize) + 0.5) - dx);
		dx += xSizes[u];
	}
	ySize = (float32)(to_height - 1) / (float32)(windHeight - 1);
	for (u = 0, dy = 0; u < (windHeight - 1); u++) {
		ySizes[u] = ( (uint16)(((float32)(u+1) * ySize) + 0.5) - dy);
		dy += ySizes[u];
	}
	/* 	fromOff = first pixel in the source image
		bufOff = first pixel in the destination image */
	fromOff = from_buffer + (from_width * from_y0) + from_x0;
	bufOff = to_buffer;
	/*	start doing the interpolation...iterate over each 
		pixel in the source image */
	for (u = 0; u < (windHeight - 1); u++) {
		for (v = 0; v < (windWidth - 1); v++) {
			/*	the corner values for this cell will be the 
				four pixels in the source image */
			ulc = *fromOff;
			urc = *(fromOff + 1);
			llc = *(fromOff + from_width);
			lrc = *(fromOff + from_width + 1);
			dy = ySizes[u];
			dx = xSizes[v];
			SetPortPoint(firstPoint,0,ulc); /* can you say 'expanded macros'? */
			SetPortPoint(secPoint,dy,urc);
			SetBresenham(v1Bres,firstPoint,secPoint);
			SetPortPoint(firstPoint,0,llc);
			SetPortPoint(secPoint,dy,lrc);
			SetBresenham(v2Bres,firstPoint,secPoint);
			/*	save the pointer to this cell so we can move to the next
				one easily, save the pointer to this line so we can move
				down the cell easily */
			cellOff = bufOff;
			/* do the interpolation for the cell */
			while (v1Bres.currPoint.h != v1Bres.toPoint.h) {
				lineOff = bufOff;
				SetPortPoint(firstPoint,0,v1Bres.currPoint.v);
				SetPortPoint(secPoint,dx,v2Bres.currPoint.v);
				SetBresenham(h1Bres,firstPoint,secPoint);
				while (h1Bres.currPoint.h != h1Bres.toPoint.h) {
					*bufOff++ = (uint8)h1Bres.currPoint.v;
					AdvanceBresenham(h1Bres);
				}
				/* reset pointer to next row */
				bufOff = lineOff + to_width;
				AdvanceBresenham(v1Bres);
				AdvanceBresenham(v2Bres);
			}
			/* reset the pointer to the next cell */
			bufOff = cellOff + (uint32)dx;
			/* one cell = one pixel in source image, so move to the next one */
			fromOff++;
		}
		/* we've reach the end of this row of cells, so move down to the next */
		bufOff += (((uint32)dy - 1) * to_width) + 1;
		/* move source pointer down a row in the source image */
		fromOff += (from_width - (windWidth - 1));
	}
	/* fill in the last column along the right hand side */
	bufOff = to_buffer + (to_width - 1);
	fromOff = from_buffer + (from_width * from_y0) + from_x0 + (windWidth - 1);
	for (u = 0; u < (windHeight - 1); u++)
		{
			ulc = *fromOff;
			llc = *(fromOff + from_width);
			dy = ySizes[u];
			SetPortPoint(firstPoint,0,ulc);
			SetPortPoint(secPoint,dy,llc);
			SetBresenham(v1Bres,firstPoint,secPoint);
			while (v1Bres.currPoint.h != v1Bres.toPoint.h) {
				*bufOff = (uint8)v1Bres.currPoint.v;
				 bufOff += to_width;
				AdvanceBresenham(v1Bres);
			}
			fromOff += from_width;
		}
	/* fill in the last row along the bottom */
	bufOff -= (to_width - 1);
	fromOff -= (windWidth - 1);
	for (v = 0; v < (windWidth - 1); v++)
		{
			ulc = *fromOff;
			urc = *(fromOff + 1);
			dx = xSizes[v];
			SetPortPoint(firstPoint,0,ulc);
			SetPortPoint(secPoint,dx,urc);
			SetBresenham(h1Bres,firstPoint,secPoint);
			while (h1Bres.currPoint.h != h1Bres.toPoint.h) {
				*bufOff++ = (uint8)h1Bres.currPoint.h;
				AdvanceBresenham(h1Bres);
			}
			fromOff++;
		}
	/* set the last element in the last row in the destination */
	*bufOff = *fromOff;
	
	vfree((char*)ySizes);
	vfree((char*)xSizes);
	return (TRUE);
	
	/* if we've failed allocation of ySizes, free up xSizes and return */
ySizesFailed:
	vfree((char*)xSizes);
	/* if we've failed allocation of xSizes, nothing to clean up */
xSizesFailed:
	return (FALSE);
}
#endif /* 0 */
