/*
 File: Voronoi.c
 Authors: K.R. Sloan
 Last Modified: 28 August 1992
 Purpose: Reads a set of x,y points,
          calculates the Delaunay triangulation
          and displays the Voronoi diagram,
          via a sgp graphics meta-file.
 NOTE: this version reads the first two numbers on each line,
       and discards the remainder of the line.  This allows us 
       to look at point sets which have information associated
       with each 2D point (like, say, a height)
 */
#include <stdio.h>
#include <math.h>
#include <DCEL.h>
#include <GrahamHull.h>
#include <Triangulate.h>

static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"Usage is\n\t%s [-h][-v][-z]\n", RoutineName);
 }

static int VERBOSE = 0;
static int DISCARDZ = -1;
static int LastPoint;
static double wL,wB,wR,wT, Cx, Cy, R;

 /*
   Read the data file and initialize
  */
static int ReadPoints(s)
 FILE *s;
 {
  int i;

  for(i=0;;i++)
   {
    if (   feof(s) || (2 != fscanf(s," %lf%*[ ,]%lf",
          &Points[i].x, &Points[i].y)) )
     {
      /*
        some programs have the convention that the last two points are
        (L,B) and (R,T) - to establish a particular window.  Look for this
        case, and fix it.
       */
      if ((wR == Points[i-1].x) && (wT == Points[i-1].y)) i--;
      if ((wL == Points[i-1].x) && (wB == Points[i-1].y)) i--;
      if ((wR == Points[i-1].x) && (wT == Points[i-1].y)) i--;
      if ((wL == Points[i-1].x) && (wB == Points[i-1].y)) i--;
       
      Cx = 0.5*(wL+wR); Cy = 0.5*(wB+wT);
      if ((wR-wL)>(wT-wB)) R = 0.5*(wR-wL); else R = 0.5*(wT-wB);
      wL = Cx - R;  wB = Cy - R;
      wR = Cx + R;  wT = Cy + R;
      return(i);
     }
    if (DISCARDZ) fscanf(s,"%*[^\n]");
    if ((0 == i) || (wL > Points[i].x)) wL = Points[i].x;
    if ((0 == i) || (wB > Points[i].y)) wB = Points[i].y;
    if ((0 == i) || (wR < Points[i].x)) wR = Points[i].x;
    if ((0 == i) || (wT < Points[i].y)) wT = Points[i].y;
    Points[i].Edges = (EdgePointer)0;
   }
 }

static void Display(s)
 FILE *s;
 {
  TrianglePointer T1, T2;
  EdgePointer E;
  double ax, ay, bx, by, ux, uy, vx, vy;
  int i;
  fprintf(s,"%%!\n%%%% created by %s\n\n",RoutineName);
  fprintf(s,"sgpColorDisplay sgpInit\n");
  fprintf(s,"%f %f %f %f sgpSetWindow\n",wL,wB,wR,wT);
  fprintf(s,"0.0 0.0 1.0 1.0 sgpSetViewport\n");  
  fprintf(s,"1 sgpGrayLevel\n");
  fprintf(s,"sgpClearScreen\n");

  fprintf(s,"\n%%%% Sites\n\n");
  fprintf(s,"0 sgpGrayLevel\n");

/*
  for (i=0;i<LastPoint;i++)
   fprintf(stdout,"[%f %f] sgpPoint\n",Points[i].x,Points[i].y);
 */
  for (i=0;i<LastPoint;i++)
   fprintf(stdout,"[%f %f] 3 sgpDot\n",Points[i].x,Points[i].y);

  fprintf(s,"\n%%%% Delaunay triangulation\n\n");
  fprintf(stdout,"0.5 sgpGrayLevel\n");
  for (i=0;i<LastPoint;i++)
   {
    E = Points[i].Edges;
    for(;E;)
     {
      if (E->Tail.V == i) /* eliminate duplicates */
       {
        fprintf(stdout,"[%f %f][%f %f] sgpLine\n",
                Points[E->Tail.V].x,Points[E->Tail.V].y,
                Points[E->Head.V].x,Points[E->Head.V].y);
       }
      if (E->Tail.V == i) E = E->Tail.Next; else E = E->Head.Next;
      if (E == Points[i].Edges) break;
     }
   }

  fprintf(s,"\n%%%% Voronoi diagram\n\n");
  fprintf(stdout,"[0.8 0.4 0.2] sgpColor\n");
  for (i=0;i<LastPoint;i++)
   {
    E = Points[i].Edges;
    for(;E;)
     {
      if (E->Tail.V == i)  /* eliminate duplicates */
       {
        if (  (T1 = E->Tail.T)
            &&(T2 = E->Head.T) )     /* need two triangles */
         {
           VoronoiPt(Points[T1->P1].x, Points[T1->P1].y,
                     Points[T1->P2].x, Points[T1->P2].y,
                     Points[T1->P3].x, Points[T1->P3].y,
                     &ux, &uy);
           VoronoiPt(Points[T2->P1].x, Points[T2->P1].y,
                     Points[T2->P2].x, Points[T2->P2].y,
                     Points[T2->P3].x, Points[T2->P3].y,
                     &vx, &vy); 
           fprintf(stdout,"[%f %f][%f %f] sgpLine\n", ux, uy, vx, vy);
         }
        else if (T1)
         {  /* on the hull... */
          double lx,ly,lw,vw;

          VoronoiPt(Points[T1->P1].x, Points[T1->P1].y,
                    Points[T1->P2].x, Points[T1->P2].y,
                    Points[T1->P3].x, Points[T1->P3].y,
                    &ux, &uy);
          ax = Points[E->Tail.V].x; ay = Points[E->Tail.V].y;
          bx = Points[E->Head.V].x; by = Points[E->Head.V].y;
          /* drop a perpendicular to the line between a and b */
          /* using homogenous coords (see N&S) */
          /* [lx,ly,lw] is the line */
          lx = ay-by;
          ly = bx-ay;
          lw = ax*by - ay*bx;
          /* the point on the line is [vx, vy, vw]
          vx = ly;
          vy = -lx;
          vw = lx*uy - ly*ux;
          /* watch for zero */

          /* it's not clear that this works, so... (watch carefully) */
          vw = 0.0;
          /* remove this when you are sure this works */


          if ( (-0.0000000001 < vw) && (vw < 0.0000000001))
           {
            vx = ux; vy = uy;	
           }
          else
           {
            /* in particular - if the Voronoi point is outside
               the triangle, this is wrong.

               the right idea is to figure out which way to go -
               either TOWARDS or AWAY from the line joining a and b

               but - I think there's something else wrong, and
               I don't have time to fix it...
             */

            vx = vx/vw; vy = vy/vw; 
           }
          fprintf(stdout,"[%f %f][%f %f] sgpLine\n", ux, uy, vx, vy);
         }
       }
      if (E->Tail.V == i) E = E->Tail.Next; else E = E->Head.Next;
      if (E == Points[i].Edges) break;
     }
   }
  fprintf(s,"\nsgpQuit\n\n%%%% -30-\n");
 }

int main (argc, argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed=0;

  Triangles = (TrianglePointer)0;
  Suspects = (EdgePointer)0;
  Hull = (PointSetPointer)0;

  RoutineName = argv[ArgsParsed++];

  while (ArgsParsed < argc)
   {
    if ('-' == argv[ArgsParsed][0])
     switch (argv[ArgsParsed++][1])
      {
       case 'z': DISCARDZ = -1; break;
       case 'v': VERBOSE = -1; break;
       default:
       case 'h': usage(); exit(-1);
      }
   }

  LastPoint = ReadPoints(stdin);
  if (VERBOSE) fprintf(stderr,"%s: Read %d points\n",RoutineName, LastPoint);
  Triangulate(LastPoint);
  if (VERBOSE) fprintf(stderr,"%s: Triangulation done\n",RoutineName);
  Display(stdout);

  exit(0);
 }
