#include <stdio.h>
#include <math.h>

#define EPSILON 1e-15

#define V2op2(X,op,Y) \
    do { X[0] op Y[0]; X[1] op Y[1]; } while (0)

#define V2op3(X,assign,Y,op,Z) \
    do { X[0] assign Y[0] op Z[0]; X[1] assign Y[1] op Z[1]; } while (0)

/*****************************************************************************
**  This routine reads in a stream of 2D polygon coordinates from the input
**  stream and returns 1 if the resulting closed polygon is convex, otherwise
**  it returns 0.  The parameter `stream' is a currently open input stream.
*****************************************************************************/

int ConvexP (stream)
    FILE *stream;
{
    auto int    looking = 2;		/* Loop Termination Status */
    auto int    factor = 0;		/* Turn Direction Factor */
    auto double A[2], B[2], C[2];	/* Current Vertices */
    auto double oldA[2], oldB[2];	/* First Two Vertices */
    auto int    xdir, ydir;		/* Current Coordinate Direction */
    auto int    xturns=0, yturns=0;	/* Coordinate Direction Changes */
    auto double dot;			/* Dot Product of Edge Norm and Edge */
    auto double U[2], V[2];		/* Edge Vectors AB and BC */

    /* Read in the first two vertices. */

    if (2 > fscanf (stream, "%lf%lf", B+0, B+1))  return 0;

    do {
	if (2 > fscanf (stream, "%lf%lf", C+0, C+1))  return 0;
    } while ((B[0] == C[0]) && (B[1] == C[1]));

    if (feof(stream)) return 0;

    V2op2 (oldA,=,B);		/* Save the first two vertices. */
    V2op2 (oldB,=,C);

    V2op3 (V,=,C,-,B);		/* Calculate the edge vector V0V1. */

    /* Set the X & Y coordinate directions if possible. */

    xdir = (B[0] < C[0]) ? 1 : (B[0] > C[0]) ? -1 : 0;
    ydir = (B[1] < C[1]) ? 1 : (B[1] > C[1]) ? -1 : 0;

    do	/* Shift the last two vertices and read in the next vertex. */
    {
	V2op2 (A,=,B);
	V2op2 (B,=,C);

	do
	{   switch (looking)
	    {   case 2:
		    if (2 > fscanf (stream, "%lf%lf", C+0, C+1))
		    {   looking = 1;
			V2op2 (C,=,oldA);
		    }
		    break;

		case 1:
		    looking = 0;
		    V2op2 (C,=,oldB);
		    break;
		
		case 0:
		    return 1;
	    }
	} while ((C[0] == B[0]) && (C[1] == B[1]));

	V2op2 (U,=,V);
	V2op3 (V,=,C,-,B);	/* Calculate the edge vector from B to C. */

	/* Get the dot product of the normal to edge AB and the vector BC.
	** Compare this result with the previous dot products.  As long as
	** the sign is the same as the previous ones (or zero), then
	** everything's cool, otherwise we found a dent in the polygon, so
	** return 0.  */

	dot = (U[0] * V[1]) - (U[1] * V[0]);
	if (fabs(dot) < EPSILON)
	    dot = 0.0;
	else
	{   if (!factor)
		factor = (dot < 0.0) ? -1 : 1;
	    else if ((factor < 0) ^ (dot < 0.0))
		return 0;
	}

	/* Check the X coordinate delta.  For a simple polygon, this can
	** change sign (direction) twice, but no more. */

#       define CHECK_DIRECTION(P,Q,dir,turns) \
	    do { \
		if (P < Q) \
		{   if (dir == -1) \
		    {   ++turns; \
			if (dot == 0.0) return 0; \
		    } \
		    dir =  1; \
		} \
		else if (P > Q) \
		{   if (dir ==  1) \
		    {   ++turns; \
			if (dot == 0.0) return 0; \
		    } \
		    dir = -1; \
		} \
	    } while (0)

	CHECK_DIRECTION (B[0],C[0], xdir, xturns);
	if (xturns > 2) return 0;

	CHECK_DIRECTION (B[1],C[1], ydir, yturns);
	if (yturns > 2) return 0;

    } while (looking);

    return 1;	/* All tests passed; polygon is simple and convex. */
}



int main ()
{
    if (ConvexP (stdin))
    {   printf ("Polygon is convex.\n");
	exit (0);
    }
    else
    {   printf ("Polygon is not convex.\n");
	exit (1);
    }
}
