/*
  File: devA/DEVICE.c
  Author: K.R. Sloan
          James Painter
  Last Modified: 10 February 1991
  Purpose: definitions for the generic graphics device used by SGP
           this version is for the adage using microcode support

  See also: devLine.c, devTriangle.c, devCircle.c, devClip.c.
           Some of the functions promised in DEVICE.h are implemented there.

           devPick always fails
 */
#include <stdio.h>

#include <sys/types.h>

#include "uEngine.h"
#include "SGP.h"
#include "DEVICE.h"

extern devClipLine();
extern devClipRectangle();


/*  Device state  - made global for convenience ... */
int devLeft, devBottom, devRight, devTop;
int devClipL, devClipB, devClipR, devClipT;
int devCurrentColor;
FILE *devOutputStream;
double devAspectRatio = 1.0;

/* Monochrome flag */
static int mono = 0;

/*
   devInit initializes the display, loading color maps if necessary
   There are two modes: color and mono.  

   devInit also fills in the "physical" size of the display

   In color mode, we have a 512x512 screen

   In mono mode, we can go to 1kx1k 

 */
int devInit(color, OutputStream)
 int color;
 FILE *OutputStream;
 {
  extern char *malloc();

  devOutputStream = OutputStream; /* but we don't use it... */

  /* Initialize the adage and set up the microcode support code */
  uEInitialize( (color == (int) ColorDisplay) ? LowRes : HiRes );
  if (color == (int) ColorDisplay)
   {
    mono = 0;
    devLeft = 0;  devBottom = 511; devRight = 511; devTop = 0; 
   }
  else
   {
    mono = 1;
    devLeft = 0;  devBottom = 1023; devRight = 1023; devTop = 0;
   }

  devSetClipWin(devLeft,devBottom,devRight,devTop);
  devCurrentColor = devColorCode(1.0, 1.0, 1.0);

  return(0); /* success */ 
 }

/*
  devClose insures that all operations are complete and closes up
 */
int devClose()
 {
  return uEFinalize();
 }

/*
  devSetClipWin establishes a clipping window
  restrictions such as (L<R), (B<T) must be enforced here!
 */
int devSetClipWin(L, B, R, T)
 int L, B, R, T;
 {
  if (L<R) { devClipL=L; devClipR=R; } else { devClipL=R; devClipR=L;} 
  if (B<T) { devClipB=B; devClipT=T; } else { devClipB=T; devClipT=B;} 

  if (devLeft < devRight)
   {
    if (devClipL < devLeft)   devClipL = devLeft;
    if (devClipR > devRight)  devClipR = devRight;
   }
  else
   {
    if (devClipL < devRight)  devClipL = devRight;
    if (devClipR > devLeft)   devClipR = devLeft;
   }

  if (devBottom < devTop)
   {
    if (devClipB < devBottom) devClipB = devBottom;
    if (devClipT > devTop)    devClipT = devTop;
   }
  else 
   {
    if (devClipB < devTop)    devClipB = devTop;
    if (devClipT > devBottom) devClipT = devBottom;
   }

  return(0); /* success */
 }

/*
   devClearScreen fills the entire screen with a particular color
 */
int devClearScreen(color)
 int color;
 {
  uEclear(color);
  return(0); /* success */
 }

/*
   the following routines draw primitive shapes on the physical device
   All of them clip against the device's clipping window.
   All of them paint in the device's current color
 */

/*
   Draw a line from (x1,y1) to (x2,y2) using Bresenham's algorithm.
   All coordinates are assumed to be expressed in raw device coordinates.
 */
int devLine(x1, y1, x2, y2)
 int x1, y1, x2, y2;		/* The coordinates of the endpoints */
 {
  int cmd[6];

  if (devClipLine(&x1,&y1,&x2,&y2)) return(0); /* success */

  uEline(x1,y1,x2,y2,devCurrentColor);
  return(0); /* success */
 }


/*
   devRect fills a rectangle - we assume this is a primitive
 */
int devRect(L, B, R, T)
 int L, B, R, T;
 {
  int cmd[6], tmp;

  if (devClipRectangle(&L, &B, &R, &T)) return(0); /* success */

  if ((L == R) && (B == T)) 
   {
    uEpoint(L, T, devCurrentColor);
   }
  else
   {
    if (L > R) { tmp = L; L = R; R = tmp; }
    if (T > B) { tmp = T; T = B; B = tmp; }
    uErect(L, T, R, B, devCurrentColor); /* note the reversal of T, B */
   }
  return(0); /* success */
 }

/*
 devDisc fills a disc - see devCircle.c
 */

/*
 devCircle draws a circle - see devCircle.c
 */

/*
 devSolidTriangle fills a triangle - see devTriangle.c
 */

/*
 devShadeTriangle shades a triangle
   with color codes specified at each vertex - see devTriangle.c
 */

/*
 devSetColor establishes the current color
 */
int devSetColor(color)
 int color;
 {
  devCurrentColor = color;
  return(0); /* success */
 }

/*
   devColorCode returns a framebuffer value - here we assume a full-color
   display, with 8-bits each of rgb, packed as:
            [00000000bbbbbbbbggggggggaaaaaaaa]
   OR, an 8-bit monochrome display, with 8-bits of i = [iiiiiiii]
   In this case, if the user program specifies a color as RGB, we convert
   to I using the CIE weights (0.30, 0.59, 0.11) 

 */ 
int devColorCode(R, G, B)
 double R,G,B; 
 {
  int devR, devG, devB;

  if (!mono) 
   {
    devR = 255.0 * R;
    devG = 255.0 * G;
    devB = 255.0 * B;
    return (  ((devR & 0xff)      )
	    | ((devG & 0xff) <<  8)
	    | ((devB & 0xff) << 16) );
   }
  else 
   {
    devR = ((0.30 * R)  + (0.59 * G) + (0.11 * B)) * 255.0;
    return (devR & 0xFF);
   }
 }

/*
  devColorDecode generates r,g,b values in [0.0, 1.0] 
 */
int devColorDecode(code, R, G, B)
 int code;
 double *R, *G, *B;
 {
  int devR, devG, devB;

  devR = (code      ) & 0xff;
  *R = (double)devR / 255.0;
  if (!mono) 
   {
    devG = (code >>  8) & 0xff;
    *G = (double)devG / 255.0;
    devB = (code >> 16) & 0xff;
    *B = (double)devB / 255.0;
   }
  else 
   {
    *G = *B = *R;
   }
  return (0);  /* success */
 }

int devPick(x,y,button)
 int *x,*y,*button;
 {
  *x = 0; *y = 0; *button = 0; return(-1); /* failure */
 }

/*       -30-         */

