/*
  File: sgpServer.c
  Authors: David Meyers,
           J. Painter,
           K.R. Sloan
  Last Modified: 15 May 1991
  Purpose: processes the "SGP" meta-file.  This is a very simple 
           PostScript-style format.  Elements in the file are either
           floating-point numbers, or keywords.  Comments are from '%'
           to end-of-line.  '[' and ']' are allowed, but ignored by this
           processor.  A "real" PostScript version requires them.

           Floating-point numbers are pushed onto a stack. 

           Almost every keyword is the name of a function exported by
           either SGP.h, 3D.h, or Lighting.h.  The "appropriate number"
           of numbers are popped from the stack, and the SGP function is
           called.

           NOTHING is pushed onto the stack as the result of a function
           call.  Therefore, there is little actual computation that
           can be done at run-time.  Changing this design decision should
           not be undertaken lightly!  Look carefully at sgpCurve, and
           sgpPolyLine. 

           The stack is ONLY used to accumulate parameters for immediate
           function calls.  Therefore, it is considered an ERROR if the
           stack is not empty immediately after a function call. 

           Almost all base level "SGP" calls can be generated by using the
           "Debug" display.  The function sgpCurve, and the
           3D, and Lighting packages do
           not emit meta-output; rather, they compute down to raw 2D SGP
           calls, which emit meta-output. 

           The higher level interfaces are included here for convenience.
           It MAY be useful to directly write the PostScript style.
            
           It is POSSIBLE that the higher level C routines will be modified 
           to notice that the Debug display is in operation, and emit 
           meta-output directly.  But don't count on it...

*/

#include <stdio.h>
#include <ctype.h>
#include "3D.h"
#include "Lighting.h"
#define STACKSIZE 2048

char *malloc();

static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"Usage is\n\t%s [-h][-v][-s][-e]\n", RoutineName);
 }

static int VERBOSE = 0;    /* don't chatter on so much */
static int StackWARNING = 0;     /* don't worry about stack */
static int ErrorWARNING = -1;    /* do notice errors       */ 

static int StackTop = 0;
static double Stack[STACKSIZE];

static int inited = 0;

/*
    General purpose utilities
 */

#define Push(Value) \
/* double Value; */ \
 { \
  if (StackTop < STACKSIZE) \
   Stack[StackTop++] = Value; \
  else { \
   (void) fprintf(stderr, "\n%s ERROR: stack overflow\n", RoutineName); \
   sgpQuit(); \
   fflush(stdout); fflush(stderr); \
   abort(); \
       } \
 }

#define Pop() /* Returns double */   \
( \
 (StackTop > 0) ? \
  (Stack[--StackTop]) : \
  ( \
   (void)fprintf(stderr, "\n%s ERROR: attempt made to pop from empty stack", \
                           RoutineName), \
  sgpQuit(), \
  fflush(stdout), fflush(stderr), \
  abort(), \
  0.0 \
 ) \
)


static double GetNumber(InputFile)
 FILE *InputFile;
 {
   register int c;
   int i, sign = 1.0, power = 0, esign = 1;
   unsigned long whole = 0, num=0,denom=1;
   double result, tmp;

   /* Discard leading white space */
   while ( (c=getc(InputFile)) != EOF &&
	    isspace( c ) )
     ;				/* Null Body */

   /* Get the optional sign */
   if (c == '+' || c == '-') {
     if (c == '-') sign = -1.0;
     c = getc(InputFile);
   }

   /* Get a stream of digits */
   while ( c != EOF && isdigit(c) ) {
     whole *= 10;
     whole += c - '0';
     c = getc(InputFile);
   }

   /* Get an optional decimal point, followed by more digits */
   if (c == '.') {
     c = getc(InputFile);
     
     /* Get some more digits */
     while (c != EOF && isdigit(c) ) {
       denom*= 10;
       num *= 10;
       num += c - '0';
       c = getc(InputFile);
     }

     /* Get the optional exponent */
     if (c == 'E' || c == 'e') {
       c = getc(InputFile);
       if (c == '+')
	 c = getc(InputFile);
       else if (c == '-') {
	 esign = -1;
	 c = getc(InputFile);
       }	 

       /* Get some more digits */
       while (c != EOF && isdigit(c) ) {
	 power *= 10;
	 power += c - '0';
	 c = getc(InputFile);
       }
     }
   }
   /* Assemble the number (simplified to make the compiler happy) */
   result  =  num;
   tmp = (double) denom;
   result /=  tmp;
   tmp = (double) whole;
   result +=  tmp;
   if (power > 0)
     if (esign > 0)
       for(i=0; i<power; i++)
	 result *= 10;
     else 
       for(i=0; i<power; i++)
	 result /= 10;
   if (sign < 0)
     result = -result;
   return result;
 }

static void GobbleComment(InputFile)
 FILE *InputFile;
 {
  char ch;

  do
   ch = getc(InputFile);
  while(ch != '\n');
 }

/*
   2D SGP functions
 */

static void CallsgpInit()
 {
  sgpDisplayType SGPDisplay;

  SGPDisplay = (sgpDisplayType) Pop();
  (void) sgpInit((sgpDisplayType) SGPDisplay);
  inited = 1;
  if (VERBOSE)
   (void) fprintf(stdout, "%s: sgpInit(%d)\n", RoutineName, (int)SGPDisplay);
 }

static void CallsgpQuit()
 {
  sgpQuit();
  inited = 0;
  if (VERBOSE)
   (void) fprintf(stdout, "%s: sgpQuit()\n", RoutineName);
 }

static void CallsgpColor()
 {
  sgpColorType Color;

  Color.b = Pop();  Color.g = Pop();  Color.r = Pop();
  (void) sgpColor(Color);
  if (VERBOSE)
   (void) fprintf(stdout, "%s: sgpColor([%8.5f, %8.5f, %8.5f])\n",
                     RoutineName, Color.r, Color.g, Color.b);
 }

static void CallsgpGrayLevel()
 {
  sgpGrayType Gray;

  Gray = Pop();
  (void) sgpGrayLevel(Gray);
  if (VERBOSE)
   (void) fprintf(stdout, "%s: sgpGrayLevel(%8.5f)\n",
                      RoutineName, Gray);
 }

static void CallsgpSetWindow()
 {
  sgpRectangleType Window;

  Window.Top     = Pop();  Window.Right   = Pop();
  Window.Bottom  = Pop();  Window.Left    = Pop();
  (void) sgpSetWindow(Window);
  if (VERBOSE)
   (void) fprintf(stdout,
    "%s: sgpSetWindow([%8.5f, %8.5f, %8.5f, %8.5f])\n",
       RoutineName,
       Window.Left, Window.Bottom, Window.Right, Window.Top);
 }

static void CallsgpSetViewport()
 {
  sgpRectangleType Viewport;

  Viewport.Top    = Pop(); Viewport.Right = Pop();
  Viewport.Bottom = Pop(); Viewport.Left  = Pop();
  (void) sgpSetViewport(Viewport);
  if (VERBOSE)
   (void) fprintf(stdout,
    "%s: sgpSetviewport([%8.5f, %8.5f, %8.5f, %8.5f])\n",
       RoutineName,
       Viewport.Left, Viewport.Bottom, Viewport.Right, Viewport.Top );
 }

static void CallsgpClearScreen()
 {
  (void) sgpClearScreen();
  if (VERBOSE)
   (void) fprintf(stdout, "%s: sgpClearScreen()\n", RoutineName);
 }

static void CallsgpPoint()
 {
  sgpPointType Point;

  Point.y = Pop();  Point.x = Pop();
  (void) sgpPoint(Point);
  if (VERBOSE)
   (void) fprintf(stdout, "%s: sgpPoint([%8.5f, %8.5f])\n",
                   RoutineName, Point.x, Point.y);
 }

static void CallsgpDot()
 {
  sgpPointType Point;
  int ScreenSize;

  ScreenSize = (int) Pop();
  Point.y = Pop();  Point.x = Pop();
  (void) sgpDot(Point,ScreenSize);
  if (VERBOSE)
   (void) fprintf(stdout, "%s: sgpDot([%8.5f, %8.5f], %d)\n",
               RoutineName,  Point.x, Point.y, ScreenSize);
 }

void CallsgpPolyLine()
 {
  int i, n;
  sgpPointType *Points;

  n = StackTop >> 1;
  if (StackWARNING && ((n << 1) != StackTop))
   fprintf(stderr,"%s WARNING: sgpPolyLine called with odd stack size\n",
                   RoutineName);  
  Points = (sgpPointType *) malloc(sizeof(sgpPointType) * (unsigned) n);
  if (0 == Points) 
   {
    fprintf(stderr,"No memory for %d Points\n",n);
    exit(-1);
   }
  for (i = n-1; i >= 0; i--)
   {
    Points[i].y = Pop();    Points[i].x = Pop();
   }

  (void) sgpPolyLine(n, Points);
  if (VERBOSE)
   {
    (void) fprintf(stdout, "%s: sgpPolyLine(%d,\n\t\t\t%d\n\t\t[",
                    RoutineName,n);
    for(i = 0; i<n; i++)
     (void) fprintf(stdout,"\n\t\t\t [%8.5f, %8.5f]",
           Points[i].x, Points[i].y);
    (void) fprintf(stdout,"])\n");
   }
  (void)free(Points);
 }

static void CallsgpLine()
 {
  sgpPointType Point1, Point2;

  Point2.y = Pop();  Point2.x = Pop();
  Point1.y = Pop();  Point1.x = Pop();

  (void) sgpLine(Point1, Point2);
  if (VERBOSE)
   (void) fprintf(stdout, "%s: sgpLine([%8.5f, %8.5f], [%8.5f, %8.5f])\n", 
                  RoutineName,
                  Point1.x, Point1.y, Point2.x, Point2.y);
 }

static void CallsgpCurve()
 {
  int i, n, m;
  sgpPointType *Points;

  n = (StackTop-1)/2;
  Points = (sgpPointType *) malloc(sizeof(sgpPointType) * (unsigned) n);
  if (0 == Points) 
   {
    fprintf(stderr,"No memory for %d Points\n",n);
    exit(-1);
   }
  for (i = n-1; i >= 0; i--)
   {
    Points[i].y = Pop();  Points[i].x = Pop();
  }
  m = (int)Pop();

  (void) sgpCurve(m, n, Points);
  if (VERBOSE)
   {
    (void) fprintf(stdout, "%s: sgpCurve(%d, %d\n\t\t\t%d\n\t\t[",
                    RoutineName,m,n);
    for(i = 0; i<n; i++)
     (void) fprintf(stdout,"\n\t\t\t [%8.5f, %8.5f]",
           Points[i].x, Points[i].y);
    (void) fprintf(stdout,"])\n");
   } 
  (void)free(Points);          
 }

static void CallsgpRectangle()
 {
  sgpRectangleType Rectangle;

  Rectangle.Top    = Pop();  Rectangle.Right = Pop();
  Rectangle.Bottom = Pop();  Rectangle.Left  = Pop();

  (void) sgpRectangle(Rectangle);
  if (VERBOSE)
   (void) fprintf(stdout, "%s: sgpRectangle([%8.5f, %8.5f, %8.5f, %8.5f])\n",
                         RoutineName,
                         Rectangle.Left, Rectangle.Bottom, 
		         Rectangle.Right, Rectangle.Top );
 }

static void CallsgpCircle()
 {
  double Radius;
  sgpPointType Center;

  Radius = Pop();
  Center.y = Pop();  Center.x = Pop();
  (void) sgpCircle(Center, Radius);
  if (VERBOSE)
   (void) fprintf(stdout, "%s: sgpCircle([%8.5f, %8.5f], %8.5f)\n",
                          RoutineName,
                          Center.x, Center.y, Radius);
 }

static void CallsgpDisc()
 {
  double Radius;
  sgpPointType Center;

  Radius = Pop();
  Center.y = Pop();  Center.x = Pop();
  (void) sgpDisc(Center, Radius);
  if (VERBOSE)
   (void) fprintf(stdout, "%s: sgpDisc([%8.5f, %8.5f], %8.5f)\n",
                          RoutineName,
                          Center.x, Center.y, Radius);
 }

static void CallsgpShadeTriangle()
 {
  sgpPointType Points[3];
  sgpColorType Colors[3];
  int i;

  for (i = 2; i >=0; i--)
   {
    Colors[i].b = Pop();    Colors[i].g = Pop();    Colors[i].r = Pop();
   }
  for (i = 2; i >=0; i--)
   {
    Points[i].y = Pop();    Points[i].x = Pop();
   }
  (void) sgpShadeTriangle(Points, Colors);
  if (VERBOSE)
   {
    (void) fprintf(stdout, "%s: sgpShadeTriangle(\n", RoutineName);
    (void) fprintf(stdout,
    "[[%8.5f,%8.5f][%8.5f,%8.5f][%8.5f,%8.5f]],\n",
     Points[0].x,Points[0].y,Points[1].x,Points[1].y,Points[2].x,Points[2].y);
    (void) fprintf(stdout,
    "[[%8.5f,%8.5f,%8.5f][%8.5f,%8.5f,%8.5f][%8.5f,%8.5f,%8.5f]])\n",
     Colors[0].r,Colors[0].g,Colors[0].b,
     Colors[1].r,Colors[1].g,Colors[1].b,
     Colors[2].r,Colors[2].g,Colors[2].b);
   }
 }

static void CallsgpSolidTriangle()
 {
  sgpPointType Points[3];
  int i;

  for (i = 2; i >=0; i--)
   {
    Points[i].y = Pop();    Points[i].x = Pop();
   }
  (void) sgpSolidTriangle(Points);
  if (VERBOSE)
   {
    (void) fprintf(stdout, "%s: sgpSolidTriangle(\n", RoutineName);
    (void) fprintf(stdout,
    "[[%8.5f,%8.5f][%8.5f,%8.5f][%8.5f,%8.5f]])\n",
     Points[0].x,Points[0].y,Points[1].x,Points[1].y,Points[2].x,Points[2].y);
   }
 }

/*
   3D functions
 */

static void CallCross3D()
 {
  VectorType3D A,B,V;

  B.dz = Pop(); B.dy = Pop(); B.dx = Pop();
  A.dz = Pop(); A.dy = Pop(); A.dx = Pop();

  V = Cross3D(A,B);
  /*
    Push(V.dx);        Push(V.dy);      Push(V.dz);
   */
  if (VERBOSE)
   (void) fprintf(stdout,
         "%s: Cross3D([%8.5f,%8.5f, %8.5f],[%8.5f,%8.5f,%8.5f])\n",
                     RoutineName, A.dx,A.dy,A.dz,B.dx,B.dy,B.dz);
 }

static void CallDot3D()
 {
  VectorType3D A,B;
  double dot;

  B.dz = Pop(); B.dy = Pop(); B.dx = Pop();
  A.dz = Pop(); A.dy = Pop(); A.dx = Pop();
  dot = Dot3D(A,B);
  /*
    Push(dot); 
   */
  if (VERBOSE)
   (void) fprintf(stdout, 
        "%s: Dot3D([%8.5f,%8.5f,%8.5f],[%8.5f,%8.5f,%8.5f])\n",
                 RoutineName, A.dx,A.dy,A.dz,B.dx,B.dy,B.dz);
 }

static void CallCamera3D()
 {
  PointType3D EyePoint;
  VectorType3D OpticalAxis, Up;

  Up.dz          = Pop(); Up.dy          = Pop(); Up.dx          = Pop();
  OpticalAxis.dz = Pop(); OpticalAxis.dy = Pop(); OpticalAxis.dx = Pop();
  EyePoint.z     = Pop(); EyePoint.y     = Pop(); EyePoint.x     = Pop();

  Camera3D(EyePoint, OpticalAxis, Up);
  if (VERBOSE)
   {
    (void) fprintf(stdout,"%s: Camera3D(\n", RoutineName);
    (void) fprintf(stdout,"\t\t\t[%8.5f,%8.5f,%8.5f],\n",
                  EyePoint.x, EyePoint.y, EyePoint.z);
    (void) fprintf(stdout,"\t\t\t[%8.5f,%8.5f,%8.5f],\n",
                  OpticalAxis.dx, OpticalAxis.dy, OpticalAxis.dz);
    (void) fprintf(stdout,"\t\t\t[%8.5f,%8.5f,%8.5f])\n",
                  Up.dx, Up.dy, Up.dz);
   }
 }

static void CallLens3D()
 {
  double f;

  f = Pop();
  (void)Lens3D(f);
  if (VERBOSE)
   (void) fprintf(stdout,"%s: Lens3D(%f)\n", RoutineName,f);
 }

static void CallWorldToViewBox3D()
 {
  PointType3D World, ViewBox;

  World.z = Pop();     World.y = Pop();    World.x = Pop();
  WorldToViewBox3D(World, &ViewBox);
  /*
    Push(ViewBox.x);        Push(ViewBox.y);      Push(ViewBox.z);  
   */
  if (VERBOSE)
   (void) fprintf(stdout,"%s: WorldToViewBox3D([%8.5f,%8.5f,%8.5f])\n",
                   RoutineName,World.x,World.y,World.z);
 }

static void CallWorld3DTosgpWorld()
 {
  PointType3D World;
  sgpPointType sgpWorld;

  World.z = Pop();     World.y = Pop();    World.x = Pop();
  World3DTosgpWorld(World, &sgpWorld);
  /*
    Push(sgpWorld.x);        Push(sgpWorld.y);
   */
  if (VERBOSE)
   (void) fprintf(stdout,"%s: World3DTosgpWorld([%8.5f,%8.5f,%8.5f])\n",
                   RoutineName,World.x,World.y,World.z);
 }

static void CallPoint3D()
 {
  PointType3D Point;

  Point.z = Pop(); Point.y = Pop(); Point.x = Pop();
  (void)Point3D(Point);
  if (VERBOSE)
   (void) fprintf(stdout,"%s: Point3D([%8.5f,%8.5f,%8.5f])\n",
                   RoutineName,Point.x,Point.y,Point.z);
 }

static void CallLine3D()
 {
  PointType3D Point1,Point2;

  Point2.z = Pop(); Point2.y = Pop(); Point2.x = Pop();
  Point1.z = Pop(); Point1.y = Pop(); Point1.x = Pop();

  (void)Line3D(Point1, Point2);
  if (VERBOSE)
   (void) fprintf(stdout,
      "%s: Line3D([%8.5f,%8.5f,%8.5f],[%8.5f,%8.5f,%8.5f])\n",
      RoutineName,Point1.x,Point1.y,Point1.z,Point2.x,Point2.y,Point2.z);
 }

static void CallShadeTriangle3D()
 {
  PointType3D Points[3];
  sgpColorType Colors[3];
  int i;

  for (i = 2; i >=0; i--)
   {
    Colors[i].b = Pop();    Colors[i].g = Pop();    Colors[i].r = Pop();
   }
  for (i = 2; i >=0; i--)
   {
    Points[i].z = Pop();    Points[i].y = Pop();    Points[i].x = Pop();
   }
  (void) ShadeTriangle3D(Points, Colors);
  if (VERBOSE)
   {
    (void) fprintf(stdout, "%s: ShadeTriangle3D(\n", RoutineName);
    (void) fprintf(stdout, 
     "\t\t[[%8.5f,%8.5f,%8.5f][%8.5f,%8.5f,%8.5f][%8.5f,%8.5f,%8.5f]],\n",
      Points[0].x,Points[0].y,Points[0].z,
      Points[1].x,Points[1].y,Points[1].z,
      Points[2].x,Points[2].y,Points[2].z);
    (void) fprintf(stdout, 
     "\t\t[[%8.5f,%8.5f,%8.5f][%8.5f,%8.5f,%8.5f][%8.5f,%8.5f,%8.5f]])\n",
      Colors[0].r,Colors[0].g,Colors[0].b,
      Colors[1].r,Colors[1].g,Colors[1].b,
      Colors[2].r,Colors[2].g,Colors[2].b);
   }
 }

static void CallSolidTriangle3D()
 {
  PointType3D Points[3];
  int i;

  for (i = 2; i >=0; i--)
   {
    Points[i].z = Pop(); Points[i].y = Pop();    Points[i].x = Pop();
   }
  (void) SolidTriangle3D(Points);
  if (VERBOSE)
   {
    (void) fprintf(stdout, "%s: SolidTriangle3D(\n", RoutineName);
    (void) fprintf(stdout, 
     "\t\t[[%8.5f,%8.5f,%8.5f][%8.5f,%8.5f,%8.5f][%8.5f,%8.5f,%8.5f]],\n",
      Points[0].x,Points[0].y,Points[0].z,
      Points[1].x,Points[1].y,Points[1].z,
      Points[2].x,Points[2].y,Points[2].z);
   }
 }


static void CallSetExposure()
 {
  double Exposure;

  Exposure = Pop();
  (void) SetExposure(Exposure);
  if (VERBOSE)
   (void) fprintf(stdout, "%s: SetExposure(%f)\n",
     RoutineName,Exposure);
 }

static void CallSetAmbient()
 {
  sgpColorType Ambient;

  Ambient.b = Pop();  Ambient.g = Pop();  Ambient.r = Pop();
  (void) SetAmbient(Ambient);
  if (VERBOSE)
   (void) fprintf(stdout, "%s: SetAmbient([%8.5f,%8.5f,%8.5f])\n",
      RoutineName,Ambient.r,Ambient.g,Ambient.b);
 }

static void CallAddLightSource()
 {
  PointType3D Location;
  sgpColorType Color;

  Color.b = Pop();  Color.g = Pop();  Color.r = Pop();
  Location.z = Pop();  Location.y = Pop();  Location.x = Pop();
  (void) AddLightSource(Location,Color);
  if (VERBOSE)
   (void) fprintf(stdout, 
     "%s: AddLightSource([%8.5f,%8.5f,%8.5f],[%8.5f,%8.5f,%8.5f])\n",
        RoutineName,Location.x,Location.y,Location.z,Color.r,Color.g,Color.b);
 }

static void CallSetSurfaceReflectance()
 {
  sgpColorType Diffuse, Specular;
  double Specularity;

  Specularity = Pop();
  Specular.g = Pop(); Specular.b = Pop(); Specular.r = Pop();
  Diffuse.g = Pop(); Diffuse.b = Pop(); Diffuse.r = Pop();
  (void) SetSurfaceReflectance(Diffuse, Specular, Specularity);
  if (VERBOSE)
   {
    (void) fprintf(stdout, "%s: SetSurfaceReflectance(\n", RoutineName);
    (void) fprintf(stdout, "\t\t[%8.5f,%8.5f,%8.5f],\n",
             Diffuse.r, Diffuse.g, Diffuse.b);
    (void) fprintf(stdout, "\t\t[%8.5f,%8.5f,%8.5f],\n",
             Specular.r, Specular.g, Specular.b);
    (void) fprintf(stdout,"\t\t%f)\n",Specularity);
   }
 }

static void CallColorAtPoint()
 {
  PointType3D Point;
  VectorType3D Normal;
  sgpColorType Color;

  Normal.dz = Pop();  Normal.dy = Pop();  Normal.dx = Pop();
  Point.z = Pop();  Point.y = Pop();  Point.x = Pop();
  Color = ColorAtPoint(Point,Normal);
  /*
    Push(Color.r); Push(Color.g); Push(Color.b);
   */
  if (VERBOSE)
   (void) fprintf(stdout,
      "%s: ColorAtPoint([%8.5f,%8.5f,%8.5f],[%8.5f,%8.5f,%8.5f])\n",
        RoutineName,Point.x,Point.y,Point.z,Normal.dx,Normal.dy,Normal.dz);
 }

static void PushsgpMonochromeDisplay()
{
  Push( (double) MonochromeDisplay );
}

static void PushsgpColorDisplay()
{
  Push( (double) ColorDisplay );
}

static void PushsgpDebugDisplay()
{
  Push( (double) DebugDisplay );
}

static void IgnoreSave() {return;}
static void IgnoreRestore(){return;}
 

typedef struct 
{
  char name[32];		/* Name MUST be first for strcmp */
  void (*func)();
} FunctionBinding;

static FunctionBinding FunctionList[] = {
    /* 2D commands - fully supported */
  { "sgpInit",        CallsgpInit, },
  { "sgpQuit",        CallsgpQuit, },
  { "sgpColor",       CallsgpColor, },
  { "sgpGrayLevel",   CallsgpGrayLevel, },
  { "sgpSetWindow",   CallsgpSetWindow, },
  { "sgpSetViewport", CallsgpSetViewport, },
  { "sgpClearScreen", CallsgpClearScreen, },
  { "sgpPoint",       CallsgpPoint, },
  { "sgpDot",         CallsgpDot, },
  { "sgpPolyLine",    CallsgpPolyLine, },
  { "sgpLine",        CallsgpLine, },
  { "sgpCurve",       CallsgpCurve, },
  { "sgpRectangle",   CallsgpRectangle, },
  { "sgpCircle",      CallsgpCircle, },
  { "sgpDisc",        CallsgpDisc, },
  { "sgpShadeTriangle", CallsgpShadeTriangle, },
  { "sgpShadeTriangle:", CallsgpShadeTriangle, }, /* Upwards compatablity */
  { "sgpSolidTriangle", CallsgpSolidTriangle, },
  { "MonochromeDisplay", PushsgpMonochromeDisplay, },
  { "ColorDisplay", PushsgpColorDisplay, },
  { "DebugDisplay", PushsgpDebugDisplay, },
  { "sgpMonochromeDisplay", PushsgpMonochromeDisplay, },
  { "sgpColorDisplay", PushsgpColorDisplay, },
  { "sgpDebugDisplay", PushsgpDebugDisplay, },

    /* 3D commands - use 'em if you can figure out how... */ 
  { "Cross3D", CallCross3D, },
  { "Dot3D", CallDot3D, },
  { "Camera3D", CallCamera3D, },
  { "Lens3D",   CallLens3D, },
  { "WorldToViewBox3D", CallWorldToViewBox3D },
  { "World3DTosgpWorld", CallWorld3DTosgpWorld, },
  { "Point3D", CallPoint3D, },
  { "Line3D",  CallLine3D, },
  { "ShadeTriangle3D", CallShadeTriangle3D, },
  { "SolidTriangle3D", CallSolidTriangle3D, },
  
  /* Lighting commands - good luck */
  { "SetExposure", CallSetExposure, },
  { "SetAmbient", CallSetAmbient, },
  { "AddLightSource", CallAddLightSource, },
  { "SetSurfaceReflectance", CallSetSurfaceReflectance, },
  { "ColorAtPoint", CallColorAtPoint, },

  /* allow save/restore */
  { "save", IgnoreSave, }, 
  { "restore", IgnoreRestore, }, 
};

#define NFUNCTIONS (sizeof(FunctionList) / sizeof(FunctionBinding) )

extern int strcmp();
#define CompareFunctionBindings strcmp

static void SortFunctionBindings()
{
  /* Sorting is only needed if you want to use binary search in
  ** DoFunctionCall.
  */
  qsort( (char *) FunctionList, NFUNCTIONS, sizeof(FunctionBinding), 
	CompareFunctionBindings );
}
  
  
static void DoFunctionCall(FunctionString)
 char *FunctionString;
  {
    FunctionBinding key, *match;
    extern char *bsearch();

    strncpy( key.name , FunctionString, 32 );
    key.name[31] = '\0';

    /* If you don't have a bsearch, write your own!  If you are lazy you
    ** can write it  a sequential search instead.
    */
    match = (FunctionBinding *) 
      bsearch( (char *) &key,  (char *)FunctionList, NFUNCTIONS, 
	      sizeof(FunctionBinding), CompareFunctionBindings );
    if (match)  {
      (*match->func)();
/*      if (inited) FlushCmds(); */
    } else if (ErrorWARNING)
         (void) fprintf(stderr, "\n%s ERROR: unrecognized function name %s\n",
                                RoutineName, FunctionString);

    if (StackWARNING && (StackTop > 0))
     {
      if (  (0 != strcmp("sgpMonochromeDisplay",FunctionString))
          &&(0 != strcmp("sgpColorDisplay", FunctionString))
          &&(0 != strcmp("sgpDebugDisplay", FunctionString))
          &&(0 != strcmp("MonochromeDisplay", FunctionString))
          &&(0 != strcmp("ColorDisplay", FunctionString))
          &&(0 != strcmp("DebugDisplay", FunctionString)) )
       (void) fprintf(stderr, 
            "\n%s WARNING: %d arguments remain on stack after Call to %s\n",
                        RoutineName, StackTop, FunctionString);
     }
  }

static void ParseFile(InputFile)
 FILE *InputFile;
 {
  char ch;
  char FunctionString[81];

  do
   {
    ch = getc(InputFile);
    switch (ch)
     {
      case '[': break;                            /* ignore brackets */
      case ']': break;
      case '%': GobbleComment(InputFile); break;  /* comments to eol */
      case ' ': break;                            /* ignore white space */
      case '\t': break;
      case '\n': break;
      default:
       if (isalpha(ch))
        {
         (void) ungetc(ch, InputFile);
         (void) fscanf(InputFile, "%80s", FunctionString);
         DoFunctionCall(FunctionString);
        }
       else if (isdigit(ch) || ('.' == ch) || ('-' == ch) || ('+' == ch))
        {
         (void) ungetc(ch, InputFile); /* put the character back */
         Push(GetNumber(InputFile)); /* we push the number on the stack */
        }
       else
        {
         if (!(feof(InputFile)))
          (void) fprintf(stderr, "Unrecognized input character\n");
        }
       break;
     } 
   }
  while (!(feof(InputFile)));
 }

main (argc, argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;

  RoutineName = argv[ArgsParsed++];

  while (ArgsParsed < argc)
   {
    if ('-' != argv[ArgsParsed][0]) { usage(); exit (-1); }
    switch (argv[ArgsParsed++][1])
      {
       case 'e': ErrorWARNING = -1; break;
       case 's': StackWARNING = -1; break;
       case 'v': VERBOSE = -1; break;
       default:
       case 'h': { usage(); exit (-1); }
      }
   }
  SortFunctionBindings();	/* So we can binary search */
  ParseFile(stdin, stdout);

  exit(0);
 } 
