/*
  File: 3D.c
  Author: K.R. Sloan
  Last Modified: 4 March 1991
  Purpose: a 3D extension to the Simple Graphics Package
 */
#include <stdio.h>
#include <math.h>
#include "3D.h"
#include "4D.h"
double fabs();

/* 3D GLOBALS */
static double epsilon  = 0.00000000001;
static double INFINITY = 10000000000.0;

typedef  struct State
                 { 
                  PointType3D EyePoint;
                  VectorType3D OpticalAxis, Up; 
                  double FocalLength, Hither;
                  TransformType4D Trans, Rot, Pers, T;
                 }
                StateType;
StateType CurrentState =
           {
            {0.0, 0.0, -1.0},            /* EyePoint on negative z axis  */
            {0.0, 0.0, 1.0},             /* Looking at the origin        */
            {0.0, 1.0, 0.0},             /* y-axis is up                 */
            0.0, 0.0,                    /* no perspective, hither at eye*/

            /* 
              be careful, these LOOK like the transpose 
              for example, the first matrix below translates by 1.0 in z
             */
            { 
              {1.0, 0.0, 0.0, 0.0},
              {0.0, 1.0, 0.0, 0.0},
              {0.0, 0.0, 1.0, 1.0}, 
              {0.0, 0.0, 0.0, 1.0},
            },   
            { 
              {1.0, 0.0, 0.0, 0.0},
              {0.0, 1.0, 0.0, 0.0},
              {0.0, 0.0, 1.0, 0.0}, 
              {0.0, 0.0, 0.0, 1.0},
            },   
            { 
              {1.0, 0.0, 0.0, 0.0},
              {0.0, 1.0, 0.0, 0.0},
              {0.0, 0.0, 1.0, 0.0}, 
              {0.0, 0.0, 0.0, 1.0},
            }   
           };

/* Private functions */

/* Exported Functions */

/*
  Clip a line against the plane perpendicular to the OpticalAxis, 
  at distance FocalLength from the EyePoint.
 */
int ClipLineHither(Point1, Point2)
 PointType3D *Point1, *Point2;
 {
  VectorType3D V1,V2;
  double l1, l2, t, h;

  V1.dx = Point1->x - CurrentState.EyePoint.x;
  V1.dy = Point1->y - CurrentState.EyePoint.y;
  V1.dz = Point1->z - CurrentState.EyePoint.z;

  V2.dx = Point2->x - CurrentState.EyePoint.x;
  V2.dy = Point2->y - CurrentState.EyePoint.y;
  V2.dz = Point2->z - CurrentState.EyePoint.z;

  /*
    |CurrentState.OpticalAxis| = 1.0 (see Camera3D, below)
   */

  l1 = Dot3D(V1,CurrentState.OpticalAxis);
  l2 = Dot3D(V2,CurrentState.OpticalAxis); 

  h = CurrentState.Hither;

  if ( (l1 <= h) && (l2 <= h)) return(0);   /* invisible */

  if (l1 <= h)
   {
    if (epsilon > (l2-l1)) t = 1.0; 
    else                   t =  (h-l1)/(l2-l1);
    Point1->x = Point1->x + (t * ((Point2->x)-(Point1->x)));
    Point1->y = Point1->y + (t * ((Point2->y)-(Point1->y)));
    Point1->z = Point1->z + (t * ((Point2->z)-(Point1->z)));
   } 
  else if (l2 <= h)
   {
    if (epsilon > (l1-l2)) t = 1.0; 
    else                   t =  (h-l2)/(l1-l2);
    Point2->x = Point2->x + (t * ((Point1->x)-(Point2->x)));
    Point2->y = Point2->y + (t * ((Point1->y)-(Point2->y)));
    Point2->z = Point2->z + (t * ((Point1->z)-(Point2->z)));
   } 
  return(1);
 }

VectorType3D Cross3D(a, b)
 VectorType3D a, b;
 {
  VectorType3D c;
  c.dx = a.dy*b.dz - a.dz*b.dy;
  c.dy = a.dz*b.dx - a.dx*b.dz;
  c.dz = a.dx*b.dy - a.dy*b.dx;
  return(c);
 }

double Dot3D(a, b)
 VectorType3D a, b;
 {
  return(a.dx*b.dx + a.dy*b.dy + a.dz*b.dz);
 } 

void Normalize3D(V)
 VectorType3D *V;
 {
  double length;

  if (fabs(V->dx) > INFINITY) { V->dx = 1.0; V->dy /= V->dx; V->dz /= V->dx; }
  if (fabs(V->dy) > INFINITY) { V->dy = 1.0; V->dz /= V->dy; V->dx /= V->dy; }
  if (fabs(V->dz) > INFINITY) { V->dz = 1.0; V->dx /= V->dz; V->dy /= V->dz; }
  length= sqrt(V->dx*V->dx + V->dy*V->dy + V->dz*V->dz);
  if (length<epsilon) length = epsilon;
  V->dx /= length; V->dy /= length; V->dz /= length;
 }

void GetCamera3D(EyePoint, OpticalAxis, Up)
 PointType3D *EyePoint;
 VectorType3D  *OpticalAxis,*Up;
 {
  *EyePoint = CurrentState.EyePoint;
  *OpticalAxis = CurrentState.OpticalAxis;
  *Up = CurrentState.Up;
 }

void Camera3D(NewEyePoint, NewOpticalAxis, NewUp)
 PointType3D NewEyePoint;
 VectorType3D NewOpticalAxis, NewUp;
 {
  VectorType3D Rz, Rx, Ry;
  PointType4D PT;
  double l;
  int i;

  /* save these two, as is */
  CurrentState.EyePoint = NewEyePoint;
  CurrentState.Up = NewUp;

  /* normalize the OpticalAxis to length = 1.0, and prepare to rotate */
  Rz = NewOpticalAxis;
  l = sqrt(Dot3D(Rz,Rz));
  if (l < epsilon)
   {
    fprintf(stderr,"Camera3D: Optical Axis is too short. Using [0 0 -1]\n");
    Rz.dx = 0.0; Rz.dy = 0.0; Rz.dz = -1.0;
   }
  else { Rz.dx /= -l; Rz.dy /= -l; Rz.dz /= -l; } /* left handed... */
   /*
     save the NORMALIZED Optical Axis
     ClipLineHither (above) depends on this!
    */
  CurrentState.OpticalAxis.dx = -Rz.dx;
  CurrentState.OpticalAxis.dy = -Rz.dy;
  CurrentState.OpticalAxis.dz = -Rz.dz;

  Rx = Cross3D(CurrentState.OpticalAxis, NewUp);
  l = sqrt(Dot3D(Rx,Rx));
  if (l < epsilon)
   {
    fprintf(stderr,"Camera3D: Axis x Up is too short. Using [1 0 0]\n");
    Rx.dx = 1.0; Rx.dy = 0.0; Rx.dz = 0.0;
   }
  else { Rx.dx /= l; Rx.dy /= l; Rx.dz /= l; }

  Ry = Cross3D(Rx, CurrentState.OpticalAxis);
  l = sqrt(Dot3D(Ry,Ry));
  if (l < epsilon)
   {
    fprintf(stderr,
     "Camera3D: (Axis x Up) x Axis is too short. Using [0 1 0]\n");
    Ry.dx = 0.0; Ry.dy = 1.0; Ry.dz = 0.0;
   }
  else { Ry.dx /= l; Ry.dy /= l; Ry.dz /= l; }

  /*
    prepare the Translation and Rotation matrices
   */ 
  PT.P[0] = NewEyePoint.x;
  PT.P[1] = NewEyePoint.y;
  PT.P[2] = NewEyePoint.z;
  PT.P[3] = -1.0; /* go the other way */
  CurrentState.Trans = Translate4D(PT);

  CurrentState.Rot = Identity4D();
  CurrentState.Rot.T[0][0] = Rx.dx;
  CurrentState.Rot.T[1][0] = Ry.dx;
  CurrentState.Rot.T[2][0] = Rz.dx; 
 
  CurrentState.Rot.T[0][1] = Rx.dy; 
  CurrentState.Rot.T[1][1] = Ry.dy;
  CurrentState.Rot.T[2][1] = Rz.dy; 

  CurrentState.Rot.T[0][2] = Rx.dz; 
  CurrentState.Rot.T[1][2] = Ry.dz;
  CurrentState.Rot.T[2][2] = Rz.dz; 
 
  /*
    compose Translation, Rotation, Perspective, and save
   */
  CurrentState.T = TxT4D(TxT4D(CurrentState.Trans, CurrentState.Rot),
                         CurrentState.Pers);
 }

void GetLens3D(FocalLength)
 double *FocalLength;
 {
  *FocalLength = CurrentState.FocalLength;
 }

void Lens3D(NewFocalLength)
 double NewFocalLength;
 {
  CurrentState.FocalLength = NewFocalLength;
  if (0.0 == CurrentState.FocalLength)
   CurrentState.Pers = Identity4D();
  else
   CurrentState.Pers = Perspective4D(-NewFocalLength); /* trust me */
  /*
    Update the composite transformation (see Camera3D, above)
   */
  CurrentState.T = TxT4D(TxT4D(CurrentState.Trans, CurrentState.Rot),
                         CurrentState.Pers);
 }

void GetHither(hither)
 double *hither;
 {
  *hither = CurrentState.Hither;
 }

void SetHither(hither)
 double hither;
 {
  if (0.0 > hither)
   {
    fprintf(stderr,"SetHither: negative Hither reset to 0.0\n");
    hither=0.0;
   }
  CurrentState.Hither = hither;
 }

void WorldToViewBox3D(World, ViewBox)
 PointType3D World;
 PointType3D *ViewBox;
 {
  PointType4D W, S;
  double w;

  W.P[0] = World.x; W.P[1] = World.y; W.P[2] = World.z; W.P[3] = 1.0;
  S = PxT4D(W, CurrentState.T);
  w = S.P[3];
  if ((w < epsilon) && (w > (-epsilon)))
   {
    fprintf(stderr,
             "WorldToViewBox3D: at infinity. Normalizing to unit length\n");
    w = sqrt(S.P[0]*S.P[0] + S.P[1]*S.P[1] + S.P[2]*S.P[2]);
    if (S.P[3] < 0.0) w = (-w);
   }

  ViewBox->x = S.P[0]/w;  ViewBox->y = S.P[1]/w;  ViewBox->z = S.P[2]/w;
 }

void World3DTosgpWorld(World3D, sgpWorld)
 PointType3D World3D;
 sgpPointType *sgpWorld;
 {
  PointType4D W, S;
  double w;

  W.P[0] = World3D.x; W.P[1] = World3D.y; W.P[2] = World3D.z; W.P[3] = 1.0;
  S = PxT4D(W, CurrentState.T);
  w = S.P[3];
  if ((w < epsilon) && (w > (-epsilon)))
   {
    fprintf(stderr,
             "World3DTosgpWorld: at infinity. Normalizing to unit length\n");
    w = sqrt(S.P[0]*S.P[0] + S.P[1]*S.P[1] + S.P[2]*S.P[2]);
    if (S.P[3] < 0.0) w = (-w);
   }
  sgpWorld->x = S.P[0]/w;  sgpWorld->y = S.P[1]/w;
 }

void Point3D(Point)
 PointType3D Point;
 {
  sgpPointType sgpWorld;

  if (0 == ClipLineHither(&Point,&Point)) return; /* invisible */
  World3DTosgpWorld(Point, &sgpWorld);
  sgpPoint(sgpWorld);
 }

void Line3D(Point1, Point2)
 PointType3D Point1, Point2;
 {
  sgpPointType sgpWorld1, sgpWorld2;

  if (0 == ClipLineHither(&Point1,&Point2)) return; /* invisible */
  World3DTosgpWorld(Point1, &sgpWorld1); 
  World3DTosgpWorld(Point2, &sgpWorld2); 
  sgpLine(sgpWorld1, sgpWorld2);
 }

void ShadeTriangle3D(Points, Colors)
 PointType3D Points[3];
 sgpColorType Colors[3];
 {
  sgpPointType sgpWorld[3];
  int i;

  /* cheap trick...for now.  Cull triangles, rather than clip them */
  for (i=0; i<3; i++)
   {
    if (0 == ClipLineHither(&Points[i],&Points[i])) return; /* trouble */
    World3DTosgpWorld(Points[i], &sgpWorld[i]);
   }
  sgpShadeTriangle(sgpWorld, Colors);
 }

void SolidTriangle3D(Points)
 PointType3D Points[3];
 {
  sgpPointType sgpWorld[3];
  int i;

  /* cheap trick...for now.  Cull triangles, rather than clip them */
  for (i=0; i<3; i++)
   {
    if (0 == ClipLineHither(&Points[i],&Points[i])) return; /* trouble */
    World3DTosgpWorld(Points[i], &sgpWorld[i]);
   }
  sgpSolidTriangle(sgpWorld);
 }
